#-------------------------------------------------------------------------------
#  pdfContainerSpool.pm:
#
#  Process AFP spool where documents have PDF container content
#  This module ensures to embed the PDF containers AS IS in the PDF output
#  (instead of rasterizing the PDF container pages as image)
#
#  Call:
#
#  On Windows: afp2web.exe -q -c -doc_cold -sp:pdfContainerSpool.pm samples\neu.afp
#
#  On Unix:    ./afp2web   -q -c -doc_cold -sp:pdfContainerSpool.pm samples/neu.afp
#
#  Author  : Fa. Maas
#
#  V100    2025-08-06    AFP-1252: Gothaer: Initial Release
#
#-------------------------------------------------------------------------------
#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;
    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }
        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }
    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
    unshift( @INC, $sScriptFilePathTmp . "/a2w" );
    unshift( @INC, $sScriptFilePathTmp . "/perl/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/perl/lib/perl5" );
    unshift( @INC, $sScriptFilePathTmp . "/perl/site/lib" );
}

use a2w::Config;
use a2w::Kernel;
use a2w::Document;
use a2w::Page;

use a2w::core::ext::Image;
use a2w::core::ext::Container;
use a2w::core::ext::Line;
use a2w::core::ext::Text;
use a2w::core::ext::Vector;

use a2w::ConfigConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;
use a2w::FontConstants;

use Time::HiRes;
use Digest::MD5;
use File::Path qw(mkpath);
use File::Copy;             # For renaming files across filesystems
use File::Spec;
use File::Basename;
use Storable qw( dclone );

use Encode;                 # To translate filename from ANSI to UTF-8
use PDF::API2;              # To extract small (when compared with AFP page size) container pages

use a2w::core::log::Logger; # Log engine

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){
    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Fetch start time
    $tStart = time;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sf") >= 0 ){
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sIndexFilePath    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::INDEXPATH );
    $sTempPath         = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::TEMPPATH );
    $sLogPath          = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGPATH );
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
    $sDocType          = $a2wConfigPar->getAttribute( "OutputFormat" );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    #---- Fetch AFP2web version
    $sA2WVersion = $a2wKernelPar->getVersion();

    #---- Instantiate Logger
    $theLogger = undef;
    $bDebugLog = $FALSE;
    if ( $bLog == $TRUE ){
        #---- Set log on log engine
        $theLogger = a2w::core::log::Logger->getSingleton();
        $theLogger->setStartTime( $sStartTime );

        #---- Register modules that has to be logged
        $theLogger->registerClasses("a2w::Main");

        #---- Open log file
        my $sLogFilenameTmp = $sSpoolFilename;
        $sLogFilenameTmp =~ s/^.*[\\\/]//;
        $sLogFilenameTmp .= ".sf.log";
        $theLogger->setFilename( $sLogFilenameTmp );
        if ( $theLogger->open( $sLogPath ) == $FALSE ){
            return ( -1, "[ERROR] Unable to open log file (" . $sLogPath . $sLogFilenameTmp . ")" );
        }
        $bLog = $theLogger->isRegistered( "a2w::Main" );

        if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sfdbg") >= 0 ){
            $theLogger->setLevel( $a2w::core::log::Logger::LEVEL_DEBUG );
            $bDebugLog = $TRUE;
        }

        $theLogger->logFunctionName( "a2w::Main", "initialize()" );
        $theLogger->logMessage( "Running $sScriptProcTmp..." );
        $theLogger->logMessage( "initialize(): Processing " . $sSpoolFilename );
        $theLogger->logMessage( "initialize(): Args: $sScriptArgsTmp, OutputFilePath: $sOutputFilePath, LogPath: $sLogPath, TempPath: $sTempPath" );
    }

    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document

    # Initalize document and page count
    $iPageCount = 0;
    $iDocCount  = 0;

    #---- Input spool file path
    $sAFPFilePath = dirname($sSpoolFilename);

    #---- Time stamp
    # (sec,min,hour,mday,mon,year,wday,yday,isdst) = localtime( time );
    @StartTime = localtime( $tStart );
    $sStartTimeStamp = sprintf( "%04d%02d%02d_%02d%02d%02d",
                              ( @StartTime[ 5 ] + 1900 ),
                              ( @StartTime[ 4 ] + 1 ),
                              @StartTime[ 3 ],
                              @StartTime[ 2 ],
                              @StartTime[ 1 ],
                              @StartTime[ 0 ]
                            );

    #---- Handles of PDF containers
    # {
    #     <container name>: <PDF::API2 handle>
    # }
    $hrefPDFHandles = {};

    # Current document info
    #
    # Syntax: {
    #     'pageCount':        <Output document page count>
    #   , 'firstPageId':      <Id of first output page as in AFP spool>
    #   , 'lastPageId':       <Id of last output page as in AFP spool>
    #   , 'pagesInfo':        <Array of AFP page ids>
    #   , 'containerInfo': {
    #         <Page number as in output document>: {
    #             'cx':       <Container X position in page>
    #           , 'cy':       <Container Y position in page>
    #           , 'name':     <Container name>
    #           , 'file':     <Container filename>
    #           , 'pgid':     <Container page number that has to be presented on current AFP page>
    #         }
    #     }
    # }
    #
    $hrefCurrentDoc = {};

    #---- Empty page constants
    $iPageResolution = 300;  # 300 dpi
    $iPageWidth      = int(( 210 / 25.4 ) * $iPageResolution); # A4 width (210 mm) in page resolution
    $iPageHeight     = int(( 297 / 25.4 ) * $iPageResolution); # A4 height (297 mm) in page resolution

    #-------------------------------------------------------------------
    #    Setting AutoSplit to true means that document splitting will
    #    done based on the standard AFP index details. In this case the 
    #    return value 2 of the afp2web() function is ignored.
    #
    #    Setting AutoSplit to false means that document splitting will
    #    be defined by the afp2web() function.
    #-------------------------------------------------------------------
    $a2wConfigPar->setAttribute( $a2w::ConfigConstants::AUTOSPLIT, "on" );

    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "initializeDoc()" );
        $theLogger->logMessage( "Name=" . $a2wDocumentPar->getName() . " Id=" . $a2wDocumentPar->getId() );
    }

    # Current document info
    #
    # Syntax: {
    #     'pageCount':        <Output document page count>
    #   , 'firstPageId':      <Id of first output page as in AFP spool>
    #   , 'lastPageId':       <Id of last output page as in AFP spool>
    #   , 'pagesInfo':        <Array of AFP page ids>
    #   , 'containerInfo': {
    #         <Page number as in output document>: {
    #             'cx':       <Container X position in page>
    #           , 'cy':       <Container Y position in page>
    #           , 'name':     <Container name>
    #           , 'file':     <Container filename>
    #           , 'pgid':     <Container page number that has to be presented on current AFP page>
    #         }
    #     }
    # }
    #
    $hrefCurrentDoc = {
        'pageCount'     => 0
      , 'firstPageId'   => 0
      , 'lastPageId'    => 0
      , 'pagesInfo'     => []
      , 'containerInfo' => {}
    };

    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "initializePage()" ); }

    #---- Increment page count
    $iPageCount++;

    $hrefCurrentDoc->{ 'pageCount' } += 1;

    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

    my $iPgIdTmp = $a2wPagePar->getParseId();
    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "afp2web()" );
        $theLogger->logMessage( "PageId=" . $iPgIdTmp );
    }

    #---- Fill in first and last page info in current doc
    my $iDocPageIdTmp = $hrefCurrentDoc->{ 'pageCount' };
    if ( $hrefCurrentDoc->{ 'firstPageId' } == 0 ){ $hrefCurrentDoc->{ 'firstPageId' } = $iPgIdTmp; }
    $hrefCurrentDoc->{ 'lastPageId' } = $iPgIdTmp;
    push( @{ $hrefCurrentDoc->{ 'pagesInfo' } }, $iPgIdTmp );

    # If page has been applied with PDF container, then remove container from output presentation
    my $bPDFPageTmp = _isPDFContainerPage( $a2wPagePar );
    if ( $bPDFPageTmp == $TRUE ){
        #---- Get first container
        my $a2wContTmp = $a2wPagePar->getFirstContainer();

        # Fill in current document applied pdf container info
        #
        # Syntax: {
        #     <page number as in output document>: {
        #         'cx':       <Container X position in page>
        #       , 'cy':       <Container Y position in page>
        #       , 'name':     <Container name>
        #       , 'file':     <Container filename>
        #       , 'pgid':     <Container page number that has to be presented on current AFP page>
        #     }
        # }
        #
        my $hrefContInfoTmp = {};

        $hrefContInfoTmp->{ 'cx' }      = $a2wContTmp->getXPos();
        $hrefContInfoTmp->{ 'cy' }      = $a2wContTmp->getXPos();
        $hrefContInfoTmp->{ 'name' }    = $a2wContTmp->getName();
        $hrefContInfoTmp->{ 'file' }    = $a2wContTmp->getFilename();
        $hrefContInfoTmp->{ 'pgid' }    = ( $a2wContTmp->getPaginatedObjectOffset() + 1 );
        $hrefCurrentDoc->{ 'containerInfo' }{ $iDocPageIdTmp } = $hrefContInfoTmp;

        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Page " . $iPgIdTmp . " has only PDF container content" );
            #$theLogger->logHashMessage( $hrefContInfoTmp );
        }

        #---- Create empty page and replace current page with it
        my $a2wEmptyPageTmp = new a2w::Page();
        $a2wEmptyPageTmp->setResolution( $iPageResolution );
        $a2wEmptyPageTmp->setWidth( $iPageWidth );
        $a2wEmptyPageTmp->setHeight( $iPageHeight );

        #---- Add empty page to current document
        $a2wDocumentPar->addPage( $a2wEmptyPageTmp, 0 ); # 0 means append
    }

    # Ensure to skip the AFP page with PDF container content alone
    return ( $bPDFPageTmp == $TRUE ) ? $SKIP : $APPEND;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "finalizePage()" ); }

    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "finalizeDoc()" ); }

    if ( $a2wDocumentPar->getPageCount() <= 0 ){
	    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Document contain no pages, skipping ..." ); }
        $hrefCurrentDoc = {}; # Cleanup document info
        return 0;
    }

    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Document: " . $a2wDocumentPar->getId() . ", Page count: " . $a2wDocumentPar->getPageCount() ); }

    #---- Recreate the core generated PDF with coded PDF container pages
    my ($iRcTmp, $sMsgTmp) = _checkAndCreateCodedPDFOutput($a2wDocumentPar);
    if ( $iRcTmp < 0 ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Failed to create coded PDF output from PDF containers, Reason: " . $sMsgTmp ); }
        $hrefCurrentDoc = {}; # Cleanup document info
        return ($iRcTmp, $sMsgTmp);
    }

    #---- Increment document count
    $iDocCount++;

    $hrefCurrentDoc = {}; # Cleanup document info

    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "finalize()" ); }

    #---- Fetch start time
    $tEnd = time;

    #---- Fetch exit status from kernel
    my $iA2WExitStatusTmp = $a2wKernelPar->getExitStatus();
	if ( $bLog == $TRUE ){ $theLogger->logMessage( "Kernel exit status: " . $iA2WExitStatusTmp ); }

    # Cleanup the opened container pdf handles
    my @arrCntKeysTmp = keys( %{ $hrefPDFHandles } );
    foreach my $k ( @arrCntKeysTmp ){
        my $pdfHandleTmp = $hrefPDFHandles->{ $k };
        if ( $pdfHandleTmp != undef ){
            $pdfHandleTmp->end();
            delete( $hrefPDFHandles->{ $k } );
        }
    }

    return 0;
}

#-----------------------------------------------------------------------
# Check and create coded PDF output (when document has PDF container applied pages)
#
# Parameter
# 1. Document handle
#
#-----------------------------------------------------------------------
sub _checkAndCreateCodedPDFOutput{
    #---- Fetch parameter
    #
    # 1. Document handle
    #
    my $a2wDocHandlePar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_checkAndCreateCodedPDFOutput()" ); }

    #---- Get content info (reports pages applied with PDF container pages)
    my $hrefDocInfoTmp = _getContentInfo( $a2wDocHandlePar );
    if ( $hrefDocInfoTmp == undef ){
        return (-1, 'Unable to get document content info')
    }

    #if ( $bLog == $TRUE ){
    #    $theLogger->logMessage( "Document Info: " );
    #    $theLogger->logHashMessage( $hrefDocInfoTmp );
    #}

    if ( $hrefDocInfoTmp->{ 'pdfpagecount' } > 0 ){
        #---- Determine split groups based on the PDF container applied pages
        my $hrefSplitInfoTmp = _detectSplitGroups( $hrefDocInfoTmp );
        if ( $hrefSplitInfoTmp == undef ){
            return (-2, 'Unable to split groups info')
        }
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Split Info: " );
            $theLogger->logHashMessage( $hrefSplitInfoTmp );
        }

        #---- Load the split group PDFs
        my ($iRcTmp, $sMsgTmp) = _loadPDFContainers( $hrefSplitInfoTmp );
        if ( $iRcTmp < 0 ){
            return ($iRcTmp, $sMsgTmp);
        }

        #---- Extract groups from core generated PDF and container PDFs
        ($iRcTmp, $sMsgTmp) = _extractGroupFiles( $hrefSplitInfoTmp );
        if ( $iRcTmp < 0 ){
            return ($iRcTmp, $sMsgTmp);
        }

        #---- Get core generated output filename
        my $sCoreFNTmp = $sOutputFilePath . $a2wDocHandlePar->getOutputFilename();

        #---- Merge the extracted group files
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( 'Files to merge:' );
            $theLogger->logHashMessage( { 'mergelist' => $hrefSplitInfoTmp->{ 'mergelist' } } );
        }

        #---- Delete the core generated output
        unlink( $sCoreFNTmp );

        #---- Merge the extracted group files for this document
        ($iRCTmp, $sMsgTmp) = _mergePDFs( $hrefSplitInfoTmp->{ 'mergelist' }, $sCoreFNTmp );
        if ( $iRcTmp < 0 ){
            return ($iRcTmp, $sMsgTmp);
        }

        #---- Cleanup the temp (extracted) files
        my @arrDelListTmp = @{ $hrefSplitInfoTmp->{ 'mergelist' } };
        foreach my $d ( @arrDelListTmp ){
            if ( -e $d ){
                unlink( $d );
            }
        }
    }
    else {
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "No pages applied with PDF container, hence not merging any" ); }
    }

    return (0, 'Success');
}

#-----------------------------------------------------------------------
# Get content info
#
# Parameter
# 1. Document handle
#
# Returns
# Hash reference
# {
#     id:               <document id>
#   , pages: [
#         {
#             id:            <Spool page id>
#           , w:             <page width>
#           , h:             <page height>
#           , r:             <page resolution>
#           , pdfpage:       <Flag indicating page has PDF container alone or not>
#           , file:          <PDF container filename or AFP2web core generated output filename>
#           , pgid:          <Page id as on 'file'>
#           , cn:            <PDF container name>
#           , cx:            <PDF container X position in AFP page>
#           , cy:            <PDF container Y position in AFP page>
#         },
#         ...
#     ]
# }
#-----------------------------------------------------------------------
sub _getContentInfo{
    #---- Fetch parameter
    #
    # 1. Document handle
    #
    my $a2wDocHandlePar = shift;

    my $hrefDocContentTmp = { 'id' => $a2wDocHandlePar->getId() };
    my @arrPagesTmp = ();

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_getContentInfo()" ); }

    #---- Iterate through the pages and collect the info
    my $iPgIdTmp = 0;
    my $a2wPgTmp = $a2wDocHandlePar->getFirstPage();

    my $hrefContInfoTmp = undef;
    my @arrPageInfoTmp = @{ $hrefCurrentDoc->{ 'pagesInfo' } };
    my $hrefDocContInfoTmp = $hrefCurrentDoc->{ 'containerInfo' };
    while ( $a2wPgTmp != undef ){
        $iPgIdTmp++;

        $hrefContInfoTmp = $hrefDocContInfoTmp->{ $iPgIdTmp };
        #if ( $bLog == $TRUE && $hrefContInfoTmp != undef ){ $theLogger->logMessage( "Container info: " . $hrefContInfoTmp ); }

        my $hrefPageTmp = {
            'id'       => $arrPageInfoTmp[ ( $iPgIdTmp - 1 ) ]
          , 'w'        => $a2wPgTmp->getWidth()
          , 'h'        => $a2wPgTmp->getHeight()
          , 'r'        => $a2wPgTmp->getResolution()
          , 'pdfpage'  => ( $hrefContInfoTmp != undef ) ? $TRUE : $FALSE
          , 'file'     => $sOutputFilePath . $a2wDocHandlePar->getOutputFilename()
          , 'name'     => $a2wDocHandlePar->getName()
          , 'pgid'     => 0
        };
        if ( $hrefPageTmp->{ 'pdfpage' } == $TRUE ){
            if ( $bLog == $TRUE ){ $theLogger->logMessage( $hrefPageTmp->{ 'id' } . " page is PDF container page" ); }
            $iPDFPgCountTmp++;
            $hrefPageTmp->{ 'cx' }   = $hrefContInfoTmp->{ 'cx' };
            $hrefPageTmp->{ 'cy' }   = $hrefContInfoTmp->{ 'cy' };
            $hrefPageTmp->{ 'name' } = $hrefContInfoTmp->{ 'name' };
            $hrefPageTmp->{ 'file' } = $hrefContInfoTmp->{ 'file' };
            $hrefPageTmp->{ 'pgid' } = $hrefContInfoTmp->{ 'pgid' };
        }
        else {
            $hrefPageTmp->{ 'pgid' } = $iPgIdTmp;
        }
        push(@arrPagesTmp, $hrefPageTmp);

        $a2wPgTmp = $a2wDocHandlePar->getNextPage();
    }

    #---- Fill in pages info
    $hrefDocContentTmp->{ 'pdfpagecount' } = $hrefCurrentDoc->{ 'pageCount' };
    $hrefDocContentTmp->{ 'pages' } = \@arrPagesTmp;

    return $hrefDocContentTmp;
}

#-----------------------------------------------------------------------
# _isPDFContainerPage
#
# Determines whether page has only the PDF container given content
#
# Rules:
# - Page must have only one object container with PDF
# - PDF content must be applied to the whole page (not partially)
#
# Returns
# true, if page has PDF container
# false, if page has no PDF container or has other additional content
#-----------------------------------------------------------------------
sub _isPDFContainerPage {
    #---- Get Parameter of _isPDFContainerPage
    # Parameters:
    # 1. Page handle
    my $objPagePar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_isPDFContainerPage()" ); }

    my $iPageIdTmp = $objPagePar->getParseId();
    my $sMsgPrefixTmp = "Page " . $iPageIdTmp . " does not have PDF container content alone, It has atleast one ";

    if ( $objPagePar->getFirstOverlay() != undef ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( $sMsgPrefixTmp . "overlay" ); }
        return $FALSE;
    }
    if ( $objPagePar->getFirstPageSegment() != undef ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( $sMsgPrefixTmp . "page segment" ); }
        return $FALSE;
    }

    # AFP page has no lines, still AFP2web seems to create dummy line object
    # to process the positioning info (if given)
    # In given samples, this is not the case, hence check following and consider
    # the page as empty
    # - Line is dummy (whose length and width will be zero)
    # - Page has only one line
    my $objLineTmp = $objPagePar->getFirstLine();
    if ( $objLineTmp != undef ){
        my $iLineCntTmp = 1;
        my $bDummyLineTmp = ( $objLineTmp->getWidth() == 0 && $objLineTmp->getLength() == 0 ) ? $TRUE : $FALSE;

        # Get lines count
        if ( $bDummyLineTmp == $FALSE ){
            $objLineTmp = $objPagePar->getNextLine();
            while ( $objLineTmp != undef ){
                $iLineCntTmp += 1;
                $objLineTmp = $objPagePar->getNextLine();
            }
        }

        if ( $iLineCntTmp > 1 ){
            if ( $bLog == $TRUE ){ $theLogger->logMessage( $sMsgPrefixTmp . "line" ); }
            return $FALSE;
        }
        # else consider no lines occurred in the page
    }

    if ( $objPagePar->getFirstText() != undef ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( $sMsgPrefixTmp . "text" ); }
        #return $FALSE; #!!!TESTING!!!
    }
    if ( $objPagePar->getFirstVector() != undef ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( $sMsgPrefixTmp . "vector" ); }
        #return $FALSE; #!!!TESTING!!!
    }
    if ( $objPagePar->getFirstImage() != undef ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( $sMsgPrefixTmp . "image" ); }
        return $FALSE;
    }

    # NOTE: This check must always be the last one
    my $objContTmp = $objPagePar->getFirstContainer();
    if ( $objContTmp != undef ){
        # If page has more than one container, then simply return false
        if ( $objPagePar->getNextContainer() != undef ){
            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Page " . $iPageIdTmp . " have more than one container" ); }
            return $FALSE;
        }

        #---- Page has only one container ----#
        # Container has PDF only when it's type name is one of the following
        # - "PDF Single Page Object"
        # - "PDF Multi Page Object"
        # - "PDF With Transparency"
		# - "PDF Multi Page With Transparency"
		my $hrefPDFListTmp = {
		    'PDF Single Page Object'           => $TRUE
		  , 'PDF Multi Page Object'            => $TRUE
		  , 'PDF With Transparency'            => $TRUE
		  , 'PDF Multi Page With Transparency' => $TRUE
		};
        my $sTypeNameTmp = $objContTmp->getObjectTypeName();
        #if ( $bLog == $TRUE ){ $theLogger->logMessage( "Type:>" . $sTypeNameTmp . "< Result:>" . $hrefPDFListTmp->{ $sTypeNameTmp } . "<" ); }

        if ( $hrefPDFListTmp->{ $sTypeNameTmp } == $TRUE ){
            if ( $bLog == $TRUE ){
                $theLogger->logMessage( "Page " . $iPageIdTmp . " is pdf container page" );
                $theLogger->logMessage( "Container info: " . $objContTmp->_toString() );
            }

            return $TRUE;
        }
    }

    return $FALSE;
}

#-----------------------------------------------------------------------
# Detect splittable groups
#
# Parameter
# 1. Document content info
#
# Returns
# Hash reference
# {
#     id:                    <document id>
#   , groups: [
#         {
#             id:            <page id>
#           , file:          <PDF container filename or AFP2web core generated output filename>
#           , container:     <Flag saying the file is of container or core generated, TRUE means container pdf>
#           , start:         <Start page id>
#           , end:           <End page id>
#           , pages:         <Reference of the start to end pages>
#         },
#         ...
#     ]
# }
#-----------------------------------------------------------------------
sub _detectSplitGroups{
    #---- Fetch parameter
    #
    # 1. Document content info
    #
    my $hrefDocInfoPar = shift;

    my $hrefSplitInfoTmp = { 'id' => $hrefDocInfoPar->{ 'id' } };
    my @arrGroupsTmp = ();
    my $hrefSplitGroupTmp = undef;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_detectSplitGroups()" ); }

    #---- Iterate through the pages and collect the split info
    my @arrPagesTmp = @{ $hrefDocInfoPar->{ 'pages' } };
    my $sPrevPDFTmp = '';
    my $iSplitPgsIdxTmp = 0;
    my $arefSplitPgsTmp = [];
    foreach my $pg ( @arrPagesTmp ){
        #if ( $bLog == $TRUE ){
        #    $theLogger->logMessage( "Page: " . $pg->{ 'id' } );
        #    $theLogger->logHashMessage( $pg );
        #}

        # Rule: Changing input file marks split
        if ( $pg->{ 'file' } ne $sPrevPDFTmp ){
            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Page: " . ( $pg->{ 'id' } + 1 ) . " marks split due to filename difference" ); }
            #if ( $bLog == $TRUE ){ $theLogger->logHashMessage( $pg ); }
            $sPrevPDFTmp = $pg->{ 'file' };

            if ( $hrefSplitGroupTmp == undef ){
                # Create new split group
                $iSplitPgsIdxTmp = 0;
                $arefSplitPgsTmp = [];
                $hrefSplitGroupTmp = { 'id' => (@arrGroupsTmp + 1) };
                $hrefSplitGroupTmp->{ 'file' } = $pg->{ 'file' };
                $hrefSplitGroupTmp->{ 'name' } = $pg->{ 'name' };
                $hrefSplitGroupTmp->{ 'container' } = $pg->{ 'pdfpage' };
                $hrefSplitGroupTmp->{ 'start' } = $pg->{ 'pgid' };
                $hrefSplitGroupTmp->{ 'end' } = $pg->{ 'pgid' };
            }
            else {
                # Add existing split group to the list
                $hrefSplitGroupTmp->{ 'pages' } = $arefSplitPgsTmp;
                push( @arrGroupsTmp, $hrefSplitGroupTmp );

                # Create new split group
                $iSplitPgsIdxTmp = 0;
                $arefSplitPgsTmp = [];
                $hrefSplitGroupTmp = { 'id' => (@arrGroupsTmp + 1) };
                $hrefSplitGroupTmp->{ 'file' } = $pg->{ 'file' };
                $hrefSplitGroupTmp->{ 'name' } = $pg->{ 'name' };
                $hrefSplitGroupTmp->{ 'container' } = $pg->{ 'pdfpage' };
                $hrefSplitGroupTmp->{ 'start' } = $pg->{ 'pgid' };
                $hrefSplitGroupTmp->{ 'end' } = $pg->{ 'pgid' };
            }
            delete( $pg->{ 'name' }); # remove the name (as it is now added at split group)
            delete( $pg->{ 'file' }); # remove the filename (as it is now added at split group)
            delete( $pg->{ 'pdfpage' }); # remove the pdfpage (as it is now added at split group)
            $arefSplitPgsTmp->[ $iSplitPgsIdxTmp++ ] = $pg;
        }
        else {
            # Rule: Changing page sequence on same pdf marks split
            if (    $pg->{ 'pdfpage' } == $TRUE
                 && ($hrefSplitGroupTmp->{ 'end' } + 1) != $pg->{ 'pgid' }
               ){
                if ( $bLog == $TRUE ){ $theLogger->logMessage( "Page: " . $pg->{ 'id' } . " marks split due to page number difference" ); }
                #if ( $bLog == $TRUE ){ $theLogger->logHashMessage( $pg ); }
                $sPrevPDFTmp = $pg->{ 'file' };
				
                # Add existing split group to the list
                $hrefSplitGroupTmp->{ 'pages' } = $arefSplitPgsTmp;
                push( @arrGroupsTmp, $hrefSplitGroupTmp );

                # Create new split group
                $iSplitPgsIdxTmp = 0;
                $arefSplitPgsTmp = [];
                $hrefSplitGroupTmp = { 'id' => ($#arrGroupsTmp + 1) };
                $hrefSplitGroupTmp->{ 'file' } = $pg->{ 'file' };
                $hrefSplitGroupTmp->{ 'name' } = $pg->{ 'name' };
                $hrefSplitGroupTmp->{ 'container' } = $pg->{ 'pdfpage' };
                $hrefSplitGroupTmp->{ 'start' } = $pg->{ 'pgid' };
                $hrefSplitGroupTmp->{ 'end' } = $pg->{ 'pgid' };
                delete( $pg->{ 'name' }); # remove the name (as it is now added at split group)
                delete( $pg->{ 'file' }); # remove the filename (as it is now added at split group)
                delete( $pg->{ 'pdfpage' }); # remove the filename (as it is now added at split group)
                $arefSplitPgsTmp->[ $iSplitPgsIdxTmp++ ] = $pg;
            }
            else {
                $hrefSplitGroupTmp->{ 'end' } = $pg->{ 'pgid' };
                delete( $pg->{ 'name' }); # remove the name (as it is now added at split group)
                delete( $pg->{ 'file' }); # remove the filename (as it is now added at split group)
                delete( $pg->{ 'pdfpage' }); # remove the pdfpage (as it is now added at split group)
                $arefSplitPgsTmp->[ $iSplitPgsIdxTmp++ ] = $pg;
            }
        }
    }
    # Add last split group to the list
    $hrefSplitGroupTmp->{ 'pages' } = $arefSplitPgsTmp;
    push( @arrGroupsTmp, $hrefSplitGroupTmp );

    $hrefSplitInfoTmp->{ 'groups' } = \@arrGroupsTmp;

    return $hrefSplitInfoTmp;
}

#-----------------------------------------------------------------------
# _fixContainerPDF
#
# - Fixes minor issues (like junk bytes in start) on container PDF using QPDF
#
# Returns
# - 0 in case of success
# - <0 in case of error
#-----------------------------------------------------------------------
sub _fixContainerPDF{
    #---- Get Parameter of _fixContainerPDF
    # Parameters:
    # 1. Container PDF filename
    my $sPDFFilenamePar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_fixContainerPDF()" ); }

    #---- Prepare command to fix the pdf
    my $sFixCmdTmp = ".bin/qpdf.sh";
    if ( $^O eq 'MSWin32' ){
        $sFixCmdTmp = ".\\.bin\\qpdf.exe";
    }
    $sFixCmdTmp .= " $sPDFFilenamePar --replace-input";
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Fix command: " . $sFixCmdTmp ); }

    #----- Execute command
    my $sConsoleTmp = `$sFixCmdTmp`;
    #if ( $bLog == $TRUE ){ $theLogger->logMessage( $sConsoleTmp ); }

    my $iRcTmp = $?;
    if ( $iRcTmp < 0 ){
        my $sErrorMsgTmp = "Unable to split PDF ($sFixCmdTmp): RC=" . $iRcTmp . "\n" . $sConsoleTmp;
        if ( $bLog == $TRUE ){ $theLogger->logMessage( $sErrorMsgTmp ); }
        return -1;
    }

    if ( $bLog == $TRUE ){ $theLogger->logMessage( 'Successfully fixed container PDF: ' . $sPDFFilenamePar ); }

    return 0;
}

#-----------------------------------------------------------------------
# _loadPDFContainers
#
# Remarks
# - Iterates through the split groups
# - For each group, gets the PDF filename and loads it using PDF::API2
# - Loaded PDF handles are filled in global variable $hrefPDFHandles
#-----------------------------------------------------------------------
sub _loadPDFContainers {
    #---- Get Parameter of _extractPDFContainerPage
    # Parameters:
    # 1. Split groups info
    my $hrefSplitGroupsPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_loadPDFContainers()" ); }

    my @arrSGrpsTmp = @{ $hrefSplitGroupsPar->{ 'groups' } };
    if ( @arrSGrpsTmp <= 0 ){
        return ( -1, 'No groups to split' );
    }

    #---- Iterate through the split groups and load the PDF of each group
    my $sErrMsgTmp = '';
    foreach my $sg ( @arrSGrpsTmp ){
        #---- Get the container PDF filename
        my $sCntNameTmp = $sg->{ 'name' };
        my $sCntFileTmp = $sg->{ 'file' };

        # File name is undefined
        if ( !defined( $sCntFileTmp ) || ( $sCntFileTmp eq "" ) ) {
            $sErrMsgTmp = "Split group " . sprintf("%03d", $sg->{ 'id' }) . ": Container PDF filename is undefined";
            if ( $bLog == $TRUE ){ $theLogger->logMessage( $sErrMsgTmp ); }
            last;
        }
        # File not found
        if ( !(-e $sCntFileTmp) ){
            $sErrMsgTmp = "Split group " . sprintf("%03d", $sg->{ 'id' }) . ": Container PDF: " . $sCntFileTmp .  " does not exist";
            if ( $bLog == $TRUE ){ $theLogger->logMessage( $sErrMsgTmp ); }
            last;
        }
        # File size is zero
        if ( -z $sCntFileTmp ){
            $sErrMsgTmp = "Split group " . sprintf("%03d", $sg->{ 'id' }) . ": Container PDF: " . $sCntFileTmp .  " size is zero";
            if ( $bLog == $TRUE ){ $theLogger->logMessage( $sErrMsgTmp ); }
            last;
        }

        #---- Fix the PDF (if it is of container, to avoid loading error due to junk bytes in start of pdf)
        if ( $sg->{ 'container' } == $TRUE ){
            my $iFixRcTmp = _fixContainerPDF( $sCntFileTmp );
            if ( $iFixRcTmp < 0 ){
                $sErrMsgTmp = "Could not fix PDF: " . $sCntFileTmp . " for loading";
                if ( $bLog == $TRUE ){ $theLogger->logMessage( $sErrMsgTmp ); }
                last;
            }
        }

        #---- Open PDF container file (if not opened already)
        my $pdfCntTmp = $hrefPDFHandles->{ $sCntNameTmp };
        if ( $pdfCntTmp == undef ){
            eval { $hrefPDFHandles->{ $sCntNameTmp } = PDF::API2->open( $sCntFileTmp ); };
            if ( $@ ){
                $sErrMsgTmp = "Split group " . sprintf("%03d", $sg->{ 'id' }) . ": Could not load container PDF: " . $sCntFileTmp . ", Reason: " . $@;
                if ( $bLog == $TRUE ){ $theLogger->logMessage( $sErrMsgTmp ); }
                last;
            }
            $pdfCntTmp = $hrefPDFHandles->{ $sCntNameTmp };
        }
        if ( $bLog == $TRUE ){ $theLogger->logMessage( 'Loaded PDF: ' . $sCntFileTmp); }
    }

    if ( $bLog == $TRUE && $sErrMsgTmp eq '' ){ $theLogger->logMessage( 'Successfully loaded the PDFs for all split groups' ); }
    return ( $sErrMsgTmp eq '' ) ? ( 0 ) : ( -1, $sErrMsgTmp );
}

#-----------------------------------------------------------------------
# _extractSinglePDFPage
#
# - Extracts the PDF page
# - Creates a new PDF with blank page (whose size match AFP page size)
# - Places the extracted PDF page on black page
# - Save the newly created PDF
#
# Returns
# - Page PDF filename
#-----------------------------------------------------------------------
sub _extractSinglePDFPage {
    #---- Get Parameter of _extractSinglePDFPage
    # Parameters:
    # 1. Split group
    # 2. AFP page info
    # 3. Container pdf handle
    my $hrefSGroupPar = shift;
    my $hrefAFPPagePar = shift;
    my $cntPDFHandlePar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_extractSinglePDFPage()" ); }

    # Assert parameter
    if ( $cntPDFHandlePar == undef ){
        $sErrMsgTmp = "Missing or invalid parameter: " . $cntPDFHandlePar;
        if ( $bLog == $TRUE ){ $theLogger->logMessage( $sErrMsgTmp ); }
        return '';
    }

    #---- Get container info
    my $sCntNameTmp = $hrefSGroupPar->{ 'name' };

    #---- Create a PDF with empty page
    my $pdfHandleTmp = PDF::API2->new();

    #---- Evaluate and set page size (based on AFP page size)
    my $iAFPRTmp = $hrefAFPPagePar->{ 'r' };
    my $iAFPWTmp = $hrefAFPPagePar->{ 'w' };
    my $iAFPHTmp = $hrefAFPPagePar->{ 'h' };
    my $iPDFWTmp = ($iAFPWTmp / $iAFPRTmp) * 72;
    my $iPDFHTmp = ($iAFPHTmp / $iAFPRTmp) * 72;
    #if ( $bLog == $TRUE ){ $theLogger->logMessage( "Page size: W=" . $iPDFWTmp . ", H=" . $iPDFHTmp ); }
    $pdfHandleTmp->mediabox($iPDFWTmp,$iPDFHTmp);

    #---- Get PDF container position in AFP page
    my $iXPosTmp = ($hrefAFPPagePar->{ 'cx' } / $iAFPRTmp) * 72;
    my $iYPosTmp = (($iAFPHTmp - $hrefAFPPagePar->{ 'cy' }) / $iAFPRTmp) * 72;

    #---- Adjust container page size
    $iYPosTmp -= $hrefAFPPagePar->{ 'ch' };

    #---- Extract the page as resource
    my $pdfPageTmp = undef;
    eval { $pdfPageTmp = $pdfHandleTmp->importPageIntoForm($cntPDFHandlePar, $hrefAFPPagePar->{ 'pgid' }); };
    if ( $@ ){
        my $sErrMsgTmp = 'Failed to import page: ' . $hrefAFPPagePar->{ 'pgid' } . ' as resource from PDF: ' . $sFilenameTmp . ', Reason: ' . $@;
        if ( $bLog == $TRUE ){ $theLogger->logMessage( $sErrMsgTmp ); }
        return '';
    }

    #---- Add extract page resource to output PDF
    my $pgHandleTmp = $pdfHandleTmp->page();
    my $pgGFXTmp = $pgHandleTmp->gfx();
    $pgGFXTmp->formimage($pdfPageTmp, $iXPosTmp, $iYPosTmp);
    #if ( $bLog == $TRUE ){ $theLogger->logMessage( "Rendered container PDF page at X=" . $iXPosTmp . ", Y=" . $iYPosTmp ); }

    #---- Save the output PDF
    my $sFilenameTmp =   $sTempPath . $sCntNameTmp
                       . '_' . sprintf("%03d", $hrefAFPPagePar->{ 'id' })
                       . '_' . sprintf("%03d", $hrefAFPPagePar->{ 'pgid' })
                       . '.pdf';
    eval { $pdfHandleTmp->saveas($sFilenameTmp); };
    if ( $@ ){
        my $sErrMsgTmp = 'Failed to extract page: ' . $hrefAFPPagePar->{ 'pgid' } . ' from PDF: ' . $sFilenameTmp . ', Reason: ' . $@;
        if ( $bLog == $TRUE ){ $theLogger->logMessage( $sErrMsgTmp ); }
        return '';
    }
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "PDF: " . $sFilenameTmp .  " with container: " . $sCntNameTmp . " page: " . $hrefAFPPagePar->{ 'pgid' } . " has been created for AFP page: " . $hrefAFPPagePar->{ 'id' } ); }

    return $sFilenameTmp;
}

#-----------------------------------------------------------------------
# _extractMultiplePDFPages
#
# - Extracts given range of pages from the PDF
#
# Returns
# - Pages pdf filename
#-----------------------------------------------------------------------
sub _extractMultiplePDFPages{
    #---- Get Parameter of _extractMultiplePDFPages
    # Parameters:
    # 1. Split group
    # 2. AFP page info
    # 3. Container pdf handle
    # 4. Starting page
    # 5. Ending page
    my $hrefSGroupPar = shift;
    my $hrefAFPPagePar = shift;
    my $cntPDFHandlePar = shift;
    my $iStartPagePar = shift;
    my $iEndPagePar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_extractMultiplePDFPages()" ); }

    # Assert parameter
    if ( $cntPDFHandlePar == undef ){
        $sErrMsgTmp = "Missing or invalid parameter: " . $cntPDFHandlePar;
        if ( $bLog == $TRUE ){ $theLogger->logMessage( $sErrMsgTmp ); }
        return '';
    }

    #---- Get container info
    my $sCntNameTmp = $hrefSGroupPar->{ 'name' };
    my $sCntFileTmp = $hrefSGroupPar->{ 'file' };

    #---- Prepare command to extract the pages
    my $sFilenameTmp =   $sTempPath . $sCntNameTmp
                       . '_' . sprintf("%03d", $hrefAFPPagePar->{ 'id' })
                       . '_' . sprintf("%03d", $iStartPagePar)
                       . '_' . sprintf("%03d", $iEndPagePar)
                       . '.pdf';

    my $sSplitCmdTmp = ".bin/qpdf.sh";
    if ( $^O eq 'MSWin32' ){
        $sSplitCmdTmp = ".\\.bin\\qpdf.exe";
    }
    $sSplitCmdTmp .= " --warning-exit-0 --no-warn --empty --pages \"$sCntFileTmp\" $iStartPagePar-$iEndPagePar -- \"$sFilenameTmp\"";
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Split command: " . $sSplitCmdTmp ); }

    #----- Execute command
    my $sConsoleTmp = `$sSplitCmdTmp`;
    #if ( $bLog == $TRUE ){ $theLogger->logMessage( $sConsoleTmp ); }

    my $iRcTmp = $?;
    if ( $iRcTmp < 0 ){
        my $sErrorMsgTmp = "Unable to split PDF ($sSplitCmdTmp): RC=" . $iRcTmp . "\n" . $sConsoleTmp;
        if ( $bLog == $TRUE ){ $theLogger->logMessage( $sErrorMsgTmp ); }
        return -1;
    }

    if ( $bLog == $TRUE ){ $theLogger->logMessage( "PDF: " . $sFilenameTmp .  " with container: " . $sCntNameTmp . " pages from " . $iStartPagePar . " to " . $iEndPagePar . " has been created" ); }

    return $sFilenameTmp;
}

#-----------------------------------------------------------------------
# Extract files from all split groups
#
# Parameter
# 1. Split groups info
#
# Returns
# Array reference having list of extracted filenames (in mergable order)
#-----------------------------------------------------------------------
sub _extractGroupFiles{
    #---- Fetch parameter
    #
    # 1. Split groups info
    #
    my $hrefSplitGroupsPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_extractGroupFiles()" ); }

    my @arrSGrpsTmp = @{ $hrefSplitGroupsPar->{ 'groups' } };
    if ( @arrSGrpsTmp <= 0 ){ return undef; }

    my @arrSplitFilesTmp = ();

    #---- Iterate through the split groups and split them as needed
    my $sErrMsgTmp = '';
    foreach my $sg ( @arrSGrpsTmp ){
        #---- Check whether container or core generated PDF
        if ( $sg->{ 'container' } == $TRUE ){
            #---- Get the container PDF filename
            my $sCntNameTmp = $sg->{ 'name' };
            my $sCntFileTmp = $sg->{ 'file' };

            #---- Get PDF container file
            my $pdfCntTmp = $hrefPDFHandles->{ $sCntNameTmp };
            if ( $pdfCntTmp == undef ){
                $sErrMsgTmp = "Split group " . sprintf("%03d", $sg->{ 'id' }) . ": Container PDF: " . $sCntFileTmp . " is not loaded";
                if ( $bLog == $TRUE ){ $theLogger->logMessage( $sErrMsgTmp ); }
                last;
            }

            #---- Iterate through the group pages
            my $bErrorTmp = $FALSE;
            my $hrefSplitTmp = undef;
            my @arrGPagesTmp = @{ $sg->{ 'pages' } };
            foreach my $gp ( @arrGPagesTmp ){
                #---- Get pdf page dimension
                my $pgCntHandleTmp = $pdfCntTmp->openpage( $gp->{ 'pgid' } );
                my ($l, $t, $iCntWTmp, $iCntHTmp) = $pgCntHandleTmp->get_mediabox();
                #if ( $bLog == $TRUE ){ $theLogger->logMessage( $sCntNameTmp . ': Page ' . sprintf("%03d", $gp->{ 'pgid' }) . ': Size: W=' . $iCntWTmp . ' H=' . $iCntHTmp ); }
                $gp->{ 'cw' } = $iCntWTmp;
                $gp->{ 'ch' } = $iCntHTmp;

                my $iWToleranceTmp = $iCntWTmp * 0.03;
                my $iHToleranceTmp = $iCntHTmp * 0.03;

                my $iAFPRTmp = $gp->{ 'r' };
                my $iAFPWTmp = $gp->{ 'w' };
                my $iAFPHTmp = $gp->{ 'h' };
                my $iPDFWTmp = ($iAFPWTmp / $iAFPRTmp) * 72;
                my $iPDFHTmp = ($iAFPHTmp / $iAFPRTmp) * 72;

                #---- Use container page as is (if the container page size is equal to AFP page size)
                if (    $iPDFWTmp >= ($iCntWTmp - $iWToleranceTmp)
                     && $iPDFWTmp <= ($iCntWTmp + $iWToleranceTmp)
                     && $iPDFHTmp >= ($iCntHTmp - $iHToleranceTmp)
                     && $iPDFHTmp <= ($iCntHTmp + $iHToleranceTmp)
                   ){
                    if ( $hrefSplitTmp == undef ){
                        $hrefSplitTmp->{ 'start' } = $gp->{ 'pgid' };
                        $hrefSplitTmp->{ 'end' } = $gp->{ 'pgid' };
                    }
                    else {
                        $hrefSplitTmp->{ 'end' } = $gp->{ 'pgid' };
                    }
                }
                #---- Extract the container page and render it on AFP page
                else {
                    #---- Split the previously collected AFP sized pages
                    if ( $hrefSplitTmp != undef ){
                        if ( $bLog == $TRUE ){ $theLogger->logMessage( 'Extracting pages from ' . $hrefSplitTmp->{ 'start' } . ' to ' . $hrefSplitTmp->{ 'end' } . ' of ' . $sCntFileTmp ); }
                        my $sFNTmp = _extractMultiplePDFPages( $sg, $gp, $pdfCntTmp, $hrefSplitTmp->{ 'start' }, $hrefSplitTmp->{ 'end' } );
                        if ( $sFNTmp eq '' ){
                            if ( $bLog == $TRUE ){ $theLogger->logMessage( $sCntNameTmp . ': Pages ' . sprintf("%d-%d", $hrefSplitTmp->{ 'start' }, $hrefSplitTmp->{ 'end' }) . ' could not be extracted' ); }
                            $bErrorTmp = $TRUE;
                            last; # inner loop: foreach my $gp ( @arrGPagesTmp )
                        }
                        push( @arrSplitFilesTmp, $sFNTmp );
                        $hrefSplitTmp = undef;
                    }

                    my $sFNTmp = _extractSinglePDFPage( $sg, $gp, $pdfCntTmp );
                    if ( $sFNTmp eq '' ){
                        if ( $bLog == $TRUE ){ $theLogger->logMessage( $sCntNameTmp . ': Page ' . sprintf("%03d", $gp->{ 'pgid' }) . ' could not be extracted' ); }
                        $bErrorTmp = $TRUE;
                        last; # inner loop: foreach my $gp ( @arrGPagesTmp )
                    }
                    push( @arrSplitFilesTmp, $sFNTmp );
                }
            } # foreach my $gp ( @arrGPagesTmp )

            #---- Split the last collected AFP sized pages
            if ( $hrefSplitTmp != undef ){
                if ( $bLog == $TRUE ){ $theLogger->logMessage( 'Extracting pages from ' . $hrefSplitTmp->{ 'start' } . ' to ' . $hrefSplitTmp->{ 'end' } . ' of ' . $sCntFileTmp ); }
                my $sFNTmp = _extractMultiplePDFPages( $sg, $gp, $pdfCntTmp, $hrefSplitTmp->{ 'start' }, $hrefSplitTmp->{ 'end' } );
                if ( $sFNTmp eq '' ){
                    if ( $bLog == $TRUE ){ $theLogger->logMessage( $sCntNameTmp . ': Pages ' . sprintf("%d-%d", $hrefSplitTmp->{ 'start' }, $hrefSplitTmp->{ 'end' }) . ' could not be extracted' ); }
                    $bErrorTmp = $TRUE;
                    last;
                }
                push( @arrSplitFilesTmp, $sFNTmp );
                $hrefSplitTmp = undef;
            }

            if ( $bErrorTmp == $TRUE ){ last; }
        }
        # Core generated PDF
        else {
            #---- Get the document PDF filename
            my $sDocNameTmp = $sg->{ 'name' };
            my $sDocFileTmp = $sg->{ 'file' };

            #---- Get PDF file
            my $pdfDocTmp = $hrefPDFHandles->{ $sDocNameTmp };
            if ( $pdfDocTmp == undef ){
                $sErrMsgTmp = "Split group " . sprintf("%03d", $sg->{ 'id' }) . ": Core PDF: " . $sDocNameTmp . " is not loaded";
                if ( $bLog == $TRUE ){ $theLogger->logMessage( $sErrMsgTmp ); }
                last;
            }

            if ( $bLog == $TRUE ){ $theLogger->logMessage( 'Extracting pages from ' . $sg->{ 'start' } . ' to ' . $sg->{ 'end' } . ' of ' . $sDocFileTmp ); }
            my $sFNTmp = _extractMultiplePDFPages( $sg, $gp, $pdfDocTmp, $sg->{ 'start' }, $sg->{ 'end' } );
            if ( $sFNTmp eq '' ){
                if ( $bLog == $TRUE ){ $theLogger->logMessage( $sDocNameTmp . ': Pages ' . sprintf("%d-%d", $sg->{ 'start' }, $sg->{ 'end' }) . ' could not be extracted' ); }
                last;
            }
            push( @arrSplitFilesTmp, $sFNTmp );
            $hrefSplitTmp = undef;
        }
    } # foreach my $sg ( @arrSGrpsTmp )

    #---- Store the list of files in split groups info
    $hrefSplitGroupsPar->{ 'mergelist' } = \@arrSplitFilesTmp;

    if ( $bLog == $TRUE && $sErrMsgTmp eq '' ){ $theLogger->logMessage( 'Successfully loaded the PDFs for all split groups' ); }
    return ( $sErrMsgTmp eq '' ) ? ( 0 ) : ( -1, $sErrMsgTmp );
}

#-----------------------------------------------------------------------
# _mergePDFs
#
# - Merge given list of PDF files
#
# Returns
# - Return code, 0 means success and all others mean error
# - Message, error message (in case of error)
#-----------------------------------------------------------------------
sub _mergePDFs{
    #---- Get Parameter of _mergePDFs
    # Parameters:
    # 1. List of PDF files
    my $arefFilesPar = shift;
    my $sOutFilePar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_mergePDFs()" ); }

    my @arrFilesTmp = @{$arefFilesPar};

    #---- Assert parameters
    # $#arrFilesTmp returns last index, hence to check empty, assert < 0
    if ( $#arrFilesTmp < 0 ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Empty or invalid files list to merge" ); }
        return (-1, "Empty or invalid files list to merge");
    }
    if ( !defined($sOutFilePar) || $sOutFilePar eq "" ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Ouptut filename is undefined" ); }
        return (-2, "Ouptut filename is undefined");
    }

    # When list has only one PDF, then return it as output PDF
    if ( $#arrFilesTmp == 0 ){
        if(!rename($arrFilesTmp[0], $sOutFilePar)){
            my $sErrMsgTmp = "Renaming " . $arrFilesTmp[0] . " as " . $sOutFilePar . " failed, Reason: " . $!;
            if ( $bLog == $TRUE ){ $theLogger->logMessage( $sErrMsgTmp ); }
            return (-3, $sErrMsgTmp);
        }

        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Renamed " . $arrFilesTmp[0] . " as " . $sOutFilePar ); }
        return (0, "Success");
    }

    #---- Build merge command
    # Syntax: qpdf --empty --pages *.pdf -- out.pdf
    my $sMergeCmdTmp = '.bin/qpdf.sh';
    if ( $^O eq 'MSWin32' ){
        $sMergeCmdTmp = '.\\.bin\\qpdf.exe'
    }
    $sMergeCmdTmp .= ' --empty --pages';
    foreach my $f ( @arrFilesTmp ){
        $sMergeCmdTmp .= ' "' . $f . '"';
    }
    $sMergeCmdTmp .= ' -- "' . $sOutFilePar . '"';

    my $sConsoleTmp = `$sMergeCmdTmp`;
    #if ( $bLog == $TRUE ){ $theLogger->logMessage( $sConsoleTmp ); }

    my $iRcTmp = $?;
    if ( $iRcTmp < 0 ){
        my $sErrorMsgTmp = "Unable to merge PDFs ($sMergeCmdTmp): RC=" . $iRcTmp . "\n" . $sConsoleTmp;
        if ( $bLog == $TRUE ){ $theLogger->logMessage( $sErrorMsgTmp ); }
        return (-4, $sErrorMsgTmp);
    }

    return (0, "Success");
}

__END__
