/**-----------------------------------------------------------------------------
 * oxsnps.js: 	OXSNPS Plugin to manage AFP2web conversion requests
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2014-2016 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V1.0.6
 *
 * History
 *  V100   25.08.2014  Initial release
 *  V101   15.10.2014  Extended to support HTTP JSON requests
 *  V102   03.11.2014  Extended to pass PDF owner pwd and user pwd given in request to A2W server
 *  V103   07.11.2014  Extended to build ScenarioConf if not set in 'res' and to pass request specific A2W props to A2W server
 *  V104   20.01.2015  Extended JSON interface to use Base64 Encoding for binary data in JSON requests/responses
 *  V105   13.03.2015  Added _logServiceRespTime()
 *  V106   26.05.2016  Extended 'syncTransform' and 'asyncTransform' HTTP APIs to handle both buffer and file list inputs
 *  V2.0.21 30.09.2022 OTS-3348: Fill A2WS 'Host' Request parameter
 *----------------------------------------------------------------------------*/
'use strict';

/**
 * Variable declaration
 * @private
 */
var packageJson 	= require(__dirname + '/package.json')
  , PLUGIN_NAME 	= packageJson.name
  , PLUGIN_VERSION  = packageJson.version // Get Server Version from package.json
  , fs 				= require('fs')
  , path 			= require('path')
  , async 			= require('async')
  , log4js 			= require('log4js')
  
  , npsServer		= require('oxsnps-core/server')
  , npsConf 		= npsServer.npsConf
  , npsConfDir 		= npsServer.npsConfDir					// holds the configuration Dir of the OTS Server 
  , npsTempDir		= npsServer.npsTempDir

  , pluginConf		= require(__dirname + '/conf/' + PLUGIN_NAME + '.js')  

  , a2wServices		= require('./services/a2wsServices.js')
  , a2ws 			= undefined										// holds an instance of AFP2web Server
  , scenarios 		= require(npsConfDir + '/OTSScenarioList.js')	//!!!
  , confs 			= require(npsConfDir + '/OTSScenarioConf.js')	//!!! holds the Transformation configurations
  , a2wsAny 		= undefined										// holds the A2WS Any Types

  , dateFormat 		= require('oxsnps-core/helpers/date_format')
  , utils 			= require('oxsnps-core/helpers/utils')
  , expressAppUtils = require('oxsnps-core/helpers/expressAppUtils')
  , FILE_SEP 		= require('path').sep
  , jsonResWriter   = require('oxsnps-core/helpers/jsonResWriter')
  ;

// Set plugin name, version and longDesc in pluginConf
pluginConf.module 	= PLUGIN_NAME; 
pluginConf.version 	= PLUGIN_VERSION; 
pluginConf.longDesc = packageJson.description;
pluginConf.log 		= pluginConf.log || {};

// Export the Plugin's Configuration Express App
exports.pluginConf = pluginConf;

// Get the Logger
var logger = log4js.getLogger(PLUGIN_NAME);
logger.setLevel(pluginConf.log.level || 'INFO'); // set log level

/**************** PUBLIC INTERFACE ***************/
/**
 * initialize: 			Initialize Plugin
 * @param  {function} 	callback(err, routes)
 */
exports.initialize = function(callback){

	var self = this;
	
	logger.debug('--->initializing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION );

	if(self._initialized) return callback();

	// httptimeout in milliseconds. 
	// Default is 5 minutes, just as apache timeout default value
	pluginConf.httptimeout = pluginConf.httptimeout || 5*60*1000;
	//pluginConf.httptimeout = 20;

	// Initialize AFP2web Server's Services.
	a2wServices.setLogLevel(pluginConf.log.level);
	a2wServices.initialize(pluginConf);

	// Initialize a2wsAny
	if(scenarios && scenarios.anytypes)
		a2wsAny	= scenarios.anytypes.a2ws || undefined	// holds the A2WS Any Types
	
	// Map the HTTP routes of the plugin
	expressAppUtils.enableRoutes(self, pluginConf.routes, function(err){
		if(err)	return callback(err);
		self._initialized = true;
		logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' initialized');
		callback(null, pluginConf.routes);
	});
}

/**
 * Activate Plugin. This function is called from pluginManager once all the enabled plugins have been initialize
 * @param  {function} 	callback  callback(err)
 */
exports.postInitialize = function(callback){
	callback();
}

/**
 * finalize: 			Finalize Plugin
 * @param  {function} 	callback(err) 
 */
exports.finalize = function(callback){

	var self = this;

	logger.debug('--->finalizing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION );

	if(!self._initialized) return callback();

	// Unmap the HTTP routes of the plugin
	expressAppUtils.disableRoutes(pluginConf.routes, function(err){
		if(err)	return callback(err);
		self._initialized = false;
		logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' finalized');
		callback();
	});
}

/**
 * main: 				Return the Plugin Readme.md file
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.main = function(req, res){
	try{

		var readmemd 	= fs.readFileSync(__dirname + '/README.md')
	      , version 	= '<h1>' +  PLUGIN_NAME + ' v' + PLUGIN_VERSION + '</h1>'
	      , marked 		= require('marked')
	      , html 		= ''
	      ;
		
		// Synchronous highlighting with highlight.js
		marked.setOptions({
		  highlight: function(code){
		  	return require('highlight').Highlight;
		    
		  }
		});

		html = marked(readmemd.toString());
		res.end('<html>' + version + html + '</html>');
	}
	catch(err){
		res.status(404).end(''+err);
	}
}

/**
 * getVersion: 			Get Version of Plugin
 * @param  {function} 	callback(err,version) 
 */
exports.getVersion = function(callback){
	return callback(null, PLUGIN_VERSION);
}

/**
 * isAlive: Check whether the A2W Server is still alive
 * @param  {function} 	callback(err,version) 
 */
exports.isAlive = function(callback){
	
	_getVersion(function(err, result){	// returns A2WS Version
    	if (err){
			//!!!logger.error('--->isAlive, Error when calling AFP2web Server Version, Reason: ' + err.message);
			return callback(null, PLUGIN_VERSION + '. AFP2web Server is NOT running, Reason: '  + err.message);
		}
		logger.debug('--->isAlive: over, Result: ' + result);
		return callback(null, PLUGIN_VERSION + '. AFP2web Server is running:\n' + result);
	});
}

/**
 * A2W Server ping API
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.ping = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);
	logger.info('--->ping: Req. id=' + res._Context.reqId);

	exports.pingService(res, function(err, result){
    	if (err){
			logger.error('--->ping, AFP2web Server is NOT running, Reason: ' + err.message);
			res.status(404).end('--->ping, AFP2web Server is NOT running, Reason: ' + err.message);
		   	_logServiceRespTime(res, 'ping');
			return;
		}
		res.end(result);
		logger.info('--->ping: Req. id=' + res._Context.reqId + ', over. Result=' + JSON.stringify(result));
	   	_logServiceRespTime(res, 'ping');
	});
}

/**
 * AFP2web Server ping service
 * @param  {function} 	callback(err,result) 
 */
exports.pingService = function(res, callback){

	// Add Context to the response instance
	res._Context = res._Context || {};

	// Add the Current Date to the Context
	res._Context.date = res._Context.date || new Date();

	// Add a Request Id to the Context
	res._Context.reqId = res._Context.reqId || utils.buildReqId(res._Context.date);

	logger.debug('--->pingService: Req. id=' + res._Context.reqId);
	_getVersion(callback);
}

/**
 * version: 			Process a HTTP Version Request
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.version = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	logger.info('--->version: Req. id=' + res._Context.reqId);
	_getVersion(function(err, resp){
		if(err){
			logger.error('--->version: Req. id=' + res._Context.reqId + ', Error! Reason: ' + err.message);
			return res.status(404).end('--->version: Req. id=' + res._Context.reqId + ', Error! Reason: ' + err.message);
		}
		logger.info('--->version: Req. id=' + res._Context.reqId + ', over');
		return res.end(resp);
	});
}

/**
 * Process a HTTP Configuration Request 
 * 	Either as HTTP GET request:
 * 	 	GET http://localhost:1029/services/:plugin/conf 							--> To retrieve the configuration
 * 	 	GET http://localhost:1029/services/:plugin/conf?save=true&conf="{...}" 	--> To pass and save the configuration
 * 	or as HTTP POST request:
 * 	 	POST http://localhost:1029/services/:plugin/conf 						--> To pass and save the configuration
 * 	 	req.body={
 * 	 		"conf": {...}
 * 	 	}
 * 	
 * @param {Request} 	req Request Object
 *                      req.query:{ // for HTTP GET request:
 *                      	"save":"false",
 *                      	"conf":"{IMPORTANT: COMPLETE JSON Conf. as String}"
 *                      }
 *                      req.body:{ // for HTTP POST request:
 *                      	"conf":{IMPORTANT: COMPLETE JSON Conf. as JSON Object}
 *                      }
 * @param {Response}    res Response Object
 */
exports.conf = function(req, res){

	var reqType 	= 'getconf'
	  , jsonConf 	= undefined
	  ;

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);
	logger.info('--->conf: Req. id=' + res._Context.reqId);
	
	if(req.method.toUpperCase() === 'POST'){
		logger.info('--->conf: body=' + JSON.stringify(req.body));
		if(!req.body || !req.body.conf) {
			logger.info('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body));
			return res.end('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body));
		}
		reqType = 'saveconf'; 		// save request
		jsonConf = req.body.conf; 	// set the json conf to be passed to saveConfService
	}
	else if(req.method.toUpperCase() === 'GET'){
		logger.info('--->conf: query=' + JSON.stringify(req.query));
		if(req.query && req.query.save && req.query.save.toLowerCase() === 'true' && req.query.conf)
		{
			reqType = 'saveconf';
			jsonConf = utils.parseJSON(unescape(req.query.conf)); // remove escaped chars and set the json conf to be passed to saveConfService
		} 
	}

	switch(reqType){
		case 'saveconf':
		    exports.saveConfService(jsonConf, function(err){
		    	logger.info('--->conf: Req. id=' + res._Context.reqId + ', over');
				res.json(err ? {'status':'error', 'message': err.message} : pluginConf);
		    }); 
			break;
		default:
		case 'getconf':
			logger.info('--->conf: Req. id=' + res._Context.reqId + ', over');
			res.json(pluginConf);
			break;
	}
}

/**
 * Store  the Plugin conf to its configuration file (i.e. .../:plugin>/conf/:plugin.js)
 * @param  {JSON}   	conf     Plugin Configuration
 * @param  {Function} 	callback
 */
exports.saveConfService = function(jsonConf, callback){

	var filename 	= undefined
	  , data 		= undefined
	  ;
	
	// PluginManager calls saveConfService without passing the conf
	if(typeof jsonConf === 'function'){
		callback = jsonConf;
		jsonConf = pluginConf;
	}

	// Assert jsonConf
	if(!jsonConf || jsonConf === null) return callback(new Error('--->saveConfService: empty Configuration'));
	logger.debug('--->saveConfService: conf:' + JSON.stringify(jsonConf));

	// Backup the current plugin conf.in the server's backup dir and store the new one in the plugin's /conf dir
	async.series([
		// Task 1: Backup the current plugin conf.
		function(nextTask){
			filename = npsConf.backupDir + FILE_SEP + dateFormat.asString('yyyy-MM-dd_hh-mm-ss', new Date()) + '_configurationOf_' + PLUGIN_NAME + '.js'; // build the Backup filename
			logger.debug('--->saveConfService: Backuping plugin configuration to ' + filename);
			data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t'); // build the plugin conf content

			// Write data to the Plugin configuration file
			fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
				if(err) logger.error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message); // log error & continue
				nextTask();
			});
		},
		// Task 2: Save the new plugin conf.
		function(nextTask){
			pluginConf = jsonConf; // we should have here a proper conf, so overwrite the current pluginConf file
			filename = path.resolve(__dirname + '/conf/' + PLUGIN_NAME + '.js'); // build the plugin configuration filename
			logger.debug('--->saveConfService: Writing plugin configuration to ' + filename);
			data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t'); // build the plugin conf content

			// Write data to Plugin configuration file
			fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
				if(err) nextTask(new Error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message));
				else nextTask();
			}); 
		}
	],
	function(err){
		callback(err);
	});
}

exports.syncTransformService = function(res, callback){
	return _syncTransform(res, callback);
}

exports.asyncTransformService = function(res, callback){
	return _asyncTransform(res, callback);
} 

exports.syncStatusService = function(res, callback){
	return _syncStatus(res, callback);
} 

/**
 * syncTransform: 		HTTP SYNC Transformation Request
 * @param  {req} 		req
 * @param  {res} 		res
 */
exports.syncTransform = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the Transformation Type (sync|async)
	res._Context.transType = 'sync';
	
	logger.info('--->syncTransform: Req. id=' + res._Context.reqId);

	var data = '';

	// V101 Begin
	res._Context.req = req.body;

	// V106 Begin
	// Handle both buffer and file list inputs
	if (res._Context.req.input !== undefined){
		if (res._Context.req.input.buffer !== undefined){
			// Input is buffer
			res._Context.inputDoc = new Buffer(res._Context.req.input.buffer, 'base64');
		}
		else if (res._Context.req.input.document !== undefined){
			// Input is file list
			res._Context.inputDoc = res._Context.req.input.document;
		}
	}
	// V106 End

	// Process Request
	_syncTransform(res, function(err, result){
		if(err){
			logger.error('--->syncTransform: Req. id=' + res._Context.reqId + ', ' + err.message);						
			// Send status code as is from A2WS
			jsonResWriter.sendJSONError(res, err.httpStatusCode /* Bad Req */,  null, err.message);
		   	_logServiceRespTime(res, 'syncTransform');
		}
		else{
			if(typeof res._Context.inputDoc == 'string'){
				jsonResWriter.sendJSONResponse(res, 200, JSON.stringify(result), result.Status, result.Status);				
			}
			else{
				jsonResWriter.sendJSONResponse(res, 200, new Buffer(result, 'binary').toString('base64'), 'Done');
			}
			logger.info('--->syncTransform: Req. id=' + res._Context.reqId + ', Response sent back, over');			
		   	_logServiceRespTime(res, 'syncTransform');
		}
	});
	// V101 End
} 

/**
 * HTTP ASYNC Transformation Request
 */
exports.asyncTransform = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the Transformation Type (sync|async)
	res._Context.transType = 'async';

	logger.info('--->asyncTransform: Req. id=' + res._Context.reqId);

	var data = '';

	// V101 Begin  
	res._Context.req = req.body;
	if(res._Context.req.input !== undefined
	   && res._Context.req.input.document !== undefined
	  ){
		// V106 Begin
		//res._Context.inputDoc = new Buffer(res._Context.req.input.document, 'binary');
		res._Context.inputDoc = res._Context.req.input.document;
		// V106 End 
	}

	_asyncTransform(res, function(err, result){
		if(err){
			logger.error('--->syncTransform: Req. id=' + res._Context.reqId + ', ' + err.message);						
			// Send status code as is from A2WS
			jsonResWriter.sendJSONError(res, err.httpStatusCode, res._Context.jobId, err.message);
		   	_logServiceRespTime(res, 'asyncTransform');
		}
		else{
			jsonResWriter.sendJSONResponse(res, 200, res._Context.jobId, result.Status);
			logger.info('--->asyncTransform: Req. id=' + res._Context.reqId + ', Response=' + JSON.stringify(result) +  ', Response sent back, over');
		   	_logServiceRespTime(res, 'asyncTransform');
		}
	});
	// V101 End
} 

/**
 * HTTP Status Request
 */
exports.getStatus = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the Transformation Type (sync|async|status)
	res._Context.transType = 'status';

	logger.info('--->getStatus: Req. id=' + res._Context.reqId);

	var data = '';

	// V101 Begin  
	res._Context.req = req.body;
	if(res._Context.req.input !== undefined
	   && res._Context.req.input.jobId !== undefined
	  ){
		res._Context.jobId = res._Context.req.input.jobId;
	}

	_syncStatus(res, function(err, result){
		if(err){
			logger.error('--->getStatus: Req. id=' + res._Context.reqId + ', ' + err.message);						
			// Send status code as is from A2WS
			jsonResWriter.sendJSONError(res, err.httpStatusCode, res._Context.jobId, err.message);
		   	_logServiceRespTime(res, 'getStatus');
		}
		else{
			jsonResWriter.sendJSONResponse(res, 200, res._Context.jobId, result.Status, result.Status);
			logger.info('--->getStatus: Req. id=' + res._Context.reqId + ', Response=' + JSON.stringify(result) + ' Response sent back, over');			
		   	_logServiceRespTime(res, 'getStatus');
		}
	});
	// V101 End
}

/************ OPTIONAL PUBLIC FUNCTIONS ***********/
/**
 * Set Log Level
 * @param {String} logLevel Log Level. values:DEBUG|INFO|WARN|ERROR
 * 
 */
exports.setLogLevel = function(logLevel){
	pluginConf.log.level = logLevel || 'INFO';
	logger.setLevel(pluginConf.log.level);
	if(a2wServices) a2wServices.setLogLevel(pluginConf.log.level);
}

/**
 * Get Log Level
 * @return {String} Log Level. values:DEBUG|INFO|WARN|ERROR
 */
exports.getLogLevel = function(){
	return pluginConf.log.level;
}

/**
 * enableRoute: 		Enable the given route
 * @param  {request} 	req
 *                      req.body = {
 *                      	"module":"sample", 
 *                      	"route": { 
 *                      		"path": "/path/to/the/service",
 *                      		"method": 	"get|post",
 *                      		"service": 	"version"
 *                      	}
 *                      }
 * @param  {function} 	callback(err)
 */
exports.enableRoute = function(req, callback){

	var self = this;

	logger.debug('--->enableRoute');

	// Enable the route if route has been passed and route has been added previously
	if(req.body.route !== undefined && 
	   req.body.route.path !== undefined &&
	   req.body.route.method !== undefined &&
	   req.body.route.service !== undefined){
		expressAppUtils.enableRoute(self, req.body.route, function(err){
			if(err){
				return callback(err);
			}
			callback();
		});
	}
	else{
		// Display a Warning
		logger.warn('--->enableRoute: Wrong parameters! Not able to enable the route, Parms: ' + JSON.stringify(req.body));
		callback();
	}
} 

/**
 * disableRoute: 		Disable the given route
 * @param  {request} 	req
 * 		            	req.body = {
 *                      	"module":"sample", 
 *                      	"route": { 
 *                      		"path": "/path/to/the/service"
 * 		            	  	}
 * 		                }
 * @param  {function} 	callback(err)
 */
exports.disableRoute = function(req, callback){

	logger.debug('--->disableRoute');

	// Disable the route
	if(req.body.route !== undefined && req.body.route.path !== undefined){
		expressAppUtils.disableRoute(req.body.route, callback);
	}
	else{
		// Display a Warning
		logger.warn('--->disableRoute: Wrong parameters! Not able to disable the route, Parms: ' + JSON.stringify(req.body));
		callback();
	}
} 

/**************** PRIVATE FUNCTIONS ***************/
// Process Version request
function _getVersion(callback){

	// Set A2WS Props
	var a2wsProps = {};

	// Add mandatory fields as AFP2web Server Props
	a2wsProps.transform_version	= 'on';
	a2wsProps.server_log_level = pluginConf.a2wslogging || pluginConf.log.level || 'INFO'; // set logging level	

	// Set Response Format to html
	a2wsProps.responseformat = 'html';

    // Call AFP2web Server's getVersion service
    a2wServices.getVersion(a2wsProps, callback);
}

/**
 * _syncTransform 		Process a Sync. Transformation request
 * @param  {Response} 	res
 *         				res._Context = {
 *         					inputDoc: ""comma separated list of files, or Buffer",
 *         					req: {
 *         						'input':  {'type': 'afp'},
 *         						'output': {'type': 'pdf'}
 *         					}
 *         					a2wsProps: {
 *         						'Any a2w prop':  	...
 *         					}
 *         				}
 * @param  {Function} 	callback(err, result)
 */
function _syncTransform(res, callback){
	if(res._Context.inputDoc !== undefined
	   && (typeof res._Context.inputDoc == 'string')
	   ){
		return _syncTransformFile(res, callback);
	}
	else{
 		return _syncTransformBuffer(res, callback);
	}
}

/**
 * _syncTransformFile 	Process a Sync. File Transformation request
 * @param  {Response} 	res
 * @param  {Function} 	callback(err, result)
 */
function _syncTransformFile(res, callback){
	
	// Assert Input Buffer	
	if(res._Context.inputDoc === undefined || res._Context.inputDoc.length <= 0){
		err = new Error( 'Invalid Request. Reason: input.url is missing !\n' + 
					  'Request:\n' + JSON.stringify(req));

		//!!!logger.error('--->_syncTransformFile: Req.Id:' + res._Context.reqId + ', Error:' + err.message);
		err.httpStatusCode = 400; /* Bad Request */
		return callback(err);
	}

	// Build Scenario Conf 
	if(res._Context.scenarioConf === undefined){
		var err = _buildScenarioConf(res);
		if(err !== null){
			//!!!logger.error('--->_syncTransformFile: Req.Id:' + res._Context.reqId + ', Error:' + err.message);
			err.httpStatusCode = 400; /* Bad Request */
			return callback(err);
		}
	}

	// Set A2WS Props
	var a2wsProps = utils.clone(res._Context.scenarioConf.props); //!!! Using a = b does NOT copy !!!
	
	if(res._Context.a2wsProps !== undefined){
		// concatenate request specific AFP2web props with scenario a2wsProps
		a2wsProps = utils.mergeJSON(a2wsProps, res._Context.a2wsProps);
	}
	
	// Set PDF Security Options
	var pdfSecOptions = _getPDFSecurityOptions(res);
	if(pdfSecOptions !== null){
		a2wsProps.PDFSecurity = pdfSecOptions;
	}
		
	a2wsProps.InputURLs = res._Context.inputDoc;
	
	// Add mandatory fields as AFP2web Server Props
	a2wsProps.OutputMode	= 'file';
	a2wsProps.server_log_level = pluginConf.a2wslogging || pluginConf.log.level || 'INFO'; // set logging level
	
	// Call AFP2web Server's syncTransformServerFile service
	a2wServices.syncTransformServerFile(a2wsProps, function(err, result){ 
		if(err) return callback(err); 
	/*??? Does not work anymore
		// Store Data in debug mode ONLY
		if(logger.isDebugEnabled()){
			var outputFilename = dateFormat.asString('yyyy-MM-dd_hh-mm-ss', new Date()) + '_' + req.input.type + '.' + req.output.type + '.' + a2wsProps.ResponseFormat;
			fs.writeFileSync(npsTempDir + '/' + outputFilename, result);
		}
	*/
		callback(null, result);
	});	
}

// Process a Sync. Buffer Transformation request
function _syncTransformBuffer(res, callback){

	// V101 Begin
	var req = res._Context.req;
	
	// Assert Input Buffer	
	if(res._Context.inputDoc === undefined || res._Context.inputDoc.length <= 0){
		err = new Error( 'Invalid Request. Reason: input buffer is missing !\n' + 
					  'Request:\n' + JSON.stringify(req));

		//!!!logger.error('--->_syncTransformBuffer: Req.Id:' + res._Context.reqId + ', Error:' + err.message);
		err.httpStatusCode = 400; /* Bad Request */
		return callback(err);
	}
	// V101 End	

	// V103 Begin
	// Build Scenario Conf 
	if(res._Context.scenarioConf === undefined){
		var err = _buildScenarioConf(res);
		if(err !== null){
			//!!!logger.error('--->_syncTransformBuffer: Req.Id:' + res._Context.reqId + ', Error:' + err.message);
			err.httpStatusCode = 400; /* Bad Request */
			return callback(err);
		}
	}
	// V103 End

	// Set A2WS Props
	var a2wsProps = utils.clone(res._Context.scenarioConf.props); //!!! Using a = b does NOT copy !!!
	
	// V103 Begin
	if(res._Context.a2wsProps !== undefined){
		// concatenate request specific AFP2web props with scenario a2wsProps
		a2wsProps = utils.mergeJSON(a2wsProps, res._Context.a2wsProps);
	}
	// V103 End
	
	// V102 Begin
	// Set PDF Security Options
	var pdfSecOptions = _getPDFSecurityOptions(res);
	if(pdfSecOptions !== null){
		a2wsProps.PDFSecurity = pdfSecOptions;
	}
	// V102 Begin
		
	// V101 Begin
	//a2wsProps.InputBuffer	= new Buffer(res._Context.inputDoc, 'base64');
	a2wsProps.InputBuffer = res._Context.inputDoc;
	// V101 End
	
	// Add mandatory fields as AFP2web Server Props
	a2wsProps.Filename		= 'Buffer';
	a2wsProps.OutputMode	= 'buffer';
	a2wsProps.server_log_level = pluginConf.a2wslogging || pluginConf.log.level || 'INFO'; // set logging level

	// Call AFP2web Server's syncTransformServerFile service
	a2wServices.syncTransformBuf(a2wsProps, function(err, result){ 
		if(err) return callback(err);

		// Store Data in debug mode ONLY
		if(logger.isDebugEnabled()){
			var outputFilename = dateFormat.asString('yyyy-MM-dd_hh-mm-ss', new Date()) + '_' + req.input.type + '.' + req.output.type;
			fs.writeFileSync(npsTempDir + '/' + outputFilename, new Buffer(result, 'binary'));
		}	
		callback(null, result);
	});
}

// Process an ASYNC Transformation request
function _asyncTransform(res, callback){

	// V101 Begin
	var req = res._Context.req;

	// Assert Input document	
	if(res._Context.inputDoc === undefined
		|| res._Context.inputDoc.length <= 0
	   ){
		err = new Error( 'Invalid Request. Reason: input document is missing !\n' + 
					  'Request:\n' + JSON.stringify(req));

		//!!!logger.error('--->_asyncTransform: Req.Id:' + res._Context.reqId + ', Error:' + err.message);
		err.httpStatusCode = 400; /* Bad Request */
		return callback(err);
	}
	// V101 End

	// V106 Begin
	if(typeof res._Context.inputDoc !== 'string'){
		err = new Error(   'Invalid input document. Reason: Buffer input is not allowed for asynchronous transformation !\n'
                         + 'Request:\n' + JSON.stringify(req));

		//!!!logger.error('--->_asyncTransform: Req.Id:' + res._Context.reqId + ', Error:' + err.message);
		err.httpStatusCode = 400; /* Bad Request */
		return callback(err);
	}
	// V106 End

	// V103 Begin
	// Build Scenario Conf 
	if(res._Context.scenarioConf === undefined){
		var err = _buildScenarioConf(res);
		if(err !== null){
			//!!!logger.error('--->asyncTransform: Req.Id:' + res._Context.reqId + ', Error:' + err.message);
			err.httpStatusCode = 400; /* Bad Request */
			return callback(err);
		}
	}
	// V103 End
	 
	// Set A2WS Props
	var a2wsProps = utils.clone(res._Context.scenarioConf.props); //!!! Using a = b does NOT copy !!!

	// V103 Begin
	if(res._Context.a2wsProps !== undefined){
		// concatenate request specific AFP2web props with scenario a2wsProps 
		a2wsProps = utils.mergeJSON(a2wsProps, res._Context.a2wsProps);
	}
	// V103 End

	// V102 Begin
	// Set PDF Security Options
	var pdfSecOptions = _getPDFSecurityOptions(res);
	if(pdfSecOptions !== null){
		a2wsProps.PDFSecurity = pdfSecOptions;
	}
	// V102 Begin

	// Add mandatory fields as AFP2web Server Props
	a2wsProps.server_log_level = pluginConf.a2wslogging || pluginConf.log.level || 'INFO'; // set logging level

	// Create a JobId
	res._Context.jobId = utils.createJobId();
	logger.debug('--->_asyncTransform: Req. id=' + res._Context.reqId + ', JobId=' + res._Context.jobId);

	// Add the JobId
	a2wsProps.JobId = res._Context.jobId;

	// V101 Begin
	// Add InputURLs (inputDoc is base64 encoded, so decode it!) Ex: ['/var/lib/apache2/fcgid/a2wserver/samples/bigfiletest.afp']
	//a2wsProps.InputURLs = new Buffer(res._Context.inputDoc, 'base64'); 
	a2wsProps.InputURLs = res._Context.inputDoc;
	// V101 End

	a2wsProps.OutputMode = 	'file';

	// Call AFP2web Server's syncTransformServerFile service
	a2wServices.asyncTransformServerFile(a2wsProps, callback);
}

// Process a Status request
function _syncStatus(res, callback){

	// V101 Begin
	var req = res._Context.req
	  , a2wsProps = {}
	  ;

	// Assert JobId	
	if (res._Context.jobId === undefined
		|| res._Context.jobId.length <= 0
	   ){
		err = new Error( 'Invalid Request. Reason: Input JobId is missing !\n' + 
					  + 'Request:\n' + JSON.stringify(req));
		//!!!logger.error('--->_syncStatus: ' + ' Req.Id:' + res._Context.reqId + ', Error:' + err.message);
		err.httpStatusCode = 400; /* Bad Request */
		return callback(err);
	}
	// V101 End

	// Add the JobId
	a2wsProps.JobId = res._Context.jobId;
		
	// V103 Begin
	if(res._Context.a2wsProps !== undefined){
		// concatenate request specific AFP2web props with scenario a2wsProps
		a2wsProps = utils.mergeJSON(a2wsProps, res._Context.a2wsProps);
	}
	// V103 End

	// Add mandatory fields as AFP2web Server Props
	a2wsProps.server_log_level = pluginConf.a2wslogging || pluginConf.log.level || 'INFO'; // set logging level

	// Call AFP2web Server's syncTransformServerFile service
	a2wServices.getStatus(a2wsProps, function(err, result){ 
		if(err) return callback(err);

		// Store Data in debug mode ONLY
		if (logger.isDebugEnabled()){
			var outputFilename = dateFormat.asString('yyyy-MM-dd_hh-mm-ss', new Date()) + '_' + req.input.type + '.' + req.output.type;
			fs.writeFileSync(npsTempDir + '/' + outputFilename, new Buffer(JSON.stringify(result)));
		}
		callback(null, result);
	});
}

// V101 Begin
// Build Scenario Conf. for AFP2web Server Request
function  _buildScenarioConf(res){

	var req 	 = undefined
	  , a2wsConf = undefined

	if(res._Context.scenarioConf !== undefined){
		return null;
	}

	req = res._Context.req;
	if (req.input === undefined || 
		req.input.type === undefined || 
		req.output.type === undefined ){
		return new Error( 'Invalid JSON Request. Reason: input.type, output.type ' + 
					 'is missing !\n' + 'JSON Request:\n' + JSON.stringify(req));
	}

	// Check whether inputType belongs to any
	if (a2wsAny && a2wsAny.indexOf(req.input.type.toLowerCase()) !== -1){
		req.input.type = 'any';
	}

	// Build Scenario Name
	res._Context.scenario = req.input.type + '2' + req.output.type;

	// Check if Scenario exists
	if (!scenarios || scenarios[res._Context.scenario] === undefined){
		return new Error( 'Scenario: ' + res._Context.scenario + ' does NOT exit!');
	}

	// Assert scenarios[res._Context.scenario].action
	if (scenarios[res._Context.scenario].action === undefined){
		return new Error( 'Missing action in Scenario: ' + res._Context.scenario);
	}
	
	a2wsConf = scenarios[res._Context.scenario].action.config;
	logger.debug('--->_buildScenarioConf: Input: ' + req.input.type + ', Output: ' + req.output.type + ', Scenario: ' + a2wsConf);

	// Assert a2wsConf
	if (a2wsConf === undefined){
		return new Error( 'Missing action.config in Scenario: ' + res._Context.scenario);
	}

	if (confs[a2wsConf] === undefined){
		return new Error( 'Configuration: ' + a2wsConf + ' does NOT exit!');
	}	

	// Add the Scenario Configuration to the request
	res._Context.scenarioConf = confs[a2wsConf];

	return null;
}

// Build A2W PDF Security Options
function _getPDFSecurityOptions(res){
	var pdfSecurityOpts = null;
	var req = res._Context.req;

	if(req.output.type === 'pdf'){
		pdfSecurityOpts = (req.output.ownerpw || '') + ',';
		pdfSecurityOpts += (req.output.userpw || '') + ',';
		// Key length is not given in SOAP/JSON request output parameters,
		// instead it is hardcoded as '128' in Servicemix service. !!! check with Daniel
		pdfSecurityOpts +=  '128,';
		pdfSecurityOpts += req.output.flags || '';
		if( pdfSecurityOpts === ',,128,' ) pdfSecurityOpts = null;	
	}
	else if(req.input.type.indexOf( 'pdf' ) >= 0 ) {	// Use indexof to check input type, instead of equal since inputype may be simply 'pdf' or 'asyncpdf'
		pdfSecurityOpts = (req.input.ownerpw || '') + ',';
		pdfSecurityOpts += (req.input.userpw || '') + ',';
		// Key length is not given in SOAP request input params. Not needed for parsing PDF inputs.
		pdfSecurityOpts +=  ',';	
		pdfSecurityOpts +=  '';	// Key length is not given in SOAP request input params 
		if(pdfSecurityOpts === ',,,') pdfSecurityOpts = null;	
	}
	
	return pdfSecurityOpts;
}

/**
 * _logServiceRespTime:  	Log a Warning if Service Response time exceeded maxresptime 
 * @param  {res} 			res     	
 * @param  {String} 		service
 */
function _logServiceRespTime(res, service){

    var respTime 	= 0
	  , os 			= require('os')
      ;

    // Check resp. time only if maxresptime is defined
   	if(pluginConf.maxresptime){		// maxresptime in milliseconds
		respTime = utils.dateDiff('x', res._Context.date, new Date()); // 'x' means returns diff. in ms

   		if(respTime > pluginConf.maxresptime){
			logger.warn('--->_logServiceRespTime: Req. id=' + res._Context.reqId + ', ' + service + 
						', Resp.Time=' + respTime + 'ms, Max.Allowed=' +
						 pluginConf.maxresptime + 'ms. Average Load=' + os.loadavg());   			
   		}
   	}
}
