/**-----------------------------------------------------------------------------
 * backup.js: 	Node.js module to backup the whole OXSNPS server
 *
 * Author    :  AFP2web Team
 * Copyright :  (C) 2015 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V1.0.0
 * 
 * History
 *  V100        21.05.2015     Initial release
 *  
 *----------------------------------------------------------------------------*/
'use strict';

var MODULE_NAME 	= 'backup'
  , MODULE_VERSION 	= '2.0.0'
  , fs 				= require('fs')
  , path 			= require('path')
  , log4js 			= require('log4js')
  , admZip 			= require('adm-zip')

  , npsServer		= require('./server')
  , npsDir	    	= npsServer.npsDir
  , npsConfDir 		= npsServer.npsConfDir
  , npsConf 		= npsServer.npsConf					// holds the configuration of the Server   
  , npsLogFile 		= npsServer.npsLogFile  
  , utils 			= require('./helpers/utils')
  , dateformat 		= require('./helpers/date_format')
  , backupDir 		= npsConf.backupDir
  ;

// Get Logger
var logger = log4js.getLogger(MODULE_NAME);
logger.setLevel(npsConf.log.level);

// Export Module Version
exports.version = MODULE_VERSION;

/**************** PUBLIC FUNCTIONS ****************/
/**
 * initialize: 			Initialize Module
 * @param  {function} 	callback(err)
 */
exports.initialize = function(callback){

	var self = this;
	
	if(self._initialized){
		return callback();
	}
	self._initialized = true;
	callback();
}

/**
 * finalize: 			Finalize Module
 * @param  {function} 	callback(err) 
 */
exports.finalize = function(callback){

	var self = this;

	if(!self._initialized){
		callback();
	}
	self._initialized = false;
	callback();
}

/**
 * version: 			Process a HTTP Version Request
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.version = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);
	logger.info('--->version: Req. id=' + res._Context.reqId);

	var data = '<h3>' +  MODULE_NAME + ' v' + MODULE_MODULE_VERSION + '</h3>';
	res.end(data);
	
	logger.info('--->version: Req. id=' + res._Context.reqId + ', sent back ' +
				MODULE_NAME + ' v' + MODULE_MODULE_VERSION + ', over');
	return;
}

/**
 * backup:  				Backup the current installation
 * @param  {req} 			req 
 *         					req.param('nodemodules'): 	true to backup the node_modules dir
 *         					req.param('test'): 			true to backup the test dir
 * @param  {res} 			res 
 */
exports.backup = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);
	logger.info('--->backup: Req. id=' + res._Context.reqId);
	
	// Set default options parms
	var options = {
		nodemodulesDir: false,	// backup the node_modules dir
		testDir:		false	// do not backup the test dir
	};

	//!!! Must be a better way to get parms from a HTTP GET req.
	// Call is http://localhost:1026/appservices/backup?nodemodules=true&test=false
	if(req.query.nodemodules !== undefined) options.nodemodulesDir 	= (req.query.nodemodules === 'true')? true: false;
	if(req.query.test  !== undefined)		options.testDir 		= (req.query.test === 'true')? true: false;

	// Backup the current installation
	exports.backupService(options, function(err, backupFilename){
		if(err){
			logger.error('--->backup: Req. id=' + res._Context.reqId + ', Error: ' + err.message);
			return res.status(404).end('--->backup: Req. id=' + res._Context.reqId + ', Error: ' + err.message);
		}
		logger.info('--->backup: Req. id=' + res._Context.reqId + ', over. Backup Filename: ' + backupFilename);
		return res.end('--->backup: Req. id=' + res._Context.reqId +  ', over. Backup Filename: ' + backupFilename);
	});
}

/**
 * backupService: 			Backup the whole OXSNPS server
 * @param  {Function} 		cb(err)
 */
exports.backupService = function(options, cb){

	logger.debug('--->backupService: Backing up the OXNSPS server...');

	// Backup the current OXSNPS server version
	var zip = new admZip();

	// Backup filename
	var backupFilename = backupDir + '/' + dateformat.asString('yyyy-MM-dd_hh-mm-ss', new Date()) + '_backup.zip' 

		// 1. Backup following directories
		, confDir 					=  'conf'
		, nodemodulesDir			=  'node_modules'
		, publicDir					=  'public'
		, testDir					=  'test' //!!! Watch out! May not be available

		// 2. Backup following files
		, gruntfileFilename			= 'Gruntfile.js'
		, historyFilename			= 'history.txt'
//!!!	, OXS_ServerFilename		= 'oxs-import-server' //!!! Get the oxsnps-server start/stop script from package.json "bin" property
		, oxsnpsFilename			= 'oxsnps'
		, packageFilename			= 'package.json'
		, processes_jsonFilename 	= 'processes.json'
		, projectParts_txtFilename 	= 'projectParts.txt'
		; 

	try{
		// Add conf dir
		zip.addLocalFolder(npsDir + '/' + confDir, 			confDir);

		// Add node_modules dir if requested
		if(options.nodemodulesDir){
			zip.addLocalFolder(npsDir + '/' + nodemodulesDir, 	nodemodulesDir);
		}

		// Add public dir
		zip.addLocalFolder(npsDir + '/' + publicDir, 		publicDir);

		//!!! Watch out! test dir may not be available
		// Add test dir if requested
		if(options.testDir){
			zip.addLocalFolder(npsDir + '/' + testDir, 			testDir);
		}

		// Add Files
		zip.addLocalFile(npsDir + '/' + gruntfileFilename);
		zip.addLocalFile(npsDir + '/' + historyFilename);
//!!!	zip.addLocalFile(npsDir + '/' + OXS_ServerFilename);
		zip.addLocalFile(npsDir + '/' + oxsnpsFilename);
		zip.addLocalFile(npsDir + '/' + packageFilename);
		zip.addLocalFile(npsDir + '/' + processes_jsonFilename);
		zip.addLocalFile(npsDir + '/' + projectParts_txtFilename);

		// Write Zip file to disk
		zip.writeZip(backupFilename);
		logger.debug('--->backupService: Current installation backed up to ' + backupFilename);
	}	  
	catch(err){
		return cb(new Error('' + err));
	}
	return cb(null, backupFilename);
}

/**
 * Set the log level of this module
 * @param  {req} 	GET req   http://localhost:1026/appservices/backup/setloglevel?loglevel=DEBUG|INFO|ERROR
 * @param  {res}	res
 */
exports.setLogLevel = function(req, res){
	var loglevel = req.query.loglevel || 'INFO';
	logger.setLevel(loglevel);
	return res.end('--->setLogLevel: Log level of ' + MODULE_NAME + ' set to ' + loglevel);
}

/**
 * Return the log level of this module
 * @param  {req} 	GET req   http://localhost:1026/appservices/backup/getloglevel
 * @param  {res}	res
 */
exports.getLogLevel = function(req, res){
	logger.info('--->getLogLevel: Getting the log level of ' + MODULE_NAME + '...');
	return res.end(logger.level.levelStr);
}

/******************  Private Functions ********************/
