/**-----------------------------------------------------------------------------
 * cache.js: Node.js module that provides APIs for caching
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2014 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V1.0.0
 * 
 * History
 *  V100        13.11.2014     Initial release
 *  V102        06.01.2015     Extended to add cron job to schedule cache deletion
 *
 *----------------------------------------------------------------------------*/
'use strict';

var MODULE_NAME 	= 'cache'
  , MODULE_VERSION 	= '2.0.0'
  , fs          	= require('fs')
  , path        	= require('path')
  , async	    	= require('async')
  , log4js	    	= require('log4js')  
  , cronjob     	= require('./cronjob')
  , npsServer		= require('./server')
  , npsDir	    	= npsServer.npsDir
  , npsConf 		= npsServer.npsConf	// holds the configuration of the OTS Server   
  , npsLogFile 		= npsServer.npsLogFile  
  , utils 			= require('./helpers/utils')
  ;

// Get Logger
var logger = log4js.getLogger(MODULE_NAME);
logger.setLevel(npsConf.log.level);

// Export Module Version
exports.version = MODULE_VERSION;

/**
 * Set the log level of this module
 * @param  {req} 	GET req   http://localhost:1026/appservices/cache/setloglevel?loglevel=DEBUG|INFO|ERROR
 * @param  {res}	res
 */
exports.setLogLevel = function(req, res){
	var loglevel = req.query.loglevel || 'INFO';
	logger.setLevel(loglevel);
	return res.end('--->setLogLevel: Log level of ' + NAME + ' set to ' + loglevel);
}

/**
 * Return the log level of this module
 * @param  {req} 	GET req   http://localhost:1026/appservices/cache/getloglevel
 * @param  {res}	res
 */
exports.getLogLevel = function(req, res){
	logger.info('--->getLogLevel: Getting the log level of ' + NAME + '...');
	return res.end(logger.level.levelStr);
}

/**
 * Constructor
 * 
 */
function cache(){};

/**
 * initialize
 * @param  {[type]}   options  
 *         				{String}   cacheDir   Fully qualified directory name. It is used as a base directory to cache file.
 *                               			  Whenever 'add' method is called a new sub-directory is created within this directory
 *                      {String}   cronTime   A cron time that defines the cron job command to clean the cache directory
 *                      {String}   maxSize    Maximum size of cache directory in Mega Bytes. 
 *                               			  When cache directory size exceeds this value, older or unused entries will be removed from cache
 *                      {String}   timeToKeep Maximum time in minutes to keep entries in cache.
 * 
 * @param  {Function} callback   A callback function that would be called after initializing the cache
 *                               callback function will have one argument "err" of node.js type "Error".
 *                               In case of success, "err" will be null.
 *                               In case of  error, "err" will have information about error occurred during initialization.
 * @return {}              
 */
cache.prototype.initialize = function(cacheDir, options, callback){
	var self = this;

	self._initialized 	= false;
	self._cacheDir 		= cacheDir;
	self._cronTime 		= options.cronTime;
	self._maxSize 		= options.maxSize;
	self._timeToKeep 	= options.timeToKeep;
	self._queues	    = [];

	// V102 Begin
	async.series([
		// Task 1: create cache directory
		function(nextTask){
			//var mode = 0777;	// NOT allowed with strict!!!
			var mode = parseInt('0777',8);	// define the mode. The leading 0 MUST be specified since mode is specified as an octal value
			utils.mkDir(self._cacheDir, mode, function(err){
				if(err){
					nextTask(new Error('Could not create cache directory "' + self._cacheDir + '". ' + err.message));
				}
				else{
					nextTask();
				}
			});
		},

		// Task 2: Create a cron job for cleaning OTS Cache
		function(nextTask){
			self.createAndStartCronJob(function(err){
				if(err){
					nextTask(new Error('Could not create cron job for cache directory deletion. Invalid cron time command. ' + err.message));
				}
				else{
					self._initialized = true;
					nextTask();
				}
			});
		}
	],	function(err){
			if(err){
				logger.error('--->initialize: Error on initializing cache, Error: ' + err.message);
			}
			return callback(err);
		}
	);
	// V102 End
};

/**
 * Add a file to the cache dir
 * @param {String}         filename   Name of the file to be added
 * @param {String|Buffer}  data       Data to write to file 
 * @param {String|Object}  fileOptions  encoding to use to write file (string to keep backward compatibility) or 
 *                                      a file options object:
 *                                    	{ 
 *										  "encoding": <encoding>
 *										  "overwrite": true | false. <Default is false>
 *										  ..., 
 *										}
 * @param {Function}       callback   A callback function 
 *                                    callback function will have one argument "err" of node.js type "Error".
 *                                    In case of success, "err" will be null.
 *                                    In case of error, "err" will have information about error occurred on directory creation
 * @return {}              
 */
cache.prototype.addFile = function(filename, data, fileOptions, callback){
	var self = this;

	if(!self._initialized){
		return callback(new Error('cache not intialized'));
	};

	if (!fileOptions ||  typeof fileOptions === 'string') {
		// contains encoding, convert into (JSON) object
		var encoding = fileOptions;
		fileOptions = {};
		fileOptions.encoding = encoding;
	}

	var dirname  = ""
	  , dirQ     = null
	  , fileInfo = {
		  'name'      : path.resolve(self._cacheDir + '/' + filename), 
		  'data'      : data,
		  'encoding'  : fileOptions.encoding,
		  'overwrite' : fileOptions.overwrite || false,
		  'action'    : _addFile
	    }
	  ;

	// Get directory name
	var matches = fileInfo.name.match(/^(.+)[\/|\\][^[\/|\\]+$/);
	// matches is an array. First item in the array is always the given input string, 
	// followed by matched parts of the given string	
	dirname = matches[1];

	// Get queue of this directory
	dirQ = self.getQ(dirname);

	// add the file to the queue
	logger.debug('--->addFile: Adding file '+ fileInfo.name + ' to "'+ dirname + '" queue');
	dirQ.push(fileInfo, function(err){
		// control comes here only after file has been added to file system
	 	return callback(err);
	});
}

/**
 * Delete a file from the cache dir
 * @param {String}         filename   Name of the file to be added
 * @param {Function}       callback   callback(err)
 */
cache.prototype.deleteFile = function(filename, callback){

	var self = this;

	if(!self._initialized) return callback(new Error('cache not intialized'));

	var dirname  = ""
	  , dirQ     = null
	  , fileInfo = {
		  'name'      : path.resolve(self._cacheDir + '/' + filename), 
		  'action'    : _deleteFile
	    }
	  ;

	// Get directory name
	var matches = fileInfo.name.match(/^(.+)[\/|\\][^[\/|\\]+$/);
	// matches is an array. First item in the array is always the given input string, 
	// followed by matched parts of the given string	
	dirname = matches[1];

	// Get queue of this directory
	dirQ = self.getQ(dirname);

	// add the file to the queue
	logger.debug('--->deleteFile: Adding ' + fileInfo.name + ' for deletion to "'+ dirname + '" queue');
	dirQ.push(fileInfo, function(err){
		// control comes here only after file has been delete from file system
	 	return callback(err);
	});
}
/**
 * Creates a new sub directory <dir>, if it does not exist in cache directory.
 * @param {String}   dir      	 Directory name.
 * @param  {Function} callback   A callback function 
 *                               callback function will have one argument "err" of node.js type "Error".
 *                               In case of success, "err" will be null.
 *                               In case of error, "err" will have information about error occurred on directory creation
 * @return {}              
 */
cache.prototype.addDir = function(dir, callback){
	var self = this;

	if(!self._initialized){
		return callback(new Error('cache not intialized'));
	};

	var dirInfo={
		'name'   : path.resolve(self._cacheDir + '/' + dir), 
		'mode'   : parseInt('0777',8),	// define the mode. The leading 0 MUST be specified since mode is specified as an octal value
		'action' : _addDir
	};

	// Get queue of this directory
	var dirQ = self.getQ(dirInfo.name);

	// add the file to the queue
	logger.debug('--->addDir: Adding directory '+ dirInfo.name + ' to "'+ dirInfo.name + '" queue');
	dirQ.push(dirInfo, function(err){
		// control comes here only after directory has been added to file system
	 	return callback(err);
	});
}

/**
 * Checks if passed <dir>+'/'+<filename> exists in cache 
 * @param  {String}   dir      Sub Directory name
 * @param  {String}   filename File name
 * @param  {Function} callback callback function. 
 *                             callback function will have two arguments.
 *                               	"err", of node.js type "Error"
 *                                	"exists", of type Boolean
 *                             In case of success,
 *                               	"err" will be null.
 *                                	"exists" will be true if file exists, else false.
 *                             In case of error,
 *                               	"err" will have information about error occcurred
 *                                	"exists" will be null.
 * @return {}           
 */
cache.prototype.hasFile = function(dir, filename, callback){
	var self = this;
	
	if(!self._initialized ){
		return callback(new Error('cache not intialized', null));
	};

	// Set the Fully Qualified name of the file
	var fqFilename = path.resolve(self._cacheDir + '/' + dir) + '/' + filename;
	fs.exists(fqFilename, function(exists){
		return callback(null, exists);
	});
}

/**
 * Checks if passed <dir>+'/'+<filename> exists in cache 
 * @param  {String}   dir      Sub Directory name
 * @param  {Function} callback callback function. 
 *                             callback function will have two arguments.
 *                               	"err", of node.js type "Error"
 *                                	"exists", of type Boolean
 *                             In case of success,
 *                               	"err" will be null.
 *                                	"exists" will be true if directory exists, else false.
 *                             In case of error,
 *                               	"err" will have information about error occcurred
 *                                	"exists" will be null.
 * @return {}           
 */
cache.prototype.hasDir = function(dir, callback){
	var self = this;
	
	if(!self._initialized ){
		return callback(new Error('cache not intialized', null));
	};

	var fqDir = path.resolve(self._cacheDir + '/' + dir);
	fs.exists(fqDir, function(exists){
		return callback(null, exists);
	});
}

/**
 * Returns the content of <dir> +'/'+<filename> from the cache
 * @param  {String}   dir      Sub Directory name
 * @param  {String}   filename Filename
 * @param  {Function} callback callback function expecting 2 arguments:
 *                             	1. err", of node.js type "Error"
 *                              2. "data", of type Buffer???
 * @return {}           
 */
cache.prototype.getFile = function(dir, filename, encoding, callback){
	var self = this;
	
	if(!self._initialized ){
		return callback(new Error('cache not intialized', null));
	};

	// Set the Fully Qualified name of the file
	var fqFilename = path.resolve(self._cacheDir + '/' + dir) + '/' + filename;
	fs.readFile(fqFilename, {'encoding': encoding || 'utf8'}, function(err, data){
		if(err){
			return callback(err);
		}
		return callback(null, data);
	});
}

/**
 * Called from the cron Job to clean the cache directory
 * @return {}              
 * Callback parameter removed since cron job does not call, passed 'onTick' function with callback function.
 * The callback was logging the status of cache deletion. That is done now within clean() function.
 */
cache.prototype.clean = function(){
	var self = this;
	logger.info('--->clean: Running the cron job to clean the OTS cache directory "' + self._cacheDir + '"...');
	if(!self._initialized){
		logger.error('--->clean: cache not intialized');
		return;
	};
	// V102 Begin
	// Delete cache dir entries that are old enough for deletion
	utils.deleteDir(self._cacheDir, false /* do not delete cache root dir */, 'n', self._timeToKeep, true /* recursively delete dir entries */, function(err){
		if(err)	logger.error('--->clean: Cron job over with error!. ' + err.message);
		else logger.info('--->clean: Cron job over!');
		return;
	});
	// V102 End
}

// V102 Begin
/**
 * Creates cron job to clean cache directory and starts it.
 * @param  {String}   cronTime Time to run cron job as defined in http://crontab.org/
 * @param  {Function} callback A callback function
 *                               callback function will have one argument "err" of node.js type "Error".
 *                               In case of success, "err" will be null.
 *                               In case of  error, "err" will have information about error occurred on running cron job
 * @return {}            
 */
cache.prototype.createAndStartCronJob = function(callback){
	var self          = this
	  , options       = {}
	  ;

	// If cronTime not specified, do not create cron job.
	if(!self._cronTime) return callback();

	// Check if cron job already exists
	cronjob.existsCronJobService(self._cacheDir, function(err, exists){
		if(err)	return callback(err);

		else if(exists){
			logger.debug('--->createAndStartCronJob: Cron job "' + self._cacheDir + '" already exists');
			return callback();
		}
		else{
			// Setup options
			options.name         = self._cacheDir;
			options.cronTime     = self._cronTime;
			options.cronFunction = self.clean;
			self.function 		 = 'oxsnps-core/cache.js->clean'; // (Optional) a String to identify the cronFunction
			options.cronOptions  = {'start':true, 'context':self};

			// Add cron job
			logger.debug('--->_createAndStartCronJob: Creating cron job to clean the OTS cache directory "' + self._cacheDir + '"...');
			cronjob.addCronJobService(options, function(err){
				return callback(err);
			});
		}
	});
}
// V102 End

/**
 * Rename the file
 * @param  {String}   oldName  Filename to be renamed
 * @param  {String}   newName  New filename
 * @param  {Function} callback A callback function
 *                               callback function will have one argument "err" of node.js type "Error".
 *                               In case of success, "err" will be null.
 *                               In case of  error, "err" will have information about error occurred on renaming the file
 * @return {}            
 */
cache.prototype.rename = function(oldName, newName, options, callback){
	var self = this
	  , state = ""
	  ;

	if(typeof options === 'function'){
		callback = options;
		options = {};
	}

	if(!self._initialized){
		return callback(new Error('cache not intialized'));
	};

	var renInfo={
		'oldName'  : path.resolve(oldName), 
		'newName'  : path.resolve(newName),
		'action'   : _rename,
		'options'  : options
	};

	async.series([
		// Task 1: Check if new file exists		
		function(nextTask){
			if(renInfo.options && renInfo.options.overwrite){
				state = '10';
			 	return nextTask();
			}
			fs.exists(renInfo.newName, function(exists){
				if(exists){
					logger.debug('--->rename: ' + renInfo.newName + ' already exists in cache directory');
					fs.unlink(renInfo.oldName, function(err){
						logger.debug('--->rename: ' + renInfo.oldName + ' deleted since new file ' + renInfo.newName + ' already exists in cache directory');
						state='end';
						nextTask();
		        	});
				}
				else{
					state = '10';
					nextTask();					
				}
			});			
		},
		// Task 2: Check if old file exists
		function(nextTask){
			switch (state) {
				case '10': {	
					// Check if old name exists
					fs.exists(renInfo.oldName, function(exists){
						if(exists){
							state='20';
							nextTask();
						}
						else{
							state = 'end';
							nextTask(new Error('Rename failed. Source file ' +  renInfo.oldName + ' does not exist'));					
						}
					});								
					break;
				}
				default: {
					nextTask();
					break;
				}
			}
		},
		// Task 3: Add Rename action to queue
		function(nextTask){
			switch (state) {
				case '20': {	
					// Get directory name
					var matches = renInfo.newName.match(/^(.+)[\/|\\][^[\/|\\]+$/)
						// matches is an array. First item in the array is always the given input string, 
						// followed by matched parts of the given string	
					  , dirname = matches[1]
					  ;

					// Get queue of this directory
					var dirQ = self.getQ(dirname);
					logger.debug('--->rename: Adding renaming of '+ renInfo.oldName + ' to ' + renInfo.newName + ' added to "'+ dirname +'" queue');
					dirQ.push(renInfo, function(err){
						// control comes here only after file has been added to file system
					 	return nextTask(err);
					});
					break;
				}
				default: {
					nextTask();
					break;
				}
			}
		}
	],	function(err){
			if(err){
				logger.error('--->rename: Renaming of ' + renInfo.oldName + ' to ' + renInfo.newName + ' failed, Error: ' + err.message);
			}
			return callback(err);
		}
	);
}

/**
 * Rename files in passed directory
 * @param  {String}   dir          Directory where files has to be renamed
 * @param  {String}   filePat      Fiel name pattern matching files to be renamed with path
 * @param  {String}   matchPat     Pattern in file name to be replace with 'replaceValue'
 * @param  {String}   replaceValue Value to replace the pattern matched in the filename
 * @param  {Function} callback     [description]
 * @return {[type]}                [description]
 *
 * Example
 * renameFiles('var/cache/nps/aee8ed57def76704f3d4d3b43911f820', 'reqid-[0-9]*.tif', 'reqid', 'docid', callback)
 */
cache.prototype.renameFiles = function(dir, filePat, matchPat, replaceValue, callback){
	var self    = this
	  ;

	if(!self._initialized){
		return callback(new Error('cache not intialized'));
	};

	var idx 	   = 0
	  , entryCount = 0
	  , errSave    = null
	  , absPath    = null
	  , regexp     = new RegExp(matchPat)
	  ;

	// Collect files from output directory that matches output filename patten and pagenr >= start page && <= end page
	var absPath = path.resolve(dir) + '/';

	fs.readdir(absPath, function (err, files){
		if(err){
	    	return callback(err);
		}
		else if(files === undefined || files.length <=0){
		  return callback(new Error("Files not found in output directory. Dir:" + absPath));
		}
		else{
	  		entryCount = files.length;
	  		files.forEach(function(file){

	  			if(errSave !== null){
	  				// An error occurred while renaming a file, so do not proceed with other files.
	  				return;
	  			}

	    		var stats = fs.statSync(absPath + file);
	    		if(stats.isFile()){
					// ensure file matched the required file pattern passed as arg
	    			var values         = file.match(filePat)
	    			  , destFilename   = ""
	    			  , srcFilename    = ""
	    			  , value          = ""
	    			  ;
					logger.debug('--->_renameFiles: checking pattern of ' + absPath + file);
	    			if(values && values.length > 0){
	    				srcFilename  = absPath + file;
				        value = file.replace(regexp, replaceValue);
				        destFilename = absPath + value;
						logger.debug('--->_renameFiles: call renaming ' + srcFilename + ' to ' + destFilename );
						self.rename(srcFilename, destFilename, function(err){
						    if(err){
						    	errSave = err;
					    		return callback(new Error('Renaming of ' + srcFilename+ ' to ' + destFilename + ' failed, Error: ' + err.message));
						    }
							else if(!--entryCount){
								return callback(null);
							}
						});
						return;
	      			}
	      			if (!--entryCount){
	        			// Invoke callback only after all entries are processed
	        			return callback(null);
	      			}
	      		}
			    else{
					if (!--entryCount){
	        			// Invoke callback only after all entries are processed
	        			return callback(null);
	      			}
	    		}
	    	});
	    }
	});
}

/**
 * Get the queue that corresponds to the passed directory.
 * If queue does not exist, create one.
 * @param  {[type]} dirname Directory name
 * @return {JSON}           Queue 
 */
cache.prototype.getQ = function(dirname){
	var self = this
	  ;

	if(!self._queues[dirname]){
		logger.debug('--->getQ: Creating queue for "' + dirname + '" directory');

		self._queues[dirname] = {};

		// Create a queue for this directory
		self._queues[dirname].queue = async.queue(function(itemInfo, next){
				itemInfo.action(itemInfo, function(err){
					next(err);
				});	 
			},
			1  	// run one task at time
		);
	}

	return self._queues[dirname].queue;
}

/******************  Private Functions ********************/	
/**
 * Adds a file to the cache directory 
 * @param {JSON}     fileInfo  Object with filename, data, encoding and overwrite props.
 * @param {Function} callback  A callback function 
 *                               callback function will have one argument "err" of node.js type "Error".
 *                               In case of success, "err" will be null.
 *                               In case of error, "err" will have information about error occurred on file creation
 * @return {}              
 */
function _addFile(fileInfo, callback){

	var skipOtherTasks = false;

	async.series([
		// Task 1: If overwrite flag is set, don't check file existence
		function(nextTask){
			if (fileInfo.overwrite === true) return nextTask();

		 	fs.exists(fileInfo.name, function(exists){
				if(exists){
					logger.debug('--->_addFile: ' + fileInfo.name + ' file already exists in cache directory');
					skipOtherTasks = true;
				}
				nextTask();
			});
		},
		
		// Task 2: Ensure directory exists, else create it
		function(nextTask){
			if (skipOtherTasks) return nextTask();

			var matches = fileInfo.name.match(/^(.+)[\/|\\][^[\/|\\]+$/)
				// matches is an array. First item in the array is always the given input string, 
				// followed by matched parts of the given string	
			  , dirname = matches[1]
			  , mode    = 0
			  ;
			mode = parseInt('0777',8);	// define the mode. The leading 0 MUST be specified since mode is specified as an octal value
			_addDir({'name':dirname, 'mode':mode}, function(err){
				nextTask(err);
			});
		},

		// Task 3: Create a file
		function(nextTask){
			if (skipOtherTasks) return nextTask();

			fs.writeFile(fileInfo.name, fileInfo.data, {'encoding': fileInfo.encoding || 'utf8'}, function(err){
				if(!err){
					logger.debug('--->_addFile: ' + fileInfo.name + ' added to cache directory'); 
				}
				nextTask(err);
			});
		}
	],	function(err){
			if(err){
				logger.error('--->addFile: Adding file ' + fileInfo.name + ' to OTS cache failed, Error: ' + err.message);
			}
			return callback(err);
		}
	);
}

/**
 * Deletes a file to the cache directory 
 * @param {JSON}     fileInfo  Object with filename
 * @param {Function} callback  callback (err)
 * @return {}              
 */
function _deleteFile(fileInfo, callback){
	fs.unlink(fileInfo.name, callback)
}

/**
 * Adds a directory to the cache directory
 * @param {JSON}     dirInfo  Object with directory name and mode
 * @param {Function} callback A callback function 
 *                               callback function will have one argument "err" of node.js type "Error".
 *                               In case of success, "err" will be null.
 *                               In case of error, "err" will have information about error occurred on directory creation
 * @return {}              
 */
function _addDir(dirInfo, callback){

	// utils.mkDir creates directory only if it does not exist
	utils.mkDir(dirInfo.name, dirInfo.mode, function(err){
		if(!err){
			logger.debug('--->_addDir: ' + dirInfo.name + ' exists or added to cache directory'); 
		}				
		return callback(err);
	});
}

/**
 * Renames the passed file 
 * @param {JSON}     dirInfo  Object with directory name and mode
 * @param {Function} callback A callback function 
 *                               callback function will have one argument "err" of node.js type "Error".
 *                               In case of success, "err" will be null.
 *                               In case of error, "err" will have information about error occurred on directory creation
 */
function _rename(renInfo, callback){

	async.series([
		// Task 1: If overwrite is false, check file exists, if exists return
		function(nextTask){
			if(renInfo.options && renInfo.options.overwrite) return nextTask();
			fs.exists(renInfo.newName, function(exists){
				if(exists){
					logger.debug('--->__rename: ' + renInfo.newName + ' already exists in cache directory');
					return callback();
				}
				nextTask();
			});
		},
		// Task 2: // Use mv to move file across partitions/devices
		function(nextTask){
			mv(renInfo.oldName, renInfo.newName, function(err){
			//!!!fs.rename(renInfo.oldName, renInfo.newName, function(err){
				if(!err) logger.debug('--->_rename: ' + renInfo.oldName +' renamed as ' + renInfo.newName);
				nextTask(err);
			});

		}

	],	function(err){
			callback(err);
		}
	);
}

/**
 * mv: copied from https://github.com/andrewrk/node-mv
 * @param  {String}   source 	source file
 * @param  {String}   dest 		destination file
 * @param  {JSON}     options 	see https://github.com/andrewrk/node-mv
 * @param  {Function} callback(err)      
 */
function mv(source, dest, options, callback){
  if(typeof options === 'function'){
    callback = options;
    options = {};
  }
  var clobber = options.clobber !== false;
  var limit = options.limit || 16;

	if(clobber){
		fs.rename(source, dest, function(err){
			if (!err) return callback();
			if (err.code !== 'EXDEV') return callback(err);
			moveFileAcrossDevice(source, dest, clobber, limit, callback);
		});
	} 
	else{
		fs.link(source, dest, function(err){
			if(err){
				if(err.code === 'EXDEV'){
					moveFileAcrossDevice(source, dest, clobber, limit, callback);
					return;
				}
				if(err.code === 'EISDIR' || err.code === 'EPERM'){
					//!!!moveDirAcrossDevice(source, dest, clobber, limit, callback);
					callback(err);//!!! instead of calling moveDirAcrossDevice
					return;
				}
				callback(err);
				return;
			}
			fs.unlink(source, callback);
		});
	}  
}

/**
 * moveFileAcrossDevice: copied from https://github.com/andrewrk/node-mv
 * @param  {String}   source 	source file
 * @param  {String}   dest 		destination file
 * @param  {JSON}     options 	see https://github.com/andrewrk/node-mv
 * @param  {[type]}   clobber 	see https://github.com/andrewrk/node-mv
 * @param  {[type]}   limit 	see https://github.com/andrewrk/node-mv
 * @param  {Function} callback(err)      
 */
function moveFileAcrossDevice(source, dest, clobber, limit, callback){
	var outFlags = clobber ? 'w' : 'wx';
	var ins = fs.createReadStream(source);
	var outs = fs.createWriteStream(dest, {flags: outFlags});
	ins.on('error', function(err){
		ins.destroy();
		outs.destroy();
		outs.removeListener('close', onClose);
		if (err.code === 'EISDIR' || err.code === 'EPERM') {
			//!!!moveDirAcrossDevice(source, dest, clobber, limit, callback);
			callback(err);//!!! instead of calling moveDirAcrossDevice
		} 
		else{
			callback(err);
		}
	});
	outs.on('error', function(err){
		ins.destroy();
		outs.destroy();
		outs.removeListener('close', onClose);
		callback(err);
	});
	outs.once('close', onClose);
	ins.pipe(outs);
	function onClose(){
		fs.unlink(source, callback);
	}
}

module.exports = cache;