/**-----------------------------------------------------------------------------
 * index.js: 	Node.js module to manage cron jobs
 *
 * Author    :  AFP2web Team
 * Copyright :  (C) 2015 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V1.0.0
 * 
 * History
 *  V100        07.05.2015     Initial release
 *  V101        11.05.2015     Added cron-job-manager
 *  
 *----------------------------------------------------------------------------*/
'use strict';

var MODULE_NAME 	= 'cronjob'
  , MODULE_VERSION 	= '2.0.0'
  , log4js 			= require('log4js')

  , npsServer		= require('./server')
  , npsDir	    	= npsServer.npsDir
  , npsConf 		= npsServer.npsConf					// holds the configuration of the OTS Server   
  , npsLogFile 		= npsServer.npsLogFile  
  , utils 			= require('./helpers/utils')
  , CronJobManager 	= require('cron-job-manager')
  , cronJobManager 	= new CronJobManager()
  ;

// Get Logger
var logger = log4js.getLogger(MODULE_NAME);
logger.setLevel(npsConf.log.level);

// Export Module Version
exports.version = MODULE_VERSION;

/**************** PUBLIC FUNCTIONS ****************/
/**
 * initialize: 			Initialize Module
 * @param  {function} 	callback(err)
 */
exports.initialize = function(callback){

	var self = this;
	
	if(self._initialized){
		return callback();
	}
	self._initialized = true;
	callback();
}

/**
 * finalize: 			Finalize Module
 * @param  {function} 	callback(err) 
 */
exports.finalize = function(callback){

	var self = this;

	if(!self._initialized){
		callback();
	}

	// Stop all cron jobs
	exports.stopAllCronJobService(function(err){
		if(err){
			return callback(err);
		}
		self._initialized = false;
		callback();
	});
}

/**
 * version: 			Process a HTTP Version Request
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.version = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);
	logger.info('--->version: Req. id=' + res._Context.reqId);

	var data = '<h3>' +  MODULE_NAME + ' v' + MODULE_VERSION + '</h3>';
	res.end(data);
	
	logger.info('--->version: Req. id=' + res._Context.reqId + ', sent back ' +
				MODULE_NAME + ' v' + MODULE_VERSION + ', over');
	return;
}

/**
 * add:  					Add a cron job that will run at the specified cronTime (ONLY FOR TESTING)
 * @param  {req} 			req 
 *         					req.body:{
 *         						name: 			name of the cron job
 *         						cronTime: 		Time to run cron job as defined in http://crontab.org/
 *         		                cronFunction: 	function to call
 *                           	[cronOptions:] 	extra options; see https://github.com/ncb000gt/node-cron/blob/master/README.md for all available
 *                            	[parms:] 		optional: function parms --> NOT IMPLEMENTED !!!
 *                          }
 * @param  {res} 			res 
 */
exports.add = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);
	logger.info('--->add: Req. id=' + res._Context.reqId);
	
	if(req.body.name === undefined || req.body.cronTime === undefined || req.body.cronFunction === undefined ){
		logger.error(__filename, '--->add: Req. id=' + res._Context.reqId + ', Wrong request: ' + JSON.stringify(req.body));
		return res.status(404).end('cronjob.add: Wrong request: ' + JSON.stringify(req.body));
	}

	var options = {
		name: 			req.body.name,
		cronTime: 		req.body.cronTime,
		cronFunction: 	exports.someTestFunction, //!!! FOR TESTING ONLY !!! req.body.cronFunction,
		cronOptions:	req.body.cronOptions || undefined
	}
 
	exports.addCronJobService(options, function(err){
		if(err){
			logger.error(__filename, '--->add: Req. id=' + res._Context.reqId + ', Error: ' + err.message);
			return res.status(404).end('--->add: Req. id=' + res._Context.reqId + ', Error: ' + err.message);
		}
		else{
			logger.info('--->add: Req. id=' + res._Context.reqId + ', over');
			return res.end('--->add: Req. id=' + res._Context.reqId +  ', over');
		}
	});
}


/**
 * addCronJobService: 		Add a cron job that will run at the specified cronTime
 * @param  {[type]}   		options 
 *         					options:{
 *         						name: 				name of the cron job
 *         						cronTime: 			Time to run cron job as defined in http://crontab.org/
 *         						cronFunction: 		function to call
 *         						[cronOptions:] 		extra options; see https://github.com/ncb000gt/node-cron/blob/master/README.md for all available
 *         					 	[parms:] 			optional: function parms --> NOT IMPLEMENTED !!!
 *         					}
 * @param  {Function} 		cb(err)
 */
exports.addCronJobService = function(options, cb){

	logger.debug(__filename, '--->addCronJobService: Adding Job: ' + JSON.stringify(options) + '...');

	try{

		cronJobManager.add(
			options.name,
			options.cronTime,
			options.cronFunction,
			options.cronOptions || null
		);
		return cb();
	}catch(err){
		return cb(err);
	}
}

/**
 * !!! NOT IMPLEMENTED YET
 * update: 					Update the specified cron job
 *         					req.body:{
 *         						name: 			name of the cron job
 *         						cronTime: 		Time to run cron job as defined in http://crontab.org/
 *         		                cronFunction: 	function to call
 *                           	[cronOptions:] 	extra options; see https://github.com/ncb000gt/node-cron/blob/master/README.md for all available
 *                            	[parms:] 		optional: function parms --> NOT IMPLEMENTED !!!
 *                          }
 * @param  {res}			res
 */
exports.update = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);
	logger.info('--->update: Req. id=' + res._Context.reqId);
	res.json({'msg':'not implemented yet'});
	logger.info('--->update: Req. id=' + res._Context.reqId + ', over');
	return;
}

/**
 * delete: 				Delete the specified cron job
 * @param  {req} 		req
 *         				req.body:{
 *         					name: name of the cron job to be deleted
 *         				}
 * @param  {res}		res
 */
exports.delete = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	logger.info('--->delete: Req. id=' + res._Context.reqId);
	logger.debug(__filename, '--->delete: Req. id=' + res._Context.reqId + ', Parms: ' + JSON.stringify(req.body));

	// 1. Assert req.body
	if(req.body.name === undefined){
		logger.error(__filename, '--->delete: Req. id=' + res._Context.reqId + ', Wrong request: ' + JSON.stringify(req.body));
		return res.status(404).end('cronjob.delete: Wrong request: ' + JSON.stringify(req.body));
	}
	
	// 2. Call the appropriate service
	exports.deleteCronJobService(req.body.name, function(err){
		if(err){
			logger.error(__filename, '--->delete: Req. id=' + res._Context.reqId + ', Request failed: ' + err.message + '. Parms:' + JSON.stringify(req.body));
			return res.status(404).end('Stop job request failed: ' + err.message + '. Parms:' + JSON.stringify(req.body));
		}
		logger.info('--->delete: Req. id=' + res._Context.reqId + ', Job ' + req.body.name + ' has been deleted');
		return res.end('Job ' + req.body.name + ' has been deleted');			
	});
}

/**
 * deleteCronJobService: 	Delete the specified cron job
 * @param  {string}   		name: name of the cron job to be deleted
 * @param  {Function} 		cb(err)
 */
exports.deleteCronJobService = function(name, cb){

	logger.debug(__filename, '--->deleteCronJobService: Deleting Job ' + name + '...');

	try{
		cronJobManager.deleteJob(name);
		return cb();
	}catch(err){
		return cb(err);
	}
}

/**
 * start: 				Start the specified cron job
 * @param  {req} 		req
 *         				req.body:{
 *         					name: name of the cron job to be started
 *         				}
 * @param  {res}		res
 */
exports.start = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	logger.info('--->start: Req. id=' + res._Context.reqId);
	logger.debug(__filename, '--->start: Req. id=' + res._Context.reqId + ', Parms: ' + JSON.stringify(req.body));

	// 1. Assert req.body
	if(req.body.name === undefined){
		logger.error(__filename, '--->start: Req. id=' + res._Context.reqId + ', Wrong request: ' + JSON.stringify(req.body));
		return res.status(404).end('cronjob.start: Wrong request: ' + JSON.stringify(req.body));
	}
	
	// 2. Call the appropriate service
	exports.startCronJobService(req.body.name, function(err){
		if(err){
			logger.error(__filename, '--->start: Req. id=' + res._Context.reqId + ', Request failed: ' + err.message + '. Parms:' + JSON.stringify(req.body));
			return res.status(404).end('Start job request failed: ' + err.message + '. Parms:' + JSON.stringify(req.body));
		}
		logger.info('--->start: Req. id=' + res._Context.reqId + ', Job ' + req.body.name + ' has been started');
		return res.end('Job ' + req.body.name + ' has been started');			
	});
}

/**
 * startCronJobService: 	Start the specified cron job
 * @param  {string}   		name: name of the cron job to be started
 * @param  {Function} 		cb(err)
 */
exports.startCronJobService = function(name, cb){

	logger.debug(__filename, '--->startCronJobService: Starting Job ' + name + '...');

	try{
		cronJobManager.start(name);
		return cb();
	}catch(err){
		return cb(err);
	}
}

/**
 * stop: 				Stop the specified cron job
 * @param  {req} 		req
 *         				req.body:{
 *         					name: name of the cron job to be stopped
 *         				}
 * @param  {res}		res
 */
exports.stop = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	logger.info('--->stop: Req. id=' + res._Context.reqId);
	logger.debug(__filename, '--->stop: Req. id=' + res._Context.reqId + ', Parms: ' + JSON.stringify(req.body));

	// 1. Assert req.body
	if(req.body.name === undefined){
		logger.error(__filename, '--->stop: Req. id=' + res._Context.reqId + ', Wrong request: ' + JSON.stringify(req.body));
		return res.status(404).end('cronjob.stop: Wrong request: ' + JSON.stringify(req.body));
	}
	
	// 2. Call the appropriate service
	exports.stopCronJobService(req.body.name, function(err){
		if(err){
			logger.error(__filename, '--->stop: Req. id=' + res._Context.reqId + ', Request failed: ' + err.message + '. Parms:' + JSON.stringify(req.body));
			return res.status(404).end('Stop job request failed: ' + err.message + '. Parms:' + JSON.stringify(req.body));
		}
		logger.info('--->stop: Req. id=' + res._Context.reqId + ', Job ' + req.body.name + ' has been stopped');
		return res.end('Job ' + req.body.name + ' has been stopped');			
	});
}

/**
 * stopCronJobService: 		Stop the specified cron job
 * @param  {string}   		name: name of the cron job to be stopped
 * @param  {Function} 		cb(err)
 */
exports.stopCronJobService = function(name, cb){

	logger.debug(__filename, '--->stopCronJobService: Stopping Job ' + name + '...');

	try{
		cronJobManager.stop(name);
		return cb();
	}catch(err){
		return cb(err);
	}
}

/**
 * stopAll: 			Stop all the cron jobs
 * @param  {req} 		req
 * @param  {res}		res
 */
exports.stopAll = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	logger.info('--->stopAll: Req. id=' + res._Context.reqId);
	logger.debug(__filename, '--->stopAll: Req. id=' + res._Context.reqId + ', Parms: ' + JSON.stringify(req.body));

	// Call the appropriate service
	exports.stopAllCronJobService(function(err){
		if(err){
			logger.error(__filename, '--->stopAll: Req. id=' + res._Context.reqId + ', Request failed: ' + err.message + '. Parms:' + JSON.stringify(req.body));
			return res.status(404).end('Stopping jobs request failed: ' + err.message + '. Parms:' + JSON.stringify(req.body));
		}
		logger.info('--->stopAll: Req. id=' + res._Context.reqId + ', All the Jobs have been stopped');
		return res.end('All the Jobs have been stopped');
	});
}

/**
 * stopAllCronJobService: 	Stop all the cron jobs
 * @param  {Function} 		cb(err)
 */
exports.stopAllCronJobService = function(cb){

	logger.debug(__filename, '--->stopAllCronJobService: Stopping Jobs...');

	try{
		cronJobManager.stopAll();
		return cb();
	}catch(err){
		return cb(err);
	}
}

/**
 * list: 				List all the cron jobs
 * @param  {req} 		req
 * @param  {res}		res
 */
exports.list = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	logger.info('--->list: Req. id=' + res._Context.reqId);
	logger.debug(__filename, '--->list: Req. id=' + res._Context.reqId + ', Parms: ' + JSON.stringify(req.body));

	// Call the appropriate service
	exports.listCronJobService(function(err, result){
		if(err){
			logger.error(__filename, '--->list: Req. id=' + res._Context.reqId + ', Request failed: ' + err.message + '. Parms:' + JSON.stringify(req.body));
			return res.status(404).end('Listing jobs request failed: ' + err.message + '. Parms:' + JSON.stringify(req.body));
		}
		logger.info('--->list: Req. id=' + res._Context.reqId + ', Jobs have been listed');
		return res.json(result);
	});
}

/**
 * listCronJobService: 		List all the cron jobs
 * @param  {Function} 		cb(err, result)
 */
exports.listCronJobService = function(cb){

	logger.debug(__filename, '--->listCronJobService: Listing Jobs...');

	try{
//!!!		var result = cronJobManager.listCrons(); //!!! SHOULD return a JSON object as [{name:<name>,crontime:<crontime>,running:<true|false>},{},...]
//		logger.debug(__filename, '--->listCronJobService: ' + result);
//		return cb(null, result);
		_list(cb);
	}catch(err){
		return cb(err);
	}
}

/**
 * exists: 				Check whether the specified cron job exists
 * @param  {req} 		req
 *         				req.body:{
 *         					name: name of the cron job to check
 *         				}
 * @param  {res}		res
 */
exports.exists = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	logger.info('--->exists: Req. id=' + res._Context.reqId);
	logger.debug(__filename, '--->exists: Req. id=' + res._Context.reqId + ', Parms: ' + JSON.stringify(req.body));

	// 1. Assert req.body
	if(req.body.name === undefined){
		logger.error(__filename, '--->exists: Req. id=' + res._Context.reqId + ', Wrong request: ' + JSON.stringify(req.body));
		return res.status(404).end('cronjob.exists: Wrong request: ' + JSON.stringify(req.body));
	}
	
	// 2. Call the appropriate service
	exports.existsCronJobService(req.body.name, function(err, result){
		if(err){
			logger.error(__filename, '--->exists: Req. id=' + res._Context.reqId + ', Request failed: ' + err.message + '. Parms:' + JSON.stringify(req.body));
			return res.status(404).json({'req': res._Context.reqId, 'error': err, 'parms': JSON.stringify(req.body)});
		}
		logger.info('--->exists: Req. id=' + res._Context.reqId + ', Job ' + req.body.name + ' has been checked');
		return res.json({'exists': result});
	});
}

/**
 * existsCronJobService: 	Check whether the specified cron job exists
 * @param  {string}   		name: name of the cron job to check
 * @param  {Function} 		cb(err,result)
 */
exports.existsCronJobService = function(name, cb){

	logger.debug(__filename, '--->existsCronJobService: Checking for Job ' + name + '...');

	try{
		var result = cronJobManager.exists(name);
		return cb(null, result);
	}catch(err){
		return cb(err);
	}
}

/**
 * Set the log level of this module
 * @param  {req} 	GET req   http://localhost:1026/appservices/jobs/setloglevel?loglevel=DEBUG|INFO|ERROR
 * @param  {res}	res
 */
exports.setLogLevel = function(req, res){
	var loglevel = req.query.loglevel || 'INFO';
	logger.setLevel(loglevel);
	return res.end('--->setLogLevel: Log level of ' + MODULE_NAME + ' set to ' + loglevel);
}

/**
 * Return the log level of this module
 * @param  {req} 	GET req   http://localhost:1026/appservices/jobs/getloglevel
 * @param  {res}	res
 */
exports.getLogLevel = function(req, res){
	logger.info('--->getLogLevel: Getting the log level of ' + MODULE_NAME + '...');
	return res.end(logger.level.levelStr);
}

/**
 * someTestFunction: 	Dummy function to test addCronJobService
 * @param  {JSON} 		options 
 */
exports.someTestFunction = function(options){
	logger.debug(__filename, '--->someTestFunction: Triggered by the cronjob service at ' + new Date() + ', options: ' + JSON.stringify(options));
}


/******************  Private Functions ********************/
function _list(cb){

	var jobList = []
	  , aJob = undefined
	  , jobs = cronJobManager.jobs
	  ;
	var manString = "";
	for (var job in jobs) {
		aJob = {
			'name': 	job,
			'function': jobs[job].context.function || job,
			'cronTime': jobs[job].cronTime.source,
			'timeZone': jobs[job].cronTime.zone,
			'running':  jobs[job].running
		}
		jobList.push(aJob)
	}                                                               
	logger.debug(__filename, '--->_list: ' + JSON.stringify(jobList));
	cb(null, jobList);
}