/**-----------------------------------------------------------------------------
 * deploy.js: 	Node.js module to deploy Server Releases
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2014 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V1.0.0
 * 
 * History
 *  V100   12.09.2014  Initial release
 *  V101   21.05.2015  Moved the backup function to backup.js
 *  				   Uses multer instead of jquery-file-upload-middleware
 *
 *----------------------------------------------------------------------------*/
'use strict';

var MODULE_NAME 	= 'deploy'
  , MODULE_VERSION 	= '2.0.0'
  , fs 				= require('fs')
  , path 			= require('path')
  , log4js 			= require('log4js') 
  ,	async 			= require('async')
  , multer  		= require('multer')
  , npsServer		= require('./server')
  , npsConf 		= npsServer.npsConf					// holds the configuration of the Server 
  , npsDir  		= npsServer.npsDir
  , npsConfDir 		= npsServer.npsConfDir
  , npsLogFile 		= npsServer.npsLogFile
  , dateFormat 		= require('./helpers/date_format')
  , utils 			= require('./helpers/utils')
  , backup 			= require('./backup')
  ;

// Get Logger
var logger = log4js.getLogger(MODULE_NAME);
logger.setLevel(npsConf.logLevel || 'INFO');

// Export Module Version
exports.version = MODULE_VERSION;

// Configure multer
npsServer.expressApp.use(
	multer({
		'dest': path.resolve(npsConf.uploadDir || npsServer.npsDir + '/uploads/'),
		'putSingleFilesInArray': true,

		// Multer limits option, see https://github.com/expressjs/multer
		'limits': {'files':1, /*'fileSize': no limit*/},

		rename: function (fieldname, filename) {
			logger.debug(__filename, '-->Renaming ' + filename + '...');
			return filename + '_' + utils.buildReqId(new Date());
		},
		onFileUploadStart: function (file) {
			console.log('-->Starting uploading ' + file.originalname + '...')
		},
		onFileUploadData: function (file, data, req, res) {
			console.log('-->' + file.fieldname + ': ' + data.length + ' bytes received.')
		},
		onFileUploadComplete: function (file) {
			logger.debug(__filename, '-->' + file.originalname + ' uploaded to ' + file.path);
		}
	}).any()
);

/**
 * Display the Upload form
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.displayform = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the HTML file to serve to the Context
	res._Context.htmlFile = '/public/deploy/index.html';

	logger.info('--->displayform: Req. id=' + res._Context.reqId);
	_serveHTML(res, '');
	logger.info('--->displayform: Req. id=' + res._Context.reqId + ', sent back, over');
	return;
}

/**
 * Process an uploaded Package
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.process = function(req, res) {

	logger.info('--->Dep. Req.: Deploy Release Request received');

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add backup to the Context
	res._Context.backup = false; 	//!!! Should come from the deploy form

	// Add backup options to the Context
	res._Context.options = {		//!!! Should come from the deploy form
		nodemodulesDir: 	false,	// backup the node_modules dir
		testDir: 			false,	// backup the test dir
	}

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);
	logger.info('--->process: Req. id=' + res._Context.reqId);

	if (req.files) { 
		logger.debug(__filename, '-->process: Req. id=' + res._Context.reqId + 
					', Uploaded Files:' + JSON.stringify(req.files.file)); /* --> Outputs an array of objects as:

		file: [
			{
				"fieldname":"file",
				"originalname":"tobeuploaded.zip",
				"name":"tobeuploaded_150521164025925-22642.zip",
				"encoding":"7bit",
				"mimetype":"application/zip",
				"path":"/opt/OXSNPS-OTS@OXSEEDR6/uploads/tobeuploaded_150521164025925-22642.zip",
				"extension":"zip",
				"size":185612,
				"truncated":false,
				"buffer":null
			}
		]
		*/

	async.series([
		// Task 1: Backup the current version
		function(nextTask){
			if(res._Context.backup){
				// First backup the current OTS version
				backup.backupService(res._Context.options, nextTask);
			}
			else{
				nextTask();
			}
		},
		// Task 2: Deploy the new version
		function(nextTask){
			_deploy(req.files.file[0], nextTask);
		},
	],	function(err){
			if(err){
				logger.error(__filename, '--->Dep. Req.: Not able to backup current installation or to deploy the new one, Reason: ' +  err.message);
				return res.status(404).end('--->Dep. Req.: Not able to backup current installation or to deploy the new one, Reason: ' +  err.message);
			}
			logger.info('--->process: Req. id=' + res._Context.reqId + ', over');
			return res.json(req.files.file);	//!!!
		}
	);		
	} 
	else{
		logger.error(__filename, '--->process: Req. id=' + res._Context.reqId + ', Error: no uploaded files.');
		return req.status(404).end('--->process: Req. id=' + res._Context.reqId + ', Error: no uploaded files.');
	}
}

/**
 * List the Packages that have been deployed so far
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.list = function(req, res){
	res.json({});
}

/**
 * Set the log level of this module
 * @param  {req} 	GET req   http://localhost:1026/appservices/deploy/setloglevel?loglevel=DEBUG|INFO|ERROR
 * @param  {res}	res
 */
exports.setLogLevel = function(req, res){
	var loglevel = req.query.loglevel || 'INFO';
	logger.setLevel(loglevel);
	return res.end('--->setLogLevel: Log level of ' + MODULE_NAME + ' set to ' + loglevel);
}

/**
 * Return the log level of this module
 * @param  {req} 	GET req   http://localhost:1026/appservices/deploy/getloglevel
 * @param  {res}	res
 */
exports.getLogLevel = function(req, res){
	logger.info('--->getLogLevel: Getting the log level of ' + MODULE_NAME + '...');
	return res.end(logger.level.levelStr);
}

/******************  Private Functions ********************/
/**
 * _deploy: 				
 * @param  {JSON Array}   	fileInfo 
 *                         	fileInfo: [
 *                         		{
 *                         			"fieldname":"file",
 *                         			"originalname":"tobeuploaded.zip",
 *                         			"name":"tobeuploaded_150521164025925-22642.zip",
 *                         			"encoding":"7bit",
 *                         			"mimetype":"application/zip",
 *                         			"path":"/opt/OXSNPS-OTS@OXSEEDR6/uploads/tobeuploaded_150521164025925-22642.zip",
 *                         			"extension":"zip",
 *                         			"size":185612,
 *                         			"truncated":false,
 *                         			"buffer":null
 *                         		}		
 *                         	] 
 * @param  {Function} 		cb(err)
 */
function _deploy(fileInfo, cb){

	// Untar/Unzip the file
	logger.debug(__filename, '_deploy: Trying to deploy ' + fileInfo.path + ' to ' + npsDir + ' ...');

	if (fileInfo.mimetype === 'application/x-compressed-tar' || 	// Google Chrome
		fileInfo.mimetype === 'application/x-gzip'	||				// Google Chrome
		fileInfo.mimetype === 'application/x-compressed'	||		// !!! Google Chrome sends this type for tgz file
		fileInfo.mimetype === 'application/x-gzip-compressed'		//  IE 9-11
	){  // must be a .tgz or a tar.gz file
		var targz = require('tar.gz');
		var compress = new targz().extract(fileInfo.path, npsDir, function(err){
			if(err){
				return cb(err);
			}
			// Exit the process in a few secs. Forever will restart it 
			logger.info('--->Dep. Req.: Deploy Release Request processed, over');
			logger.info('--->Restarting the OXSNPS Server in 3 seconds...');
			setTimeout(function(){process.exit(0);}, 3000);	// restart OTS
		});
	}
	else if (fileInfo.mimetype === 'application/zip' ||				// Google Chrome
			 fileInfo.mimetype === 'application/x-zip-compressed'	//  IE 9-11
		 ){  // must be a .zip file
		try{
			var admZip	= require('adm-zip');
			var zip = new admZip(fileInfo.path);
			zip.extractAllTo(npsDir, /*overwrite*/true);

			// Exit the process in a few secs. Forever will restart it 
			logger.info('--->Dep. Req.: Deploy Release Request processed, over');
			logger.info('--->Restarting the OXSNPS Server in 3 seconds...');
			setTimeout(function(){process.exit(0);}, 3000);	// restart OTS
		}	  
		catch(err){
			return cb(err);
		}
	}
	else{
		cb(new Error('Incorrect Release Package (' + fileInfo.originalName + 
					 '). Wrong type:' + fileInfo.mimetype +
					 '. Expects a .zip (type:application/zip), ' +
					 'a .tgz (type;application/x-compressed-tar) file ' + 
					 'or a tar.gz (type;application/x-gzip) file'));
	}
}

function _serveHTML(res, body){

	// Read the HTML file, set some nps... vars, and send back the modified HTML
	fs.readFile(npsDir + res._Context.htmlFile, 'utf8', function(err, html){
		if(err){
			return res.status(404).end(npsDir + res._Context.htmlFile + ' not found!' );
		}
		html = html.replace(/&npsVersion;/g, npsServer.npsVersion);	// replace version
		html = html.replace(/&npsName;/g, npsServer.npsDesc);		// replace description
		html = html.replace(/&npsBody;/g, body);					// replace body
		html = html.replace(/&npsPublicDir;/g, npsDir + '/public');					// replace body
		res.end(html);
	});
}
