/**-----------------------------------------------------------------------------
 * error.js: 	oxsnps error class 
 *
 * Author    :  AFP2web Team
 * Copyright :  (C) 2015 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V1.0.0
 * 
 * History
 *  V100        07.05.2015     Initial release
 *  
 *----------------------------------------------------------------------------*/
'use strict';

var MODULE_NAME 	= 'error'
  , MODULE_VERSION 	= '1.0.0'
  , log4js 			= require('log4js')
  , util 			= require('util')
  , npsServer		= require('./server')
  , npsDir	    	= npsServer.npsDir
  , npsConf 		= npsServer.npsConf	// holds the configuration of the OTS Server   
  , npsConfDir  	= npsDir + '/conf'
  , npsLogFile 		= npsServer.npsLogFile  
  , utils 			= require('./helpers/utils')
  , supplant 		= require('supplant')
  , subs 			= new supplant()
  , errorObjs 		= {}
  ;

// Get Logger
var logger = log4js.getLogger(MODULE_NAME);
logger.setLevel(npsConf.logLevel || 'INFO');

// Export Module Version
exports.version = MODULE_VERSION;

module.exports=error;

/**************** PUBLIC FUNCTIONS ****************/
/**
 * initialize: 			Initialize Module
 * @param  {function} 	callback(err)
 */
module.exports.initialize = function(callback){

	var self = this;
	
	if(self._initialized){
		return callback();
	}
	self._initialized = true;
	callback();
}

/**
 * finalize: 			Finalize Module
 * @param  {function} 	callback(err) 
 */
module.exports.finalize = function(callback){

	var self = this;

	if(!self._initialized){
		callback();
	}
	self._initialized = false;
	callback();
}

/**
 * version: 			Process a HTTP Version Request
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
module.exports.version = function(req, res){
	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);
	logger.info('--->version: Req. id=' + res._Context.reqId);

	var data = '<h3>' +  NAME + ' v' + VERSION + '</h3>';
	res.end(data);
	
	logger.info('--->version: Req. id=' + res._Context.reqId + ', Version sent back, over');

	return;
}

/**
 * Set the log level of this module
 * @param  {req} 	GET req   http://localhost:1026/appservices/backup/setloglevel?loglevel=DEBUG|INFO|ERROR
 * @param  {res}	res
 */
module.exports.setLogLevel = function(req, res){
	var loglevel = req.query.loglevel || 'INFO';
	logger.setLevel(loglevel);
	return res.end('--->setLogLevel: Log level of ' + NAME + ' set to ' + loglevel);
}

/**
 * Return the log level of this module
 * @param  {req} 	GET req   http://localhost:1026/appservices/backup/getloglevel
 * @param  {res}	res
 */
module.exports.getLogLevel = function(req, res){
	logger.info('--->getLogLevel: Getting the log level of ' + NAME + '...');
	return res.end(logger.level.levelStr);
}

/**
 * Error constructor
 * @param  {JSON} errDetails            	Error details json
 *                                       	{
 *                                       		'code': 	<code>,
 *                                       		'module': 	<Module name>,
 *                                       		'func': 	<function name>,
 *                                       		'parms': 	<Parameter values to replace in error message defined in error.json>
 *                                       	}
 * @param  {Object} implementationContext   Optional. If given, all frames above implementationContext, 
 *                                          including implementationContext, will be omitted from the generated stack trace. 
 * @return {error}                       	returns error object
 *
 * Refer to http://www.bennadel.com/blog/2828-creating-custom-error-objects-in-node-js-with-error-capturestacktrace.htm
 */  
function error(errDetails, implementationContext){

	var msg 		= ''
	  , modErrors 	= undefined
	  ;

    // Ensure that errDetails exists to prevent refernce errors.
    errDetails = (errDetails || {});

    // Set Properties
    this.code 		= (errDetails.code || '' );
	this.module 	= (errDetails.module || '');    
	this.func 		= (errDetails.func || '');
	this.isAppError = true;
	this.parms 		= (errDetails.parms || {});

	// Get error object of a module
	modErrors = errorObjs[this.module];
	if(modErrors){
    	if(!modErrors[this.code]) 
    		this.message = 'Error code "' + this.code + '" not found for module ' + this.module;
    	else if(!modErrors[this.code].msg)
    		this.message = 'Error messsage not defined for error code "' + this.code + '" for module ' + this.module;
    	else
			this.message = subs.text(modErrors[this.code].msg,this.parms);
	}
	else{
		this.message = 'Error messages not found for ' + this.module;
	}

 	// Capture the current stacktrace and store it in the property "this.stack". By
    // providing the implementationContext argument, we will remove the current
    // constructor (or the optional factory function) line-item from the stacktrace; this
    // is good because it will reduce the implementation noise in the stack property.
    // --
    // Rad More: https://code.google.com/p/v8-wiki/wiki/JavaScriptStackTraceApi#Stack_trace_collection_for_custom_exceptions
    Error.captureStackTrace(this, (implementationContext || error));
};

/**
 * Factory function to create error object
 * @param  {JSON} errDetails            	Error details json
 *                                       	{
 *                                       		'code': 	<code>,
 *                                       		'module': 	<Module name>,
 *                                       		'func': 	<function name>,
 *                                       		'parms': 	<Parameter values to replace in error message defined in error.json>
 *                                       	}
 *                                          including implementationContext, will be omitted from the generated stack trace. 
 * @return {error}                       	returns error object
 *
 */
module.exports.createError = function(errDetails){
    // NOTE: We are overriding the "implementationContext" so that the createAppError()
    // function is not part of the resulting stacktrace.
    return(new error(errDetails, exports.createError));

}

/**
 * Register plugin/module error objects with error class
 * @param  {String} moduleName 	Module name
 * @param  {JSON} 	errorObj   	Plugin/Module JSON error object
 */
module.exports.registerErrorObj = function(moduleName, errorObj){
	if(moduleName && moduleName.length >0){
		errorObj = errorObj || {};
		errorObjs[moduleName] = errorObj;
	}
    return;
}

/******************  Private Functions ********************/

// Inherit from node.js Error class
util.inherits(error, Error);
