/**-----------------------------------------------------------------------------
 * bufferStream.js: Node.js module that provides buffer streaming
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2014 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V1.0.0
 * 
 * History
 *  V100        14.02.2017     Initial release
 *
 * Courtesy:
 * https://www.bennadel.com/blog/2681-turning-buffers-into-readable-streams-in-node-js.htm
 *----------------------------------------------------------------------------*/
'use strict';

var MODULE_NAME 	= 'bufferStream'
  , MODULE_VERSION 	= '1.0.0'
  , fs          	= require('fs')
  , stream 			= require( "stream" )
  , util 			= require( "util" )
  , log4js	    	= require('log4js')  
  , npsServer   	= require('../server')
  , npsDir	    	= npsServer.npsDir
  , npsConf 		= npsServer.npsConf	// holds the configuration of the OTS Server   
  , npsLogFile 		= npsServer.npsLogFile  
  ;

// Get Logger
var logger = log4js.getLogger(MODULE_NAME);
logger.setLevel(npsConf.log.level);

// Export Module Version
exports.version = MODULE_VERSION;

/**
 * Constructor
 * 
 */
// Turn the given source Buffer into a Readable stream.
function bufferStream(source){

    if(!Buffer.isBuffer(source)){
        throw( new Error( "Source must be a buffer." ) );
    }

    // Super constructor.
    stream.Readable.call( this );

    this._source = source;

    // I keep track of which portion of the source buffer is currently being pushed
    // onto the internal stream buffer during read actions.
    this._offset = 0;
    this._length = source.length;

    // When the stream has ended, try to clean up the memory references.
    this.on( "end", this._destroy.bind(this) );

}

util.inherits(bufferStream, stream.Readable);


// I attempt to clean up variable references once the stream has been ended.
// --
// NOTE: I am not sure this is necessary. But, I'm trying to be more cognizant of memory
// usage since my Node.js apps will (eventually) never restart.
bufferStream.prototype._destroy = function() {
    this._source = null;
    this._offset = null;
    this._length = null;
};


// I read chunks from the source buffer into the underlying stream buffer.
// --
// NOTE: We can assume the size value will always be available since we are not
// altering the readable state options when initializing the Readable stream.
bufferStream.prototype._read = function(size){

    // If we haven't reached the end of the source buffer, push the next chunk onto
    // the internal stream buffer.
    if(this._offset < this._length){
        this.push(this._source.slice(this._offset, (this._offset + size)));
        this._offset += size;
    }

    // If we've consumed the entire source buffer, close the readable stream.
    if(this._offset >= this._length){
        this.push( null );
    }
};

module.exports = bufferStream;