/**-----------------------------------------------------------------------------
 * server.js :  Node.js module that creates an application server
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2014 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V2.0.X
 * 
 * History
 *  V2.0.35     01.09.2025  AFP-1259: Updated httpUtils.js with the httpUtils.js of oxsnps-core V5.0.x
 *  V2.0.34     06.01.2025 AFP-1228: server.js Extended to use express middileware instead of body-parser to fix vulnerability issues related with body-parser module. 
 *                                   pluginManager.js: _intializePlugin(): Extended to support base plugins adopted to oxsnps-core 5.x
 *  V2.0.33     20.06.2023 AFP-1156: Fix vulnerability issues. Switched modules to the following version as given below
 *                          		"moment": "2.29.4",
 *                                  "multer": "1.4.5-lts.1",
 *                                  "supplant": "0.2.0-1-oxs-extn",
 *  V100        25.08.2014  Initial release
 *
 *----------------------------------------------------------------------------*/
'use strict';

var path        = require('path')
  , npsDir      = path.resolve(__dirname + '../../../')
  , packageJson = require(npsDir + '/package.json')
  , npsName     = packageJson.name || 'OXSNPS Server'
  , npsVersion  = packageJson.version || 'unknown'  // Get Server Version from package.json
  , npsDesc     = packageJson.description || 'OXSNPS Server'
  , express     = require('express')

  , favicon = require('serve-favicon')
  //, bodyParser  = require('body-parser')  // V2.0.34
  //, method_override = require('method-override') //!!! DON'T USE IT 
  , serve_index = require('serve-index')

  , log4js      = require('log4js')
  , nodeCache   = require('node-cache')  
  , npsConfDir  = npsDir + '/conf'
  , npsConf     = require(npsConfDir + '/server-conf.js')
  , npsPort     = npsConf.port || process.env.PORT || 3000
  , mode        = parseInt('0777',8)
  , utils       = require('./helpers/utils')
  , app         = express()
  , server      = undefined 
  ;

// Create log dir if not existing
npsConf.logDir = npsConf.logDir || '/var/log/nps';
utils.mkDirSync(npsConf.logDir, mode);

// Create Temp dir if not existing
npsConf.tempDir = path.resolve((npsConf.tempDir?npsConf.tempDir:'/tmp/nps'));
utils.mkDirSync(npsConf.tempDir, mode);

// Create Backup dir if not existing
npsConf.backupDir = path.resolve((npsConf.backupDir?npsConf.backupDir:'./backups'));
utils.mkDirSync(npsConf.backupDir, mode);

// Create Cache dir if not existing but required
if(npsConf.cache){
  npsConf.cache.dir = (npsConf.cache.dir?npsConf.cache.dir:'/var/cache/nps');
  utils.mkDirSync(npsConf.cache.dir, mode);
  
  // Export the Server Cache Directory
  exports.npsCacheDir = path.resolve(npsConf.cache.dir);

  // Create and export a Memory Cache
  //!!! Should be moved to the cache core service
  exports.npsMemCache = new nodeCache({"stdTTL":npsConf.cache.memory.ttl, "checkperiod":npsConf.cache.memory.checkperiod});  
}

// Export the Express App
exports.expressApp = app;

// Export the Application Name.
exports.npsName = npsName;

// Export the Application Version.
exports.npsVersion = npsVersion;

// Export the Server Description
exports.npsDesc = npsDesc;

// Export the Server Directory
exports.npsDir = npsDir;

// Export the Server Temp Directory
exports.npsTempDir = path.resolve(npsConf.tempDir);

// Export the Server Configuration Directory
exports.npsConfDir = npsConfDir;

// Export the Server Log File
exports.npsLogFile = path.resolve(npsConf.logDir) + '/npsServer.log'; //!!! NOT USED when controlled from PM2

// Export the Server Configuration
exports.npsConf = npsConf;

// Export the Server Port
exports.npsPort = npsPort;

// Set log properties as defined in pluginConf.log4js
npsConf.log.level = npsConf.log.level || 'INFO';
log4js.configure(npsConf.log, {});

// Get the Logger
var logger = log4js.getLogger(npsName);
logger.setLevel(npsConf.log.level); // MUST be calles although it is passed to log4js.configure(...) !?

/**
 * init:              Initialize the Server
 * @param  {function}  callback(err)
 */
function _init(callback){

  logger.debug(__filename, 'init: Initializing OXSNPS Server...');
  logger.info('Starting ' + (npsDesc || 'OXSNPS Server') + '...');
  
  // Log Hardware Info
  logger.info(utils.getHardwareInfo());

  // Log Environment Info
  logger.info(utils.getEnvironmentInfo());

  // Log /etc/hosts
  logger.info(utils.getHostsSync());

  // Log Mount points Info
  logger.info(utils.getMountInfoSync());

  // Create npsTempDir
  utils.mkDir(exports.npsTempDir, parseInt('0777',8), function(err){
    if(err){
      logger.error(__filename, 'init: Error when creating Temp Dir ' + exports.npsTempDir + ', Message: ' + err.message);
    }
  });

  // Set the parameters need for the sub-modules
  app.set('port', npsPort);

  // Set favicon
  app.use(favicon(npsDir + '/public/img/favicon.ico'));

  // Debug/Log based on the running environment
  //!!! Set to dev does not work since devDependencies like morgan are not installed, so set it to xxx
/*!!!  
  if(process.env.ENVIRONMENT === 'xxx'){ // 'dev', 'int', 'prod'
    var morgan = require('morgan');
    app.use(morgan('dev'));
  }
*/
  // Set Max. JSON request length to be 25 MB
  // V2.0.34 Begin
  //app.use(bodyParser.json({limit: '25mb'}));
  app.use(express.json({limit: '25mb'}));
  //app.use(bodyParser.urlencoded({extended: false}));
  app.use(express.urlencoded({extended: false}));
  // V2.0.34 End
  //app.use(method_override); //!!! DON'T USE IT. It screws up the routing process

  //!!! It seems that app.get('/', core.main); must be defined BEFORE app.use(express.static(path.join(npsDir, 'public')));
  //!!! The reason is, since we don't specify a path in app.use(express.static...), it is / per default.
  var core = require('oxsnps-core')
    ;
  app.get('/', core.main); 

  // Allow static files stored in the public dir to be served (ex. /css/style.css, /img/oxseed_logo.png,...)
  app.use(express.static(path.join(npsDir, 'public')));

  // Allow to browse the Server log directory
  // 1. Map http://server:port/log to the log dir
  app.use('/log', serve_index(path.resolve(npsConf.logDir),{icons:true})); 

  // 2. Allow static files stored in npsConf.logDir to be served
  app.use('/log', express.static(path.resolve(npsConf.logDir)));

  // Allow to browse the Server doc directory
  // 1. Map http://server:port/doc to the doc dir
  app.use('/doc', serve_index(path.join(npsDir, './doc'),{icons:true})); 

  // 2. Allow static files stored in the doc Dir to be served
  app.use('/doc', express.static(path.join(npsDir, './doc')));

  // Allow to browse the Server backups directory
  // 1. Map http://server:port/backup to the backups dir
  app.use('/backup', serve_index(path.resolve(npsConf.backupDir),{icons:true})); 

  // 2. Allow static files stored in npsConf.backupDir to be served
  app.use('/backup', express.static(path.resolve(npsConf.backupDir)));

  // Allow to browse the /var directory
  // 1. Map http://server:port/var to the var dir
  app.use('/var', serve_index('/var',{icons:true})); 

  // 2. Allow static files stored in /var to be served
  app.use('/var', express.static('/var'));

  // Allow to browse the /opt directory
  // 1. Map http://server:port/opt to the var dir
  app.use('/opt', serve_index('/opt',{icons:true})); 

  // 2. Allow static files stored in /opt to be served
  app.use('/opt', express.static('/opt'));

  // Allow to browse the /usr directory
  // 1. Map http://server:port/usr to the var dir
  app.use('/usr', serve_index('/usr',{icons:true})); 

  // 2. Allow static files stored in /usr to be served
  app.use('/usr', express.static('/usr'));

  // Allow to browse the /tmp directory
  // 1. Map http://server:port/tmp to the var dir
  app.use('/tmp', serve_index('/tmp',{icons:true})); 

  // 2. Allow static files stored in /tmp to be served
  app.use('/tmp', express.static('/tmp'));

  // Allow to browse the /mnt directory
  // 1. Map http://server:port/mnt to the var dir
  app.use('/mnt', serve_index('/mnt',{icons:true})); 

  // 2. Allow static files stored in /mnt to be served
  app.use('/mnt', express.static('/mnt'));

  // Allow to browse the /mnt directory
  // 1. Map http://server:port/mnt to the var dir
  app.use('/oxsnps', serve_index('/',{hidden:true, icons:true, view:'details'})); 

  // 2. Allow static files stored in /mnt to be served
  app.use('/oxsnps', express.static('/'));

  // Define routes to get version, history...
  app.get('/ping',    core.ping);
  app.get('/version', core.getVersion);
  app.get('/isalive', core.isAlive);
  app.get('/history', core.getHistory);
  app.get(['/appservices', '/appservices/routes'], core.getRoutes);
  app.get('/appservices/core/getLogLevel', core.getLogLevel);
  app.get('/appservices/core/setloglevel', core.setLogLevel);
  app.get('/quit',    core.quit);
  app.get('/http',    core.processHttpReq);

  // Add Plugin routes
  core.init(function(err){
    return callback(err);
  });

  // Add Error Handling
  // Development error handler...will print stacktrace
  if (app.get('env') === 'development') {
    app.use(function(err, req, res, next) {
      res.status(err.status || 500).end('Error: ' + err.message + '\nStatus: ' + err.status + '\nStack: ' + err.stack);
    });
  }
  else{ // Production error handler...no stacktraces leaked to user
    app.use(function(err, req, res, next) {
      res.status(err.status || 500).end('Error: ' + err.message + '\nStatus: ' + err.status);
    });
  }
}

// Initialize the Server Application
_init(function(err){
  if(err){
    logger.error('Error while initializing ' + exports.npsDesc + ' v' + exports.npsVersion
           + ' on port ' + app.get('port') + ', Reason: ' + err.message);
    process.exit(0);
  }
  // Start the server     
  server = app.listen(app.get('port'), function(err){
    if(err){
      logger.error('Error on starting ' + exports.npsDesc + ' v' + exports.npsVersion
             + ' on port ' + app.get('port')+ ', Reason: ' + err.message);
      process.exit(0);
    }
    logger.info(exports.npsDesc + ' v' + exports.npsVersion +  ' is listening on port ' + app.get('port'));
  });
});

// This function is called when the server should die gracefully, i.e. wait for existing connections
var shutdown = function(){
  logger.warn('Shutting down gracefully.');
  server.close(function(){
    logger.warn('Closed out remaining connections.');
    process.exit(0);
  });
  
  // If after 
  setTimeout(function(){
    logger.warn('Could not close connections in time, forcefully shutting down.');
    process.exit(0);
  }, 10*1000);
}

// Listen for TERM signal .e.g. kill 
process.on ('SIGTERM', shutdown);

// Listen for INT signal e.g. Ctrl-C
process.on ('SIGINT', shutdown);   