/**-----------------------------------------------------------------------------
 * oxsnps.js: 	OXSEED NodeJS Server HTML2any Plugin
 *
 * Author    :  AFP2web Team
 * Copyright :  (C) 2015 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * 
 * History
 * 1.0.8 	13.10.2023		AFP-1170: oxsnps-html2any.conf: Removed unneeded log config
 * 1.0.7 	16.09.2022		AFP-1033: OTS V2 Update Delivery
 * 1.0.6    22.02.2019    - Optionally do a pre-process to remove/modify links or references to internet resources in the html file before passing it to wkhtmltopdf tool
 * 							JiRa: OTS-2460
 * 1.0.4    08.12.2017    - transformservice() writes output to a <tempdir>/output.pdf file.
 * 							When transformservice is called parellely for N requests, 
 * 							N requests will try to write to same file 'output.pdf' that in turn
 *                          would lead to 'Output size is invalid' error
 *                          Fix: Get ouput file name from caller or 
 *                          generate a uniuque output filename and write to it.
 *                          JiRa: OTS-1990

 * 1.0.3    31.07.2017    - Forced wkhtmltopdf module to write output to file instead of stdout+pipe for streaming,
 *                          Since for some html inputs (refer RE__document_not_returned.zip attachment of OTS-1478)
 *                          html to pdf transformation hangs when output is streamed using stdout+pipe
 *                          JiRa: OTS-1478
 *
 * 1.0.2a   14.07.2017    - Updated to wkhtmltopdf v0.3.4
 *                        - Used async API from wkhtmltopdf module
 *                        - Passed options (where options are camelCased of commandline name without '-') to wkhtmltopdf
 *                        JiRa: OTS-1572
 *
 * 1.0.2    08.08.2016    Extended oxsnps-html2any plugin with proper event handlers for stream "error" and "finish" events  to catch conversion errors
 *
 * 1.0.1    07.03.2016    Extended to support TIFF B/W output
 *
 * 1.0.0                  Initial release
 *
 *----------------------------------------------------------------------------*/
'use strict';

/**
 * Variable declaration
 * @private
 */
var packageJson 	= require(__dirname + '/package.json')
  , PLUGIN_NAME 	= packageJson.name
  , PLUGIN_VERSION	= packageJson.version // Get Server Version from package.json
  , fs 				= require('fs')
  , path 			= require('path')
  , async 			= require('async')
  , log4js		    = require('log4js')
  , wkhtmltopdf 	= require('wkhtmltopdf')

  , npsServer		= require('oxsnps-core/server')
  , npsConf 		= npsServer.npsConf  
  , npsTempDir		= npsServer.npsTempDir

  , pluginConf		= require(__dirname + '/conf/' + PLUGIN_NAME + '.js')

  , dateFormat 		= require('oxsnps-core/helpers/date_format')
  , utils 			= require('oxsnps-core/helpers/utils')
  , expressAppUtils = require('oxsnps-core/helpers/expressAppUtils')
  , pluginManager 	= require('oxsnps-core/pluginManager')  
  , FILE_SEP 		= require('path').sep  
  , htmlRules 		= undefined		// v1.0.6
  ;

// Set plugin name, version and longDesc in pluginConf
pluginConf.module 	= PLUGIN_NAME; 
pluginConf.version 	= PLUGIN_VERSION; 
pluginConf.longDesc = packageJson.description;

// Export the Plugin's Configuration Express App
exports.pluginConf = pluginConf;

// Get the Logger
var logger = log4js.getLogger(PLUGIN_NAME);
logger.setLevel(pluginConf.log.level || 'INFO'); // set log level

/**************** PUBLIC FUNCTIONS ****************/
/**
 * Initialize Plugin
 * @param  {function} 	callback  callback(err, routes)
 */
exports.initialize = function(callback){

	var self 			= this
	  , appPluginConf	= (npsConf.plugins && npsConf.plugins[PLUGIN_NAME] ? npsConf.plugins[PLUGIN_NAME] : {})		// v1.0.6

	logger.debug('--->initializing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION );
	
	if(self._initialized) return callback();

	// v1.0.6 Begin
	// Merge application specific configuration with plugin configuration
	pluginConf = utils.mergeJSON(pluginConf, appPluginConf)
	
	htmlRules = pluginConf.htmlRules		// can be a string or a JSON object
	
	if(typeof htmlRules === 'string'){
		let envvar = pluginConf.htmlRules
		if(!process.env[envvar]) return callback(new Error('Missing environment variable: ' + envvar))

		htmlRules = utils.parseJSON(process.env[envvar])
		if(htmlRules === null) return callback(new Error('Unable to parse environment variable: ' + 
										envvar + ', value: ' + process.env[envvar]))
	} 
	logger.debug('--->initialize: htmlRules: ' + JSON.stringify(htmlRules))
	// v1.0.6 End
	
	// Map the HTTP routes of the plugin
	expressAppUtils.enableRoutes(self, pluginConf.routes, function(err){
		if(err)	return callback(err);
		self._initialized = true;
		logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' initialized');
		callback(null, pluginConf.routes);
	});
}

/**
 * Activate Plugin. This function is called from pluginManager once all the enabled plugins have been initialize
 * @param  {function} 	callback  callback(err)
 */
exports.postInitialize = function(callback){
	callback();
}

/**
 * Finalize Plugin
 * @param  {function} 	callback callback(err) 
 */
exports.finalize = function(callback){

	logger.debug('--->finalizing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION );
	var self = this;

	if(!self._initialized) return callback();

	// Unmap the HTTP routes of the plugin
	expressAppUtils.disableRoutes(pluginConf.routes, function(err){
		if(err) return callback(err);
		self._initialized = false;
		logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' finalized');
		callback();
	});
}

/**
 * Return Plugin Version
 * @param  {function} 	callback callback(err,version)
 */
exports.getVersion = function(callback){
	return callback(null, PLUGIN_VERSION);
}

/**
 * Process a HTTP Version Request
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.version = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);
	logger.info('--->version: Req. id=' + res._Context.reqId);

	var data = '<h3>' +  PLUGIN_NAME + ' v' + PLUGIN_VERSION + '</h3>';
	res.end(data);
	
	logger.info('--->version: Req. id=' + res._Context.reqId + ', sent back ' +
				PLUGIN_NAME + ' v' + PLUGIN_VERSION + ', over');
	return;
}

/**
 * Process a HTTP Configuration Request 
 * 	Either as HTTP GET request:
 * 	 	GET http://localhost:1029/services/<plugin>/conf 							--> To retrieve the configuration
 * 	 	GET http://localhost:1029/services/<plugin>/conf?save=true&conf="{...}" 	--> To pass and save the configuration
 * 	or as HTTP POST request:
 * 	 	POST http://localhost:1029/services/<plugin>/conf 						--> To pass and save the configuration
 * 	 	req.body={
 * 	 		"conf": {...}
 * 	 	}
 * 	
 * @param {Request} 	req Request Object
 *                      req.query:{ // for HTTP GET request:
 *                      	"save":"false",
 *                      	"conf":"{<IMPORTANT: COMPLETE JSON Conf. as String>}"
 *                      }
 *                      req.body:{ // for HTTP POST request:
 *                      	"conf":{<IMPORTANT: COMPLETE JSON Conf. as JSON Object>}
 *                      }
 * @param {Response}    res Response Object
 */
exports.conf = function(req, res){

	var reqType 	= 'getconf'
	  , jsonConf 	= undefined
	  ;

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);
	logger.info('--->conf: Req. id=' + res._Context.reqId);
	
	if(req.method.toUpperCase() === 'POST'){
		logger.info('--->conf: body=' + JSON.stringify(req.body));
		if(!req.body || !req.body.conf) {
			logger.info('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body));
			return res.end('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body));
		}
		reqType = 'saveconf'; 		// save request
		jsonConf = req.body.conf; 	// set the json conf to be passed to saveConfService
	}
	else if(req.method.toUpperCase() === 'GET'){
		logger.info('--->conf: query=' + JSON.stringify(req.query));
		if(req.query && req.query.save && req.query.save.toLowerCase() === 'true' && req.query.conf)
		{
			reqType = 'saveconf';
			jsonConf = utils.parseJSON(unescape(req.query.conf)); // remove escaped chars and set the json conf to be passed to saveConfService
		} 
	}

	switch(reqType){
		case 'saveconf':
		    exports.saveConfService(jsonConf, function(err){
		    	logger.info('--->conf: Req. id=' + res._Context.reqId + ', over');
				res.json(err ? {'status':'error', 'message': err.message} : pluginConf);
		    }); 
			break;
		default:
		case 'getconf':
			logger.info('--->conf: Req. id=' + res._Context.reqId + ', over');
			res.json(pluginConf);
			break;
	}
}

/**
 * Store  the Plugin conf to its configuration file (i.e. .../<plugin>/conf/<plugin>.js)
 * @param  {JSON}   	conf     Plugin Configuration
 * @param  {Function} 	callback
 */
exports.saveConfService = function(jsonConf, callback){

	var filename 	= undefined
	  , data 		= undefined
	  ;
	
	// PluginManager calls saveConfService without passing the conf
	if(typeof jsonConf === 'function'){
		callback = jsonConf;
		jsonConf = pluginConf;
	}

	// Assert jsonConf
	if(!jsonConf || jsonConf === null) return callback(new Error('--->saveConfService: empty Configuration'));
	if(logger.isDebugEnabled()) logger.debug('--->saveConfService: conf:' + JSON.stringify(jsonConf));

	// Backup the current plugin conf.in the server's backup dir and store the new one in the plugin's /conf dir
	async.series([
		// Task 1: Backup the current plugin conf.
		function(nextTask){
			filename = npsConf.backupDir + FILE_SEP + dateFormat.asString('yyyy-MM-dd_hh-mm-ss', new Date()) + '_configurationOf_' + PLUGIN_NAME + '.js'; // build the Backup filename
			if(logger.isDebugEnabled()) logger.debug('--->saveConfService: Backuping plugin configuration to ' + filename);
			data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t'); // build the plugin conf content

			// Write data to the Plugin configuration file
			fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
				if(err) logger.error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message); // log error & continue
				nextTask();
			});
		},
		// Task 2: Save the new plugin conf.
		function(nextTask){
			pluginConf = jsonConf; // we should have here a proper conf, so overwrite the current pluginConf file
			filename = path.resolve(__dirname + '/conf/' + PLUGIN_NAME + '.js'); // build the plugin configuration filename
			if(logger.isDebugEnabled()) logger.debug('--->saveConfService: Writing plugin configuration to ' + filename);
			data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t'); // build the plugin conf content

			// Write data to Plugin configuration file
			fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
				if(err) nextTask(new Error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message));
				else nextTask();
			}); 
		}
	],
	function(err){
		callback(err);
	});
}

/**
 * HTTP GET request to transform HTML to PDF or TIF
 * @param  {Request}  	req Request Object
 *                      req.query: {
 *                      	'inputDoc':      <html file name>,
 *                      	'outputType':    <'PDF'|'TIF'>,
 *                      	'transformOptions': [OPTIONAL] <wkhtmltopdf options for output generation>,
 *                      }
 * @param  {Response} 	res Response Object
 */
exports.transform = function(req, res){
    // V1.0.2a Begin
	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the temp Dir to the Context
	res._Context.tempDir = npsTempDir + '/' + res._Context.reqId;

	logger.info('--->transform: Req. id=' + res._Context.reqId);

    if(req.query.inputDoc === undefined){
        return res.status(404).end('--->transform: Req. id=' + res._Context.reqId + ', Error! Reason: HTML input (inputDoc) is not given');
    }
	res._Context.inputDoc = req.query.inputDoc || undefined;
	res._Context.outputType = req.query.outputType || 'pdf';
	res._Context.outputFilename = req.query.outputFilename;
	res._Context.transformOptions = req.query.transformOptions || {};
	/*
    logger.debug('--->transform: Req.Id=' + res._Context.reqId + ', ' + 
    			'OutputType=' + res._Context.outputType + 
    			((typeof res._Context.inputDoc === 'string') ? (', Inputfilename=' + res._Context.inputDoc) : '' ) +
	    		(res._Context.outputFilename ? (', OutputFilename=' + res._Context.outputFilename) : '' ) +
    			', TransformOptions= ' + JSON.stringify(res._Context.transformOptions))
	*/
	// Call the transform service
	exports.transformService(res, function(err, result){
		if(err){
			logger.error('--->transform: Req. id=' + res._Context.reqId + ', Error! Reason: ' + err.message);
			return res.status(404).end('--->transform: Req. id=' + res._Context.reqId + ', Error! Reason: ' + err.message);
		}			
		logger.info('--->transform: Req. id=' + res._Context.reqId + ', over');
		if(result) res.setHeader('Content-Type', 'application/' + res._Context.outputType);
		return res.end(result);
	}); 
    // V1.0.2a End
}

/**
 * Service to transform HTML to PDF or TIF
 * @param  {Response}   res
 *                      res._Context:{
 *                      	'inputDoc':      <html file name or html buffer>,
 *                      	'outputFilename': Optional. If given output will be written to <outputFilename> else returned as buffer
 *                      	'outputType':    <'PDF'|'TIF'>,
 *                      	'transformOptions': [OPTIONAL] <wkhtmltopdf options for output generation>,
 *                      }
 *
 * where
 * For "res_Context.transformOptions", All of the command line options are supported as documented on the page https://wkhtmltopdf.org/usage/wkhtmltopdf.txt
 * The options are camelCased instead-of-dashed as in the command line tool. Example. if option is --page-size, then it should be specified as {'pageSize: 'letter'}
 * Note that options that do not have values, must be specified as a boolean, e.g. debugJavascript: true. Reference 1: https://www.npmjs.com/package/wkhtmltopdf#options
 *
 * @param  {Function} 	callback(err, result)
 */
exports.transformService = function(res, callback){
	// 1.0.2a Begin
	var html2PDFOpts	= {}
	  , inFilename 	= undefined 
	  , inURIFile 	= undefined 
	  , outFilename = undefined 
	  , outputType  = res._Context.outputType || 'pdf'
	  , mode        = parseInt('0777',8)
	  , result      = undefined 
	  , stream 		= undefined
	  , saveErr 	= undefined
	  , orgInputDoc = undefined
	  , inputDoc	= undefined
	  , min = 10000 
      , max = 100000
    ;	  
	  ;
	// 1.0.2a End

	// Task 1: Validate the parameters
    // Assert for input document
    if(res._Context.inputDoc === undefined) return callback(new Error('oxsnps-html2any: Invalid request. HTML input (res._Context.inputDoc) is not given'));

    // Assert for output type
    outputType = outputType.toLowerCase();
    if(   outputType !== 'pdf'
       && outputType !== 'tif'
       && outputType !== 'tiff'
       && outputType !== 'tif_bw' // V1.0.1 Change
      ) 
    	return callback(new Error('oxsnps-html2any: html to ' + outputType + ' transformation not yet supported'));

    // Get htmltopdf options
    html2PDFOpts = res._Context.transformOptions || html2PDFOpts;
    logger.info('--->transformService: Req. Id=' + res._Context.reqId + ', html to ' + outputType + ' transformation' + 
    			((typeof res._Context.inputDoc === 'string') ? (', Inputfilename=' + res._Context.inputDoc) : '' ) +
	    		(res._Context.outputFilename ? (', OutputFilename=' + res._Context.outputFilename) : '' ) +
		    	', Options: ' + JSON.stringify(html2PDFOpts));
	// 1.0.6 Begin
	orgInputDoc = inputDoc = res._Context.inputDoc
	// 1.0.6 End
	
    async.series([
    	// Create temp. dir.
		// 1.0.6 Begin		
        function(nextTask){
			logger.debug('--->transformService: Req. Id=' + res._Context.reqId + ', Creating request temp. directory ' + res._Context.tempDir +'...');
			utils.mkDir(res._Context.tempDir, mode, function(err){
				if(err) return nextTask(new Error('Could not make directory ' + res._Context.tempDir + '. ' + err.message));
				return nextTask()
			});
		},
		// Task 1: Do Preprocess
        function(nextTask){
			if(!htmlRules || htmlRules.length <=0) return nextTask()
			_doPreprocess(res, function(err, result){	// returns a filename that has modified html content 
				if(!err && result) inputDoc = result
				nextTask(err)
			});
		},		
		// 1.0.6 End
		// Task 2: Build input filename. Use MD5 id if buffer is given 
        function(nextTask){		
    		if(typeof inputDoc === 'string'){
    			inFilename 	= path.resolve(inputDoc);
    			inURIFile 	= 'file://' + path.resolve(inputDoc); 
				if(typeof orgInputDoc === 'string')	outFilename = res._Context.tempDir + FILE_SEP + utils.getLastToken(path.resolve(orgInputDoc), FILE_SEP) + '.pdf';
				else outFilename = inFilename + '.pdf'
    			nextTask();
    		}
    		else{
    			logger.debug('--->transformService: Req. Id=' + res._Context.reqId + ', Creating MD5 ID of input buffer...');
				utils.getObjMD5Id(res._Context.inputDoc, function(err, md5Id){
					if(err) return nextTask(err);
					// build unique filename
					inFilename = res._Context.tempDir + FILE_SEP + md5Id + '_'+
									dateFormat.asString("yyMMddhhmmssSSS", new Date()) + 
									'-' + Math.floor(Math.random() * (max - min + 1) + min) + '.html';
					inURIFile 	= 'file://' + inFilename;
					outFilename = inFilename + '.pdf'
					logger.debug('--->transformService: Req. Id=' + res._Context.reqId + ', Writing input html data to ' + inFilename +'...');
					// Write the data to the specified file
					fs.writeFile(inFilename, res._Context.inputDoc, {'encoding': 'binary'}, nextTask);					
					//nextTask();
				});
    		}
    	},
    	// 1.0.4 End
		/*
        // Task 2: Store the html data to the temp dir
        function(nextTask){
			logger.debug('--->transformService: Req. Id=' + res._Context.reqId + ', Creating request temp. directory ' + res._Context.tempDir +'...');
			utils.mkDir(res._Context.tempDir, mode, function(err){
				if(err) return nextTask(new Error('Could not make directory ' + res._Context.tempDir + '. ' + err.message));
				if(typeof res._Context.inputDoc === 'string') return nextTask();
				logger.debug('--->transformService: Req. Id=' + res._Context.reqId + ', Writing input html data to ' + inFilename +'...');
				// Write the data to the specified file
				fs.writeFile(inFilename, res._Context.inputDoc, {'encoding': 'binary'}, nextTask);
			});
		},
		*/
		// Task 3: Start the html2pdf transformation
		function(nextTask){
			// 1.0.4 Begin
			// Pass output file name on html2PDFOpts to avoid (stdout + pipe) streaming of output
			if(res._Context.outputFilename && outputType === 'pdf') outFilename = res._Context.outputFilename;
			
			html2PDFOpts.output = outFilename;  // V1.0.3 Change
			// 1.0.4 End
			logger.debug('--->transformService: Req. Id=' + res._Context.reqId + 
					', Starting html2pdf transformation of ' + inURIFile + ' to ' + html2PDFOpts.output + '...');

			// V1.0.2a Begin
			wkhtmltopdf(inURIFile, html2PDFOpts, function(err, stream){
				if(err) return nextTask(err);

				// V1.0.3 Begin
				/*
				if(!stream) return nextTask(new Error("Null stream from wkhtmltopdf callback"));

				stream.pipe(fs.createWriteStream(outFilename));
				stream.on('finish', function(err){ // V1.0.2 Change
					logger.debug('--->transformService: Req. Id=' + res._Context.reqId + ', "finish" event called. ' + (err ? err.message : '') );
					if (saveErr) return nextTask(saveErr);
					return nextTask(err);
				});
				stream.on('error', function(err){ // V1.0.2 Change
					logger.debug('--->transformService: Req. Id=' + res._Context.reqId + ', "error" event called. ' + err.message);
					saveErr = err;
					// Since 'finish' event will be triggered even in case of error, do not call nextTask here.
					// It will be called in 'finish' event, 
					//nextTask(err);
				});
				*/
				fs.stat(outFilename, function(err, stat){
					if(err) return nextTask(err);
					if(stat.size <= 0) return nextTask(new Error('oxsnps-html2any: html2pdf transformation failed. Reason: Output size is invalid (' + stat.size + ')'));
					logger.debug('--->transformService: Req. Id=' + res._Context.reqId + ', Finished html2pdf transformation of ' + inURIFile + ' to ' + outFilename + ', FileSize=' + stat.size);
					nextTask();
				});
				// V1.0.3 End
			});
			// V1.0.2a End
		},
		// Task 3: Convert PDF to TIFF if required
		function(nextTask){
			if(!res._Context.outputFilename) return nextTask();
			if(outputType === 'pdf') return nextTask();   
			// Convert PDF to TIFF
			logger.debug('--->transformService: Req. Id=' + res._Context.reqId + ', Starting pdf2tif transformation of ' + 
						 outFilename + ' to ' + res._Context.outputFilename + '...');
			res._Context.inputDoc = outFilename;
			_pdf2tif(res, function(err){
				if(err) return nextTask(new Error('Could not convert ' + outFilename + ' to ' + res._Context.outputFilename + '. ' + err.message));
				logger.debug('--->transformService: Req. Id=' + res._Context.reqId + ', Finished pdf2tif transformation of '+ outFilename + ' to ' + res._Context.outputFilename);
				nextTask();
			});	
		},		
		// 1.0.4 End
		// Task 4: Convert PDF to TIFF if required
		function(nextTask){
			if(res._Context.outputFilename) return nextTask();
			// Read PDF file
			fs.readFile(outFilename, function(err, pdfData){
				if(err)	return nextTask(new Error('Could not read from ' + outFilename + ' file. ' + err.message));
				logger.debug('--->transformService: Req. Id=' + res._Context.reqId + ', PDF Data Length: ' +  pdfData.length);

				// If output type is tif, convert PDF to tif
				// V1.0.1 Begin
				//if(outputType === 'tif'){
				if(outputType === 'tif' || outputType === 'tiff' || outputType === 'tif_bw'){
				// V1.0.1 End
					// Convert PDF to TIFF
					logger.debug('--->transformService: Req. Id=' + res._Context.reqId + ', Starting pdf2tif transformation...');
					res._Context.inputDoc = pdfData;
					_pdf2tif(res, function(err, tifData){
						if(err) return nextTask(new Error('Could not convert "' + outFilename + '" to tif, ' + err.message));
						logger.debug('--->transformService: Req. Id=' + res._Context.reqId + ', Finished pdf2tif transformation.' + 
										(tifData ? (' TIF Data Length: ' +  tifData.length) : ''));
						result = tifData;
						nextTask();
					});	
				}
				else{
					// outputType is PDF
					result = pdfData;
					nextTask();
				}			
			});
		}
	],	function(err){
			if(err) logger.error('--->transformService: Req. Id=' + res._Context.reqId + ', html to ' + outputType + ' transformation failed. Reason:' + err.message);
			else logger.info('--->transformService: Req. Id=' + res._Context.reqId + ', html to ' + outputType + ' transformation over.');
			return callback(err, result);	// return the output data
		}
	);
}

/************ OPTIONAL PUBLIC FUNCTIONS ***********/
/**
 * Set Log Level
 * @param {String} logLevel Log Level. values:DEBUG|INFO|WARN|ERROR 
 */
exports.setLogLevel = function(logLevel){
	pluginConf.log.level = logLevel || 'INFO'
	logger.setLevel(pluginConf.log.level);
}

/**
 * Get Log Level
 * @return {String} Log Level. values:DEBUG|INFO|WARN|ERROR
 */
exports.getLogLevel = function(){
	return pluginConf.log.level
}

/**
 * Enable the given route
 * @param  {Request} 	req
 *                      req.body = {
 *                      	"module":"sample", 
 *                      	"route": { 
 *                      		"path": "/path/to/the/service",
 *                      		"method": 	Optional. Values are "get|post|dw|queue",
 *                      		"service": 	"version"
 *                      	}
 *                      }
 * @param  {function} 	callback callback(err)
 */
exports.enableRoute = function(req, callback){

	var self = this;

	if(logger.isDebugEnabled()) logger.debug('--->enableRoute: Route=' + JSON.stringify(req.body.route));

	if(!req.body.route) return callback(new Error('route parameter not given'));

	req.body.route.method = req.body.route.method || 'get';

	// Enable the route
	switch(req.body.route.method.toLowerCase()){
		case 'post':
		case 'get':
			// Enable HTTP Route
			expressAppUtils.enableRoute(self, req.body.route, callback);
			break;
		/*
		case 'dir': // Enable DIR route
	   		// Delete Queued Jobs before enabling route 
	   		_cleanJobs('inactive', function(err){
   				if(err){
   					logger.error('--->enableRoute Error: ' + err.message);
   					return callback(err);
   				}
				pluginManager.callPluginService({name:'oxsnps-dirwatcher',service:'addListenersService'}, [req.body.route], callback);
	   		});			
			break;
		//Queue routes not applicable for this plugin	
		case 'queue': 	// Enable Queue Route
	   		// Delete Queued Jobs before enabling route 
	   		_cleanJobs('inactive', function(err){
   				if(err){
   					logger.error('--->enableRoute Error: ' + err.message);
   					return callback(err);
   				}
				pluginManager.callPluginService({name:'oxsnps-ibmmq', service:'addListenersService'}, [req.body.route], callback);
	   		});
			break;
		*/	
		default:
			//logger.warn('--->enableRoute: Wrong parameters(unknown route.method)! Not able to enable the route, Parms: ' + JSON.stringify(req.body));
			callback(new Error('Wrong parameters(unknown route.method)'));
	}
} 

/**
 * Disable the given route
 * @param  {Request} 	request Request json has following structure.
 * 		            	req.body = {
 *                      	"module":"sample", 
 *                      	"route": { 
 *                      		"path": "/path/to/the/service",
 *                      		"method": 	Optional. Values are "get|post|dw|queue"
 * 		            	  	}
 * 		                }
 * @param  {function} 	callback  callback(err)
 */
exports.disableRoute = function(req, callback){

	if(logger.isDebugEnabled()) logger.debug('--->disableRoute: Route=' + JSON.stringify(req.body.route));
	
	if(!req.body.route) return callback(new Error('route parameter not given'));

	req.body.route.method = req.body.route.method || 'get';

	// Disable the route
	switch(req.body.route.method.toLowerCase()){
		case 'post':
		case 'get':
			// Disable HTTP Route
			expressAppUtils.disableRoute(req.body.route, callback);
			break;
		/*
		case 'dir': 
			// Disable DIR Route
			pluginManager.callPluginService({name:'oxsnps-dirwatcher',service:'removeListenersService'}, [req.body.route], function(err){
				if(err) return callback(err);
		   		// Delete Queued Jobs if route is disabled successfully
		   		_cleanJobs('inactive', function(err){
	   				if(err) logger.error('--->disableRoute Error: ' + err.message);
					callback();
		   		});			
			});
			break;

		// Queue Routes are not applicable for this plugin

		case 'queue':
			// Disable Queue Route
			pluginManager.callPluginService({name:'oxsnps-ibmmq', service:'removeListenersService'}, [req.body.route], callback);
			break;
		*/	
		default:
			//logger.warn('--->disableRoute: Wrong parameters(unknown route.method)! Not able to disable the route, Parms: ' + JSON.stringify(req.body));
			callback(new Error('Wrong parameters(unknown route.method)'));
	}
} 

/**************** PRIVATE FUNCTIONS ***************/
/**
 * Call afp2any syncTransform to convert PDF to tiff(bw)
 * @param  {JSON}     res       Response Object
 *                    res._Context:{
 *                    	inputDoc: <Buffer>
 *                    	...
 *                    }
 * @param  {Function} callback  Callback(err,data)
 * @return {}             
 */
function _pdf2tif(res, callback){

	var pluginName = 'oxsnps-afp2any'
	  , module = null
	  // V1.0.1 Begin
	  , outputType = res._Context.outputType.toLowerCase()
	  // V1.0.1 End 
	  ;

	// Set needed request parameters to call afp2any plugin
	res._Context.req = {
		'input':{'type': 'pdf'},
		// V1.0.1 Begin
		//'output':{'type': 'tif'}
		'output':{'type': outputType}
		// V1.0.1 End 
	};

	// Clear any scenarioConf set in res._Context, by any other plugin that called afp2any to process the request
	res._Context.scenarioConf = undefined;

	// Clear any a2wsProps set by any other plugin that called afp2any to process the request
	res._Context.a2wsProps = undefined;
	if(res._Context.outputFilename){
		res._Context.a2wsProps = {
			'OutputFilePath':  path.dirname(res._Context.outputFilename),
			'FilenamePattern': '"' + path.basename(res._Context.outputFilename, FILE_SEP) + '"',
		}
	}

	// Get the handle of the requested plugin if enabled
	pluginManager.getModule(pluginName, function(err, module){
		if(err) return callback(err);

		// Try to call the requested Service from the given Plugin
		try{
			module['syncTransformService'](res, function(err, data){
				callback(err, data);
			});
		}catch(err){
			return callback(new Error('Calling ' + pluginName + '.syncTransformService() failed.' + err.message));
		}
	});
}

// 1.0.6 Begin	
/**
 * Preprocess HTML content
 * Mainly used to remove/modify references that need network access
 * @param  {Response}   res
 *                      res._Context:{
 *                      	'inputDoc':      <html file name or html buffer>,
 *                      }
 *
 * @param  {Function} 	callback(err, modifiedFilename)
 */
function _doPreprocess(res, callback){

	let newContent	= undefined
	  , outFilename	= undefined
	  
	async.series([
		// Get HTML content in buffer
		function(nextTask){
			if(typeof res._Context.inputDoc === 'string'){
				if(logger.isDebugEnabled()) logger.debug('--->_doPreprocess: Req. Id=' + res._Context.reqId + ', Reading a file: ' + res._Context.inputDoc)
				fs.readFile(res._Context.inputDoc, {'encoding':'utf8'}, function(err, data){
					if(err) return nextTask(new Error('Unable to a read file "' + res._Context.inputDoc + '": ' + err.message))
					res._Context.htmlBuf = data
					return nextTask()
				})
				return
			}
			res._Context.htmlBuf = res._Context.inputDoc	// we already have html in a buffer
			return nextTask()
		},
		// Apply html rules
		function(nextTask){
			newContent = _applyHTMLRules(res)
			nextTask()
		},		
		// Write data to the file
		function(nextTask){
			outFilename = res._Context.tempDir + FILE_SEP + res._Context.reqId + '_modified.html' 			
			if(logger.isDebugEnabled()) logger.debug('--->_doPreprocess: Req. Id=' + res._Context.reqId + ', Writing modified html content to a file: ' + outFilename)
			fs.writeFile(outFilename, newContent, {'encoding': 'utf8'}, function(err){
				if(err) return nextTask(new Error('Unable to write to "' + outFilename + '": ' + err.message))
				nextTask()
			})
		}
	],
	function(err){
		callback(err, outFilename)
	})
}

/**
 * Apply html rules specified in configuration to HTML content
 * Mainly used to remove/modify references that need network access
 * @param  {Response}   res
 *                      res._Context:{
 *                      	'htmlBuf':      <html buffer>,
 *                      }
 *
 * @param  {Function} 	callback(err, modifiedFilename)
 */
function _applyHTMLRules(res){

	let htmlBuf		= res._Context.htmlBuf
	  , patExists 	= false
	  , regexp 		= undefined
	
	/*
	"htmlRules" :[
		{
			"pattern": "pattern",
			"flags": "ig",
			"value": '$1title=$2$3'
		},
		{
			"pattern": /pattern2/ig
			"value": '$1$2'
		}		
	]
	*/
	if(logger.isDebugEnabled()) logger.debug('--->_applyHTMLRules: Req. Id=' + res._Context.reqId + ', Started to apply html rules..')
	htmlRules.forEach(function(item){
		if(!item.pattern) return
		if(typeof item.pattern == 'string') regexp = new RegExp(item.pattern, (item.flags || "ig"))
		else regexp = item.pattern
        patExists = regexp.test(htmlBuf)
        while(patExists){
            htmlBuf = htmlBuf.replace(regexp, item.value||'')
            patExists = regexp.test(htmlBuf)
        }
    })
	if(logger.isDebugEnabled()) logger.debug('--->_applyHTMLRules: Req. Id=' + res._Context.reqId + ', Finished applying html rules')
	return htmlBuf
}
// 1.0.6 End
