/**-----------------------------------------------------------------------------
 * oxsnps.js:  oxsnps-officeclient:  A client for sending HTTP requests and receiving responses to/from the oxs-office-server.
 *
 * Author    :  AFP2web Team
 * Copyright :  (C) 2024-2025 by Maas Holding GmbH
 * Email     :  support@maas.de
 * 
 * History
 *
 * V1.0.0      01.09.2025    AFP-1259: Implement oxsnps-officeclient plugin
 *----------------------------------------------------------------------------*/
'use strict'

/**
 * Variable declaration
 * @private
 */
var packageJson      = require(__dirname + '/package.json')
  , PLUGIN_NAME      = packageJson.name
  , PLUGIN_VERSION   = packageJson.version // Get Server Version from package.json
  , pluginConf       = require(__dirname + '/conf/' + PLUGIN_NAME + '.js')

  , fs               = require('fs')
  , os               = require('os')
  , async            = require('async')  
  , path             = require('path')
  , log4js           = require('log4js')
  , FILE_SEP         = path.sep

  , npsServer        = require('oxsnps-core/server')
  , npsConf          = npsServer.npsConf
  , npsTempDir       = npsServer.npsTempDir

  , pluginManager    = require('oxsnps-core/pluginManager')  
  , bufferStream     = require('oxsnps-core/helpers/bufferStream')    
  , dateFormat       = require('oxsnps-core/helpers/date_format')
  , utils            = require('oxsnps-core/helpers/utils')
  , httpUtils        = require('oxsnps-core/helpers/httpUtils')
  , expressAppUtils  = require('oxsnps-core/helpers/expressAppUtils')
  , SERVER_NAME      = "oxs-office-server"
  , initError        = undefined

// Set plugin name, version and longDesc in pluginConf
pluginConf.module    = PLUGIN_NAME 
pluginConf.version   = PLUGIN_VERSION 
pluginConf.longDesc  = packageJson.description
pluginConf.log       = pluginConf.log || {}

// Export the Plugin's Configuration Express App
exports.pluginConf   = pluginConf
exports.router       = undefined
exports.ERROR	     = 'error'
exports.SUCCESS	     = 'success'
exports.OP_MODE_BUF  = 'buf'
exports.OP_MODE_FILE = 'file'

// Get the Logger
let logger = log4js.getLogger(PLUGIN_NAME)
utils.log4jsSetLogLevel(logger, (pluginConf.log.level || 'INFO'))

/**************** PUBLIC FUNCTIONS ****************/
/**
 * Initialize Plugin
 * @param  {function}     callback  callback(err, routes)
 */
exports.initialize = function(callback){

    let self          = this
      , envParseErr   = undefined
      , appPluginConf = (npsConf.plugins && npsConf.plugins[PLUGIN_NAME] ? npsConf.plugins[PLUGIN_NAME] : {})    
      
    initError = undefined

    function _handleError(err){
        initError = err
        self._initialized = false 
        return callback(err, pluginConf.routes)
    }          
      
    if(logger.isDebugEnabled()) logger.debug('--->initializing ' + PLUGIN_NAME + ' v' + PLUGIN_VERSION)

    if(self._initialized) return callback()

    // Merge application specific configuration with plugin configuration
    pluginConf = utils.mergeJSON(pluginConf, appPluginConf)

    envParseErr = _parseEnvs()
    if(envParseErr) return _handleError(envParseErr)

    pluginConf.maxresptime = pluginConf.maxresptime || 60000
    pluginConf.maxresptime = pluginConf.maxresptime * 1

    // For oxsnps-core version  >= V4.x.x
    if (typeof(expressAppUtils.setupRouter) === 'function'){    
        // Add a router for the plugin routes and its middlewares
        expressAppUtils.setupRouter(self, function(err, router){
            if(err) return _handleError(err)
            exports.router = router
            self._initialized = true
            logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v' + PLUGIN_VERSION + ' initialized')
            return callback(null, pluginConf.routes)
        })
        return 
    }

    // For oxsnps-core version <= 3.x.x
	// Map the HTTP routes of the plugin
	expressAppUtils.enableRoutes(self, pluginConf.routes, function(err){
		if(err)	return callback(err);
		self._initialized = true;
		logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' initialized');
		callback(null, pluginConf.routes);
	});    
}

/**
 * Activate Plugin. This function is called from pluginManager once all the enabled plugins have been initialize
 * @param  {function}     callback  callback(err)
 */
exports.postInitialize = function(callback){
    let self  = this
      , plVer = 'Plugin ' + PLUGIN_NAME + ' v' + PLUGIN_VERSION

    if(!self._initialized)
        return callback(new Error('Can\'t activate ' + plVer + ' as long as it has not been initialized!'))
    
    if(logger.isDebugEnabled()) logger.debug('Activating ' + plVer + ' ...')
    return callback()
}

/**
 * Finalize Plugin
 * @param  {function}     callback callback(err) 
 */
exports.finalize = function(callback){

    let self = this

    if(logger.isDebugEnabled()) logger.debug('--->finalizing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION )
    if(!self._initialized) return callback()

    async.series([
        // Disable all routes
        function(nextTask){
            async.each(pluginConf.routes, 
                function(route, next){
                    let req = {"body": {'module': PLUGIN_NAME, "route": route}}
                    exports.disableRoute(req, next)
                },
                function(err){
                    return nextTask(err)
                }
            )
        }
    ],
    function(err){
        if(err) return callback(err)
        self._initialized = false
        logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' finalized')
        callback()
    })

}

/**
 * Return Plugin Version
 * @param  {function}     callback callback(err,version)
 */
exports.getVersion = function(callback){
    return callback(null, PLUGIN_VERSION)
}

/**
 * Process a HTTP Version Request
 * @param  {Request}      req Request Object
 * @param  {Response}     res Response Object
 */
exports.version = function(req, res){

    let data = undefined

    // Add Context to the response instance
    res._Context = {}

    // Add the Current Date to the Context
    res._Context.date = new Date()

    // Add a Request Id to the Context
    res._Context.reqId = utils.buildReqId(res._Context.date)
    logger.info('--->version: Req. Id=' + res._Context.reqId)

    data = '<h3>' +  PLUGIN_NAME + ' v' + PLUGIN_VERSION + '</h3>'
    res.end(data)

    logger.info('--->version: Req. Id=' + res._Context.reqId + ', sent back ' +
                PLUGIN_NAME + ' v' + PLUGIN_VERSION + ', over')
    return
}

/**
 * Process a HTTP Configuration Request 
 *     Either as HTTP GET request:
 *          GET http://localhost:1029/services/<plugin>/conf                             --> To retrieve the configuration
 *          GET http://localhost:1029/services/<plugin>/conf?save=true&conf="{...}"     --> To pass and save the configuration
 *     or as HTTP POST request:
 *          POST http://localhost:1029/services/<plugin>/conf                         --> To pass and save the configuration
 *          req.body={
 *              "conf": {...}
 *          }
 *     
 * @param {Request}     req Request Object
 *                      req.query:{ // for HTTP GET request:
 *                          "save":"false",
 *                          "conf":"{<IMPORTANT: COMPLETE JSON Conf. as String>}"
 *                      }
 *                      req.body:{ // for HTTP POST request:
 *                          "conf":{<IMPORTANT: COMPLETE JSON Conf. as JSON Object>}
 *                      }
 * @param {Response}    res Response Object
 */
exports.conf = function(req, res){

    let reqType     = 'getconf'
      , jsonConf     = undefined
      
    // Add Context to the response instance
    res._Context = {}

    // Add the Current Date to the Context
    res._Context.date = new Date()

    // Add a Request Id to the Context
    res._Context.reqId = utils.buildReqId(res._Context.date)
    logger.info('--->conf: Req. id=' + res._Context.reqId)
    
    if(req.method.toUpperCase() === 'POST'){
        logger.info('--->conf: body=' + JSON.stringify(req.body))
        if(!req.body || !req.body.conf) {
            logger.info('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body))
            return res.end('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body))
        }
        reqType = 'saveconf'         // save request
        jsonConf = req.body.conf     // set the json conf to be passed to saveConfService
    }
    else if(req.method.toUpperCase() === 'GET'){
        logger.info('--->conf: query=' + JSON.stringify(req.query))
        if(req.query && req.query.save && req.query.save.toLowerCase() === 'true' && req.query.conf)
        {
            reqType = 'saveconf'
            jsonConf = utils.parseJSON(unescape(req.query.conf)) // remove escaped chars and set the json conf to be passed to saveConfService
        } 
    }

    switch(reqType){
        case 'saveconf':
            exports.saveConfService(jsonConf, function(err){
                logger.info('--->conf: Req. id=' + res._Context.reqId + ', over')
                res.json(err ? {'status':ERROR, 'message': err.message} : pluginConf)
            }) 
            break
        default:
        case 'getconf':
            logger.info('--->conf: Req. id=' + res._Context.reqId + ', over')
            res.json(pluginConf)
            break
    }
}

/**
 * Store  the Plugin conf to its configuration file (i.e. .../<plugin>/conf/<plugin>.js)
 * @param  {JSON}       conf     Plugin Configuration
 * @param  {Function}     callback
 */
exports.saveConfService = function(jsonConf, callback){

    let filename     = undefined
      , data         = undefined
      
    // PluginManager calls saveConfService without passing the conf
    if(typeof jsonConf === 'function'){
        callback = jsonConf
        jsonConf = pluginConf
    }

    // Assert jsonConf
    if(!jsonConf || jsonConf === null) return callback(new Error('--->saveConfService: empty Configuration'))
    if(logger.isDebugEnabled()) logger.debug('--->saveConfService: conf:' + JSON.stringify(jsonConf))

    // Backup the current plugin conf.in the server's backup dir and store the new one in the plugin's /conf dir
    async.series([
        // Task 1: Backup the current plugin conf.
        function(nextTask){
            filename = npsConf.backupDir + FILE_SEP + dateFormat.asString('yyyy-MM-dd_hh-mm-ss', new Date()) + '_configurationOf_' + PLUGIN_NAME + '.js' // build the Backup filename
            if(logger.isDebugEnabled()) logger.debug('--->saveConfService: Backuping plugin configuration to ' + filename)
            data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t') // build the plugin conf content

            // Write data to the Plugin configuration file
            fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
                if(err) logger.error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message) // log error & continue
                nextTask()
            })
        },
        // Task 2: Save the new plugin conf.
        function(nextTask){
            pluginConf = jsonConf // we should have here a proper conf, so overwrite the current pluginConf file
            filename = path.resolve(__dirname + '/conf/' + PLUGIN_NAME + '.js') // build the plugin configuration filename
            if(logger.isDebugEnabled()) logger.debug('--->saveConfService: Writing plugin configuration to ' + filename)
            data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t') // build the plugin conf content

            // Write data to Plugin configuration file
            fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
                if(err) nextTask(new Error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message))
                else nextTask()
            }) 
        }
    ],
    function(err){
        callback(err)
    })
}

/**
 * Process a HTTP GET Ping Request
 * 
 * @param  {Request}      req Request Object
 * @param  {Response}     res Response Object
 */
exports.ping = function(req, res){

    let lp = undefined

    // Add Context to the response instance
    res._Context = {};

    // Add a Request Id to the Context
    res._Context.reqId = utils.buildReqId(new Date())
    
    lp    = '--->ping: Req. id=' + res._Context.reqId + ', '
     
    logger.info(lp)

    // Call the ping service
    exports.pingService(res, function(err, result){
        if(err){
            logger.error(lp + 'oxs-office-server is not running, Reason: ' + + err.message)
            return res.status(404).end(lp + 'oxs-office-server is not running, Reason: ' + + err.message)
        }            
        logger.info(lp + 'over!')
        return res.send(result)
    })
}

/**
 * ping 
 *
 * @param  {Response}     res Response Object
 * @param  {Function}     callback(err,result)
 */
exports.pingService = function(res, callback){

    let self    = this
      , lp      = '--->pingService: Req. id=' + res._Context.reqId + ', '

    if(!self._initialized) return callback(new Error(PLUGIN_NAME + ' plugin is not initialized. ' + (initError ? initError.message:'')))

    res._Context.req = {'api': pluginConf.office.uris.ping, httpMethod: 'GET'}
    
    res._Context.req.httpMethod = res._Context.req.httpMethod || 'GET'
    exports.processService(res, function(err, result){
        if(err){
            logger.error(lp + 'oxs-office-server Error, HTTP Status Code: ' + (err.statusCode || err.code) + ', Reason: ' + err.message)
            return callback(new Error('oxs-office-server Error, HTTP Status Code: ' + (err.statusCode || err.code) + ', Reason: ' + err.message))
        }
        logger.info(lp + 'Result: ' + JSON.stringify(result))
        return callback(null, result||'')
    })
}

/**    
 * Office document transform service
 * @param  {Request}      req Request Object
 *                           req.body:{
 *                              'input':{
 *                                  'filename': <Fully Qualified input filename>
 *                                  'buffer': <input data as buffer>
 *                                  'type': Input file type like 'doc', 'docx', 'odt', 'rtf' or 'xls', 'xlsx', 'ods', 'csv'. Optional when filename parameter is given. Default is to take it from file extn
 *                                  'readOptions':{   
 *                                      'password': <Password for encrupted documents>
 *                                  }
 *                              },
 *                              'output':{
 * 									'mode': 'buf' |'file'. Default is 'file'
 *                                  'type': Output type like 'pdf', 'tif'. Default is 'pdf'
 *                                  'path': <FQ output path> when mode is 'file'
 *                                  'transformOptions':{     // libre office filter options given as key value pairs 
 *                                      'EncryptFile':          true | false,
 *                                      'DocumentOpenPassword': <Document Open Password>,
 *                                      'UserPassword':         <User Permissions password>,
 *                                      'SelectPdfVersion':       <0: standard, 1: PDF/A>
 *                                      ....
 *                                  }
 *                              },
 *                              // Exec options to use while invoking office CLI
 *                              'execOptions':{   // Refer to https://nodejs.org/api/child_process.html#child_processexecfilefile-args-options-callback
 *                                  'timeout': <Timeout in milliseconds before killing the process.>. Default is 60 seconds, If value is 0, process runs indefinitely and will NOT be killed automatically if in case it is hanging.
 *                                  //env: { ...process.env, LANG: 'en_US.UTF-8' },
 *                              },
 *                              // number of times to retry the office conversion 
 *                              'retry':{
 *                                  'count': <number of times to retry>,
 *                                  'interval': <Delay in milliseconds between retry attempts>
 *                              },
 *                              engine: 'libreoffice' | 'msoffice'. Default is 'libreoffice'
 *                           }
 * @param  {Response}     res Response Object
 * On Error:
 *  Error message sent as string 
 * On Suceess:
 *  For buffer output, output buffer piped to response object 
 *  For file output, the following JSON returned
 *      {
 *          filename: <FQ output filename>
 *      }
 */ 
exports.transform = function(req, res){	
    
    let lp = undefined

	// Add Context to the response instance
	res._Context = {};

	// Add a Request Id to the Context
    res._Context.reqId = utils.buildReqId(new Date())
    
    lp = '--->transform: Req. id=' + res._Context.reqId + ', '
    logger.info(lp + 'Query Parms: ' + JSON.stringify(req.query) + ', Path Parms: ' + JSON.stringify(req.params) + ', Req.body: ' + JSON.stringify(req.body) + (req.headers['content-type'] ? ', Content-type: ' + req.headers['content-type'] :'') )

    res._Context.o2a = {'parms': req.body }

	logger.debug(lp + 'Parms: ' + JSON.stringify(res._Context.o2a.parms))

	// Call the transform service
	exports.transformService(res, function(err, result){
		if(err){
			logger.error(lp + 'Reason: ' + err.message + ', Parms: ' + JSON.stringify(res._Context.o2a.parms))
			return res.status(404).end(lp + 'Reason: ' + err.message)
        }	
        
        if(Buffer.isBuffer(result)) {
            res._Context.o2a.buffer = result
            _sendBufferResponse(res, function(err){
                if(err) logger.error(lp + 'Error when streaming output buffer, ' + err.message)
                logger.info(lp + 'over!. Result: <Buffer>, length: ' + Buffer.byteLength(result))
                return 
            })
            return 
        }

		logger.info(lp + 'over!. Result: '+ JSON.stringify(result))
		return res.json(result)
	})
}

/**    
 * Office document transform service
 * @param    {JSON}          res  Object
 *                           res._Context.o2a.parms:{
 *                              'input':{
 *                                  'filename': <Fully Qualified input filename>
 *                                  'buffer': <input data as buffer>
 *                                  'type': Input file type like 'doc', 'docx', 'odt', 'rtf' or 'xls', 'xlsx', 'ods', 'csv'. Optional when filename parameter is given. Default is to take it from file extn
 *                                  'readOptions':{   
 *                                      'password': <Password for encrupted documents>
 *                                  }
 *                              },
 *                              'output':{
 * 									'mode': 'buf' |'file'. Default is 'file'
 *                                  'type': Output type like 'pdf', 'tif'. Default is 'pdf'
 *                                  'path': <FQ output path> when mode is 'file'
 *                                  'transformOptions':{     // libre office filter options given as key value pairs 
 *                                      'EncryptFile':          true | false,
 *                                      'DocumentOpenPassword': <Document Open Password>,
 *                                      'UserPassword':         <User Permissions password>,
 *                                      'SelectPdfVersion':       <0: standard, 1: PDF/A>
 *                                      ....
 *                                  }
 *                              },
 *                              // Exec options to use while invoking office CLI
 *                              'execOptions':{   // Refer to https://nodejs.org/api/child_process.html#child_processexecfilefile-args-options-callback
 *                                  'timeout': <Timeout in milliseconds before killing the process.>. Default is 60 seconds, If value is 0, process runs indefinitely and will NOT be killed automatically if in case it is hanging.
 *                                  //env: { ...process.env, LANG: 'en_US.UTF-8' },
 *                              },
 *                              // number of times to retry the office conversion 
 *                              'retry':{
 *                                  'count': <number of times to retry>,
 *                                  'interval': <Delay in milliseconds between retry attempts>
 *                              },
 *                              engine: 'libreoffice' | 'msoffice'. Default is 'libreoffice'
 *                           }
 * @param    {Function}      callback    callback(err)
 * On Suceess:
 * For buffer output:
 *  res._Context.o2a.result = <Buffer >
 * For file output
 *  res._Context.o2a.result={
 *      filename: <FQ output filename>
 *  }
 */    
exports.transformService = function(res, callback){

    let self        = this 
      , parms       = res._Context.o2a && res._Context.o2a.parms      
      , logParms    = undefined
      , bufferLen   = undefined
      , mode    	= parseInt('0777',8)
      , lp          = '--->transformService: Req. id=' + res._Context.reqId + ', '

    if(!self._initialized) return callback(new Error(PLUGIN_NAME + ' plugin is not initialized. ' + (initError ? initError.message:'')))  // 3.0.0
    
    res._Context.tempDir = path.join(npsTempDir, res._Context.reqId)

    bufferLen= (parms.input && parms.input.buffer) ? Buffer.byteLength(parms.input.buffer): undefined
    logParms = {...parms,
        input: {...parms.input, bufferLen:bufferLen, buffer: undefined}
    }

    if(logger.isDebugEnabled()) logger.debug(lp + 'Starting to process the request, parms: ' + JSON.stringify(logParms))

    res._Context.req        = res._Context.req    || {}
    res._Context.req.api    = 'transform'
    res._Context.req.httpMethod = 'POST'

	async.series([
        // assert Parms
		function(nextTask){
            _assertParms(res, nextTask)
		},        
		// Create the temp dir if input buffer is given
		function(nextTask){
            if(parms.input.buffer || parms.output.mode === exports.OP_MODE_BUF){
				if(logger.isDebugEnabled()) logger.debug(lp + 'Create a directory: ' + res._Context.tempDir)            
				return utils.mkDir(res._Context.tempDir, mode, nextTask)				
			}
			return nextTask() 
		},
		// Write input buffer to a file 
		function(nextTask){
            if(parms.input.buffer) return _buffer2file(res, nextTask)
            return nextTask() 
		},
		// Call oxs-office-server
		function(nextTask){
            res._Context.o2a.reqParms = {...parms}
            res._Context.o2a.reqParms.input.buffer = undefined  // since input.filenae is set delete input.buffer
            exports.processService(res, nextTask)
		},
		function(nextTask){
            if(parms.output.mode === exports.OP_MODE_FILE) return nextTask()
            res._Context.o2a.filename   = res._Context.o2a.result.outputFilename
            _file2buffer(res, function(err){
                if(err) return nextTask(err)
                res._Context.o2a.result = res._Context.o2a.buffer
                nextTask()
            })
		}
	],
	function(err){
        if(err) return callback(err)
        if(logger.isDebugEnabled())  logger.debug(lp + 'Finsihed processing the request, result: ' 
            + (Buffer.isBuffer(res._Context.o2a.result) ? '<Buffer>, length:' + Buffer.byteLength(res._Context.o2a.result) : JSON.stringify(res._Context.o2a.result)))
		callback(err, res._Context.o2a.result)
	})    
}

/**
 * Base service
 *
 * @param {Response}    res Response Object
 *                        res._Context.o2a.reqParms:{
 *                        }
 * @param  {Function}     callback(err)
 *
 */
 exports.processService = function(res, callback){ 

    let options         = undefined
      , retryOptions    = undefined
      , serviceURI      = undefined
      , self            = this
      , parms           = res._Context.o2a.reqParms    
      , lp              = '--->processService: Req. id=' + res._Context.reqId + ', '

    if(!self._initialized) return callback(new Error(PLUGIN_NAME + ' plugin is not initialized. ' + (initError ? initError.message:'')))
    
    options = utils.clone(pluginConf.office.server)
    
    options.path = pluginConf.office.server.context
    options.path += pluginConf.office.uris[res._Context.req.api]
    serviceURI = options.path

    options.reqId = res._Context.reqId
    if(res._Context.retryOptions && res._Context.retryOptions[SERVER_NAME]) retryOptions = res._Context.retryOptions[SERVER_NAME]
    else if(res._Context.retryOptions && res._Context.retryOptions['default']) retryOptions = res._Context.retryOptions['default']
    else retryOptions = pluginConf.office.server.retryOptions 

    options.retryOptions = retryOptions     // overwrite retryoptions in givne server env.variable with passed retryOptions // 3.0.4

    switch(res._Context.req.httpMethod.toUpperCase()){
        case 'POST':
            if(logger.isDebugEnabled()) logger.debug(lp + 'API: ' + res._Context.req.api + ', options: '+ JSON.stringify(options) + ', parms: ' + JSON.stringify(parms))
            res._Context.startTime = new Date()
            httpUtils.postURL(options, parms, retryOptions, function(err, result){
                _logServiceRespTime(res, serviceURI)

                if(result){
                    if(logger.isDebugEnabled()) logger.debug(lp + 'Result: ' + JSON.stringify(result))
                    if(typeof result === 'string') result = utils.parseJSON(result) || result     
                    if(result && logger.isDebugEnabled()) logger.debug(lp + 'Result(JSON): ' + JSON.stringify(result))		
                    res._Context.o2a.result = result			
                }
                else if(logger.isDebugEnabled()) logger.debug(lp + 'Result: undefiend')

                if(err) return callback(new Error(err.message + ', API: ' + res._Context.req.api + ', Req. options: ' + JSON.stringify(options) + ', parms: ' + JSON.stringify(parms) + ', result: ' + JSON.stringify(result)))
                return callback(null, result)
            })         
            break;
        case 'GET':
            options.path += "?o2a="+JSON.stringify(parms)
            if(logger.isDebugEnabled()) logger.debug(lp + ' API: ' + res._Context.req.api + ', options: '+ JSON.stringify(options) + ', parms: ' + JSON.stringify(parms))
            res._Context.startTime = new Date()
            httpUtils.getURL(options, retryOptions, function(err, result){
                _logServiceRespTime(res, serviceURI)
                
                if(err) return callback(new Error(err.message + ', API: ' + res._Context.req.api + ', Req. options: ' + JSON.stringify(options) + ', parms: ' + JSON.stringify(parms)))

                if(logger.isDebugEnabled()) logger.debug(lp + 'Result: ' + JSON.stringify(result))
                if(result && typeof result === 'string') result = utils.parseJSON(result)                   // result is a JSON string
                if(logger.isDebugEnabled()) logger.debug(lp + 'Result(JSON): ' + JSON.stringify(result))

                res._Context.o2a.result = result
                return callback(null, result)
            })         
            break;   
        default:     
            return callback(new Error(lp +' ' + res._Context.req.httpMethod + ' not supported')) 
    }
    return
} 

/************ OPTIONAL PUBLIC FUNCTIONS ***********/
/**
 * Set Log Level
 * @param {String} logLevel Log Level. values:DEBUG|INFO|WARN|ERROR 
 */
exports.setLogLevel = function(logLevel){
    pluginConf.log.level = logLevel || 'INFO'
    utils.log4jsSetLogLevel(logger, pluginConf.log.level)
}

/**
 * Get Log Level
 * @return {String} Log Level. values:DEBUG|INFO|WARN|ERROR
 */
exports.getLogLevel = function(){
    return pluginConf.log.level
}

/**
 * Enable the given route
 * @param  {Request}     req
 *                      req.body = {
 *                          "module":"sample", 
 *                          "route": { 
 *                              "path": "/path/to/the/service",
 *                              "method":     Optional. Values are "get|post|dw|queue",
 *                              "service":     "version"
 *                          }
 *                      }
 * @param  {function}     callback callback(err)
 */
exports.enableRoute = function(req, callback){

    let self = this

    if(logger.isDebugEnabled()) logger.debug('--->enableRoute: Route=' + JSON.stringify(req.body.route))

    if(!req.body.route) return callback(new Error('route parameter not given'))

    req.body.route.method = req.body.route.method || 'get'

    // Enable the route
    switch(req.body.route.method.toLowerCase()){
        case 'dir': // Enable Dir Route
            pluginManager.callPluginService({name:'oxsnps-dirwatcher',service:'addListenersService'}, [req.body.route], callback)
            break
        case 'queue': // Enable Queue Route
               // Delete Queued Jobs before enabling route 
            pluginManager.callPluginService({name:'oxsnps-ibmmq', service:'addListenersService'}, [req.body.route], callback)
            break
        default:
            // Enable HTTP Route
            expressAppUtils.enableRoute(self, req.body.route, callback)
    }
} 

/**
 * Disable the given route
 * @param  {Request}     request Request json has following structure.
 *                         req.body = {
 *                          "module":"sample", 
 *                          "route": { 
 *                              "path": "/path/to/the/service",
 *                              "method":     Optional. Values are "get|post|dw|queue"
 *                               }
 *                         }
 * @param  {function}     callback  callback(err)
 */
exports.disableRoute = function(req, callback){
    let self = this

    if(logger.isDebugEnabled()) logger.debug('--->disableRoute: Route=' + JSON.stringify(req.body.route))
    
    if(!req.body.route) return callback(new Error('route parameter not given'))

    req.body.route.method = req.body.route.method || 'get'

    // Disable the route
    switch(req.body.route.method.toLowerCase()){
        // Disable Dir Route
        case 'dir': 
            pluginManager.callPluginService({name:'oxsnps-dirwatcher',service:'removeListenersService'}, [req.body.route], callback)
            break
        case 'queue':
            // Disable Queue Route
            pluginManager.callPluginService({name:'oxsnps-ibmmq', service:'removeListenersService'}, [req.body.route], callback)
            break
        default:
            // Disable HTTP Route
            expressAppUtils.disableRoute(self, req.body.route, callback)
    }
} 

/**************** PRIVATE FUNCTIONS ***************/
function _assertParms(res, callback){

    let lp    = '--->_assertParms: Req. id=' + res._Context.reqId + ', '  
      , parms = res._Context.o2a && res._Context.o2a.parms

    if(logger.isDebugEnabled()) logger.debug(lp + 'Validating the parameters...')
    if(!parms) return callback(new Error('Invalid request, parameters are missing'))      

    if(!parms.input) return callback(new Error('Invalid request, parms.input parameter is missing'))      

    // Assert input file
    if(!parms.input.filename && !parms.input.buffer)
        return callback(new Error('Invalid request, neither parms.input.filename nor parms.input.buffer parameter is given'))      

    if(parms.input.buffer && !parms.input.type)
        return callback(new Error('Invalid request, parms.input.type parameter is missing'))      

    // For buffer output setup a temp filename since oxs-office-server has only filein-fileout API
    if(parms.input.buffer){
        res._Context.o2a.buffer     = parms.input.buffer
        parms.input.filename        = res._Context.o2a.filename   = path.join(res._Context.tempDir, res._Context.reqId + '_input.' + parms.input.type)    // input file name to oxs-office-server
    }
    //if(!parms.output) return callback(new Error('Invalid request, parms.output parameter is missing'))      

    parms.output        = parms.output          || {}
    parms.output.mode   = parms.output.mode     || exports.OP_MODE_BUF
    parms.output.type   = parms.output.typemode || 'pdf'
    parms.output.transformOptions = parms.output.transformOptions || {}

    // Assert output path in case of fileoutput
    if(parms.output.mode === exports.OP_MODE_FILE && !parms.output.path)
        return callback(new Error('Invalid request, parms.output.path parameter is missing'))      

    if(parms.output.mode === exports.OP_MODE_BUF) parms.output.path = path.join(res._Context.tempDir)   // output file path to oxs-office-server

    if(parms.output.path) parms.output.path = path.resolve(parms.output.path)

    // Exec options to use while invoking office CLI
    parms.execOptions         = parms.execOptions || pluginConf.office.execOptions  || {}
    parms.execOptions.timeout = parms.execOptions.timeout || 60000    // Timeout in milliseconds before killing the process.

    // number of times to retry the office conversion 
    parms.retry           = parms.retry           || pluginConf.office.retry || {} 
    parms.retry.count     = parms.retry.count     || 1      // <number of times to retry conversion>, count =1 no retry
    parms.retry.interval  = parms.retry.interval  || 1000   // <Delay in milliseconds between retry attempts>

    parms.engine    = parms.engine  || pluginConf.engine || 'libreoffice'
    return callback()
}

// write buffer to a file
function _buffer2file (res, callback){

	let lp  = '--->_buffer2file: Req. Id=' + res._Context.reqId + ', ' 
    
    if(logger.isDebugEnabled()) logger.debug(lp + 'Writing buffer of length: ' + Buffer.byteLength(res._Context.o2a.buffer) + ' to a file: ' + res._Context.o2a.filename)
    fs.writeFile(res._Context.o2a.filename, res._Context.o2a.buffer, (err) => {
        if(err) return callback(new Error('Unable to write buffer of length: ' + Buffer.byteLength(res._Context.o2a.buffer) + ' to the file: ' + res._Context.o2a.filename + ', ' + err.message))

        if(logger.isDebugEnabled()) logger.debug(lp + 'Buffer written to the file: ' + res._Context.o2a.filename)
        return callback()
    })
}

// Read file into a buffer
function _file2buffer (res, callback){

    let lp  = '--->_file2buffer: Req. Id=' + res._Context.reqId + ', ' 
   
    if(logger.isDebugEnabled()) logger.debug(lp + ', Reading a file: ' + res._Context.o2a.filename)
    fs.readFile(res._Context.o2a.filename, function(err, data){
        if(err) return callback(new Error('Unable to read from the file: ' + res._Context.o2a.filename + ', ' + err.message))
        res._Context.o2a.buffer = data 
        if(logger.isDebugEnabled()) logger.debug(lp + 'Read the file: ' + res._Context.o2a.filename + ' into a buffer. Bufer length: ' + Buffer.byteLength(res._Context.o2a.buffer))
        return callback()
    })
}

// Send buffer as response
function _sendBufferResponse (res, callback){

	let callbackInvoked = false
	  , bufStream		= undefined
      , lp              = '--->_sendBufferResponse: Req. Id=' + res._Context.reqId + ', ' 

	// Read the PDF file
	try{
		bufStream = new bufferStream(res._Context.o2a.buffer)
  
		if(res._Context.header && res._Context.header.contentType) res.setHeader('Content-Type', res._Context.header.contentType)
        else res.setHeader('Content-Type', 'application/pdf')
        if(res._Context.filename) res.setHeader('Content-Disposition', 'filename=' + res._Context.filename)	// V2.4.6

		if(logger.isDebugEnabled()) logger.debug(lp + 'Streaming buffer...')
	    
		bufStream.on('end', function(){
			logger.debug(lp +'Buffer stream "end" event called.')
			if(!callbackInvoked){ 
				callbackInvoked = true
				callback()
			}			
		});
		bufStream.on('error', function(err){
			if(!callbackInvoked){ 
				callbackInvoked = true
				callback(err)
			}			
			else{			
				logger.error(lp + 'Buffer stream "error" event called. ' + err.message)
			}
		});
		// There is no need for error event handlers for response object since error event is not emitted.
		// Refer to below text from https://www.bennadel.com/blog/2823-does-the-http-response-stream-need-error-event-handlers-in-node-js.htm
		// HTTP response stream will never emit an error event; or, at the very least, that such an error 
		// event will never crash the server. So, while you have to bind an "error" event handler to just about
		// every other streams in Node.js, it would seem that you can safely assume the HTTP response stream will
		// never error, no matter what happens. And, that even if the response stream closes, you can continue 
		// to write to it / pipe to it without consequence.	 
		/*
		res.on('error', function(err){
			logger.error();
		});
		*/
		bufStream.pipe(res)

	}
	catch(err){
		callback(err)
	}
}

function _parseEnvs(res, service){

    let envvar  = undefined
      , context = undefined
      , lp      = '--->_parseEnvs: '
      

    /**
     * Process pluginConf.office value
     *
     *     "office":{
     *        "server":{
     *            "hostname":    "localhost",
     *            "port": 80,
     *            "context": "office", // context will be prepended to the URL as http://<server>:<port>/<context>/....
     *            "httptimeout": 30000 // request Timeout in milliseconds; default is 10000 milli seconds
     *        },
     *        'uris': {
     *            "transform":     "/services/office/transform",
     *            "ping":         "/services/office/ping"
     *        }
     *    }
     */
    pluginConf.office    = pluginConf.office || {} 

    // If pluginConf.office.server is a string, parse it (i.e. "{\"hostname\":\"localhost\", \"port\":80, \"httptimeout\":10000,  \"context\":\"chrome\"}",)
    // Envar: <OXSNPS_SERVER_PREFIX> + "OFFICE_SERVER"
    if(typeof pluginConf.office.server === 'string'){
        envvar = pluginConf.office.server
        if(!process.env[envvar]) return new Error('Missing environment variable: ' + envvar)

        pluginConf.office.server = utils.parseJSON(process.env[envvar])
        if(pluginConf.office.server === null) return new Error('Unable to parse environment variable: ' + envvar + ', value: ' + process.env[envvar])
    } 

    pluginConf.office.server = pluginConf.office.server || {}

    // Set protocol
    pluginConf.office.server.protocol = pluginConf.office.server.protocol || 'http:'

    // Set hostname
    pluginConf.office.server.hostname = pluginConf.office.server.hostname || 'localhost'

    // Set Port
    pluginConf.office.server.port = pluginConf.office.server.port || 80

    // Request Timeout in milliseconds, default is 10000
    pluginConf.office.server.httptimeout = pluginConf.office.server.httptimeout || 60000
    
    pluginConf.office.server.respEncoding = pluginConf.office.server.respEncoding ||  "utf-8"
    
    // Ensure that context is having leading slash
    context = pluginConf.office.server.context || ''
    if(context.length > 0 && !context.startsWith('/')) context = '/' + context
    pluginConf.office.server.context = context
    
    pluginConf.office.uris    = pluginConf.office.uris || {}    

    // number of times to retry the office conversion 
    if(typeof pluginConf.office.retry === 'string'){
        envvar = pluginConf.office.retry
        if(!process.env[envvar]) return new Error('Missing environment variable: ' + envvar)

        pluginConf.office.retry = utils.parseJSON(process.env[envvar])
        if(pluginConf.office.retry === null) return new Error('Unable to parse environment variable: ' + envvar + ', value: ' + process.env[envvar])
    } 

    // number of times to retry the office conversion 
    pluginConf.office.retry         = pluginConf.office.retry       	|| {}
    pluginConf.office.retry.count   = pluginConf.office.retry.count 	|| 5            // <number of times to retry conversion>
    pluginConf.office.retry.interval= pluginConf.office.retry.interval	|| 10000        // <Delay in milliseconds between retry attempts>

    // Options to use with exec command when invoking libreoffice headless 
    // Refer to https://nodejs.org/api/child_process.html#child_processexecfilefile-args-options-callback                
    if(typeof pluginConf.office.execOptions === 'string'){
        envvar = pluginConf.office.execOptions
        if(!process.env[envvar]) return new Error('Missing environment variable: ' + envvar)

        pluginConf.office.execOptions = utils.parseJSON(process.env[envvar])
        if(pluginConf.office.execOptions === null) return new Error('Unable to parse environment variable: ' + envvar + ', value: ' + process.env[envvar])
    } 
    pluginConf.office.execOptions           = pluginConf.office.execOptions         || {}
    pluginConf.office.execOptions.timeout   = pluginConf.office.execOptions.timeout || 60000

    if(logger.isDebugEnabled()) logger.debug(lp + 'pluginConf.office: ' + JSON.stringify(pluginConf.office))

    return 
}

function _logServiceRespTime(res, service){

    let respTime     = 0
    
    // Check resp. time only if maxresptime is defined
       if(pluginConf.maxresptime){
        respTime = utils.dateDiff('x', res._Context.startTime, new Date()) // 'x' means returns diff. in ms

           if(respTime > pluginConf.maxresptime * 1000){
            logger.warn('--->' + service + ': Req.Id=' + res._Context.reqId + ', ' + ' Resp. Time: ' + respTime/1000 + 's, Max. allowed: '+ pluginConf.maxresptime + 's. Average Load: ' + os.loadavg())
           }
       }
}

