/**-----------------------------------------------------------------------------
 * oxsnps.js: 	
 *
 * Author    :  AFP2web Team
 * Copyright 	:  (C) 2019-2020 by Maas Holding GmbH
 * Email		:  support@maas.de
 * 
 * History
 * V4.0.5	 11.12.2024  	OXS-16233: Extended to convert "addDefaultArgs" parameter value to boolean data type (For Delticom: recognise incoming electronic invoices)
 * V4.0.4	 09.10.2023  	OTS-3491: Added files for Swagger Interface
 * V4.0.2	 14.10.2022  	Added the qpdf command as 1st arg if no specified
 * V4.0.1	 27.07.2022  	OXS-13569: Pack the QPDF tool in the oxsnps-qpdf plugin
 * V4.0.0	 20.05.2022  	OTS-3271: EXtended to adopt to oxsnps-core >=5.0.7 that disables routes if plugin can not be initialized
 * V3.0.2    14.03.2022    	OTS-3256: Extended to add plugin defined default qpdf options in addition to the caller qpdf options.  Default qpdf options are defined in plugin conf file through in processe env. variable.
 *                                    This is done to suppress qpdf warning that takes more space in the log files. Default options are ["--no-warn", "--warning-exit-0"]
 *                                    Calling plugin can turn off this feature by setting res._Context.qpdf.addDefaultArgs=false
 *           14.03.2022    	OTS-3259: Extended to handle input filenames having a single quote in it. Escape input filenames by surrounding input filenames with double quotes. If the input filename has double quotes, escape it like "\""
 * V3.0.0    09.04.2020    	OTS-2727: Extended oxsnps-qpdf to capture stdout/stderr messages
 * V2.0.0    26.02.2020    	OTS-2699: Extended to use oxsnps-core@4.1.0
 *
 * V1.0.0    04.10.2019    	Initial release
 *----------------------------------------------------------------------------*/
'use strict'

/**
 * Variable declaration
 * @private
 */
var packageJson		= require(__dirname + '/package.json')
  , PLUGIN_NAME		= packageJson.name
  , PLUGIN_VERSION	= packageJson.version // Get Server Version from package.json
  , async			= require('async')  
  , fs				= require('fs')
  , log4js			= require('log4js')
  , path			= require('path')
  , FILE_SEP 		= path.sep

  , npsServer		= require('oxsnps-core/server')
  , npsConf 		= npsServer.npsConf
  , npsLogDir 		= npsConf.logDir

  , pluginConf		= require(__dirname + '/conf/' + PLUGIN_NAME + '.js')
  , pluginManager 	= require('oxsnps-core/pluginManager')  
  , dateFormat 		= require('oxsnps-core/helpers/date_format')
  , utils 			= require('oxsnps-core/helpers/utils')
  , httpUtils		= require('oxsnps-core/helpers/httpUtils')
  , expressAppUtils = require('oxsnps-core/helpers/expressAppUtils')
  , initError       = undefined	// V4.0.0   // to be used when pingservice is implmented

  //, qpdfTool		= require('node-qpdf')

// Set plugin name, version and longDesc in pluginConf
pluginConf.module 	= PLUGIN_NAME 
pluginConf.version 	= PLUGIN_VERSION 
pluginConf.longDesc = packageJson.description
pluginConf.log 		= pluginConf.log || {}

// Export the Plugin's Configuration Express App
exports.pluginConf 	= pluginConf
exports.router      = undefined     // V200 Change

// Get the Logger
let logger = log4js.getLogger(PLUGIN_NAME)
utils.log4jsSetLogLevel(logger, (pluginConf.log.level || 'INFO'))

/**************** PUBLIC FUNCTIONS ****************/
/**
 * Initialize Plugin
 * @param  {function} 	callback  callback(err, routes)
 */
exports.initialize = function(callback){

	let self = this
	  , mode = parseInt('0777',8)
	  , appPluginConf	= (npsConf.plugins && npsConf.plugins[PLUGIN_NAME] ? npsConf.plugins[PLUGIN_NAME] : {})	
      
    // V4.0.0 Begin
    initError = undefined

    function _handleError(err)
    {
        initError = err
        self._initialized = false 
        return callback(err, pluginConf.routes)
    }          
    // V4.0.0 End
      
	if(logger.isDebugEnabled()) logger.debug('--->initializing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION )

	if(self._initialized) return callback()

    pluginConf.defaultArgs = pluginConf.defaultArgs || ["--no-warn", "--warning-exit-0"]
        
	if(typeof pluginConf.defaultArgs === 'string'){
		let envvar = pluginConf.defaultArgs
		if(process.env[envvar]){
            pluginConf.defaultArgs = utils.parseJSON(process.env[pluginConf.defaultArgs])
            if(!pluginConf.defaultArgs) return _handleError(new Error('Unable to parse environment variable: ' + envvar + ', value: ' + process.env[envvar]))
        }
        else pluginConf.defaultArgs = ["--no-warn", "--warning-exit-0"]
	} 

    if(pluginConf.defaultArgs && !Array.isArray(pluginConf.defaultArgs)) pluginConf.defaultArgs = [pluginConf.defaultArgs]

	// Merge application specific configuration with plugin configuration
	exports.pluginConf = pluginConf = utils.mergeJSON(pluginConf, appPluginConf)

	// Add a router for the plugin routes and its middlewares
	//expressAppUtils.enableRoutes(self, pluginConf.routes, function(err){ // V200 Change
	expressAppUtils.setupRouter(self, function(err, router){ // V200 Change
		if(err)	return _handleError(err)
		exports.router = router // V200 Change
		self._initialized = true
		logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' initialized')
		//callback(null, pluginConf.routes)
		callback(null, {'routes': pluginConf.routes, 'swagger':{"name": PLUGIN_NAME, "url": '/' + PLUGIN_NAME + '/api/rest.yaml'}})	// V4.0.4

	})
}

/**
 * Activate Plugin. This function is called from pluginManager once all the enabled plugins have been initialize
 * @param  {function} 	callback  callback(err)
 */
exports.postInitialize = function(callback){
	callback()
}

/**
 * Finalize Plugin
 * @param  {function} 	callback callback(err) 
 */
exports.finalize = function(callback){

	let self = this
	  , req  = {}

	if(logger.isDebugEnabled()) logger.debug('--->finalizing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION )
	if(!self._initialized) return callback()

	async.series([
		// Disable all routes
		function(nextTask){
			req.body = {'module': PLUGIN_NAME}
			async.each(pluginConf.routes, 
				function(route, next){
					req.body.route = route
					exports.disableRoute(req, next)
				},
				function(err){
					return nextTask(err)
				}
			)
		}
	],
	function(err){
		if(err) return callback(err)
		self._initialized = false
		logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' finalized')
		callback()
	})

}

/**
 * Return Plugin Version
 * @param  {function} 	callback callback(err,version)
 */
exports.getVersion = function(callback){
	return callback(null, PLUGIN_VERSION)
}

/**
 * Process a HTTP Version Request
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.version = function(req, res){

	let data = undefined

	// Add Context to the response instance
	res._Context = {}

	// Add the Current Date to the Context
	res._Context.date = new Date()

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date)
	logger.info('--->version: Req. Id=' + res._Context.reqId)

	data = '<h3>' +  PLUGIN_NAME + ' v' + PLUGIN_VERSION + '</h3>'
	res.end(data)

	logger.info('--->version: Req. Id=' + res._Context.reqId + ', sent back ' +
				PLUGIN_NAME + ' v' + PLUGIN_VERSION + ', over')
	return
}

/**
 * Process a HTTP Configuration Request 
 * 	Either as HTTP GET request:
 * 	 	GET http://localhost:1029/services/<plugin>/conf 							--> To retrieve the configuration
 * 	 	GET http://localhost:1029/services/<plugin>/conf?save=true&conf="{...}" 	--> To pass and save the configuration
 * 	or as HTTP POST request:
 * 	 	POST http://localhost:1029/services/<plugin>/conf 						--> To pass and save the configuration
 * 	 	req.body={
 * 	 		"conf": {...}
 * 	 	}
 * 	
 * @param {Request} 	req Request Object
 *                      req.query:{ // for HTTP GET request:
 *                      	"save":"false",
 *                      	"conf":"{<IMPORTANT: COMPLETE JSON Conf. as String>}"
 *                      }
 *                      req.body:{ // for HTTP POST request:
 *                      	"conf":{<IMPORTANT: COMPLETE JSON Conf. as JSON Object>}
 *                      }
 * @param {Response}    res Response Object
 */
exports.conf = function(req, res){

	let reqType 	= 'getconf'
	  , jsonConf 	= undefined
	  
	// Add Context to the response instance
	res._Context = {}

	// Add the Current Date to the Context
	res._Context.date = new Date()

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date)
	logger.info('--->conf: Req. id=' + res._Context.reqId)
	
	if(req.method.toUpperCase() === 'POST'){
		logger.info('--->conf: body=' + JSON.stringify(req.body))
		if(!req.body || !req.body.conf) {
			logger.info('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body))
			return res.end('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body))
		}
		reqType = 'saveconf' 		// save request
		jsonConf = req.body.conf 	// set the json conf to be passed to saveConfService
	}
	else if(req.method.toUpperCase() === 'GET'){
		logger.info('--->conf: query=' + JSON.stringify(req.query))
		if(req.query && req.query.save && req.query.save.toLowerCase() === 'true' && req.query.conf)
		{
			reqType = 'saveconf'
			jsonConf = utils.parseJSON(unescape(req.query.conf)) // remove escaped chars and set the json conf to be passed to saveConfService
		} 
	}

	switch(reqType){
		case 'saveconf':
		    exports.saveConfService(jsonConf, function(err){
		    	logger.info('--->conf: Req. id=' + res._Context.reqId + ', over')
				res.json(err ? {'status':'error', 'message': err.message} : pluginConf)
		    }) 
			break
		default:
		case 'getconf':
			logger.info('--->conf: Req. id=' + res._Context.reqId + ', over')
			res.json(pluginConf)
			break
	}
}

/**
 * Store  the Plugin conf to its configuration file (i.e. .../<plugin>/conf/<plugin>.js)
 * @param  {JSON}   	conf     Plugin Configuration
 * @param  {Function} 	callback
 */
exports.saveConfService = function(jsonConf, callback){

	let filename 	= undefined
	  , data 		= undefined
	  
	// PluginManager calls saveConfService without passing the conf
	if(typeof jsonConf === 'function'){
		callback = jsonConf
		jsonConf = pluginConf
	}

	// Assert jsonConf
	if(!jsonConf || jsonConf === null) return callback(new Error('--->saveConfService: empty Configuration'))
	if(logger.isDebugEnabled()) logger.debug('--->saveConfService: conf:' + JSON.stringify(jsonConf))

	// Backup the current plugin conf.in the server's backup dir and store the new one in the plugin's /conf dir
	async.series([
		// Task 1: Backup the current plugin conf.
		function(nextTask){
			filename = npsConf.backupDir + FILE_SEP + dateFormat.asString('yyyy-MM-dd_hh-mm-ss', new Date()) + '_configurationOf_' + PLUGIN_NAME + '.js' // build the Backup filename
			if(logger.isDebugEnabled()) logger.debug('--->saveConfService: Backuping plugin configuration to ' + filename)
			data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t') // build the plugin conf content

			// Write data to the Plugin configuration file
			fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
				if(err) logger.error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message) // log error & continue
				nextTask()
			})
		},
		// Task 2: Save the new plugin conf.
		function(nextTask){
			pluginConf = jsonConf // we should have here a proper conf, so overwrite the current pluginConf file
			filename = path.resolve(__dirname + '/conf/' + PLUGIN_NAME + '.js') // build the plugin configuration filename
			if(logger.isDebugEnabled()) logger.debug('--->saveConfService: Writing plugin configuration to ' + filename)
			data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t') // build the plugin conf content

			// Write data to Plugin configuration file
			fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
				if(err) nextTask(new Error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message))
				else nextTask()
			}) 
		}
	],
	function(err){
		callback(err)
	})
}

/**
 * 
 * HTTP Request for PDF processing
 * HTTP request:
 * http://localhost:1029/services/qpdf/process
 * 	
 * @param	{Request}		req	Request Object
 *								req.body:{
 *									'args':[
 * 										// Reference: http://qpdf.sourceforge.net/files/qpdf-manual.html#ref.using
 * 										<command line options of qpdf>
 *									],
 *                                  addDefaultArgs: true|false. Default is true. true => Add oxsnps-qpdf plugin defined default args, in addition to the caller qpdf args.  false => Do not add default args
 *								}
 * @param	{Response}		res	Response Object
 * On Success: returns a JSON {'code':code, 'stdout':stdout,'stderr':stderr}
 * On Error: returns a JSON {'code':code, 'message':<error message>, stdout':stdout,'stderr':stderr}
 * 
 */
exports.process = function(req, res){

	let ctDate	= new Date()
	  , reqId	= utils.buildReqId(ctDate)
	  , parms	= httpUtils.getRequestParms(req)

	// Build request context
	res._Context = {
		'date': 		ctDate,
		'reqId': 		reqId,
		'route':{
			'path':		req.path,
			'method':	req.method || 'POST'
		},
		'path':			req.path, // url path, ex. services/qpdf/process
		// Create a separate JSON object to store qpdf values so that it does not mingle with other service's context values
		'qpdf': {
            'args':		parms.args || [],
            'addDefaultArgs': (parms.addDefaultArgs !== undefined) ? parms.addDefaultArgs : true  // V3.0.2 OTS-3256
		}
    }
    
	// Process Request
	logger.info('--->process: Req. id=' + res._Context.reqId + ', res._Context.qpdf: ' + JSON.stringify(res._Context.qpdf))
	exports.processService(res, function(err,result){
		if(err){
            logger.error('--->process: Req. id=' + res._Context.reqId + ', ' + err.message + (result ? (', Result: '+JSON.stringify(result)) : ''))
            if(result){
                result.message = err.message
                return res.json(result) // since we have result object, set res status to 200. client will check resutl.stderr/result.code and takes action
            }
			return res.status(404).end(err.message)
		} 
        //res.json({'status': 'success', 'args': res._Context.qpdf.args})
        res.json(result)
		logger.info('--->process: Req. id=' + res._Context.reqId + ', over')
	})
}

/**
 * Process given PDF document
 * @param	{Response}	res			Response Object
 *									res._Context:{
 *										'qpdf':{ 
 *											'args':[
 * 												// Reference: http://qpdf.sourceforge.net/files/qpdf-manual.html#ref.using
 * 												<command line options of qpdf>
 *											],
 *                                          addDefaultArgs: true|false. Default is true. true => Add oxsnps-qpdf plugin defined default args, in addition to the caller qpdf args.  false => Do not add default args
 *										}
 *									}
 * @param	{function}	callback	callback(err,result)
 * result = { "code": 127, "stdout": "", "stderr": "/bin/sh: 1: /usr/lib/node_modules/oxs-qpdf-server/node_modules/oxsnps-qpdf/qpdf/bin/qpdf1: not found\n"}
 * result = { "code": 127, "stdout": "", "stderr": "/usr/lib/node_modules/oxs-qpdf-server/node_modules/oxsnps-qpdf/qpdf/bin/qpdf: error while loading shared libraries: libqpdf.so.28: cannot open shared object file: No such file or directory\n"}
 * result = { "code": 2,   "stdout": "", "stderr": "open /home/mathi/qpdftest/samples/insure1.pdf: No such file or directory\n"}
 * result = { "code": 0,   "stdout": "3\n",  "stderr": ""}	// page count result
 */
exports.processService = function(res, callback){

    let resTmp      = undefined
      , qpdfResult  = undefined
      , addDefaultArgs  = (res._Context.qpdf && res._Context.qpdf.addDefaultArgs !== undefined) ? res._Context.qpdf.addDefaultArgs : true  // V3.0.2
      , self = this

    if(!self._initialized) return callback(new Error(PLUGIN_NAME + ' plugin is not initialized. ' + (initError ? initError.message:'')))  // 2.0.0

    if(addDefaultArgs && typeof(addDefaultArgs) === 'string') addDefaultArgs = utils.stringToBoolean(addDefaultArgs)	// 4.0.5

	if(logger.isDebugEnabled()) logger.debug('--->processService: Req.Id=' + res._Context.reqId + ', res._Context.qpdf:' + JSON.stringify(res._Context.qpdf))
	async.series([
		// Task 1: Assert parameters
		function(nextTask){
			/**
			 * Assert the required attributes
			 * res._Context.qpdf
			 * res._Context.qpdf.args
			 * ...
			 */
			if(!res._Context.qpdf){
				return nextTask(new Error('--->processService: Req.Id=' + res._Context.reqId + ', Missing qpdf in res._Context:' + JSON.stringify(res._Context)))
			}
			if(!res._Context.qpdf.args){
				return nextTask(new Error('--->processService: Req.Id=' + res._Context.reqId + ', Missing args in res._Context.qpdf:' + JSON.stringify(res._Context.qpdf)))
			}
			return nextTask()
		},
		// Task 2: Invoke QPDF
		function(nextTask){

            resTmp = { 
                '_Context': {
                    'reqId': 		res._Context.reqId,
                    'cmd': 		'/bin/sh',
                    "args": 	["-c"]
                }
            }

			// V4.0.1 Begin Add the path to the qpdf command if not specified
			if(res._Context.qpdf.args[0] === 'qpdf'){
				res._Context.qpdf.args[0] = __dirname + '/qpdf/bin/qpdf'
			}
			else{ // Add the qpdf command as 1st arg
				// This must be done so that in future calls, the qpdf command won't have to be specified by the caller
				res._Context.qpdf.args.unshift(__dirname + '/qpdf/bin/qpdf') // V4.0.2
			}
			// V4.0.1 End

            // Add default args
            if(addDefaultArgs) res._Context.qpdf.args = res._Context.qpdf.args.concat(pluginConf.defaultArgs)   // V3.0.2  OTS-3256

            // Surround all string arguments with double quotes. If the string argument has double quote in it, escape it
            res._Context.qpdf.args = res._Context.qpdf.args.map(arg => escapeArg(res, arg))      // V3.0.2  OTS-3259

          	//resTmp._Context.args = resTmp._Context.args.concat(res._Context.qpdf.args) // does not work
            resTmp._Context.args.push(res._Context.qpdf.args.join(' '))
            
            if(logger.isDebugEnabled()) logger.debug('--->processService: Req.Id=' + res._Context.reqId + ', Calling oxnps-spawn.executeService(), Parms:' + JSON.stringify(resTmp._Context))
            pluginManager.callPluginService({'name': 'oxsnps-spawn', 'service':'executeService'}, resTmp, function(err, result){
                qpdfResult = result                
                if(err) return nextTask(new Error('oxsnps-spawn Error: ' + err.message + ', Parms:' + JSON.stringify(resTmp._Context)))
                if(logger.isDebugEnabled()) logger.debug('--->processService: Req. id=' + res._Context.reqId + ', qpdf result: ' + JSON.stringify(qpdfResult))
                return nextTask()
            })
		}
	],
	function(err){
		logger.debug('--->processService: Req. id=' + res._Context.reqId + ((err) ? ' failed. Reason: ' + err.message : ' is over'))
		return callback(err, qpdfResult)
	})
}

/************ OPTIONAL PUBLIC FUNCTIONS ***********/
/**
 * Set Log Level
 * @param {String} logLevel Log Level. values:DEBUG|INFO|WARN|ERROR 
 */
exports.setLogLevel = function(logLevel){
	pluginConf.log.level = logLevel || 'INFO'
	utils.log4jsSetLogLevel(logger, pluginConf.log.level)
}

/**
 * Get Log Level
 * @return {String} Log Level. values:DEBUG|INFO|WARN|ERROR
 */
exports.getLogLevel = function(){
	return pluginConf.log.level
}

/**
 * Enable the given route
 * @param  {Request} 	req
 *					  req.body = {
 *					  	"module":"sample", 
 *					  	"route": { 
 *					  		"path": "/path/to/the/service",
 *					  		"method": 	Optional. Values are "get|post|dw|queue",
 *					  		"service": 	"version"
 *					  	}
 *					  }
 * @param  {function} 	callback callback(err)
 */
exports.enableRoute = function(req, callback){

	let self = this

	if(logger.isDebugEnabled()) logger.debug('--->enableRoute: Route=' + JSON.stringify(req.body.route))

	if(!req.body.route) return callback(new Error('route parameter not given'))

	req.body.route.method = req.body.route.method || 'get'

	// Enable the route
	switch(req.body.route.method.toLowerCase()){
		case 'dir': // Enable Dir Route
			pluginManager.callPluginService({name:'oxsnps-dirwatcher',service:'addListenersService'}, [req.body.route], callback)
			break
		case 'queue': // Enable Queue Route
	   		// Delete Queued Jobs before enabling route 
			pluginManager.callPluginService({name:'oxsnps-ibmmq', service:'addListenersService'}, [req.body.route], callback)
			break
		default:
			// Enable HTTP Route
			expressAppUtils.enableRoute(self, req.body.route, callback)
	}
} 

/**
 * Disable the given route
 * @param  {Request} 	request Request json has following structure.
 * 						req.body = {
 *					  	"module":"sample", 
 *					  	"route": { 
 *					  		"path": "/path/to/the/service",
 *					  		"method": 	Optional. Values are "get|post|dw|queue"
 * 						  	}
 * 						}
 * @param  {function} 	callback  callback(err)
 */
exports.disableRoute = function(req, callback){
	let self = this  // V200 Change

	if(logger.isDebugEnabled()) logger.debug('--->disableRoute: Route=' + JSON.stringify(req.body.route))
	
	if(!req.body.route) return callback(new Error('route parameter not given'))

	req.body.route.method = req.body.route.method || 'get'

	// Disable the route
	switch(req.body.route.method.toLowerCase()){
		// Disable Dir Route
		case 'dir': 
			pluginManager.callPluginService({name:'oxsnps-dirwatcher',service:'removeListenersService'}, [req.body.route], callback)
			break
		case 'queue':
			// Disable Queue Route
			pluginManager.callPluginService({name:'oxsnps-ibmmq', service:'removeListenersService'}, [req.body.route], callback)
			break
		default:
			//logger.warn('--->disableRoute: Wrong parameters(unknown route.method)! Not able to disable the route, Parms: ' + JSON.stringify(req.body))
			//callback(new Error('Wrong parameters(unknown route.method)'))
			// Disable HTTP Route
			//expressAppUtils.disableRoute(req.body.route, callback) // V200 Change
			expressAppUtils.disableRoute(self, req.body.route, callback) // V200 Change
	}
} 

/**************** PRIVATE FUNCTIONS ***************/
// V3.0.2  OTS-3259 Begin 
/**
 * Escape input filenames by surrounding input filenames with double quotes. If the input filename has double quote, escape it like "\""
 * @param {*} str string
 */
function escapeArg(res, str){

    let patDoubleQuote  = new RegExp(/^"(.*)"$/)         // pattern to check if string is surrounded by double quote
      , patSingleQuote  = new RegExp(/^'(.*)'$/)         // pattern to check if string is surrounded by single quote
      , match           = undefined
      , orgStr       = str

	if(typeof(str) !== "string") return str

    // Sometimes, calling plugins surround args (like filenames) with doube quotes or single quote. 
    // To handle such cases, find if string is surrounded by double quote or by single quote. If yes, remove them and get only the actual string. This is done to escape double quotes that occur only in the value not the surrounding quotes

    // check if str has surrounding double quote
    match = str.match(patDoubleQuote)

    //  if not, check str has surrounding single quote
    if(!match || match.length <=0) match = str.match(patSingleQuote)

    //If string is surrounded by double quote or by single quote, remove them and get only the actual string.
	if(match && match.length >=2) str = match[1]

    //console.log("str: " + str + ", str.indexOf('\"'): " + str.indexOf('"'))

    //if str contains double quote, escape it
    if(str.indexOf('"') >=0) str = str.replace(/\"/g, '\\"')
 
    str = '"'+ str + '"'
    if(logger.isDebugEnabled()) logger.debug('--->escapeArg: Req. id=' + res._Context.reqId + ', orgStr: ' + orgStr + ', escaped str: ' + str)
	return str
}
// V3.0.2  OTS-3259 End

/**
 * Log a Warning if Service Response time exceeded maxresptime 
 * @param  {Response}	res  	Response Object
 * @param  {String}		service service
 * @private
 */
function _logServiceRespTime(res, service){

	let respTime 	= 0
	  , os 			= require('os')
	  

	// Check resp. time only if maxresptime is defined
   	if(pluginConf.maxresptime){
		respTime = utils.dateDiff('x', res._Context.date, new Date()) // 'x' means returns diff. in ms

   		if(respTime > pluginConf.maxresptime * 1000){
			logger.warn('--->' + service + ': Req.Id=' + res._Context.reqId + ', '
						+ ' Resp. Time: ' + respTime/1000 + 's, Max. allowed: '
						+ pluginConf.maxresptime + 's. Average Load: ' + os.loadavg())
   		}
   	}
}
