'use strict'
/**
 * -----------------------------------------------------------------------------
 * index.js: 	
 *
 * Author    :  AFP2web Team
 * Copyright :  (C) 2020 by Maas Holding GmbH
 * Email     :  support@maas.de
 * 
 * History
 * V1.0.8    26.08.2024    OXS-16051: Extended to catch errors of log.write() promise that lead to server restarts.
 *                                    Enhanced with an optional retry feature for handling logging API errors
 * V1.0.7    14.06.2022    OXS-13737: Upgrade @google-cloud/logging to the latest version 10.0.4
 *                                    Pass maximum entry size("maxEntrySize") value in options parameter, when creating a log object that in turn enables @google-cloud/logging to truncate log entries that exceeds <maxEntrySize>
 * V1.0.6    07.06.2022    OXS-13737: Truncate the log message if it exceeds 255k. 
 *                         Refer to https://github.com/googleapis/nodejs-logging/issues/520, https://cloud.google.com/logging/quotas
 * V1.0.5    10.05.2022    Updated the README.MD with latest lables (specified in "appLabels" object) used with "google" appender
 * V1.0.3/V1.0.4    05.05.2021    OTS-3080:  1. Upgraded to use "@google-cloud/logging" module to v"9.2.2"
 * V1.0.2    04.11.2020    Added metadata.labels.tenant
 * V1.0.0    24.01.2020    Initial release
 *                                           2. Refactored maas-log4js-stackdriver-appender, to fix issues that caused resource labels to become empty while adding log statements for "google" appender
 *----------------------------------------------------------------------------
 */

/**
 * A log object from server-conf.js
 *  "log": {
 *      "appenders": {
 *			"google": {
 *				"type":  'maas-log4js-stackdriver-appender',
 *				"credentials":{
 *					"projectId": 'oxs-develop-docker',
 *					"keyFilename": '/app/logsecret/oxs-develop-docker-b8d9468810a6.json'
 *				},
 *				"resource": {
 *					"appLabels":{
 *						"k8s-pod/app":exports.getServerName(),
 *						"k8s-pod/io_kompose_service":exports.getServerName()
 *					},
 *					"type": "k8s_container",
 *					"labels": {
 *						"container_name":	exports.getServerName(),
 *						"cluster_name": 	"development",
 *						"pod_name": 		process.env["HOSTNAME"],
 *						"project_id":		"oxs-develop-docker",
 *						"namespace_name":	"development",
 *						"location":			"europe-west3-b",
 *					}
 *				},
 *				//"logFileName": 'syslog',
 *				//"logFileName": 'stderr',
 *				"logFileName": 'stdout',
 *				"layout": {
 *                  "type":								"pattern",
 *                  // see https://log4js-node.github.io/log4js-node/layouts.html
 *                  //"pattern": "[%d] [%p] %c - %m"
 *                  // Use following pattern to log ONLY the first parameter of the logger.error() statement
 *                  "pattern": '[%p] %c - %x{data0}',
 *                  "tokens": {
 *                      "data0": function(loggingEvent){return loggingEvent.data[0]}
 *                  }
 *				}
 *			},
 *          "consoleappender": {
 *              "type": "console",
 *              "layout": {
 *                  "type": "pattern",
 *                  "pattern": "[%d] [%p] %c - %m"
 *				}
 *          }
 *      },
 *      "categories": {
 *          "default": {
 *              //"appenders":["google", "consoleappender"],
 *              "appenders":["google"],
 *              "level": process.env[process.env.OXSNPS_SERVER_PREFIX + "LOG_LEVEL"] || "INFO"
 *          }
 *      }
 *	}
 * 
 * A log entry should look as follow:
 * metadata: {
 *      insertId: ".........0HtipAecvG63AkA5dDsJS5A"
 *      labels: {
 *          k8s-pod/app: "oxs-test-server"
 *          k8s-pod/io_kompose_service: "oxs-test-server"
 *      }
 *      logName: "projects/oxs-develop-docker/logs/stdout"
 *      receiveTimestamp: "2020-01-24T14:03:19.212853457Z"
 *      resource: {
 *           labels: {
 *              cluster_name: "development"
 *              container_name: "oxs-test-server"
 *              location: "europe-west3-b"
 *              namespace_name: "development"
 *              pod_name: "oxs-test-server-844f7966c5-sqttd"
 *              project_id: "oxs-develop-docker"
 *          }
 *          type: "k8s_container"
 *      }
 *      severity: "WARNING"
 *      textPayload: "[2020-01-24T14:01:52.177] [WARN] oxsnps-dmy - --->process: Req. id=200124140145766-58069, Logging at level: warn"
 *      timestamp: "2020-01-24T14:03:13.042000055Z"
 *  }
 */

const {Logging} = require('@google-cloud/logging')
const deepmerge = require('deepmerge') // V102
const MAX_LABEL_COUNT = 64    // maximum number of labels allowed in "metadata" log entry object // V1.0.3
// const MAX_LOG_ENTRY_SIZE = 256 *1024  // V1.0.6   
const MAX_LOG_ENTRY_SIZE = 100 *1024  // V1.0.7   Refer to  https://issuetracker.google.com/issues/141996314?pli=1. 
const retry = require('async-retry')


const writeLogWithRetry = async (loggingEvent, resource, layout, credentials, log, logOptions) => {
    
    let entry           = undefined
       , attemptCount   = 0 
       , retryCount     = logOptions && logOptions.retryOptions && logOptions.retryOptions.count || 0

    await retry(async (bail, attempts) => {

        entry = undefined
        attemptCount = attempts

        try{
            // Ex. If the log statement is: logger.error('2. error!', {'tenant': 'wackler'}) ...
            // ...then data = ['2. error!', {'tenant': 'wackler'}]
            let data = loggingEvent.data || []  // V102

            let metadata = {
                /* The way it shoud be
                labels:{
                    "k8s-pod/app": "oxs-xxx-server",
                    "k8s-pod/io_kompose_service": "oxs-xxx-server"
                },
                */
                labels:{},          // V1.0.3   
                resource: {
                    type: "global",
                    labels: {
                        pod_name: process.env["HOSTNAME"] || ''	// V1.0.3
                    }
                },
                severity: loggingEvent.level.levelStr === "WARN" ? "WARNING" : loggingEvent.level.levelStr
            }

            // Add custom labels from server-conf.js (i.e google.resource.appLabels) to metadata.labels
            if (resource){
                metadata.resource = _mergeJSON(resource,{})  //!!!IMPORTANT: We MUST copy resource to metadata.resource using mergeJSON in order to NOT alter the resource obj.
                if(resource.appLabels) metadata.labels = _mergeJSON(resource.appLabels,{})
            }

            /**
             * Custom Labels are optional. When passed thru the log statement, 
             * they are passed as last JSON object in the LoggingEvent.data array and must have a property called 'glabels'. 
             * Ex: logger.error("some", "error", "occured", {'glabels':{'tenant':'wackler','inputchannel:'IC1'}})
             */
            // Add custom labels passed thru logger.<level> statement
            if(data && data.length > 1 && data[data.length-1] && typeof data[data.length-1] === 'object' && data[data.length-1]['glabels']){
                _addProps(metadata.labels, data[data.length-1]['glabels'])   // ex. data[1] = {'tenant':'wackler', 'inputchannel': 'emailImport'}
                loggingEvent.data.pop() // remove the last JSON object from data array so that it does not get writtent in the log message
            }

            entry = log.entry(metadata, layout(loggingEvent)) // metadata.resource is needed in log.entry it seems
            /*
            // V1.0.6 Begin
            // Restrict log entry data to 255KB. 
            // Refer to https://github.com/googleapis/nodejs-logging/issues/520, https://cloud.google.com/logging/quotas
            if(entry && entry.data && JSON.stringify(entry.data).length > 150*1024){
				// for testing 
                console.log('-->writeLogWithRetry: entry: ', entry, ', type of entry.data: ' + typeof(entry.data))
            }
            if(entry && entry.data && typeof(entry.data) === 'string' && entry.data.length > MAX_LOG_ENTRY_SIZE) entry.data = entry.data.substring(0, MAX_LOG_ENTRY_SIZE)
            // V1.0.6 End
            */
            
            delete metadata.resource // delete BEFORE calling log.write but AFTER calling log.entry. Otherwise it does not work
            // console.log('--->writeLogWithRetry: entry:' + JSON.stringify(entry))
            // console.log('--->writeLogWithRetry: metadata:' + JSON.stringify(metadata))
            // V1.0.8 Begin
            // try-catch block does not work. log.write seems to be a promise so we have to use the below style to catch it
            //log.write(entry, metadata)
            // console.log('--->writeLogWithRetry: attempts: ' + attemptCount + ', writing the entry')
            await log.write(entry, metadata)
            // V1.0.8 End
        //}
        //catch(excep){
        //    console.log('-->maas-log4js-stackdriver-appender/index.js: Warning! Error while writing the log entry. Reason: ' + ((excep && excep.message) ? excep.message.substring(0, 1000) + '... truncated' : 'excep object is undefined') + ', Log Data: ' + ((entry && entry.data) ? entry.data.substring(0, 1000) + '... truncated' : ''))
        //}        
        } catch (err) {
            let retryCodes = logOptions && logOptions.retryOptions && logOptions.retryOptions.errorCodes || []
            if(retryCodes.length > 0 && retryCodes.includes(err.code)){
                if(attemptCount >= retryCount){
                    console.log('-->writeLogWithRetry: Google Logging API error: ' + err.message + ' err.code: ' + err.code + ', Number of attempts: ' + attemptCount + ' exceeding the configured retry count: ' + retryCount)
                    bail(err) // Bail if it's a non-retryable error                    
                }
                else {
                    console.warn('-->writeLogWithRetry: Google Logging API error: ' + err.message + ' err.code: ' + err.code + 
                            ', attempt: ' + attemptCount + ', retrying after ' + (logOptions.retryOptions.minTimeout || logOptions.retryOptions.interval || 2000) + ' ms')
                    throw err // Retry the operation
                }
            }else {
                console.error('-->writeLogWithRetry: Google Logging API error: ' + err.message + ' err.code: ' + err.code + ', error code not in the retry list: ' + JSON.stringify(retryCodes))
                bail(err) // Bail if it's a non-retryable error
            }
        }
    }, {
        retries:    retryCount,
        factor:     logOptions && logOptions.retryOptions && logOptions.retryOptions.factor     || 1,    
        minTimeout: logOptions && logOptions.retryOptions && logOptions.retryOptions.minTimeout || logOptions && logOptions.retryOptions && logOptions.retryOptions.interval   || 2000,
        maxTimeout: logOptions && logOptions.retryOptions && logOptions.retryOptions.maxTimeout || logOptions && logOptions.retryOptions && logOptions.retryOptions.interval   || 2000,
        randomize:  logOptions && logOptions.retryOptions && logOptions.retryOptions.randomize  || false
    }).catch((err) => {
        console.error('-->writeLogWithRetry: Warning! Error while writing the log entry. Reason: ' + ((err && err.message) ? err.message.substring(0, 1000) + '... truncated.' : 'err object is undefined.') + ' err.code: ' + err.code + ', Number of retries: ' + attemptCount + ', Log Data: ' + ((entry && entry.data) ? entry.data.substring(0, 1000) + '... truncated' : ''))
    })
}

function stackdriverAppender(logFileName, resource, layout, credentials, logOptions) {

    let options = {'maxEntrySize': MAX_LOG_ENTRY_SIZE}      // V1.0.7

    const logging = new Logging(credentials)
    const log = logging.log(logFileName || "syslog", options)  // V1.0.7

    return function (loggingEvent) {
        writeLogWithRetry(loggingEvent, resource, layout, credentials, log, logOptions).catch((err) => {
            console.error('--->stackdriverAppender: Unexpected error during logging:', err)
        })
    }
}

/**
 * Add properties from <srcObj> to <targetObj>
 * @param {JSON} targetObj 
 * @param {JSON} srcObj 
 * 
 * @google-cloud/logging v9.2.2 reports the following error when labels count in metadata exceeds 64 
 *      ERROR: (node:1) UnhandledPromiseRejectionWarning: Error: 3 INVALID_ARGUMENT: Log entry has a maximum number of 64 labels 
 *      ...at Object.callErrorFromStatus (/app/node_modules/@grpc/grpc-js/build/src/call.js:31:26)
 * So we have to restrict the number lables added to metadata to 64 (i.e. MAX_LABEL_COUNT).
 */
function _addProps(targetObj, srcObj){

    let name    = ''
      , keys    = undefined
      , labelCount  = 0
    
    if(typeof srcObj !== 'object') return
    keys = Object.keys(targetObj) || []
    
    labelCount = keys.length

    for(name in srcObj){
        if(labelCount >= MAX_LABEL_COUNT) break
        labelCount++
        targetObj[name]= srcObj[name]
    }
}

/**
 * Appender's config function, which is called by  log4js
 * @param {JSON} config   Appender's configuration object   
 * @param {JSON} layouts  Appender's layout module      // Refer to https://github.com/log4js-node/log4js-node/blob/master/docs/writing-appenders.md
 */
function configure(config, layouts) {
    let logOptions = undefined

    var layout = layouts.basicLayout
    if (config.layout) {
        layout = layouts.layout(config.layout.type, config.layout)
    }

    logOptions = _parseLogOptions(config)
    if(logOptions){
        config.credentials.clientConfig = {
            "interfaces": {
                "google.logging.v2.LoggingServiceV2": {
                    "methods": {
                        'WriteLogEntries': {
                            "timeout_millis":  logOptions.timeout || 60000, // Set timeout to 60 seconds (60,000 milliseconds)		//https://github.com/googleapis/google-cloud-node/blob/118b622db780e25e63bf5e19fefa16d9782a10ab/packages/logging/src/v2/logging_service_v2_client_config.json#L44
                        }
                    }
                }
            }
        }
    }
    //console.log("config.credentials.clientConfig: " + JSON.stringify(config.credentials.clientConfig))

    return stackdriverAppender(
        config.logFileName,
        config.resource,
        layout,
        config.credentials,
        logOptions
    )
}

function _parseLogOptions(config){

    let options = config.options
      , envvar  = undefined

    if(!options){
        console.log('--->_parseLogOptions: Google Logging options undefined')
        return undefined
    }

    if(typeof options === 'string'){
        envvar = options
        if(!process.env[envvar]){
            console.error('--->_parseLogOptions: Missing environment variable: ' + envvar)
            return undefined
        }

        try {
            options = JSON.parse(process.env[envvar])
        } catch (err) {
            console.error('--->_parseLogOptions: Error on parsing environment variable: ' +  envvar + ', error: ' + err.message + ', value: ' + process.env[envvar])
            return undefined
        }

        if(options === null){
            console.error('--->_parseLogOptions: Error on parsing environment variable: ' + envvar + ', value: ' + process.env[envvar])
            return undefined
        }

    } 
    /*
    console.log("options.retryOptions.count: " + (options.retryOptions && options.retryOptions.count ? options.retryOptions.count : undefined) + ', type: ' + typeof(options.retryOptions.count))
    console.log("options.retryOptions.factor: " + options.retryOptions.factor + ', type: ' + typeof(options.retryOptions.factor))
    console.log("options.retryOptions.interval: " + options.retryOptions.interval + ', type: ' + typeof(options.retryOptions.interval))
    console.log("options.retryOptions.randomize: " + options.retryOptions.randomize + ', type: ' + typeof(options.retryOptions.randomize))
    */
    return options
}

/** V102
 * From https://svn.oxseco.net/svn/AFP2web/OXSNPS/base/oxsnps-core/trunk/helpers/utils.js
 * Merge two objects obj1 & obj2,  and return a new merged object with the elements from both obj1 & obj2.
 * If an element at the same key is present for both obj1 and obj2, the value from obj2 will appear in the result.
 * 
 * @param  {JSON} obj1    JSON   A single JSON object or array of objects
 * @param  {JSON} obj2    JSON   Optional.  If obj1 is array of objects, then this parameter is options parameter else a JSON object
 * @param  {JSON} options JSON   Optional. Options. Refer to https://www.npmjs.com/package/deepmerge
 * @return {JSON}      Returns merged JSON object.
 */
function _mergeJSON(obj1, obj2, options){
    if(obj1 && Array.isArray(obj1)){
      options = obj2
      return deepmerge.all(obj1, options)
    }
    return deepmerge(obj1, obj2, options)
}
 
module.exports.configure = configure
