/**
 * Whenever the word "object" is used here, it usually means business object: for example process, workbasket or record.
 * Object is identified by 2 properties: type and id.
 * Type can be "process", "workbasket", "record" - and actually anything, as long as user of this client recognizes them.
 * Id is unique identification of the object among other objects of the same type.
 * @module
 */

/**
 * Defines notification from the client on the object.
 * It can be received either in callback of subscribe() call or by directly fetching list of pending notifications
 * of the object via {@link getPendingNotifications}
 *
 * @param type type of object
 * @param id id of object
 * @param relations relations of object
 * @param action action performed on object
 * @param phase "started" or "stopped" if it's persistent notification; undefined if it's volatile notification
 * @param data additional data
 * @param user user which made notification
 * @constructor
 */
var Notification = function (type, id, relations, action, phase, data, user) {
    this.object = {
        type: type,
        id: id,
        relations: relations
    };
    this.action = action;
    this.phase = phase;
    this.data = data;
    this.user = user;
};

/**
 * @param socket websocket to use
 * @constructor
 */
function ConcurrentAccessClient(socket) {
    this.socket = socket;
}

/**
 * Subscribe to events on object with given type and id.
 * Clients which subscribe on specified object will receive notifications thrown by other clients using
 * notify*() methods if those notifications concern this subscription, i.e. if
 * - notify* is called for an object on which this client subscribed
 * - notify* is called for an object which has this object in relations
 *
 * @param type type of object to subscribe to
 * @param id id of object to subscribe to
 * @param cb callback that will be called upon receiving the notification concerning this subscription
 *  cb will be called with the {@link Notification} object
 */
ConcurrentAccessClient.prototype.subscribe = function (type, id, cb) {

    function isMessageForMe(object){
        if (object.type == type && object.id == id)
            return true;
        for (var key in object.relations) {
            var whom = object.relations[key];
            if (whom.type == type && whom.id == id)
                return true;
        }
        return false;
    }

    var callback = function (message) {
        try {
            message = JSON.parse(message);
            if (message.command == "notify" && isMessageForMe(message.data.object)) {
                cb(message.data);
            }
        } catch (e) {
            console.error(e.stack || e).bind(console);
        }
    };
    callback.prototype.name = constructCallbackName(type, id);

    this.socket.on("message", callback);
    this.socket.emit("message", createStandartMessage("subscribe", type, id));
    console.log('Subscribed:', "type='" + type + "'"+ " id='" + id + "'");
};

/**
 * Un-subscribe from the events on given object
 * @param type type of object to un-subscribe from
 * @param id id of object to un-subscribe from
 */
ConcurrentAccessClient.prototype.unsubscribe = function (type, id) {
    this.socket.emit("message", createStandartMessage("unsubscribe", type, id));
    var nameToDelete = constructCallbackName(type, id);
    for (var index in this.socket._callbacks.$message) {
        if (this.socket._callbacks.$message.hasOwnProperty(index)){
            if (this.socket._callbacks.$message[index].prototype && this.socket._callbacks.$message[index].prototype.name == nameToDelete)
                this.socket._callbacks.$message[index] = Function.prototype;
        }
    }
    console.log('Un-subscribed:', "type='" + type + "'"+ " id='" + id + "'");
};

/**
 * Notify concerned parties about action on given object.
 * All concerned parties (i.e. other connected clients) will receive a Notification object which will include data that is provided here.
 * Whether a party can be considered "concerned" is defined by its subscriptions and by relations passed in this call.
 * Namely, the following connected clients will receive notification:
 * - subscribed to the given object (specified by type/id)
 * - subscribed to any of the objects in the relations passed in `data`
 *
 * Structure of `relations`:
 * [
 *      {"type": "workbasket", "id": "W1"}
 * ]
 *
 * `data` can contain arbitrary custom data that you wish to pass to subscribers
 *
 * @param type type of object
 * @param id id of object
 * @param action action done on the object
 * @param relations relations of the object
 * @param data additional data
 */
ConcurrentAccessClient.prototype.notify = function (type, id, action, relations, data) {
    var notifyMessage = {
        command: "notify",
        data: {
            notification: new Notification(type, id, relations, action, undefined, data, this.user)
        }
    };
    this.socket.emit("message", notifyMessage);
    //console.log("Notified");
};

/**
 * Notify concerned parties about start of specified action on specified object.
 * Unlike notify, this has permanent results: the object will stay in the action
 * until this client does not call {@link notifyStop} or until this client disconnects.
 * @param type type of object
 * @param id id of object
 * @param action action done on the object
 * @param relations relations of the object
 * @param data additional data
 */
ConcurrentAccessClient.prototype.notifyStart = function (type, id, action, relations, data) {
    var notifyMessage = {
        command: "notify",
        data: {
            notification: new Notification(type, id, relations, action, "started", data, this.user)
        }
    };
    this.socket.emit("message", notifyMessage);
    //console.log("Started action");
};

/**
 * Notify concerned parties about stop of specified action on specified object.
 * Use it in conjunction with {@link notifyStart} to mark the end of an action.
 * This will be called automatically if client disconnects.
 * @param type type of object
 * @param id id of object
 * @param action action done on the object
 * @param relations relations of the object
 * @param data additional data
 */
ConcurrentAccessClient.prototype.notifyStop = function (type, id, action, relations, data) {
    var notifyMessage = {
        command: "notify",
        data: {
            notification: new Notification(type, id, relations, action, "stopped", data, this.user)
        }
    };
    this.socket.emit("message", notifyMessage);
    //console.log("Stopped action");
};

/**
 * Return the list of notifications current set on the object.
 * It is defined by {@link notifyStart}/{@link notifyStop} calls from all the clients.
 * @param type type of object
 * @param id id of object
 * @return Promise with list of Notifications set on this object
 */
ConcurrentAccessClient.prototype.getRunningNotifications = function (type, id) {
    //console.log("Get status.");

    function promise(resolve, reject){
        this.socket.on("message", function(message) {
            try {
                message = JSON.parse(message);
                if (message.command == "runningNotifications") {
                    resolve(message.data.runningNotifications.map(function (notification){
                        return notification.message;
                    }));
                }
            } catch (e) {
                console.error(e.stack || e).bind(console);
                reject(e);
            }
        });
    }
    this.socket.emit("message", createStandartMessage("getRunningNotifications", type, id));
    return new Promise(promise.bind(this));
};

ConcurrentAccessClient.prototype.registerUser = function(user){
    this.user = user;
};

function createStandartMessage(command, type, id) {
    return {
        command: command,
        data: {
            type:type,
            id:id
        }
    }
}

function constructCallbackName(type, id) {
    return [type, id].join("$$");
}

module.exports = ConcurrentAccessClient;