"use strict";

/**
* @constructor
* */
function MessageOperationHandler(){
    var self = this;
    this._handlers = {
        post: self.post,
        start: self.start,
        download: self.download,
        default: self.default
    };
}

/**
 *
 * @param operations object to extend current operations
 */
MessageOperationHandler.prototype.extendOperations = function(operations){
    Object.assign(this._handlers, operations);
};

/**
 * Handle incoming messages
 * Usage example: wssClient.messageExchangeManager.subscribe(wssClient.messageOperationHandler.handler.bind(wssClient.messageOperationHandler));
 * Don't forgot bind context wssClient.messageOperationHandler.
 * @param message
 */
MessageOperationHandler.prototype.handler = function(message){
    try {
        message = JSON.parse(message);
        if (this._handlers[message.operation]) {
            this._handlers[message.operation].call(this, message);
        } else {
            this._handlers.default.call(this, message);
        }
    } catch (e){
        console.error(e);
    }
};

/**
 *
 * @param object object of windows properties. See http://www.w3schools.com/jsref/met_win_open.asp -> Parameter Values -> specs
 * @returns {*} Right string of windows params or incoming param @object
 * @private
 */
MessageOperationHandler.prototype._createWindowParams = function(object){
    if (typeof(object) == "object") {
        return Object.keys(object).reduce(function (string, param) {
            if (object.hasOwnProperty(param))
                string += param + "=" + object[param] + ",";
            return string;
        }, "");
    } else {
        return object;
    }
};

/**
 * Validate incoming message for required fields
 * @param object object to check fields existence
 * @param requiredFields array of fields
 * @returns {boolean} is valid
 */
MessageOperationHandler.prototype.validateMessage = function(object, requiredFields){
    return requiredFields.every(function(field){
        return object.hasOwnProperty(field);
    });
};

/**
 * start operation handler
 * @param message
 */
MessageOperationHandler.prototype.start = function(message){
    var requiredParameters = ["url", "windowName"];
    if (this.validateMessage(message.operationParameters, requiredParameters)) {
        var url = message.operationParameters.url;
        var windowName = message.operationParameters.windowName;

        if (this._windows[windowName] && !this._windows[windowName].closed){
            this.post(message);
        } else {
            var paramsString = this._createWindowParams(message.operationParameters.windowParams);
            this._windows[windowName] = window.open(url, windowName, paramsString);
        }
    } else {
        errorMessage(requiredParameters);
    }
};

/**
 * post operation handler
 * @param message
 * @returns {*} Exit code flow if MessageOperationHandler doesn't consist window with target name
 */
MessageOperationHandler.prototype.post = function (message){
    var requiredParameters = ["windowName"];
    if (this.validateMessage(message.operationParameters, requiredParameters)) {
        var windowName = message.operationParameters.windowName;

        if (!this._windows[windowName])
            return console.warn("There's no window with id: " + windowName);

        //If second argument isn't '*' - will check target window url and second argument for equals.
        //Window have to consist listener: window.addEventListener("message", console.log.bind(console), false);

        this._windows[windowName].postMessage(message.operationParameters.postData, "*");
    } else {
        errorMessage(requiredParameters);
    }
};

/**
 * download operation handler
 * @param message
 */
MessageOperationHandler.prototype.download = function(message){
    var requiredParameters = ["url"];
    if (this.validateMessage(message.operationParameters, requiredParameters)) {
        var paramsString = this._createWindowParams(message.operationParameters.windowParams);
        this._windows._toDownloadCount++;
        window.open(message.operationParameters.url, "download_" + this._windows._toDownloadCount, paramsString);
    } else {
        errorMessage(requiredParameters);
    }
};

MessageOperationHandler.prototype.default = Function.prototype;

/**
 * Consist opened windows
 * @type {{_toDownloadCount: number}}
 * @private
 */
MessageOperationHandler.prototype._windows = {
    /**
     * count of opened windows for download
     */
    _toDownloadCount: 0
};

function errorMessage(params){
    console.warn("Incoming message is missing required parameters: " + params.join(", "));
}

module.exports = MessageOperationHandler;