"use strict";
var io = require('socket.io-client');
var Promise = require('es6-promise').Promise;
var url = require("url");

var ConcurrentAccessManager = require('./ConcurrentAccessManager');
var ChatNoticeManager = require('./ChatNoticeManager');
var MessageExchangeManager = require('./MessageExchangeManager');
var MessageOperationHandler = require("./MessageOperationHandler");

/**
 * Empty constructor. See WSSClient#connect
 * @constructor
 */
var WSSClient = function (socket, mandant, token) {
    this.socket = socket;
    this.concurrentAccessManager = new ConcurrentAccessManager(socket);
    this.chatNoticeManager = new ChatNoticeManager(socket);
    this.messageExchangeManager = new MessageExchangeManager(socket, mandant, token);
    this.messageOperationHandler = new MessageOperationHandler();
};

/**
 * Disconnect from web-socket
 */
WSSClient.prototype.disconnect = function () {
    this.socket.disconnect();
    this.socket = null;
};

/**
 * Return unique session id of current WSS session.
 */
WSSClient.prototype.getSessionId = function () {
    return this.socket ? this.socket.id : null;
};

/**
 * Establish connection to WSS at given address for given user of token.
 * address can be either absolute or relative. You can only use relative address in browser environment, in which
 * case client will use current host as a base.
 * If address is absolute, host will be used as transport path for WS-connection, and path will be prepended to `path`
 * option in WSS connection.
 *
 * @param address
 * @param token
 * @param mandant
 * @return {*}
 */
WSSClient.connect = function(opts) {
    var address = opts.address;
    var token = opts.token;
    var userId = opts.userId;
    var mandant = opts.mandant;

    return new Promise(function(resolve, reject) {
        address = getHostAndPath(address);
        console.log("Using this address for connect:", address);
        var client = new WSSClient(io(address.host, {
            path: address.path,
            'force new connection': true,
            multiplex: false,
            query: "mandant=" + mandant + "&userId=" + userId + (token ? "&token=" + token : "")
        }), mandant, token);
        client.socket.on('connect', function () {
            // pause to allow check for validity of token
            setTimeout(function() {
                if (client.getSessionId()) {
                    console.log('WSS socket connected');
                    resolve(client);
                } else {
                    reject('Could not connect');
                }
            }, 500);
        });

        client.socket.on('disconnect', function () {
            console.log('WSS socket disconnected');
        });
    });
};

function slashModifier(pathname){
    if (pathname[0] !== "/")
        pathname = "/" + pathname;
    if (pathname[pathname.length - 1] == "/")
        pathname = pathname.substr(0,pathname.length - 1);
    return pathname;
}

function getHostAndPath(address) {
    address = url.parse(address);
    var defaultPath = '/socket.io';
    var location = global.location;

    if (address.host) {
        return {
            path: address.pathname ? slashModifier(address.pathname) + defaultPath : defaultPath,
            host: (address.protocol || "https:") + "//" + address.host
        }
    }

    if (location) {
        var pathname = address.pathname[0] != "/" ? slashModifier(location.pathname) + slashModifier(address.pathname) : slashModifier(address.pathname);
        return {
            path: location.pathname ? pathname + defaultPath : defaultPath,
            host: location.origin
        }
    }

    return {
        host: address,
        path: defaultPath
    }
}

module.exports = WSSClient;