/**-----------------------------------------------------------------------------
 * oxsnps.js: 	Plugin to process nils utility requests
 *
 * Author    :  AFP2web Team
 * Copyright :  (C) 2017 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  1.0.0
 * 
 * History
 *  V3.0.0   10.07.2025  HNDMS-1916: Deliver oxs-bpm-server based on oxs-server architecture (DMS3)
 *  						Extended plugin / services
 *  						1. To adopt to oxsnps-cre 5.x.x
 *  						2. To use oxsnps-job instead of "kue" module directly
 *  						3. Added env.variable in ci.bpm.js whereever needed
  *  V2.0.0   31.05.2021  OTS-2699: Extend base and custom import/export plugins to use oxsnps-core@4.1.0
 * V1.0.0   15.08.2018  OTS-2336: Implement a "getStack" service to oxs-bpm-server
 *  
 *----------------------------------------------------------------------------*/
'use strict'

/**
 * Variable declaration
 * @private
 */
let packageJson		= require(__dirname + '/package.json')
  , PLUGIN_NAME		= packageJson.name
  , PLUGIN_VERSION	= packageJson.version // Get Server Version from package.json
  , async			= require('async')  
  , fs				= require('fs')
  , log4js			= require('log4js')
  , os				= require('os')  
  , path			= require('path')
  , url     		= require('url')
  , npsServer		= require('oxsnps-core/server')
  , npsConfDir 		= npsServer.npsConfDir					// holds the configuration Dir of the OTS Server   
  , npsConf 		= npsServer.npsConf
  , npsName   		= npsServer.npsName
  , npsVersion   	= npsServer.npsVersion
  , npsLogFile 		= npsServer.npsLogFile
  , npsLogDir 		= npsConf.logDir
  , npsTempDir		= npsServer.npsTempDir

  , pluginConf		= require(__dirname + '/conf/' + PLUGIN_NAME + '.js')
  , pluginErrors	= require(__dirname + '/conf/error.json')  
  , dms2common 		= undefined

  , dateFormat 		= require('oxsnps-core/helpers/date_format')
  , utils 			= require('oxsnps-core/helpers/utils')
  , httpUtils		= require('oxsnps-core/helpers/httpUtils')
  , expressAppUtils = require('oxsnps-core/helpers/expressAppUtils')
  , pluginManager 	= require('oxsnps-core/pluginManager')  
  , error 			= require('oxsnps-core/error').createError
  , regError 		= require('oxsnps-core/error').registerErrorObj
  
  //, kue 			= require('kue')		// V3.0.0
  //, jobs 			= kue.createQueue(npsConf.kue)  // V3.0.0
  , initError       = undefined 
  , oxsnpsJob 		= undefined // V3.0.0  

  , pluginLogDir	= undefined
  , FILE_SEP 		= require('path').sep
  , processHandlerCreated 	= false    
  , responseObjs	= {}	// to store response objects of HTTP requests.  
  , defTenant		= npsConf.tenants && npsConf.tenants.length > 0 ? npsConf.tenants[0] : 'hallesche' // V3.0.0

// Strings used to identify different nils services
let GET_STAPEL	= 'getStapel'
  
// Set plugin name, version and longDesc in pluginConf
pluginConf.module 	= PLUGIN_NAME 
pluginConf.version 	= PLUGIN_VERSION 
pluginConf.longDesc = packageJson.description
pluginConf.log 		= pluginConf.log || {}

// Export the Plugin's Configuration Express App
exports.pluginConf 	= pluginConf
exports.router      = undefined     // For oxsnps-core V4.1.x

// Set log properties as defined in pluginConf.log4js
//log4js.configure(pluginConf.log, {})

// Get the Logger
let logger = log4js.getLogger(PLUGIN_NAME)
utils.log4jsSetLogLevel(logger, (pluginConf.log.level || 'INFO'))

// V3.0.0
/* Add this to catch the Kue/Redis errors */
// jobs.on('error', function(err){
// 	logger.error('--->_createJob: CRITICAL Kue Error, Reason: ' + err.message)
// 	return
// })

/**************** PUBLIC FUNCTIONS ****************/
/**
 * Initialize Plugin
 * @param  {function} 	callback  callback(err, routes)
 */
exports.initialize = function(callback){

	let self = this
      , mode = parseInt('0777',8)
      , appPluginConf	= (npsConf.plugins && npsConf.plugins[PLUGIN_NAME] ? npsConf.plugins[PLUGIN_NAME] : {})	
	  
    // V3.0.0 Begin
    function _handleError(err){
        initError = err
        self._initialized = false 
        return callback(err, pluginConf.routes)
    }          
    // V3.0.0 End

	if(self._initialized) return callback();
	initError = undefined	// V3.0.0

    if(logger.isDebugEnabled()) logger.debug('--->initializing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION )
        
    pluginConf		= utils.mergeJSON(pluginConf, appPluginConf)    
	pluginLogDir	= path.resolve(npsLogDir) + FILE_SEP + PLUGIN_NAME

    exports.pluginConf = pluginConf

    /*
    if(logger.isDebugEnabled()) logger.debug('--->initializing pluginConf.routes[3]:',  pluginConf.routes[3])
    if(logger.isDebugEnabled()) logger.debug('--->initializing typeof(pluginConf.routes[3].maxJobs):',  typeof(pluginConf.routes[3].maxJobs) )
    if(logger.isDebugEnabled()) logger.debug('--->initializing pluginConf:',  pluginConf )
*/
	// Register plugin error object with oxsnps-core/error class
	regError(PLUGIN_NAME, pluginErrors)
	
	async.series([
		// check and create Plugin Log directory
	    function(nextTask){
			logger.debug('--->initialize: "' + pluginLogDir + '" does not exist, creating it...')
			utils.mkDir(pluginLogDir, mode, function(err){
				if(err) return nextTask(new Error('Unable to create directory "'+ 	pluginLogDir +
												'", needed to store Stored Procedure results. ' + err.message))
				return nextTask()
			})
		},
	    function(nextTask){
			// Map the HTTP routes of the plugin
            //expressAppUtils.enableRoutes(self, pluginConf.routes, nextTask)
            expressAppUtils.setupRouter(self, function(err, router){    // V2.0.0
                exports.router = router     // V2.0.0
                return nextTask(err)
            })                           
	    }
	],
	function(err, results){
		if(err) return _handleError(err) // 3.0.0
		self._initialized = true
		logger.info('\t\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' initialized')
		// callback(null, pluginConf.routes)
		callback(null, {'routes': pluginConf.routes, 'swagger': undefined /*{"name": PLUGIN_NAME, "url": '/' + PLUGIN_NAME + '/api/rest.yaml'}*/})
	})
}

/**
 * Activate Plugin. This function is called from pluginManager once all the enabled plugins have been initialize
 * @param  {function} 	callback  callback(err)
 */
exports.postInitialize = function(callback){

	let self	= this
	  , resTmp	= {'_Context':{'plugin': PLUGIN_NAME}}
	  , redisErr= undefined

	if(logger.isDebugEnabled()) logger.debug('Activating ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + '...')

	async.series([
		// V3.0.0 Begin
	    function(nextTask){
            if(!self._initialized) return nextTask(initError)
            nextTask()
        },        
	    function(nextTask){
			pluginManager.getModule('oxsnps-job', function(err, module){
				if(err)	return nextTask(err)
				oxsnpsJob = module
				nextTask()
			})
        },        
		function(nextTask){
			oxsnpsJob.isSchedulerReadyService(resTmp, function(err, connected){
                if(err) redisErr = err
                return nextTask()
            })
        },
		// Clean any active or inactive job before enabling the processing of new jobs
		// Remove any active Jobs BEFORE trigerring new ones
		function(nextTask){
            // _cleanJobs('active', nextTask);
            if(redisErr) return nextTask()
            resTmp._Context.jobState = 'active'
            oxsnpsJob.cleanJobsService(resTmp, nextTask)    // V3.0.0 
		},
		// Remove any inactive Jobs BEFORE trigerring new ones
		function(nextTask){
            if(redisErr) return nextTask()
            //_cleanJobs('inactive', nextTask);
            resTmp._Context.jobState = 'inactive'
            oxsnpsJob.cleanJobsService(resTmp, nextTask)    // V3.0.0 
        },
        // Remove failed Jobs BEFORE trigerring new ones
        function(nextTask){
            if(redisErr) return nextTask()

            resTmp._Context.jobState = 'failed'
            oxsnpsJob.cleanJobsService(resTmp, nextTask)
        },		
		// V3.0.0 End
		// Get dms2common module
	    function(nextTask){
			pluginManager.getModule('dms2.common', function(err, module){
				if(!err)	dms2common = module
				return nextTask(err)
			})
	    },
		// V3.0.0 
		function(nextTask){
			if(redisErr || processHandlerCreated) return nextTask(); // if handler already added, just return
            processHandlerCreated = true;
            //_addKuehandlers(nextTask);
            resTmp._Context.routes = pluginConf.routes
            oxsnpsJob.registerJobHandlersService(resTmp, nextTask)  
		}        		
	],
	function(err){
		// V3.0.0 
        if(redisErr || err){
            if(!initError) initError = redisErr || err
            self._initialized = false
            logger.error('--->postInitialize: Unable to activate the plugin: ' + PLUGIN_NAME + ' v' + PLUGIN_VERSION + '. Reason: ' + (redisErr ? redisErr.message : err.message))
            return _disableAllRoutes(initError, callback)
        }
		return callback()
	})
}

/**
 * Prefinalize Plugin
 * @param  {Response}  	res Response Object
 * @param  {function} 	callback callback(err) 
 */
exports.preFinalize = function(res, callback){

	let self 		= this
	  , softQuit 	= false

	if(logger.isDebugEnabled()) logger.debug('--->preFinalizing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION)
	if(!self._initialized) return callback();

	if(res._Context.query && res._Context.query.soft && res._Context.query.soft.toLowerCase() === 'true') softQuit = true;
	if(logger.isDebugEnabled()) logger.debug('--->preFinalize: Req. id=' + res._Context.reqId + ', softQuit=' + softQuit)

	async.series([
		// Disable all routes. It also deletes Queued Jobs
		function(nextTask){
			async.each(pluginConf.routes, 
				function(route, next){
					let req = {'body': {'module': PLUGIN_NAME, 'route': route}}
					exports.disableRoute(req, next);
				},
				function(err){
					return nextTask(err);
				}
			);
		},
		// Wait for job completion, if soft quit is triggered.
		function(nextTask){
			if(!softQuit) return nextTask();
			// Wait for job completion 			//_waitTillJobCompletion(res, nextTask);
            res._Context.plugin = PLUGIN_NAME
			// Wait for job completion
			oxsnpsJob.waitTillJobCompletion(res, nextTask)
		}
	],
	function(err){
		if(err) return callback(err);
		logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' prefinalize over');
		callback();
	});	
}

/**
 * Finalize Plugin
 * @param  {function} 	callback callback(err) 
 */
exports.finalize = function(callback){

	var self = this

	logger.debug('--->finalizing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION );
	if(!self._initialized) return callback();

	self._initialized = false;
	logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' finalized');
	callback();
}

/**
 * Return Plugin Version
 * @param  {function} 	callback callback(err,version)
 */
exports.getVersion = function(callback){
	return callback(null, PLUGIN_VERSION)
}

/**
 * Process a HTTP Version Request
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.version = function(req, res){

	let data = undefined

	// Add Context to the response instance
	res._Context = {}

	// Add the Current Date to the Context
	res._Context.date = new Date()

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date)
	logger.info('--->version: Req. Id=' + res._Context.reqId)

	data = '<h3>' +  PLUGIN_NAME + ' v' + PLUGIN_VERSION + '</h3>'
	res.end(data)

	logger.info('--->version: Req. Id=' + res._Context.reqId + ', sent back ' +
				PLUGIN_NAME + ' v' + PLUGIN_VERSION + ', over')
	return
}

/**
 * Process a HTTP Configuration Request 
 * 	Either as HTTP GET request:
 * 	 	GET http://localhost:1029/services/<plugin>/conf 							--> To retrieve the configuration
 * 	 	GET http://localhost:1029/services/<plugin>/conf?save=true&conf="{...}" 	--> To pass and save the configuration
 * 	or as HTTP POST request:
 * 	 	POST http://localhost:1029/services/<plugin>/conf 						--> To pass and save the configuration
 * 	 	req.body={
 * 	 		"conf": {...}
 * 	 	}
 * 	
 * @param {Request} 	req Request Object
 *                      req.query:{ // for HTTP GET request:
 *                      	"save":"false",
 *                      	"conf":"{<IMPORTANT: COMPLETE JSON Conf. as String>}"
 *                      }
 *                      req.body:{ // for HTTP POST request:
 *                      	"conf":{<IMPORTANT: COMPLETE JSON Conf. as JSON Object>}
 *                      }
 * @param {Response}    res Response Object
 */
exports.conf = function(req, res){

	let reqType 	= 'getconf'
	  , jsonConf 	= undefined

	// Add Context to the response instance
	res._Context = {}

	// Add the Current Date to the Context
	res._Context.date = new Date()

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date)
	logger.info('--->conf: Req. id=' + res._Context.reqId)
	
	if(req.method.toUpperCase() === 'POST'){
		logger.info('--->conf: body=' + JSON.stringify(req.body))
		if(!req.body || !req.body.conf) {
			logger.info('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body))
			return res.end('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body))
		}
		reqType = 'saveconf' 		// save request
		jsonConf = req.body.conf 	// set the json conf to be passed to saveConfService
	}
	else if(req.method.toUpperCase() === 'GET'){
		logger.info('--->conf: query=' + JSON.stringify(req.query))
		if(req.query && req.query.save && req.query.save.toLowerCase() === 'true' && req.query.conf)
		{
			reqType = 'saveconf'
			jsonConf = utils.parseJSON(unescape(req.query.conf)) // remove escaped chars and set the json conf to be passed to saveConfService
		} 
	}

	switch(reqType){
		case 'saveconf':
		    exports.saveConfService(jsonConf, function(err){
		    	logger.info('--->conf: Req. id=' + res._Context.reqId + ', over')
				res.json(err ? {'status':'error', 'message': err.message} : pluginConf)
		    }) 
			break
		default:
		case 'getconf':
			logger.info('--->conf: Req. id=' + res._Context.reqId + ', over')
			res.json(pluginConf)
			break
	}
}

/**
 * Store  the Plugin conf to its configuration file (i.e. .../<plugin>/conf/<plugin>.js)
 * @param  {JSON}   	conf     Plugin Configuration
 * @param  {Function} 	callback
 */
exports.saveConfService = function(jsonConf, callback){

	let filename 	= undefined
	  , data 		= undefined
	  
	// PluginManager calls saveConfService without passing the conf
	if(typeof jsonConf === 'function'){
		callback = jsonConf
		jsonConf = pluginConf
	}

	// Assert jsonConf
	if(!jsonConf || jsonConf === null) return callback(new Error('--->saveConfService: empty Configuration'))
	if(logger.isDebugEnabled()) logger.debug('--->saveConfService: conf:' + JSON.stringify(jsonConf))

	// Backup the current plugin conf.in the server's backup dir and store the new one in the plugin's /conf dir
	async.series([
		// Task 1: Backup the current plugin conf.
		function(nextTask){
			filename = npsConf.backupDir + FILE_SEP + dateFormat.asString('yyyy-MM-dd_hh-mm-ss', new Date()) + '_configurationOf_' + PLUGIN_NAME + '.js' // build the Backup filename
			if(logger.isDebugEnabled()) logger.debug('--->saveConfService: Backuping plugin configuration to ' + filename)
			data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t') // build the plugin conf content

			// Write data to the Plugin configuration file
			fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
				if(err) logger.error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message) // log error & continue
				nextTask()
			})
		},
		// Task 2: Save the new plugin conf.
		function(nextTask){
			pluginConf = jsonConf // we should have here a proper conf, so overwrite the current pluginConf file
			filename = path.resolve(__dirname + '/conf/' + PLUGIN_NAME + '.js') // build the plugin configuration filename
			if(logger.isDebugEnabled()) logger.debug('--->saveConfService: Writing plugin configuration to ' + filename)
			data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t') // build the plugin conf content

			// Write data to Plugin configuration file
			fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
				if(err) nextTask(new Error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message))
				else nextTask()
			}) 
		}
	],
	function(err){
		callback(err)
	})
}

/**
 * Process getStack HTTP Request 
 * HTTP POST Request:
 * 		http://oxs-bpm-server:<port>/nils/getstack
 * 		Req. body
 * 			<XML request>
 * @param {Request} 	req Request Object
 * @param {Response}    res Response Object
 */
exports.getStapel = function(req, res){

	let ctDate 			= new Date()
	  , resTmp 			= undefined
	  , route 			= undefined
	  , reqId 			= utils.buildReqId(ctDate)
	  , reqOptions		= undefined

	if(logger.isDebugEnabled()) logger.debug('--->getStapel: Req.Id=' + reqId +  ', ' + (req.method ? (req.method +', '): '') + 'URL: ' + req.url)

    try{
        reqOptions = url.parse(req.url)
        if(!reqOptions) return res.status(400).send('Unable to parse request URL string. url: ' + req.url)
    }catch(err){
        return res.status(400).send('Unable to parse a URL string. url: ' + req.url + '. ' + err.message)
    }
    /*
        reqOptions = {
            "protocol": null,
            "slashes": null,
            "auth": null,
            "host": null,
            "port": null,
            "hostname": null,
            "hash": null,
            "search": "?enc=utf-8",
            "query": "enc=utf-8",
            "pathname": ""/services/ci.bpm/updateprocesshistory"",
            "path": ""/services/ci.bpm/updateprocesshistory?xxxx",
            "href": ""/services/ci.bpm/updateprocesshistory?xxxx"
        }
    */
    
    route = {
        'path':     reqOptions.pathname,        // "/services/ci.bpm/updateprocesshistory"
		'method':   req.method || 'POST'
    }
	// Build res context 
	resTmp = {
	   '_Context' :{
			'plugin': PLUGIN_NAME,  // V3.0.0
   			'nils': {
				//'service' : GET_STAPEL,
   				'err': { // set module name and function name in err object
	   				'module': PLUGIN_NAME, 
	   				'func': 'getStapel'
				},
				'req':{
					'parms': {}
				},
			},
			'route': 		route,
	   		'path': 		route.path,
	   		'date' : 		ctDate,
	   		'reqId': 		reqId,
	   		'service': 		'nilsService', //!!! IMPORTANT: will be called when processing the job
	   		'titleInfo':	'Service: ' + GET_STAPEL
		}
	}

	// Get request parameters
	resTmp._Context.nils.req.parms = _getRequestParms(req)

	// Since we store resTmp in Kue Job data,  Store HTTP Response obj in "responseObjs" json instead resTmp._Context
	// Othewise while creating job, we get an error "Unable to create a kue job, Reason: Converting circular structure to JSON"
	responseObjs[resTmp._Context.reqId] = res

	logger.info('--->getStapel: Req.Id=' + resTmp._Context.reqId + ', Parms=' + JSON.stringify(resTmp._Context.nils.req.parms))
	oxsnpsJob.createJobService(resTmp)    // V3.0.0
}

/**
 * REST HTTP GET API to get a stack
 * HTTP GET  Request:
 * 		http://<oxs-bpm-server>:<port>/services/nils.utils/stack/:stackid
 * @param {Request} 	req Request Object
 * @param {Response}    res Response Object
 */
exports.getStapelREST = function(req, res){

	res._Context = {} 											// add Context to the response instance
	res._Context.date = new Date()								// add the Current Date to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date)	// add a Request Id to the Context

	// Build the res._Context.nils as used in _callService
	res._Context.nils = {
		'stackId': req.params.stackid,		
		'service': GET_STAPEL,
		'functionName': 'processRESTreq',
		'err':{
			'module': PLUGIN_NAME
		}
	}

	logger.info('--->getStapelREST: Req.Id=' + res._Context.reqId + ', Parms=' + JSON.stringify(res._Context.nils))
	
	_callService(res, function(err){
		if(err) return res.status(404).end('--->getStapelREST: Req.Id=' + res._Context.reqId + ', Request failed, Reason: ' + err.mesage)
		res.json(res._Context.nils.spResult)
	})
}

/**
 * Service to handle requests from nils 
 * @param  {Response}  	res Response Object
 * 						res._Context':{
 * 							'nils': {
 * 								'service' : <service name>,
 * 								'req':{
 * 									'parms': 
 * 										{
 * 											xml: <SOAP XML string>
 *										}
 * @param  {Function} 	callback(err)
 */
exports.nilsService = function(job, callback){

	var res 			= job.data.res  
	  , totalTasks 		= 4
	  , pendingTasks 	= totalTasks

	// When Job is saved, kue converts res._Context.date to string, so convert it back to date object
	if(res._Context.date && typeof res._Context.date === 'string') res._Context.date = new Date(res._Context.date)

	// Add the Current Date to the Context
	res._Context.date = res._Context.date || new Date()

	// Add a Request Id to the Context
	res._Context.reqId = res._Context.reqId || utils.buildReqId(res._Context.date)

	// Add the temp Dir to the Context
	res._Context.tempDir = res._Context.tempDir || npsTempDir + FILE_SEP + res._Context.reqId
	
	res._Context.nils		= res._Context.nils 		|| {}
	res._Context.nils.req	= res._Context.nils.req 	|| {'parms': {}}
	res._Context.nils.err	= res._Context.nils.err		|| {'module': PLUGIN_NAME}
	 
	if(logger.isDebugEnabled()) logger.debug('--->nilsService: Req.Id=' + res._Context.reqId + ', Parms=' + JSON.stringify(res._Context.nils.req.parms))
	async.series([
		// Task 1: Parse XML
		function(nextTask){
			job.progress(totalTasks - --pendingTasks, totalTasks)
			_parseXML(res, nextTask)
		},
		// Task 2: Call service
		function(nextTask){
			job.progress(totalTasks - --pendingTasks, totalTasks)
			_callService(res, nextTask)
		},
		// Task 3: Convert result JSON to SOAP XML
		function(nextTask){
			_resultJSON2XML(res, nextTask)
		},
		// Task 4: Send HTTP Response
		function(nextTask){
			job.progress(totalTasks - --pendingTasks, totalTasks)
			_sendHTTPResponse(res, nextTask)
		}
	],
	function(err, results){
		job.progress(totalTasks - --pendingTasks, totalTasks)
		if(err){
			logger.error('--->nilsservice: Req.Id=' + res._Context.reqId +  ', Error, Reason: ' + err.message + 
								 (res._Context.nils.infoText ? ( ', ' + res._Context.nils.infoText) : ''))
			res._Context.err = err
			_sendHTTPResponse(res, function(retErr){
				if(retErr){ 
					logger.error('--->nilsservice: Req.Id=' + res._Context.reqId + ', Unable to send error response. ' + retErr.message)
					return callback(err, 'Unable to send error response. ' + retErr.message + '. Request error: ' + err.message)
				}
				callback(err)
			})
			return
		}
		logger.info('--->nilsservice: Req.Id=' + res._Context.reqId + 
					  ', Finished processing "' + res._Context.nils.service + '" request ')	
		callback()
	})
}

/************ OPTIONAL PUBLIC FUNCTIONS ***********/
/**
 * Set Log Level
 * @param {String} logLevel Log Level. values:DEBUG|INFO|WARN|ERROR 
 */
exports.setLogLevel = function(logLevel){
    let servicesDir = path.resolve(__dirname + '/services') 

	pluginConf.log.level = logLevel || 'INFO'
    utils.log4jsSetLogLevel(logger, pluginConf.log.level)
    
    // Set Log level for services
	utils.getFileList(servicesDir, '.js', true/*caseSensitive*/, false/*recursive*/, function(err,result){

		if(err) return logger.error('--->setLogLevel: Unable to set log level for plugin services. Reason: ' + err.message)

		if(logger.isDebugEnabled())
			logger.debug('--->setLogLevel: servicesDir: ' + servicesDir + ', Files: ' + JSON.stringify(result))
        
        if(!result || result.length <=0) return

		// Loop thru the list of services and set log level
		async.eachOfLimit(result,  5,
			function(filename, index, next){
				try{
					let module = require(filename) // load service
					if(!module) return logger.error('--->setLogLevel: Error loading service: ' + filename + ', require statement failed')
					if(typeof(module.setLogLevel) !== 'function') return next()
                    module.setLogLevel(logLevel)
                    next()
				}catch(err){
                    logger.error('--->setLogLevel: Error loading service: ' + filename + ', ' + err.message) 
                    return next()
				}				
			},
			function(){
				return
			}
		)
	})
}

/**
 * Get Log Level
 * @return {String} Log Level. values:DEBUG|INFO|WARN|ERROR
 */
exports.getLogLevel = function(){
	return pluginConf.log.level
}

/**
 * Enable the given route
 * @param  {Request} 	req
 *					  req.body = {
 *					  	"module":"sample", 
 *					  	"route": { 
 *					  		"path": "/path/to/the/service",
 *					  		"method": 	Optional. Values are "get|post|dw|queue",
 *					  		"service": 	"version"
 *					  	}
 *					  }
 * @param  {function} 	callback callback(err)
 */
exports.enableRoute = function(req, callback){

	let self	= this 
	  , resTmp = {'_Context':{'plugin': PLUGIN_NAME, 'jobState': 'inactive'}}

	if(!self._initialized) return callback(new Error(PLUGIN_NAME + ' plugin not initialized. ' + (initError ? initError.message: '')))

	if(logger.isDebugEnabled()) logger.debug('--->enableRoute: Route=' + JSON.stringify(req.body.route))

	if(!req.body.route) return callback(new Error('route parameter not given'))

	req.body.route.method = req.body.route.method || 'get'

	resTmp._Context.jobTypepattern = '^' + PLUGIN_NAME + '_' + req.body.route.path +'$'
	// Enable the route
	switch(req.body.route.method.toLowerCase()){
		case 'dir': // Enable Dir Route
			// pluginManager.callPluginService({name:'oxsnps-dirwatcher',service:'addListenersService'}, [req.body.route], callback)
			break
		case 'queue': // Enable Queue Route
	   		// Delete Queued Jobs before enabling route 
			// pluginManager.callPluginService({name:'oxsnps-ibmmq', service:'addListenersService'}, [req.body.route], callback)
			break
		default:
            // Enable HTTP Route
            if(!self.router) return callback(new Error('Unable to enable a route, since router instance not initialized for plugin:' + PLUGIN_NAME))   // return error since we can not enabel a route since router is undefined
			// Enable HTTP Route
			oxsnpsJob.cleanJobsService(resTmp, function(err){
				if(err){
					logger.error('--->enableRoute Error: ' + err.message);
					return callback(new Error('--->enableRoute Error: ' + err.message));
				}
                return expressAppUtils.enableRoute(self, req.body.route, callback)
            })
			break
	}
} 

/**
 * Disable the given route
 * @param  {Request} 	request Request json has following structure.
 * 						req.body = {
 *					  	"module":"sample", 
 *					  	"route": { 
 *					  		"path": "/path/to/the/service",
 *					  		"method": 	Optional. Values are "get|post|dw|queue"
 * 						  	}
 * 						}
 * @param  {function} 	callback  callback(err)
 */
exports.disableRoute = function(req, callback){

    let self	= this
	  , resTmp  = {'_Context':{'plugin': PLUGIN_NAME, 'jobState': 'inactive'}}

	if(!self._initialized) return callback(new Error(PLUGIN_NAME + ' plugin not initialized. ' + (initError ? initError.message: '')))

	if(logger.isDebugEnabled()) logger.debug('--->disableRoute: Route=' + JSON.stringify(req.body.route))
	
	if(!req.body.route) return callback(new Error('route parameter not given'))

	req.body.route.method = req.body.route.method || 'get'

	resTmp._Context.jobTypepattern = '^' + PLUGIN_NAME + '_' + req.body.route.path +'$'

	// Disable the route
	switch(req.body.route.method.toLowerCase()){
		// Disable Dir Route
		case 'dir': 
			// pluginManager.callPluginService({name:'oxsnps-dirwatcher',service:'removeListenersService'}, [req.body.route], callback)
			break
		case 'queue':
			// Disable Queue Route
			// pluginManager.callPluginService({name:'oxsnps-ibmmq', service:'removeListenersService'}, [req.body.route], callback)
			break
		default:
            // Disable HTTP Route
            if(!self.router) return callback(new warn('Unable to disable a route, since router instance not initialized for plugin: ' + PLUGIN_NAME))    // return warning and not error, since route is already not accessible since router is undefiend
			expressAppUtils.disableRoute(self, req.body.route, function(err){
				if(err) return callback(err)
				// Delete Jobs if route is disabled successfully
				oxsnpsJob.cleanJobsService(resTmp, function(err){
	   				if(err) logger.error('--->disableRoute Error: ' + err.message)
					return callback()
		   		})			
            })            
			break
	}
} 

/**************** PRIVATE FUNCTIONS ***************/

/**
 * Parse XML
 * @param  {Response}	res 		Response Object
 * @param  {Function}	callback 	callback(err)
 */
function _parseXML(res,  callback){

	let nilsErr 			= res._Context.nils.err
	  , parseOptions 		= undefined
	  , keys 				= undefined
	  , filename			= pluginLogDir + FILE_SEP + res._Context.reqId + '_soapreq.xml'

	nilsErr.func = '_parseXML'

	parseOptions = {
		explicitArray: 		false, 
		valueProcessors: 	[utils.replaceHexWithUTF8Char],
		attrValueProcessors:[utils.replaceHexWithUTF8Char]
		// by default strips namespace prefixes
	}

	async.series([
		// Write input XML to a file
		function(nextTask){
			if(logger.isDebugEnabled()){
				logger.debug('--->_parseXML: Req.Id=' + res._Context.reqId + ', Writing SOAP XML request to ' + filename + '...')
				fs.writeFile(filename, res._Context.nils.req.parms.xml, function(err){
					if(err){ 
						// Continue process even if writing has failed for files, created to store input XML messages
						logger.error('--->_parseXML: Req.Id=' + res._Context.reqId + 
									', Unable to write SOAP XML request message to ' + filename + '. ' + err.message)
					}		
					return
				});
			}
			nextTask()
		},		
		function(nextTask){
			if(logger.isDebugEnabled()) logger.debug('--->nilsService: Req.Id=' + res._Context.reqId + ', Parsing XML request... ')
			utils.xml2json(res._Context.nils.req.parms.xml, parseOptions, function(err, result){
				if(err){
					nilsErr.code 	= 'E1005'
					nilsErr.parms 	= {'msg':err.message}
					return callback(error(nilsErr))
				} 

				if(logger.isDebugEnabled()) logger.debug('--->_parseXML: Req.Id=' + res._Context.reqId + ', Input JSON=' + JSON.stringify(result))
				res._Context.nils.inputJson = result
				if(result && result['Envelope'] && result['Envelope']['Body']){
					keys = Object.keys(result['Envelope']['Body'])
					if(keys && keys.length >0) res._Context.nils.service = keys[0]
				}
				if(!res._Context.nils.service){
					nilsErr.code 	= 'E1002'
					nilsErr.parms 	= {'reason': 'Request does not contain service name '}
					return callback(error(nilsErr))
				}
				res._Context.nils.infoText = 'Service:' + res._Context.nils.service
		
				if(logger.isDebugEnabled()) logger.debug('--->_parseXML: Req.Id=' + res._Context.reqId + ', Service=' + res._Context.nils.service)
				return callback()
			})
		}
	],	function(err){
		callback(err)
	});	  					
}

/**
 * Call Service Module
 * @param  {Response}	res 		Response Object
 * @param  {Function}	callback 	callback(err, result)
 */
function _callService(res, callback){
	
	let module 			= undefined
	  , moduleName 		= __dirname + FILE_SEP + 'services' + FILE_SEP + res._Context.nils.service + '.js'
	  , retValue 		= undefined
	  , nilsErr 		= res._Context.nils.err  	  
	  , functionName	= undefined
	  

	// Get module handle	  
	nilsErr.func = '_callService'	  
	nilsErr.code	= 'E1004'
	try{
		module = require(moduleName) // load service module
        if(!module){
			nilsErr.parms = {'pluginName': moduleName, 'msg': 'Module handle is null'}
			return callback(error(nilsErr))
		}
    }catch(err){
		nilsErr.parms = {'pluginName': moduleName, 'msg': err.message}
		return callback(error(nilsErr))
	}    
		  
	nilsErr.code	= 'E1003'
	
	functionName = res._Context.nils.functionName || 'process'

	if(typeof(module[functionName]) !== 'function'){
		nilsErr.parms = {'pluginName': moduleName, 'pluginService': functionName, 'msg': functionName + '() function does not exist' }
		return callback(error(nilsErr))
	}

	// Call service 
	try{
		module[functionName](res, callback)		// V2.4.2
	}
	catch(err){
		nilsErr.parms = {'pluginName': moduleName, 'pluginService': functionName, 'msg': err.message }		
		return callback(error(nilsErr))
	}
}

/**
 * Convert result JSON to SOAP XML
 * @param  {Response}	res 		Response Object
 * @param  {Function}	callback 	callback(err, result)
 */
function _resultJSON2XML(res, callback){

	let buildOpts 		= undefined
	  , enc				= npsConf.encoding || 'binary'
	  , nilsErr 		= res._Context.nils.err
	  , outputFilename	= pluginLogDir + FILE_SEP + res._Context.reqId + '_soapres.xml'

	nilsErr.func 	= '_resultJSON2XML'
	nilsErr.code 	= 'E1007'
		
	buildOpts={
		renderOpts:{
			'pretty': (logger.isDebugEnabled()?true:false)	// pretty print if debug mode 
		},
		xmldec: {
			'version': 		'1.0',
			'encoding': 	utils.getEncoding(enc)
		}
	}			

	async.series([
		// convert JSON to XML
		function(nextTask){
			if(logger.isDebugEnabled()) logger.debug('--->_resultJSON2XML: Req.Id=' + res._Context.reqId + ', Converting response JSON to XML...')

			utils.json2xml(res._Context.nils.soapResult, buildOpts, function(err, result){
				if(err){
					nilsErr.parms = {'json': JSON.stringify(res._Context.nils.soapResult), 'msg': err.message}
					return nextTask(error(nilsErr))
				}
				res._Context.nils.soapResultXML = result  // XML string (encoded in utf8)
				if(logger.isDebugEnabled()) logger.debug('--->_resultJSON2XML: Req.Id=' + res._Context.reqId + ', XML Response:' + res._Context.nils.soapResultXML)
				nextTask()
			})
		},

		// Write message to a file always
		function(nextTask){
			if(logger.isDebugEnabled()){
				logger.debug('--->_resultJSON2XML: Req.Id=' + res._Context.reqId + ', Writing SOAP XML response to ' + outputFilename + '...');
				fs.writeFile(outputFilename, res._Context.nils.soapResultXML, function(err){
					if(err){ 
						// Continue process even if writing has failed for files, created to store MQ messages
						logger.error('--->_resultJSON2XML: Req.Id=' + res._Context.reqId + 
									', Unable to write SOAP XML response message to ' + outputFilename + '. ' + err.message);
					}		
					return;
				});
			}
			nextTask();
		}
	],	function(err){
			callback(err);
		}
	);	  		
}

/**
 * Send HTTP Response
 * @param  {Response}   res 		Response object
 * @param  {Function} 	callback 	callback(err)
 */
function _sendHTTPResponse(res, callback){
	
	let resObj = responseObjs[res._Context.reqId]

	if(!resObj){
		logger.error('--->_sendHTTPResponse: Req. Id=' + res._Context.reqId + ', Response Object is null')		
		return callback()
	} 

	responseObjs[res._Context.reqId] = undefined 
	
	resObj.setHeader('Content-Type', 'text/xml');		

	if(res._Context.err)
		resObj.status(404).send(res._Context.err.message)
	else 
		resObj.send(res._Context.nils.soapResultXML)
	
	logger.info('--->_sendHTTPResponse: Req. Id=' + res._Context.reqId + ', ' + res._Context.nils.service + ' request  over.')
	return callback()
}

/**
 * Get request parameters
 * @return  JSON 	parms  Parameter json with key/value pairs
 * 
 */
function _getRequestParms(req){

	let  parms 		  = {}
	  , contenttype = undefined 

	if(!req) return parms

	if(req.headers) contenttype = req.headers['content-type']
	
	if(logger.isDebugEnabled()) logger.debug('--->_getRequestParms: Contenttype: ' + contenttype)

	// Get Query parameters if any, and add to req.parms
	if(req.query){
		Object.keys(req.query).forEach(function(key) {
			parms[key] = req.query[key]
		})
	}
	// Get parameters from body if any, and add to req.parms
  if(req.method && req.method.toLowerCase() === 'post' && req.body){
		if(contenttype && contenttype.toLowerCase().indexOf('xml')>=0){ // 'text/xml', 'application/xml', '*/xml'
			parms.xml = req.body
			if(logger.isDebugEnabled()) logger.debug('--->_getRequestParms: req.body:\n' + req.body)
		}
		else{ // 'application/json' ?		
			Object.keys(req.body).forEach(function(key) {
				parms[key] = req.body[key]
			})
		}
	}
	return parms
}

/**
 * Delete any existing active/inactive jobs of that plugin only
 * @param  {String}   method   active|inactive
 * @param  {String}   pattern  Job type pattern
 * @param  {Function} callback
 */
/*
function _cleanJobs(method, pattern, callback){

	var jobCount = 0
	  , belongsToPlugin = new RegExp('^' + PLUGIN_NAME, 'i') // same as /^ci.wps_/i which means starting with 'ci.wps_' should match ci.wps_QDS001N.REQ.WPSDN.TOH2WDMS.DS29
	  

	if(typeof pattern === 'string') belongsToPlugin = new RegExp(pattern, 'i')
	else callback = pattern

	jobs[method](function(err, ids){
	    if(err){
	    	logger.error('--->_cleanJobs: Error when fetching a ' + method + ' job, Reason: ' + err.message)
	    	return callback(err)
	    }
		jobCount = ids.length
		logger.debug('--->_cleanJobs: ' + method + ' Job Count=' + jobCount)
		if(jobCount <= 0) return callback()
		ids.forEach(function(id){
		    kue.Job.get(id, function(err, job){
			    if(err) logger.error('--->_cleanJobs: Error when fetching a remaining ' + method + ' job')

			    // If job.type starts with PLUGIN_NAME, then delete it
			    else if(belongsToPlugin.test(job.type)){
				    job.remove(function(err){
				    	if (err) logger.error('--->_cleanJobs: Error when trying to remove an ' + method + ' job, id: ' + job.id)
				    	else logger.info('--->_cleanJobs: Removed remaining ' + method + ' job, id: ' + job.id)
				    })
			    }
			    if(--jobCount <= 0) return callback()
		    })
	  	})
	})
}
*/

/**
 * Create and queue a Job
 * @param  {JSON} 	res
 *         			res._Context:{
 *         				'queueParms' :{
 *         					'manager': 		<manager name>,
 *         					'queue': 		<queue name>,
 *         					'messageId': 	<messageId>,
 *         					'encoding': 	'utf8'
 *         				},
 *         				'ciXXXReq':{
 *         				},
 *         				'date' : 	 <current date>,
 *         				'reqId': 	 <ReqId>,
 *         				'service': 	 <the service that will be called once the job will be triggered>,
 *         				'titleInfo': <Title that occurs in kue-dashboard>
 *         			}
 */
/*
function _createJob(res){

	// Create and queue the job
	var job = jobs.create(PLUGIN_NAME + '_' + res._Context.path, { // ex.: ci.wps_QDS001N.REQ.WPSDN.TOH2WDMS.DS29	// V2.4.2
		'title': res._Context.reqId + ', ' + res._Context.titleInfo,
		'jobid': res._Context.reqId,
		'res': 	 res
	}).removeOnComplete(true).save(function(err){
		if(err) return logger.error('_createJob: Unable to create a kue job, Reason: ' + err.message)
		if(logger.isDebugEnabled()) 
			logger.debug('--->_createJob: Req.Id=' + res._Context.reqId + ', Job created. Kue Job.Id=' + job.id + ', params:' + JSON.stringify(job.data))
	})

	// Add Job event handlers
	job.on('enqueue', function(result){
		if(logger.isDebugEnabled()){
			_getPendingJobsCount(job.type, function(err,pendingJobs){
				logger.debug('--->_createJob: Req.Id=' + job.data.res._Context.reqId + ', Job queued. ' + 
							 'Pending Jobs: ' + pendingJobs + ' for ' + job.type)
			})
		}
	})	
	job.on('start', function(result){
		if(logger.isDebugEnabled()){
			_getPendingJobsCount(job.type, function(err,pendingJobs){
				logger.debug('--->_createJob: Req.Id=' + job.data.res._Context.reqId + ', Job started. Kue Job.Id=' + job.id +
							 ', Pending Jobs: ' + pendingJobs + ' for ' + job.type)
			})
		}

	})
	job.on('complete', function(result){
		if(logger.isDebugEnabled()){
			_getPendingJobsCount(job.type, function(err,pendingJobs){
				logger.debug('--->_createJob: Req.Id=' + job.data.res._Context.reqId + ', Job completed. ' + 
							 'Pending Jobs: ' + pendingJobs + ' for ' + job.type)
			})
		}
	})
	job.on('failed', function(errorMessage){
		if(logger.isDebugEnabled()){
			_getPendingJobsCount(job.type, function(err,pendingJobs){
				logger.debug('--->_createJob: Req.Id=' + job.data.res._Context.reqId + ', Job failed. ' + 
							 'Pending Jobs: ' + pendingJobs + ' for ' + job.type + ', Reason=' + errorMessage)
			})
		}
	})			
	job.on('remove', function(result){
		if(logger.isDebugEnabled())
			logger.debug('--->_createJob: Req.Id=' + job.data.res._Context.reqId + ', Job removed. Kue Job.Id=' + job.id)	
	})
}	
*/

/**
 * Add a queue handler for each and every mqueue route defined in the plugin's configuration
 * @param {Function} callback
 */
/*
function _addKuehandlers(callback){

	// Add a queue handler for each and every mqueue route defined in the plugin's configuration
	async.each(pluginConf.routes,
		function(route, nextTask){
			if(route.path){ // add route if only if method is queue (add job.type as ci.wps_QDS001N.REQ.WPSDN.TOH2WDMS.DS29)	// V2.4.2
				jobs.process(PLUGIN_NAME + '_' + route.path, route.maxJobs || 1, function(job, done){
					setTimeout(function(){exports[job.data.res._Context.service](job, done) // call the service passing res to it
					},0)
				})
			}
			nextTask()
		},
		function(err){
			callback(err)
		}
	)
}
*/

/**
 * Delete any existing active/inactive jobs of that plugin only
 * @param  {String}   method   active|inactive
 * @param  {Function} callback
 */
/*
function _getPendingJobsCount(jobType, callback){

	var jobCount 		= 0
	  , pendingJobsCount= 0
	  , belongsToPlugin = new RegExp('^' + PLUGIN_NAME, 'i') // same as /^ci.wps_/i which means starting with 'ci.wps_' should match ci.wps_QDS001N.REQ.WPSDN.TOH2WDMS.DS29
	  , method 			= 'inactive'
	  

	jobs[method](function(err, ids){
	    if(err){
	    	logger.error('--->_getPendingJobsCount: Error when fetching a ' + method + ' job, Reason: ' + err.message)
	    	return callback(err, 'unknown')
	    }
		jobCount = ids.length
		if(jobCount <= 0) return callback(null, pendingJobsCount)
		ids.forEach(function(id){
		    kue.Job.get(id, function(err, job){
			    if(err) logger.error('--->_getPendingJobsCount: Error when fetching a ' + method + ' job')
			    else if(job.type === jobType){
			    	pendingJobsCount++
			    }
			    if(--jobCount <= 0) return callback(null, pendingJobsCount)
		    })
	  	})
	})
}
*/

/**
 * Get Job count for a passed job state and type
 * @param  {[String]}   jobState Job state like active|inactive
 * @param  {[String]}   jobType  Job Type
 * @param  {Function} 	callback callback(err, jobCount)
  */
 /*
function _getJobsCount(jobState, jobType, callback){

	var jobsCount = 0

	if(!jobState) return callback(new Error('_getJobsCount: Job state parameter is missing'))
	if(!jobType)  return callback(new Error('_getJobsCount: Job type parameter is missing'))
	
	logger.debug('--->_getJobsCount: State=' + jobState + ', Gettting Job count ...')	
	jobs[jobState](function(err, ids){
	    if(err) return callback(new Error('--->_getJobsCount: Unable to fetch ' + jobState + ' job ids, ' + err.message))
		
		logger.debug('--->_getJobsCount: State=' + jobState + ', Count='+ ids.length)	
		if(ids.length <= 0) return callback(null, 0)

		logger.debug('--->_getJobsCount: State=' + jobState + ', JobType=' + jobType + ', Finding Job count ...')			
		async.each(ids, 
			function(id, next){
			    kue.Job.get(id, function(err, job){
				    if(err) return next(new Error('--->_getJobsCount: Unable to fetch a ' + jobState + ' job(id=' + id + ')'))
				    if(job.type === jobType) jobsCount++
			    })
			},
			function(err){
				if(!err) logger.debug('--->_getJobsCount: State=' + jobState + ', JobType=' + jobType + ', Count=' + jobsCount)	
				return callback(err, jobsCount)
			}
		)
	})
}
*/

/**
 * Log a Warning if Service Response time exceeded maxresptime 
 * @param  {Response}	res  	Response Object
 * @param  {String}		service service
 * @private
 */
function _logServiceRespTime(res, service){

	let respTime 	= 0
	  , os 			= require('os')
	  

	// Check resp. time only if maxresptime is defined
   	if(pluginConf.maxresptime){
		respTime = utils.dateDiff('x', res._Context.date, new Date()) // 'x' means returns diff. in ms

   		if(respTime > pluginConf.maxresptime * 1000){
			logger.warn('--->' + service + ': Req.Id=' + res._Context.reqId + ', '
						+ ' Resp. Time: ' + respTime/1000 + 's, Max. allowed: '
						+ pluginConf.maxresptime + 's. Average Load: ' + os.loadavg())
   		}
   	}
}

// V3.0.0 Begin
/**
 * Disable all plugin routes
 * @param {*} err 
 * @param {*} callback 
 */
function _disableAllRoutes(err, callback){

    let resTmp  = undefined

    resTmp = {
        '_Context': {
            'reqId': utils.buildReqId(new Date()),
            'disableRoutesReq': {
                'module': PLUGIN_NAME,
                'reason': {
                    'error':{'msg': err.message}
                }    
            }
        }
    }
   
    if(logger.isDebugEnabled()) logger.debug('--->_disableAllRoutes: Req. id=' + resTmp._Context.reqId + ', Disabling all the routes of a plugin: ' + resTmp._Context.disableRoutesReq.module)

    pluginManager.disablePluginRoutesService(resTmp, function(disableErr){
        if(disableErr) logger.error('--->_disableAllRoutes: Req. id=' + resTmp._Context.reqId + ', Unable to disable all the routes of a plugin: ' + PLUGIN_NAME + '. Reason: ' + disableErr.message)
        else logger.warn('--->_disableAllRoutes: Req. id=' + resTmp._Context.reqId + ', Disabled all the routes of a plugin: ' + resTmp._Context.disableRoutesReq.module)
        callback(err, pluginConf.routes)    // return original error passed as parameter to this function
    })
}
// V3.0.0 End
