/**-----------------------------------------------------------------------------
 * getStapel.js: Service to get stapel information
 *
 * Author    :  AFP2web Team
 * Copyright :  (C) 2018 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  1.0.0
 * 
 * History
 *----------------------------------------------------------------------------*/
'use strict'

/**
 * letiable declaration
 * @private
 */
let async           = require('async')  
  , fs				= require('fs')
  , log4js          = require('log4js')
  , path            = require('path')  
  , packageJson     = require(__dirname + '/../package.json')
  , PLUGIN_NAME     = packageJson.name
  , npsServer       = require('oxsnps-core/server')
  , npsConf         = npsServer.npsConf
  , npsLogDir       = npsConf.logDir
  , npsLogFile      = npsServer.npsLogFile
  , pluginConf      = require(__dirname + '/../conf/' + PLUGIN_NAME + '.js')
  , dateFormat      = require('oxsnps-core/helpers/date_format')
  , error 			= require('oxsnps-core/error').createError
  , utils           = require('oxsnps-core/helpers/utils')
  , httpUtils  		= require('oxsnps-core/helpers/httpUtils')    
  , pluginManager   = require('oxsnps-core/pluginManager') 
  , FILE_SEP        = require('path').sep  
  , dms2common      = undefined
  , SERVICE_NAME    = 'getStapel'
  , logger          = log4js.getLogger(PLUGIN_NAME + '/' + SERVICE_NAME)    
  , pluginLogDir	= undefined
  , SP_INFO_MAX_LEN = 32000
  , sp232Defaults   = undefined
  
utils.log4jsSetLogLevel(logger, (pluginConf.log.level || 'WARN'))

/**
 * Initialize Plugin
 */
exports.initialize = function(){

    let self = this
    
    if(self._initialized) return callback()

    logger.debug('--->initializing getStapel.js...' )

    pluginLogDir = path.resolve(npsLogDir) + FILE_SEP + PLUGIN_NAME

    self._initialized = true
}

exports.setLogLevel = function(logLevel){
	utils.log4jsSetLogLevel(logger, (logLevel || 'WARN'))
}

/**
 * Function to process getStapel request
 * @param  {Response}	res 		Response Object
 * 	                    res._Context.nils = {
 *                          'req':{
 *                  			'parms': {
 *                  				'stackId': <stackId>
 *                  			}
 *                      	}, 
 *                  		'service':      'getStapel',
 *                  		'functionName': 'processRESTreq'
 *                  		'err':{
 *                  			'module':   'nils.utils'
 *                  		}
 *                  	}
 * @param  {Function}	callback 	callback(err, result)
 */
exports.processRESTreq = function(res, callback){

    let self        = this
      , nilsErr 	= res._Context.nils.err  	  

      if(!self._initialized) exports.initialize()

    async.series([
        // Get dms2common handle
        function(nextTask){
            if(dms2common) return nextTask()
            if(logger.isDebugEnabled()) logger.debug('--->getStapelREST: Req.Id=' + res._Context.reqId + ', Getting dms2common plugin handle...')
            pluginManager.getModule('dms2.common', function(err, module){
                if(err){
                    nilsErr.func 	= 'getStapelREST'	  
                    nilsErr.code    = 'E1004'
                    nilsErr.parms   = {'pluginName': 'dms2.common', 'msg': err.message}
                    return nextTask(error(nilsErr))
                }
                dms2common = module
                return nextTask()
            })
        },        
        // call SP
        function(nextTask){
            _callSP(res, nextTask)
        }
    ],
    function(err){
        callback(err)        
    })
}

/**
 * Function to process getStapel request
 * @param  {Response}	res 		Response Object
 * @param  {Function}	callback 	callback(err, result)
 */
exports.process = function(res, callback){

    let self        = this
      , retValue    = undefined
      , nilsErr 	= res._Context.nils.err  	  
      , req         = res._Context.nils.inputJson['Envelope']['Body'][SERVICE_NAME]

    if(!self._initialized) exports.initialize()

    // extract and assert parameters
    retValue = _assertParms(res)
    if(retValue && retValue instanceof Error) return callback(retValue)

    res._Context.nils.stackId = req.SmfStackId
    res._Context.nils.infoText += ', StackId: ' + res._Context.nils.stackId

    logger.info('--->process: Req.Id=' + res._Context.reqId + ', Starting processing getStapel request, ' + res._Context.nils.infoText)    

    async.series([
        // Get dms2common handle
        function(nextTask){
            if(dms2common) return nextTask()
            if(logger.isDebugEnabled()) logger.debug('--->process: Req.Id=' + res._Context.reqId + ', Getting dms2common plugin handle...')
            pluginManager.getModule('dms2.common', function(err, module){
                if(err){
                    nilsErr.func 	= 'process'	  
                    nilsErr.code	    = 'E1004'
                    nilsErr.parms    = {'pluginName': 'dms2.common', 'msg': err.message}
                    return nextTask(error(nilsErr))
                }
                dms2common = module
                return nextTask()
            })
        },        
        // call SP
        function(nextTask){
            _callSP(res, nextTask)
        },
        // Build result
        function(nextTask){
            _buildResult(res, nextTask) 
        }         
    ],
    function(err){
        callback(err)
    })
}

/************ OPTIONAL PUBLIC FUNCTIONS ***********/

/**************** PRIVATE FUNCTIONS ***************/

/**
 * Assert History input XML parametes
 */
function _assertParms(res){

    let nilsErr  = res._Context.nils.err
      , req     = res._Context.nils.inputJson['Envelope']['Body'][SERVICE_NAME]

	nilsErr.func = '_assertParms'
    nilsErr.code	= 'E1002'
     
    if(!req.SmfStackId || req.SmfStackId.length <=0){
		nilsErr.parms 	= {'reason': 'Request does not have SmfStackId element or its value is empty'}
		return error(nilsErr)
    }
    return    
}

/**
 * Call SP
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
function _callSP(res, callback){

	let spName 			= undefined
      , nilsErr 	    = res._Context.nils.err
      , args            = undefined
      , startTime       = undefined
      
	// Build SP args array
	args = [res._Context.nils.stackId]
    
	// Get Stored Procedure name
	spName = 'SP012_B01_NSQL_DS_R_DMS_GETHOLESCS'  
	if(npsConf.db2 && npsConf.db2.storedProcedures && npsConf.db2.storedProcedures.spGetStack)
		spName = npsConf.db2.storedProcedures.spGetStack
	
	res._Context.sp = {
		'name':     spName,
        'parms':    args,
        'resultFilename': pluginLogDir + FILE_SEP + res._Context.reqId + '.json'
	}

    nilsErr.func 	= '_callSP'      
    nilsErr.code     = 'E1006'
    nilsErr.parms 	= {}

    if(logger.isDebugEnabled()) logger.debug('--->_callSP: Req.Id=' + res._Context.reqId + ', Calling SP: ' + res._Context.sp.name + 
                    ' to get stack information, Parms: ' + JSON.stringify(res._Context.sp.parms))

    startTime   = new Date()

	// Call stored procedure to get Document List
	dms2common.callSP(res, function(err, result){		
		if(err){
            nilsErr.parms 	= {'msg': 	err.message}
            return callback(error(nilsErr))
        }
        /*
            result =
            [

                {
                    "resultSet":[
                        {
                            "ERRORCLASS":"null",
                            "ERRORNUMBER":0,
                            "ERRORMSG1":"null",
                            "ERRORMSG2":"null",
                            "ERRORMSG3":"null"
                        }
                    ]
                },
                {
                    "resultSet":[
                        {
                            "FTVG_SPARTITION_ID":0,
                            "FTVG_TSPROCESS_ID":"2013-07-23-09.15.16.532308",
                            "FTVG_CPUBLICWB_AK":" ",
                            "FTVG_SPROCESTYP_RI":15,
                            "FTVG_SPRIORITY":3,
                            "FTVG_CRECORDID_RI":"17937740_",
                            "FTVG_CWORKBASKET_AK":"XE00117_",
                            "FTVG_SWORKBASKETTYP_RI":1,
                            "FTVG_VCLONGINFO":"",
                            "FTVG_CSCNR":"08",
                            "FTVG_CVDNR":"64",
                            "FTVG_CVWHNR":"0000",
                            "FTDO_SPARTITION_ID":0,
                            "FTDO_TSDOCUMENT_ID":"2013-07-23-09.14.53.989442",
                            "FTDO_SDOCUTYPE_RI":320,
                            "FTDO_TSLASTUPDATE":"2014-01-24-11.31.28.013297",
                            "FTDO_SCOUNTPAGE":1,
                            "FTDO_DINOUTDATE":"2013-07-23",
                            "FTDO_CLOCKSTATE":" ",
                            "FTDO_CKOPIE":"O",
                            "FTDO_SSEQNRPROCESS":1,
                            "FTDI_CDOCID":"000",
                            "FTDO_IDOCUMENTKEY":1,
                            "FTDO_TSORIGDOCU_RI":"2013-07-23-09.14.53.989442",
                            "FTDI_CRBETRAG":"29,9",
                            "FTDI_CWAEHRUNG":"EU",
                            "FTDI_CDATUMVON":"2006-11-1 ",
                            "FTDI_CDATUMBIS":"2006-12-2 ",
                            "FTDI_CRKRKHNR":"",
                            "FTDI_VCDIAGNOSETEXT":"Knick-Spreizfu Gastroenteriti "
                        },
                        {
                            "FTVG_SPARTITION_ID":0,
                            "FTVG_TSPROCESS_ID":"2013-07-23-09.15.16.532308",
                            "FTVG_CPUBLICWB_AK":" ",
                            "FTVG_SPROCESTYP_RI":15,
                            "FTVG_SPRIORITY":3,
                            "FTVG_CRECORDID_RI":"17937740_",
                            "FTVG_CWORKBASKET_AK":"XE00117_",
                            "FTVG_SWORKBASKETTYP_RI":1,
                            "FTVG_VCLONGINFO":"",
                            "FTVG_CSCNR":"08",
                            "FTVG_CVDNR":"64",
                            "FTVG_CVWHNR":"0000",
                            "FTDO_SPARTITION_ID":0,
                            "FTDO_TSDOCUMENT_ID":"2013-07-23-09.14.53.966310",
                            "FTDO_SDOCUTYPE_RI":390,
                            "FTDO_TSLASTUPDATE":"2014-01-24-11.31.28.021146",
                            "FTDO_SCOUNTPAGE":2,
                            "FTDO_DINOUTDATE":"2013-07-23",
                            "FTDO_CLOCKSTATE":" ",
                            "FTDO_CKOPIE":"O",
                            "FTDO_SSEQNRPROCESS":2,
                            "FTDI_CDOCID":"000",
                            "FTDO_IDOCUMENTKEY":1,
                            "FTDO_TSORIGDOCU_RI":"2013-07-23-09.14.53.966310",
                            "FTDI_CRBETRAG":"122,0",
                            "FTDI_CWAEHRUNG":"EU",
                            "FTDI_CDATUMVON":"2006-12-2 ",
                            "FTDI_CDATUMBIS":"1900-01-01",
                            "FTDI_CRKRKHNR":"",
                            "FTDI_VCDIAGNOSETEXT":""
                        },
                        {
                            "FTVG_SPARTITION_ID":0,
                            "FTVG_TSPROCESS_ID":"2013-07-23-09.15.16.532308",
                            "FTVG_CPUBLICWB_AK":" ",
                            "FTVG_SPROCESTYP_RI":15,
                            "FTVG_SPRIORITY":3,
                            "FTVG_CRECORDID_RI":"17937740_",
                            "FTVG_CWORKBASKET_AK":"XE00117_",
                            "FTVG_SWORKBASKETTYP_RI":1,
                            "FTVG_VCLONGINFO":"",
                            "FTVG_CSCNR":"08",
                            "FTVG_CVDNR":"64",
                            "FTVG_CVWHNR":"0000",
                            "FTDO_SPARTITION_ID":0,
                            "FTDO_TSDOCUMENT_ID":"2013-07-23-09.14.53.979337",
                            "FTDO_SDOCUTYPE_RI":390,
                            "FTDO_TSLASTUPDATE":"2014-01-24-11.31.28.023870",
                            "FTDO_SCOUNTPAGE":2,
                            "FTDO_DINOUTDATE":"2013-07-23",
                            "FTDO_CLOCKSTATE":" ",
                            "FTDO_CKOPIE":"O",
                            "FTDO_SSEQNRPROCESS":3,
                            "FTDI_CDOCID":"000",
                            "FTDO_IDOCUMENTKEY":1,
                            "FTDO_TSORIGDOCU_RI":"2013-07-23-09.14.53.979337",
                            "FTDI_CRBETRAG":"93,0",
                            "FTDI_CWAEHRUNG":"EU",
                            "FTDI_CDATUMVON":"2006-12-2 ",
                            "FTDI_CDATUMBIS":"1900-01-01",
                            "FTDI_CRKRKHNR":"",
                            "FTDI_VCDIAGNOSETEXT":"Knick-Spreizfu "
                        },
                        {
                            "FTVG_SPARTITION_ID":0,
                            "FTVG_TSPROCESS_ID":"2013-07-23-09.15.16.532308",
                            "FTVG_CPUBLICWB_AK":" ",
                            "FTVG_SPROCESTYP_RI":15,
                            "FTVG_SPRIORITY":3,
                            "FTVG_CRECORDID_RI":"17937740_",
                            "FTVG_CWORKBASKET_AK":"XE00117_",
                            "FTVG_SWORKBASKETTYP_RI":1,
                            "FTVG_VCLONGINFO":"",
                            "FTVG_CSCNR":"08",
                            "FTVG_CVDNR":"64",
                            "FTVG_CVWHNR":"0000",
                            "FTDO_SPARTITION_ID":0,
                            "FTDO_TSDOCUMENT_ID":"2013-07-23-09.14.53.999518",
                            "FTDO_SDOCUTYPE_RI":320,
                            "FTDO_TSLASTUPDATE":"2014-01-24-11.31.28.026540",
                            "FTDO_SCOUNTPAGE":1,
                            "FTDO_DINOUTDATE":"2013-07-23",
                            "FTDO_CLOCKSTATE":" ",
                            "FTDO_CKOPIE":"O",
                            "FTDO_SSEQNRPROCESS":4,
                            "FTDI_CDOCID":"000",
                            "FTDO_IDOCUMENTKEY":2,
                            "FTDO_TSORIGDOCU_RI":"2013-07-23-09.14.53.999518",
                            "FTDI_CRBETRAG":"180,4",
                            "FTDI_CWAEHRUNG":"EU",
                            "FTDI_CDATUMVON":"2006-05-0 ",
                            "FTDI_CDATUMBIS":"2006-10-1 ",
                            "FTDI_CRKRKHNR":"",
                            "FTDI_VCDIAGNOSETEXT":"Myalgi Blockierung L5/S Dysfunktion der Papillarmuskel "
                        },
                        {
                            "FTVG_SPARTITION_ID":0,
                            "FTVG_TSPROCESS_ID":"2013-07-23-09.15.16.532308",
                            "FTVG_CPUBLICWB_AK":" ",
                            "FTVG_SPROCESTYP_RI":15,
                            "FTVG_SPRIORITY":3,
                            "FTVG_CRECORDID_RI":"17937740_",
                            "FTVG_CWORKBASKET_AK":"XE00117_",
                            "FTVG_SWORKBASKETTYP_RI":1,
                            "FTVG_VCLONGINFO":"",
                            "FTVG_CSCNR":"08",
                            "FTVG_CVDNR":"64",
                            "FTVG_CVWHNR":"0000",
                            "FTDO_SPARTITION_ID":0,
                            "FTDO_TSDOCUMENT_ID":"2013-07-23-09.14.53.987559",
                            "FTDO_SDOCUTYPE_RI":320,
                            "FTDO_TSLASTUPDATE":"2014-04-15-12.57.20.405210",
                            "FTDO_SCOUNTPAGE":2,
                            "FTDO_DINOUTDATE":"2013-07-23",
                            "FTDO_CLOCKSTATE":" ",
                            "FTDO_CKOPIE":"O",
                            "FTDO_SSEQNRPROCESS":5,
                            "FTDI_CDOCID":"000",
                            "FTDO_IDOCUMENTKEY":2,
                            "FTDO_TSORIGDOCU_RI":"2013-07-23-09.14.53.987559",
                            "FTDI_CRBETRAG":"150,8",
                            "FTDI_CWAEHRUNG":"EU",
                            "FTDI_CDATUMVON":"2006-12-1 ",
                            "FTDI_CDATUMBIS":"2006-12-1 ",
                            "FTDI_CRKRKHNR":"",
                            "FTDI_VCDIAGNOSETEXT":"AusschluÃŸ eines Verdachtsfalle "
                        },
                        {
                            "FTVG_SPARTITION_ID":0,
                            "FTVG_TSPROCESS_ID":"2013-07-23-09.15.16.532308",
                            "FTVG_CPUBLICWB_AK":" ",
                            "FTVG_SPROCESTYP_RI":15,
                            "FTVG_SPRIORITY":3,
                            "FTVG_CRECORDID_RI":"17937740_",
                            "FTVG_CWORKBASKET_AK":"XE00117_",
                            "FTVG_SWORKBASKETTYP_RI":1,
                            "FTVG_VCLONGINFO":"",
                            "FTVG_CSCNR":"08",
                            "FTVG_CVDNR":"64",
                            "FTVG_CVWHNR":"0000",
                            "FTDO_SPARTITION_ID":0,
                            "FTDO_TSDOCUMENT_ID":"2013-07-23-09.14.53.952156",
                            "FTDO_SDOCUTYPE_RI":10,
                            "FTDO_TSLASTUPDATE":"2014-01-24-11.31.28.031911",
                            "FTDO_SCOUNTPAGE":2,
                            "FTDO_DINOUTDATE":"2013-07-23",
                            "FTDO_CLOCKSTATE":" ",
                            "FTDO_CKOPIE":"O",
                            "FTDO_SSEQNRPROCESS":6,
                            "FTDI_CDOCID":"000",
                            "FTDO_IDOCUMENTKEY":1,
                            "FTDO_TSORIGDOCU_RI":"2013-07-23-09.14.53.952156",
                            "FTDI_CRBETRAG":"576,2",
                            "FTDI_CWAEHRUNG":"EU",
                            "FTDI_CDATUMVON":"1900-01-01",
                            "FTDI_CDATUMBIS":"1900-01-01",
                            "FTDI_CRKRKHNR":"",
                            "FTDI_VCDIAGNOSETEXT":""
                        },
                        {
                            "FTVG_SPARTITION_ID":0,
                            "FTVG_TSPROCESS_ID":"2013-07-23-09.15.16.532310",
                            "FTVG_CPUBLICWB_AK":" ",
                            "FTVG_SPROCESTYP_RI":15,
                            "FTVG_SPRIORITY":3,
                            "FTVG_CRECORDID_RI":"17937740_",
                            "FTVG_CWORKBASKET_AK":"XE00117_",
                            "FTVG_SWORKBASKETTYP_RI":1,
                            "FTVG_VCLONGINFO":"",
                            "FTVG_CSCNR":"08",
                            "FTVG_CVDNR":"64",
                            "FTVG_CVWHNR":"0",
                            "FTDO_SPARTITION_ID":0,
                            "FTDO_TSDOCUMENT_ID":"2014-07-10-12.38.27.769803",
                            "FTDO_SDOCUTYPE_RI":10,
                            "FTDO_TSLASTUPDATE":"2014-07-10-12.38.27.769803",
                            "FTDO_SCOUNTPAGE":2,
                            "FTDO_DINOUTDATE":"2013-07-23",
                            "FTDO_CLOCKSTATE":" ",
                            "FTDO_CKOPIE":"C",
                            "FTDO_SSEQNRPROCESS":1,
                            "FTDI_CDOCID":"0001",
                            "FTDO_IDOCUMENTKEY":1,
                            "FTDO_TSORIGDOCU_RI":"2013-07-23-09.14.53.952156",
                            "FTDI_CRBETRAG":"576,27",
                            "FTDI_CWAEHRUNG":"EUR",
                            "FTDI_CDATUMVON":"1900-01-01",
                            "FTDI_CDATUMBIS":"1900-01-01",
                            "FTDI_CRKRKHNR":"",
                            "FTDI_VCDIAGNOSETEXT":""
                        },
                        {
                            "FTVG_SPARTITION_ID":0,
                            "FTVG_TSPROCESS_ID":"2013-07-23-09.15.16.532310",
                            "FTVG_CPUBLICWB_AK":" ",
                            "FTVG_SPROCESTYP_RI":15,
                            "FTVG_SPRIORITY":3,
                            "FTVG_CRECORDID_RI":"17937740_",
                            "FTVG_CWORKBASKET_AK":"XE00117_",
                            "FTVG_SWORKBASKETTYP_RI":1,
                            "FTVG_VCLONGINFO":"",
                            "FTVG_CSCNR":"08",
                            "FTVG_CVDNR":"64",
                            "FTVG_CVWHNR":"0",
                            "FTDO_SPARTITION_ID":0,
                            "FTDO_TSDOCUMENT_ID":"2014-07-10-12.38.27.963963",
                            "FTDO_SDOCUTYPE_RI":320,
                            "FTDO_TSLASTUPDATE":"2014-07-10-12.38.27.963963",
                            "FTDO_SCOUNTPAGE":1,
                            "FTDO_DINOUTDATE":"2013-07-23",
                            "FTDO_CLOCKSTATE":" ",
                            "FTDO_CKOPIE":"C",
                            "FTDO_SSEQNRPROCESS":2,
                            "FTDI_CDOCID":"0007",
                            "FTDO_IDOCUMENTKEY":1,
                            "FTDO_TSORIGDOCU_RI":"2013-07-23-09.14.53.989442",
                            "FTDI_CRBETRAG":"29,95",
                            "FTDI_CWAEHRUNG":"EUR",
                            "FTDI_CDATUMVON":"2006-11-13",
                            "FTDI_CDATUMBIS":"2006-12-21",
                            "FTDI_CRKRKHNR":"",
                            "FTDI_VCDIAGNOSETEXT":"Knick-Spreizfuß Gastroenteritis "
                        },
                        {
                            "FTVG_SPARTITION_ID":0,
                            "FTVG_TSPROCESS_ID":"2013-07-23-09.15.16.532310",
                            "FTVG_CPUBLICWB_AK":" ",
                            "FTVG_SPROCESTYP_RI":15,
                            "FTVG_SPRIORITY":3,
                            "FTVG_CRECORDID_RI":"17937740_",
                            "FTVG_CWORKBASKET_AK":"XE00117_",
                            "FTVG_SWORKBASKETTYP_RI":1,
                            "FTVG_VCLONGINFO":"",
                            "FTVG_CSCNR":"08",
                            "FTVG_CVDNR":"64",
                            "FTVG_CVWHNR":"0",
                            "FTDO_SPARTITION_ID":0,
                            "FTDO_TSDOCUMENT_ID":"2014-07-10-12.38.27.971391",
                            "FTDO_SDOCUTYPE_RI":390,
                            "FTDO_TSLASTUPDATE":"2014-07-10-12.38.27.971391",
                            "FTDO_SCOUNTPAGE":2,
                            "FTDO_DINOUTDATE":"2013-07-23",
                            "FTDO_CLOCKSTATE":" ",
                            "FTDO_CKOPIE":"C",
                            "FTDO_SSEQNRPROCESS":3,
                            "FTDI_CDOCID":"0004",
                            "FTDO_IDOCUMENTKEY":1,
                            "FTDO_TSORIGDOCU_RI":"2013-07-23-09.14.53.966310",
                            "FTDI_CRBETRAG":"122,00",
                            "FTDI_CWAEHRUNG":"EUR",
                            "FTDI_CDATUMVON":"2006-12-27",
                            "FTDI_CDATUMBIS":"1900-01-01",
                            "FTDI_CRKRKHNR":"",
                            "FTDI_VCDIAGNOSETEXT":""
                        },
                        {
                            "FTVG_SPARTITION_ID":0,
                            "FTVG_TSPROCESS_ID":"2013-07-23-09.15.16.532310",
                            "FTVG_CPUBLICWB_AK":" ",
                            "FTVG_SPROCESTYP_RI":15,
                            "FTVG_SPRIORITY":3,
                            "FTVG_CRECORDID_RI":"17937740_",
                            "FTVG_CWORKBASKET_AK":"XE00117_",
                            "FTVG_SWORKBASKETTYP_RI":1,
                            "FTVG_VCLONGINFO":"",
                            "FTVG_CSCNR":"08",
                            "FTVG_CVDNR":"64",
                            "FTVG_CVWHNR":"0",
                            "FTDO_SPARTITION_ID":0,
                            "FTDO_TSDOCUMENT_ID":"2014-07-10-12.38.27.987439",
                            "FTDO_SDOCUTYPE_RI":390,
                            "FTDO_TSLASTUPDATE":"2014-07-10-12.38.27.987439",
                            "FTDO_SCOUNTPAGE":2,
                            "FTDO_DINOUTDATE":"2013-07-23",
                            "FTDO_CLOCKSTATE":" ",
                            "FTDO_CKOPIE":"C",
                            "FTDO_SSEQNRPROCESS":4,
                            "FTDI_CDOCID":"0005",
                            "FTDO_IDOCUMENTKEY":1,
                            "FTDO_TSORIGDOCU_RI":"2013-07-23-09.14.53.979337",
                            "FTDI_CRBETRAG":"93,00",
                            "FTDI_CWAEHRUNG":"EUR",
                            "FTDI_CDATUMVON":"2006-12-27",
                            "FTDI_CDATUMBIS":"1900-01-01",
                            "FTDI_CRKRKHNR":"",
                            "FTDI_VCDIAGNOSETEXT":"Knick-Spreizfuß "
                        },
                        {
                            "FTVG_SPARTITION_ID":0,
                            "FTVG_TSPROCESS_ID":"2013-07-23-09.15.16.532310",
                            "FTVG_CPUBLICWB_AK":" ",
                            "FTVG_SPROCESTYP_RI":15,
                            "FTVG_SPRIORITY":3,
                            "FTVG_CRECORDID_RI":"17937740_",
                            "FTVG_CWORKBASKET_AK":"XE00117_",
                            "FTVG_SWORKBASKETTYP_RI":1,
                            "FTVG_VCLONGINFO":"",
                            "FTVG_CSCNR":"08",
                            "FTVG_CVDNR":"64",
                            "FTVG_CVWHNR":"0",
                            "FTDO_SPARTITION_ID":0,
                            "FTDO_TSDOCUMENT_ID":"2014-07-10-12.38.27.989896",
                            "FTDO_SDOCUTYPE_RI":320,
                            "FTDO_TSLASTUPDATE":"2014-07-10-12.38.27.989896",
                            "FTDO_SCOUNTPAGE":1,
                            "FTDO_DINOUTDATE":"2013-07-23",
                            "FTDO_CLOCKSTATE":" ",
                            "FTDO_CKOPIE":"C",
                            "FTDO_SSEQNRPROCESS":5,
                            "FTDI_CDOCID":"0008",
                            "FTDO_IDOCUMENTKEY":2,
                            "FTDO_TSORIGDOCU_RI":"2013-07-23-09.14.53.999518",
                            "FTDI_CRBETRAG":"180,44",
                            "FTDI_CWAEHRUNG":"EUR",
                            "FTDI_CDATUMVON":"2006-05-05",
                            "FTDI_CDATUMBIS":"2006-10-13",
                            "FTDI_CRKRKHNR":"",
                            "FTDI_VCDIAGNOSETEXT":"Myalgie Blockierung L5/S1 Dysfunktion der Papillarmuskeln "
                        },
                        {
                            "FTVG_SPARTITION_ID":0,
                            "FTVG_TSPROCESS_ID":"2013-07-23-09.15.16.532310",
                            "FTVG_CPUBLICWB_AK":" ",
                            "FTVG_SPROCESTYP_RI":15,
                            "FTVG_SPRIORITY":3,
                            "FTVG_CRECORDID_RI":"17937740_",
                            "FTVG_CWORKBASKET_AK":"XE00117_",
                            "FTVG_SWORKBASKETTYP_RI":1,
                            "FTVG_VCLONGINFO":"",
                            "FTVG_CSCNR":"08",
                            "FTVG_CVDNR":"64",
                            "FTVG_CVWHNR":"0",
                            "FTDO_SPARTITION_ID":0,
                            "FTDO_TSDOCUMENT_ID":"2014-07-10-12.38.27.998770",
                            "FTDO_SDOCUTYPE_RI":320,
                            "FTDO_TSLASTUPDATE":"2014-07-10-12.38.27.998770",
                            "FTDO_SCOUNTPAGE":2,
                            "FTDO_DINOUTDATE":"2013-07-23",
                            "FTDO_CLOCKSTATE":" ",
                            "FTDO_CKOPIE":"C",
                            "FTDO_SSEQNRPROCESS":6,
                            "FTDI_CDOCID":"0006",
                            "FTDO_IDOCUMENTKEY":2,
                            "FTDO_TSORIGDOCU_RI":"2013-07-23-09.14.53.987559",
                            "FTDI_CRBETRAG":"150,87",
                            "FTDI_CWAEHRUNG":"EUR",
                            "FTDI_CDATUMVON":"2006-12-12",
                            "FTDI_CDATUMBIS":"2006-12-12",
                            "FTDI_CRKRKHNR":"",
                            "FTDI_VCDIAGNOSETEXT":"Ausschluß eines Verdachtsfalles "
                        }
                    ]
                }
            ]
        */
        res._Context.nils.spRespTime = utils.dateDiff('x', startTime, new Date()); // 'x' means returns diff. in ms
        res._Context.nils.spResult = result       		
        logger.debug('--->_callSP: Req.Id=' + res._Context.reqId + ', SPName: ' + res._Context.sp.name + ', Result: ' + JSON.stringify(result))
                                  
		callback()			
	})
}

/**
 * Returns a Processs as a JSON Object
 * @param  {Response}	res 		Response Object
 * @return {JSON}	Process JSON
 */
function _buildProcess(res){

    let row     = res._Context.nils.spRow
      , process =  undefined
    
	process = {	// definition of a Process
        "Partition": 				row.FTVG_SPARTITION_ID, 
        "VorgangKey": 				row.FTVG_TSPROCESS_ID,
        "VorgangTypKey": 			row.FTVG_SPROCESTYP_RI,
        "VorgangPrio": 				row.FTVG_SPRIORITY,
        "Versicherungsnummer": 		row.FTVG_CRECORDID_RI,
        "Arbeitskorb": 				row.FTVG_CWORKBASKET_AK,
        "ArbeitskorbTyp": 			row.FTVG_SWORKBASKETTYP_RI,
        "Betreff": 					row.FTVG_VCLONGINFO,
        "SC": 						row.FTVG_CSCNR,
        "VD": 						row.FTVG_CVDNR,
        "GruppenNummer": 			row.FTVG_CVWHNR,
        "DokumentListe":{
            "Dokument": undefined	// docArray
        }
    }
	return process
}

/**
 *  Returns a Document as a JSON Object
 * @param  {Response}	res 		Response Object
 * @return {JSON}	Document JSON
 */
function _buildDoc(res){

    let row     = res._Context.nils.spRow
      , doc     = undefined
      , idx     = undefined
      
    doc = {	// definition of a Document
        "Partition": 				row.FTDO_SPARTITION_ID,
        "DokumentKey": 				row.FTDO_TSDOCUMENT_ID,
        "DokumentTypKey": 			row.FTDO_SDOCUTYPE_RI,
        "LetzteAenderung": 			row.FTDO_TSLASTUPDATE,
        "AnzahlSeiten": 			row.FTDO_SCOUNTPAGE,
        "EingangDatum": 			row.FTDO_DINOUTDATE,
    //	"SperrKz": 					row.FTDO_CLOCKSTATE,	// will be added below only if not empty
    //	"KopieKz": 					row.FTDO_CKOPIE,		// will be added below only if not empty
        "LaufendeNrImVorgang": 		row.FTDO_SSEQNRPROCESS,
        "DocID": 					row.FTDI_CDOCID,
        "VpKey": 					row.FTDO_IDOCUMENTKEY,
        "OriginalDokumentKey": 		row.FTDO_TSORIGDOCU_RI,
        //"RechnungBetrag": 			row.FTDI_CRBETRAG.replace(",","."), // replaces , thru .  str.replace(",",".")
        "RechnungBetragWaehrung": 	row.FTDI_CWAEHRUNG,
        "BehandlungVonDatum": 		row.FTDI_CDATUMVON,
        "BehandlungBisDatum": 		row.FTDI_CDATUMBIS,
        "KrankenhausNr": 			row.FTDI_CRKRKHNR,
        "DiagnoseTextListe":{
            "DiagnoseText": []	// diagnosticArray
        }
	}

    if(row.FTDI_CRBETRAG && row.FTDI_CRBETRAG.length >0) doc.RechnungBetrag = row.FTDI_CRBETRAG.replace(",","."); // replaces , thru .  str.replace(",",".")

	// Add SperrKz if not Empty
	if(row.FTDO_CLOCKSTATE !== '') doc.SperrKz = row.FTDO_CLOCKSTATE

	// Add KopieKz if not Empty
	if (row.FTDO_CKOPIE !== '') doc.KopieKz = row.FTDO_CKOPIE
    
    // Build diagnoseTextListe
    row.FTDI_VCDIAGNOSETEXT = row.FTDI_VCDIAGNOSETEXT.trim() || ''
    if(row.FTDI_VCDIAGNOSETEXT.length <= 50)  doc.DiagnoseTextListe.DiagnoseText[0] = row.FTDI_VCDIAGNOSETEXT   // already trimmed
    else{
		for(idx = 0; idx < row.FTDI_VCDIAGNOSETEXT.length; idx=idx+50){
            doc.DiagnoseTextListe.DiagnoseText.push(row.FTDI_VCDIAGNOSETEXT.substr(idx, 50).trim()) // removes whitespace from both sides of the string
		}
    }
	return doc
}

/**
 * Build Result
 * @param  {Response}   res Response Object
 * @param  {Function} callback  callback(err, result)
 */
function _buildResult(res, callback){

    let spResult    = res._Context.nils.spResult[1].resultSet
      , jsonResult  = undefined
      , currentProcessId = ""
      , processList = []	// is an array of Processes
      , docArray    = undefined
      , process     = undefined
      , doc         = undefined

    jsonResult = {
        "soapenv:Envelope":{
            "$":{
                "xmlns:soapenv":"http://schemas.xmlsoap.org/soap/envelope/",
                "xmlns:soapenc":"http://schemas.xmlsoap.org/soap/encoding/",
                "xmlns:xsd":"http://www.w3.org/2001/XMLSchema",
                "xmlns:xsi":"http://www.w3.org/2001/XMLSchema-instance"
            },
            "soapenv:Body":{
                "ns1:getStapelResponse":{
                    "$":{
                        "xmlns:ns1":"http://hallesche.de/dms/nils/services/"
                    },
                    "GetStapelResult":{
                        "VorgangsListe":{
                            /*
                            "Vorgang":[
                                {
                                    "Partition":"0",
                                    "VorgangKey":"2018-10-01-15.39.36.845606",
                                    "VorgangTypKey":"64",
                                    "VorgangPrio":"2",
                                    "Versicherungsnummer":"26234628",
                                    "Arbeitskorb":"POSTEING",
                                    "ArbeitskorbTyp":"2",
                                    "Betreff":"",
                                    "SC":"05",
                                    "VD":"62",
                                    "GruppenNummer":"0",
                                    "DokumentListe":{
                                        "Dokument":[
                                            {
                                                "Partition":"0",
                                                "DokumentKey":"2018-10-01-15.38.29.462470",
                                                "DokumentTypKey":"76",
                                                "LetzteAenderung":"2018-10-01-15.39.36.867239",
                                                "AnzahlSeiten":"1",
                                                "EingangDatum":"2018-10-01",
                                                "LaufendeNrImVorgang":"1",
                                                "DocID":"",
                                                "VpKey":"2",
                                                "OriginalDokumentKey":"",
                                                "RechnungBetrag":"0",
                                                "RechnungBetragWaehrung":"",
                                                "BehandlungVonDatum":"1900-01-01",
                                                "BehandlungBisDatum":"1900-01-01",
                                                "KrankenhausNr":"",
                                                "DiagnoseTextListe":{
                                                    "DiagnoseText":""
                                                },
                                                "SperrKz":""
                                            },                                                
                                        ]
                                    }
                                },
    
                            ]
                            */
                        }
                    }
                }
            }
        }
    }    

    // Build process and document list 
    spResult.forEach(function(row){
        if(!row) return
        res._Context.nils.spRow = row

        if(currentProcessId !== row.FTVG_TSPROCESS_ID){
            // it is a new Process

            // Save the new current Process id
            currentProcessId = row.FTVG_TSPROCESS_ID

			// Build a new Process Object
            process = _buildProcess(res)
            processList.push(process)

            process.DokumentListe.Dokument = docArray = []
        }
        // Build document Object
        doc = _buildDoc(res)
        if(docArray) docArray.push(doc)
    })
    
    //set process list in JSON result
    jsonResult["soapenv:Envelope"]["soapenv:Body"]["ns1:getStapelResponse"]["GetStapelResult"]["VorgangsListe"].Vorgang = processList

    res._Context.nils.soapResult = jsonResult
    if(logger.isDebugEnabled()) logger.debug('--->_buildResult: Req.Id=' + res._Context.reqId + ', Result: ' + JSON.stringify(res._Context.nils.soapResult))
    callback()
}