/**-----------------------------------------------------------------------------
 * oxsnps.js: 	
 *
 * Author    :  AFP2web Team
 * Copyright :  (C) 2017 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  1.0.0
 * 
 * History
 *  V2.0.19		09.05.2024	isServersAlive(): Modified /isalive?skipRetry=true|false as /isalive?skipretry=true|false (url + query string must be lowercase as convention)
 *  V2.0.18		22.04.2024	OTS-3550: Extend /isAlive API to skip retrying
 *  V2.0.17		18.04.2024	OTS-3550: _assertServers(): Extend to use retry options from assert environment variable when assertion is called at server startup
 *  V2.0.16		16.04.2024	OTS-3550: oxsnps-core: Extend httpUtils.getURL()/httpUtils.postURL() APIs with endless retry options
 *  V2.0.15		12.04.2024	OTS-3548: Extend assert environment variable with "retryOptions" property
 *  V2.0.14		26.10.2023	isServersAlive: Extend to return the server name and version if there are no servers to assert (Applicabel for stateless servers )
 *  V2.0.13		19.10.2023	OTS-3503: Extend to return the server assertion error to oxsnps-core so that core can act based on "stopOnInitErrors"
 *  V2.0.12		28.07.2023	Extend _assertServers() to pass tenant always to the ping services and to get tenant name in the following order 
                               1. From env "HAL_ASSERTION_ASSERT_ARCHIVE_SERVER":  {"plugin":"oxsnps-archive","parms":{'tenant': "hallesche"}}
                               2. From "tenants" property in server-conf.js 
 *  V2.0.11		21.03.2023	OTS-3434: oxsnps-core: Extend to use same context for the core service APIs
 *  V2.0.10		29.07.2022	OTS-3329:  Extended _assertServers() to pass parameters defined in an assert environment variable (Ex: {"plugin":"oxsnps-afp2any", "parms":{'envvar': "HAL_ASSERTION_ONLINE_A2WSERVER"}}) to the plugin pingService()
 *  V2.0.9		19.07.2022	OXS-xxxx:  Extended isAlive() to return 500 as HTTP status code when there are dependency server errors
 *  V2.0.8		15.07.2022	OXS-13840: Extended isAlive() to add oxsnps server name with version along with text "All servers are running" when there are no errors. 
 *                                          Response: "All dependent servers are running in DMS3 Assertion Server@3.0.4"
 *  V2.0.7		13.07.2022	OXS-13840: Extended isAlive() to return simple text "All servers are running" when there are no errors 
 *  V2.0.6		06.07.2022	OXS-13815: Extended to return assertion results as a JSON having both success and failure cases
 *                                     Added an HTTP API to test if events are triggered for the given file path (symlink) 
 *  V2.0.5		18.05.2022	OTS-3271: Extend Ping Service of oxsnps base plugins to use "initialized" flag before sending requests to HA servers
 *  V2.0.4		05.05.2022	OTS-3284:  isAlive() HTTP API should report only failures and do not need to take any actions like disabling the routes.	//V2.0.4
 *  V2.0.2		05.05.2022	OTS-3284:  Added isAlive() HTTP API and beforeServerStart() service
 *  V2.0.1      18.06.2020  OXS-10750:  Extend to pass plugin name to oxsnps-statefilewatcher.getRouteStateInfo() function, since its API is changed
 *  V2.0.0      20.02.2020  OTS-2699:   Extend base and custom import/export plugins to use oxsnps-core@4.1.x
 *  V1.0.0      03.10.2019  Initial version
 *----------------------------------------------------------------------------*/
'use strict'

/**
 * Variable declaration
 * @private
 */
var packageJson		= require(__dirname + '/package.json')
  , PLUGIN_NAME		= packageJson.name
  , PLUGIN_VERSION	= packageJson.version // Get Server Version from package.json
  , async			= require('async')  
  , fs				= require('fs')
  , log4js			= require('log4js')
  , os				= require('os')  
  , path			= require('path')

  , npsServer		= require('oxsnps-core/server')
  , npsConf 		= npsServer.npsConf
  /*
  , npsName   		= npsServer.npsName
  , npsVersion   	= npsServer.npsVersion
  , npsLogFile 		= npsServer.npsLogFile
  , npsLogDir 		= npsConf.logDir
  , npsTempDir		= npsServer.npsTempDir
    */
  , pluginConf		= require(__dirname + '/conf/' + PLUGIN_NAME + '.js')

  , dateFormat 		= require('oxsnps-core/helpers/date_format')
  , utils 			= require('oxsnps-core/helpers/utils')
  , expressAppUtils = require('oxsnps-core/helpers/expressAppUtils')
  , httpUtils		= require('oxsnps-core/helpers/httpUtils')  
  , pluginManager 	= require('oxsnps-core/pluginManager')    
  , warn 			= require('oxsnps-core/helpers/warn')  
  , FILE_SEP 		= require('path').sep
  , serverList		= undefined
  , initError       = undefined	// V2.0.5
  , symlinkResponse    = {}        // Hash used to store infomration about triggered file events for each request	// v2.0.6

// Set plugin name, version and longDesc in pluginConf
pluginConf.module 	= PLUGIN_NAME 
pluginConf.version 	= PLUGIN_VERSION 
pluginConf.longDesc = packageJson.description
pluginConf.log 		= pluginConf.log || {}

// Export the Plugin's Configuration Express App
exports.pluginConf 	= pluginConf
exports.router = undefined     // V2.0.0

// V2.0.6
const STATUS_ERROR      = 'Error'
    , STATUS_RUNNING    = 'Running'
    , STATUS_ENABLED    = 'Enabled'
    , DW_MAX_WAIT_TIME  = 60000             // Given in milliseconds, 60000 => 1min
    , DW_EVENT_CHECK_INTERVAL   = 5000      // Given in milliseconds 
    , DUMMY_FILENAME    = "dummy.txt"

// Get the Logger
let logger = log4js.getLogger(PLUGIN_NAME)
utils.log4jsSetLogLevel(logger, (pluginConf.log.level || 'INFO'))

/**************** PUBLIC FUNCTIONS ****************/
/**
 * Initialize Plugin
 * @param  {function} 	callback  callback(err, routes)
 */
exports.initialize = function(callback){

	let self = this
	  , mode = parseInt('0777',8)
	  , appPluginConf	= (npsConf.plugins && npsConf.plugins[PLUGIN_NAME] ? npsConf.plugins[PLUGIN_NAME] : {})	
	  
	if(logger.isDebugEnabled()) logger.debug('--->initializing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION )

	if(self._initialized) return callback()

	// Merge application specific configuration with plugin configuration
    exports.pluginConf = pluginConf = utils.mergeJSON(pluginConf, appPluginConf)    // V2.0.0

	// Get assertion server list from env. variable list
	serverList = _getServerList()
    
    pluginConf.templates 				= pluginConf.templates || {}
	pluginConf.templates.isalive        = pluginConf.templates.isalive || path.join('conf', 'isalive.html')
	if(!path.isAbsolute(pluginConf.templates.isalive)) pluginConf.templates.isalive = path.join(__dirname, pluginConf.templates.isalive)
    pluginConf.templates.isalive  = path.resolve(pluginConf.templates.isalive) 	

    pluginConf.maxOccurences                = pluginConf.maxOccurences || {}
    pluginConf.maxOccurences.assertServers  = pluginConf.maxOccurences.assertServers || 10
    pluginConf.maxOccurences.assertServers = pluginConf.maxOccurences.assertServers * 1

    if(logger.isDebugEnabled()) logger.debug('--->initialize: pluginConf.maxOccurences.assertServers: ' + pluginConf.maxOccurences.assertServers + ', type: ' + typeof(pluginConf.maxOccurences.assertServers))
	// Setup router for the plugin
    expressAppUtils.setupRouter(self, function(err, router){   // V2.0.0
		if(err){
            // V2.0.5 Begin
            initError = err
            self._initialized = false 
            return callback(err, pluginConf.routes)
        }
        // V2.0.5 End
        exports.router = router     // V2.0.0
        self._initialized = true
		logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' initialized')
        //callback(null, pluginConf.routes)
        callback(null, {'routes': pluginConf.routes, 'swagger':{"name": PLUGIN_NAME, "url": '/' + PLUGIN_NAME + '/api/rest.yaml'}})
	})
}

/**
 * Activate Plugin. This function is called from pluginManager once all the enabled plugins have been initialize
 * @param  {function} 	callback  callback(err)
 */
exports.postInitialize = function(callback){
	callback()
}

/**
 * Finalize Plugin
 * @param  {function} 	callback callback(err) 
 */
exports.finalize = function(callback){

	let self = this
	  , req  = {}

	if(logger.isDebugEnabled()) logger.debug('--->finalizing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION )
	if(!self._initialized) return callback()

	async.series([
		// Disable all routes
		function(nextTask){
			req.body = {'module': PLUGIN_NAME}
			async.each(pluginConf.routes, 
				function(route, next){
					req.body.route = route
					exports.disableRoute(req, next)
				},
				function(err){
					return nextTask(err)
				}
			)
		}
	],
	function(err){
		if(err) return callback(err)
		self._initialized = false
		logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' finalized')
		callback()
	})

}

/**
 * Return Plugin Version
 * @param  {function} 	callback callback(err,version)
 */
exports.getVersion = function(callback){
	return callback(null, PLUGIN_VERSION)
}

/**
 * Process a HTTP Version Request
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.version = function(req, res){

	let data = undefined

	// Add Context to the response instance
	res._Context = {}

	// Add the Current Date to the Context
	res._Context.date = new Date()

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date)
	logger.info('--->version: Req. Id=' + res._Context.reqId)

	data = '<h3>' +  PLUGIN_NAME + ' v' + PLUGIN_VERSION + '</h3>'
	res.end(data)

	logger.info('--->version: Req. Id=' + res._Context.reqId + ', sent back ' +
				PLUGIN_NAME + ' v' + PLUGIN_VERSION + ', over')
	return
}

/**
 * Process a HTTP Configuration Request 
 * 	Either as HTTP GET request:
 * 	 	GET http://localhost:1029/services/<plugin>/conf 							--> To retrieve the configuration
 * 	 	GET http://localhost:1029/services/<plugin>/conf?save=true&conf="{...}" 	--> To pass and save the configuration
 * 	or as HTTP POST request:
 * 	 	POST http://localhost:1029/services/<plugin>/conf 						--> To pass and save the configuration
 * 	 	req.body={
 * 	 		"conf": {...}
 * 	 	}
 * 	
 * @param {Request} 	req Request Object
 *                      req.query:{ // for HTTP GET request:
 *                      	"save":"false",
 *                      	"conf":"{<IMPORTANT: COMPLETE JSON Conf. as String>}"
 *                      }
 *                      req.body:{ // for HTTP POST request:
 *                      	"conf":{<IMPORTANT: COMPLETE JSON Conf. as JSON Object>}
 *                      }
 * @param {Response}    res Response Object
 */
exports.conf = function(req, res){

	let reqType 	= 'getconf'
	  , jsonConf 	= undefined
	  
	// Add Context to the response instance
	res._Context = {}

	// Add the Current Date to the Context
	res._Context.date = new Date()

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date)
	logger.info('--->conf: Req. id=' + res._Context.reqId)
	
	if(req.method.toUpperCase() === 'POST'){
		logger.info('--->conf: body=' + JSON.stringify(req.body))
		if(!req.body || !req.body.conf) {
			logger.info('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body))
			return res.end('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body))
		}
		reqType = 'saveconf' 		// save request
		jsonConf = req.body.conf 	// set the json conf to be passed to saveConfService
	}
	else if(req.method.toUpperCase() === 'GET'){
		logger.info('--->conf: query=' + JSON.stringify(req.query))
		if(req.query && req.query.save && req.query.save.toLowerCase() === 'true' && req.query.conf)
		{
			reqType = 'saveconf'
			jsonConf = utils.parseJSON(unescape(req.query.conf)) // remove escaped chars and set the json conf to be passed to saveConfService
		} 
	}

	switch(reqType){
		case 'saveconf':
		    exports.saveConfService(jsonConf, function(err){
		    	logger.info('--->conf: Req. id=' + res._Context.reqId + ', over')
				res.json(err ? {'status':'error', 'message': err.message} : pluginConf)
		    }) 
			break
		default:
		case 'getconf':
			logger.info('--->conf: Req. id=' + res._Context.reqId + ', over')
			res.json(pluginConf)
			break
	}
}

/**
 * Store  the Plugin conf to its configuration file (i.e. .../<plugin>/conf/<plugin>.js)
 * @param  {JSON}   	conf     Plugin Configuration
 * @param  {Function} 	callback
 */
exports.saveConfService = function(jsonConf, callback){

	let filename 	= undefined
	  , data 		= undefined
	  
	// PluginManager calls saveConfService without passing the conf
	if(typeof jsonConf === 'function'){
		callback = jsonConf
		jsonConf = pluginConf
	}

	// Assert jsonConf
	if(!jsonConf || jsonConf === null) return callback(new Error('--->saveConfService: empty Configuration'))
	if(logger.isDebugEnabled()) logger.debug('--->saveConfService: conf:' + JSON.stringify(jsonConf))

	// Backup the current plugin conf.in the server's backup dir and store the new one in the plugin's /conf dir
	async.series([
		// Task 1: Backup the current plugin conf.
		function(nextTask){
			filename = npsConf.backupDir + FILE_SEP + dateFormat.asString('yyyy-MM-dd_hh-mm-ss', new Date()) + '_configurationOf_' + PLUGIN_NAME + '.js' // build the Backup filename
			if(logger.isDebugEnabled()) logger.debug('--->saveConfService: Backuping plugin configuration to ' + filename)
			data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t') // build the plugin conf content

			// Write data to the Plugin configuration file
			fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
				if(err) logger.error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message) // log error & continue
				nextTask()
			})
		},
		// Task 2: Save the new plugin conf.
		function(nextTask){
			pluginConf = jsonConf // we should have here a proper conf, so overwrite the current pluginConf file
			filename = path.resolve(__dirname + '/conf/' + PLUGIN_NAME + '.js') // build the plugin configuration filename
			if(logger.isDebugEnabled()) logger.debug('--->saveConfService: Writing plugin configuration to ' + filename)
			data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t') // build the plugin conf content

			// Write data to Plugin configuration file
			fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
				if(err) nextTask(new Error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message))
				else nextTask()
			}) 
		}
	],
	function(err){
		callback(err)
	})
}

// V2.0.2  Begin
/**
 * Process a isAlive HTTP Request
 * 	
 * @param {Request} 	req Request Object
 * 		   GET URL: 	<server>[:port]/services/assert/isalive[?tenant=condor]
 * @param {Response}    res Response Object
 */
exports.isServersAlive = function(req, res){

	let parms      = undefined
	  , serverList = undefined
	  , msg        = undefined

    parms = httpUtils.getRequestParms(req) || {}

	res._Context = {} 									// add Context to the response instance
    res._Context.reqId = utils.buildReqId(new Date())	// add a Request Id to the Context

    logger.info('--->isServersAlive: Req. id=' + res._Context.reqId + ', parms: ' + JSON.stringify(parms))

    res._Context.assert = {
        'conditions':   npsConf.startConditions ? utils.clone(npsConf.startConditions) : {},
        //'tenant':       parms['tenant'] || npsConf.tenant
        'module':       npsConf.importPlugins ? npsConf.importPlugins[0]: undefined
    }
    if(parms.skipretry === undefined) parms.skipretry = true   // default is to skip retry
    if(parms.skipretry && typeof(parms.skipretry) === 'string') parms.skipretry = utils.stringToBoolean(parms.skipretry)  // v2.0.19
    res._Context.assert.skipRetry = parms.skipretry

	// V2.0.14 Begin
	if(res._Context.assert.conditions.servers) serverList = Object.keys(res._Context.assert.conditions.servers)
		
	if(serverList === undefined || serverList.length === 0){
		// there are no dependency servers to assert so return only this server version
		msg = npsServer.npsDesc + '@' + npsServer.npsVersion + ' running'
		logger.info('--->isServersAlive: Req. id=' + res._Context.reqId + ', ' + msg)
		res.setHeader('Content-Type', 'text/plain')
		return res.send(msg)	//  sample output: DMS3 Assertion Server@3.0.4 running
	}
    // V2.0.14 End

    res._Context.assert.conditions.actions = undefined  // isAlive should report only failures and do not need to take any actions.	//V2.0.4

    exports.isServersAliveService(res, function(err, html){
        if(err){
            if(html){
                logger.error('--->isServersAlive: Req. id=' + res._Context.reqId + ', Error occurred for some servers, Response: ' + JSON.stringify(res._Context.assert.results))
                res.setHeader('Content-Type', 'text/html')
                return res.status(500).send(html)
            }
            logger.info('--->isServersAlive: Req. id=' + res._Context.reqId + ', Server Response:', res._Context.assert.results)
            logger.error('--->isServersAlive: Req. id=' + res._Context.reqId + ', Reason:' + err.message)
            return res.status(500).send(err.message)
        }
        else{
            logger.info('--->isServersAlive: Req. id=' + res._Context.reqId + ', All servers are running, Response: ' + JSON.stringify(res._Context.assert.results))
            //res.setHeader('Content-Type', 'text/html')
            //return res.send(html)
            res.setHeader('Content-Type', 'text/plain')
            //res.send('All servers are running')	// V2.0.7
            //res.send('All dependent servers are running in ' + npsServer.npsName + '@' + npsServer.npsVersion)	// V2.0.8  sample output: All dependent servers are running in hal-assertion-server@3.0.4
            res.send('All dependent servers are running in ' + npsServer.npsDesc + '@' + npsServer.npsVersion)	// V2.0.8  sample output: All dependent servers are running in DMS3 Assertion Server@3.0.4 (as in HTML output)
        }
    })
}

/**
 * Assert Servers before starting an application server
 * 	
 * @param {Response}    res Response Object
 * @param  {Function}	callback 	callback(err)
 */
exports.isServersAliveService = function(res, callback){
    
    let htmlResult  = undefined
      , assertErr   = undefined
      , self        = this
      , serverinfo  = undefined

    serverinfo = {
        'name':     npsServer.npsDesc,
        'version':   npsServer.npsVersion,
        'props':{
            'hostname': os.hostname() + ', ' + os.platform() + ' ' + os.arch() + ' ' + os.release(),
            'ip':       utils.networkInterfaces() ||'',
            'date':     dateFormat.getAsString(new Date(), "DD.MM.YYYY HH:mm")   
        }
    }

    async.series([
        // Initialize plugin if not initialized yet
        function(nextTask){
            if(!self._initialized) return exports.initialize(nextTask)
            return nextTask()
        },
        // Create a directory
        function(nextTask){
            exports.assertJobConditions(res, function(err){
                if(err) assertErr = err    // errors are already logged
                return nextTask()
            })
        },      
        // Build html output
        function(nextTask){
            // Read the HTML file, set some nps... vars, and send back the modified HTML
            fs.readFile(pluginConf.templates.isalive, 'utf8', function(err, html){
                if(err) return nextTask(new Error('HTML Template file: ' + pluginConf.templates.isalive + ' not found!'))
                htmlResult = html.replace(/&data;/, JSON.stringify(res._Context.assert.results))	// replace &data with actual data
                htmlResult = htmlResult.replace(/&serverinfo;/, JSON.stringify(serverinfo))	// replace &data with actual data
                nextTask()
            })
        }
	],
	function(err){
        callback(err || assertErr, htmlResult)
	})
}

/**
 * Assert Servers before starting an application server
 * 	
 * @param {Response}    res Response Object
 * @param  {Function}	callback 	callback(err)
 */
exports.beforeServerStart = function(res, callback){
    
    // fill res._Context from server-conf.startConditions
    res._Context.assert = {
        'conditions':   utils.clone(npsConf.startConditions),
        //'tenant':       npsConf.tenant,
        'module':       npsConf.importPlugins ? npsConf.importPlugins[0]: undefined,
        'useAssertRetry': true   // 2.0.17       // At server startup, use retry options given in the assert env. variable that does not have retrycount that in turn retries till it succeeds (Ex: "OXS_MAIL_ASSERT_QPDF_SERVER": {"plugin":"oxsnps-qpdfclient", "retryOptions":{"interval":30000}} )
    }

    exports.assertJobConditions(res, function(err, errorList){
		return callback(err)
		/*
        if(err){
            if(npsConf.startConditions && npsConf.startConditions.actions && npsConf.startConditions.actions.includes('stopServer')) return callback(err)
            logger.error('--->beforeServerStart: Req.Id=' + res._Context.reqId + ', Error while running server start assertion tests. ' + err.message)
            return callback()
        }
        return callback()
		*/
    })

}
// V2.0.2  End
/*
 * Assert Job Conditions
 * @param  {Response}	res 		Response Object
 *                      res._Context={
 *                      	reqId: <reqId>,
 *                     		'assert':{
 *       						// Job conditions as specified in channel props
 *       						"conditions": {
 *                     				'statefile': <true|false>,  Optional.Default is true.
 *                     				'servers': {
 *                     					'<server_name>': <true|false>,
 *                     			    	'<server_name>': <true|false>
 *                     				},
 *                     				'actions':[
 *                     					'logWarning',
 *                     					'disableRoute
 *                     				]
 *                     			}, 
 *                     			'tenant': <tenant>,
 *                     			'module': <PluginName>,
 *                     			'route': Optional. [<route>]	
 *                     			'route': {
 *									'tenant' : <'tenant'> Optional
 *									'path' : <'route path'>, 
 *									'subscription' : <'subscription'>  
 *                     			}
 *                     		 }
 *                      }
 * @param  {Function}	callback 	callback(err)
 */
exports.assertJobConditions = function(res, callback){

    let self            = this
      , assertResult    = undefined
      , errmsg          = undefined

    if(!self._initialized) return callback(new Error(PLUGIN_NAME + ' plugin is not initialized. ' + (initError ? initError.message:'')))   // 2.0.5

    if(!res._Context.assert || !res._Context.assert.conditions) return callback()

    //res._Context.assert.errors = []
    res._Context.assert.results = []    // V2.0.6
    
	async.series([
		// Check if state file is enabled.
		function(nextTask){
            // state file checking is off
            if(res._Context.assert.conditions.statefile === undefined || res._Context.assert.conditions.statefile === false) return nextTask() 	// state file checking is not enabled
            //if(res._Context.assert.conditions.statefile !== undefined && res._Context.assert.conditions.statefile === false) return nextTask() 	// state file checking is not enabled

            if(!res._Context.assert.route || !res._Context.assert.module) return nextTask()
            res._Context.sfw = {
                'route' :   res._Context.assert.route,
                'plugin':   res._Context.assert.module  // V2.0.1
            }
            assertResult = {'type': 'statefile', 'status': STATUS_ENABLED}        // V2.0.6
            
	    	//pluginManager.getRouteStateInfo(res._Context.assert.route, function(err, result){
			pluginManager.callPluginService({name:'oxsnps-statefilewatcher',service:'getRouteStateInfo'}, res, function(err, result){
	    		if(err){
                    assertResult.status = STATUS_ERROR
                    assertResult.message = 'Unable to get Route state information. ' + err.message + ', Route: ' + JSON.stringify(res._Context.assert.route)
	    			//res._Context.assert.errors.push('statefile: Unable to get Route state information. ' + err.message + ', Route: '+ JSON.stringify(res._Context.assert.route))	// V2.0.2  
                    logger.error('--->assertJobConditions: Req.Id=' + res._Context.reqId + ', Statefile: ' + assertResult.message)
	    		} 
	    		else if(!result){
                    assertResult.status = STATUS_ERROR
                    assertResult.message = 'Route state file not available. Hostname: ' + os.hostname() + 
                                            (res._Context.assert.route.tenant ? (', tenant: '+ res._Context.assert.route.tenant) : '') +  
                                            ', plugin: ' + res._Context.assert.module + ', Route: ' + JSON.stringify(res._Context.assert.route)
                    logger.error('--->assertJobConditions: Req.Id=' + res._Context.reqId + ', Statefile: ' + assertResult.message)
	    			//res._Context.assert.errors.push('statefile: Route state file not available. Hostname: ' + os.hostname() + 
					//							        (res._Context.assert.route.tenant ? (', tenant: '+ res._Context.assert.route.tenant) : '') +  
                    //                                    ', plugin: ' + res._Context.assert.module + ', Route: '+ JSON.stringify(res._Context.assert.route)) // V2.0.2  
	    		}	    		
	    		else if(result && result.state && result.state.toLowerCase() !== pluginManager.ENABLE){
                    assertResult.status = STATUS_ERROR
                    assertResult.message = 'State of '+ res._Context.assert.route.path + ' route is not "enable". Route state file: ' + result.filename + 
                                    (res._Context.assert.route.tenant ? (', tenant: '+ res._Context.assert.route.tenant) : '') +  ', State: '+ result.state
                    logger.error('--->assertJobConditions: Req.Id=' + res._Context.reqId + ', Statefile: ' + assertResult.message)
	    			//res._Context.assert.errors.push('statefile: State of '+ res._Context.assert.route.path + 
                    //                                    ' route is not "enable". Route state file: ' + result.filename + (res._Context.assert.route.tenant ? (', tenant: '+ res._Context.assert.route.tenant) : '') +  ', State: '+ result.state)   // V2.0.2  
                    // Assume there are Kue jobs 
                    // For first job, HA server assertion failed, state file rename as disablebyerror and error message written to state file
                    // When we run the second kue job, state file is already disabled. And if we write the message 'state file is disabled' in the state file again, the actual original error message will be lost.
                    // So do not overwrite the actual message is state file is already disablebyerror
                    if(result.state.toLowerCase() === pluginManager.DISABLEBYERROR) res._Context.assert.repeatError = true  

    		    }
                else res._Context.assert.route.stateInfo = result
                
                res._Context.assert.results.push(assertResult)
	    		nextTask()
	    	})
		},
		// Check if all servers are alive
		function(nextTask){
			_assertServers(res, nextTask)       // fills errors in res._Context.assert.errors
		},
		// If any of servers not alive or state file not enabled, do specified actions like disabling the route
		function(nextTask){
            res._Context.assert.errormsg = _getErrorMsg(res)
            if(res._Context.assert.errormsg && res._Context.assert.errormsg.length > 0) return _doFailureActions(res, nextTask)
            return nextTask()
		}
	],
	function(err){
        if(res._Context.assert.errormsg && res._Context.assert.errormsg.length > 0){
            // for older version compatibility for OXS.eco, combine all errors and return an error object
            //res._Context.assert.error = new Error(res._Context.assert.errors.join('\n'))   // V2.0.2  
            //return callback(res._Context.assert.error, res._Context.assert.errors)	// V2.0.2  
            return callback(new Error(res._Context.assert.errormsg))	// V2.0.2  
        }
        callback()
	})
}

/**
 * Process a HTTP Request to test if events are triggered for the given file path (symlink) 
 * 	
 * GET URL: 	<server>[:port]/services/assert/symlink?path=<path>[&options=<JSON String>][&maxwaittime=<maxwaittime>]
 * @param  {Request}  	req Request Object
 * 	                    req:{
 * 	                        query:{ 		
 * 							    'path':	        <path to test>
 * 							    'options':	    [OPTIONAL] <Chokidar options>. Refer to options passed to watch() function at https://www.npmjs.com/package/chokidar. If not given default will be used.
 * 							    'maxwaittime':[OPTIONAL] <Time to wait for event in milliseconds>. Default is 60000 (1min)
 * 	                        }
 *                      }
 * @param  {Response} 	res Response Object
 */
exports.assertSymlink = function(req, res){

    let parms   = undefined
      , errmsg  = undefined
      , tmp     = undefined
      , resp    = undefined

    res._Context = {}
    res._Context.reqId = utils.buildReqId(new Date())
  
    logger.info('--->assertSymlink: Req. id=' + res._Context.reqId + ', Query Parms: ' + JSON.stringify(req.query) + ', Path Parms: ' + JSON.stringify(req.params))

    parms = httpUtils.getRequestParms(req)

    if(parms.options){
        tmp = utils.parseJSON(parms.options) 
        if(!tmp){
            errmsg = 'options parameter value is not a valid JSON. options: ' + parms.options
            logger.error('--->assertSymlink: Req. id=' + res._Context.reqId + ', Reason: ' + errmsg + ', Parms: ' + JSON.stringify(parms))
            return res.status(400).end(errmsg)
        }	        
        parms.options = tmp
    }

    if(parms.events){
        tmp = utils.parseJSON(parms.events) 
        if(!tmp){
            errmsg = 'events parameter value is not a valid JSON. events: ' + parms.events
            logger.error('--->assertSymlink: Req. id=' + res._Context.reqId + ', Reason: ' + errmsg + ', Parms: ' + JSON.stringify(parms))
            res.setHeader('Content-Type', 'application/text') 
            return res.status(400).end(errmsg)
        }	        
        parms.events = tmp
    }    

    res._Context.assert = {'parms': parms}
    
    exports.assertSymlinkService(res, function(err, result){
        if(err){
            logger.error('--->assertSymlink: Req. id=' + res._Context.reqId + ', Reason: ' + err.message + ', Parms: ' + JSON.stringify(parms))
            return res.status(404).send(err.message)
        }
        else{
            resp = symlinkResponse[res._Context.assert.parms.path] || {}
            symlinkResponse[res._Context.assert.parms.path] = []
            logger.info('--->assertSymlink: Req. id=' + res._Context.reqId + ', Response:', resp)
            //res.setHeader('Content-Type', 'application/json') 
            res.json(resp)
        }
    })
}

/*
 * Service to test if events are triggered for the given file path (symlink) 
 * @param  {Response}	res 		Response Object
 *                      res._Context={
 *                      	reqId: <reqId>,
 *                     		'assert':{
 *       						// Job conditions as specified in channel props
 *       						"parms": {
 *                     				'path': <path>
 *                     				'options': {      // Optional. Refer to options passed to watch() function at https://www.npmjs.com/package/chokidar
                                        "depth": 			0,
                                        "binaryInterval": 	10000,
                                        "followSymlinks": 	false,
                                        "ignoreInitial": 	false,
                                        "ignorePermissionErrors": false,
                                        "interval": 		10000,
                                        "persistent": 		true,
                                        "usePolling": 		true
 *                     					'<server_name>': <true|false>,
 *                     			    	'<server_name>': <true|false>
 *                     				},
 *                                  'maxwaittime': <Optional, time to wait for an event in milliseconds>
 *                     			}, 
 *                      }
 * @param  {Function}	callback 	callback(err)
 */
exports.assertSymlinkService = function(res, callback){

    let self        = this
      , route       = undefined
      , chokidarOptions = undefined
      , mode 			= parseInt('0777',8)

    if(logger.isDebugEnabled()) logger.debug('--->assertSymlinkService: Req. id=' + res._Context.reqId + ', res._Context.assert.parms.path: ' + res._Context.assert.parms.path 
        + (res._Context.assert.parms.maxwaittime ? (', maxwaittime: ' + res._Context.assert.parms.maxwaittime) : '')    
        + (res._Context.assert.parms.interval ? (', interval: ' + res._Context.assert.parms.interval) : '')    
        + (res._Context.assert.parms.events ? (', events: ' + JSON.stringify(res._Context.assert.parms.events)) : '')
        + (res._Context.assert.parms.options ? (', options: ' + JSON.stringify(res._Context.assert.parms.options)) : ''))

    res._Context.assert.interval = res._Context.assert.parms.interval         
    if(!res._Context.assert.interval) res._Context.assert.interval = DW_EVENT_CHECK_INTERVAL

    res._Context.assert.maxwaittime = res._Context.assert.parms.maxwaittime
    if(!res._Context.assert.maxwaittime) res._Context.assert.maxwaittime = DW_MAX_WAIT_TIME    

    res._Context.assert.parms.events = res._Context.assert.parms.events || ["add", "addDir", "change", "unlink"]

    chokidarOptions = res._Context.assert.parms.options
    if(!chokidarOptions) 
        chokidarOptions = {         // if anything is changed here, default value given for parameter "options" described in rest.yaml should also be changed
            "depth": 			0,
            "binaryInterval": 	10000,  // in milliseconds
            //"followSymlinks": 	true,  // default: true
            "ignoreInitial": 	false,
            "ignorePermissionErrors": false,
            "interval": 		10000,      // in milliseconds
            //"persistent": 		true,       //  default: true
            "usePolling": 		true            // default: false
        }
    // Build DW Route
    route = [{
        "path": 	res._Context.assert.parms.path, 
        "method":   "dir",
        "events": 	res._Context.assert.parms.events,        // add, addDir, change, unlink, unlinkDir
        "allowed": "(\/.+)",            // matches any file
        "ignorecase": true,                     // if true, ignore case when checking allowed dirs.; default is false
        "options": chokidarOptions,
        "action": 	{
            "props":    {
                "module":  "oxsnps-assert",
                "service": "dwSymlink"
            },
            'parms':{
                'reqId': res._Context.reqId,
            }
        }
    }]

    res._Context.assert.filename = path.join(res._Context.assert.parms.path,  DUMMY_FILENAME)

	async.series([
        // Initialize plugin if not initialized yet
        function(nextTask){
            if(!self._initialized) return exports.initialize(nextTask)
            return nextTask()
        },
        // Create a directory
        function(nextTask){
            if(logger.isDebugEnabled()) logger.debug('--->assertSymlinkService: Req. id=' + res._Context.reqId + ', Creating a directory: '+ res._Context.assert.parms.path)
            utils.mkDir(res._Context.assert.parms.path, mode, function(err){
                if(err) return nextTask(new Error('Unable to create a directory: ' +  res._Context.assert.parms.path + ', ' + err.message))
                return nextTask()
            })
        },        
        // Add dw listener for the given path
        function(nextTask){
            if(logger.isDebugEnabled()) logger.debug('--->assertSymlinkService: Req. id=' + res._Context.reqId + ', Adding DW listener, Route: '+ JSON.stringify(route))
            pluginManager.callPluginService({name:'oxsnps-dirwatcher',service:'addListenersService'}, route, function(err){
                if(err) return nextTask(new Error('Unable to add directory watcher listener. ' + err.message + ', Route: ' + JSON.stringify(route)))
                return nextTask()
            })
        },
        // Create a dummy empty file 
        function(nextTask){
            if(logger.isDebugEnabled()) logger.debug('--->assertSymlinkService: Req. id=' + res._Context.reqId + ', Creating a file:' + res._Context.assert.filename)
            res._Context.assert.userEvents = symlinkResponse[res._Context.assert.parms.path] || []
            symlinkResponse[res._Context.assert.parms.path] = []
			fs.writeFile(res._Context.assert.filename, JSON.stringify(new Date()), function(err){
                if(err) return nextTask(new Error('Unable to create a file: ' + res._Context.assert.filename + ', ' + err.message))
                return nextTask()
                // wait for a given time, for an event occurrence
                //setTimeout(function(){return nextTask()},maxwaittime)
			})
        },
        // wait for a given time, for an event occurrence
        function(nextTask){        
            res._Context.assert.retryCount = res._Context.assert.maxwaittime / res._Context.assert.interval
            res._Context.assert.attempts = 0
            _waitForEvent(res, nextTask)
        }
	],
	function(err){
        if(err) return callback(err)
        callback(err)
	})
}

/**
 * File/directory  events handler
 */
exports.dwSymlink = function(event, dirOrFile, route, callback){
    
    let parms = undefined

    parms = route.action.parms || {}
    
	// Event is also triggered for the root path, so ignore it
	if(dirOrFile === route.path){
		if(logger.isDebugEnabled()) logger.debug('--->dwSymlink: Called for root path: ' + dirOrFile + ', ignoring it')
		return callback()
    }
    
    logger.info('--->dwSymlink: Event: ' + event + ', Path: ' + dirOrFile + ', RootPath: ' + route.path + ', Parms: ' + JSON.stringify(parms))
    
    symlinkResponse[route.path] = symlinkResponse[route.path] || []
    symlinkResponse[route.path].push({
        'event':    event,
        'path':     dirOrFile,
        //'rootPath': route.path,
        //'reqId':    parms.reqId
    })
    return callback()
}

/**
 * Process a HTTP Request to remove the directory watcher listener created for the passed path
 * 	
 * DELET URL: 	<server>[:port]/services/assert/symlink?path=<path>
 * @param  {Request}  	req Request Object
 * 	                    req:{
 * 	                        query:{ 		
 * 							    'path':	        <path to test>
 * 	                        }
 *                      }
 * @param  {Response} 	res Response Object
 */
exports.removeSymlink = function(req, res){

    let parms   = undefined

    res._Context = {}
    res._Context.reqId = utils.buildReqId(new Date())
  
    logger.info('--->removeSymlink: Req. id=' + res._Context.reqId + ', Query Parms: ' + JSON.stringify(req.query) + ', Path Parms: ' + JSON.stringify(req.params))

    parms = httpUtils.getRequestParms(req)

    res._Context.assert = {'parms': parms}
    
    exports.removeSymlinkService(res, function(err, result){
        if(err){
            logger.error('--->removeSymlink: Req. id=' + res._Context.reqId + ', Reason: ' + err.message + ', Parms: ' + JSON.stringify(parms))
            return res.status(500).end(err.message)
        }
        else{
            logger.info('--->removeSymlink: Req. id=' + res._Context.reqId + ', over')
            //res.setHeader('Content-Type', 'application/json') 
            res.send("Directory listener removed")
        }
    })
}

/*
 * Remove the directory watcher listener created for the passed path
 * @param  {Response}	res 		Response Object
 *                      res._Context={
 *                      	reqId: <reqId>,
 *                     		'assert':{
 *       						// Job conditions as specified in channel props
 *       						"parms": {
 *                     				'path': <path>
 *                     			}, 
 *                      }
 * @param  {Function}	callback 	callback(err)
 */
exports.removeSymlinkService = function(res, callback){

    let self        = this
      , route       = undefined

    if(logger.isDebugEnabled()) logger.debug('--->removeSymlinkService: Req. id=' + res._Context.reqId + ', res._Context.assert.parms.path: ' + res._Context.assert.parms.path)

    // Build DW Route
    route = [{
        "path": 	res._Context.assert.parms.path, 
    }]

	async.series([
        // Initialize plugin if not initialized yet
        function(nextTask){
            if(!self._initialized) return exports.initialize(nextTask)
            return nextTask()
        },
        // Remove dw listener for the given path
        function(nextTask){
            if(logger.isDebugEnabled()) logger.debug('--->removeSymlinkService: Req. id=' + res._Context.reqId + ', Removing DW listener, Route: '+ JSON.stringify(route))
            pluginManager.callPluginService({name:'oxsnps-dirwatcher',service:'removeListenersService'}, route, function(err){
                if(err) return nextTask(new Error('Unable to remove directory listener. ' + err.message + ', Route: ' + JSON.stringify(route)))
                delete symlinkResponse[res._Context.assert.parms.path]
                return nextTask()
            })
        }
	],
	function(err){
        callback(err)
	})
}

/************ OPTIONAL PUBLIC FUNCTIONS ***********/
/**
 * Set Log Level
 * @param {String} logLevel Log Level. values:DEBUG|INFO|WARN|ERROR 
 */
exports.setLogLevel = function(logLevel){
	pluginConf.log.level = logLevel || 'INFO'
	utils.log4jsSetLogLevel(logger, pluginConf.log.level)
}

/**
 * Get Log Level
 * @return {String} Log Level. values:DEBUG|INFO|WARN|ERROR
 */
exports.getLogLevel = function(){
	return pluginConf.log.level
}

/**
 * Enable the given route
 * @param  {Request} 	req
 *					  req.body = {
 *					  	"module":"sample", 
 *					  	"route": { 
 *					  		"path": "/path/to/the/service",
 *					  		"method": 	Optional. Values are "get|post|dw|queue",
 *					  		"service": 	"version"
 *					  	}
 *					  }
 * @param  {function} 	callback callback(err)
 */
exports.enableRoute = function(req, callback){

	let self = this

	if(logger.isDebugEnabled()) logger.debug('--->enableRoute: Route=' + JSON.stringify(req.body.route))

	if(!req.body.route) return callback(new Error('route parameter not given'))

	req.body.route.method = req.body.route.method || 'get'

	// Enable the route
	switch(req.body.route.method.toLowerCase()){
		case 'dir': // Enable Dir Route
			pluginManager.callPluginService({name:'oxsnps-dirwatcher',service:'addListenersService'}, [req.body.route], callback)
			break
		case 'queue': // Enable Queue Route
	   		// Delete Queued Jobs before enabling route 
			pluginManager.callPluginService({name:'oxsnps-ibmmq', service:'addListenersService'}, [req.body.route], callback)
			break
		default:
			// Enable HTTP Route
			expressAppUtils.enableRoute(self, req.body.route, callback)
	}
} 

/**
 * Disable the given route
 * @param  {Request} 	request Request json has following structure.
 * 						req.body = {
 *					  	"module":"sample", 
 *					  	"route": { 
 *					  		"path": "/path/to/the/service",
 *					  		"method": 	Optional. Values are "get|post|dw|queue"
 * 						  	}
 * 						}
 * @param  {function} 	callback  callback(err)
 */
exports.disableRoute = function(req, callback){

    let self = this

	if(logger.isDebugEnabled()) logger.debug('--->disableRoute: Route=' + JSON.stringify(req.body.route))
	
	if(!req.body.route) return callback(new Error('route parameter not given'))

	req.body.route.method = req.body.route.method || 'get'

	// Disable the route
	switch(req.body.route.method.toLowerCase()){
		// Disable Dir Route
		case 'dir': 
			pluginManager.callPluginService({name:'oxsnps-dirwatcher',service:'removeListenersService'}, [req.body.route], callback)
			break
		case 'queue':
			// Disable Queue Route
			pluginManager.callPluginService({name:'oxsnps-ibmmq', service:'removeListenersService'}, [req.body.route], callback)
			break
		default:
			//logger.warn('--->disableRoute: Wrong parameters(unknown route.method)! Not able to disable the route, Parms: ' + JSON.stringify(req.body))
			//callback(new Error('Wrong parameters(unknown route.method)'))
			// Disable HTTP Route
			expressAppUtils.disableRoute(self, req.body.route, callback) // V2.0.0
	}
} 

/**************** PRIVATE FUNCTIONS ***************/

/*
 * Check if state file is enabled for the route
 * @param  {Response}	res 		Response Object
 *                      res._Context={
 *                      	reqId: <reqId>,
 *                     		'job':{
 *        						"enable": "on",     // on: Check assertions, off: Skip all assertions
 *       						// List server assertions to be done before starting a server
 *       						"conditions": {
 *                     				'statefile': <true|false>,
 *                     				'servers': {
 *                     					'<server_name>': <true|false>,
 *                     			    	'<server_name>': <true|false>
 *                     				},
 *                     				'actions':[
 *                     					'logWarning',
 *                     					'disableRoute
 *                     				]
 *                     			}, 
 *                     			'tenant': <tenant>,
 *                     			'module': <PluginName>,
 *                     			'route': Optional. [<route>]
 *                     		 }
 *                      }
 * @param  {Function}	callback 	callback(err)
 */
function _assertServers(res, callback){
	
	let tenant = res._Context.assert.tenant || ((npsConf.tenants && Array.isArray(npsConf.tenants)) ? npsConf.tenants[0] : undefined)	// V2.0.12
	
	if(!res._Context.assert.conditions.servers) return callback()

    //async.each(Object.keys(res._Context.assert.conditions.servers), 
    //async.eachSeries(Object.keys(res._Context.assert.conditions.servers),     
    async.eachOfLimit(Object.keys(res._Context.assert.conditions.servers), pluginConf.maxOccurences.assertServers,    
		function(server, index, next){
            let envname		= undefined
            , serverInfo	= undefined
            , assertResult  = undefined
            , retryOptions  = undefined
      
			if(!server ) return next()
			if(!res._Context.assert.conditions.servers[server]) return next()
			envname = process.env['OXSNPS_SERVER_PREFIX'] + 'ASSERT_' + server
            if(!serverList || !serverList[server] || !serverList[server].plugin) {
                assertResult = {'server': server, 'status': STATUS_ERROR, 'message': envname + ' env. variable not found. Or plugin property is not given in ' + envname + ' env. variable'}        // V2.0.6
                res._Context.assert.results.push(assertResult)
                logger.error('--->_assertServers: Req.Id=' + res._Context.reqId + ', Server: ' + server + ', ' + assertResult.message)
                //res._Context.assert.errors.push(envname + ' env. variable not found. Or plugin property is not given in ' + envname + ' env. variable')
                return next()
            }
            serverInfo = serverList[server]
			let resTmp = {
			   '_Context' :{
                    'reqId': 		res._Context.reqId,
					'tenant':		(serverInfo.parms && serverInfo.parms.tenant) || tenant,	// V2.0.12	// res._Context.assert.tenant 
					'orderParms':	{'tenant': (serverInfo.parms && serverInfo.parms.tenant) || tenant},	// V2.0.12		// {'tenant': res._Context.assert.tenant},
					'plugin':		serverInfo.plugin,
                    'service':		serverInfo.service 		|| 'pingService',
                    'parms':        serverInfo.parms,   // V2.0.10
					//'retryCount': 	(serverInfo.retryCount === undefined) ? 5 : serverInfo.retryCount,		// number of time to check the server
					'interval': 	serverInfo.interval		|| 1000,			// retry interval in milliseconds on how often to check the server
					'job': 			utils.clone(res._Context.assert),
					'serverName': 	server,
                    //'count': 		0,
                    //'retryOptions': {'default': retryOptions} // V2.1.5 // 2.0.17
				}
			}
            // 2.0.17
            if(res._Context.assert.useAssertRetry){
                // V2.1.5
                retryOptions            = (serverInfo && serverInfo.retryOptions) ? utils.clone(serverInfo.retryOptions)  : {}  // V2.0.16
                //retryOptions.count      = retryOptions.count        || 5  // it is optional   // V2.0.16
                retryOptions.errorCodes = retryOptions.errorCodes   || ["all"]    // "all" => retry for all error
                retryOptions.interval   = retryOptions.interval     || 30000 // 30 secons
                // V2.1.5
                if(retryOptions.count === 0) retryOptions.count = undefined // V2.0.16
			
                resTmp._Context.retryOptions = {'default': retryOptions} // V2.1.5
            }
            // 2.0.17
            // 2.0.18
            else if(res._Context.assert.skipRetry){
                resTmp._Context.retryOptions = {'default': {'skip':true}} 
            }
            // V2.0.18
			_assertServer(resTmp, function(err){
                assertResult = {'server': server, 'status': STATUS_RUNNING}        // V2.0.6
                if(err){
                    assertResult.status     = STATUS_ERROR
                    assertResult.message    = err.message
                    logger.error('--->_assertServers: Req.Id=' + res._Context.reqId + ', Server: ' + server + ', ' + err.message)
					// V2.0.2  Begin
                    //res._Context.assert.errors.push(err.message)
                    if(res._Context.assert.route && res._Context.assert.route.stateInfo) res._Context.assert.route.stateInfo.state = pluginManager.DISABLEBYERROR
                }
                else logger.info('--->_assertServers: Req.Id=' + res._Context.reqId + ', Server: ' + server + ', Running')
                res._Context.assert.results.push(assertResult)
				// V2.0.2  End
                next()
            })
		},
		function(){
		    callback()
		}
	)
}

/**
 * Check if all servers are alive
 * @param  {Response}	res 		Response Object
 * @param  {Function}	callback 	callback(err, result)
 */
function _assertServerOld(res, callback){

	res._Context.count++

	logger.debug('--->_assertServer: Req.Id=' + res._Context.reqId + 
					', Calling ' + res._Context.plugin + '.' + res._Context.service + ' to assert server: ' + res._Context.serverName)
	
	pluginManager.callPluginService({name:res._Context.plugin, service:res._Context.service}, res, function(err){
		if(!err) return callback()

		if(res._Context.count >= res._Context.retryCount){
            return callback(new Error(err.message + ', Number of times tried: ' + res._Context.count + ', retryCount: ' + res._Context.retryCount +
                    (res._Context.interval ? (', Interval: ' + res._Context.interval + ' milliseconds') : '')))	// V2.0.2  
		}
		else{
			logger.warn('--->_assertServer: Req.Id=' + res._Context.reqId + 
						', Server: ' + res._Context.serverName + ', ' + err.message + ', Number of times tried: ' + res._Context.count + ', Trying again after ' + res._Context.interval + ' milliseconds')
			setTimeout(function(){ _assertServer(res, callback)}, res._Context.interval)
		}
	})
}
// V2.1.5
/**
 * Check if all servers are alive
 * @param  {Response}	res 		Response Object
 * @param  {Function}	callback 	callback(err, result)
 */
function _assertServer(res, callback){

    logger.debug('--->_assertServer: Req.Id=' + res._Context.reqId + ', Assert server: ' + res._Context.serverName + ', Parms: ' + JSON.stringify(res._Context))
    pluginManager.callPluginService({name:res._Context.plugin, service:res._Context.service}, res, function(err){
        callback(err)
    })
}
// V2.1.5
/**
 * Do specified actions in case of server failures
 * @param  {Response}	res 		Response Object
 * @param  {Function}	callback 	callback(err, result)
 */
function _doFailureActions(res, callback){

    let actions = res._Context.assert.conditions.actions
      //, errmsg  = res._Context.assert.errors.length >0 ? res._Context.assert.errors.join('\n') : undefined
	
	if(!actions || actions.length <=0){
        //logger.error('--->_doFailureActions: Req.Id=' + res._Context.reqId + ', ' + errmsg)
        return callback()
    }

	// Log server ping error as warning if action contains "logWarning" else log it as error
	if(actions.includes('logWarning')) logger.warn('--->_doFailureActions: Req.Id=' + res._Context.reqId + ', ' + res._Context.assert.errormsg)
	else logger.error('--->_doFailureActions: Req.Id=' + res._Context.reqId + ', ' + res._Context.assert.errormsg)

	async.each(actions, 
		function(action, next){
			switch(action.toLowerCase()){
				//case 'stopserver':	// not supported
				case 'logwarning':
					next() 
					break
                case 'disableroute':
                    _disableRoutes(res, next)
                    break
                case 'disableallroutes':                
                    _disablePluginRoutes(res, next)
                    break
				default:	
					logger.warn('--->_doFailureActions: Req.Id=' + res._Context.reqId + ', Invalid action(' + action + ')')
					next() 
					break
			}
		},
		function(err){
			if(err) logger.error('--->_doFailureActions: Req.Id=' + res._Context.reqId + ', ' + err.message)
		    callback(err)
		}
	)
}

/**
 * Check if all servers are alive
 * @param  {Response}	res 		Response Object
 * @param  {Function}	callback 	callback(err, result)
 */
function _disableRoutes(res, callback){

    let route   = res._Context.assert.route
      //, errmsg  = res._Context.assert.errors.length >0 ? res._Context.assert.errors.join('\n') : undefined

	if(!route || !route.method) return callback()

    if(route.method !== 'dir' && route.method !== 'queue' && route.method.toLowerCase() !== 'gcpubsub') return callback()

    route.stateInfo 		= route.stateInfo 			|| {}
    route.stateInfo.state 	= route.stateInfo.state 	|| pluginManager.DISABLEBYERROR
    route.stateInfo.plugin 	= route.stateInfo.plugin 	|| res._Context.assert.module
    route.stateInfo.route 	= route.subscription 		|| route.path
    route.stateInfo.tenant 	= route.stateInfo.tenant	|| route.tenant
    var resTmp = {
        '_Context' :{
            'reqId': res._Context.reqId,
            'disableRouteReq': {
                'body':{
                    'module': res._Context.assert.module,
                    'route' : route,
                    'reason': {
                        'repeatError': res._Context.assert.repeatError,
                        'desc':  'HA Server / State File assertion failed',
                        'error':{
                            //'code':   // not used 
                            'msg': res._Context.assert.errormsg
                        }
                    }
                    
                }
            }
        }
    }
    pluginManager.disableRouteService(resTmp, function(err){
        if(err && err instanceof warn)
            logger.warn('--->_disableRoutes: Req.Id=' + res._Context.reqId + ', Unable to disable a Route. ' + err.message)
        else if(err) 
            logger.error('--->_disableRoutes: Req.Id=' + res._Context.reqId + ', Unable to disable a Route. ' + err.message)
            callback()
    })
}

/*
 * Get assertion server list from env. variables matching pattern <OXSNPS_SERVER_PREFIX>_ASSERT_<name>
 */
function _getServerList(){

	let name		= undefined
      , pattern 	= '^' + process.env['OXSNPS_SERVER_PREFIX'] + 'ASSERT_(.*)$'
	  , serverList	= {}
	  , value		= undefined	
	  , result		= undefined	

	pattern = new RegExp(pattern)	
	
	// iterate through env. list and get assertion server list
	for(name in process.env){
		value	= process.env[name]
		result 	= name.match(pattern)
		// result => [ "OXS_BATCH_IMPORT_ASSERT_A2WS_SERVER", "A2WS_SERVER"]
		if(result && result.length >=2){
			serverList[result[1]] = JSON.parse(value)
		}
	}
	
	if(Object.keys(serverList).length <=0) serverList = undefined		// Object.keys returns an empty array if there are no keys in serverList
	
	if(logger.isDebugEnabled()){
		if(!serverList) logger.debug('--->getServerList: There are no env. variables matching ' +  process.env['OXSNPS_SERVER_PREFIX'] + 'ASSERT_<server name>')
		else logger.debug('--->assertJobConditions: Assert Server Info: ' +  JSON.stringify(serverList))
	}

	return serverList
}

/**
 * Disable plugin routes
 * @param  {Response}	res 		Response Object
 * @param  {Function}	callback 	callback()
 */
function _disablePluginRoutes(res, callback){

    let errmsg  = res._Context.assert.errors.length >0 ? res._Context.assert.errors.join('\n') : undefined
      , resTmp  = undefined

	if(!res._Context.assert.module) return callback()

    resTmp = {
        '_Context': {
            'reqId': res._Context.reqId,
            'disableRoutesReq': {
                'module': res._Context.assert.module,
                'reason': {
                    'error':{
                        'msg': res._Context.assert.errormsg
                    }
                }    
            }
        }
    }

    if(logger.isDebugEnabled()) logger.debug('--->_disablePluginRoutes: Req. id=' + resTmp._Context.reqId + ', Disabling all the routes of a plugin: ' + resTmp._Context.disableRoutesReq.module)
        
    pluginManager.disablePluginRoutesService(resTmp, function(err){
        if(err && err instanceof warn) logger.warn('--->_disablePluginRoutes: Req.Id=' + resTmp._Context.reqId + ', ' + err.message)
        else if(err) logger.error('--->_disablePluginRoutes: Req.Id=' + resTmp._Context.reqId + ', Unable to disable plugin routes. Reason: ' + err.message)
        else logger.info('--->_disablePluginRoutes: Req.Id=' + resTmp._Context.reqId + ', Disabled all the routes of a plugin: '  + resTmp._Context.disableRoutesReq.module)
        callback()            
    })    
}

/**
 * Get Error message from server assertion results
 * @param {*} res 
 * @param {*} callback 
 */
function _getErrorMsg(res){

    let errmsg = undefined

    res._Context.assert.results.forEach(function(assertResult){
        if(!assertResult || assertResult.status !== STATUS_ERROR) return 
        if(errmsg && errmsg.lenngth >0) errmsg += os.EOL + assertResult.message
        else errmsg  = assertResult.message
    })

    return errmsg     
}

/**
 * Wait till an event occurs or time exceeds given max wait time
 * @param {*} res 
 * @param {*} callback 
 */
function _waitForEvent(res, callback){

    if(symlinkResponse[res._Context.assert.parms.path] && symlinkResponse[res._Context.assert.parms.path].length > 0){
        symlinkResponse[res._Context.assert.parms.path] = res._Context.assert.userEvents.concat(symlinkResponse[res._Context.assert.parms.path])
        return callback()
    }

    res._Context.assert.attempts++
    if(res._Context.assert.attempts > res._Context.assert.retryCount) return callback(new Error('No specified event (' + JSON.stringify(res._Context.assert.parms.events) + ') has been triggered for the path: ' + res._Context.assert.filename + ', within ' + res._Context.assert.maxwaittime + ' ms'))
    
    if(logger.isDebugEnabled()) logger.debug('--->_waitForEvent: Req. id=' + res._Context.reqId + ', No specified event (' + JSON.stringify(res._Context.assert.parms.events) + ') triggered yet, Number of attempts: ' + res._Context.assert.attempts + ', scheduled to check again after the interval: ' + res._Context.assert.interval + 'ms')
    setTimeout(
        function(){return _waitForEvent(res, callback)},
        res._Context.assert.interval
    )    
}

/**
 * Log a Warning if Service Response time exceeded maxresptime 
 * @param  {Response}	res  	Response Object
 * @param  {String}		service service
 * @private
 */
function _logServiceRespTime(res, service){

	let respTime 	= 0
	  , os 			= require('os')
	  

	// Check resp. time only if maxresptime is defined
   	if(pluginConf.maxresptime){
		respTime = utils.dateDiff('x', res._Context.date, new Date()) // 'x' means returns diff. in ms

   		if(respTime > pluginConf.maxresptime * 1000){
			logger.warn('--->' + service + ': Req.Id=' + res._Context.reqId + ', '
						+ ' Resp. Time: ' + respTime/1000 + 's, Max. allowed: '
						+ pluginConf.maxresptime + 's. Average Load: ' + os.loadavg())
   		}
   	}
}
