/**-----------------------------------------------------------------------------
 * error.js: 	oxsnps error class 
 *
 * Author    :  AFP2web Team
 * Copyright :  (C) 2015 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V1.0.0
 * 
 * History
 *  V100        07.05.2015     Initial release
 *  
 *----------------------------------------------------------------------------*/
'use strict';

var MODULE_NAME 	= 'error'
  , MODULE_VERSION 	= '1.0.0'
  , log4js 			= require('log4js')
  , os 				= require('os')
  , util 			= require('util')
  , npsServer		= require('./server')
  , npsConf 		= npsServer.npsConf				// server-conf.js of oxs-xxx-server
  , utils 			= require('./helpers/utils')
  , supplant 		= require('supplant')
  , subs 			= new supplant()
  , errorObjs 		= {}

// Get Logger
var logger = log4js.getLogger(MODULE_NAME);
utils.log4jsSetLogLevel(logger, (utils.getAppLogLevel(npsConf.log) || 'INFO'));

// Export Module Version
exports.version = MODULE_VERSION;

module.exports=error;

/**************** PUBLIC FUNCTIONS ****************/
/**
 * initialize: 			Initialize Module
 * @param  {function} 	callback(err)
 */
module.exports.initialize = function(callback){

	var self = this;
	
	if(self._initialized){
		return callback();
	}
	self._initialized = true;
	callback();
}

/**
 * finalize: 			Finalize Module
 * @param  {function} 	callback(err) 
 */
module.exports.finalize = function(callback){

	var self = this;

	if(!self._initialized){
		callback();
	}
	self._initialized = false;
	callback();
}

/**
 * version: 			Process a HTTP Version Request
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
module.exports.version = function(req, res){
	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);
	logger.info('--->version: Req. id=' + res._Context.reqId);

	var data = '<h3>' +  NAME + ' v' + VERSION + '</h3>';
	res.end(data);
	
	logger.info('--->version: Req. id=' + res._Context.reqId + ', Version sent back, over');

	return;
}

/**
 * Set the log level of this module
 * @param  {req} 	GET req   http://localhost:1026/appservices/backup/setloglevel?loglevel=DEBUG|INFO|ERROR
 * @param  {res}	res
 */
module.exports.setLogLevel = function(req, res){
	var loglevel = req.query.loglevel || 'INFO';
	utils.log4jsSetLogLevel(logger, loglevel);
	return res.end('--->setLogLevel: Log level of ' + NAME + ' set to ' + loglevel);
}

/**
 * Return the log level of this module
 * @param  {req} 	GET req   http://localhost:1026/appservices/backup/getloglevel
 * @param  {res}	res
 */
module.exports.getLogLevel = function(req, res){
	logger.info('--->getLogLevel: Getting the log level of ' + NAME + '...');
	return res.end(logger.level.levelStr);
}

/**
 * Error constructor
 * @param  {JSON} errDetails            	Error details json
 *                                       	{
 *                                       		'code': 	<code>,
 *                                       		'module': 	<Module name>,
 *                                       		'func': 	<function name>,
 *                                       		'parms': 	<Parameter values to replace in error message defined in error.json>
 *                                       	}
 * @param  {Object} implementationContext   Optional. If given, all frames above implementationContext, 
 *                                          including implementationContext, will be omitted from the generated stack trace. 
 * @return {error}                       	returns error object
 *
 * Refer to http://www.bennadel.com/blog/2828-creating-custom-error-objects-in-node-js-with-error-capturestacktrace.htm
 */  
function error(errDetails, implementationContext){

	var msg 		= ''
	  , modErrors 	= undefined
	  ;

    // Ensure that errDetails exists to prevent refernce errors.
    errDetails = (errDetails || {});

    // Set Properties
    this.code 		= (errDetails.code || '' );
	this.module 	= (errDetails.module || '');    
	this.func 		= (errDetails.func || '');
	this.isAppError = true;
	this.parms 		= (errDetails.parms || {});

	// Get error object of a module
	modErrors = errorObjs[this.module];
	if(modErrors){
    	if(!modErrors[this.code]) 
    		this.message = 'Error code: ' + this.code + ' not found for plugin: ' + this.module;
    	else if(!modErrors[this.code].msg)
    		this.message = 'Error messsage not defined for error code: ' + this.code + ' for plugin: ' + this.module;
    	else
			this.message = subs.text(modErrors[this.code].msg,this.parms);
	}
	else{
		this.message = 'No error messages found for plugin: ' + this.module;
	}

	this.message = _buildHostnameAndIp() + ', Message: ' + this.message 
	
 	// Capture the current stacktrace and store it in the property "this.stack". By
    // providing the implementationContext argument, we will remove the current
    // constructor (or the optional factory function) line-item from the stacktrace; this
    // is good because it will reduce the implementation noise in the stack property.
    // --
    // Rad More: https://code.google.com/p/v8-wiki/wiki/JavaScriptStackTraceApi#Stack_trace_collection_for_custom_exceptions
    Error.captureStackTrace(this, (implementationContext || error));
};

/**
 * Factory function to create error object
 * @param  {JSON} errDetails            	Error details json
 *                                       	{
 *                                       		'code': 	<code>,
 *                                       		'module': 	<Module name>,
 *                                       		'func': 	<function name>,
 *                                       		'parms': 	<Parameter values to replace in error message defined in error.json>
 *                                       	}
 *                                          including implementationContext, will be omitted from the generated stack trace. 
 * @return {error}                       	returns error object
 *
 */
module.exports.createError = function(errDetails){
    // NOTE: We are overriding the "implementationContext" so that the createAppError()
    // function is not part of the resulting stacktrace.
    return(new error(errDetails, exports.createError));

}

/**
 * Register plugin/module error objects with error class
 * @param  {String} moduleName 	Module name
 * @param  {JSON} 	errorObj   	Plugin/Module JSON error object
 */
module.exports.registerErrorObj = function(moduleName, errorObj){
	if(moduleName && moduleName.length >0){
		errorObj = errorObj || {};
		errorObjs[moduleName] = errorObj;
	}
    return;
}

/******************  Private Functions ********************/
// see https://github.com/indutny/node-ip/blob/43e442366bf5a93493c8c4c36736f87d675b0c3d/lib/ip.js#L98
//
// ### function _ipAddress(name, family)
// #### @name {string|'public'|'private'} **Optional** Name or security
//      of the network interface.
// #### @family {ipv4|ipv6} **Optional** IP family of the address (defaults
//      to ipv4).
//
// Returns the address for the network interface on the current system with
// the specified `name`:
//   * String: First `family` address of the interface.
//             If not found see `undefined`.
//   * 'public': the first public ip address of family.
//   * 'private': the first private ip address of family.
//   * undefined: First address with `ipv4` or loopback address `127.0.0.1`
//
function _ipAddress(name, family){

    var interfaces = os.networkInterfaces()
    var all
    
    // Default to `ipv4`
    family = _normalizeFamily(family)
    
    // If a specific network interface has been named, return the address
    if(name && name !== 'private' && name !== 'public'){
        var res = interfaces[name].filter(function(details){
        var itemFamily = details.family.toLowerCase()
        return itemFamily === family
        })
        if(res.length === 0)
        return undefined
        return res[0].address
    }
    
    var all = Object.keys(interfaces).map(function (nic){
        // Note: name will only be `public` or `private` when this is called
        var addresses = interfaces[nic].filter(function (details){
        details.family = details.family.toLowerCase()
        if(details.family !== family || _ipIsLoopback(details.address)){
            return false
        } else if(!name){
            return true
        }
    
        return name === 'public' ? _ipIsPrivate(details.address) :
            _ipIsPublic(details.address)
        })
    
        return addresses.length ? addresses[0].address : undefined
    }).filter(Boolean)
    
    return !all.length ? _ipLoopback(family) : all[0]
}

function _ipIsPrivate(addr){
    return /^(::f{4}:)?10\.([0-9]{1,3})\.([0-9]{1,3})\.([0-9]{1,3})$/i
        .test(addr) ||
        /^(::f{4}:)?192\.168\.([0-9]{1,3})\.([0-9]{1,3})$/i.test(addr) ||
        /^(::f{4}:)?172\.(1[6-9]|2\d|30|31)\.([0-9]{1,3})\.([0-9]{1,3})$/i
        .test(addr) ||
        /^(::f{4}:)?127\.([0-9]{1,3})\.([0-9]{1,3})\.([0-9]{1,3})$/i.test(addr) ||
        /^(::f{4}:)?169\.254\.([0-9]{1,3})\.([0-9]{1,3})$/i.test(addr) ||
        /^f[cd][0-9a-f]{2}:/i.test(addr) ||
        /^fe80:/i.test(addr) ||
        /^::1$/.test(addr) ||
        /^::$/.test(addr)
}
  
function _ipIsPublic(addr){
   return !_ipIsPrivate(addr)
}
  
function _ipIsLoopback(addr){    
    return /^(::f{4}:)?127\.([0-9]{1,3})\.([0-9]{1,3})\.([0-9]{1,3})/
        .test(addr) || /^fe80::1$/.test(addr) || /^::1$/.test(addr) || /^::$/.test(addr)
}

function _ipLoopback(family){

    // Default to `ipv4`
    family = _normalizeFamily(family)
  
    if(family !== 'ipv4' && family !== 'ipv6'){
      throw new Error('family must be ipv4 or ipv6')
    }
    return family === 'ipv4' ? '127.0.0.1' : 'fe80::1'
}

function _normalizeFamily(family){
    return family ? family.toLowerCase() : 'ipv4'
 }
    
function _getHostname(){
    return os.hostname().split('.')[0]
}

function _buildHostnameAndIp(){
    return 'hostname: ' + _getHostname() + ', privateIp: ' + _ipAddress('private')  + ', publicIp: ' + _ipAddress('public')
}

function _extendErrorMessage(message){
    return _buildHostnameAndIp() + ', Message: ' + message
}

/**
 * Define an Error class that extends Node.js Error class by adding info about the server's Hostname and Ip
 * 
 * ex.: new oxsError('some error')
 * ...may log:
 *  hostname: LM20, privateIp: 127.0.0.1, publicIp: 10.0.2.15, Message: some error
 */
class oxsError extends Error {
    constructor(message) {
        super(_extendErrorMessage(message))
        this.name = "oxsError"
    }
}
module.exports.oxsError = oxsError

// Inherit from node.js Error class
util.inherits(error, Error);
