/**-----------------------------------------------------------------------------
 * date_format.js:  Node.js module that helps to get date as formatted string
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2014 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V1.0.0
 * 
 * History
 *  V100   26.08.2014  Initial release
 *
 *----------------------------------------------------------------------------*/
'use strict';
var moment = require("moment")
  ;

exports.ISO8601_FORMAT = "yyyy-MM-dd hh:mm:ss.SSS";
exports.ISO8601_WITH_TZ_OFFSET_FORMAT = "yyyy-MM-ddThh:mm:ssO";
exports.DATETIME_FORMAT = "dd MM yyyy hh:mm:ss.SSS";
exports.ABSOLUTETIME_FORMAT = "hh:mm:ss.SSS";

function padWithZeros(vNumber, width) {
  var numAsString = vNumber + "";
  while (numAsString.length < width) {
    numAsString = "0" + numAsString;
  }
  return numAsString;
}
  
function addZero(vNumber) {
  return padWithZeros(vNumber, 2);
}

/**
 * Formats the TimeOffest
 * Thanks to http://www.svendtofte.com/code/date_format/
 * @private
 */
function offset(date) {
  // Difference to Greenwich time (GMT) in hours
  var os = Math.abs(date.getTimezoneOffset());
  var h = String(Math.floor(os/60));
  var m = String(os%60);
  if (h.length == 1) {
    h = "0" + h;
  }
  if (m.length == 1) {
    m = "0" + m;
  }
  return date.getTimezoneOffset() < 0 ? "+"+h+m : "-"+h+m;
}

exports.asString = function(/*format,*/ date) {
  var format = exports.ISO8601_FORMAT;
  if (typeof(date) === "string") {
    format = arguments[0];
    date = arguments[1];
  }

  var vDay = addZero(date.getDate());
  var vMonth = addZero(date.getMonth()+1);
  var vYearLong = addZero(date.getFullYear());
  var vYearShort = addZero(date.getFullYear().toString().substring(2,4));
  var vYear = (format.indexOf("yyyy") > -1 ? vYearLong : vYearShort);
  var vHour  = addZero(date.getHours());
  var vMinute = addZero(date.getMinutes());
  var vSecond = addZero(date.getSeconds());
  var vMillisecond = padWithZeros(date.getMilliseconds(), 3);
  var vTimeZone = offset(date);
  var formatted = format
    .replace(/dd/g, vDay)
    .replace(/MM/g, vMonth)
    .replace(/y{1,4}/g, vYear)
    .replace(/hh/g, vHour)
    .replace(/mm/g, vMinute)
    .replace(/ss/g, vSecond)
    .replace(/SSS/g, vMillisecond)
    .replace(/O/g, vTimeZone);
  return formatted;

};

/**
 * Get passed date or current date and time as local string
 * @param  {Date}   date  Date
 * @return {String}      Date as local string like 2015-11-15T18:53:45.382+05:30
 */
exports.getAsUTCString = function(date){

  var dateTmp = null;
  
  if(date) dateTmp = moment(date);
  else dateTmp = moment(new Date());
  return dateTmp.format('YYYY-MM-DDTHH:mm:ss.SSSZ');
}

/**
 * Get passed date or current date and time in a given format
 * @param  {Date}     date  Date
 * @param  {String}   format  Format string. Refer to http://momentjs.com/docs/
 * @return {String}   Date as string in given format
 */
exports.getAsString = function(date, format, inputFormat){

  var dateTmp = null;
  
  if(date && inputFormat) dateTmp = moment(date, inputFormat);
  else if(date) dateTmp = moment(date);
  else dateTmp = moment(new Date());

  return dateTmp.format(format);
}

/**
 * Convert date string dd:MM:YYYY to YYYY<delimiter>MM<delimiter>dd 
 * @param  {String} datestr   Date String in dd:MM:YYYYY
 * @param  {String} delimiter Delimiter
 * @return {String}           Date string in YYYY<delimiter>MM<delimiter>dd  format
 */
exports.ddmmyyyy2yyyymmdd = function(datestr, delimiter){

  var ret = undefined;

  if(!datestr) return datestr;
  ret = datestr.substring(4,8) + delimiter +
          datestr.substring(2,4) + delimiter +
          datestr.substring(0,2);
          
  return ret;
}

/**
 * Convert date string YYYY<delimiter>MM<delimiter>DD to DD<delimiter>MM<delimiter>YYYY
 * @param  {String} datestr   Date String in YYYY<delimiter>MM<delimiter>DD
 * @param  {String} delimiter Delimiter
 * @return {String}           Date string in DD<delimiter>MM<delimiter>YYYY  format
 */
exports.yyyymmdd2ddmmyyyy = function (datestr, delimiter){

  var ret = undefined;

  if(!datestr) return datestr;
  
  if(!delimiter) delimiter = '.';

  ret = datestr.substring(8,10) + delimiter +
          datestr.substring(5,7) + delimiter +
          datestr.substring(0,4);
          
  return ret;
}

/**
 * Subtract specified years from the given date
 * @param  {Date}     date  Date
 * @param  {Integer}  years Number of years to deduct
 * @return {Date}           Date
 */
exports.subtractYears = function(date, years){
  var dateTmp = null;
  
  if(date) dateTmp = moment(date);
  else dateTmp = moment(new Date());
  return new Date(dateTmp.subtract(years, 'years'));
}

/**
 * Subtract specified months from the given date
 * @param  {Date}     date  Date
 * @param  {Integer}  months Number of months to deduct
 * @return {Date}           Date
 */
exports.subtractMonths = function(date, months){
  var dateTmp = null;
  
  if(date) dateTmp = moment(date);
  else dateTmp = moment(new Date());
  return new Date(dateTmp.subtract(months, 'months'));
}

/**
 * Subtract specified days from the given date
 * @param  {Date}     date  Date
 * @param  {Integer}  days Number of days to deduct
 * @return {Date}           Date
 */
exports.subtractDays = function(date, days){
  var dateTmp = null;
  
  if(date) dateTmp = moment(date);
  else dateTmp = moment(new Date());
  return new Date(dateTmp.subtract(days, 'days'));
}

/**
 * Subtract specified hours from the given date
 * @param  {Date}     date  Date
 * @param  {Integer}  hours Number of hours to deduct
 * @return {Date}           Date
 */
exports.subtractHours = function(date, hours){
  var dateTmp = null;
  
  if(date) dateTmp = moment(date);
  else dateTmp = moment(new Date());
  return new Date(dateTmp.subtract(hours, 'hours'));
}

/**
 * Add specified minutes from the given date
 * @param  {Date}     date  Date
 * @param  {Integer}  hours Number of hours to deduct
 * @return {Date}           Date
 */
exports.addMinutes = function(date, minutes){
    var dateTmp = null;
    
    if(date) dateTmp = moment(date);
    else dateTmp = moment(new Date());
    return new Date(dateTmp.add(minutes, 'minutes'));
  }

// V5.0.35 Begin
/**
 * Add specified seconds from the given date
 * @param  {Date}     date  Date
 * @param  {Integer}  seconds Number of seconds to add
 * @return {Date}           Date
 */
exports.addSeconds = function(date, seconds){
  let dateTmp = undefined
  
  if(date) dateTmp = moment(date)
  else dateTmp = moment(new Date())
  return new Date(dateTmp.add(seconds, 'seconds'))
}

/**
* Add specified seconds from the given date
* @param  {Date}     date  Date
* @param  {Integer}  seconds Number of seconds to add
* @return {Date}           Date
*/
exports.subtractSeconds = function(date, seconds){
  let dateTmp = undefined
  
  if(date) dateTmp = moment(date)
  else dateTmp = moment(new Date())
  return new Date(dateTmp.subtract(seconds, 'seconds'))
}

// V5.0.35 End