/**-----------------------------------------------------------------------------
 * expressAppUtils.js: Node.js module that provides function to handle Express HTTP routes
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2014 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V1.0.0
 * 
 * History
 *  V4.1.0      18.02.2020      OTS-2692: Add middlewares at run time based on the server requirements
 *  V100        02.03.2015      Initial release
 *
 *----------------------------------------------------------------------------*/
'use strict'

var MODULE_NAME		= 'expressAppUtils'
  , MODULE_VERSION 	= '1.0.0'
  , async	    	= require('async')
  , express         = require('express')
  , log4js	    	= require('log4js')  
  , url         	= require('url')
  , utils       	= require('./utils')
  , npsServer   	= require('../server')
  , expressApp		= npsServer.expressApp			// is the Express App
  , npsConf 		= npsServer.npsConf				// server-conf.js of oxs-xxx-server
  
// Get Logger
var logger = log4js.getLogger(MODULE_NAME)
utils.log4jsSetLogLevel(logger, (utils.getAppLogLevel(npsConf.log) || 'INFO'))

/**
 * enableRoutes: 		Add HTTP Routes defined in the plugin configuration file
 * @param  {module} 	handle of the module
 * @param  {json} 		routes
 *         				routes = [
 *         					{
 *         						"enable": "on|off"
 *         						"path": 	"/path/to/the/service",
 *         						...,
 *         					}
 *         					{...},
 *         				]
 * @param  {function} 	callback(err)
 */
exports.enableRoutes = function(module, routes, callback){

	logger.debug('-->enableRoutes: Adding HTTP following routes: ' + JSON.stringify(routes))

	// Assert routes
	if(!routes || routes.length <0)	return callback() // routes are optional
		
    // Loop through all routes
	async.each(routes,
		function(route, nextTask){
            route.enable = route.enable || 'on'
            route.method = route.method || 'get'
			// Add HTTP route if only it's enabled and method (get,post,...) is defined
			if(route.enable === 'on' && _isHTTPRoute(route)){
				return exports.enableRoute(module, route, nextTask)
			} 
			nextTask()
		},
		function(err){
			return callback(err)
		}
	)
}

/**
 * enableRoute: 		Enable the given http route
 * @param  {module} 	handle of the module
 * @param  {json} 		route
 * 			        	route = {
 * 			        		"path": 	"/path/to/the/services",
 * 			        	 	"method": 	"get|post",
 * 			           		"service": 	"version",
 *                 		}
 * @param  {function}	callback(err)
 * @param  {boolean} 	bInit 	!!!
 */
exports.enableRoute = function(module, route, callback, bInit){

    //let path    = undefined

	logger.debug('--->enableRoute: Route: ' + JSON.stringify(route))
	
	route.method = route.method || 'get'
	if(bInit === undefined){//!!! NOT FULLY IMPLEMENTED
		bInit = true
	}

	if(!route.path || !route.method || !route.service)
		return callback(new Error('HTTP route does not have value for either "path" or "method" or "service"'))

	try{
        if(module.router){
            //path    = route.path.replace(module.router.basePath, '')
            module.router[route.method](route.path, module[route.service])	
        }
        else{
            // First add the route to the Express App list of routes
            //!!! Allow route.path the be an array of path!!!
            expressApp[route.method](route.path, module[route.service])	

            // Last ensure to move the '/*' route to the end of the list of routes
    //		if(!bInit){//!!! NOT FULLY IMPLEMENTED
                exports.moveDefaultAppRoute()
    //		}
        }
	}
	catch(err){
		return callback(err)
	}
	callback()
}

// V4.1.0 Begin
/**
 * @param  {plugin}     plugin      plugin
 *                      plugin : {
 *                          basePath:  <basePath. String | regular exprn>,
 *                          pluginConf: {
 *                              routes: [
 *                                  {
 *                                      "path": 	<path>,
 *                                      "method": 	<http method>,
 *                                      "service": 	<service function>
 *                                  },
 *                              ],
 *                              middlewares: [
 *                                  {
 *                                      "name":     <middleware name>,
 *                                      "handler":  <handler name>
 *                                      "parms":    <Optional. middleware parms. Used when middleware is applied for all route paths>
 *                                      "routes":[  <Optional. Apply middleware to specific route paths>
 *                                          {
 *                                              "path": <Optional. String or Array of paths. If given, middleware is applied to the specified path(s). Else to all paths>
 *                                              "parms": <middleware parms>
 *                                          },
 *                                       ]
 *                                  }
 *                              ],
 *                          }
 *                        }
 * @param  {function} 	callback    callback(err, router) 
 */
exports.setupRouter = function(plugin, callback){

    let router      = undefined
      , pluginConf  = plugin ? plugin.pluginConf : undefined
      , parms       = undefined
      //, rbasepath   = undefined

    //if(!plugin || !pluginConf || !plugin.basePath) return callback(new Error('Missing pluginConf.routes/plugin.basePath'))
    if(!plugin || !pluginConf) return callback(new Error('Missing pluginConf'))
    
    plugin.router = router = express.Router()
    //router.basePath = plugin.basePath || ''

	async.series([
		// Add middlewares.  
		function(nextTask){
            if(!pluginConf.middlewares) return nextTask()   // middlewares are optional
            parms = {
                'router':       plugin.router,
                'moduleName':   pluginConf.module,
                'middlewares':  pluginConf.middlewares
            }
            exports.addMiddlewares(parms, nextTask)
		},
		// Add routes to router
		function(nextTask){
            if(!pluginConf.routes || pluginConf.routes.length <=0) return nextTask()  // There can be plugins without routes 

            async.eachSeries(pluginConf.routes, function(route, next){
                if(!route || !route.path || !_isHTTPRoute(route)) return next()
                route.method    = route.method || 'get'
                route.enable    = route.enable || 'on'

                /*
                rbasepath = ''
                if(typeof (router.basePath) ===  "string") rbasepath = router.basePath
                else {
                    // else it is reg.exp
                    rbasepath = route.path.match(router.basePath)
                    if(rbasepath && rbasepath.length > 0) rbasepath = rbasepath[0]
                    else return next(new Error('Unable to add route:' + route.path +', Invaid plugin base path: ' + router.basePath))
                }                
                path = route.path.replace(rbasepath, '')  // Since base path is given while adding router to express app, remove it from route.path
                if(logger.isDebugEnabled()) logger.debug('-->setupRouter: Adding route, plugin: ' + pluginConf.module + ', route: ' + rbasepath + path + ', method: ' + route.method)
                */
               if(logger.isDebugEnabled()) logger.debug('-->setupRouter: Adding route, plugin: ' + pluginConf.module + ', route: ' + route.path + ', method: ' + route.method)
                try{
                    router[route.method](route.path, plugin[route.service])
                    next()
                }
                catch(err){
                    return next(new Error('Unable to add route, ' + err.message + ', plugin: ' + pluginConf.module + ', route: ' + route.path  + ', method: ' + route.method))
                }
            }, function(err) {
                if(err) return nextTask(err)
                // Add router to express APP 
                //expressApp.use(router.basePath, router, exports.routeNotAvailable)
                expressApp.use(router, 
                    function(req, res, next){
                        return next()   // go to next router
                    }
                )
                exports.moveDefaultAppRoute()   //Ensure that the '/*' route is at the end of the list of routes
                nextTask()
            })			
		}
	],
	function(err){
		callback(err, router)
	})
}
  
/**
 * @param  {JSON}     parms      
 *                          parms = {
 *                              router: <router>,
 *                              moduleName: <module>,
 *                              middlewares: [
 *                                  {
 *                                      "name":     <middleware name>,
 *                                      "handler":  <Optional. handler name>
 *                                      "parms":    <Optional. middleware parms. Used when middleware is applied for all route paths>
 *                                      "routes":[  <Optional. Apply middleware to specific route paths>
 *                                          {
 *                                              "path": <Optional. String or Array of paths. If given, middleware is applied to the specified path(s). Else to all paths>
 *                                              "parms": <Optional. middleware parms>
 *                                          },
 *                                       ]
 *                                  }
 *                              ],
 *                          }
 * @param  {function} 	callback    callback(err, router) 
 */
/** 
 *  Ex: 
    {
        router: <router>,
        moduleName: "oxsnps-test",
        "middlewares": <Optional. Array of middlewares used by the plugin> 
        [
            // Add static route for <serverDir>/restish
            // sample URL: http://localhost:9000/restish/restish.yaml, http://localhost:9000/restish/restish.json
            {
                "name": "express",  
                "handler": "static",
                "routes":[
                    {
                        "path": ["/restish" ],      // Actual path is <serverDir>/restish. Files (restish.yaml, restish.json) to be served should be in this directory
                        "parms": path.join(npsDir, 'restish')
                    }        
                ]
            },            
            // Set favicon
            {
                "name": "serve-favicon",   
                "parms": npsDir + '/public/img/favicon.ico'
            },
            // Add serve-index for specific route paths
            {
                "name": "serve-index",  
                "parms": "",
                "routes":[
                    {
                        "path": "/oxsnps",		//  Allow to browse "/" directory
                        "parms": "/"
                    },
                    {
                        "path": "/public",			// Allow to browse public of server installed directory
                        "parms": npsDir + "/public"
                    }        
                ]
            },
            // Add body-parser.json for specific route paths
            {
                "name": "body-parser",
                "handler": "json"
                "routes":[
                    {
                        "path": "/services/afp2any/transform",
                        "parms": {"limit": "2mb"}  // Limit request body size to  2 MB
                    },
                    {
                        "path": "/services/nopodofo/*",
                        "parms": {"limit": "25mb"}  // Limit request body size to  25 MB
                    }            
                ]
            },
            // Add body-parser.urlencoded() applied to all route paths
            {
                "name": "body-parser",
                "handler": "urlencoded",
                "parms": {"extended":false}
            },
            // Add oxsnps-upload specific route paths
            {
                "name": "oxsnps-upload",
                "handler": "register",
                "routes":[
                    {
                        "path": "/services/test/pdfupload",     // upload only PDF files to npsDir + "pdfuploads"
                        "parms": {
                            destination: 	function(req, file, callback){return callback(null, path.resolve(npsServer.npsDir + '/pdfuploads/'))}, // to set the dirname where to store the uploaded file
                            filename: 		function(req, file, callback){return callback(null,file.originalname)}, // the filename to use when storing the uploaded file
                            fileFilter: 	function(req, file, callback){if(path.extname(file.originalname).toLowerCase() === '.pdf') return callback(null,true) return callback(null,false)}, // the function to call to filter files to upload
                        }  
                    },
                ]
            },
            {
                "name": "oxsnps-upload",
                "handler": "register",
                "routes":[
                    {
                        "path": "/services/test/txtupload",         // upload only PDF files to npsDir + "txtuploads"
                        "parms": {
                            destination: 	function(req, file, callback){return callback(null, path.resolve(npsServer.npsDir + '/txtuploads/'))}, // to set the dirname where to store the uploaded file
                            filename: 		function(req, file, callback){return callback(null,file.originalname)}, // the filename to use when storing the uploaded file
                            fileFilter: 	function(req, file, callback){if(path.extname(file.originalname).toLowerCase() === '.txt') return callback(null,true) return callback(null,false)}, // the function to call to filter files to upload
                        }  
                    },
                ]
            }		
        ]
    }
 *  
 */
exports.addMiddlewares = function (parms, callback){

    let module          = undefined
       , handler        = undefined
       , router         = parms.router
       , moduleName     = parms.moduleName
       , middlewares    = parms.middlewares
       , error          = undefined
       , path           = undefined
       , route          = undefined
       //, rbasepath      = undefined // route base path
       //, rtrBasepath      = undefined // router base path

    if(!middlewares || middlewares.length <=0) return callback()

    async.eachSeries(middlewares, function(middleware, next){

        if(!middleware || !middleware.name || middleware.name.length <=0) return next(new Error('Missing middleware name. middleware: ' + JSON.stringify(middleware) + ', plugin: ' + moduleName))

        try{
            module = require(middleware.name)   // !!! if middleware is oxsnps plugin, get plugin handle through PluginManager. Should be done after the task: OTS-2050 - oxsnps: Extend oxsnps-core to specify the order of plugin loading 
        }catch(err){
            return next(new Error('Unable to load middlware: '  + middleware.name + ', ' +  err.message + ', for plugin: ' + moduleName))
        }

        handler = middleware.handler        

        // If middleware.routes is not given, add middleware without any specific route so that it applies to all routes paths
        if(!middleware.routes || middleware.routes.length <=0){
            if(logger.isDebugEnabled()) logger.debug('-->addMiddlewares: Adding middleware:' + middleware.name + (handler ? '.' + handler : '' ) + 
                    ', plugin: ' + moduleName + ', parms:' + ((typeof middleware.parms === "string") ? middleware.parms : JSON.stringify(middleware.parms)))
            try{        
                if(handler) router.use(module[handler](middleware.parms))
                else  router.use(module(middleware.parms))
                next()
            }catch(err){
                return next(new Error('Unable to load middlware: '  + middleware.name + (handler ? '.' + handler : '' ) +  ', ' +  err.message + 
                    ', for plugin: ' + moduleName + ', parms:' + ((typeof middleware.parms === "string") ? middleware.parms : JSON.stringify(middleware.parms))))
            }
            return
        }

        //rtrBasepath = router.basePath || ''

        // Add middleware for specific routes
        for(let ridx=0; ridx < middleware.routes.length; ridx++){
            route = middleware.routes[ridx]
            if(!route) continue
            
            if(route.path && route.path.length > 0){
				// route.path can be a string or an array of paths
                //if(!Array.isArray(route.path)) route.path = [route.path]

                //for(let idx=0; idx < route.path.length; idx++){
                    /*
                    if(typeof (rtrBasepath) ===  "string") rbasepath = rtrBasepath
                    else {
                        // else it is reg.exp
                        rbasepath = route.path[idx].match(rtrBasepath)
                        if(rbasepath && rbasepath.length > 0) rbasepath = rbasepath[0]
                        else return next(new Error('Unable to add middleware:' + middleware.name + (handler ? '.' + handler : '' ) + ', for route: ' + route.path[idx] +', Invaid plugin base path: ' + rtrBasepath))
                    }
    
					path = route.path[idx].replace(rbasepath, '')
					if(logger.isDebugEnabled()) logger.debug('-->addMiddlewares: Adding middleware:' + middleware.name + (handler ? '.' + handler : '' ) +
                        ', plugin: ' + moduleName + ', path: ' + rbasepath + path  + ', parms:' + ((typeof route.parms === "string") ? route.parms : JSON.stringify(route.parms)))     
                    */               
                   //path = route.path[idx]
                   if(logger.isDebugEnabled()) logger.debug('-->addMiddlewares: Adding middleware:' + middleware.name + (handler ? '.' + handler : '' ) +
                       ', plugin: ' + moduleName + ', path: ' + JSON.stringify(route.path)  + ', parms:' + ((typeof route.parms === "string") ? route.parms : JSON.stringify(route.parms)))     

                    try{
                        if(handler) router.use(route.path, module[handler](route.parms))      // route.use("/test/json", body-parser.json({..}))
                        else router.use(route.path, module(route.parms))                      // route.use("/ci.test/conf", serve-index( <pluginsDir> + "ci.test/conf"))
                    }catch(err){
                        error= new Error('Unable to add middleware:' + middleware.name + (handler ? '.' + handler : '' ) + ', ' + err.message + 
                                    ', plugin: ' + moduleName + ', path: ' + JSON.stringify(route.path)  +
                                    ', parms:' + ((typeof route.parms === "string") ? route.parms : JSON.stringify(route.parms)))
                        break
                    }
                //}
                // break  or continue outer for 
				if(error) break; else continue;
            }

			// route.path can be optional. Add middleware without any path so that it applies to all paths
            if(logger.isDebugEnabled()) logger.debug('-->addMiddlewares: Adding middleware:' + middleware.name + (handler ? '.' + handler : '' ) +  
                    ', plugin: ' + moduleName + ', parms:' + ((typeof route.parms === "string") ? route.parms : JSON.stringify(route.parms)))            
            try{
                if(handler) router.use(module[handler](route.parms))            // route.use( body-parser.json({..}))
                router.use(module(route.parms))                                 // route.use( serve-favicon(npsDir + '/public/img/favicon.ico')
            }catch(err){
                error= new Error('Unable to add middleware:' + middleware.name + (handler ? '.' + handler : '' ) + ', ' + err.message + 
                            ', plugin: ' + moduleName + ', parms:' + ((typeof route.parms === "string") ? route.parms : JSON.stringify(route.parms)))
                break
            }
        }

        next(error)
    },function(err){
        return callback(err)
    })			
}
// V4.1.0 End

/**
 * disableRoutes: 		Remove HTTP Routes
 * @param  {json} 		routes
 *         				routes = [
 *         					{
 *         						"enable": "on|off"
 *         						"path": 	"/path/to/the/service",
 *         						...,
 *         					}
 *         					{...},
 *         				]
 * @param  {function} 	callback(err)
 */
exports.disableRoutes = function(module, routes, callback){

    if(arguments.length === 2){
        callback = routes
        routes = module        
    }
    
	logger.debug('-->disableRoutes: Removing following HTTP routes: ' + JSON.stringify(routes))

	// Loop through all routes
	async.each(routes,
		function(route, nextTask){
			// Add HTTP route if only it's enabled and method (get,post,...) is defined
			if(route.enable === 'on' && _isHTTPRoute(route)){
                if(arguments.length === 2) exports.disableRoute(route, nextTask)	// V4.1.0
				else exports.disableRoute(module, route, nextTask)
			} 
			else{
				nextTask()
			}
		},
		function(err){
			return callback(err)
		}
	)
}

/**
 * disableRoute: 	Disable the given HTTP route
 * @param  {json} 	route
 * 			        route = { 
 * 			        	"path": 	"/path/to/the/services",
 * 						"method": 	OPTIONAL http method GET|POST|PUT|...
 * 			        }
 * @param  {function} 	callback(err)
 */
exports.disableRoute = function(module, route, callback){

	logger.debug('--->disableRoute: route: ' + JSON.stringify(route))
    
    if(!_isHTTPRoute(route)) return callback()

	try{
        if(arguments.length === 3)	// V4.1.0
            _removeRouterRoute(module.router, route.path, route.method) // Remove the route from router
        else{   
            // Remove the route from the app (for older compatibility)
            callback = route
            route = module
            _removeAppRoute(route.path, route.method)  
        }
	}   
	catch(err){
		return callback(err)
	}
	callback()
}

/**
 * moveDefaultAppRoute:  	Ensure that the '/*' route is at the end of the list of routes
 */
exports.moveDefaultAppRoute = function(){
	_removeAppRoute('/*')
	expressApp['get']('/*', exports.routeNotAvailable)
	expressApp['post']('/*', exports.routeNotAvailable)			
}

/**
 * routeNotAvailable: 	Process a unmapped route request
 * @param  {request}  	req Request Object
 * @param  {response} 	res Response Object
 */
exports.routeNotAvailable = function(req, res){
	var retMsg = ''
	  , url_parts = url.parse(req.url, true)

	retMsg = req.method + ' ' + (req.originalUrl || url_parts['pathname']) + ' route is not available or not enabled'
	logger.info('--->routeNotAvailable: ' + retMsg)
	return res.status(404).end(retMsg)
}

/***** Private Functions ***********/
/**
 * _removeAppRoute: 	Remove the given route from the Express App
 * @param  {string} 	route   The route to be removed
 * @param  {string} 	method  HTTP method Optional. 
 */
function _removeAppRoute(route, method){

	var stack = expressApp._router.stack	// get the Express App router stack

	// Remove passed route
	for(var i = stack.length - 1; i >= 0; i--){
		if(stack[i].route && stack[i].route.path){
			if(method){
				// if method is given, search for path having that method
				if(stack[i].route.path === route &&
					stack[i].route.methods && stack[i].route.methods[method]){
					logger.info('deleting ' + route + ' route using ' + method + ' method')
	  				stack.splice(i, 1)					
	  				break
				}
			}
			else if(stack[i].route.path === route){
				logger.debug('deleting ' + route)
	  			stack.splice(i, 1)
	  			//break // do not break here since some path may be in 'get' and 'post'
			}
		}
		else{ // Could be a route added dynamically through the proxy module
			if(stack[i].name && stack[i].regexp){ // name="$oxseed$orderbase$tenny" & regexp is defined
				let namedFunction = route.replace(/\//g,"$") // replace all the / by $
				if (stack[i].name === namedFunction){
					logger.debug('deleting ' + route + ' whose name is: ' + namedFunction)
					stack.splice(i, 1)
					break
				}
			}
		}
	}
}

// V4.1.0 Begin
/**
 * _removeRouterRoute: 	Remove the given route from the Express Router instance
 * @param  {Router}     router  Express Router instance
 * @param  {string} 	route   The route to be removed
 * @param  {string} 	method  HTTP method Optional. 
 */
function _removeRouterRoute(router, route, method){

    let stack = router.stack	// get the Express App router stack
    
    //route = route.replace(router.basePath, '')

	// Remove passed route
	for(var i = stack.length - 1; i >= 0; i--){
		if(stack[i].route && stack[i].route.path){
            if(logger.isDebugEnabled()) logger.debug(stack[i].route.path)
			if(method){
				// if method is given, search for path having that method
				if(stack[i].route.path === route &&
					stack[i].route.methods && stack[i].route.methods[method]){
					logger.info('deleting ' + route + ' route using ' + method + ' method')
	  				stack.splice(i, 1)
	  				break
				}
			}
			else if(stack[i].route.path === route){
				logger.debug('deleting ' + route)
	  			stack.splice(i, 1)
	  			//break // do not break here since some path may be in 'get' and 'post'
			}
		}
        else{ // Could be a route added dynamically through the proxy module
			if(stack[i].name && stack[i].regexp){ // name="$oxseed$orderbase$tenny" & regexp is defined
                let namedFunction = route.replace(/\//g,"$") // replace all the / by $
                if(logger.isDebugEnabled()) logger.debug("namedFunction: " + namedFunction + ", stack[i].name: " + stack[i].name)
				if (stack[i].name === namedFunction){
					logger.debug('deleting ' + route + ' whose name is: ' + namedFunction)
					stack.splice(i, 1)
					break
				}
			}
		}
	}
}

function _isHTTPRoute(route){
    let method = route.method || 'get' 

    method = method.toLowerCase()

    switch(method){
        // refer to https://www.w3.org/Protocols/rfc2616/rfc2616-sec9.html
        // https://www.tutorialspoint.com/http/http_methods.htm
        case 'get':
        case 'post':
        case 'delete':
        case 'put':
        case 'head':
        case 'options':
        case 'trace':
        case 'connect':
            return true
        default: 
            return false
    }
}
// V4.1.0 End