/**-----------------------------------------------------------------------------
 * httpUtils.js:  Node.js module that provides http utility functions
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2014 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V1.0.0
 * 
 * History
 *  V100   07.11.2014  Initial release
 *  V101   06.02.2015  Added postURL & processURL
 *  V102   25.02.2015  Added https support
 *  V103   17.12.2018  Ensured to always specify body length in uploadFileAsBuffer
 * 
 *  V4.1.9 	  11.09.2020  OTS-2812: oxsnps-core: Extend httpUtils.getURL() and postURL() function to retry failed requests based retry options parameter
 *----------------------------------------------------------------------------*/
'use strict';

var MODULE_NAME		= 'httpUtils'
  , fs      = require('fs')
  , http    = require('http')
  , https   = require('https')
  , log4js  = require('log4js')
  , url     = require('url')
  , request = require('request')
  , isHttps = /^https/i  
  , utils   = require('./utils')
  , npsServer = require('../server')
  , npsConf = npsServer.npsConf	  

// Get Logger
var logger = log4js.getLogger(MODULE_NAME);
utils.log4jsSetLogLevel(logger, (utils.getAppLogLevel(npsConf.log) || 'INFO'))
//utils.log4jsSetLogLevel(logger, 'debug')

/**
 * getURL:                  Get data from the passed URL
 * @param  {object|string}  options         http.request options, can be an object or a string
 * @param  {object}         retryOptions    {
 *                                              'interval': <interval in milliseconds>,
 *                                              'count': <count>,
 *                                              'errorCodes' : <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          }
 * @param  {function}       callback(err, data)
 */
// V4.1.9 Begin
exports.getURL = function(options, retryOptions, callback){

  let req           = undefined
    , httptimeout   = undefined
    , respEncoding  = undefined 
    , reqId         = undefined 
   // , errTmp        = undefined
    , protocol      = undefined
    , cbInvoked   = false //V4.1.28
    , statusCode    = undefined

    if(typeof retryOptions === 'function'){
        callback = retryOptions
        retryOptions = undefined
    }

    if(logger.isDebugEnabled()) logger.debug('--->getURL: ' + (options.reqId ? ('Req.Id=' + options.reqId +', ') : '') + 'options:' + JSON.stringify(options) + 
                ', retryOptions: ' + (retryOptions ?  JSON.stringify(retryOptions) : 'undefined')) 

    try{
        if(typeof options === 'string') options = url.parse(options)
        if(!options) return callback(new Error('Unable to parse URL string. options: ' + options))
    }catch(err){
        return callback(new Error('Unable to parse a URL string. options: ' + options + '. ' + err.message))
    }

    if(!options.path) return callback(new Error('getURL: options.path is missing, options: ' + JSON.stringify(options)))

    //if(options.encodeURI !== undefined && options.encodeURI === true) options.path = utils.encodeURIData(options.path)

    function _chkAndRetryGet(err, resData, resp){
        // V4.1.28 Begin
        if(cbInvoked === true){
          /*
          console.log('--->_chkAndRetryGet: ' 
              + (reqId ? ('Req.Id=' + reqId +', ') : '')
              + (reqId ? ('Req.Id=' + reqId) : '')
              + (err ? (', Error code: ' + err.code + ', Error: ' + err.message) : '')
              + ', options:' + JSON.stringify(options) 
              + ', request already received response(status/error code: '+ statusCode + '). Ignoring this repeat response')
          */          
          logger.warn('--->_chkAndRetryGet: ' 
              + (reqId ? ('Req.Id=' + reqId) : '')
              + (err ? (', Error code: ' + err.code + ', Error: ' + err.message) : '')
              + ', options:' + JSON.stringify(options) 
              + ', request already received response(previous status/error code: '+ statusCode + '). Ignoring this repeat response')
          return 
        }
        cbInvoked = true 
        if(err && !statusCode) statusCode = err.code || err.statusCode || err.message
        // V4.1.28 End
        if(!err) return callback(null, resData, resp)

        if(!retryOptions) return callback(err, resData, resp)

        // V5.0.28
        if(retryOptions && retryOptions.skip === true){ // no need to retry.   
            if(logger.isDebugEnabled()) logger.debug('--->getURL: ' + (options.reqId ? ('Req.Id=' + options.reqId +', ') : '') + 'options:' + JSON.stringify(options) + 
                ', retryOptions: ' + (retryOptions ?  JSON.stringify(retryOptions) : 'undefined') + 
                ', Skipped retrying the request'
                ) 
            return callback(err, resData, resp)  
        }

        let retryArgs = {
            'reqId':        reqId,
            'respEncoding': respEncoding,
            'functionName': 'getURL',
            'retryOptions': utils.clone(retryOptions),  // 5.0.26
            'options':      options,
            'err':          err, 
            'resData':      resData,
            'resp':   resp,
            'callback':     callback
        }
        return _retry(retryArgs)
    }     

    httptimeout     = options.timeout || options.httptimeout || undefined
    reqId           = options.reqId
    respEncoding    =  options.respEncoding
    options.reqId   = undefined
    options.respEncoding = undefined
    protocol = options.protocol
    options.protocol     = undefined

    try{
        // Set up the request
        if(protocol && isHttps.test(protocol)){
            req = https.request(options)
        }
        else{
            req = http.request(options)
        }
        req.end()
    }
    catch(err){
        //return callback(err)
        //console.log('--->getURL: request creation error. ' + (reqId ? (', Req.Id=' + reqId) : '') + (err ? (', Error code: ' + err.code + ', Error: ' + err.message) : ''))
        return _chkAndRetryGet(err)
    }

    // Timeout handling
    if(httptimeout){
        req.setTimeout(httptimeout, function(){
            //console.log('--->getURL: request creation error. ' + (reqId ? (', Req.Id=' + reqId) : '') + (err ? (', Error code: ' + err.code + ', Error: ' + err.message) : ''))
            req.abort();
            // req.abort will trigger 'error' event which invokes callback().  So do not invoke callback() here. Refer to http://stackoverflow.com/questions/6214902/how-to-set-a-timeout-on-a-http-request-in-node
        })
    }

    req.on('response', function(resp){

        let data = '';  // response of a request as a concatenation of chunks
        if(respEncoding && respEncoding === 'raw') data = []
        else if(respEncoding) resp.setEncoding(respEncoding)
        else resp.setEncoding('binary')

        resp.on('data', function(chunk){
            // Concatenate the chunks
            if(Array.isArray(data)) data.push(chunk)
            else data += chunk
        })

        // Once over, send the data through callback
        resp.on('end', function(){
            if (resp.statusCode === 200){
                // 200 means everything went ok
                //console.log('--->getURL: ' + (reqId ? ('Req.Id=' + reqId) : '') +  ', resp.statusCode:' + resp.statusCode)
                // V4.1.28
                if(cbInvoked=== true){
                  /*
                    console.log('--->getURL: ' + (reqId ? ('Req.Id=' + reqId) : '')
                        + ', resp.statusCode:' + resp.statusCode
                        + ', options:' + JSON.stringify(options)
                                    + ', request callback already invoked. Ignoring the response')
                  */
                  logger.warn('--->getURL: ' + (reqId ? ('Req.Id=' + reqId) : '')
                      + ', resp.statusCode:' + resp.statusCode
                      + ', options:' + JSON.stringify(options)
                      + ', request already received response(previous status/error code: '+ statusCode + '). Ignoring this repeat response')
                  return 
                }
                cbInvoked = true
                if(!statusCode) statusCode = resp.statusCode
                // V4.1.28 End    
                if(Array.isArray(data)) return callback(null, Buffer.concat(data), resp)
                return callback(null, data, resp)    // we are not returning result as JSON like {'headers': resp.header, 'data':data} to keep compatibility with plugins that expect only <data> not a JSON obj.
            }
            // error case
            //return callback(new Error('postURL: Failed to get complete data. Data: ' + data), null)            
            let errTmp = new Error('getURL: Failed to get complete data. StatusCode: ' + resp.statusCode + ', Data: ' + (typeof data === 'string' ? data : data.toString()))
            errTmp.statusMessage = data.toString()
            errTmp.code = resp.statusCode
            errTmp.statusCode = resp.statusCode
            //console.log('--->getURL: Response error occured. ' + (reqId ? (', Req.Id=' + reqId) : '') + ', Error code: ' + errTmp.code)            
            return _chkAndRetryGet(errTmp, data, resp)
        })
    })

    // Error handling 
    req.on('error', function(err){
        let errTmp = undefined

        if(err.code === 'ECONNRESET')
            errTmp = new Error('Request timeout(' + httptimeout + ' ms) occurred. ' + err.message)
        else errTmp = err

        errTmp.code = err.code
        //errTmp.statusCode = err.code
        //console.log('--->getURL: request error event occured. ' + (reqId ? (', Req.Id=' + reqId) : '') + ', Error code: ' + errTmp.code)
        return _chkAndRetryGet(errTmp)
        /*
        if(err.code === 'ECONNRESET')
            return callback(new Error('Request timeout(' + httptimeout + ' ms) occurred. Reason: ' + err.message));
        callback(err);
        */
    })
} 
// V4.1.9 End
// V4.1.9 Begin
/**
 * [postURL description]
 * @param  {object | string}  options       http.request options, can be an object or a string
 * @param  {object}           parms         Optional. parameters to be posted. 
 * @param  {object}           retryOptions  Optional. 
 *                                          {
 *                                              'interval': <interval in milliseconds>,
 *                                              'count': <count>,
 *                                              'errorCodes' : <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          }
 * @param  {function}         callback(errr, data)
 */
exports.postURL = function(options, parms, retryOptions, callback){

    let body            = undefined
      , req             = undefined
      , httptimeout     = undefined
      , respEncoding    = undefined 
      , reqId           = undefined
      //, errTmp          = undefined
      , protocol      = undefined
      , cbInvoked     = false   // 4.1.28
      , statusCode      = undefined // 4.1.28

    if(arguments.length === 2){ // postURL = function(options,  callback){
        callback = parms
        parms           = undefined
        retryOptions    = undefined
    }
    else if(arguments.length === 3){ 
        // postURL = function(options,  parms |retryOptions, callback){
        callback = retryOptions
        if(typeof parms === 'object' && parms.count !== undefined && parms.interval !== undefined){
            retryOptions = parms
            parms        = undefined
        }
        else retryOptions = undefined            
    }

    if(logger.isDebugEnabled()) logger.debug('--->postURL: ' + (options.reqId ? ('Req.Id=' + options.reqId +', ') : '') + 'options:' + JSON.stringify(options) +
             //', parms: ' + (parms ?  JSON.stringify(parms) : 'undefined') +  
             ', retryOptions: ' + (retryOptions ?  JSON.stringify(retryOptions) : 'undefined') +  
             ', callback: ' + typeof callback)     

    try{
        if(typeof options === 'string') options = url.parse(options)
        if(!options) return callback(new Error('Unable to parse URL string. options: ' + options))
    }catch(err){
        return callback(new Error('Unable to parse a URL string. options: ' + options + '. ' + err.message))
    }

    if(!options.path) return callback(new Error('postURL: options.path is missing, options: ' + JSON.stringify(options)))
    
    //if(options.encodeURI !== undefined && options.encodeURI === true) options.path = utils.encodeURIData(options.path)

	if(parms && Buffer.isBuffer(parms)) body = parms
    else if(parms && typeof parms === 'object') body = JSON.stringify(parms)
    else if(parms) body = parms + ''  // make parms as string

    function _chkAndRetryPost(err, resData, resp){
          // V4.1.28 Begin
          if(cbInvoked === true){
            /*
          console.log('--->_chkAndRetryPost: ' 
              + (reqId ? ('Req.Id=' + reqId) : '')
              + (err ? (', Error code: ' + err.code + ', Error: ' + err.message) : '')
              + ', options:' + JSON.stringify(options)
              + ', request already received response(status/error code: '+ statusCode + '). Ignoring this repeat response')
              */          
          logger.warn('--->_chkAndRetryPost: ' 
              + (reqId ? ('Req.Id=' + reqId) : '')
              + (err ? (', Error code: ' + err.code + ', Error: ' + err.message) : '')
              + ', options:' + JSON.stringify(options)
              + ', request already received response(previous status/error code: '+ statusCode + '). Ignoring this repeat response')
          return 
        }
        cbInvoked = true 
        if(err && !statusCode) statusCode = err.code || err.statusCode || err.message
        // V4.1.28 End      
        if(!err) return callback(null, resData, resp)

        if(!retryOptions) return callback(err, resData, resp)

        // V5.0.28
        if(retryOptions && retryOptions.skip === true){     // no need to retry.   

            if(logger.isDebugEnabled()) logger.debug('--->postURL: ' + (options.reqId ? ('Req.Id=' + options.reqId +', ') : '') + 'options:' + JSON.stringify(options) +
            ', retryOptions: ' + (retryOptions ?  JSON.stringify(retryOptions) : 'undefined') +  
            ', Skipped retrying the request')     

            return callback(err, resData, resp)
        }
        
        let retryArgs = {
            'reqId':        reqId,
            'respEncoding': respEncoding,
            'functionName': 'postURL',
            'retryOptions': utils.clone(retryOptions),	// 5.0.26
            'options':      options,
            'parms':        parms,
            'err':          err, 
            'resData':      resData,
            'resp':   resp,
            'callback':     callback
        }
        return _retry(retryArgs)
    }

    options.method  = options.method || 'post'
    options.headers = options.headers || {}
    if(body){
        options.headers['Content-Type']     = options.headers['Content-Type']   || 'application/json'
        options.headers['Content-Length']   = options.headers['Content-Length'] || Buffer.byteLength(body)
    }

    httptimeout     = options.timeout || options.httptimeout || undefined
    reqId           = options.reqId
    respEncoding    = options.respEncoding
    options.reqId   = undefined
    protocol        = options.protocol
    options.protocol     = undefined
    options.respEncoding = undefined

    try{
        // Set up the request
        if(protocol && isHttps.test(protocol)) req = https.request(options)
        //if (isHttps.test(options)) 
        else req = http.request(options)
        // Post the data
        if(body) req.write(body)
        req.end()
    }
    catch(err){
        //return callback(err)
        //console.log('--->postURL: request creation error. ' + (reqId ? (', Req.Id=' + reqId) : '') + (err ? (', Error code: ' + err.code + ', Error: ' + err.message) : ''))
        return _chkAndRetryPost(err)
    }

    // Timeout handling
    if(httptimeout){
        req.setTimeout(httptimeout, function(){
            //console.log('--->postURL: request timed out. ' + (reqId ? (', Req.Id=' + reqId) : '') + ', timeout ' + httptimeout )
            req.abort()
            // req.abort will trigger 'error' event which invokes callback(). So do not invoke callback() here. Refer to http://stackoverflow.com/questions/6214902/how-to-set-a-timeout-on-a-http-request-in-node
            //callback(new Error('Request timeout(' + httptimeout + ' ms). occurred.'));
        })
    }

    // Response handling
    req.on('response', function(resp){

        let data = '';  // response of a request as a concatenation of chunks
        if(respEncoding && respEncoding === 'raw') data = []
        else if(respEncoding) resp.setEncoding(respEncoding)
        else resp.setEncoding('binary')

        resp.on('data', function(chunk){
            // Concatenate the chunks
            if(Array.isArray(data)) data.push(chunk)
            else data += chunk
        })

        // Once over, send the data through callback
        resp.on('end', function(){
            if (resp.statusCode === 200){
                // 200 means everything went ok
                //console.log('--->postURL: ' + (reqId ? ('Req.Id=' + reqId) : '') +  ', resp.statusCode:' + resp.statusCode)
                // V4.1.28
                if(cbInvoked=== true){
                  /*
                  console.log('--->postURL: ' + (reqId ? ('Req.Id=' + reqId) : '')
                      + ', resp.statusCode:' + resp.statusCode
                      + ', options:' + JSON.stringify(options)
                                  + ', request callback already invoked. Ignoring the response')
                  */
                  logger.warn('--->postURL: ' + (reqId ? ('Req.Id=' + reqId) : '')
                      + ', resp.statusCode:' + resp.statusCode
                      + ', options:' + JSON.stringify(options)
                      + ', request already received response(previous status/error code: '+ statusCode + '). Ignoring this repeat response')
                  return 
                }
                cbInvoked = true
                if(!statusCode) statusCode = resp.statusCode
                // V4.1.28 End                
                if(Array.isArray(data)) return callback(null, Buffer.concat(data), resp)
                return callback(null, data, resp)    // we are not returning result as JSON like {'headers': resp.header, 'data':data} to keep compatibility with plugins that expect only <data> not a JSON obj.
            }
            // error case
            //return callback(new Error('postURL: Failed to get complete data. Data: ' + data), null)            
            let errTmp = new Error('postURL: Failed to get complete data.  StatusCode: ' + resp.statusCode + ', Data: ' + (typeof data === 'string' ? data : data.toString()))
            errTmp.code         = resp.statusCode
            errTmp.statusCode   = resp.statusCode
            errTmp.statusMessage = data.toString()
            return _chkAndRetryPost(errTmp, data, resp)
        })
    })
  
      // Error handling 
    req.on('error', function(err){
        let errTmp = undefined

        /*
        if(err.code === 'ECONNRESET')
            return callback(new Error('Request timeout(' + httptimeout + ' ms) occurred. ' + err.message))
        callback(err)
        */
        if(err.code === 'ECONNRESET')
            errTmp = new Error('Request timeout(' + httptimeout + ' ms) occurred. ' + err.message)
        else errTmp = err
   
        errTmp.code = err.code
        //errTmp.statusCode = err.code        
        //console.log('--->postURL: request error event occured. ' + (reqId ? (', Req.Id=' + reqId) : '') + ', Error code: ' + errTmp.code)
       return _chkAndRetryPost(errTmp)
    })  
} 

// V4.1.9 End
exports.putURL = function(options, parms, retryOptions, callback){
    if(options && typeof options === 'object') options.method  = 'put'
    exports.processURL(options, parms, retryOptions, callback)  // V5.0.7
}

// V5.0.7 Begin
exports.deleteURL = function(options, parms, retryOptions, callback){
  if(options && typeof options === 'object') options.method  = 'delete'
  exports.processURL(options, parms, retryOptions, callback)
}
// V5.0.7 End

/**
 * Service to setup a http(s) Request
 * @param  {object | string}  options  http.request options, can be:
 *                                       protocol: Protocol to use. Defaults to 'http:'.
 *                                       hostname: To support url.parse() hostname is preferred over host
 *                                       port: Port of remote server. Defaults to 80.
 *                                       method: A string specifying the HTTP request method (GET,POST,PUT,HEAD); defaults to 'GET'.
 *                                       path: Request path. Defaults to '/'. 
 *                                       headers: An object containing request headers.
 *                                       ...and more, see https://nodejs.org/docs/latest-v0.12.x/api/http.html#http_http_request_options_callback
 * @param  {JSON}             parms    parameters to be posted. Must be JSON
 * @param  {function}         callback(err, result)
 *
 *
 * TODO: Why parms must be JSON ?
 * TODO: How to pass the timeout value ?
 * TODO: Fix resp.setEncoding('binary'); // !!! This should be passed from the Request. Default could be binary
 */
// V4.1.9 Begin
// TODO
exports.processURLNew = function(options, parms, callback){    // todo

    if(typeof options === 'string' 
        || (typeof options === 'object' && options.method === undefined)
        || (typeof options === 'object' && options.method && options.method.toLowerCase() === 'get'))
        return exports.getURL(options, callback)
    return  exports.postURL(options, parms, callback)    
}
// V4.1.9 End

exports.processURLOld = function(options, parms, callback){    // !!! will be OBSOLETE. use getURL() or postURL()

  var body        = undefined
    , req         = null
    , httptimeout = undefined
    ;

  // Any parm specified?
  if (typeof(parms) === "function"){
    callback = parms;
    parms = undefined;
  }

  // If options is a string, parse it (for ex.: http://localhost:84/services/version.html)
  if (typeof options === 'string') options = url.parse(options);

  // Is options specified?
  if(options === undefined){  // no go!
    return callback(new Error('processURL: options is undefined'));
  }

  // Set get as default method
  options.method === options.method || 'get';

  // Lowercase the method name
  options.method = options.method.toLowerCase();

  // Add Content-Type as json if parms is specified
  if(parms){
    body = JSON.stringify(parms);
    options.headers = options.headers || {'Content-Type':'application/json','Content-Length': Buffer.byteLength(body)};
  }

  try{
    // Set up the request
    if(isHttps.test(options.protocol)) req = https.request(options);   
    else req = http.request(options);   

    // Write the body if specified
    if(body) req.write(body);

    // Flush the request
    req.end();
  }
  catch(err){
    return callback(err);
  }

  if(options.httptimeout) httptimeout = options.httptimeout;
  // Timeout handling
  if(httptimeout){
    req.setTimeout(httptimeout, function(){
        req.abort();
        /*
           req.abort will trigger 'error' event which invokes callback(). 
           So do not invoke callback() here
           Refer to http://stackoverflow.com/questions/6214902/how-to-set-a-timeout-on-a-http-request-in-node
         */
    });
  }

  // Response handling
  req.on('response', function(resp){
    var data = '';  // response of a request as a concatenation of chunks
    resp.setEncoding('binary'); // !!! This should be passed from the Request. Default could be binary

    resp.on('data', function(chunk){
      data += chunk; // concatenate the chunks
    });

    // Once over, send the data through callback
    resp.on('end', function(){
      // 200 means everything went ok
      if (resp.statusCode === 200) return callback(null, data);
      return callback(new Error('processURL: Failed to get complete data. Data: ' + data));
    }); 
  });
  
  // Error handling 
  req.on('error', function(err){
    if(err.code === 'ECONNRESET')
      return callback(new Error('Request timeout(' + httptimeout + ' ms) occurred. Reason: ' + err.message));
    callback(err);
  });  
} 

/**
 * uploadFileAsBuffer upload of a buffer
 * @param  {json}     options
 *                    options ={
 *                      hostname:     "<hostname>",
 *                      [port:        "<port>",]
 *                      method:       "<method>, POST or PUT"
 *                      [path:        "<URI>"],
 *                      [httptimeout:        "<Request timeout in milliseconds>"],
 *                    }
 * @param  {Buffer}   body            "<buffer to be uploaded>",
 * @param  {Function} callback
 */
exports.uploadFileAsBuffer = function(options, body, callback){     // !!! will be OBSOLETE. use postURL()

  var req         = undefined
    , httptimeout = undefined

  // Is options specified?
  if(!options){  // no go!
    return callback(new Error('uploadFileAsBuffer: options is missing'))
  }
   
  // If options is a string, parse it (for ex.: http://localhost:1029/services/oxsnps-.../version)
  if (typeof options === 'string'){
    options = url.parse(options, true, true);
  }
  else{
    if(!options.path) 
      return callback(new Error('uploadFileAsBuffer: options.path is missing, options: ' + JSON.stringify(options)))

    // Encode Path (i.e. escape special chars like space)
    options.path = encodeURI(options.path);
  }

  // Add the content length, if not specified
  if(!options.headers){
      options.headers = {
      'Content-Length': Buffer.byteLength(body)
    }
  }
  else if(!options.headers['Content-Length']){
    options.headers['Content-Length'] = Buffer.byteLength(body)
  }

  try{
    // Set up the request
    if(isHttps.test(options)) req = https.request(options);   
    else req = http.request(options);   
   
    // Post the data
    req.write(body);
    req.end();
  }
  catch(err){
    return callback(err);
  }

  if(options !== null && typeof options === 'object' && options.httptimeout) httptimeout = options.httptimeout;

  // Timeout handling
  if(httptimeout){
    req.setTimeout(httptimeout, function(){
        req.abort();
        // req.abort will trigger 'error' event which invokes callback(). 
        // So do not invoke callback() here
        // Refer to http://stackoverflow.com/questions/6214902/how-to-set-a-timeout-on-a-http-request-in-node
        //callback(new Error('Request timeout(' + httptimeout + ' ms). occurred.'));
    });
  }

  // Response handling
  req.on('response', function(resp){
    var data = '';  // response of a request as a concatenation of chunks
    //!!!resp.setEncoding('binary');

    resp.on('data', function(chunk){
      // Concatenate the chunks
      data += chunk;
    });

    // Once over, send the data through callback
    resp.on('end', function(){
      if(resp.statusCode === 200) return callback(null, data); // 200 means everything went ok
      return callback(new Error('postURL: Failed to upload buffer. Data: ' + data), resp);
    }); 
  });
  
  // Error handling 
  req.on('error', function(err){
    if(err.code === 'ECONNRESET')
      return callback(new Error('Request timeout(' + httptimeout + ' ms) occurred. Reason: ' + err.message));
    callback(err);
  });
} 

exports._uploadFile = function(options, parms, callback){

  var formData = {}
    , url = 'http://' + options.hostname + (options.port ? ':' + options.port : '') + (options.path ? options.path : '/');
  //console.log('uploadFile: url: ' + url + ',  formData:' + JSON.stringify(formData));

  // Assert buffer
  if(!parms.buffer){
    return callback(new Error('parms.buffer undefined.'));
  }

  // Assert parms
  parms.filename    = parms.filename || 'unknown_file';
  parms.contentType = parms.contentType || 'application/pdf';

  if(options.method === 'POST'){
    // Post the request
    var req = request.post({'url':url, 'formData': formData}, function(err, httpResponse, data) {
      if (err) {
        return callback(err);
      }
      //console.log('uploadFile: Upload successful! Server responded:', data);
      return callback(null, data);
    });

    // Add buffer to the form data (see http://stackoverflow.com/questions/25344879/uploading-file-using-post-request-in-node-js)
    var form = req.form();
    form.append('file', new Buffer(new Uint8Array(parms.buffer)), {filename: parms.filename, contentType: parms.contentType});  
  }
  else if(options.method === 'PUT'){
    // Put the request
    var req = request.put({'url':url, 'formData': formData}, function(err, httpResponse, data) {
      if (err) {
        return callback(err);
      }
      //console.log('uploadFile: Upload successful! Server responded:', data);
      return callback(null, data);
    });

    // Add buffer to the form data (see http://stackoverflow.com/questions/25344879/uploading-file-using-post-request-in-node-js)
    var form = req.form();
    form.append('file', new Buffer(new Uint8Array(parms.buffer)), {filename: parms.filename, contentType: parms.contentType});  
  }
  else{
    return callback(new Error('HTTP method undefined: ' + options.method));
  }
}

/**
 * !!! NOT COMPLETED !!! has to be checked before being used
 * 
 * postMultipartRequest 
 * @param  {json}       props    
 *                      props = {
 *                        filename:     "<filename>",
 *                        buffer:       "<file buffer>",
 *                        contentType:  "application/pdf",
 *                        encoding:     "binary",
 *                        hostname:     "<hostname>",
 *                        port:         "<port>",
 *                        method:       "<method>"
 *                        path:         "<URI>"
 *                      }
 * @param  {Function}   callback
 */
exports.postMultipartRequest = function(props, callback){

  // Build the POST Req Parameters
  var crlf = "\r\n",
    //boundary = '---------------------------9r-SzFGlBRvRAc8RJbwAJ4i-Idwoujo2F-EPQ6v', // Boundary: "--" + up to 70 ASCII chars + "\r\n"
    min = 100000000000000,
    max = 1000000000000000,
    //boundary = '---------------------------' + Math.floor(Math.random() * (max - min + 1) + min),
    boundary = 'oswhTEndxHsTZddqkYwAg6-sbh6HP0e5R',
    delimiter = crlf + "--" + boundary,
    preamble = "", // ignored. a good place for non-standard mime info
    epilogue = "", // ignored. a good place to place a checksum, etc
    
    bufferHeader = [
      'Content-Disposition: form-data; name="InputBuffer"; filename="' + props.filename + '"' + crlf,
      //'Content-Type: application/octet-stream' + crlf,
      'Content-Type: ' + props.contentType + crlf,
      'Content-Transfer-Encoding: ' + props.encoding + crlf,
    ],
    
    optionHeader1 = [
      'Content-Disposition: form-data; name='
    ],
    optionHeader2 = [
      'Content-Type: text/plain; charset=US-ASCII' + crlf,
      'Content-Transfer-Encoding: 8bit' + crlf,
    ],            
    closeDelimiter = delimiter + "--" + crlf, // crlf is VERY IMPORTANT here
    
  // Here is how the multipartBody MUST be built 
  // multipartBody  = preamble
  //          + 1..n[delimiter + crlf + optionHeader1 + optionName + crlf + optionHeader2 + crlf + optionValue]
  //          + delimiter + crlf + bufferHeader + crlf + InputBuffer
  //          + closeDelimiter + epilogue
  // Concatenate preamble
  multipartBody = Buffer.concat([
    new Buffer(preamble)
  ]);
      
  // Concatenate InputBuffer...if provided
  if (props.buffer){
    multipartBody = Buffer.concat([
      multipartBody,
      new Buffer(delimiter + crlf + bufferHeader.join('') + crlf),
      props.buffer
    ]);
  }

  // Concatenate closeDelimiter + epilogue
  multipartBody = Buffer.concat([
    multipartBody,
    new Buffer(closeDelimiter + epilogue)
  ]); 
          
  // Set the POST Req Options
  var post_options = {
    hostname: props.hostname,
    port:     props.port,
    path:     props.path,
    method:   props.method,
    headers:  {
      'Content-Type': 'multipart/form-data; boundary=' + boundary,
      'Content-Length': multipartBody.length
    }
  };


  // Set up the request
  var post_req = http.request(post_options);

  // Post the data
  post_req.write(multipartBody);
  post_req.end();

  // V101 Begin
  if( props.path.indexOf( 'async') >= 0 ){
    // If it is a async request, do not wait for response and return immediately 
    return callback(null);
  }
  // V101 End
  else{
    // Response handling
    post_req.on('response', function(res) {

      var response = '';        // response of a request as a concatenation of chunks
      res.setEncoding('binary');
      res.on('data', function (chunk) {
        // Concatenate the chunks
        response += chunk;
      });

      // Call the callback function passing the response
      res.on('end', function () {
        if (res.statusCode === 200){ // HTTP status code (200, 404, 500, ...), 200 means everything went ok
          return callback(null, response);
        }
        else{
          return callback(response);
        }
      }); 
    });
    
    // Error handling 
    post_req.on('error', function(error) {
      return callback(error);
    }); 
  }         
}

/**
 * Get request parameters
 * @param  {Request}  	req Request Object
 * @return  JSON 	parms  Parameter json with key/value pairs
 */
exports.getRequestParms = function(req){

	let  parms 		= {}
	  , contenttype = undefined 

	if(!req) return parms

	if(req.headers) contenttype = req.headers['content-type']
	
	// Get Query parameters if any, and add to req.parms
	if(req.query){
		Object.keys(req.query).forEach(function(key) {
			parms[key] = req.query[key]
		})
	}
	// Get parameters from req.params if any, and add to req.parms
	if(req.params){
		Object.keys(req.params).forEach(function(key) {
			parms[key] = req.params[key]
		})
	}	
	// Get parameters from body if any, and add to req.parms
	if(req.method && req.method.toLowerCase() === 'post' && req.body){
		if(contenttype && contenttype.toLowerCase().indexOf('xml')>=0) // 'text/xml', 'application/xml', '*/xml'
      parms.xml = req.body
    // V5.0.19 Begin
    else if(contenttype && contenttype.toLowerCase().indexOf('text')>=0) // 'text/*', 'text/plain', '*/xml'
      parms.text = req.body
    // V5.0.19 End
    else if(contenttype && contenttype.toLowerCase().indexOf('json')>=0){ // 'application/json', 'text/plain', '*/xml'            
			Object.keys(req.body).forEach(function(key) {
				parms[key] = req.body[key]
			})
    }
    else parms.text = req.body  // V5.0.19
	}
	return parms
}

// V4.1.9 Begin
/**
 * retry the http request
 * @return  JSON 	retryArgs  retry Arguments
    retryArgs = {
            'reqId':        reqId,
            'respEncoding': respEncoding,
            'functionName': 'getURL',
            'retryOptions': {
                'interval': <interval in milliseconds>,
                'count': <count>,
                'errorCodes' : <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
            },
            'options':      <request options>,
            'err':          err, 
            'callback':     callback
        }
 */
function _retry(retryArgs){

    let reqId           = retryArgs.reqId
      , respEncoding    = retryArgs.respEncoding
      , functionName    = retryArgs.functionName
      , retryOptions    = retryArgs.retryOptions
      , options         = retryArgs.options
      , parms           = retryArgs.parms
      , err             = retryArgs.err
      , resData         = retryArgs.resData
      , resp            = retryArgs.resp
      , callback        = retryArgs.callback
      , retryForAllErrors     = false   // v5.0.25

    if(!err) return callback(null, resData, resp)

    if(!retryOptions) return callback(err, resData, resp)

    if(retryOptions && retryOptions.skip === true) return callback(err, resData, resp)  // no need to retry.   // V5.0.28

    options.reqId = reqId
    options.respEncoding = respEncoding
    
    if(retryOptions.errorCodes && retryOptions.errorCodes.length === 1 && retryOptions.errorCodes[0].trim().toLowerCase() === 'all') retryForAllErrors = true   // v5.0.25

    if(retryOptions.attemptCount === undefined) retryOptions.attemptCount = 1	// 5.0.26
        
    if(retryForAllErrors === false){    // v5.0.25
        if(!retryOptions.errorCodes || !retryOptions.errorCodes.includes(err.code)){
            logger.warn('--->_retry: ' + 
                (reqId ? ('Req.Id=' + reqId +', ') : '') +
                (err.code === undefined ? '': ('Error code: ' + err.code)+ ', ') + 
                'Error: ' + err.message + 
                ', options:' + JSON.stringify(options) + 
                ', request not scheduled again since the error code(' + err.code + ') is not in retry error codes list')
            return callback(err, resData, resp)
        }
    }

    retryOptions.interval   = retryOptions.interval || 1000
    //let retryOptionsCopy = utils.clone(retryOptions)      
    //retryOptionsCopy.orgCount = retryOptions.orgCount || retryOptions.count
    //-- retryOptionsCopy.count
	
	if(retryOptions.count !== undefined && retryOptions.attemptCount >= retryOptions.count){    // 5.0.26 //retryOptions.count is set to undefiend by the caller to retry till we succeed
    //if(retryOptionsCopy.count <=0){
        logger.error('--->_retry: ' + 
            (reqId ? ('Req.Id=' + reqId +', ') : '') +
            (err.code === undefined ? '': ('Error code: ' + err.code)+', ') + 
            'Error: ' + err.message +
            ', options:' + JSON.stringify(options) + 
			', Number of times retried: ' + retryOptions.attemptCount + ', interval: '+ retryOptions.interval +' ms.')			
		//  ', Number of times retried: ' + retryOptionsCopy.orgCount + ', interval: '+ retryOptions.interval +' ms.')
        //return callback(new Error(err.message + ', Number of times tried: ' + retryOptionsCopy.orgCount + ', interval: '+ retryOptions.interval +' milliseconds.'))
        return callback(err, resData, resp)
    }

    // retry the request
    if(retryOptions.count === undefined){
        // V5.0.29
        // for endless retrying log warning with error message and options
        logger.warn('--->_retry: ' + (reqId ? ('Req.Id=' + reqId +', ') : '') +
                (err.code === undefined ? '': ('Error code: ' + err.code +', ')) + 'Error: ' + err.message + ', ' +
                (retryOptions.attemptCount === 1 ? 'options:' + JSON.stringify(options) : ('hostname: ' + options.hostname + ', path: ' + options.path)) + ', ' +
                //', retryOptions.errorCodes:' + JSON.stringify(retryOptions.errorCodes) + 
                //', Retry Attempt:'  + ((retryOptionsCopy.orgCount - retryOptionsCopy.count)+1) + ', scheduled again after ' + retryOptions.interval + ' ms.')
                'Retry Attempt:'  + (retryOptions.attemptCount +1) + ', scheduled again after ' + retryOptions.interval + ' ms.')
    }
    else{
      // V5.0.29
        logger.warn('--->_retry: ' + (reqId ? ('Req.Id=' + reqId +', ') : '') +
                (err.code === undefined ? '': ('Error code: ' + err.code +', ')) + // ' Error: ' + err.message +
               // (retryOptions.attemptCount === 1 ? ' Error: ' + err.message : '') + 
                (retryOptions.attemptCount === 1 ? 'options:' + JSON.stringify(options) : ', hostname: ' + options.hostname + ', path: ' + options.path) + ', ' +
                //', retryOptions.errorCodes:' + JSON.stringify(retryOptions.errorCodes) + 
                //', Retry Attempt:'  + ((retryOptionsCopy.orgCount - retryOptionsCopy.count)+1) + ', scheduled again after ' + retryOptions.interval + ' ms.')
                'Retry Attempt:'  + (retryOptions.attemptCount +1) + ', scheduled again after ' + retryOptions.interval + ' ms.')

    }
	  retryOptions.attemptCount += 1	// 5.0.26
    setTimeout(
        function(){
            if(parms) return exports[functionName](options, parms, utils.clone(retryOptions), callback)
            return exports[functionName](options, utils.clone(retryOptions), callback)
        },
        retryOptions.interval
    )        
}     
// V4.1.9 Begin

// V5.0.7 Begin
/**
 * Process HTTP URL
 * @param  {object | string}  options       http.request options, can be an object or a string
 * @param  {object}           parms         Optional. parameters to be posted. 
 * @param  {object}           retryOptions  Optional. 
 *                                          {
 *                                              'interval': <interval in milliseconds>,
 *                                              'count': <count>,
 *                                              'errorCodes' : <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          }
 * @param  {function}         callback(errr, data)
 */
exports.processURL = function(options, parms, retryOptions, callback){

  let body            = undefined
    , req             = undefined
    , httptimeout     = undefined
    , respEncoding    = undefined 
    , reqId           = undefined
    //, errTmp          = undefined
    , protocol      = undefined
    , cbInvoked     = false   // 4.1.28
    , statusCode      = undefined // 4.1.28

  if(arguments.length === 2){ // processURL = function(options,  callback){
      callback = parms
      parms           = undefined
      retryOptions    = undefined
  }
  else if(arguments.length === 3){ 
      // processURL = function(options,  parms |retryOptions, callback){
      callback = retryOptions
      if(typeof parms === 'object' && parms.count !== undefined && parms.interval !== undefined){
          retryOptions = parms
          parms        = undefined
      }
      else retryOptions = undefined            
  }

  if(logger.isDebugEnabled()) logger.debug('--->processURL: ' + (options.reqId ? ('Req.Id=' + options.reqId +', ') : '') + 'options:' + JSON.stringify(options) +
           //', parms: ' + (parms ?  JSON.stringify(parms) : 'undefined') +  
           ', retryOptions: ' + (retryOptions ?  JSON.stringify(retryOptions) : 'undefined') +  
           ', callback: ' + typeof callback)     

  try{
      if(typeof options === 'string') options = url.parse(options)
      if(!options) return callback(new Error('Unable to parse URL string. options: ' + options))
  }catch(err){
      return callback(new Error('Unable to parse a URL string. options: ' + options + '. ' + err.message))
  }

  if(!options.path) return callback(new Error('--->processURL: options.path is missing, options: ' + JSON.stringify(options)))
  
  //if(options.encodeURI !== undefined && options.encodeURI === true) options.path = utils.encodeURIData(options.path)

if(parms && Buffer.isBuffer(parms)) body = parms
  else if(parms && typeof parms === 'object') body = JSON.stringify(parms)
  else if(parms) body = parms + ''  // make parms as string

  function _chkAndRetry(err, resData, resp){
          // V4.1.28 Begin
          if(cbInvoked === true){
            /*
          console.log('--->_chkAndRetry: ' 
              + (reqId ? ('Req.Id=' + reqId) : '')
              + (err ? (', Error code: ' + err.code + ', Error: ' + err.message) : '')
              + ', options:' + JSON.stringify(options)
              + ', request already received response(status/error code: '+ statusCode + '). Ignoring this repeat response')
              */          
          logger.warn('--->_chkAndRetry: ' 
              + (reqId ? ('Req.Id=' + reqId) : '')
              + (err ? (', Error code: ' + err.code + ', Error: ' + err.message) : '')
              + ', options:' + JSON.stringify(options)
              + ', request already received response(previous status/error code: '+ statusCode + '). Ignoring this repeat response')
          return 
        }
        cbInvoked = true 
        if(err && !statusCode) statusCode = err.code || err.statusCode || err.message
        // V4.1.28 End          
      if(!err) return callback(null, resData, resp)

      if(!retryOptions) return callback(err, resData, resp)
      
      // V5.0.28
      if(retryOptions && retryOptions.skip === true){   // no need to retry.   
        if(logger.isDebugEnabled()) logger.debug('--->processURL: ' + (options.reqId ? ('Req.Id=' + options.reqId +', ') : '') + 'options:' + JSON.stringify(options) +
        ', retryOptions: ' + (retryOptions ?  JSON.stringify(retryOptions) : 'undefined') +  
        ', Skipped retrying the request')

          return callback(err, resData, resp)
      }

      let retryArgs = {
          'reqId':        reqId,
          'respEncoding': respEncoding,
          'functionName': 'processURL',
          'retryOptions': retryOptions,
          'options':      options,
          'parms':        parms,
          'err':          err, 
          'resData':      resData,
          'resp':   resp,
          'callback':     callback
      }
      return _retry(retryArgs)
  }

  options.method  = options.method || 'post'
  options.headers = options.headers || {}
  if(body){
      options.headers['Content-Type']     = options.headers['Content-Type']   || 'application/json'
      options.headers['Content-Length']   = options.headers['Content-Length'] || Buffer.byteLength(body)
  }

  httptimeout     = options.timeout || options.httptimeout || undefined
  reqId           = options.reqId
  respEncoding    = options.respEncoding
  options.reqId   = undefined
  protocol        = options.protocol
  options.protocol     = undefined
  options.respEncoding = undefined

  try{
      // Set up the request
      if(protocol && isHttps.test(protocol)) req = https.request(options)
      //if (isHttps.test(options)) 
      else req = http.request(options)
      // Post the data
      if(body) req.write(body)
      req.end()
  }
  catch(err){
      //return callback(err)
      //console.log('--->postURL: request creation error. ' + (reqId ? (', Req.Id=' + reqId) : '') + (err ? (', Error code: ' + err.code + ', Error: ' + err.message) : ''))
      return _chkAndRetry(err)
  }

  // Timeout handling
  if(httptimeout){
      req.setTimeout(httptimeout, function(){
          //console.log('--->postURL: request timed out. ' + (reqId ? (', Req.Id=' + reqId) : '') + ', timeout ' + httptimeout )
          req.abort()
          // req.abort will trigger 'error' event which invokes callback(). So do not invoke callback() here. Refer to http://stackoverflow.com/questions/6214902/how-to-set-a-timeout-on-a-http-request-in-node
          //callback(new Error('Request timeout(' + httptimeout + ' ms). occurred.'));
      })
  }

  // Response handling
  req.on('response', function(resp){

      let data = '';  // response of a request as a concatenation of chunks
      if(respEncoding && respEncoding === 'raw') data = []
      else if(respEncoding) resp.setEncoding(respEncoding)
      else resp.setEncoding('binary')

      resp.on('data', function(chunk){
          // Concatenate the chunks
          if(Array.isArray(data)) data.push(chunk)
          else data += chunk
      })

      // Once over, send the data through callback
      resp.on('end', function(){
          if (resp.statusCode === 200){
                // 200 means everything went ok
                //console.log('--->postURL: ' + (reqId ? ('Req.Id=' + reqId) : '') +  ', resp.statusCode:' + resp.statusCode)
                // V4.1.28
                if(cbInvoked=== true){
                  /*
                  console.log('--->postURL: ' + (reqId ? ('Req.Id=' + reqId) : '')
                      + ', resp.statusCode:' + resp.statusCode
                      + ', options:' + JSON.stringify(options)
                                  + ', request callback already invoked. Ignoring the response')
                  */
                  logger.warn('--->postURL: ' + (reqId ? ('Req.Id=' + reqId) : '')
                      + ', resp.statusCode:' + resp.statusCode
                      + ', options:' + JSON.stringify(options)
                      + ', request already received response(previous status/error code: '+ statusCode + '). Ignoring this repeat response')
                  return 
                }
                cbInvoked = true
                if(!statusCode) statusCode = resp.statusCode
                // V4.1.28 End                
              if(Array.isArray(data)) return callback(null, Buffer.concat(data), resp)
              return callback(null, data, resp)    // we are not returning result as JSON like {'headers': resp.header, 'data':data} to keep compatibility with plugins that expect only <data> not a JSON obj.
          }
          // error case
          //return callback(new Error('processURL: Failed to get complete data. Data: ' + data), null)            
          let errTmp = new Error('--->processURL: Failed to get complete data.  StatusCode: ' + resp.statusCode + ', Data: ' + (typeof data === 'string' ? data : data.toString()))
          errTmp.code         = resp.statusCode
          errTmp.statusCode   = resp.statusCode
          errTmp.statusMessage = data.toString()
          return _chkAndRetry(errTmp, data, resp)
      })
  })

    // Error handling 
  req.on('error', function(err){
      let errTmp = undefined
      /*
      if(err.code === 'ECONNRESET')
          return callback(new Error('Request timeout(' + httptimeout + ' ms) occurred. ' + err.message))
      callback(err)
      */
      if(err.code === 'ECONNRESET')
          errTmp = new Error('Request timeout(' + httptimeout + ' ms) occurred. ' + err.message)
      else errTmp = err
 
      errTmp.code = err.code
      //errTmp.statusCode = err.code
      //console.log('--->postURL: request error event occured. ' + (reqId ? (', Req.Id=' + reqId) : '') + ', Error code: ' + errTmp.code)
     return _chkAndRetry(errTmp)
  })  
} 
// V5.0.7 End