/**-----------------------------------------------------------------------------
 * utils.js:  Node.js module that provides utility functions
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2021 by Maas Holding GmbH
 * Email     :  support@maas.de
 * 
 * History
 *  V100   18.09.2014  Initial release
 *  V101   30.09.2014  getFileList refactored to handle sub directories recursively
 *  V102   16.10.2014  Added getAttrTextFromHTML() method
 *  V103   06.11.2014  Added checkAndCreateDir() and parseJSON() methods
 *  V104   07.11.2014  Added mergeJSON() method
 *  V105   13.11.2014  Added symlink() method
 *  V106   05.01.2015  Added deleteDir() methods using async methods
 *  V107   09.01.2015  Added getDocId() and getDocIdFromRegExp() functions
 *  V108   22.01.2015  Extended to have configuration to turn on/off caching
 *  V109   28.01.2015  Added getOutputJSONFilename() method
 *  V110   19.10.2015  Replace ('blueimp-md5').md5 module with 'crypto-md5' module, 
 *                     since ('blueimp-md5').md5 gives following error when used for big files(>20 KB) 
 *                     ................
 *                     /opt/test/node_modules/blueimp-md5/js/md5.js:235
 *                       return unescape(encodeURIComponent(input));
 *                       ^
 *                     URIError: URI malformed
 *                     at encodeURIComponent (native)
 *                     at str2rstr_utf8 (/opt/test/node_modules/blueimp-md5/js/md5.js:235:25)
 *                     ................
 *                     Note: MD5 Id will not change even if we change the module used to generate MD5 Id
 * 
 * V3.0.24 19.07.2018  (OXS-8448) Extended getDocId() to use value of given list of parameters in the query string of input.url, 
 *                                to generate unique id along with host,post and service path in input.url parameter
 * V3.0.32 14.11.2018  utils.js: Extended mergeJSON() function to use "deepmerge" module for merging JSON objects
 * V3.0.33 02.03.2020  mkdirp interface changed from v0.5.1 to v1.0.3. mkdirp does not use anymore callback but promise
 * V3.0.35 29.01.2018  utils.js: Added checkFileAccess function to check accessibility of a file
 * V3.0.35-1 12.03.2020	OTS-2710: Extended deleteDir() function  to delete the files and directories one by one and not in parallel to avoid taking more heap memory
 * V4.1.8 		01.07.2020	OTS-2753 oxsnps-core/utils.js: Extend utils.getMountInfoSync() and utils.getHostsSync() to support Windows OS 
 * V3.0.35-2 28.08.2020	    OTS-2889: 1. utils.deleteDir() deletes EMPTY directories without checking if it is older than the given time. This could lead to deleting temp. directories created for currently running requests that have not written any file when cron job is running. Extended to check the time when deleting the directories.
 *                                    2. If a directory creation time is NOT older than configured time, do not go down the directory (that is skip that directory and its sub-directories and files from deletion)
 * V4.0.11	 12.10.2020	    OTS-2906: Extended to get username and password to get authorization token to use it with configuration server/ordersystem APIs from technical user JS file specified in OXSECO_TECH_USER env. variable
 * ....
 *  Refer to ./server.js for history of V4.1.17 onwards
 *----------------------------------------------------------------------------*/
'use strict';

var MODULE_NAME = 'utils'
  , path        = require('path')
  , fs          = require('fs')
  , fse         = require('fs-extra')
  , chardet 	= require('chardet')  
  , oldMask     = process.umask(0)  // oldMask not used anywhere, but this statement is needed for mode parameter passed to utils.mkdir() function
  , date_format = require('./date_format')
  , log4js      = require('log4js')
  // , cheerio     = require('cheerio')            // to parse HTML  
  , mkdirp      = require('mkdirp')  
  , async       = require('async')
  , deepmerge   = require('deepmerge')
  , md5         = require('crypto-md5')
  , url         = require('url')
  , os          = require('os')
  , xml2js      = require('xml2js')         // to convert xml to json and vice versa      
  , cp          = require('child_process')
  , rimraf      = require('rimraf')
  //, request     = require('request')    // 4.1.17
  , underscore  = require('underscore')
  , mv          = require('mv')
  , FILE_SEP    = require('path').sep  
  // , detectCharacterEncoding = require('detect-character-encoding')   // commented and module removed from oxsnps-core sicne it takes long time to build
  , iconv       = require('iconv-lite')
  , CRLF_Tabs   = '\n\t\t\t\t\t\t\t'
  , hex2utf8    = require(path.resolve(__dirname + '/../../../conf/hex2utf8.js'))

// Get Logger
var logger = log4js.getLogger(MODULE_NAME);
//!!!_log4jsSetLogLevel(logger, (_getAppLogLevel(npsConf.log) || 'INFO')) //!!! not needed since server.js calls utils.log4jsSetLogLevel

/*
  Delete dirs if their date is older than (today's date + interval)
    dirs:     array of dirs
    interval: if the dir date is older than (today's date + interval (in seconds)) then delete the dir.
*/
exports.deleteDirs = function(dirs, interval){

  //logger.debug(__filename, 'deleteDirs: ' + dirs.length + ' temp dir(s) queued to be deleted...');

  if (dirs.length <= 0){
    return; // over since array is empty
  }

  // Delete the dir and its content if the dir date is older than (today's date + interval)
  _deleteDir(dirs, new Date(), interval);
}

/**
 * Delete recursively invalid Soft links SYNC 
 * @param  {String} dir
 * @return {void}
 */
exports.deleteInvalidSoftLinksSync = function(dir){

  if(fs.existsSync(dir)){
    fs.readdirSync(dir).forEach(function(file, index){
      var curDir = dir + "/" + file;

      if (!fs.existsSync(curDir)){
        // If the Soft Link is invalid, delete it!
        try{
          // If the Soft Link maps to a non existing dir, it will be deleted else an exception will be thrown
          fs.unlinkSync(curDir); 
        }
        catch(err){
          // Ignore the exception that will be thrown if the soft link is invalid
        }
      }
      else if(fs.lstatSync(curDir).isDirectory()){ 
        deleteInvalidSoftLinksSync(curDir); // recurse
      }
    });
  }
}

/**
 * Delete recursively invalid Soft links
 * Based on the very nicely structured https://github.com/AvianFlu/ncp/blob/master/lib/ncp.js
 * @param  {string}     dir
 * @param  {function}   callback(err)
 */
exports.deleteInvalidSoftLinks = function(dir, callback){

  var errs      = null
    , started   = 0
    , finished  = 0
    , running   = 0 // indicates how many times process has been called recursively
    ;

  // Assert Callback function
  callback = (typeof callback === 'function') ? callback : function(){};
  
  process(dir);

  function process(source) {
    
    started++;
    running++;
    fs.lstat(source, function (err, stats) {
      if (err) {
        return onError(err);
      }
      if (stats.isDirectory()) {
        return onDir(source);
      }
      else if (stats.isFile()) {
        return onFile(source);
      }
      else if (stats.isSymbolicLink()) {
        return onLink(source);
      }
    });
  }

  function onFile(file){
    // Ignore files
    //console.log('Processing file ' + file);
    return cb();
  }
  
  function onDir(dir){
    //console.log('Processing dir  ' + dir);
    fs.readdir(dir, function (err, items){
      if(err){
        return onError(err);
      }
      items.forEach(function(item){
        return process(dir + '/' + item);
      });
      return cb();
    });
  }

  function onLink(link){
    //console.log('Processing link ' + link);
    fs.exists(link, function(exists){
      if(!exists){  // If the Soft Link is invalid, delete it!
        //console.log('Deleting  link ' + link);
        fs.unlink(link, function(err){
          if(err){
            return onError(err);
          }
          return cb();          
        });
      }
      return cb();
    });
  }

  function onError(err) {
    if (!errs) {
      errs = [];
    }
    if (typeof errs.write === 'undefined') {
      errs.push(err);
    }
    else { 
      errs.write(err.stack + '\n\n');
    }
    return cb();
  }

  function cb() {
    running--;
    finished++;
    if ((started === finished) && (running === 0)) {
      return errs ? callback(errs) : callback(null);
    }
  }
}
/**
 * listDirsSync       List the dirs contained in the passed root dir    
 * @param  {String}   dir      the root dir
 * @param  {Array}    dirs     list of dirs found
 * @param  {Boolean}  bResolve If true add the FQ path to the list
 */
exports.listDirsSync = function(dir, dirs, bResolve){

  var curDir = '';

  if (bResolve === undefined) bResolve = true;

  if (fs.existsSync(dir)){
    fs.readdirSync(dir).forEach(function(file, index){
      curDir = path.resolve(dir) + '/' + file;
      if (fs.lstatSync(curDir).isDirectory()){ // recurse
        if(bResolve){
          dirs.push({ 'name': curDir})  // Add the dir to the list         
        }
        else{
          dirs.push({ 'name': file})  // Add the dir to the list        
        }
      }
    });
  }
}

/*
  Delete the dirs and subdirs and and their content
    dir: the root dir
    bDeleteDir: is true if dir must be deleted
*/
exports.deleteDirSync = function(dir, bDeleteDir){ /* bDeleteDir is true if dir must be deleted */

  if (fs.existsSync(dir)){
    fs.readdirSync(dir).forEach(function(file, index){
      var curDir = dir + "/" + file;
      if (fs.lstatSync(curDir).isDirectory()){ // recurse
        exports.deleteDirSync(curDir, true);
      } else { // delete file
        fs.unlinkSync(curDir);
      }
    });
    if (bDeleteDir === true){
      fs.rmdirSync(dir);  // delete all other dirs than the root dir
    }
  }
}

/* 
  Evaluate the date diff between 2 dates  
    datepart: 
      'y' --> years,    'm' --> months, 'w' --> weeks, 
      'd' --> days,     'h' --> hours,  'n' --> minutes, 
      's' --> seconds,  's' --> seconds 'x' --> milliseconds 
*/    
exports.dateDiff = function(datepart, fromdate, todate){

  datepart = datepart.toLowerCase();  
  var diff = todate - fromdate; 
  var divideBy = { 
        w:604800000, 
        d:86400000, 
        h:3600000, 
        n:60000, 
        s:1000,
        x:1
      };  

  return Math.floor(diff/divideBy[datepart]);
}

// Pad the passed number to the passed width and returns it as string
exports.padWithZeros = function(vNumber, width){

  var numAsString = vNumber + "";
  while (numAsString.length < width) {
    numAsString = "0" + numAsString;
  }
  return numAsString;
}

// Create a Request Id as yyMMddhhmmssSSS-x where x is random integer between min (inclusive) and max (inclusive)
exports.buildReqId = function(date){
    
    let digitFactor = 0x1000000

    // Since digitFactor is 7 digits in hex,  Math.floor((1 + Math.random()) * digitFactor).toString(16) will return 7 hex digits. 
    // Since we need only six digits as request id suffix, we strip the first digit using substring(1) fucntion (Math.floor((1 + Math.random()) * digitFactor).toString(16).substring(1))
    // If we need more hex digits in request id suffix part, accordingly increase the digits in the digitFactor. Ex: if 7 digits are needed, digitFactor should be 0x10000000

    return  date_format.asString("yyMMddhhmmssSSS", date) + '-' + Math.floor((1 + Math.random()) * digitFactor).toString(16).substring(1)
}

/**
 * Retuns files in a directory in an ARRAY 
 * @Params:
 * dir  :  String. Name of the directory
 * ext  : File extension like ".jar"
 * caseSensitive:  true: Case sensivie ext comparision, false: No case sensivie ext comparision
 * recursive: Boolean: true: Recursively call subdirectories to find files. false:Find files only in given directory 
 * callback    :  A callback function 
 *             callback function will have two arguments.
 *                "err", of node.js type "Error"
 *                "result", of type JSON
 *             In case of success,
 *                "err" will be null.
 *                "result" a JSON array containing files in that directory
 *             In case of error,
 *                "err" will have information about error occurred during method execution
 *                "result" will be null.
 */
exports.getFileList = function(dir, ext, caseSensitive, recursive, callback){
  var self    = this
    , absPath = path.resolve(dir) + FILE_SEP
    , extArg  = ext
    , fileExt = undefined
    ;

  if(!caseSensitive) extArg    = ext.toLowerCase();

  fs.readdir(absPath, function (err, files){
    if(err){
        callback(err, null);
        return;
    }

    var retFiles=[];
    var errSave= "";

    if(files === undefined){
      callback(new Error("Invalid directory name. Dir:" + absPath), null);
      return;
    }
    else{
      var entryCount = files.length;
      if(!entryCount){
        return callback(null, retFiles);
      }

      files.forEach(function(file){
        var stats = fs.statSync(absPath + file);
        fileExt = path.extname(file);
        if(!caseSensitive) fileExt = fileExt.toLowerCase();
        if (stats.isFile() && fileExt === extArg){
          retFiles.push( absPath + file );
          if (!--entryCount){
            // Invoke callback only after all entries are processed
            callback(null, retFiles);
            return;
          }
        }
        else if(stats.isDirectory() && recursive){
          self.getFileList(absPath + file, ext, caseSensitive, recursive, function(err, res){
              if(err){
                errSave += err.message;
              }
              else if(res !== undefined && res !== null){
                retFiles = retFiles.concat(res);
              }

              if (!--entryCount){
                // Invoke callback only after all dir entries are processed
                if( errSave !== "" ){
                  callback(new Error(errSave), null);  
                }
                else{
                  callback(null, retFiles);
                }
              }
          });
        }
        else{
          if (!--entryCount){
            // Invoke callback only after all entries are processed
            callback(null, retFiles);
            return;
          }
        }
      });
    }
  });
}
//V101 End

// Evaluate the length of an utf-8 string
exports.strLength = function(str){
  // returns the byte length of an utf8 string
  var s = str.length;
  for (var i=str.length-1; i>=0; i--) {
    var code = str.charCodeAt(i);
    if (code > 0x7f && code <= 0x7ff) s++;
    else if (code > 0x7ff && code <= 0xffff) s+=2;
  }
  return s;
}
 
exports.clone = function(a){
   return JSON.parse(JSON.stringify(a));
}

// Create a Unique JobId
exports.createJobId = function(){
  return date_format.asString('yyMMddhhmmssSSS', new Date());
}

// V102 Begin
// Get Text of an Attribute placed under given tag from AFP2web Server Response Body
// cheerio module used in this fucntion leads vulnerability issues
// This function was earlier used in oxsnps-afp2any and now it obsolete 
/*
exports.getAttrTextFromHTML = function(html, tagName, attrName, attrType){

  // Parse HTML
  var parsedHTML = cheerio.load(html);

  // Get contents under tag. For example <pre>
  var tag = parsedHTML(tagName, 'body');

  // Filter contents for the specific attribute
  var targetLine = "" + tag.contents().
          filter(function(i, el){
            if(el.type === attrType && el.data.indexOf(attrName)>=0){ 
                  return true;
              }
          });
  var attrText = null;

  if(targetLine !== null && targetLine.length>0){
    attrText = targetLine.substring(targetLine.indexOf(':') + 1).trim();
  }
  return attrText;
}
// V102 End
*/
// cheerio module used in this fucntion leads vulnerability issues
// This function was earlier used in oxsnps-afp2any and now it obsolete 
/**
 * [getHTMLContent description]
 * @param  {JSON} parms
 *                parms:{
 *                  "htmlBuf":  "<html>...</html>",
 *                  "context":  "head",
 *                  "selector": "meta",
 *                  "pattern":   "^i3\\_.*" -->some reg expr
 *                }
 */
/*
exports.getHTMLContent = function(parms){

  logger.debug('-->getHTMLContent: parms: ' + JSON.stringify(parms));

  var result          = []
    , regexprPattern  = new RegExp(parms.pattern)
    , keys            = undefined
    ;

  // Parse HTML
  var metaTagList = cheerio(parms.selector, parms.context, parms.htmlBuf);
  if(!metaTagList || metaTagList.length <= 0) return undefined; // selector not found in the context!

  logger.debug('-->getHTMLContent: metaTagList length: ' + metaTagList.length);
  keys = Object.keys(metaTagList);

  // Ex.: metaTagList[1].attribs={"name":"i3_accountNumber","content":"12345678"}
  keys.forEach(function(key){
    if(metaTagList[key].attribs && metaTagList[key].attribs.name){
      if(regexprPattern.test(metaTagList[key].attribs.name)){
        logger.debug('-->getHTMLContent: name:' + metaTagList[key].attribs.name + ', value:' + metaTagList[key].attribs.content);
        result.push({'name':metaTagList[key].attribs.name,'value':metaTagList[key].attribs.content});
      }
    } 
  });
  if(result.length<=0) return undefined;
  return result;
}
*/

// V103 Begin
/**
 * [checkAndCreateDir description]
 * @param  {String}   dir      [description]
 * @param  {String}   mode     [description]
 * @param  {Function} callback [description]
 * @return {}
 */
exports.checkAndCreateDir = function(dir, mode, callback){
    return exports.mkDir(dir, mode, callback);
}

/**
 * mkDir              Create a dir if it does not exist
 * @param  {String}   dir      
 * @param  {String}   mode     
 * @param  {Function} callback(err)
 */
exports.mkDir = function(dir, mode, callback){
  // V3.0.33 Begin
  // mkdirp interface changed from v0.5.1 to v1.0.3. mkdirp does not use anymore callback but promise
  // mkdirp(dir, [opts]) -> Promise<String | undefined>
  //mkdirp.mkdirp(dir, mode,/*works since we set process.umask(0)*/ callback);
  mkdirp(dir, mode).then(made => {callback()}).catch(err => {callback(err)})
  // V3.0.33 End
}

/**
 * mkDirSync          Create a dir SYNC if it does not exist
 * @param  {String}   dir      
 * @param  {String}   mode     
 * @result            throw an exception in case of error
 */
exports.mkDirSync = function(dir, mode){

  mkdirp.sync(dir, mode/*works since we set process.umask(0)*/);
}

/**
 * Parse a string as JSON
 * @param  {String} str [description]
 * @return {JSON}     [description]
 */
exports.parseJSON = function(str){
    try {
        return JSON.parse(str);
    } catch (ex) {
        return null;
    }
} 
// V103 End

/**
 * Parse a string as JSON
 * @param  {String} str [description]
 * @return {JSON}     [description]
 */
exports.parseJSONFile = function(filename, readOptions, callback){

  if(!callback){
    callback    = readOptions
    readOptions = undefined
  }

  if(!readOptions) readOptions = {'encoding': 'utf8'}

	//console.log('--->parseJSONFile: Reading from file: ' + filename)

	fs.readFile(filename, readOptions, function(err, data){
		if(err) return callback(new Error('Unable to read file: ' + filename + '. ' + err.message))

		data = exports.parseJSON(data);
		if(!data) return callback(new Error('Unable to parse file: ' + filename))

		//console.log('--->parseJSONFile: Req.Id=' + res._Context.reqId + ', data: ' +  JSON.stringify(data))
		callback(null, data)
	});	
} 


/**
 * Return the path of a file (similar to unix dirname cmd)
 * @param  {String} str the fully qualified name of the file (ex. mnt/inputchannels/mail/out.pdf)
 * @return {String}     the fully qualified name of the path to the file (ex. mnt/inputchannels/mail)
 */
exports.getDirname = function(str){
    try {
        return path.dirname(str);
    } catch (ex) {
        return undefined;
    }
} 

// V104 Begin
/**
 * Merge obj2 Json with Obj1 Json. If any entry of obj2 already exists in obj1, obj1 entry will be replaced with obj2 entry.
 * @param  {JSON} obj1 JSON to which obj2 entries has to be merged
 * @param  {JSON} obj2 JSON
 * @return {JSON}      Returns merged JSON object.
 */
exports.mergeJSON_obsolete = function(obj1, obj2){
  if(obj2 === undefined || obj2 === null){
    return obj1;    
  }
  
  if(obj1 === undefined || obj1 === null){
    obj1 = {};
  }

  for(var attrname in obj2){
    obj1[attrname] = obj2[attrname];
  }
  return obj1;
}
// V104 End

// V3.0.32 Begin
/**
 * Merge two objects obj1 & obj2,  and return a new merged object with the elements from both obj1 & obj2.
 * If an element at the same key is present for both obj1 and obj2, the value from obj2 will appear in the result.
 * 
 * @param  {JSON} obj1    JSON   A single JSON object or array of objects
 * @param  {JSON} obj2    JSON   Optional.  If obj1 is array of objects, then this parameter is options parameter else a JSON object
 * @param  {JSON} options JSON   Optional. Options. Refer to https://www.npmjs.com/package/deepmerge
 * @return {JSON}      Returns merged JSON object.
 */
exports.mergeJSON = function(obj1, obj2, options){

  if(obj1 && Array.isArray(obj1)){
    options = obj2
    return deepmerge.all(obj1, options)
  }
  return deepmerge(obj1, obj2, options)
}
// V3.0.32 End

/**
 * Create async. a Symbolic / Soft Link
 * @param  {String}   srcpath  Source Path
 * @param  {String}   dstpath  Destinattion path
 * @param  {String}   type     'dir', 'file', or 'junction' (default is 'file')
 * @param  {Function} callback A callback function
 *                               callback function will have one argument "err" of node.js type "Error".
 *                               In case of success, "err" will be null.
 *                               In case of error, "err" will have information about error occurred on link creation
 * @return {}
 * 
 */
exports.symlink = function(srcpath, dstpath, type, callback){
    return _symlink(srcpath, dstpath, type, callback);
}

// V106 Begin
/**
 * Delete directories and files that are older than given interval
 * @param  {String}   dir        Root Directory
 * @param  {Boolean}   bDeleteDir Flag to say whether to delete Root directory or not
 * @param  {String}   datePart   String that specifies the unit of interval(like days, months,...)
 * @param  {Number}   interval   Interval
 * @param  {Boolean}   recursive  Flag to delete directories recursively or not
 * @param  {JSON}      timeToUse  {
 *                          'dir':  <fs stat time properyt to use like ctime. mtime, to find age for directroy entities>, 
 *                          'file': <fs stat time properyt to use like ctime. mtime, to find age for file entities> 
 *                      }
 * 
 * @param  {Function} callback   callback(err) 
 */
exports.deleteDir = function(dir, bDeleteDir, datePart, interval, recursive, timeToUse, callback){

  var fileDate    = ''
    , now         = new Date()
    , entryCount  = undefined
    , absPath     = path.resolve(dir) + '/'
    , diff        = undefined
    , filetime    = undefined
    ;

  // V4.1.34
  if(typeof timeToUse === 'function'){
    callback = timeToUse
    timeToUse = {'dir': 'birthtime', 'file': 'mtime'}
  }

  timeToUse     = timeToUse       || {}
  timeToUse.dir = timeToUse.dir   ? timeToUse.dir.toLowerCase()   : 'birthtime'
  timeToUse.file= timeToUse.file  ? timeToUse.file.toLowerCase()  : 'mtime'
  // V4.1.34

  if(logger.isDebugEnabled()) logger.debug('-->deleteDir: dir: '+ dir + ', now: ' + now + ', datePart:>' + datePart + '<, interval:' + interval + ', timeToUse: ' + JSON.stringify(timeToUse));

  fs.readdir(absPath, function(err, files){
    if(err){
      if(err.code === 'ENOENT'){
        if(logger.isDebugEnabled()) logger.debug('--->deleteDir: Warning! ' + absPath +' not found.');
        return callback();
      }
      else return callback(err);
    }

    entryCount = files ? files.length : 0;

    if(logger.isDebugEnabled()) logger.debug('-->deleteDir: Dir: ' + absPath + ', entryCount:' + entryCount);
    // 	V3.0.35-1 Begin
    //async.each(files, function (file, next){
    async.eachSeries(files, function (file, next){
    // 	V3.0.35-1 End
          file = absPath + file;
          fs.stat(file, function(err, stat){
              if(err){
                if(err.code === 'ENOENT') logger.debug('--->deleteDir: Warning! Unable to get statistics of file. ' + file +' file not found.');
                else logger.warn('--->deleteDir: Unable to get statistics of file(' + file +') ' + err.message);
                setTimeout(function(){next()}, 0);
                // return next(err);
              } 
              else if(stat.isDirectory() && recursive){
                // 3.0.35-2 Begin
                // Get Directory creation time.  
                //fileDate =  new Date(stat.birthtime); 
                // V4.1.34
                filetime = stat[timeToUse.dir] || stat.birthtime  // V4.1.34
                fileDate =  new Date(filetime)  // V4.1.34
                diff = exports.dateDiff(datePart, fileDate, now)
				        if(logger.isDebugEnabled()) logger.debug('-->deleteDir: Dir: ' + file + ', ' + timeToUse.dir + ': '  + fileDate + ', diff: ' + diff);
                // V4.1.34                
                //if(diff < 0){                  logger.warn('-->deleteDir: Skipping ' + file + ' from deletion since it is created after invoking delete dir');                }
                if(diff < interval){
                  //skip dirs that are not older enough for deletion
                  if(logger.isDebugEnabled()) logger.debug('-->deleteDir: Skipping ' + file + ' from deletion since it not older enough');
                  return setTimeout(function(){next()}, 0);
                }                
                // 3.0.35-2 End
                // push the recursive call to the next event loop
                setTimeout(
                    function(){
                        exports.deleteDir(file, true, datePart, interval, recursive, function(err){
                            if(err){ 
                                if(err.code === 'ENOENT') logger.debug('--->deleteDir: Unable to delete directory. ' + file + ' dir not found.');
                                else logger.warn('--->deleteDir: Unable to delete directory (' + file +'). ' + err.message);
                                // return next(err);
                            }
                            next();
                        });
                    },
                    0
                )
              }
              else if(stat.isFile()){
                // Get Last Modification time
                //fileDate =  new Date(stat.mtime); 
                // V4.1.34
                filetime = stat[timeToUse.file] || stat.mtime   // V4.1.34
                fileDate =  new Date(filetime)                 // V4.1.34
                diff = exports.dateDiff(datePart, fileDate, now)
				        if(logger.isDebugEnabled()) logger.debug('-->deleteDir: File: ' + file + ', ' + timeToUse.file + ': '  + fileDate + ', diff: ' + diff);
                // V4.1.34                
                // if(diff < 0){                   logger.warn('-->deleteDir: Skipping ' + file + ' from deletion since it is created after invoking delete dir1');               }
                if(diff < interval){
                  //skip files that are not older enough for deletion
                  if(logger.isDebugEnabled()) logger.debug('-->deleteDir: Skipping ' + file + ' from deletion since it not older enough');
                  return setTimeout(function(){next()}, 0);
                }
                if(logger.isDebugEnabled()) logger.debug('-->deleteDir: Deleting ' + file);
                fs.unlink(file, function(err){
                  if(err){
                    if(err.code === 'ENOENT') logger.debug('--->deleteDir: Unable to delete file. ' + file + ' file not found.');
                    else logger.warn('--->deleteDir: Unable to delete file (' + file +'). ' + err.message);
                  }
                  setTimeout(function(){next()}, 0);
                });
              }
              else setTimeout(function(){next()}, 0);
          });
        },
        function(err){
          if(err) return callback(err);
          
          if(!bDeleteDir) return callback();  

          // Delete directory
          fs.readdir(absPath, function(err, files){
            if(err){
              if(err.code === 'ENOENT'){
                if(logger.isDebugEnabled()) logger.debug('--->deleteDir: Warning! ' + absPath +' not found.');
                return callback();
              }
              else return callback(err);
            }
            entryCount = files ? files.length : 0;

            if(entryCount !== 0)  return callback(); // dir. not empty, do not delete dir.

            // Directory is older that is why we come here so no need to check the time of directory again
            if(logger.isDebugEnabled()) logger.debug('-->deleteDir: Deleting ' + absPath+ '...');
            fs.rmdir(absPath, function(err){
                if(err){
                  if(err.code === 'ENOENT'){
                    if(logger.isDebugEnabled()) logger.debug('--->deleteDir: Warning! ' + absPath +' not found.');
                    return callback();
                  }
                  else return callback(err);
                }
                return callback();
            });
         });
      }
    );
  });
}
// V106 End

// V107 Begin
/**
 * Get document id of the request
 * @param  {JSON}   req Request  Object
 * @param   {JSON}  options   Options object	 // V3.0.24
 *                  options : {
 *                    // List of parameters in the query string of input.url parameter, whose values are to be used to generate unique id along with host,post and service path in input.url parameter (refer to OXS-8448)
 *                    "docidParms" : [] // applicable only for req.input.url
 *                  }
 * @return {String}     Documen id in case of success else error object
 */
exports.getDocId = function(req, options){

  if(req.input === undefined){
    return (new Error("request does not contain input document parameters"));
  }

  // V108 Begin
  if(req.input.buffer){
    var srcBuf = new Buffer(req.input.buffer, 'base64')
      , tmpBuf = srcBuf.slice(0, 1024);
    return md5(tmpBuf, 'hex');  
  }
  else if(req.input.filename){
    return md5(req.input.filename, 'hex');
  }
  else if(req.input.url){
    var urlTmp = req.input.url.trim()
      , idx    = urlTmp.indexOf('?')
      , docidTxt = urlTmp
      , urlObj     = undefined
      ;

    if(idx>=0){
      docidTxt = urlTmp.substring(0,idx);
    }
    // V3.0.24 Begin
    // Add value of parameters given for docid genearation to the docidTxt
    if(options && options.docidParms && options.docidParms.length>0){
      urlObj = url.parse(req.input.url, true);
      if(urlObj.query){
        options.docidParms.forEach(function(parm){
          if(urlObj.query[parm]) docidTxt += '-' + urlObj.query[parm];
        })
      }
    }
    //console.log('docidTxt: '+ docidTxt)
    // V3.0.24 End
    return md5(docidTxt, 'hex');
    /*
    var urlParts = url.parse(req.input.url);
    if(urlParts.path && (urlParts.path.length > 0)){
      return md5(urlParts.path);
    }
    else{
     return (new Error("input.url does not contain any path to get the document"));
    }
    */
  }
  else{
    return (new Error("request does not contain input document parameters"));
  }
  /*
  // if docId + url is given, extract docId from url
  if(req.input.docId && req.input.url){
    return this.getValueUsingRegExp(req.input.docId, req.input.url);
  }
  else if(req.input.docId){
    // Return docId, if passed in the request
    return req.input.docId;
  }
  else{
    return null;
  }
  */
 // V108 End
 }

// V110 Begin
/**
 * Read the pass file content and get MD5 Id
 * @param  {String}   filename  File Name
 * @param  {Function}   callback(err, md5Id)
 */
 exports.getMD5Id = function(filename, callback){
  var md5Id = null;

  if(filename === undefined || filename === null || filename.length <=0){
    return callback(new Error("Could not generate MD5Id, passed filename is null or empty"));
  }
  fs.readFile(filename, function(err, data){
    if(err){
      return callback(new Error('Could not generate MD5Id, unable to read "' + filename + '". ' + err.message));
    }
    md5Id = md5(data, 'hex');
    return callback(null, md5Id);
  });
}
// V110 End

/**
 * Get MD5 Id for the passed string
 * @param  {String}   data  data
 * @param  {Function}   callback(err, md5Id)
 */
//!!! Extend it for any object like buffer/JSON
 exports.getObjMD5Id = function(data, callback){

  var md5Id = null;

  if(data && data.length > 0 ){
    md5Id = md5(data, 'hex');
    return callback(null, md5Id);
  }else{
    return callback(new Error("Could not generate MD5Id, passed data is null or empty"));
  }
}

// 
/**
 * Get value from input using regular expression
 * @param  {String} regExpr Regular Expression String
 * @return {String}         If regular expression is valid returns a value extracted from inputString 
 *                          else returns regular expression itself.
 */
exports.getValueUsingRegExp = function(regExpr, inputString){
  if(regExpr === null || inputString === null){
    return null;
  }

  var regexp = new RegExp(regExpr)
    , result = inputString.match(regexp)    // Refer http://msdn.microsoft.com/en-us/library/ie/7df7sf4x%28v=vs.94%29.aspx
    , retValue = ""
    ;

  //console.log('-->getValueUsingRegExp:RegExp: "' + regExpr + '", result:' + JSON.stringify(result));
  if(result && result.length > 1){
    // Values to be extracted start from index 1.
    for(var index=1; index<result.length; index++)
    {
        if(retValue.length > 0){
          retValue += '-';
        }
        retValue += result[index];
    }
  }
  else{
      retValue = regExpr;
  }
  //console.log('-->_getDocIdFromRegExp: evaluated value is "' + retValue + '"');
  return retValue;
}
// V107 End

/**
 * Get Cache filename suffix
 * @param  {JSON} req                  Request Object
 * @param  {JSON} cacheFilenamePattern JSON Array having info about parameter names used to build cache filename suffix
 * @return {String}                    Returns cache filename suffix
 */
exports.getCacheFilenameSuffix = function(req, cacheFilenamePattern){ 
  var name = ""
    ;

  if(req.props === undefined || req.props === null){
    return "";
  }  

  if(cacheFilenamePattern === undefined || cacheFilenamePattern.length <=0){
    return "";
  }

  var value          = null
    , filenameSuffix = ""
    ;

  for(var i = 0; i < cacheFilenamePattern.length; i++){
    // Get Parameter Name
    name = cacheFilenamePattern[i].name;

    // Get Parameter value
    if(req.props[name]){
      value = req.props[name];
    }
    else
    {
      value = cacheFilenamePattern[i].defaultValue;
    }

    if(cacheFilenamePattern[i].type && cacheFilenamePattern[i].type.toLowerCase() === 'bool'){
      if(cacheFilenamePattern[i].defaultValue && cacheFilenamePattern[i].defaultValue.toLowerCase() !== value.toLowerCase()){
        filenameSuffix +=  "-" + cacheFilenamePattern[i].shortForm;
      }
    }
    else{
      if(cacheFilenamePattern[i].valuePattern && value !== null && value.length > 0){
        // Get Parameter value applying regular expression
        value = exports.getValueUsingRegExp(cacheFilenamePattern[i].valuePattern, value);
      }
      if(cacheFilenamePattern[i].replacePattern && value !== null && value.length > 0){
        // Replace given pattern in parameter value with value given in conf.
        //var regexp = XRegExp(cacheFilenamePattern[i].replacePattern.pattern);
        //value = XRegExp.replace(value, regexp, cacheFilenamePattern[i].replacePattern.value, 'all');
        var regexp = new RegExp(cacheFilenamePattern[i].replacePattern.pattern);
        value = value.replace(regexp, cacheFilenamePattern[i].replacePattern.value);
      }     

      filenameSuffix +=  "-" + cacheFilenamePattern[i].shortForm + value;
    }
  }
  return filenameSuffix;
}


/**
 * Get Output JSON Filename 
 * @param  {Response} res              Response Object
 * @param  {Response} req              Request Object
 * @param  {JSON} cacheFilenamePattern JSON Array having info about parameter names used to build cache filename suffix
 * @return {String}                    Returns Output JSON Filename 
 */
exports.getOutputJSONFilename = function(res, req, cacheFilenamePattern){

    var filenameSuffix = exports.getCacheFilenameSuffix(req, cacheFilenamePattern)
      , jsonFilename   = ""
      ;

  // Set the Output  filename (<docId dir>/<docId>-<pageNr>.<res._Context.outputType>)
  jsonFilename = (res._Context.cacheSubdir || res._Context.docId) + '/' + res._Context.docId + '-' + res._Context.pageNr  + filenameSuffix + '.json';
  return jsonFilename;
}

// V109 Begin
/**
 * Get Output JSON Filename 
 * @param  {Response} res Response Object
 * @return {String}       Returns Output JSON Filename 
 */ 
exports.getOutputJSONFilename_del = function(res){
  var req           = res._Context.req
    , jsonFilename  = ""
    , otherProps    = ""
    ;

  if(req.props){
    // Segserver/OCR request parameters
    if(req.props.rotation && req.props.rotation !== 0){
      otherProps += '-r' + req.props.rotation;
    }
    if(req.props.deskew && req.props.deskew.toLowerCase() === 'yes'){
      otherProps += '-d';
    }
    if(req.props.max_width || req.props.max_height){
      otherProps += '-os';  
      if(req.props.max_width){
        otherProps += req.props.max_width;  
      }
      if(req.props.max_width && req.props.max_height){
        otherProps += 'x' ;
      }     
      if(req.props.max_height){
        otherProps += req.props.max_height; 
      }     
    }
    if(req.props.antialias && req.props.antialias.toLowerCase() === 'on'){
      otherProps += '-aa';
    }
    if(req.props.jpgQuality){
      otherProps += '-jq' + req.props.jpgQuality;
    }
    if(req.props.pngCompressionFactor){
      otherProps += '-pcf' + req.props.pngCompressionFactor;
    }   

    // Image service properties
    if(req.props.Rotation && req.props.Rotation !== 0){
      otherProps += '-r' + req.props.Rotation;
    }
    if(req.props.OutputSize){
      /*
      var idx = req.props.OutputSize.indexOf(",");
      if(idx >= 0){
        var osw = req.props.OutputSize.substring(0, idx);
        var osh = req.props.OutputSize.substring(idx+1);
        otherProps += '-os' + osw + 'x' + osh;
      } 
      */
      // Replace ',' char' with 'x' char
      otherProps += '-os' + req.props.OutputSize.replace(/,/g, 'x');
    }
    if(req.props.JPEGQuality){
      otherProps += '-jq' + req.props.JPEGQuality;
    }
    if(req.props.FormatType){
      otherProps += '-ft' + req.props.FormatType;
    }
    if(req.props.Resolution){
      otherProps += '-pr' + req.props.Resolution;
    }
    if(req.props.OutputScale){
      otherProps += '-oscale' + req.props.OutputScale;
    }
    if(req.props.Color && req.props.Color.toLowerCase() === 'on'){
      otherProps += '-c' + req.props.Color;
    }
  }

  // Set the Output  filename (<docId dir>/<docId>-<pageNr>.<res._Context.outputType>)
  jsonFilename = res._Context.docId + '/' + res._Context.docId + '-' + res._Context.pageNr  + otherProps + '-' + res._Context.outputType + '.json';
  return jsonFilename;
}

/**
 * removeTrailingSlash:   Remove the trailing slash if any
 * @param  {string}       path
 * @return {string}       path without the trailing slash
 */
exports.removeTrailingSlash = function(path){
    return path.replace(/\/$/, "");
} 

/**
 * Check if passed value is positive integer
 * @param  {[number|string]}  inputValue value to be verified
 * @return {Boolean}            Returns trun if passed value is integer and > 0
 */
exports.isPositiveInteger = function(value){
  var ret = /^[1-9]\d*$/.test(value);
  return ret;
}

exports.isValidPageRange_doesnotwork = function(value){
  var ret = /^[1-9]\d*-[1-9]\d*/.test(value);
  return ret;
}

/**
 * [isValidPageRange description]
 * @param  {[type]}  value [description]
 * @return {Boolean}       [description]
 */
exports.isValidPageRange = function(value){
  if(value === null || value.length <= 0){
    return false;
  }

  var idx = value.indexOf('-');   

  if(idx<0){
    return false;
  }
  var startPage = value.substring(0, idx);
  if(startPage && !exports.isPositiveInteger(startPage)){
    return false;
  }

  var endPage = value.substring(idx+1); 
  if(endPage && !exports.isPositiveInteger(endPage)){
    return false;
  }
  return true;
  /*
    10-=> true
    0-0=> false
    0=> false
    -5=> true
    -55=> true
    672-=> true
    2-3=> true
    25-300=> true
    25+300=> false
    xyz=> false
    123=> false
  */  
}

/**
 * getHardwareInfo
 * @return {String}     Hardware Info
 */
exports.getHardwareInfo = function(){

  var cpus = os.cpus()
    , CRLF_Tabs = '\n\t\t\t\t\t\t\t'
    , hardwareInfo = 'Server Info - Hardware:'
                   + CRLF_Tabs + 'Hostname: ' + os.hostname() + ', ' + os.platform() + ' ' + os.arch() + ' ' + os.release()
                   + CRLF_Tabs + 'Average Load: ' + os.loadavg()
                   + CRLF_Tabs + 'Total Mem.: ' + (os.totalmem()/1024/1024/1024).toFixed(2) + 'GB, Free Mem.: ' + (os.freemem()/1024/1024/1024).toFixed(2) + 'GB.'
    ;
  
  for (var i = 0; i < cpus.length; i++) {
    hardwareInfo += CRLF_Tabs + 'CPU ' + (i+1) +', Model: ' + cpus[i].model;
  };
  
  hardwareInfo += CRLF_Tabs + 'Network Interfaces: ' + _networkInterfaces();

  /* Other Hardware Info
    logger.info(os.endianness());
    logger.info(os.type());
    logger.info(os.tmpdir());
    logger.info(os.uptime());
    logger.info(os.cpus());
    logger.info(os.networkInterfaces());
  */
  return hardwareInfo;
}

/**
 * getEnvironmentInfo   
 * @return {String}     Environment Info
 */
exports.getEnvironmentInfo = function(){

  /* Let's try to beautify the output as:
    {
    "TERM":"xterm",
    "SHELL":"/bin/bash",
    "XDG_SESSION_COOKIE":"d1938ee4627d4f0a3849455d00000007-1461134155.149860-472728795",
    "USER":"root",
    "LD_LIBRARY_PATH":"./node_modules/oxsnps-ibmmq/services/",
    "LS_COLORS":"rs=0:di=01;34:ln=01;36:... <cut> ...:*.spx=00;36:*.xspf=00;36:",
    "SVN_USERNAME":"daniel",
    "SUDO_USER":"adminuser",
    "SUDO_UID":"1000",
    "USERNAME":"root",
    "PATH":"/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin:/usr/games",
    "MAIL":"/var/mail/root",
    "PWD":"/opt/OXS.IMPORT",
    "UV_THREADPOOL_SIZE":"24",
    "LANG":"en_US.UTF-8",
    "SVN_PASSWORD":"EkLj9/3I1BF6f8x4Y0Ko",
    "NODE_PATH":"/usr/lib/nodejs:/usr/lib/node_modules:/usr/share/javascript",
    "HOME":"/root",
    "SUDO_COMMAND":"/bin/su",
    "SHLVL":"2",
    "LOGNAME":"root",
    "LESSOPEN":"| /usr/bin/lesspipe %s",
    "DISPLAY":":0",
    "SUDO_GID":"1000",
    "LESSCLOSE":"/usr/bin/lesspipe %s %s",
    "COLORTERM":"gnome-terminal",
    "XAUTHORITY":"/home/adminuser/.Xauthority",
    "_":"/usr/bin/node"
    }  
  */    
  // Walk thru the JSON Object, sort it and build the Environment Info message
  return exports.walk(process.env);
}

/**
 * Returns info about the Server Environment
 * @param  {Object}     obj           A Javascript Object
 * @return {String}     Environment Info
 * 
 * see http://am.aurlien.net/post/1221493460/sorting-javascript-objects
 */
exports.walk = function(obj){

  var environmentInfo = 'Server Info - Environment:'
    , temp_array = []
    ;

  // Make an array out of the obj
  for (var key in obj) {
    if (obj.hasOwnProperty(key)) {
      temp_array.push(key + '=' + obj[key]);
    }
  }
  temp_array.sort(); // sort the array

  // Build the result
  for (var i=0; i<temp_array.length; i++) {
    environmentInfo += CRLF_Tabs + temp_array[i];
  }
  return environmentInfo;
}

/**
 * Returns info about the Server's mount points
 * @return {String}     Mount points Info
*/
exports.getMountInfoSync = function(){

  // V4.1.8 Begin
  if(os.platform() === 'win32'){
    try{
      // Refer to https://docs.microsoft.com/en-us/windows/win32/cimwin32prov/win32-logicaldisk, https://stackoverflow.com/questions/26729187/how-can-get-hard-disk-information-using-cmd      
      var stdout = 'Server Info - Drive: ' + cp.execSync('wmic logicaldisk'); 
      stdout = stdout.replace(/(\r\r\n\r\r\n|\r\r\n|\r?\n|\r)$/, '');     // remove last new line
      stdout = stdout.replace(/\r\r\n|\r?\n|\r/g, CRLF_Tabs);// replace new lines with CRLF_Tabs
      return stdout;
    }
    catch(err){
      return 'Drive information: ERROR when getting the drive information, Reason: ' + err.message;
    }
  }
  // V4.1.8 End

  try{
    var stdout = 'Server Info - Mount:\n' + cp.execSync('mount'); // execute mount command to get the mount points
    stdout = stdout.replace(/(\r?\n|\r)$/, '');     // remove last new line
    stdout = stdout.replace(/\r?\n|\r/g, CRLF_Tabs);// replace new lines with CRLF_Tabs
    return stdout;
  }
  catch(err){
    return 'Mount information: ERROR when getting the mount points, Reason: ' + err.message;
  }
}

/**
 * Returns info about the Server's /etc/hosts
 * @return {String}     /etc/hosts Info
 */
exports.getHostsSync = function(){

	// V4.1.8 Begin
  // Windows OS
  if(os.platform() === 'win32'){

    // Refer to https://support.rackspace.com/how-to/modify-your-hosts-file/ 
    // Host file location on Windows  (Widnows 7, 8, 10, Windows NT, Windows 2000, and Windows XP) is 'c:/Windows/System32/Drivers/etc/hosts'
    let filename  = undefined   
      , version   = undefined
      , systemdir = undefined
      , sysDirve  = undefined
  
    /*
      Windows env. variables
        HOMEDRIVE=C:
        SystemDrive=C:
        SystemRoot=C:\Windows
        windir=C:\Windows
    */
    systemdir = process.env.SystemRoot || process.env.windir
    if(!systemdir){
      sysDirve = process.env.SystemDrive || process.env.HOMEDRIVE
      systemdir = sysDirve ? (sysDirve + FILE_SEP + "Windows") : undefined
    }
    if(systemdir)  filename = path.join(systemdir, '/System32/drivers/etc/hosts')

    filename = filename || 'c:/Windows/System32/Drivers/etc/hosts'  // if env. variables are not found, fall back to default path

    try{
      version = 'Server Info - Hosts ('+ filename + '):\n' + fs.readFileSync(filename)
      version = version.replace(/(\r?\n|\r)$/, '')     // remove last new line
      version = version.replace(/\r?\n|\r/g, CRLF_Tabs)// replace new lines with CRLF_Tabs
      return version
    }
    catch(err){
      return 'Hosts information: ERROR when getting the content of "' + filename + '".  ' + err.message
    }
  }
  // V4.1.8 End

  // Linux OS
  try{
    var stdout = 'Server Info - Hosts (/etc/hosts):\n' + cp.execSync('cat /etc/hosts'); // execute cat command to get/etc/hosts content
    stdout = stdout.replace(/(\r?\n|\r)$/, '');     // remove last new line
    stdout = stdout.replace(/\r?\n|\r/g, CRLF_Tabs);// replace new lines with CRLF_Tabs
    return stdout;
  }
  catch(err){
    return 'Hosts (/etc/hosts) information: ERROR when getting the content of /etc/hosts, Reason: ' + err.message;
  }
}

/**
 * pad:               Pad the String (str) in the Length (length) using the Char (char)
 * @param  {string}   str
 * @param  {int}      length
 * @param  {char}     char
 * @return {string}   the modified str
 */
exports.pad = function(str, length, char){
  while (str.length < length){
    str = char + str;
  }
  return str;
}

/**
 * Convert XML file to JSON string
 * @param  {String}   xmlData         XML Data
 * @param  {Object}   parseOptions    Optional. XML Parser Options. Refer to https://www.npmjs.com/package/xml2js
 * @param  {Function} callback        callback(err, jsonObject)
 */
exports.xml2json = function(xmlData, parseOptions, callback){
  
  var parser = undefined
    , cbInvoked = false 

  if(!callback){
    callback = parseOptions;
    parseOptions = undefined;
  }

  // V4.1.27 Begin
  function _chkAndInvokeCallback(err, result){
    if(cbInvoked){
        let xmlDataTmp = xmlData ? JSON.stringify(xmlData) : ''
        logger.error('-->xml2json: callback invoked again, error: ' + (err ? err.message: '') + ', xmlData: ' + (xmlDataTmp  ? xmlDataTmp.substring(0, 4096): '') + (result ? ', result: '+ JSON.stringify(result) : ''))
    }
    else{
        cbInvoked = true
        callback(err, result)
    }
  }
  // V4.1.27 End

  if(!parseOptions){
    parseOptions = {
      explicitArray:false, 
      tagNameProcessors:[xml2js.processors.stripPrefix]
    }    
  };

  if(!parseOptions.tagNameProcessors) parseOptions.tagNameProcessors = [xml2js.processors.stripPrefix];

  try{
    parser = new xml2js.Parser(parseOptions);
    //parser.parseString(xmlData, callback);
    parser.parseString(xmlData, _chkAndInvokeCallback);    // V4.1.27
  }
  catch(err){
    //return callback(new Error('XML to JSON conversion failed. ' + err.message));
    return _chkAndInvokeCallback(new Error('XML to JSON conversion failed. ' + err.message));    // V4.1.27
  }
}

/**
 * Convert XML file to JSON string
 * @param  {String}   xmlFile         Fully qualified XML filename
 * @param  {Object}   readOptions     File Read options like {encoding:'utf8'}
 * @param  {Object}   parseOptions    Optional. XML Parser Options. Refer to https://www.npmjs.com/package/xml2js
 * @param  {Function} callback        callback(err, jsonObject)
 */
exports.xmlFile2json = function(xmlFile, readOptions, parseOptions, callback){

  var fileEnc 		= undefined
    , result 			= undefined
    , passedEnc 		= undefined
    , fileContent 	= undefined
    , regEx           = undefined
    , regExResult     = undefined
    , xmlDefaultEnc   = undefined
    ;

  /*      
    http://www.w3schools.com/xml/xml_syntax.asp
     -- UTF-8 is the default character encoding for XML documents.
     -- XML Attribute Values Must be Quoted
  */
  // Reg. exp to find encoding attribute value from xml content
  regEx           = /\s*encoding\s*=\s*"\s*([^\s]+)\s*"/i
  xmlDefaultEnc   = 'utf-8'; // UTF-8 is the default character encoding for XML documents.

  if(!readOptions) readOptions = {};

  if(!callback){
    callback = parseOptions;
    parseOptions = undefined;
  }

  if(!parseOptions) parseOptions = {};

  // Get Passed encoding
  passedEnc = readOptions.encoding || null; // Refer to fs.readFile in fs.js from node.js
  if(passedEnc) passedEnc = passedEnc.toLowerCase();

  async.series([
    // Read the file
    function(nextTask){
      fs.readFile(xmlFile, readOptions, function(err, data){
        if(err){
          return nextTask(new Error('Unable to read "' + xmlFile + '", ReadOptions=' + JSON.stringify(readOptions) +', ' + err.message));
        }
        fileContent = data;
        nextTask();
      });
    },

    // Get encoding from file content
    function(nextTask){
      regExResult = fileContent.match(regEx);
      if(regExResult && regExResult.length > 1){
        fileEnc = regExResult[1]; 
      }
      else{
        fileEnc = xmlDefaultEnc;
      }
      fileEnc = fileEnc.toLowerCase();
      if(fileEnc.match(/ISO-8859-\d+/i)) fileEnc = 'binary';
      nextTask();
    },

    // If encoding specified in readOptions is not same as encoding specified within XML file 
    // read the XML file again using the encoding specified within XML file 
    function(nextTask){
      //console.log('filenc:>'+ fileEnc +'<');
      //console.log('passedEnc:>'  + passedEnc +'<');

      if(passedEnc && passedEnc !== fileEnc){
        readOptions.encoding = fileEnc;
        //console.log('reading file again using ' + JSON.stringify(readOptions));
        fs.readFile(xmlFile, readOptions, function(err, data){
          if(err){
            return nextTask(new Error('Unable to read "' + xmlFile + '", ReadOptions=' + JSON.stringify(readOptions) +', ' + err.message));
          }
          fileContent = data;
          nextTask();          
        });
      }
      else nextTask();
    },    

    // convert xml 2 json
    function(nextTask){
      exports.xml2json(fileContent, parseOptions, function(err, result){
        if(err){
          return callback(new Error('Unable to parse XML data read from "' + xmlFile + '". ' + err.message));
        }
        callback(null,result);
      });
    }

  ],  function(err){
      return callback(err, result)
    }
  );    
}

/**
 * Convert JSON as XML
 * @param  {JOSN}       json            JSON Object
 * @param  {JSON}       xmlBuildOptions XML Build Options. Refer to https://www.npmjs.com/package/xml2js
 * @param  {Function}   callback        callback(err, xmlData)
 */
exports.json2xml = function(json, xmlBuildOptions, callback){
  
  var xmlData = null
    , builder = null
    ;

  if(!callback){
    callback = xmlBuildOptions;
    xmlBuildOptions = undefined;
  }

  if(!xmlBuildOptions){
    xmlBuildOptions = {
        renderOpts:{
          'pretty': false // pretty print or not
        },
        xmldec: {
          'version':'1.0',
          'encoding': exports.getEncoding('binary')
        }
    }    
  };

  // Convert JSON to XML
  builder = new xml2js.Builder(xmlBuildOptions);
  
  try{
    xmlData = builder.buildObject(json); /* xmlData is not a buffer it is a string in UTF8) */
  }
  catch(err){
    return callback(new Error('JSON to XML conversion failed. ' + err.message));
  }
  
  return callback(null, xmlData);
}

/**
 * Write passed JSON as XML to given filename
 * @param  {JOSN}       json            JSON Object
 * @param  {String}     xmlFilename     XML Filename to write to
 * @param  {JSON}       xmlBuildOptions XML Build Options. Refer to https://www.npmjs.com/package/xml2js
 * @param  {JSON}       writeOptions    File Write options.
 * @param  {Function}   callback        callback(err, xmlData)
 */
exports.json2xmlFile = function(json, xmlFilename, writeOptions, xmlBuildOptions, callback){
  
  if(!writeOptions) writeOptions = {};

  if(!callback){
    callback        = xmlBuildOptions;
    xmlBuildOptions = undefined;
  }

  exports.json2xml(json, xmlBuildOptions, function(err, xmlData){
    if(err) return callback(err);
    fs.writeFile(xmlFilename, xmlData, writeOptions, function(err1){
      if(err1) return callback(new Error('Writing JSON to  "' + xmlFilename + '" failed. ' + err1.message));
      return callback(null, xmlData);
    }); 
  });
}

/**
 * Get equivalent standard encoding string that corresponds to passed node.js encoding
 * @param  {String} nodeJSencoding Encoding as specified by node.js
 * @return {String}                returns equivalent standard encoding string that corresponds to passed node.js encoding
 */
exports.getEncoding = function(nodeJSencoding){
  // Refer to https://nodejs.org/api/buffer.html to get list of node.js supported encodings
  // Refer to https://en.wikipedia.org/wiki/Character_encoding
  switch(nodeJSencoding.toLowerCase()){
    case 'hex':
      return 'hex';

    case 'base64':
      return 'base64';

    case 'utf8':
    case 'utf-8':
      return 'UTF-8';

    case 'ascii':
      return 'ASCII';

    case 'binary':
      return 'iso-8859-1';

    case 'ucs2':
    case 'ucs-2':
      return 'UCS-2';

    case 'utf16le':
    case 'utf-16le':
      return 'UTF-16';

    default:
      return new Error('Unknown encoding: ' + nodeJSencoding);
  }
}

/**
 * Get Last Token separated by given delimiter
 * @param  {String} string    Input String
 * @param  {String} delimiter Delimiter character or string
 * @return {String}           Last token of string separated by given delimiter
 */
exports.getLastToken = function (string, delimiter){
  var idx       = 0
    , retValue  = string
    ;

  if(string === undefined || string === null || string.length === 0) return retValue;

  idx   = string.lastIndexOf(delimiter);

  if(idx>0) retValue = string.substring(idx+1);
  return retValue;
}

/**
 * Iterate a JSON and call <handleKey>  function for each key
 * @param  {[JSON]} obj           JSON Object
 * @param  {String} parentObjName Parent Object name. For Root or top level object it is string
 * @param  {Function} handleKey   Function that gets key name and its value.
 * @return {[type]}               None
 */
exports.iterateJSON=function(obj, parentObjName, handleKey){
  var value = ''
    , name  = ''
    ;

  for(name in obj){
    value = obj[name];
    if(parentObjName !== null && parentObjName.length > 0){
      name = parentObjName + '.'  + name;
    }
    if(value instanceof Object){        // V5.0.35  Works only for node8 and not for node16
      exports.iterateJSON(value, name, handleKey);
    }
    else if(typeof(value) === "object"){    // V5.0.35  Works for node16
      exports.iterateJSON(value, name, handleKey);
    }    
    else{
      handleKey(name, value);
    }
  }
}

/**
 * Convert booelan string to boolean 
 * @param  {String} string Input String
 * @return {boolean}        
 */
exports.stringToBoolean=function(string){
    switch(string.toLowerCase().trim()){
        case "true": case "yes": case "1": return true;
        case "false": case "no": case "0": case null: return false;
        default: return Boolean(string);
    }
}

/**
 * Strip a trial comma in the passed string
 * @param  {[String]} str Input Sting
 * @return {[String]}     String with stripped trailing comma, if any              
 */
exports.stripTrailingComma=function(str){
  var idx = 0;

  if(str && str.length > 0){
    idx = str.lastIndexOf(',');
    if(idx>=0 && idx === (str.length -1)) str = str.substring(0, idx);
  }
  return str;
}

/**
 * Check whether a string matches a pattern specified as regular exp.
 * @param  {String}     str
 * @param  {Reg. Exp.}  pattern
 * @return {Boolean}    Returns true if str matches pattern
 */
exports.strEndsWith=function(str, pattern){
//!!!IMPORTANT NOTE: just looks for now if str ends with pattern (see $)
  if (str.match(pattern + '$')) return true;
    return false;
}

/**
 * Copy a file or directory. 
 * @param  {String}   src      Source file or dir.
 * @param  {String}   dest     Destination file or dir.
 * @param  {JSON}     options  Refer to https://www.npmjs.com/package/fs-extra#outputjsonfile-data-callback
 * @param  {Function} callback callback(err)
 */
exports.copyFile=function(src, dest, options, callback){
  fse.copy(src, dest, options, callback);
}

/**
 * Encode string data passed as part of Get or Post http requests
 * @param  {String} str   Data to be encoded 
 * @return {String}       Encoded data
 */
exports.encodeURIData=function(str){
  /*
    Refer to  
    https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/encodeURIComponent
    https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/encodeURI

    encodeURI() and encodeURIComponent() escapes all characters except the following: alphabetic, decimal digits, - _ . ! ~ * ' ( )
    encodeURI does not encode "&", "+", and "=" chars, which are treated as special characters in GET and POST requests
    however encodeURIComponent() does encode these characters. 
   */
  return encodeURIComponent(str);
}

/**
 * Search of %HH where HH is a hex digit and replace with corresponding utf-8 char
 * @param  {String} str input string
 * @return {String}     String with replaced values for %HH 
 */
//!!! To be moved to customer specific common module
exports.replaceHexWithUTF8Char=function(str) {

  var regexp1    = /%u{0,1}([0-9A-F]{1,8})/ig
    , regexp    = /%u([0-9A-F]{1,4})|%([0-9A-F]{1,2})/ig
    , result    = str.match(regexp)  
    , ret       = undefined
    , utf8char  = undefined
    ;

  if(result && result.length > 0){      
    //console.log(str);
    str = str.replace(regexp, function(match, p1, p2){
      if(p1) p1 = p1.toUpperCase();
      if(p2) p1 = p2.toUpperCase();
      utf8char = hex2utf8[p1];
      //console.log('utils:'+'p1:' + p1 +', ' + p2 + ', utf8char:' + utf8char );
      if(utf8char){
        // We use Buffer since in hex2utf8.js we can define utf8 char(lefthand side) as array of bytes too. 
        // It is useful when we can not enter utf8 char in js file as character.
        ret = new Buffer(utf8char).toString();
      }
      else{
        // hex2utf8 does not contain mapping for p1.
        ret = match;  
      }
      return ret;
    }); 
    //console.log(str);
  }
  return str;
};

/**
 * Pad string with given 'pad' char for specified length
 * @param  {String} str   Input String
 * @param  {Int}    len   Padding Length
 * @param  {String} pad   Padding Character
 * @return {String}       Padded String
 */
exports.padLeft = function(str, len, pad){
    pad = typeof pad === 'undefined' ? '0' : pad + '';
    str = str + '';
    while(str.length < len) str = pad + str;
    return str;
}

/**
 * Ensure passed file is readable on return
 * @param  {String}   filename   Filename
 * @param  {int}   retryCount Retry count. Number of time to check for file readabiliy
 * @param  {int}   interval   Time in milliseconds, to pause between the file readable checks
 * @param  {int}   filesize   Optional. Used internally by recursive calls
 * @param  {Function} callback   callback(err)
 *                                 If file is ready for reading, err=null
 *                                 else err object contains error information
 */
exports.ensureFileIsReadable = function(filename, retryCount, interval, filesize, callback){
  var newSize = 0;

  if(typeof filesize === 'function'){
    callback = filesize;
    filesize = 0;
  }
  
  //console.log('filename:' + filename +', retryCount:'+ retryCount +', filesize:' + filesize);

  // fs.open() and fs.readFile() returns control, even if file is being written, so not used
  // Get File statistics
  fs.stat(filename, function(err, stats){
    if(retryCount == 0 && err) 
        return callback(new Error(filename + ' is not accessible. ' + err.message));

    if(err){
       //console.log('filename:' + filename + ' not readable yet. ' + err.message + ', retryCount:'  + retryCount + ', filesize:' + filesize + ', scheduled to check again...');
    }
    else{  
      newSize = stats.size;

      // if file size is same as prev. iteration file size, file is fully written 
      if(filesize !== 0 && filesize === newSize)
           return callback();

      // File is not accessible even after waiting for specified time
      if(retryCount == 0)
        return callback(new Error(filename + ' is not accessible even after waiting for specified time'));

       //console.log('filename:' + filename + ' not readable yet. ' + ', retryCount:'  + retryCount + ', filesize:' + newSize + ', scheduled to check again...');
    }
    // File not found still or still file is being written, so call _ensureFileIsReady after a delay
    setTimeout(function(){
        exports.ensureFileIsReadable(filename, --retryCount, interval,  newSize, callback);
      },
      interval
    );      
   });
}

/**
 * Replace not allowed chars in simple filename with '-' or passed replacement char
 * @param  {String}   simpleFilename    Simple filename without path
 * @param  {String}   replaceChar       Replace Char
 * @return {String}                     
 */
exports.replaceInvalidCharsInFilename = function(simpleFilename, replaceChar){
  
  if(!replaceChar) replaceChar = '-';

  // Refer to http://www.mtu.edu/umc/services/web/cms/characters-avoid/
  // to see Characters to Avoid in Directories and Filenames
  simpleFilename = simpleFilename.replace(/[#<$+%>!`&*‘|{?"=}/:\@ ]/g, replaceChar); 

  return simpleFilename;
}

/**
 * DeleteDirectory recursively
 * @param  {String|glob pattern}  dir      Directory
 * @param  {JSON}                 opts     Options. Refer to https://github.com/isaacs/rimraf
 * @param  {Function}             callback callback(err)
 */
exports.deleteDirectory = function (dir, opts, callback){
  rimraf(dir, opts, callback);
}

/**
 * Retuns files in a directory in an ARRAY matching the pattern
 * @Params:
 * dir  :  String. Name of the directory
 * ext  : File extension like ".jar"
 * caseSensitive:  true: Case sensivie ext comparision, false: No case sensivie ext comparision
 * recursive: Boolean: true: Recursively call subdirectories to find files. false:Find files only in given directory 
 * callback    :  A callback function 
 *             callback function will have two arguments.
 *                "err", of node.js type "Error"
 *                "result", of type JSON
 *             In case of success,
 *                "err" will be null.
 *                "result" a JSON array containing files in that directory
 *             In case of error,
 *                "err" will have information about error occurred during method execution
 *                "result" will be null.
 */
exports.getFiles = function(dir, pattern, recursive, callback){
  var absPath     = path.resolve(dir) + FILE_SEP
    , retFiles    = []
    , fqFile      = undefined
    ;

  // If pattern is string, convert pattern to regular expression  
  if(typeof(pattern) === 'string') pattern = new RegExp(pattern);

  fs.readdir(absPath, function (err, files){
    if(err) return callback(err, retFiles);

    if(files === undefined) 
        return callback(new Error("Invalid directory name. Dir:" + absPath), retFiles);

    if(!files.length) return callback(null, retFiles);

    async.forEachOfLimit(files, 1,
      function(file, index, next){
          fqFile = absPath + file;
          fs.stat(fqFile, function(err, stat){
              if(err)
                next(new Error('--->getFiles: Unable to get statistics of file(' + fqFile +'). ' + 
                              err.message));
              else if(stat.isDirectory() && recursive){
                exports.getFiles(fqFile, pattern, recursive, function(err, result){
                  if(err) return next(err);
                  if(result) retFiles = retFiles.concat(result);
                  next();
                });
              }
              else if(stat.isFile() && file.match(pattern)){
                retFiles.push(fqFile);
                next();
              }
              else next();
            });
        },
        function(err){
          return callback(err, retFiles);
        }
    );        
  });
}

/**
 * Performs an optimized deep comparison between the two objects, to determine if they should be considered equal. 
 * Refer to http://underscorejs.org/#rest
 * @param  {Object}   json1 JSON 
 * @param  {Object}   json2 JSON
 * @return {boolean}        Returns true if both JSON have same props and values
 *                          else returns false
 */
exports.jsonEqual = function(json1, json2){
  return underscore.isEqual(json1, json2)
}

/**
 * Move 'soruce' dir or file to 'dest' location
 * @param  {String}   source  source dir/file
 * @param  {String}   dest    destination dir/file
 * @param  {JSON}     options   see https://github.com/andrewrk/node-mv
 * @param  {Function} callback(err)      
 */
exports.move = function(source, dest, options, callback){
  mv(source, dest, options, callback)
}

/**
 * Check if file exiss
 * @param  {String}   file     Filename
 * @param  {Function} callback callback(err, exists)
 */
exports.fileExists = function(file, callback){
  fs.stat(file, function(err, stats){
      if(err){
          if(err.code === 'ENOENT') return callback(null, false);
          return callback(err);
      }
      return callback(null, stats.isFile());
  });
}

/**
 * Check if directory exists
 * @param  {String}   dir      Directory
 * @param  {Function} callback calback(err, exists)
 */
exports.dirExists = function(dir, callback){
    fs.stat(dir, function(err, stats){
        if(err){
            if(err.code === 'ENOENT') return callback(null, false);
            return callback(err);
        }
        return callback(null, stats.isDirectory());
    });
}

/**
 * Check if file/directory exists with the passed name
 * @param  {String}   entry      File or Directory Name
 * @param  {Function} callback calback(err, exists)
 */
exports.fsEntryExists = function(entry, callback){
    fs.stat(entry, function(err, stats){
        if(err){
            if(err.code === 'ENOENT') return callback(null, false);
            return callback(err);
        }
        return callback(null, (stats.isFile() || stats.isDirectory()));
    });
}

/**
 * Retuns fs entries(dirs and files) in a directory in an ARRAY matching the pattern
 * @Params:
 *   dir  :  String. Name of the directory
 *   options:{
 *     'recursive': true: Recursively call subdirectories to find dir and files. 
 *                  false:Find files only in given directory 
 *     'dirPattern': Directory pattern matching exp,
 *     'filePattern': file pattern matching exp
 *   }
 * callback    :  callback(err, dirs, files)
 *             callback function will have three arguments.
 *                "err", of node.js type "Error"
 *                "dirs", list of directories matching 'dirPattern'
 *                "files", list of files matching 'filePattern'
 */
exports.getFsEntries = function(dir, options,  callback){
  
  var absPath     = path.resolve(dir) + '/'
    , retFiles    = []
    , retDirs    = []
    , fqFile      = undefined
    , options     = options || {}
    , dirPattern  = options.dirPattern
    , filePattern  = options.filePattern
    , recursive = options.recursive || false
    ;

  // If pattern is string, convert pattern to regular expression  
  if(filePattern && typeof(filePattern) === 'string') filePattern = new RegExp(filePattern);
  if(dirPattern && typeof(dirPattern) === 'string') dirPattern = new RegExp(dirPattern);

  fs.readdir(absPath, function (err, files){
    if(err) return callback(err, retDirs, retFiles);

    if(files === undefined) 
        return callback(new Error("Invalid directory name. Dir:" + absPath), retDirs, retFiles);

    if(!files.length) return callback(null, retDirs, retFiles);

    async.forEachOfLimit(files, 1,
      function(file, index, next){
          fqFile = absPath + file;
          fs.stat(fqFile, function(err, stat){
              if(err) next(new Error('--->getFsEntries: Unable to get statistics of (' 
                            + fqFile +'). ' + err.message));
              else if(stat.isDirectory()){
                if(dirPattern && file.match(dirPattern)) retDirs.push(fqFile);
                if(recursive){
                  exports.getFsEntries(fqFile, options, function(err, result1, result2){
                    if(err) return next(err);
                    if(result1) retDirs = retDirs.concat(result1);
                    if(result2) retFiles = retFiles.concat(result2);
                    next();
                  });
                }
                else return next();
              }
              else if(stat.isFile() && filePattern && file.match(filePattern)){
                retFiles.push(fqFile);
                next();
              }
              else next();
            });
        },
        function(err){
          return callback(err, retDirs, retFiles);
        }
    );        
  });
}

/** 
 * Delete directory and its sub directories recursively if they are empty 
 * 
 * @param  {String} dir         Directory to be deleted
 * @param  {String} options     options = {
 *                                'deleteRootDir': true | false // default is true. true: delete passed 'dir' false: do not delete passed 'dir'
 *                              }
 * @param  {function} callback  callback(err)
 */
 exports.deleteEmptyDirs = function(dir, options, callback){

  var entryCount  = undefined
    , absPath     = path.resolve(dir) + FILE_SEP
  ;

  options = options || {'deleteRootDir': true}
  
  fs.readdir(absPath, function(err, files){
      if(err){
          if(err.code === 'ENOENT'){
              logger.debug('--->deleteEmptyDirs: Warning! ' + absPath +' not found.');
              return callback();
          }
          else return callback(err);
      }

      entryCount = files ? files.length : 0;

      logger.debug('-->deleteEmptyDirs: Path=' + absPath + ', entryCount=' + entryCount);
      if( entryCount === 0){
          if(!options.deleteRootDir) return callback();  
          logger.debug('-->deleteEmptyDirs: Deleting ' + absPath+ '...');
          fs.rmdir(absPath, function(err){
              if(err){
                  if(err.code === 'ENOENT'){
                      logger.debug('--->deleteEmptyDirs: Warning! ' + absPath +' not found.');
                      return callback();
                  }
                  else return callback(err);
              }
              return callback();
          });
          return;
      }

      async.each(files, function (file, next){
          file = absPath + file;
          fs.stat(file, function(err, stat){
              if(err){
                  if(err.code === 'ENOENT') logger.debug('--->deleteEmptyDirs: Warning! Unable to get statistics of file. ' + file + ' file not found.');
                  else logger.warn('--->deleteEmptyDirs: Unable to get statistics of file(' + file +') ' + err.message);
                  return next();
                  // return next(err);
              } 
              if(stat.isDirectory()){
                  exports.deleteEmptyDirs(file, {'deleteRootDir': true}, function(err){
                      if(err){
                          if(err.code === 'ENOENT') logger.debug('--->deleteEmptyDirs: Unable to delete directory. ' + file + ' dir not found.');
                          else logger.warn('--->deleteEmptyDirs: Unable to delete directory (' + file +'). ' + err.message);
                          // return next(err);
                      }
                      return next();
                  });
                  return;
              }
              if(stat.isFile()) return next()
              next();
          });
      },
      function(err){
          if(err) return callback(err);
        
          if(!options.deleteRootDir) return callback();  

          // Delete directory
          fs.readdir(absPath, function(err, files){
              if(err){
                  if(err.code === 'ENOENT'){
                      logger.debug('--->deleteEmptyDirs: Warning! ' + absPath +' not found.');
                      return callback();
                  }
                  else return callback(err);
              }
              entryCount = files ? files.length : 0;
              logger.debug('-->deleteEmptyDirs: Path=' + absPath+ ', entryCount='+ entryCount);

              if(entryCount !== 0)  return callback(); // dir. not empty, do not delete dir.

              logger.debug('-->deleteEmptyDirs: Deleting ' + absPath+ '...');
              fs.rmdir(absPath, function(err){
                  if(err){
                      if(err.code === 'ENOENT'){
                          logger.debug('--->deleteEmptyDirs: Warning! ' + absPath +' not found.');
                          return callback();
                      }
                      else return callback(err);
                  }
                  return callback();
              });
          });
      });
  });
};

/** 
 * Delete directory and its parent directories recursively if they are empty 
 * 
 * @param  {String} fromDir Starting directory to be deleted
 * @param  {String} toDir   Parent directory upto which empty directories are to be deleted
 * @param  {JSON}   options Not used now. (For later usage)
 */
exports.deleteEmptyParentDirs = function(fromDir, toDir, options, callback){

  var entryCount  = undefined;

  // Get absolute path and add file separator
  fromDir  = path.resolve(fromDir) + FILE_SEP
  toDir    = path.resolve(toDir) + FILE_SEP

  fs.readdir(fromDir, function(err, files){
      if(err){
          if(err.code === 'ENOENT'){
              logger.debug('--->deleteEmptyParentDirs: Warning! ' + fromDir +' not found.')
              return callback()
          }
          else return callback(err)
      }

      entryCount = files ? files.length : 0
      logger.debug('-->deleteEmptyParentDirs: Path=' + fromDir + ', entryCount:' + entryCount)

      if( entryCount <=0){
          if(fromDir === toDir) return callback()
          // no entries in this directory, delete directory and call _deleteEmptyParentDirs() with its parent directory 
          logger.debug('-->deleteEmptyParentDirs: Deleting ' + fromDir+ '...');
          fs.rmdir(fromDir, function(err){
              if(err){
                if(err.code === 'ENOENT'){
                  logger.debug('--->deleteEmptyParentDirs: Warning! ' + fromDir +' not found.');
                  return callback();
                }
                else return callback(err);
              }
              fromDir  = path.dirname(fromDir)
              exports.deleteEmptyParentDirs(fromDir, toDir, options, callback)                  
          });
          return 
      }

      async.each(files, function (file, next){
          file = fromDir + file;
          fs.stat(file, function(err, stat){
              if(err){
                if(err.code === 'ENOENT') logger.debug('--->deleteEmptyParentDirs: Warning! Unable to get statistics of file. ' + file +' file not found.');
                else logger.warn('--->deleteEmptyParentDirs: Unable to get statistics of file(' + file +') ' + err.message);
                next();
                // return next(err);
              } 
              else if(stat.isDirectory()){
                  exports.deleteEmptyDirs(file, {'deleteRootDir': true}, function(err){
                      if(err){
                          if(err.code === 'ENOENT') logger.debug('--->deleteEmptyParentDirs: Unable to delete directory. ' + file + ' dir not found.');
                          else logger.warn('--->deleteEmptyParentDirs: Unable to delete directory (' + file +'). ' + err.message);
                          // return next(err);
                      }
                      return next();
                  });
              }
              else if(stat.isFile()){
                  // next();
                  // dir has files and can not be deleted
                  return callback()
              }
              else next();
          });
        },
        function(err){
          if(err) return callback(err);

          // We have reached upto 'toDir', do not delete further up and return
          if(fromDir === toDir) return callback()
          
          // Read current directory (after removing empty dirs)
          fs.readdir(fromDir, function(err, files){
              if(err){
                  if(err.code === 'ENOENT'){
                      logger.debug('--->deleteEmptyParentDirs: Warning! ' + fromDir +' not found.');
                      return callback();
                  }
                  else return callback(err);
              }
              entryCount = files ? files.length : 0;

              if(entryCount !== 0)  return callback(); // dir. not empty, do not delete dir.

              logger.debug('-->deleteEmptyParentDirs: Deleting ' + fromDir+ '...');
              fs.rmdir(fromDir, function(err){
                  if(err){
                      if(err.code === 'ENOENT'){
                          logger.debug('--->deleteEmptyParentDirs: Warning! ' + fromDir +' not found.');
                          return callback();
                      }
                      else return callback(err);
                  }
                  fromDir  = path.dirname(fromDir)
                  exports.deleteEmptyParentDirs(fromDir, toDir, options, callback)                  
              });
          });
      });        
  });
}

/** 
 * Encode buffer to given 'outputEnc'
 * 
 * @param  {Buffer} buffer    input buffer to be encoded
 * @param  {JSON}   options   {
 *                              'inputEnc':   Optional. If not given, input encoding will be deteced
 *                              'outputEnc':  Optional. Default is UTF-8
 *                              'outputAsString':  Optional. Default is false. 
 *                                                 If true output is returned as JS string which is encoded in UTF-8, 
 *                                                 else output is returned as buffer encoded using 'outputEnc'
 *                            }
 * @param  {JSON}   options Not used now. (For later usage)
 */
// 4.1.25 Begin
// Modified to use chardet instead of detectCharacterEncoding module
exports.encodeBuffer = function(buffer, options){
  
  let retValue  = undefined
    , str       = undefined

  options = options || {}

  //If outputEnc not given, set outputEnc to 'UTF-8'
  options.outputEnc = options.outputEnc || 'UTF-8'

  //If inputEnc not given, detect input encoding
  if(!options.inputEnc){
    //retValue = detectCharacterEncoding(buffer)
    //if(!retValue || !retValue.encoding) return new Error('unable to detect input buffer encoding')
    //options.inputEnc = retValue.encoding
    retValue = chardet.detect(buffer)
    if(!retValue || retValue.length <=0) return new Error('unable to detect input buffer encoding')
    if(logger.isDebugEnabled()) logger.debug('--->encodeBuffer: Input Encoding: ' + retValue)
    options.inputEnc = retValue
  } 

  if(options.outputEnc.toLowerCase() === options.inputEnc.toLowerCase() && !options.outputAsString) return buffer;
  
  // Convert buffer to JS string
  str = iconv.decode(buffer, options.inputEnc)

  if(options.outputAsString) return str

  // convert JS string to buffer
  retValue = iconv.encode(str, options.outputEnc);
 
  return retValue
}
// 4.1.25 End

/** Get Auth. token.
 * For DMT env, returns JWT token else return config. server token
 * 
 * @param  {Response} 	res Response Object
 *         				res._Context = {
 *                    		login:{
 *         						tenant: 	<Tenant>,
 *         						username: 	<Username>,
 *         						password: 	<Password>
 *         					},
 *                          "retryOptions":{
 *                                          "default":{
 *									            "interval": 10000,      // 10 secs  <interval in milliseconds>,
 *									            "count": 5,		                  
 *									            "errorCodes" : [502]   // <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          },
 *                                          "oxs-turntable-server":{  // <Optional>
 *									            "interval": 10000,      // 10 secs  <interval in milliseconds>,
 *								            	"count": 5,		                  
 *								            	"errorCodes" : [502]   // <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          },
 *                                          "configserver":{  // <Optional>
 *									            "interval": 10000,      // 10 secs  <interval in milliseconds>,
 *								            	"count": 5,		                  
 *								            	"errorCodes" : [502]   // <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          }

 *							},
 *         			  }
 * @param  {Function} callback(err, auth_token) 
 *      For DMT env, returns JWT token else return config. server token
 */
exports.getAuthToken = function(res, callback){

  let envname       = process.env.OXSNPS_SERVER_PREFIX + "DMT_ENV"
    , dmtEnv        = process.env[envname] ? process.env[envname] : true
    , loginOptions  = undefined
    , auth_token    = undefined
    , pluginManager = require('oxsnps-core/pluginManager')
    //, url           = ''
    , techUser      = undefined
    , loginOptionsDup   = undefined
	, username		= undefined		
	, password		= undefined
    , servername    = (dmtEnv ? 'oxs-turntable-server' : 'configserver')
    // V4.1.17 Begin
    , retryOptions  = undefined 
    , tturl         = undefined
    , reqBody       = undefined
    , statusCode    = undefined
    , result        = undefined
    , httpUtils     = require('./httpUtils')  // we can not do this at the top since httpUtils.js calls utils.js
    //, defTimeout    = 20000
    //, timeout      = process.env.OXSECO_HTTP_TIMEOUT || defTimeout
    // V4.1.17 End

    // V4.1.18 Begin
// if timeout is string, convert it to a number
/*
  if(typeof timeout === 'string'){
    timeout = parseInt(timeout)
    if(isNaN(timeout)) timeout = defTimeout
  }
  */
    // V4.1.18 End
  if(typeof dmtEnv === 'string') dmtEnv = exports.stringToBoolean(dmtEnv)

  // getAuthToken is called by oxsnps-ordersystem, oxsnps-archive and ce.wac.export. these plugins does not pass res._Context.login and only pass res._Context.tenant
  // It is only a workaround, once those plugins are modified, this "if" will be removed
  if(!res._Context.login){
	res._Context.login ={'tenant': res._Context.tenant}	
  }
  
    // V4.1.17 Begin
    if(res._Context.retryOptions && res._Context.retryOptions[servername]) retryOptions = res._Context.retryOptions[servername]
    else if(res._Context.retryOptions && res._Context.retryOptions['default']) retryOptions = res._Context.retryOptions['default']
    else retryOptions = {
        "interval": 10000,
        "count": 5,
        "errorCodesDesc": "List of error codes for which retry should be done",
        "errorCodes": [
            "ECONNREFUSED",
            404,    // page not found
            502,    // bad gateway
            504     // gateway timeout
        ]
   }     // V2.0.5
   // V4.1.17 End
  
  async.series([
    // If not DMT Env, Get config. server token
    function(nextTask){
		if(dmtEnv) return nextTask()
		let resTmp 	= {'_Context':{'reqId': res._Context.reqId, 'tenant':res._Context.login.tenant, 'retryOptions': res._Context.retryOptions}}	 // V4.1.17
		if(logger.isDebugEnabled()) logger.debug('--->getAuthToken: Req.Id=' + res._Context.reqId + ', Getting config. server token, tenant: ' + resTmp._Context.tenant)
		pluginManager.callPluginService({name:'oxsnps-cs',service:'signInToTenant'}, resTmp, function(err, result){
			if(err) return nextTask(err)
			auth_token = result
			if(logger.isDebugEnabled()) logger.debug('--->getAuthToken: Req.Id=' + res._Context.reqId + ', Tenant: ' + res._Context.login.tenant + ', Config.Server Auth_token: ' + auth_token)
			nextTask()
		})
    },
	// V4.1.17 Begin
    function(nextTask){
        if(!dmtEnv) return nextTask()
       // V5.0.17/V4.1.29 Begin
       loginOptions = _getLoginOptions(res)
       if(loginOptions instanceof Error) return nextTask(err)       
       /*
        if(process.env.ENTRY_SERVICE_PREFIX && process.env.ENTRY_SERVICE_PREFIX.startsWith('http')) // ENTRY_SERVICE_PREFIX value startswith http do not add http prefix
            tturl = process.env.ENTRY_SERVICE_PREFIX    // this is done for testing it locally
        else tturl = 'http://' + (process.env.ENTRY_SERVICE_PREFIX || 'entry1/oxseedint') + '/'

        if(!tturl.endsWith('/')) tturl += '/'
        tturl += 'login/services/tt.auth/login'
        tturl += '?client=' + res._Context.login.tenant
        tturl = url.parse(tturl)

        /*
            ENTRY_SERVICE_PREFIX:https://apps-dev.oxseco.net/ox
            tturl ={
            protocol: 'https:',
            slashes: true,
            auth: null,
            host: 'apps-dev.oxseco.net',
            port: null,
            hostname: 'apps-dev.oxseco.net',
            hash: null,
            search: '?client=condor',
            query: 'client=condor',
            pathname: '/ox/login/services/tt.auth/login',
            path: '/ox/login/services/tt.auth/login?client=condor',
            href: 'https://apps-dev.oxseco.net/ox/login/services/tt.auth/login?client=condor' }


            ENTRY_SERVICE_PREFIX:entry1/oxseedint
            tturl = {
            protocol: 'http:',
            slashes: true,
            auth: null,
            host: 'entry1',
            port: null,
            hostname: 'entry1',
            hash: null,
            search: '?client=condor',
            query: 'client=condor',
            pathname: '/oxseedint/login/services/tt.auth/login',
            path: '/oxseedint/login/services/tt.auth/login?client=condor',
            href: 'http://entry1/oxseedint/login/services/tt.auth/login?client=condor' }
        * /

       loginOptions = {
            'protocol':             (tturl.protocol || process.env.PROTOCOL || 'http'),
            'hostname':             tturl.host,
            'method':               'POST',
            'path':                 tturl.path,
            'headers':              {'Content-type': 'application/json'},
            'rejectUnauthorized':   false,
            'respEncoding':         "utf-8",        
            'checkServerIdentity':  function(host, cert) {return true;},
            'timeout':              timeout, /* timeout * /
        }
        */
       // V5.0.17/V4.1.29 End
        // V4.0.11 Begin      
        if(res._Context.login.username){
            username    = res._Context.login.username   
            password    = res._Context.login.password 
        }
        if(!username || !password) {  // V4.1.31
			res._Context.retDefault = true
            techUser = exports.getOXSEcoTechUser(res)
			res._Context.retDefault = undefined	// reset value 
			if(!techUser) return nextTask(new Error('Unable to get JWT token.'))			
            if(techUser && techUser instanceof Error) return nextTask(new Error('Unable to get JWT token. ' + techUser.message))
            if(techUser){
                username    = techUser.username   
                password    = techUser.password 
            }/*
            //if(!username || !password) return nextTask(new Error('Unable to get JWT token. username or password to get JWT token is undefined/empty')) 
            else{
                // to be removed once either OXSECO_TECH_USER or OXSECO_DEFAULT_TECH_USER is defined in all server environments
                username    = 'support.user'
                password    = 'owbxlhVx5x' 
            }
			*/
        }
		if(!username || !password) return nextTask(new Error('Unable to get JWT token. username or password to get JWT token is undefined/empty')) 
        reqBody = {
            'client':   res._Context.login.tenant,
            'username': username,
            'password': password
        }        
        // V4.0.11 End
        loginOptionsDup 		= exports.clone(loginOptions)
        loginOptionsDup.reqBody	= exports.clone(reqBody)
        loginOptionsDup.reqBody.password = loginOptionsDup.reqBody.password.substring(0,5) +'*****'
    
        if(logger.isDebugEnabled()) logger.debug('--->getAuthToken: Req.Id=' + res._Context.reqId + ', Getting JWT token, Parms: ' + JSON.stringify(loginOptionsDup) + ', retryOptions: ' + JSON.stringify(retryOptions))
        httpUtils.postURL(loginOptions, reqBody, retryOptions, function (err, body, resp){
            statusCode = (err ? (err.statusCode || err.code) : 200)

            /*
            [2021 - 04 - 07T16: 29: 53.929][DEBUG]utils - --->getAuthToken: Req.Id = 210407162952944 - 89642, HTTP status code: 200, 
            response: {
                "token": "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpZCI6ImthdGphQG1hYXMuZGUiLCJjbGllbnQiOiJjb25kb3IiLCJyb2xlSWQiOiJzdXBlcnVzZXIiLCJsYXN0TG9nZ2VkSW5UaW1lIjoiMjAyMS0wNC0wN1QxMDo1NTo1NC4yMjBaIiwiY3NfYWNjZXNzX3Rva2VuIjoiY29uZG9yLTNiMWZhYzIwLTk3NjMtMTFlYi05OWJlLWU5NWIzMTZmNjkxMyIsImV4cGlyZXNJbiI6Mjg4MDAsImlhdCI6MTYxNzc5MzE5NSwiZXhwIjoxNjE3ODIxOTk1fQ.2sHGYV_MZUMrErkjR2z8V1MMATOUM0JBU1rKBsFNhss",
                "token_type": "Bearer",
                "refresh_token": "MGNGlkLv25vk5IiANxO9xyHgRQaSNzHAv99zCqhMVmyCAOQCiH6NX0zKuXpcXHk20f50FZzgfeZeZBJNPk1rvkcFALOxwPQjtkvltKbRmG7Q46UpdPiQ1UMYBTUOFBxSrkVK6Wpvx84voqQhJQMGtWIixc9HPGG5opiqNMHOTwHfrYOtxNtDvHgEaZtB2YuIWrJN22Re2lvCJoyeWxx8P25fULMVb3zejyjzi3NxCJpaNF6JtP9LaiLdw9GUIR9r",
                "success": true
            }
            */
            if(logger.isDebugEnabled()) logger.debug('--->getAuthToken: Req. Id=' + res._Context.reqId + ', HTTP status code: ' + statusCode + ', response: ' + body)

            if(err) return nextTask(new Error('Unable to get JWT token. HTTP status code: ' + statusCode + ', '+ err.message + ', req.options: ' + JSON.stringify(loginOptionsDup))) 
            
            if(!body) return nextTask(new Error('Unable to get JWT token. HTTP status code: ' + statusCode + ', Response body is null, req.options: ' + JSON.stringify(loginOptionsDup))) 

            try{
                result = exports.parseJSON(body)
            } catch (ex){
                return nextTask(new Error('Unable to get JWT token. Error on parsing JSON string from response body. ' +  ex.message + ', Response body: ' + body + ', Parms: ' + JSON.stringify(loginOptionsDup)))
            }            

            if(!result || !result.token || !result.token_type) 
                return nextTask(new Error('Unable to get JWT token. Token or token_type is missing. Response body : ' + body + ', Parms: ' + JSON.stringify(loginOptionsDup)))

            auth_token = result.token_type + ' ' + result.token
            if(logger.isDebugEnabled()) logger.debug('--->getAuthToken: Req.Id=' + res._Context.reqId + ', Tenant: ' + res._Context.login.tenant + ', JWT Token: ' + auth_token)
            nextTask()
        })
    },
    /*
    // If DMT Env, Get JWT token
    function(nextTask){
        if(!dmtEnv) return nextTask()

        if(process.env.ENTRY_SERVICE_PREFIX && process.env.ENTRY_SERVICE_PREFIX.startsWith('http')) // ENTRY_SERVICE_PREFIX value startswith http do not add http prefix
            url = process.env.ENTRY_SERVICE_PREFIX    // this is done for testing it locally
        else url = 'http://' + (process.env.ENTRY_SERVICE_PREFIX || 'entry1/oxseedint') + '/'

        if(!url.endsWith('/')) url += '/'
        url += 'login/services/tt.auth/login'
      
        loginOptions = {
            "url": url,
            "qs":{
            'client': res._Context.login.tenant
            },
            "timeout": process.env.OXSECO_HTTP_TIMEOUT || 20000,
            "rejectUnauthorized": false,
            "checkServerIdentity": function(){return true;},
            'headers': {
                'Content-type': 'application/json'
            }
        }
		// V4.0.11 Begin      
        if(res._Context.login.username){
            username    = res._Context.login.username   
            password    = res._Context.login.password 
        }
        else {
            techUser = exports.getOXSEcoTechUser(res)
            if(techUser && techUser instanceof Error) return nextTask(new Error('Unable to get JWT token. ' + techUser.message))
            if(techUser && techUser.username){
                username    = techUser.username   
                password    = techUser.password 
            }
            //if(!username || !password) return nextTask(new Error('Unable to get JWT token. username or password to get JWT token is undefined/empty')) 
            else{
                // to be removed once OXSECO_TECH_USER is defined
                username    = 'support.user'
                password    = 'owbxlhVx5x' 
            }
        }
        loginOptions.body = {
            'client':   res._Context.login.tenant,
            'username': username,
            'password': password
        }        
		// V4.0.11 End
        loginOptionsDup = exports.clone(loginOptions)
        loginOptionsDup.body.password = loginOptionsDup.body.password.substring(0,5) +'*****'

        loginOptions.body = JSON.stringify(loginOptions.body)

        if(logger.isDebugEnabled()) logger.debug('--->getAuthToken: Req.Id=' + res._Context.reqId + ', Getting JWT token Parms: ' + JSON.stringify(loginOptionsDup))
        request.post(loginOptions, function (err, response) {		// Refer to https://confluence.oxseco.net/confluence/display/OPD/JWT+Bearer+Token
            if(err) return nextTask(new Error('Unable to get JWT token. ' + err.message + ', Parms: ' + JSON.stringify(loginOptionsDup))) 
            
            if(response.statusCode !== 200) return nextTask(new Error('Unable to get JWT token. HTTP StatusCode:' + response.statusCode + ', Parms: ' + JSON.stringify(loginOptionsDup)))

            let loginResponse = JSON.parse(response.body)
            if(!loginResponse || !loginResponse.token || !loginResponse.token_type) 
            return nextTask(new Error('Unable to get JWT token. Token or token_type is missing. Response body:' + response.body + ', Parms: ' + JSON.stringify(loginOptionsDup)))

            auth_token = loginResponse.token_type + ' ' + loginResponse.token
            if(logger.isDebugEnabled()) logger.debug('--->getAuthToken: Req.Id=' + res._Context.reqId + ', Tenant: ' + res._Context.login.tenant + ', JWT Token: ' + auth_token)
            nextTask()
        })
    }
    */
	// V4.1.17 End
  ],  function(err){
      return callback(err, auth_token)
    }
  )
}

// V4.0.11 Begin
/**
 * Get tech user information for a tenant defined in env. variable "OXSECO_TECH_USER"
 * @param  {response}  res - Response Object
 *                     res._Context:{
 *                      	tenant: <tenant>
 *                      	// retDefault: true: If user/password not found for the given tenant in techinai user list, 
 *                      	// return default user/pwd. This user/pwd exists for all tenants. And this flag is only a temp. workaround till tech user list issues are fixed
 *                      	retDefault: true | false.  If user/password not found for the given tenant, return default user/pwd. This user/pwd exists for all tenants
 *                     }
 * @retun  {JSON || Error object}   On success, a JSON with tech user infor. On Error returns Error object
 *  techUser = {
 *      username: <username>,
 *      password: <password>
 *  }
 */
exports.getOXSEcoTechUser = function(res){

    let techUser    = process.env['OXSECO_TECH_USER']
      , tenant      = undefined
      , jsonTmp     = undefined
	  , envvar		= "OXSECO_DEFAULT_TECH_USER"	
	  , retValue	= undefined

    //if(!res || !res._Context) return // tenant for which technica user info is needed is not passed 
    //if(!techUser) return        // OXSECO_TECH_USER is not defined. It is a case for Hallesch or for On Premise customers
	
	if(res && res._Context) tenant = res._Context.login && res._Context.login.tenant || res._Context.tenant
  if(!tenant) return new Error('--->getOXSEcoTechUser: Missing tenant parameter.') // tenant for which technica user info is needed is not passed 

	if(techUser){
		if(typeof techUser === 'string'){
			try{
				jsonTmp = require(techUser)
				/*
				module.exports = {
					<tenant1> :{
						"username":"support.user",
						"password":"owbxlhVx5x"
					},
					<tenant2> :{
						"username":"support.user",
						"password":"owbxlhVx5x"
					}	
				} 
				*/           
				if(!jsonTmp) new Error('Unable to parse file: ' + techUser)
			}catch(err){
				return new Error('Unable to parse file: ' + techUser + ', ' + err.message)
			}
		}
		else jsonTmp = techUser
		
    if(jsonTmp) retValue = jsonTmp[tenant || "default"]   // V4.1.31
    }
    // V4.1.21 Begin
	// if there is no techinai user list defiend for tennats, return default user/pwd from the "OXSECO_DEFAULT_TECH_USER" env.variable (JS secret file)
	else if(process.env[envvar]){
		if(process.env[envvar] && typeof process.env[envvar] === 'string'){
			// OXSECO_DEFAULT_TECH_USER = {'username': <username>, 'password': <password>, 'expire-in': ''}	=> secret file
			retValue = exports.parseJSON(process.env[envvar])
			if(!retValue){
				logger.warn('--->getOXSEcoTechUser: Req. Id=' + res._Context.reqId + ', Unable to parse JSON string: ' + process.env[envvar] + ', of an env. variable: ' + envvar)
				return new Error('--->getOXSEcoTechUser: Req. Id=' + res._Context.reqId + ', Unable to parse JSON string: ' + process.env[envvar] + ', of an env. variable: ' + envvar)
			}
		} 
		else retValue = process.env[envvar]
    }
	if(!retValue && res._Context.retDefault === true){
		retValue = {
			// to be removed once either OXSECO_TECH_USER or OXSECO_DEFAULT_TECH_USER is defined in all server environments
			'username': 'support.user',
			'password':	'owbxlhVx5x',
			'expire-in': -1	
		}
    }
    // V4.1.21 End
	return retValue
}
// V4.0.11 End

//Set the Log Level
exports.log4jsSetLogLevel = _log4jsSetLogLevel
exports.getAppLogLevel = _getAppLogLevel

exports.networkInterfaces = _networkInterfaces      // v5.0.9

/**
 * Search for chars defined in utf82hex and replace with %HH representation
 * @param  {String} str input string
 * @return {String}     String with %HH valuesfor special chars defined in utf82hex
 */
/*
exports.replaceUTF8withHex = function(str){

  var ret = '';
  for(var i = 0; i < str.length; i++){
    //console.log('"' + str[i] + '"=>'  + utf82hex[str[i]]);
    if(utf82hex[str[i]]){
      ret = ret + '%' + utf82hex[str[i]].toUpperCase();
    }
    else ret = ret + [str[i]];
  }
  return ret;
};
*/

// V4.1.24 Begin

/**
 * Detect encoding of a buffer or filename
 * @param {Buffer|String}   inputDoc    Buffer or filename
 * @param {JSON}            options     chardet options. Refer to https://github.com/runk/node-chardet. Ex: { sampleSize: 32 }    
 * @param {function}        callback    callback(null|err, encoding)
 */
exports.detectEncoding = function(inputDoc, options, callback){
    
    let result = undefined

    if(typeof options === 'function'){
        callback = options
        options = undefined
    }
    options = options || {}

    // inputDoc is buffer
    if(Buffer.isBuffer(inputDoc, options)){ 
        result = chardet.detect(inputDoc)
        return callback(null, result)        
    }

    // inputDoc is Filename
    chardet.detectFile(inputDoc, options)
        .then(result => {
            return callback(null, result)
        })
        .catch(err => { 
	    	return callback(err)
	    })        
 }
// V4.1.24 End

// V5.0.10 Begin
/**
 * Get IP Addresses
 * @return {Array}     Return the IP Addresses as an array
 */
exports.getIPAddresses = function(){

    let ifaces = os.networkInterfaces()
     , addr = []
  
    Object.keys(ifaces).forEach(function(ifname){
        if(!ifname || !ifaces[ifname]) return
        ifaces[ifname].forEach(function(iface){
            if ('IPv4' !== iface.family || iface.internal !== false) return; // skip over internal (i.e. 127.0.0.1) and non-ipv4 addresses
            addr.push(iface.address)
        })
    })
    return addr
  }
  // V5.0.10 End

/**************** PRIVATE FUNCTIONS ***************/

/**
 * Set the Log Level for the given logger object
 *
 * @param  {logger}   logger    Log4j logger
 * @param  {String}   level     level to set
 */
function _log4jsSetLogLevel(logger, level) {
  level = level || "INFO"
  if (logger.setLevel)  // Log4j 1.1.1 version supports setLevel function.
    return logger.setLevel(level)
  logger.level = level  // Log4j >= 3.0.6 version supports level property.
}

/**
 * Get the Log Level for the given log object
 *
 * @param  {JSON}    logconf    Application log object
 */
function _getAppLogLevel(log){
  if (!log) return undefined
  if (log.categories && log.categories.default) return log.categories.default.level // Log4js >= 3.0.6
  return log.level // Log4js 1.1.1
}


/**
 * _networkInterfaces:  Return the Network Interfaces
 * @return {String}     Network Interfaces as string
 */
function _networkInterfaces(){

  //console.log('----------------->_networkInterfaces...');

  var ifaces = os.networkInterfaces()
    , alias = 0
    , networkInterfaces = ''
    ;


  Object.keys(ifaces).forEach(function (ifname){

    alias = 0;

    ifaces[ifname].forEach(function (iface) {
      if ('IPv4' !== iface.family || iface.internal !== false) {
        // skip over internal (i.e. 127.0.0.1) and non-ipv4 addresses
        return;
      }

      if (alias >= 1) {
        // this single interface has multiple ipv4 addresses
        //console.log(ifname + ':' + alias, iface.address);
        if(networkInterfaces.length > 0) networkInterfaces += ', ' + ifname + ': ' + alias + ', ' + iface.address + ''; // v5.0.9
        else networkInterfaces += ifname + ': ' + alias + ', ' + iface.address + '';
      } else {
        // this interface has only one ipv4 adress
        //console.log(ifname, iface.address);
        if(networkInterfaces.length > 0) networkInterfaces += ', ' + ifname + ': ' + iface.address + '';    // v5.0.9
        else networkInterfaces += ifname + ': ' + iface.address + '';
      }
    });
  });
  return networkInterfaces;
}

/*
  Delete the dir and its content if the dir date is older than (today's date + interval)
    dirs: array of dirs
    now:  today's date
    interval: if the dir date is older than (today's date + interval (in seconds)) then delete the dir.
*/
function _deleteDir(dirs, now, interval){
  
  // Any dir?
  if (dirs.length <= 0){
    return; // over since array is empty
  }

  // Get the first dir to check
  var dir = dirs[0]; 

  // If no date passed, try to extract it from the name of the dir
  // works only for /opt/OTS/nodeDocServer/temp/140912110440272-78578
  if (dir.date === undefined){

    // Extract the relative dir out of the fully qualified dir (the last 21 digits)
    // /opt/OTS/nodeDocServer/temp/140912110440272-78578
    var relDir = dir.name.substring(dir.name.length - 21, dir.name.length);

    // Extract the date from the relative dir
    dir.date = new Date(20 + exports.padWithZeros(relDir.substring(0,  2), 2),    // year (4 digits year)
                             exports.padWithZeros(relDir.substring(2,  4)-1 ,2),  // month (month starts with 0)
                             exports.padWithZeros(relDir.substring(4,  6), 2),    // day
                             exports.padWithZeros(relDir.substring(6,  8), 2),    // hours
                             exports.padWithZeros(relDir.substring(8,  10), 2),   // minutes
                             exports.padWithZeros(relDir.substring(10, 12), 2),   // seconds
                             exports.padWithZeros(relDir.substring(12, 15), 3)    // milliseconds
                        );
  }

  // If dir is older that interval seconds, then delete it
  if (exports.dateDiff('s', dir.date, now) >= interval){
    //logger.debug(__filename, '_deleteDir: deleting ' + dir.name + ', date: ' + dir.date + '...');
    try{
      exports.deleteDirSync(dir.name, true);  // false means delete the content of the temp/ dir but NOT the the temp/ dir
      dirs.shift();         // remove dir from array
    }
    catch(err){
      //logger.debug(__filename, '_deleteDir: Error when deleting ' + dir.name + ', Message: ' + err.message);
    }
      return _deleteDir(dirs, now, interval); // delete the next dir in the array
  }
}

/**
 * Low level function to create async. a Symbolic / Soft Link
 * @param  {String}   srcpath  Source Path
 * @param  {String}   dstpath  Destinattion path
 * @param  {String}   type     'dir', 'file', or 'junction' (default is 'file')
 * @param  {Function} callback A callback function
 *                               callback function will have one argument "err" of node.js type "Error".
 *                               In case of success, "err" will be null.
 *                               In case of error, "err" will have information about error occurred on link creation
 * @return {}
 * 
 */
function _symlink(srcpath, dstpath, type, callback){  

  type = type || 'file';      // assert type

  // Create a Symbolic / Soft Link from tsPath (<cachedir>/objects) to the temp dir (res._Context.tempDir)
  fs.symlink(srcpath, dstpath, type, callback);
} 

/** This code would not work since there is no guranteed order in which keys in JSON are returned
 *  Refer to http://book.mixu.net/node/ch5.html
 */
/*
exports.getOutputJSONFilename_del = function(res){
  
  var req        = res._Context.req
  , jsonFilename = ""
  , otherProps   = ""
  , keyTmp       = ""
  , value        = ""
  ;
  
  var prefix={
    "pngcompressionfactor": "pcf",  
    "rotation":    "r",
    "dskew":       "d",
    "max_width":   "mw",
    "max_height":  "mh",
    "antialias":   "aa",
    "jpgquality":  "jq",
    "resolution":  "pr",
    "formattype":  "ft",
    "jpegquality": "jq",
    "outputsize":  "os",
    "outputscale": "oscale",
    "color":       "c"
  };

  for(var keyTmp in req.props){
    value = req.props[keyTmp] + "";
    if(value.toLowerCase() === 'off' || value.toLowerCase() === 'no'){
      //do nothing  
    }
    else if(value.toLowerCase() === 'on' || value.toLowerCase() === 'yes'){
      otherProps += '-' + prefix[keyTmp];
    }
    else{
        otherProps += '-' + prefix[keyTmp] + value;
    }
  }

  // Set the Output  filename (<docId dir>/<docId>-<pageNr>.<res._Context.outputType>)
  jsonFilename = res._Context.docId + '/' + res._Context.docId + '-' + res._Context.pageNr  + otherProps + '-' + res._Context.outputType + '.json';
  return jsonFilename;  
}
*/

// V109 End
// V3.0.35 Begin
/**
 * Check file accessibility
 * @param  {Response} 	res Response Object
 *         				res._Context = {
 *                    		fa:{
 *         						path: 	        Fully qualified file or directory,
 *         						mode            File accessibility constant.  Refer to https://nodejs.org/api/fs.html#fs_fs_access_path_mode_callback
 *         						retryOptions:   Optioanl. Retry options. retryOptions={ 'interval':<interval in milliseconds>, 'count':<count>}
 *         					}
 *         			  }
 * @return  {function}   callback(err)   If file is accessible <err> is null.  If file is not accessible <err> object will be passed back
 */
exports.checkFileAccess = function(res, callback){

  let retryOptions = res._Context.fa.retryOptions

  // set default retryOptions, if not passed as parameter
  retryOptions             = retryOptions || {}
  retryOptions.interval    = retryOptions.interval     || 1000        // 1000 milli seconds
  if(retryOptions.count === undefined)   retryOptions.count  = 5

  fs.access(res._Context.fa.path, res._Context.fa.mode, function(err){
    if(!err) return callback()

    // File is not accessible even after waiting for specified time
    if(retryOptions.count <= 0) return callback(new Error(res._Context.fa.path + ' is not accessible. ' + err.message))

    // Error occured. log warn message
    logger.warn('--->checkFileAccess: ' + (res._Context.reqId ? ('Req.Id=' + res._Context.reqId +', ') : '') +
    'Path:' + res._Context.fa.path + ' is not accessible yet. ' + err.message + ', scheduled to check again after ' + retryOptions.interval + ' milliseconds. Remaining retry count:'  + retryOptions.count)

    // File not found  or File is not accessible, so call checkFileAccess after a delay
    -- retryOptions.count
    res._Context.fa.retryOptions = retryOptions
    setTimeout(function(){
          exports.checkFileAccess(res, callback)
        },
        retryOptions.interval
    )
  })
}
// V3.0.35 End

// V4.1.19 Begin  
/**
 * Return additional log parameters 
 * @param  {Response} 	res Response Object
 *         				res._Context = {
 *                    		log:{
 *         						labels:	 [Array of labels to add log statements],
 *         						context: JSON containng values for lables,
 *         					},
 *         			  }
 * @return  Returns additional log parameters as JSON
 */
exports.getAddlLogParms = function(res){

	let glabels = {}
	
	if(!res || !res._Context || !res._Context.log || !res._Context.log.labels || !res._Context.log.context) return glabels
    
	res._Context.log.labels.forEach(function(label){
        if(!label) return
		glabels[label] = res._Context.log.context[label] || ''
    })
	//logger.info('--->getAddlLogParms: glabels: '+ JSON.stringify(glabels))
	return {'glabels': glabels} // V4.1.20
}
// V4.1.19 End

// V5.0.6 Begin
/**
 */
exports.isHTTPRoute = function(route){
    let method = route.method || 'get' 

    method = method.toLowerCase()

    switch(method){
        // refer to https://www.w3.org/Protocols/rfc2616/rfc2616-sec9.html
        // https://www.tutorialspoint.com/http/http_methods.htm
        case 'get':
        case 'post':
        case 'delete':
        case 'put':
        case 'patch':		// to apply partial modifications to a resource.
        case 'head':
        case 'options':
        case 'trace':
        case 'connect':
            return true
        default: 
            return false
    }
}
// V5.0.6 End

// V5.0.17/V4.1.29 Begin
function _getLoginOptions(res){

  //V5.0.18 - Begin
  //let servicePath = 'login/services/tt.auth/login?client=' + res._Context.login.tenant
  let servicePath = (process.env.TT_LOGIN_URL_SUFFIX || 'login/services/tt.auth/login') + '?client=' + res._Context.login.tenant
  //V5.0.18 - End
    , tturl       = undefined
    , tmp         = undefined
    , loginOptions= undefined
    , defTimeout    = 20000
    , timeout      = process.env.OXSECO_HTTP_TIMEOUT || defTimeout
    // V4.1.17 End

    // V4.1.18 Begin
    // if timeout is string, convert it to a number
    if(typeof timeout === 'string'){
        timeout = parseInt(timeout)
        if(isNaN(timeout)) timeout = defTimeout
    }

    if (process.env.TT_HOST_IP_CONTEXT){
        /**
            * process.env.TT_HOST_IP_CONTEXT could come in two formats
            * 1. string: <hostname>:<port>
            * 2. JSON: "{'hostname': <hostname>, 'port':<port>, 'context':<context>, 'timeout':<timeout>}""
        */
        try{
            loginOptions = JSON.parse(process.env.TT_HOST_IP_CONTEXT)
        }
        catch(err){
            // value of process.env.TT_HOST_IP_CONTEXT could be a string in format <hostname>:<port>, so ignore the error
            if(logger.isDebugEnabled()) logger.debug('--->_getLoginOptions: Error parsing env var: TT_HOST_IP_CONTEXT. value: ' + process.env.TT_HOST_IP_CONTEXT + ', ' + err.message)
        }
        if(!loginOptions){
            // value of process.env.TT_HOST_IP_CONTEXT could be a string in format <hostname>:<port>, so split and get host.port
            tmp = process.env.TT_HOST_IP_CONTEXT.split(':')
            loginOptions = {
                'hostname':   tmp[0], 
                'port':   tmp[1],
            }
        }

        loginOptions.context = loginOptions.context || '/'
        if(!loginOptions.context.startsWith('/')) loginOptions.context = '/' + loginOptions.context
        if(!loginOptions.context.endsWith('/')) loginOptions.context += '/'
        loginOptions.path = loginOptions.context + servicePath

        loginOptions.protocol           = loginOptions.protocol || process.env.PROTOCOL || 'http'
        loginOptions.method             = 'POST'
        loginOptions.headers            = {'Content-type': 'application/json'}
        loginOptions.rejectUnauthorized = false
        loginOptions.respEncoding       = loginOptions.respEncoding || "utf-8"
        loginOptions.checkServerIdentity= function(host, cert) {return true;}
        loginOptions.timeout            = loginOptions.timeout || timeout /* timeout */
        return loginOptions
    }

    if(process.env.ENTRY_SERVICE_PREFIX && process.env.ENTRY_SERVICE_PREFIX.startsWith('http')) // ENTRY_SERVICE_PREFIX value startswith http do not add http prefix
        tturl = process.env.ENTRY_SERVICE_PREFIX    // this is done for testing it locally
    else tturl = 'http://' + (process.env.ENTRY_SERVICE_PREFIX || 'entry1/oxseedint') + '/'

    if(!tturl.endsWith('/')) tturl += '/'
    tturl += servicePath
    tturl = url.parse(tturl)

    /*
        ENTRY_SERVICE_PREFIX:https://apps-dev.oxseco.net/ox
        tturl ={
        protocol: 'https:',
        slashes: true,
        auth: null,
        host: 'apps-dev.oxseco.net',
        port: null,
        hostname: 'apps-dev.oxseco.net',
        hash: null,
        search: '?client=condor',
        query: 'client=condor',
        pathname: '/ox/login/services/tt.auth/login',
        path: '/ox/login/services/tt.auth/login?client=condor',
        href: 'https://apps-dev.oxseco.net/ox/login/services/tt.auth/login?client=condor' }


        ENTRY_SERVICE_PREFIX:entry1/oxseedint
        tturl = {
        protocol: 'http:',
        slashes: true,
        auth: null,
        host: 'entry1',
        port: null,
        hostname: 'entry1',
        hash: null,
        search: '?client=condor',
        query: 'client=condor',
        pathname: '/oxseedint/login/services/tt.auth/login',
        path: '/oxseedint/login/services/tt.auth/login?client=condor',
        href: 'http://entry1/oxseedint/login/services/tt.auth/login?client=condor' }
    */

    loginOptions = {
        'protocol':             (tturl.protocol || process.env.PROTOCOL || 'http'),
        'hostname':             tturl.host,
        'method':               'POST',
        'path':                 tturl.path,
        'headers':              {'Content-type': 'application/json'},
        'rejectUnauthorized':   false,
        'respEncoding':         "utf-8",        
        'checkServerIdentity':  function(host, cert) {return true;},
        'timeout':              timeout /* timeout */
    }
    return loginOptions
}
// V5.0.17/V4.1.29 End

// 5.0.32 Begin
/**
 * Get file stat with retry options
 * @param  {Response} 	res Response Object
 *         				res._Context = {
 *                    		fs:{
 *         						path: 	        Fully qualified file or directory,
  *         					retryOptions:   Optioanl. Retry options. retryOptions={ 'interval':<interval in milliseconds>, 'count':<count>}
 *         					}
 *         			  }
 * @return  {function}   callback(err)   
 */
exports.getFileStat = function(res, callback){

	let retryOptions = res._Context.fs.retryOptions || {}
	  , retErr		 = undefined

	// set default retryOptions, if not passed as parameter
	retryOptions.interval    = retryOptions.interval     || 1000        // 1000 milli seconds
	if(retryOptions.count === undefined)   retryOptions.count  = 5

	fs.stat(res._Context.fs.path, function(err, stat){
		if(!err) return callback(null, stat)
		
		if(retryOptions.attempts === undefined) retryOptions.attempts = 1
		else retryOptions.attempts++

		// File is not accessible even after waiting for specified time
		if(retryOptions.attempts >= retryOptions.count){
			retErr = new Error('fs.stat failed, Path: ' + res._Context.fs.path + ', Error code: ' + err.code + ', Msg: ' + err.message + ', Retry count: ' + retryOptions.attempts)
			retErr.code = err.code
			return callback(retErr)
		}

		// Error occured. log warn message
		logger.warn('--->getFileStat: ' + (res._Context.reqId ? ('Req.Id=' + res._Context.reqId +', ') : '') +
			'fs.stat failed, Path: ' + res._Context.fs.path + ', Error code: ' + err.code + ', Msg: ' + err.message + ', Attempt: ' + (retryOptions.attempts +1) + ', scheduled after ' + retryOptions.interval + ' milliseconds.')

		// File not found  or File is not accessible, so call getFileStat after a delay
		res._Context.fs.retryOptions = retryOptions
		setTimeout(function(){
			  exports.getFileStat(res, callback)
			},
			retryOptions.interval
		)
	})
}
// 5.0.32 End