/**-----------------------------------------------------------------------------
 * index.js: 	Node.js module to handle request processing
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2021 by Maas Holding GmbH
 * Email     :  support@maas.de
 * 
 * History
 *  V5.0.2	12.11.2021	Added exports.heapdump(req, res)
 *  V5.0.0	06.10.2021	OTS-3151: Moves core services definition to conf/oxsnps-core.js
 *  V4.1.0	18.02.2020  OTS-2692  Remove deploy services from oxsnps-core since oxsnps-deploy will take care of deploy services
 *  					Refer to ../server.js
 *  V3.0.39	09.08.2019  OXS-9525  Extend to complete the running jobs before closing the application
 *  V3.0.16 14.02.2018  (OTS-2049)  Extended to do application specific tasks after starting the server  
 *  V3.0.13 11.12.207 	(OTS-1996) Extended core with softquit functionality to gracefully stop server
 *  V102   	26.01.2015	Added the pluginManager services                        
 *  V101   26.08.2014  	Changes done
 *                     		1. Ensure 'init' function modified to return only after 
 *                        	   initialization is over through 'callback' function
 *                     		2. Extended to include version information about MSOffice 
 *                        	   Converter Wrapper service and PDF Converter Wrapper
 *                        	   service in response to version request 
 *                     		3. Added a 'quit' function to handle 'quit' request. 
 *                        	   'quit' function will take care of stopping java services. 
 *                     		4. Implemented MSofficeConverter service
 *                     		5. Implemented PDF MergeAndSecure service
 *                     		6. Implemented A2W Async. Transformation service 
 *                     		7. Implemented Job Status request service
 *                     		8. Extended to send proper SOAP response in case of errors
 *                     		9. In case of A2W Server error, parsed error html response 
 *                        	   to get error information and built error SOAP response 
 *                        	   for sync. and async. transformations.
 *  V100   09.08.2014  Initial release
 *----------------------------------------------------------------------------*/
'use strict';

var MODULE_NAME 	= 'core'
  , MODULE_VERSION 	= '5.0.0'
  , fs 				= require('fs')
  , helmet			= require('helmet')  
  , path 			= require('path')
  , log4js 			= require('log4js')
  ,	async 			= require('async')
  , util 			= require('util')
  , url     		= require('url')
  , npsServer		= require('./server')  
  , npsDir			= npsServer.npsDir
  , npsConf 		= npsServer.npsConf				// server-conf.js of oxs-xxx-server
  , npsTempDir 		= npsConf.tempDir
  , pluginManager 	= require('./pluginManager')
  , cronjob 		= require('./cronjob')
  , utils 			= require('./helpers/utils')
  , httpUtils		= require('./helpers/httpUtils')
  , expressAppUtils = require('./helpers/expressAppUtils')
  , shutdown		= require('./shutdown')
  , pluginJSfilename= 'oxsnps.js'
  , coreModules 	= require('./conf/oxsnps-core.js') // add any core service as '/appservices/<coreModule>/<service>'
  /*
  	coreModules = [
			{	// PluginManager services
				name: 	    'pluginManager',
				baseRoute: 	'/appservices/pm',
				routes: [
					{method: 'get', 	path: '/appservices/pm/version', 			service: 'version'}, 
					...
				]
			},
			{	// Cronjob services
				name: 	    'cronjob',
				baseRoute: 	'/appservices/jobs',
				routes: 	[...]
			},
			{	// Cache services
				name: 	    'cache',
				baseRoute: 	'/appservices/cache',
				routes: 	[...]
			}
		]
  */

const v8 = require('v8') // V5.0.2 

// Get Logger
var logger = log4js.getLogger(MODULE_NAME);
utils.log4jsSetLogLevel(logger, (utils.getAppLogLevel(npsConf.log) || 'INFO'));

// Export Module Version
exports.version = MODULE_VERSION;

// V5.0.35 Begin
/**
 * OXS-15459: CSP headers for NodeJs express application
 * Set content security policies in response header when given in server-conf.js
 */
// V4.1.33 Begin
exports.setResponseHeaders = function(){

	let resHeaders	= npsConf.responseHeaders
      , envvar		= undefined
      , app 	    = npsServer.expressApp

	if(!npsConf.responseHeaders) return 	// response headers are not configurued so just return

	if(typeof npsConf.responseHeaders === 'string'){
		envvar = npsConf.responseHeaders
		logger.info('--->setResponseHeaders: Env.var: ' + envvar + ', Value: ' + process.env[envvar])
		if(!process.env[envvar]) return // response headers are not configurued so just return

		try{
			resHeaders = JSON.parse(process.env[envvar])
		}catch(err){
			return new Error('Unable to parse environment variable: ' + envvar + ', reason: ' + err.message + ', value: ' + process.env[envvar] )
		}
		if(resHeaders === null) return new Error('Unable to parse environment variable: ' + envvar + ', value: ' + process.env[envvar])
	} 

	if(resHeaders){
		logger.info('--->setResponseHeaders: Response headers for security are set. Header Value: ' + JSON.stringify(resHeaders))
		app.use(helmet(resHeaders))
	}
	return undefined
}
// V4.1.33 End
// V5.0.35 End

/**
 * init: Initializes Server services 
 * @Params:
 * callback    :  A callback function that would be called after initialization is over
 *                callback function will one argument "err", of node.js type "Error"
 *                In case of success,
 *                    "err" will be null.
 *               In case of error,
 *                  "err" will have information about intialization error
 */
exports.init_old = function(callback){
	
	logger.debug('init: Initializing routes...');

	var self 	= this
	  , app 	= npsServer.expressApp
	  , options = {}
	  , envvar  = undefined

	//V5.0.15 Begin
	// Get plugin list from env. variable if given
	if(typeof npsConf.pluginList === "string"){
		envvar = npsConf.pluginList
		npsConf.pluginList = utils.parseJSON(process.env[envvar])
		if(npsConf.pluginList === null) return callback(new Error('Error on parsing environment variable: ' + envvar + ', value: ' + process.env[envvar]))
	}
	//V5.0.15 End

		// Set up default parms used by cleanTempDir
	if(npsConf.tempDirProps){
		// Set up the cron job options
		npsConf.tempDirProps.cronTime 	= npsConf.tempDirProps.cronTime || '00 00 * * * *'; // Run cron at 0th min of every hour

		// Set up options
		npsConf.tempDirProps.interval 	= npsConf.tempDirProps.interval || 30;  // delete files older than 30 mins
		npsConf.tempDirProps.unit 		= npsConf.tempDirProps.unit ||'n';   // n => minutes
	}
	else{
		// Set up default options
		//npsConf.tempDirProps = {'plugin':'', 'service': '', 'parms':{'parm1':''}, cronOptions:{'cronTime': '00 00 * * * *', 'interval': 30, 'unit': 'n'}};
		npsConf.tempDirProps = {'cronTime': '00 00 * * * *', 'interval': 30, 'unit': 'n'};
	}

	// Clean the temp dir
	exports.cleanTempDir(function(err){
		// Error already logged by cleanTempDir. Ignore the error since we are deleting temp. directories
		//if(err) return callback(err);	

		// Initialize the cronjob module
		cronjob.initialize(function(err){
			if(err) return callback(err)

			// Set up a Cron Job to clean the temp dir
			options.name         = npsTempDir
			options.cronFunction = self.cleanTempDir
			self.function 		 = 'oxsnps-core/index.js->cleanTempDir' // (Optional) a String to identify the cronFunction
			options.cronOptions  = {'start': true, 'context': self}
			options.cronTime     = npsConf.tempDirProps.cronTime

			// Add the Cron job
			logger.debug('--->_createAndStartCronJob: Creating cron job to clean the temp directory (' + npsTempDir + ')...')
			cronjob.addCronJobService(options, function(err){
				if(err) return callback(err)
			})
			// V4.1.0 Begin
            if(npsConf.middlewares){
            	// Add middlewares given in server-conf.js
                expressAppUtils.addMiddlewares({router:app, middlewares:npsConf.middlewares, moduleName:'oxsnps-core'}, function(err){
                    if(err) return callback(err)
                })
            }
			// V4.1.0 End            
			// Enable the core services
			async.each(coreModules,
				function(coreModule, nextTask){

					// Load the core module
					let error = _loadCoreModule(coreModule) 
					if(error) return nextTask(error)
					expressAppUtils.enableRoutes(coreModule.handle, coreModule.routes, function(err){
						if(err) return nextTask(err)
						nextTask()
					})
				},
				function(err){
					if(err){
						logger.error('init: Initializing core services failed, Reason: ' + err.message)
						return callback(err)
					}
				}
			)

			// Feed application specific configuration files, if exist
			//!!!TODO: We should call readPluginConfFiles first for the base plugins and then for the custom plugins
			//!!!TODO: Just pass the path to readPluginConfFiles where to find the conf files
			pluginManager.readPluginConfFiles(function(err){
				if(err) return callback(err);
				logger.debug('-->init:  plugins confn.: ' + JSON.stringify(npsConf.plugins))
				// Enable plugins
				pluginManager.enablePlugins(function(err){
					if(err) return callback(err);
					// get application startup plugin
					if(npsConf.appStartPlugin){
						//if application startup plugin exists, do application specific startup tasks
						_doAppStartupTasks(function(err){
							if(err) return callback(err);
							// Activate plugins
							return pluginManager.activatePlugins(callback);
						});
					}
					else pluginManager.activatePlugins(callback); 	// Activate plugins
				});
			})
		})
	})
}

// V5.0.16 Begin
exports.init = function(callback){
	let app 	= npsServer.expressApp
	  , envvar  = undefined
      , result  = undefined
	
	logger.debug('init: Initializing routes...')

	//V5.0.15 Begin
	// Get plugin list from env. variable if given
	if(typeof npsConf.pluginList === "string"){
		envvar = npsConf.pluginList
		npsConf.pluginList = utils.parseJSON(process.env[envvar])
		if(npsConf.pluginList === null) return callback(new Error('Error on parsing environment variable: ' + envvar + ', value: ' + process.env[envvar]))
	}
	//V5.0.15 End

	// Set up default parms used by cleanTempDir
	if(npsConf.tempDirProps){
		// Set up the cron job options
		npsConf.tempDirProps.cronTime 	= npsConf.tempDirProps.cronTime || '00 00 * * * *'; // Run cron at 0th min of every hour

		// Set up options
		npsConf.tempDirProps.interval 	= npsConf.tempDirProps.interval || 30;  // delete files older than 30 mins
		npsConf.tempDirProps.unit 		= npsConf.tempDirProps.unit ||'n';   // n => minutes
		// V4.1.34
		npsConf.tempDirProps.timeToUse     = npsConf.tempDirProps.timeToUse       || {}
		npsConf.tempDirProps.timeToUse.dir = npsConf.tempDirProps.timeToUse.dir   ? npsConf.tempDirProps.timeToUse.dir.toLowerCase()   : 'birthtime'
		npsConf.tempDirProps.timeToUse.file= npsConf.tempDirProps.timeToUse.file  ? npsConf.tempDirProps.timeToUse.file.toLowerCase()  : 'mtime'	  
		// V4.1.34
	}
	else{
		// Set up default options
		//npsConf.tempDirProps = {'plugin':'', 'service': '', 'parms':{'parm1':''}, cronOptions:{'cronTime': '00 00 * * * *', 'interval': 30, 'unit': 'n'}};
		npsConf.tempDirProps = {'cronTime': '00 00 * * * *', 'interval': 30, 'unit': 'n', timeToUse: {'dir': 'birthtime', 'file': 'mtime'}};	// V4.1.34
	}

    async.series([
        // Clean the temp dir
        function(nextTask){	// V5.0.30
			_parseEnvVars(function(err){	// V5.0.30
				// Error already logged by cleanTempDir. Ignore the error since we are deleting temp. directories
				//if(err) return nextTask(err);	
				return nextTask(err)
			})
        },		
        // Clean the temp dir
        function(nextTask){
			exports.cleanTempDir(function(err){
				// Error already logged by cleanTempDir. Ignore the error since we are deleting temp. directories
				//if(err) return nextTask(err);	
				return nextTask()
			})
        },
   		// Initialize the cronjob module
		function(nextTask){
            cronjob.initialize(nextTask)
        },
		// Set up a Cron Job to clean the temp dir
        function(nextTask){
            exports.addCronJobForTempDir(nextTask)
        },
		// Add server level cron jobs
        function(nextTask){
            exports.addCronJobs(nextTask)
        },                                                
        // Add middlewares given in server-conf.js
        function(nextTask){
            if(npsConf.middlewares)
  // V5.0.21 Begin
            if(npsConf.envname && npsConf.disableSwagger && npsConf.disableSwagger.includes(npsConf.envname.toLowerCase())){
                // remove swagger middleware
                result = npsConf.middlewares.filter(function(item){if(item && item.id && item.id.toLowerCase() === 'swagger') return false; else return true;})
                npsConf.middlewares = result
            }
// V5.0.21 Begin
			expressAppUtils.addMiddlewares({router:app, middlewares:npsConf.middlewares, moduleName:'oxsnps-core'}, function(err){
				if(npsConf.stopOnInitErrors) return nextTask(err)  // V5.0.23
				if(err) logger.error('-->init: ' + err.message)	// V5.0.23
				return nextTask()					
			})
            //return nextTask()            
        },
        // Enable the core services
        function(nextTask){
            _enableCoreServices(nextTask)
        },
        // Feed application specific configuration files, if exist
        //!!!TODO: We should call readPluginConfFiles first for the base plugins and then for the custom plugins
        //!!!TODO: Just pass the path to readPluginConfFiles where to find the conf files
        function(nextTask){
            pluginManager.readPluginConfFiles(function(err){
				if(!err) logger.debug('-->init: server specific plugins configuration.: ' + JSON.stringify(npsConf.plugins))

				if(npsConf.stopOnInitErrors) return nextTask(err)	// V5.0.23
				if(err) logger.error('-->init: ' + err.message)		// V5.0.23
				return nextTask()					
				
                //return nextTask(err)
            })
        },
        // Enable plugins
        function(nextTask){
            pluginManager.enablePlugins(nextTask)
        },
        //if application startup plugin exists, do application specific startup tasks
        function(nextTask){
            if(npsConf.appStartPlugin) return _doAppStartupTasks(nextTask)
            return nextTask()
        },
        function(nextTask){
            pluginManager.activatePlugins(nextTask)
        }
    ],
    function(err){
        return callback(err)
    })
}
// V5.0.16 End

// V3.0.16 Begin
/**
 * Perform tasks to be done after start of server
 * @param  {Function} callback callback(err)
 */
exports.afterServerStart = function(callback){
	pluginManager.afterServerStart(callback);
};
// V3.0.16 End

// Process the default Request (i.e. localhost/)
exports.main = function(req, res){
	logger.debug('main:\nreq: ' + req.url + '\nheaders:' + JSON.stringify(req.headers)  + '\nbody:'+ JSON.stringify(req.body));
	return _getVersion(res);
}

// V5.0.12 Begin
exports.swagger = function(req, res){

    let indexHtmlFile = path.join(npsDir, 'swagger', 'index.html')
	//console.log(indexHtmlFile )

	// Read the HTML file, set some nps... vars, and send back the modified HTML
	fs.readFile(indexHtmlFile, 'utf8', function(err, html){
		if(err) return res.status(404).end(indexHtmlFile + ' not found!' )
		res.send(html)
	})
}
// V5.0.12 End

// Process a ping request (i.e. localhost/ping)
exports.ping = function(req, res){
	res.end();
}

/**
 * getVersion: Returns the Server version the version of its components
 * @Params: 
 *    req: Request Object
 *    res: Response Object 
 */
exports.getVersion = function(req, res){
	return _getVersion(res);
}

/**
 * isAlive: Check whether the Servers behing the plugins are still alive
 * @Params: 
 *    req: Request Object
 *    res: Response Object 
 */
exports.isAlive = function(req, res){
	return _isAlive(res);
}

/** V5.0.2
 * Dump heap 
 * @param  {req} 	GET req   http://<server>:<port>/http?parm={"method":"post","url": "http://locahost:1030/postkorb/online/postkorb/services/pk.frombpm/existGeVo","body": "{\"geVoTypKey\": 2,\"ordnungsBegriffe\":[{\"definitionId\":901,\"value\":\"67\",\"sequenceNo\":1,\"uiPriority\":1}]}"}
 * @param  {res}	res
 */
exports.heapdump = function(req, res){

	let dumpDir     = undefined
      , filename    = undefined
      , opFileStream= undefined
      , mode		= parseInt('0777',8)
      , nextTaskInvoked = false
	  , versionParts    = undefined
      , nodejsVers      = undefined
 
	res._Context = {} 									// add Context to the response instance
	res._Context.reqId = utils.buildReqId(new Date())	// add a Request Id to the Context

    logger.info('--->heapdump: Req.id=' + res._Context.reqId + ', Processing a heapdump request, req.query: ' + JSON.stringify(req.query))

	// V4.1.25 Begin
    if(process.versions.node) versionParts = process.versions.node.split('.')       //  process.versions.node: '10.16.0',

    if(versionParts && versionParts.length > 0 && versionParts[0] && isNaN(versionParts[0]) === false) nodejsVers = parseInt(versionParts[0])

    if(!nodejsVers || nodejsVers <= 12){
        res.end('Heap dump API is not supported for node.js version: ' + process.versions.node + '. It is supported for node.js versions >=12')
        return
    }
	// V4.1.25 End 	
    dumpDir = req.query.dir || npsTempDir
    filename = path.join(dumpDir, res._Context.reqId) + '.heapsnapshot'

	async.series([
		// Task 1: Create the output directory 
		function(nextTask){
            logger.info('--->heapdump: Req.id=' + res._Context.reqId + ', Creating a directory: ' + dumpDir)
            utils.mkDir(dumpDir, mode, nextTask)
		},
		// Task 2: Dump heap
		function(nextTask){
            logger.info('--->heapdump: Req.id=' + res._Context.reqId + ', Calling v8.writeSnapshot()..')  
            const stream = v8.getHeapSnapshot()
            opFileStream = fs.createWriteStream(filename)

	    	// Add event listeners for destination file
			opFileStream.on('finish', function(){
				if(logger.isDebugEnabled()) logger.debug('--->heapdump: Req. id=' + res._Context.reqId + ', finished writing file: ' + filename)
				if(!nextTaskInvoked){
					nextTaskInvoked = true
					return nextTask()
				}
			})
			opFileStream.on('error', function(err){
				if(nextTaskInvoked) return logger.error('--->heapdump: Req. id=' + res._Context.reqId + ', unable to  write to file: ' + filename + ', ' + err.message)
				nextTaskInvoked = true
				return nextTask(err)
			})

			// Add event listeners for source
			stream.on('end', function(){
				if(logger.isDebugEnabled()) logger.debug('--->heapdump: Req. id=' + res._Context.reqId + ', finished reading from snapshot stream ')
			})
			stream.on('error', function(err){	// error handling 
				if(nextTaskInvoked) return logger.error('--->heapdump: Req. id=' + res._Context.reqId + ', unable to read from snapshot stream, ' + err.message)
				nextTaskInvoked = true
				return nextTask(err)
			})  

            stream.pipe(opFileStream)
		}
	],
	function(err){
		if(err){
            err.message = 'Error when dumping heap, filename: ' + filename + ', Reason: ' + err.message
            logger.error('--->heapdump: Req.id=' + res._Context.reqId + ', ' + err.message)
            return res.end(err.message)
        }
        logger.info('--->heapdump: Req.id=' + res._Context.reqId + ', heap dumped to file: ' + filename)
        res.end('Heap dumped to file: ' + filename)
		return
	})	
}

// Returns Server History
exports.getHistory = function(req, res) {
	
	var historyFile 	= undefined;

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	logger.info('--->History Req. id=' + res._Context.reqId + ', Server History requested');
	historyFile = fs.createReadStream(npsDir + '/' + 'history.txt');

	historyFile.on('end', function(){
		logger.info('--->History Req. id=' + res._Context.reqId + ', over');
	});

	historyFile.on('error', function(err){
		logger.error('--->History Req. id=' + res._Context.reqId + ', ' + 
				', Unable to read history text. ' + err.message);
		res.end('Unable to read history text. ' + err.message);
	});

	// There is no need for error event handlers for response object since error is not emitted.
	// Refer to below text from https://www.bennadel.com/blog/2823-does-the-http-response-stream-need-error-event-handlers-in-node-js.htm
	// HTTP response stream will never emit an error event; or, at the very least, that such an error 
	// event will never crash the server. So, while you have to bind an "error" event handler to just about
	// every other streams in Node.js, it would seem that you can safely assume the HTTP response stream will
	// never error, no matter what happens. And, that even if the response stream closes, you can continue 
	// to write to it / pipe to it without consequence.	 
	/*
	res.on('error', function(err){
		logger.error('--->History Req. id=' + res._Context.reqId + ', ' + 
				', Unable to return history text. ' + err.message);
	});
	*/
	// Send the output
	historyFile.pipe(res);

	//setTimeout(function(){historyFile.pipe(res);}, 10*1000);  // for testing
}

/**
 * getRoutes 			Returns the list of core services
 * @param  {Request} 	req Express request
 * @param  {Response} 	res Express response
 */
exports.getRoutes = function(req, res) {

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	logger.info('--->getRoutes Req. id=' + res._Context.reqId + ', List of Core Services requested');
	res.json(coreModules);
	logger.info('--->getRoutes Req. id=' + res._Context.reqId + ', List of Core Services returned');
}

/**
 * Set the log level of this module
 * @param  {req} 	GET req   http://localhost:1026/appservices/deploy/setloglevel?loglevel=DEBUG|INFO|ERROR
 * @param  {res}	res
 */
exports.setLogLevel = function(req, res){
	var loglevel = req.query.loglevel || 'INFO';
	utils.log4jsSetLogLevel(logger, loglevel);
	return res.end('--->setLogLevel: Log level of ' + MODULE_NAME + ' set to ' + loglevel);
}

/**
 * Return the log level of this module
 * @param  {req} 	GET req   http://localhost:1026/appservices/deploy/getloglevel
 * @param  {res}	res
 */
exports.getLogLevel = function(req, res){
	logger.info('--->getLogLevel: Getting the log level of ' + MODULE_NAME + '...');
	return res.end(logger.level.levelStr);
}

/**
 * Called from the cron Job to clean the temp directory
 */
exports.cleanTempDir = function(callback){

	var self = this
	  , timeToUse = npsConf.tempDirProps.timeToUse || {}	// V4.1.34

	logger.info('--->cleanTempDir: Cleaning the temp directory (' + npsTempDir + ')...');

	// Delete temp dirs and files that are older than 30mns
	utils.deleteDir(npsTempDir, false /* do not delete the root dir */, npsConf.tempDirProps.unit /* mns */, npsConf.tempDirProps.interval || npsConf.tempDirProps.timeToKeep, true /* recursively delete dir entries */, timeToUse, function(err){	// V4.1.34
		if(err){
			logger.error('--->cleanTempDir: Cleaning the temp directory (' + npsTempDir + ') failed: ' + err.message);
		} 
		else{
			logger.info('--->cleanTempDir: Temp directory (' + npsTempDir + ') cleaned.');
		}
		if(callback) callback(err);
	});
}

/**
 * quit: Terminate Server
 * Graceful quit: http://localhost:1033/quit/?soft=true
 * Hard quit: http://localhost:1033/quit
 * @Params: 
 *    req: Request Object
 *    res: Response Object 
 */
exports.quit = function(req, res) {

	var quitMsg = undefined;

// V3.0.13 Begin
	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	logger.info('--->quit: Req. id=' + res._Context.reqId +
		((req.query && req.query.soft) ? (', Params='+ JSON.stringify(req.query))  : '') + 
		', Leaving Server...' );

	res._Context.query = req.query;

	async.series([
		// Finalize the cronjob module
		function(nextTask){
			cronjob.finalize(function(err){
				if(err) logger.	error('quit, Error: ' + err);
				nextTask();
			});
		},
		// Deactivate plugins
		function(nextTask){
			pluginManager.deactivatePlugins(res, nextTask);
		},
		// Disable Plugins
		function(nextTask){
			pluginManager.disablePlugins(res, nextTask);
		}		
	],
	function(err){
		if(err){
			// Build Response
			logger.error('quit, Error on closing the server.: ' + err);
			quitMsg = 'Error on closing the server. ' + err;
			res.status(500);
		}else{
			quitMsg = 'OXSNPS Server quit successfully.';
			logger.info('--->Quit Req.: ' + quitMsg);
		} 

		// Send  Response
		res.end('<html><head><title>OXSNPS Server</title></head>' +
				'<body><br><h2>'+ quitMsg + '</h2></body></html>'
			);
		// Terminate the process
		process.exit(0);
	});
// V3.0.13 End
}

/**
 * Return the log level of this module
 * @param  {req} 	GET req   http://<server>:<port>/http?parm={"method":"post","url": "http://locahost:1030/postkorb/online/postkorb/services/pk.frombpm/existGeVo","body": "{\"geVoTypKey\": 2,\"ordnungsBegriffe\":[{\"definitionId\":901,\"value\":\"67\",\"sequenceNo\":1,\"uiPriority\":1}]}"}
 * @param  {res}	res
 */
exports.processHttpReq = function(req, res){

	var parm 	= undefined
	  , options = undefined
	  , helpmsg	= 'This API expects following parameters:<br>' +
	  			  '<br>parm={"method":"&lt;method&gt;","url":"&lt;url&gt;","body": "{&lt;body&gt;}<br>' +
	  			  '<br>where:' +
	  			  '<br>\t\t"&lt;method&gt;":\t\t get|post|put|head (a http verb; optional, defaults to get)' + 
	  			  '<br>\t\t"&lt;url&gt;":\t\t the url to call' + 
	  			  '<br>\t\t"&lt;body&gt;":\t\t the body of the request; optional' + 
	  			  '<br><br>Examples:<br>' + 
	  			  '\t\tGET http://&lt;hostname&gt;:&lt;port&gt;/http?parm={"method":"post","url":"http://locahost:1030/...",' + 
	  			  '"body":{&lt;some JSON object&gt;}}<br><br>'
	  ;
	logger.info('--->processHttpReq: Processing req.query: ' + JSON.stringify(req.query));

	if(Object.keys(req.query).length === 0){
		logger.warn('--->processHttpReq: Wrong Request, the parm parameter is missing, leaving...');
		return res.end('<html><body>--->processHttpReq: Wrong Request, the "parm" parameter is missing!<br><br>' + helpmsg + '</body></html>');
	}

	logger.debug('--->processHttpReq: Processing req.query.parm: ' + JSON.stringify(req.query.parm));
	parm = utils.parseJSON(req.query.parm);
	if(parm === null){
		logger.warn('--->processHttpReq: Wrong Request, the "parm" parameter is missing, leaving...');
		return res.end('<html><body>--->processHttpReq: Wrong Request, the "parm" parameter is missing!<br><br>' + helpmsg + '</body></html>');
	}

	logger.debug('--->processHttpReq: parm as JSON Object: ' + JSON.stringify(parm));
	if(parm.url === undefined){
		logger.warn('--->processHttpReq: Wrong Request, the "url" parameter is missing, leaving...');
		return res.end('<html><body>--->processHttpReq: Wrong Request, the "url" parameter is missing!<br><br>' + helpmsg + '</body></html>');
	}

	// Ensure parm.method is define
	parm.method = parm.method || 'get' 

	logger.debug('--->processHttpReq: method:' + parm.method.toLowerCase() + ', url:' + parm.url + ', body:' + (parm.body? JSON.stringify(parm.body) : 'not passed'));
	/*
		if(parm.method.toLowerCase() === 'post' && !parm.body){
			logger.warn('--->processHttpReq: Wrong Request, if the post method is used, body must be passed, leaving...');
			return res.end('<html><body>--->processHttpReq: Wrong Request, if the post method is used, body must be passed!<br><br>' + helpmsg + '</body></html>');
		}
	 */	

	// Parse the URL
	options = url.parse(parm.url);
	if(options === undefined){
		logger.warn('--->processHttpReq: Wrong Request, the given "url" parameter (' + parm.url + ') is not a valid url, leaving...');
		return res.end('<html><body>--->processHttpReq: Wrong Request, the given "url" parameter (' + parm.url + ') is not a valid url!<br><br>' + helpmsg + '</body></html>');
	}
	logger.debug('--->processHttpReq: url as JSON Object: ' + JSON.stringify(options));

	// Add the method to options
	options.method = parm.method

	// Execute the HTTP(s) request
	httpUtils.processURL(options, parm.body, function(err, data){
		if(err){
			logger.error('--->processHttpReq: Executing HTTP request with options:' + JSON.stringify(options) +
						 ' and parms:' + JSON.stringify(parm.body) + ' failed, Error: ' + err.message);
			return res.end('<html><body>Executing HTTP request with options:' + 
						   JSON.stringify(options) + ' failed, Error: ' + err.message + '</body></html>');

		}
		logger.info('--->processHttpReq: req.query: ' + JSON.stringify(req.query) + ' successfully processed. ' +
					'httpUtils.processURL returned ' + data.length + ' bytes');						 
		return res.end(data); 
	});
}

// V3.0.39 Begin
exports.setupShutdownOptions = function (server){
	shutdown.setupShutdownOptions(server)
	return
}
// V3.0.39 End

/******************  Private Functions ********************/
function _isAlive(res){

	var data 	= ''
	  , plugins = pluginManager.getPluginsList()
	  , module 	= null
	  , bError 	= false
	  ;

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	logger.info('--->isAlive. Req. id=' + res._Context.reqId);

	// Assert plugins parameter
	if(plugins === undefined || plugins === null|| (typeof plugins !== "object") || (!util.isArray(plugins)) || (plugins.length <= 0)){
		logger.info('--->isAlive. Req. id=' + res._Context.reqId + ', over.');
		return res.status(404).end('No Plugin available.');
	}

	// Loop through all the plugins
	async.each(plugins,
		function(plugin, nextTask){
			if(plugin.enable === undefined || plugin.enable !== 'on'){
        		data += plugin.module + ' not enabled\n';
				nextTask();
			}
			else{
				//  Get plugin version
				logger.debug('_isAlive: Getting ' + plugin.module + ' version...');

				try{
					// Get the module handle
					module = require(path.resolve(plugin.plugindir) + '/' + plugin.module + '/' + pluginJSfilename);

					if (typeof(module.isAlive) === "function"){
						// Call isAlive
						module.isAlive(function(err, result){
				        	if (err){
								logger.error('_isAlive, ' + plugin.module + ' Version Error: ' + err.message);
				        		data += plugin.module + ' Error: ' + err.message + '\n';
								bError = true;
				        	}
				        	else{
				        		data += plugin.module + ' v' + result + '\n';
				        	}
							nextTask();
						});
					}
					else{				
						// Get the version of the module
						module.getVersion(function(err, result){
				        	if (err){
								logger.error('_isAlive, ' + plugin.module + ' Version Error: ' + err.message);
				        		data += plugin.module + ' Error: ' + err.message + '\n';
								bError = true;
				        	}
				        	else{
				        		data += plugin.module + ' v' + result + '\n';
				        	}
							nextTask();
						});
					}
				}catch(err){
					logger.error('_isAlive, ' + plugin.module + ' Version Error: ' + err.message);
	        		data += plugin.module + ' Error: ' + err.message + '\n';
					bError = true;
					nextTask();
				}
			}
		},
		function(err){
			logger.debug('_isAlive: Plugin Info:\n' + data);
	        if (bError){
				logger.error('--->isAlive. Req. id=' + res._Context.reqId + ', at least one Component is not alive.');
				return res.status(500).end('--->isAlive. Req. id=' + res._Context.reqId + ' Error: At least one Component is not alive:\n' + data);
			}
			logger.info('--->isAlive. Req. id=' + res._Context.reqId + ', over.');
			res.end('All the Components are alive:\n' + data);
		}
	);
}

function _getVersion(res){

	var data 	= ''
	  , plugins = pluginManager.getPluginsList()
	  , module 	= null
	  ;

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the HTML file to server to the Context
	res._Context.htmlFile = '/public/index.html';

	logger.info('--->Ver. Req. id=' + res._Context.reqId + ', Servers Version required');

	// Assert plugins parameter
	if(plugins === undefined || plugins === null|| (typeof plugins !== "object") || (!util.isArray(plugins)) || (plugins.length <= 0)){
		logger.info('--->Ver. Req. id=' + res._Context.reqId + ', over.');
		// Send back HTTP Response 
		return _serveHTML(res, 'Plugins are not available.');
	}

	// Loop through all the plugins
	async.each(plugins,
		function(plugin, nextTask){
			if(plugin.enable === undefined || plugin.enable !== 'on'){
				nextTask();
			}
			else{
				//  Get plugin version
				logger.debug('_getVersion: Getting ' + plugin.module + ' version...');

				try{
					// Get the module handle
					module = require(path.resolve(plugin.plugindir) + '/' + plugin.module + '/' + pluginJSfilename);

					// Get the version of the module
					module.getVersion(function(err, result){
			        	if (err){
							logger.error('_getVersion,' + plugin.module + ' Version Error: ' + err);
							data += '<h3>' +  plugin.module + ' Version Error: ' + err + '</h3>';
			        	}
			        	else{
							// result => v1.0.x.x
							logger.debug('_getVersion: ' + plugin.module + ' Version: ' + result);
							data += '<h3>' +  plugin.module + ' v' + result + '</h3>';
						}
						nextTask(err);
					});
				}catch(err){
					nextTask(err);
				}
			}
		},
		function(err){
	        if (err){
				logger.error('_getVersion: Error: ' + err);
				logger.info('--->Ver. Req. id=' + res._Context.reqId + ', over.');
				return _serveHTML(res, 'Error when retrieving a Server Version: ' + err.message);
			} 
			logger.info('--->Ver. Req. id=' + res._Context.reqId + ', Servers Version returned');

			// Send back HTTP Response 
			_serveHTML(res, data);
		}
	);
}

function _serveHTML(res, body){

	// Read the HTML file, set some nps... vars, and send back the modified HTML
	fs.readFile(npsDir + res._Context.htmlFile, 'utf8', function(err, html){
		if(err){
			return res.status(404).end(npsDir + res._Context.htmlFile + ' not found!' );
		}
		html = html.replace(/&npsVersion;/g, npsServer.npsVersion);	// replace version
		html = html.replace(/&npsName;/g, npsServer.npsDesc);		// replace description
		html = html.replace(/&npsBody;/g, body);					// replace body
		res.end(html);
	});
}

/**
 * Call functions of application startup plugin to do application specific startup tasks
 * @param  {Function} callback callback(err)
 */
function _doAppStartupTasks(callback){

	var module 	= undefined
	  , res 	= {}
	  ;

	// Add Context to the response instance
	res._Context =   {};

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(new Date());

	async.series([
		// Get startup plugin module
		function(nextTask){
			pluginManager.getModule(npsConf.appStartPlugin, function(err, result){
				if(err) return nextTask(err);
				module = result;
				nextTask();
			});
		},
		// call beforeServerStart() function if exists.
		function(nextTask){
			if(module.beforeServerStart && typeof(module.beforeServerStart) === 'function'){
                logger.info('---> _doAppStartupTasks: Starting application startup tasks, Calling ' + npsConf.appStartPlugin + '.beforeServerStart() ...')
                return module.beforeServerStart(res, nextTask)
            }
			nextTask();	
		}		
	],
	function(err){
		if(npsConf.stopOnInitErrors) return callback(err)		// V5.0.23
		if(err) logger.error('--->_doAppStartupTasks: ' + err.message) // V5.0.23
		return callback()		
		//return callback(err)
	});
}

// V5.0.16 Begin
// Set up a Cron Job to clean the temp dir
exports.addCronJobForTempDir = function(callback){
	let self 	= this
		, options = {}

	// Set up a Cron Job to clean the temp dir
	options.name         = npsTempDir
	options.cronFunction = self.cleanTempDir
	self.function 		 = 'oxsnps-core/index.js->cleanTempDir' // (Optional) a String to identify the cronFunction
	options.cronOptions  = {'start': true, 'context': self}
	options.cronTime     = npsConf.tempDirProps.cronTime

	// Add the Cron job
	logger.debug('--->_addCronJobForTempDir: Creating cron job to clean the temp directory (' + npsTempDir + ')...')
	cronjob.addCronJobService(options, function(err){
        if(npsConf.stopOnInitErrors) return callback(err)		// V5.0.23
        if(err) logger.error('--->addCronJobForTempDir: Unable to  Create cron job to clean the temp directory: ' + npsTempDir + '. Reason: ' + err.message)
        return callback()
	})
}

// Log server alive status
exports.logServerAliveStatus = function(callback){
	let self 	= this
	  , options	= npsConf.cronjobs['logServerAliveStatus']
	  , raw 	= Math.floor(process.uptime())	// uptime in seconds 
      , days 	= 0
      , hours 	= 0
      , minutes = 0
	  , seconds = 0
	  , status	= undefined

	// set defaults
    options.logLevel	= options.logLevel	|| 'info'
    options.logLevel    = options.logLevel.toLowerCase()
	options.message 	= options.message	|| "'Server is running for ' + (days >0 ? days + ' days, ': '') + (hours >0 ? hours + ' hours, ': '') + minutes + ' mins and ' + seconds + ' secs'"

	if(logger.isDebugEnabled()) logger.debug('--->logServerAliveStatus: options: ' + JSON.stringify(options) + ', uptime: ' + raw + ' secs')
    //raw = require('os').uptime()
	//courtesy: https://www.tabnine.com/code/javascript/functions/uptime
	days = Math.floor(raw/86400)
	raw -= days*86400
	hours = Math.floor(raw/3600)
	raw -= hours*3600
	minutes = Math.floor(raw/60)
	raw -= minutes*60
	seconds = Math.floor(raw)

	try{
		status = eval(options.message)
		logger[options.logLevel](status)
	}catch(err){
		logger.error('Unable to log server alive status, ' + err.message)
	}

	if(callback) return callback(null, status)
	return status
}

// Add server level cron jobs
exports.addCronJobs = function(callback){
    let cronjobNames= undefined
       , self 	    = this

	npsConf.cronjobs = npsConf.cronjobs || {}

	// Add default cron jobs
	if(npsConf.cronjobs.logServerAliveStatus === undefined)
		npsConf.cronjobs.logServerAliveStatus = { "cronTime": '0 0 0 * * *' }

	if(logger.isDebugEnabled()) logger.debug('--->_addCronJobs: Adding server level cron jobs, cronjobs: ' + JSON.stringify(npsConf.cronjobs))

	cronjobNames = Object.keys(npsConf.cronjobs)
	
	async.each(cronjobNames,
		function(cronjobName, next){
            let options = npsConf.cronjobs[cronjobName]
              , envvar  = undefined

            if(!cronjobName || !options) return next()

            if(typeof options === 'string'){
                envvar = options
                if(!process.env[envvar]) return next(new Error('Unable to add a cron job: '+  cronjobName + ', missing environment variable: ' + envvar))
        
                options = utils.parseJSON(process.env[envvar])
                if(options === null) return next(new Error('Unable to add a cron job: '+  cronjobName + ', error on parsing environment variable: ' + envvar + ', value: ' + process.env[envvar]))
                npsConf.cronjobs[cronjobName] = options
            }
            if(logger.isDebugEnabled()) logger.debug('--->_addCronJobs: cronjob: ' + JSON.stringify(options))
            
            if(options.enable && options.enable.toLowerCase() === 'off') return next()
            if(!options.cronTime) return next()

			options.name 		=  options.name || cronjobName
			options.cronFunction=  self[options.function || cronjobName]
		    //self.function		=  cronjob.function || cronjobName	// (Optional) a String to identify the cronFunction
			options.cronOptions  = {'start': true, 'context': {'name':'cronjobName'}}
			//options.cronTime     = options.cronTime
			
			// Add the Cron job
			if(logger.isDebugEnabled()) logger.debug('--->_addCronJobs: Adding a cron job, options: ' + JSON.stringify(options))
			cronjob.addCronJobService(options, function(err){
				if(err) return next('Unable to add a cron job, options: ' + JSON.stringify(options))
				if(logger.isDebugEnabled()) logger.debug('--->_addCronJobs: Addded the cron job, name: ' + options.name)
				return next()
			})
		},
		function(err){
			if(npsConf.stopOnInitErrors) return callback(err)		// V5.0.23
			if(err) logger.error('--->_addCronJobs: ' + err.message)		// V5.0.23
			return callback()					
			//return callback(err)
		}
	)	

}

// Enable the core services
function _enableCoreServices(callback){
	// V4.1.0 End            
	// Enable the core services
	logger.debug('--->_enableCoreServices: Enabling the core services...')
	async.each(coreModules,
		function(coreModule, nextTask){
			// Load the core module
			let error = _loadCoreModule(coreModule) 
			if(error) return nextTask(error)
			expressAppUtils.enableRoutes(coreModule.handle, coreModule.routes, function(err){
				if(err) return nextTask(err)
				nextTask()
			})
		},
		function(err){
			if(err) logger.error('_enableCoreServices: Initializing core services failed, Reason: ' + err.message)
			else logger.debug('--->_enableCoreServices: Enabled the core services')

			if(npsConf.stopOnInitErrors) return callback(err)	// V5.0.23
			return callback()

			//return callback(err)
		}
	)
}
// V5.0.16 End

function _loadCoreModule(coreModule){
	try{
		// Load the given core module
		let filename = path.resolve(__dirname + '/' + coreModule.name + '.js') // require('./conf/oxsnps-core.js')
		coreModule.handle = require(filename)
		logger.info('---> Core Module: ' + coreModule.name + ' loaded.')
	}catch(err){
		return err
	}
}

// V5.0.30 Begin
function _parseEnvVars(callback){

	var envvar  = undefined

	//V5.0.15 Begin
	// Get plugin list from env. variable if given
	if(typeof npsConf.pluginList === "string"){
		envvar = npsConf.pluginList
		npsConf.pluginList = utils.parseJSON(process.env[envvar])
		if(npsConf.pluginList === null) return callback(new Error('Error on parsing environment variable: ' + envvar + ', value: ' + process.env[envvar]))
	}
	//V5.0.15 End

	//V5.0.30 Begin
	
	// Parse env.variable value given for npsConf.startConditions.servers
	if(npsConf.startConditions && npsConf.startConditions.servers && typeof npsConf.startConditions.servers === "string"){
		envvar = npsConf.startConditions.servers
		npsConf.startConditions.servers = utils.parseJSON(process.env[envvar])
		if(npsConf.startConditions.servers === null) return callback(new Error('Error on parsing environment variable: ' + envvar + ', value: ' + process.env[envvar]))
	}
    logger.debug('--->_parseEnvVars: npsConf.startConditions.servers: ' + (npsConf.startConditions ? JSON.stringify(npsConf.startConditions.servers) : ''))
//    logger.debug('--->_parseEnvVars: npsConf.startConditions.servers: ', npsConf.startConditions.servers)
//    logger.debug('--->_parseEnvVars: typeof (npsConf.startConditions.servers): ' + typeof (npsConf.startConditions.servers))
    return callback()
	//V5.0.30 End	
}
// V5.0.30 End