/**-----------------------------------------------------------------------------
 * pluginManager.js: 	Node.js module to manage plugins
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2014 by Maas Holding GmbH
 * Email     :  support@maas.de
 * Version   :  V5.0.0
 * 
 * History
 * Refer to ../server.js
 *  V5.0.6/V5.0.7  	27.04.2022  OTS-3238: pluginManager.js: Disable the routes of a plugin if it can't be initialized. 
 *                                    Added disablePluginRoutesService() function to disable all routess of a plugin
 *  V4.1.15  	18.12.2020  OTS-2942: pluginManager.js: Extend to pass the reason to rename the state file to disablebyerror to oxsnps-statfilewatcher.updateRouteStatefile
 *  V4.1.7  	18.06.2020  OXS-10750: pluginManager.js: Extend to check if oxnps-statefilewatcher plugin exists befor using it to to create state files
 *  V3.0.39  	01.08.2018  OXS-9479:  pluginManager.js: Extend to support tenant based state files
 *  V3.0.32  	14.11.2018  OTS-2326:  pluginManager.js: Extend to provide an option to skip saving plugin configuration
 *  V3.0.23 	22.05.2018  			Turntable extenstion: Log entries added thru 'http-proxy-middleware'
 *  V3.0.22 	08.05.2018  OTS-2159: 	TFS-163113: Log level of hal-archivegw cannot be set thru the server's admin web gui
 *  V3.0.21 	25.04.2018  			Extended not to create state files for HTTP routes with 'delete' and 'put' as HTTP methods
 *  V3.0.16   	14.02.2018  OTS-2049:  Extended to do application specific tasks after starting the server  
 *  V3.0.13 	11.12.2017 	OTS-1996: 	Extended core with softquit functionality to gracefully stop server
 *  V3.0.14   	03.10.2018  OTS-1981: 	Fixed a bug that when route state file name is manually changed to <statefile>.disablebystop, sets route state file name as <statefile>.disable
 *  V100   		24.02.2015  			Initial release
 *----------------------------------------------------------------------------*/
'use strict';

var MODULE_NAME		= 'pluginManager'
  , MODULE_VERSION	= '2.0.0'
  , fs          	= require('fs')
  , path        	= require('path')
  , log4js			= require('log4js')
  ,	async 			= require('async')
  , os       		= require('os')

  , npsServer		= require('./server')  
  , expressApp		= npsServer.expressApp				// is the Express App
  , npsDir			= npsServer.npsDir					
  , npsConf 		= npsServer.npsConf					// server-conf.js of oxs-xxx-server
  , npsConfDir 		= npsServer.npsConfDir
  , warn 			= require('./helpers/warn')  
  , utils 			= require('./helpers/utils')
  , expressAppUtils = require('./helpers/expressAppUtils')
  , pluginJSfilename= 'oxsnps.js'						// Mandatory js filename of an OXSNPS Plugin
  , FILE_SEP 		= require('path').sep
  , plugins 		= [] 								// holds the list of the plugins
  , stopInProgress 	= false
  , savePluginConf	= ((npsConf['oxsnps-core'] && npsConf['oxsnps-core'].plugins && npsConf['oxsnps-core'].plugins.saveconf)) ? npsConf['oxsnps-core'].plugins.saveconf : false 	// V3.0.32, OTS-2326
  , sfwPluginName   = 'oxsnps-statefilewatcher' // V4.1.7
  , appname			= npsServer.appname			// V5

/*
	Structure of the plugins array
    "plugins":[
		{
			"module": 		"Plugin Name (ex.: afp2any)",
			"plugindir": 	"FQ name of the the plugin root directory (ex.: /path/to/pluginDir/)",
			"enable": 		"on|off",
			"desc": 		"Short description of the plugin",
			"longDesc": 	"Long description of the plugin",
			"logLevel":		"DEBUG|INFO|ERROR",
			"routes": [
				{
					"path": 	"/path/to/the/service",
					"enable": 	"on/off"
					// HTTP routes:
					"method": 	"get/post",
					"service": 	"name of the plugin service to be mapped to path",
					// dirwacther routes:
					"events": 	"...to watch for",
					"options": "see node,js chokidar options",
					"action": 	{
		                "interface": "[http | local]: http or local call of a service of a module",
		                "props":    "the props required by the interface",
		                "props":    {
							"module":  "module/plugin name",
							"module":  "./local/ic1",
							"service": "ic1Service"
		                },
		                "parms":    "the props required by the service",
				        "parms":	{
				        	"prop1": "prop1",
				        	"prop2": "prop2"
				        }
					}
				}
			]
		},
		{...}
	]
*/

// swagger config object
let swagger = {"urls": []}	// V4.1.22

// Get Logger
let logger = log4js.getLogger(MODULE_NAME);
utils.log4jsSetLogLevel(logger, (utils.getAppLogLevel(npsConf.log) || 'INFO'));

// Export Module Version
exports.version = MODULE_VERSION;

exports.DEACTIVATED 	= 'deactivated';
exports.ENABLE 			= 'enable';
exports.DISABLE 		= 'disable';
exports.DISABLEBYERROR 	= 'disablebyerror';
exports.DISABLEBYSTOP 	= 'disablebystop';

/**
 * version: 		Process a Version Request
 * @param  {req}	req Request Object
 * @param  {res}	res Response Object
 */
exports.version = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);
	logger.info('--->version: Req. Id=' + res._Context.reqId);

	var data = '<h3>' +  MODULE_NAME + ' v' + MODULE_VERSION + '</h3>';
	res.end(data);
	
	logger.info('--->version: Req. Id=' + res._Context.reqId + ', Version sent back, over');
}

/**
 * enablePlugin: 		Enable the given plugin.
 * @param  {req} 		req as GET or POST 
 * 		   GET URL: 	<server>[:port]/appservices/pm/enable?module=oxsnps-afp2any
 * 		   POST URL: 	<server>[:port]/appservices/pm/enable
 * 		   POST parms:	req.body = {
 * 		            		"module":"oxsnps-afp2any", 
 * 		              	}
 * @param  {res} 		res
 */
exports.enablePlugin = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the plugin name to the Context
	if(req.method === 'GET') res._Context.module = req.query.module;
	else res._Context.module = req.body.module;

	logger.info('--->enable: Req. Id=' + res._Context.reqId);
	if(logger.isDebugEnabled()) logger.debug('--->enable: Req. Id=' + res._Context.reqId + ', Parms: ' + (req.method === 'GET'?JSON.stringify(req.query):JSON.stringify(req.body)));

	// Get Plugin
	_getPlugin(res._Context.module, function(err, plugin){
		if(err){
			logger.error('--->enablePlugin: Req. Id=' + res._Context.reqId + ', Plugin enable request failed: ' + err.stack + '. Parms:' + JSON.stringify(req.body));
			return res.status(404).end('Plugin enable request failed: ' + err.stack + '. Parms:' + JSON.stringify(req.body));
		}

		// If the plugin is already enabled (plugin is loaded only when it is enabled), leave
		if(plugin.enable === undefined || plugin.enable === 'on'){
			return res.end('Plugin ' + res._Context.module + ' already enabled');
		}

		//  Enable the plugin (means enable the routes of the plugin)
		_enablePlugin(plugin, function(err){

			// Ensure to move the '/*' route to the end of the list of routes
			expressAppUtils.moveDefaultAppRoute();

			if(err){
				logger.error('--->enable: Req. Id=' + res._Context.reqId + ', Plugin enable request failed. Reason:' + err.stack);
				return res.status(404).end('Plugin enable request failed. Reason:' + err.stack);
			}
			if(logger.isDebugEnabled()){
				_logAppRoutes();
			}
			logger.info('--->enable: Req. Id=' + res._Context.reqId + ', over');			
			return res.end('done');
		});
	});
}

/**
 * Load the given plugin.
 * @param  {req} 		req as GET or POST 
 * 		   GET URL: 	<server>[:port]/appservices/pm/load?module=oxsnps-afp2any
 * 		   POST URL: 	<server>[:port]/appservices/pm/load
 * 		   POST parms:	req.body = {
 * 		            		"module":"oxsnps-afp2any", 
 * 		              	}
 * @param  {res} 	res
 */
exports.loadPlugin = function(req, res){
    
	var dirs = []
	  , plugindir = npsDir + '/node_modules/' // default plugin dir
	  , groups = []
      , bPluginFound = false;

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the plugin name to the Context
	if(req.method === 'GET') res._Context.module = req.query.module;
	else res._Context.module = req.body.module;

	logger.info('--->load: Req. Id=' + res._Context.reqId);
	if(logger.isDebugEnabled()) logger.debug('--->load: Req. Id=' + res._Context.reqId + ', Parms: ' + (req.method === 'GET'?JSON.stringify(req.query):JSON.stringify(req.body)));

	// Is the Plugin already loaded?
	_getPlugin(res._Context.module, function(err, plugin){
		if(!err){
			logger.info('--->loadPlugin: Req. Id=' + res._Context.reqId + ', Plugin already enabled.');
			return res.end('Plugin already enabled.');
		}

		// If the plugin has not been loaded, load it
		// Get the list of the subdirs contained in the npsConf.pluginDirs array
		if(npsConf.pluginDirs){
		    if(npsConf.pluginDirs instanceof Array){
		        for (var i = 0; i< npsConf.pluginDirs.length; i++){
		            utils.listDirsSync(path.resolve(npsDir + '/' + npsConf.pluginDirs[i]), dirs, true);
		        };   
		    }
		    else{   
		        // Not an array, so it must be one dir only
		        plugindir = path.resolve(npsDir + '/' + npsConf.pluginDirs);
		        utils.listDirsSync(plugindir, dirs, true);
		    }
		}
		else{
		    // Use the default plugin dir
		    utils.listDirsSync(plugindir, dirs, true);
		}

		// Loop through all the dirs
		async.each(dirs,
			function(dir, nextTask){

				// Is it the dir of the plugin?
			    groups = dir.name.match(/(.*\/)(.*)/);
				if(groups[2] === res._Context.module){
					if(fs.existsSync(dir.name + '/' + pluginJSfilename)){ // dir.name is the FQ name of the plugin
						bPluginFound = true;
						// Load and initialize the plugin
						_loadPlugin(dir.name, nextTask);
					}
					else{
						nextTask();
					}
				}
				else{
					nextTask();
				}
			},
			function(err){

				// Ensure to move the '/*' route to the end of the list of routes
				expressAppUtils.moveDefaultAppRoute();

				if(err){
					logger.error('--->loadPlugin: Req. id=' + res._Context.reqId + ', Loading Plugin ' + res._Context.module + ' failed, Reason:' + err.stack);
					return res.status(404).end('Loading Plugin ' + res._Context.module + ' failed, Reason:' + err.stack);
				}
				if(bPluginFound){
					return res.end('Plugin ' + res._Context.module + ' loaded.');
				}
				return res.status(404).end('Plugin ' + res._Context.module + ' not found!');
			}
		);
	});
}

/**
 * Disable the given plugin
 * @param  {req} 		req as GET or POST 
 * 		   GET URL: 	<server>[:port]/appservices/pm/disable?module=oxsnps-afp2any
 * 		   POST URL: 	<server>[:port]/appservices/pm/disable
 * 		   POST parms:	req.body = {
 * 		            		"module":"oxsnps-afp2any", 
 * 		              	}
 * @param  {res} 		res
 */
exports.disablePlugin = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	// Add the plugin name to the Context
	if(req.method === 'GET') res._Context.module = req.query.module;
	else res._Context.module = req.body.module;

	logger.info('--->disable: Req. Id=' + res._Context.reqId);
	if(logger.isDebugEnabled()) logger.debug('--->disable: Req. Id=' + res._Context.reqId + ', Parms: ' + (req.method === 'GET'?JSON.stringify(req.query):JSON.stringify(req.body)));

	// Get Plugin
	_getPlugin(res._Context.module, function(err, plugin){
		if(err){
			logger.error('--->disablePlugin: Req. Id=' + res._Context.reqId + ', Plugin disable request failed: ' + err.stack + '. Parms:' + JSON.stringify(req.body));
			return res.status(404).end('Plugin disable request failed: ' + err.stack + '. Parms:' + JSON.stringify(req.body));
		}

		// If Plugin is already disabled, leave
		if(plugin.enable && plugin.enable !== 'on'){
			return res.end('Plugin ' + res._Context.module + ' already disabled');
		}

		// V3.0.13 Begin
		// deactivate plugin
		_deactivatePlugin(res, plugin, function(err){
			if(err){
				logger.error('--->disable: Req. Id=' + res._Context.reqId + ', Plugin disable request failed. ' + err.stack);
				return res.status(404).end('Plugin disable request failed. ' + err.stack);
			}
			// V3.0.13 End
			// Disable the plugin
			_disablePlugin(plugin, function(err){
				if(err){
					logger.error('--->disable: Req. Id=' + res._Context.reqId + ', Plugin disable request failed. ' + err.stack);
					return res.status(404).end('Plugin disable request failed. ' + err.stack);
				}
				if(logger.isDebugEnabled()){
					_logAppRoutes();
				}
				logger.info('--->disable: Req. Id=' + res._Context.reqId + ', over');
				return res.end('done');
			});
		});
	});
}

/**
 * Enable the given route
 * @param  {req} 		req as GET or POST 
 * 		   GET URL: 	<server>[:port]/appservices/pm/enableroute?module=oxsnps-afp2any&route={...}
 * 		   POST URL: 	<server>[:port]/appservices/pm/enableroute
 * 		   POST parms:	req.body = {
 * 		            		"module":"oxsnps-afp2any", 
 * 		            	 	"route": { 
 * 		            		 	"tenant": <tenant> Optional, 
 * 		            		 	"path": "/path/to/the/route",
 * 		            		 	"method": <Optional. Values are 'get'|'post'|'dw'|'queue'>. 
 * 		            		 			  No default. If given method+path will be used to find the route 
 * 		            		 			  else path alone is used to find the route.
 * 		            		 			  This is useful since same path (especially HTTP Paths) can be used 
 * 		            		 			  with different methods(get|post)
 * 		            		  	...
 * 		            	 	}
 * 		              	}
 * @param  {res}		res
 */
exports.enableRoute = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	req.body = req.body || {};
	
	// Add Get Parameters to req.body
	if(req.method === 'GET'){
		req.body.module = req.query.module;
		req.body.route = utils.parseJSON(req.query.route);
	}

	res._Context.enableRouteReq = {
		'body':{
			'module': req.body.module,
			'route' : req.body.route
		}
	};

	exports.enableRouteService(res, function(err){
		if(err && err instanceof warn) return res.end(err.message);
		else if(err) return res.status(404).end('Unable to enable a Route. Req. Id=' + 
											res._Context.reqId + ', ' + err.message);
		res.end('Plugin ' + req.body.module + ', ' + (req.body.route.method || 'get') + ' route ' + 
					req.body.route.path + ' has been enabled');
	});
}

/**
 * Enable the given route
 * @param  {res} 		Response Object
 * 		   POST parms:	req._Context = {
 * 		   					"enableRouteReq"."body" : {
 * 		            			"module": "oxsnps-afp2any", 
 * 		            	 		"route": { 
 * 	                 	           		 	"tenant": <tenant> Optional,  
 * 	                 	           		 	"path": "/path/to/the/route",
 * 		                                 	"method": <Optional. Values are 'get'|'post'|'dw'|'queue'>. 
 * 		            		 	                      No default. If given method+path will be used to find the route 
 * 		            		 	                      else path alone is used to find the route.
 * 		            		 			              This is useful since same path (especially HTTP Paths) can be used 
 * 		            		 			              with different methods(get|post)
 * 		            		  	             ...
 * 		            	 	             }
 * 		              	        }
 * 		              	}
 * @param  {Function} 	callback(err)
 */
exports.enableRouteService = function(res, callback){

	var fqPlugin 		= ''
	  , req 			= undefined
	  , msg 			= undefined
	  , module 			= undefined
	  , routeStateObj 	= undefined
	  , reqRouteState 	= undefined
	  , saveWarn 		= undefined	  
      , resTmp			= undefined
      , sfwExists       = _pluginExists(sfwPluginName)  // V4.1.7
	  ;

	// Add Context to the response instance
	res._Context = res._Context || {};

	// Add the Current Date to the Context
	res._Context.date = res._Context.date || new Date();

	// Add a Request Id to the Context
	res._Context.reqId = res._Context.reqId || utils.buildReqId(res._Context.date);

	req = res._Context.enableRouteReq || {};

	logger.info('--->enableRouteService: Req. Id=' + res._Context.reqId + ', Parms:' + JSON.stringify(req.body));

	// 1. Assert Parms
	if(req.body === undefined || req.body.module === undefined || req.body.route === undefined){
		logger.error('--->enableRouteService: Req. Id=' + res._Context.reqId + 
					 ', Wrong request: Parms:' + JSON.stringify(req));
		return callback(new Error('Wrong request: Parms:' + JSON.stringify(req)));
	}
	// V3.0.14 Begin
	routeStateObj = req.body.route.stateInfo || undefined; // req.body.route.stateInfo given only from plugins and not for HTTP requests
	if(routeStateObj) reqRouteState = routeStateObj.state; // Save route state passed in routeStateObject
	req.body.route.stateInfo = undefined; 	// Remove state info from route, otherwise it will be written to plugin conf file
	// V3.0.14 End
	
	res._Context.sfw = {'route': req.body.route, 'plugin':req.body.module}	// V4.1.7
	
	async.series([
		// Get Route State Info
	    function(nextTask){
			if(!sfwExists) return nextTask()	// V4.1.7
	    	if(!routeStateObj || !routeStateObj.filename || !routeStateObj.plugin){	
				//exports.getRouteStateInfo(req.body.route.path, function(err, result){
				exports.callPluginService({name:'oxsnps-statefilewatcher',service:'getRouteStateInfo'}, res, function(err, result){
					if(err) return nextTask(err);
					if(result) routeStateObj = result;
					// Check if route is deactivated, if deactivated route can not be enabled or disabled
					if(routeStateObj && routeStateObj.state && routeStateObj.state.toLowerCase() === exports.DEACTIVATED){
						logger.warn('--->enableRouteService: Req. id=' + res._Context.reqId + 
						(routeStateObj.tenant ? (', tenant:'+  routeStateObj.tenant) : '') +
						', plugin: ' + req.body.module + ', route: ' + req.body.route.path + ' can not be enabled since it is deactivated.');
						return nextTask(new warn((routeStateObj.tenant ? ('tenant:'+  routeStateObj.tenant + ', ') : '') + 'plugin: ' + req.body.module + ', route: ' + req.body.route.path + ' can not be enabled since it is deactivated'));
					}
					nextTask();
				});
			} else nextTask();
		},
		// Get Plugin
	    function(nextTask){
	    	_getPlugin(req.body.module, function(err, result){
				if(err){
					logger.error('--->enableRouteService: Req. Id=' + res._Context.reqId + ', Request failed: ' + err.message + ' Parms:' + JSON.stringify(req.body));
					return nextTask(new Error(err.message + ', Parms:' + JSON.stringify(req.body)));
				}
   				// Add the plugin to the Context
				res._Context.plugin = result;
	    		nextTask();
	    	});
	    },
	    // Disable Route
	    function(nextTask){
	    	_enableRoute(res, function(err, result){
	    		// V3.0.14 Begin
	    		if(err && err instanceof warn && res._Context.route && res._Context.route.enable === 'on'){
	    			// Route is already enabled. Still we have to ensure route state file has enable status
	    			// so save warn and proceed further
					saveWarn = err;
					return nextTask();;	    		
				}
				// V3.0.14 End
				return nextTask(err);
	    	});
	    },

	    // Save plugin configuration file
	    function(nextTask){
	    	if(saveWarn) return nextTask(); // V3.0.14
			if (savePluginConf && res._Context.pluginHandle.saveConfService && typeof(res._Context.pluginHandle.saveConfService) === 'function'){	// added "savePluginConf" flag for OTS-2326
				res._Context.pluginHandle.saveConfService(function(err){
					if(err){
						logger.warn('--->enableRouteService: Req. Id=' + res._Context.reqId + ', ' + err.message);
						return nextTask(new warn('Plugin ' + req.body.module + ', ' + (req.body.route.method || 'get') + ' route ' + req.body.route.path + ' disabled1. But unable to save plugin configuration. ' + err.message));
					}
					nextTask()
				});
			}
			else nextTask();
	    },	    

	    // Update route state file
	    function(nextTask){
	    	if(!routeStateObj) return nextTask(); // state file not available

            if(res._Context.route && utils.isHTTPRoute(res._Context.route)) return nextTask()	// V5.0.6 state files not yet supported for HTTP routes 
			// state file already in proper state. No need to udpate it
	    	//if(routeStateObj.state === exports.ENABLE) return nextTask();

			// Update route state file
			routeStateObj.state = reqRouteState || exports.ENABLE; // V3.0.14
			//_updateRouteStatefile(routeStateObj, function(err){
            resTmp = {
                '_Context': {
                    'reqId':    res._Context.reqId,
                    'sfw':  {'routeStateObj': routeStateObj}
                }
            }
            
			exports.callPluginService({name:'oxsnps-statefilewatcher',service:'updateRouteStatefile'}, resTmp, function(err, result){
				if(err){
					logger.warn('--->enableRouteService: Req. id=' + res._Context.reqId + ', ' + err.message);
					return nextTask(new warn('Plugin ' + req.body.module + ', ' + (req.body.route.method || 'get') + ' route ' + req.body.route.path + ' disabled. But state in route state file ' + routeStateObj.filename + 'can not be updated. ' + err.message));
				}
				return nextTask();
			});
	    }	    	    
	],
	function(err, results){
		if(saveWarn) return callback(saveWarn); 	// V3.0.14
		if(!err){
			logger.info('--->enableRouteService: Req. Id=' + res._Context.reqId + 
					' Plugin ' + req.body.module + ', ' + (req.body.route.method || 'get') + ' route ' + req.body.route.path + ' has been enabled');
		}
		callback(err);
	});
}

/**
 * Disable the given route
 * @param  {req} 		req as GET or POST 
 * 		   GET URL: 	<server>[:port]/appservices/pm/disableroute?module=oxsnps-afp2any&route={...}
 * 		   POST URL: 	<server>[:port]/appservices/pm/disableroute
 * 		   POST parms:	req.body = {
 * 		            		"module":"oxsnps-afp2any", 
 * 		            	 	"route": { 
 * 		            		 	"path": "/path/to/the/route",
 * 		            		 	"method": <Optional. Values are 'get'|'post'|'dw'|'queue'>. 
 * 		            		 			  No default. If given method+path will be used to find the route 
 * 		            		 			  else path alone is used to find the route.
 * 		            		 			  This is useful since same path (especially HTTP Paths) can be used 
 * 		            		 			  with different methods(get|post)
 * 		            		  	...
 * 		            	 	},
 * 		            	 	"reason": { // optional
 * 		            		 	"reqId": res._Context.reqId,
 * 		            		 	"desc":  res._Context.proJob.error.server + ' assertion failed',
 * 		            		 	"error":{
 * 		            		 	    //"code":   // not used 
 * 		            		 	    "msg': res._Context.proJob.error.msg
 * 		            		 	}
 * 		            	 	}
 * 		              	}
 * @param  {res}		res
 */
exports.disableRoute = function(req, res){

	// Add Context to the response instance
	res._Context = {};

	// Add the Current Date to the Context
	res._Context.date = new Date();

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date);

	req.body = req.body || {};

	// Add Get Parameters to req.body
	if(req.method === 'GET'){
		req.body.module = req.query.module;
		req.body.route = utils.parseJSON(req.query.route);
	}

	res._Context.disableRouteReq = {
		'body':{
			'module': req.body.module,
            'route' : req.body.route,
            'reason' : req.body.reason  // V4.1.15
		}
	};

	exports.disableRouteService(res, function(err){
		if(err && err instanceof warn) return res.end(err.message);
		else if(err) return res.status(404).end('Unable to disable a Route. Req. Id=' + res._Context.reqId + ', ' + err.message);
		res.end('Plugin ' + req.body.module + ', ' + (req.body.route.method || 'get') + ' route ' + 
				req.body.route.path + ' has been disabled');
	});
}

/**
 * Disable the given proxy route
 * @param  {req} 		req as GET
 * 		   GET URL: 	<server>[:port]/appservices/pm/disableroute?route="/path/to/the/proxy/route"
 * 		            		 	
 */
exports.disableProxyRoute = function(req, res){

	// Add Context to the response instance
	res._Context = {}

	// Add the Current Date to the Context
	res._Context.date = new Date()

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date)

	// Add the proxy route to the Context
	res._Context.route = {'path': req.query.route}

	exports.disableProxyRouteService(res, function(err){
		if(err) return res.status(404).end('--->disableProxyRoute: Req. Id=' + res._Context.reqId + 
										   ',  Unable to disable Route; ' + res._Context.route + '. Reason: ' + err.message)
		res.end('--->disableProxyRoute: Req. Id=' + res._Context.reqId + 
		 		', Route ' + JSON.stringify(res._Context.route) + ' has been disabled');
	});
}

/**
 * Disable the given proxy route
 * @param  {res} 		Response Object
 * 		   POST parms:	req._Context.route": { 
 * 	                 		"path": "/path/to/the/proxy/route"
 * 		              	}
 * @param  {Function} 	callback(err)
 */
exports.disableProxyRouteService = function(res, callback){

	if(res._Context.route === undefined) // assert route
		return callback(new Error('Wrong request: Parms:' + JSON.stringify(res._Context)))

	expressAppUtils.disableRoute(res._Context.route, callback)
}

/**
 * Disable the given route
 * @param  {res} 		Response Object
 * 		   POST parms:	req._Context = {
 * 		   					"disableRouteReq"."body" : {
 * 		            			"module": "oxsnps-afp2any", 
 * 		            	 		"route": { 
 * 	                 	           		 	"tenant": <tenant> Optional,   
 * 	                 	           		 	"path": "/path/to/the/route", 
 * 		                                 	"method": <Optional. Values are 'get'|'post'|'dw'|'queue'>. 
 * 		            		 	                      No default. If given method+path will be used to find the route 
 * 		            		 	                      else path alone is used to find the route.
 * 		            		 			              This is useful since same path (especially HTTP Paths) can be used 
 * 		            		 			              with different methods(get|post)
 * 		            		  	             ...
 * 		            	 	             }
 * 		            	 	    "reason": { // optional
 * 		            		 	    "reqId": res._Context.reqId,
 * 		            		 	    "desc":  res._Context.proJob.error.server + ' assertion failed',
 * 		            		 	    "error":{
 * 		            		 	        //"code":   // not used 
 * 		            		 	        "msg': res._Context.proJob.error.msg
 * 		            		 	    }
 * 		            	 	    }
 * 		              	}
 * @param  {Function} 	callback(err)
 */
exports.disableRouteService = function(res, callback){

	var fqPlugin 		= ''
	  , req 			= undefined
	  , msg 			= undefined
	  , module 			= undefined
	  , routeStateObj 	= undefined
	  , plugin 			= undefined
	  , reqRouteState 	= undefined
	  , saveWarn 		= undefined
      , resTmp			= undefined
      , sfwExists       = _pluginExists(sfwPluginName)  // V4.1.7
	  ;

	// Add Context to the response instance
	res._Context = res._Context || {};

	// Add the Current Date to the Context
	res._Context.date = res._Context.date || new Date();

	// Add a Request Id to the Context
	res._Context.reqId = res._Context.reqId || utils.buildReqId(res._Context.date);

	req = res._Context.disableRouteReq || {};

	logger.info('--->disableRouteService: Req. Id=' + res._Context.reqId + ', Parms:' + JSON.stringify(req.body));

	// 1. Assert Parms
	if(req.body === undefined || req.body.module === undefined || req.body.route === undefined){
		logger.error('--->disableRouteService: Req. Id=' + res._Context.reqId + 
					 ', Wrong request: Parms:' + JSON.stringify(req));
		return callback(new Error('Wrong request: Parms:' + JSON.stringify(req)));
	}

	routeStateObj = req.body.route.stateInfo || undefined; // req.body.route.stateInfo given only from plugins and not for HTTP requests
	if(routeStateObj) reqRouteState = routeStateObj.state; // Save route state passed in routeStateObject
	req.body.route.stateInfo = undefined; 	// Remove state info from route, otherwise it will be written to plugin conf file
	
    res._Context.sfw = {'route': req.body.route, 'plugin':req.body.module}	// V4.1.7
	
	async.series([
		// Get Route State Info
	    function(nextTask){
			if(!sfwExists) return nextTask()	// V4.1.7
	    	if(!routeStateObj || !routeStateObj.filename || !routeStateObj.plugin){

				//exports.getRouteStateInfo(req.body.route.path, function(err, result){
				exports.callPluginService({name:'oxsnps-statefilewatcher',service:'getRouteStateInfo'}, res, function(err, result){
					if(err) return nextTask(err);
					if(result) routeStateObj = result;
					nextTask();
				});
	    	} else nextTask();
		},
		// Get Plugin
	    function(nextTask){
	    	_getPlugin(req.body.module, function(err, result){
				if(err){
					logger.error('--->disableRouteService: Req. Id=' + res._Context.reqId + ', Request failed: ' + err.message + ' Parms:' + JSON.stringify(req.body));
					return nextTask(new Error(err.message + ', Parms:' + JSON.stringify(req.body)));
				}
   				// Add the plugin to the Context
				res._Context.plugin = result;
	    		nextTask();
	    	});
	    },
	    // Disable Route
	    function(nextTask){
	    	_disableRoute(res, function(err, result){
	    		if(err && err instanceof warn && res._Context.route && res._Context.route.enable !== 'on'){
	    			// Route is already disabled. Still we have to update the route state file with
	    			// proper state(disabled, disablebyerror, disablebystop), so save warn and proceed further
					saveWarn = err;
					return nextTask();;	    		
				}
				if(err)	return nextTask(err);
				res._Context.route.enable = 'off'; 	// set enable 'off' 
	    		nextTask();
	    	});	    	
	    },

	    // Save plugin configuration file
	    function(nextTask){
	    	if(saveWarn) return nextTask();
			if (savePluginConf && res._Context.pluginHandle.saveConfService && typeof(res._Context.pluginHandle.saveConfService) === 'function'){	// added "savePluginConf" flag for OTS-2326
				res._Context.pluginHandle.saveConfService(function(err){
					if(err){
						logger.warn('--->disableRouteService: Req. Id=' + res._Context.reqId + ', ' + err.message);
						return nextTask(new warn('Plugin ' + req.body.module + ', ' + (req.body.route.method || 'get') + ' route ' + req.body.route.path + ' disabled1. But unable to save plugin configuration. ' + err.message));
					}
					nextTask()
				});
			}
			else nextTask();
	    },	    

	    // Update route state file
	    function(nextTask){
            if(!routeStateObj) return nextTask(); // state file not available
            
            if(res._Context.route && utils.isHTTPRoute(res._Context.route)) return nextTask()	// V5.0.6 state files not yet supported for HTTP routes 

			// state file already in proper state. No need to udpate it
	    	//if(routeStateObj.state === exports.DEACTIVATED || routeStateObj.state === exports.DISABLE) return nextTask();

			// Update route state file
			routeStateObj.state = reqRouteState || exports.DISABLE;
			//_updateRouteStatefile(routeStateObj, function(err){
            resTmp = {
                '_Context': {
                    'reqId':    res._Context.reqId,
                    'sfw':  {'routeStateObj': routeStateObj, 'reason': req.body.reason}	// V4.1.15
                }
            }
			exports.callPluginService({name:'oxsnps-statefilewatcher',service:'updateRouteStatefile'}, resTmp, function(err, result){
				if(err){
					logger.warn('--->disableRouteService: Req. Id=' + res._Context.reqId + ', ' + err.message);
					return nextTask(new warn('Plugin ' + req.body.module + ', ' + (req.body.route.method || 'get') + ' route ' + req.body.route.path + ' disabled. But state in route state file ' + routeStateObj.filename + 'can not be updated. ' + err.message));
				}
				return nextTask();
			});
	    }	    	    
	],
	function(err, results){
		if(saveWarn) return callback(saveWarn);
		else if(!err){
			logger.info('--->disableRouteService: Req. Id=' + res._Context.reqId + 
					' Plugin ' + req.body.module + ', ' + (req.body.route.method || 'get') + ' route ' + req.body.route.path + ' has been disabled');
		}
		callback(err);
	});
}

/**
 * Return the list of the plugin routes
 * @param  {req} 	GET req
 * @param  {res}	res
 */
exports.getRoutes = function(req, res){

	logger.info('--->getRoutes: Getting the list of the Plugins and their Routes...');

	// Assert plugins array
	if(plugins){
		res.json(plugins);
	}
	else{
		res.json([]); // return an empty array
	}
	logger.info('--->getRoutes: List of the Plugins and their Routes returned, over.');
}

/**
 * Return the log level of the given plugin
 * @param  {req} 	GET req   http://localhost:1026/appservices/pm/getloglevel?module=oxsnps-afp2any
 * @param  {res}	res
 */
exports.getLogLevel = function(req, res){

	logger.info('--->getLogLevel: Getting the log level of a plugin or of the Plugin Manager...');

	//!!! Must be a better way to get parms from a HTTP GET req.
	if(req.query.module === undefined){
		return res.status(404).end('Module name is missing. API call: http://<server>[:<port>/appservices/pm/getloglevel?module=<moduleName>');
	}

	var moduleName 	= req.query.module;

	// Assert parms
	if(!moduleName){
		return res.status(404).end('Missing module, Parms: ' + JSON.stringify(req.query));
	}

	// If the module is Plugin Manager
	if(moduleName.toLowerCase() === 'pm' || moduleName.toLowerCase() === 'plugin manager'){
		return res.end(logger.level.levelStr);
	}
	else{	
		// Get the handle of the given module if enabled
		exports.getModule(moduleName, function(err, module){
	    	if(err) return res.status(404).end(''+err);

			// Call getLogLevel service of the plugin if it exists
			if (typeof(module.getLogLevel) === 'function'){

				// Try to call getloglevel
				try{
			    	return res.end(module.getLogLevel());
					logger.info('--->getLogLevel: Log level of ' + moduleName + ' is ' + loglevel);
				}catch(err){
					return res.status(404).end('Calling ' + moduleName + ' .getloglevel() failed.' + err.stack);
				}
			}
			else{
				logger.warn('--->getLogLevel: ' + moduleName + '.getloglevel not available.');
				return res.status(404).end('Plugin ' + moduleName + '.getloglevel not available.');
			}
		});
	}
}

/**
 * Set the log level of the given plugin
 * @param  {req} 	GET req   http://localhost:1026/appservices/pm/setloglevel?module=oxsnps-afp2any&loglevel=DEBUG|INFO|ERROR
 * @param  {res}	res
 */
exports.setLogLevel = function(req, res){

	logger.info('--->setLogLevel: Setting the log level of a plugin...');

	//!!! Must be a better way to get parms from a HTTP GET req.
	if(req.query.module === undefined){
		return res.status(404).end('Module name is missing. API call: http://<server>[:<port>/appservices/pm/setloglevel?module=<moduleName>loglevel=DEBUG|INFO|ERROR');
	}

	var moduleName 	= req.query.module
	  , loglevel 	= req.query.loglevel || 'INFO'
	  ;

	// Assert parms
	if(!moduleName || !loglevel ){
		return res.status(404).end('Missing module or logLevel, Parms: ' + JSON.stringify(req.query));
	}

	// If the module is Plugin Manager
	if(moduleName.toLowerCase() === 'pm' || moduleName.toLowerCase() === 'plugin manager'){
		utils.log4jsSetLogLevel(logger, loglevel);
		return res.end(loglevel);
	}
	else{	
		// Get the handle of the given module if enabled
		exports.getModule(moduleName, function(err, module){
	    	if(err) return res.status(404).end(''+err);

			// Call setLogLevel service of the plugin if it exists
			if (typeof(module.setLogLevel) === 'function'){

				// Try to call setLoglevel
				try{
					module.setLogLevel(loglevel);
					// Get plugin object
					_getPlugin(moduleName, function(err, plugin){
						if(err) return res.status(404).end(''+err);
						// Save plugin Configuration file
						if (savePluginConf && module.saveConfService && typeof(module.saveConfService) === 'function'){	// added "savePluginConf" flag for OTS-2326
							module.saveConfService(function(err){
								if(err) return res.end('--->setLogLevel: Unable to set log level to ' + loglevel + ', Reason: ' + err.message);
								plugin.logLevel = loglevel // update the GUI JSON Structure

								res.end('--->setLogLevel: Log level of ' + moduleName + ' set to ' + loglevel);
							});
						}
						else{
							plugin.logLevel = loglevel // update the GUI JSON Structure
							return res.end('--->setLogLevel: Log level of ' + moduleName + ' set to ' + loglevel);
						}
					});
				}catch(err){
					return res.status(404).end('Calling ' + moduleName + ' .setLoglevel() failed.' + err.stack);
				}
			}
			else{
				logger.warn('--->setLogLevel: ' + moduleName + '.setLoglevel not available.');
				return res.status(404).end('Plugin ' + moduleName + '.setLoglevel not available.');
			}
		});
	}
}

exports.logHTTPRoutes = function(req, res){
	res.json(_logAppRoutes())
}

/**
 * Enable the plugins found in the dirs listed in npsConf.pluginDirs
 * @param  {function}   callback(err)
 *
 */
exports.enablePlugins = function(callback){
	let pluginname = ''
	logger.info('-->Enabling Plugins...');

	// Search the plugin dir for plugins
	var dirs 			= []
	  , fqPluginName	= undefined

	if(appname){ // V5
		let pluginList = npsConf.pluginList
		for (let i = 0; i< pluginList.length; i++){
			for (let j = 0; j< npsConf.pluginDirs.length; j++){
				fqPluginName = path.resolve(npsDir + FILE_SEP + npsConf.pluginDirs[j] + FILE_SEP + pluginList[i])
				if(fs.existsSync(fqPluginName)){
					dirs.push({'name':fqPluginName})
					break
				}
			}
		}
	}
	else{	// backward compatibility
		// Get the list of the subdirs contained in the npsConf.pluginDirs array
		for (var i = 0; i< npsConf.pluginDirs.length; i++){
			// {name:"/home/daniel/DEV/oxs-server/node_modules/ce.wps"}
			utils.listDirsSync(path.resolve(npsDir + '/' + npsConf.pluginDirs[i]), dirs, true)
		}
	}

    // V5.0.6 Begin
    // Loop through all the dirs
    //async.each(dirs,
    async.eachSeries(dirs,
        function(dir, next){
			pluginname = dir.name
            if(fs.existsSync(dir.name + '/' + pluginJSfilename)){ // dir.name is the FQ name of the plugin
                // Load and initialize the plugin
                return _loadPlugin(dir.name, next)
            }
            next()
        },
        function(err){
            // Add the '/*' route to the end of the list of routes to map undefined routes
            expressApp['get']('/*', expressAppUtils.routeNotAvailable)
            expressApp['post']('/*', expressAppUtils.routeNotAvailable)

            if(logger.isDebugEnabled()){
                _logPlugins()
                _logAppRoutes()
            }
            if(err) logger.error('--->enablePlugins: Enable Plugins failed. ' + (pluginname? 'Plugin: '+  path.basename(pluginname)+ ', ' :'') + err.message)
            callback(err)
        }
    )

    /*
	async.series([
		// Load the plugins
		function(nextTask){
			// Loop through all the dirs
            async.each(dirs,
            //async.eachSeries(dirs,
				function(dir, next){
					if(fs.existsSync(dir.name + '/' + pluginJSfilename)){ // dir.name is the FQ name of the plugin
						// Load and initialize the plugin
						_loadPlugin(dir.name, next)
					}
					else next()
				},
				function(err){
					// Add the '/*' route to the end of the list of routes to map undefined routes
					expressApp['get']('/*', expressAppUtils.routeNotAvailable)
					expressApp['post']('/*', expressAppUtils.routeNotAvailable)

					if(logger.isDebugEnabled()){
						_logPlugins()
						_logAppRoutes()
					}
					if(err) logger.error('--->enablePlugins: Enable Plugins failed: ' + err.message)
					nextTask(err)
				}
			)
        },
		// Get state file information for all plugins
		function(nextTask){
			sfwExists       = _pluginExists(sfwPluginName)  // V4.1.7
			if(!sfwExists) return nextTask()	
			res._Context = {'sfw': {}}
			exports.callPluginService({name:'oxsnps-statefilewatcher',service:'getStateFilesInfo'}, res, function(err, result){
				if(err) return nextTask(new Error('Unable to load state of routes. ' + err.message))
				routeStateInfo = result || {}
				logger.info('-->enablePlugins: RouteStateInfo=' + JSON.stringify(routeStateInfo))
				nextTask()
			})
		},
		// Update channel(route) enable propery, with state specified in route state file
		function(nextTask){
			if(!sfwExists) return nextTask()        // V4.1.7
			async.eachSeries(plugins,
				function(plugin, next){
					if(plugin.enable !== 'on') return next()
					res._Context = {
						'sfw': {
							'plugin':			plugin.module, 
							'routeStateInfo': 	routeStateInfo,
							'routes': 			plugin.routes
						}
					}
					exports.callPluginService({name:'oxsnps-statefilewatcher',service:'updateRoutesWithStateInfo'}, res, next)
				},
				function(err){
					return nextTask(err)
				}
			)
        }

	],
	function(err){
		callback(err)
    })
    */
    // V5.0.6 End    
}

exports.activatePlugins = function(callback){
 	
	logger.info('-->Activating Plugins...');

    // Loop through all plugins
    //async.each(plugins,	// 5.0.27
	async.eachSeries(plugins,
		function(plugin, nextTask){

			if(plugin.enable !== 'on') return nextTask(); // do nothing if plugin is disabled

			// Activate the plugin
			_activatePlugin(plugin, nextTask);
		},
		function(err){
			// While initializing plugins, we read all route state files of all plugins and keep in'routeStateInfo' to decide whether to add route or not. 
			// After initializing, reset routeStateInfo in memory. 
			// When route state info is needed later, it should be read from state file
			//routeStateInfo 	= undefined;
			if(err) logger.error('--->activatePlugins: Activating Plugins failed. ' + err.message);
			callback(err);
		}
	);
}

// V3.0.16 Begin
/**
 * Perform tasks to be done after start of server
 * @param  {Function} callback callback(err)
 */
exports.afterServerStart = function(callback){

	var fqPlugin	= undefined
      , module 		= undefined
      , initError   = undefined
      , resTmp      =  undefined
	  ;

	logger.info('-->afterServerStart: Starting tasks to do after start of server...');

	// Loop through all plugins
	async.eachSeries(plugins,
		function(plugin, nextTask){
  			// Assert plugin 
			  if(!plugin || !plugin.plugindir || !plugin.module)
				return nextTask(new Error('Plugin configuration parameter is null'));

			if(plugin.enable !== 'on') return nextTask(); // do nothing if plugin is disabled

			fqPlugin = plugin.plugindir + plugin.module + '/' + pluginJSfilename;
		
            if(logger.isDebugEnabled()) logger.debug('-->afterServerStart: Starting after server start tasks for ' + fqPlugin + '  ...');
            module 		= undefined
            initError   = undefined
			// 5.0.6 Begin
            /*
			// Get the module handle
			try{
				module = require(fqPlugin);
				if(!module) return nextTask(new Error('Error loading Plugin ' + fqPlugin));
			}catch(err){
				return nextTask(new Error('Error loading Plugin ' + fqPlugin + '. ' + err.stack));
			}
			// Call afterServerStart service of the plugin if it exists
			if (typeof(module.afterServerStart) === 'function'){
		
				// Call afterServerStart
				try{
					module.afterServerStart(nextTask);
				}catch(err){
					nextTask(new Error('Calling ' + plugin.module + ' .afterServerStart() failed.' + err.stack));
				}
			}
			else{
				if(logger.isDebugEnabled()) logger.debug('--->afterServerStart: ' + plugin.module + '.afterServerStart() not available.');
				nextTask();
            }	
            */

            plugin.initErrors = plugin.initErrors || []

            async.series([
                function(next){
                    // Get the module handle
                    try{
                        module = require(fqPlugin)
                        //if(!module) return next(new Error('Error loading Plugin ' + fqPlugin));
                        if(!module) initError = new Error('Error loading Plugin ' + fqPlugin)
                        if(!module && npsConf.stopOnInitErrors) return next(initError)
                    }catch(err){
                        if(err && npsConf.stopOnInitErrors) return next(new Error('Error loading Plugin ' + fqPlugin + '. ' + err.stack))
                        initError = err
                    }
                    if(initError){
                        logger.error('--->afterServerStart: Error on calling afterServerStart() for plugin: ' + plugin.module +  ((module.pluginConf && module.pluginConf.version) ? ('v' + module.pluginConf.version) : '')  + '. Reason: ' + initError.message)
                        plugin.initErrors.push(initError.message)
                    }
                    return next()
                },                
                function(next){
                    if(initError || typeof(module.afterServerStart) !== 'function') return next()
                    // Activate the plugin
                    try{
                        module.afterServerStart(function(err){
                            if(err && npsConf.stopOnInitErrors) return next(new Error('Error on calling afterServerStart() for plugin ' + fqPlugin + '. ' + err.message))
                            if(err){
                                logger.error('--->afterServerStart: Error on calling afterServerStart() for plugin: ' + plugin.module +  ((module.pluginConf && module.pluginConf.version) ? ('v' + module.pluginConf.version) : '')  + '. Reason: ' + err.message)
                                plugin.initErrors.push(err.message)
                                initError = err
                            }
                            next()
                        })
                    }catch(err){
                        if(err && npsConf.stopOnInitErrors) return next(new Error('Calling ' + plugin.module + ' .afterServerStart() failed.' + err.message))
                        if(err){
                            logger.error('--->afterServerStart: Error on calling afterServerStart() for plugin: ' + plugin.module +  ((module.pluginConf && module.pluginConf.version) ? ('v' + module.pluginConf.version) : '')  + '. Reason: ' + err.message)
                            plugin.initErrors.push(err.message)
                            initError = err
                        }
                        next()
                    }
                },
                // in case of plugin initialization errors, disable plugin routes 
                function(next){
                    if(npsConf.stopOnInitErrors || !initError) return next()		  
                    // log error
                    //logger.error('--->afterServerStart: Error on calling afterServerStart() for plugin: ' + plugin.module +  ((module.pluginConf && module.pluginConf.version) ? ('v' + module.pluginConf.version) : '')  + '. Reason: ' + initError.message)
                    // disable plugin routes
                    resTmp = {
                        '_Context': {
                            'reqId': utils.buildReqId(new Date()),
                            'disableRoutesReq': {
                                'module': plugin.module
                            }
                        }
                    }
                
                    if(logger.isDebugEnabled()) logger.debug('--->afterServerStart: Req. id=' + resTmp._Context.reqId + ', Disabling all the routes of a plugin: ' + resTmp._Context.disableRoutesReq.module)
        
                    exports.disablePluginRoutesService(resTmp, function(err){
                        // V5.0.7 Begin
                        if(err && err instanceof warn) logger.warn('--->afterServerStart: Req.Id=' + resTmp._Context.reqId + ', ' + err.message)
                        else if(err) logger.error('--->afterServerStart: Req.Id=' + resTmp._Context.reqId + ', Unable to disable plugin routes. Reason: ' + err.message)    // plugin name comes in error message
                        else logger.info('--->afterServerStart: Req.Id=' + resTmp._Context.reqId + ', Disabled all the routes of a plugin: '  + resTmp._Context.disableRoutesReq.module)
                        // V5.0.7 End
                        next()            
                    })
                }
            ],
            function(err){
                plugin.initErrors = undefined  // afterServerStart() is the last server statup task, so reset plugin initErrors. 
                nextTask(err)
            })
           // V5.0.6 End                
		},
		function(err){
			callback(err);
		}
	);
}
// V3.0.16 End

// V3.0.13 Begin
/**
 * Deactivate plugins
 * @param  {Response}  	res Response Object
 * @param  {Function} callback callback(err)
 */
exports.deactivatePlugins = function(res, callback){
 	
	logger.info('-->Deactivating Plugins...');

	// Loop through all plugins
	async.each(plugins,
		function(plugin, nextTask){

			if(plugin.enable !== 'on') return nextTask(); // do nothing if plugin is disabled

			// Activate the plugin
			_deactivatePlugin(res, plugin, nextTask);
		},
		function(err){
			if(err) logger.error('--->deactivatePlugins: Deactivating Plugins failed. ' + err.stack);
			callback(err);
		}
	);
}
// V3.0.13 End

/**
 * Disable all plugins
 * @param  {function} 	callback(err)
 */
exports.disablePlugins = function(res, callback){
	
	var module = null;

	logger.info('-->Disabling Plugins...');

	stopInProgress = true; 	// V3.0.13 Begin
	// Loop through all plugins
	async.each(plugins,
		function(plugin, nextTask){

			if(plugin.enable !== 'on'){
				nextTask();
			}
			else{ // Disable the plugin
				_disablePlugin(plugin, nextTask);
			}
		},
		function(err){
			if(err){
				logger.error('--->disablePlugins: Disable Plugins failed. ' + err.stack);
			}
			callback(err);
		}
	);
}

/**
 * isPluginEnabled: 	Checks if plugin is enabled or not.
 * @param  {string}  	pluginName Name of the plugin
 * @return {boolean}    Returns true if plugin is enabled, else false.
 */
exports.isPluginEnabled = function(pluginName){

	if(pluginName === undefined || pluginName === null){
		return false;
	}	

	// Check if the plugin is in the list and enabled
	for (var i = plugins.length - 1; i >= 0; i--) {
		if(plugin[i].module === pluginName && plugin[i].enable === 'on'){
			return true;
		}
	};
	return false;
}

/**
 * Returns the list of plugins
 * @return {json} 	plugins List of plugins
 */
exports.getPluginsList = function(){
	return plugins;
}

/**
 * Returns the handle of the requested plugin if enabled
 * @param  {string} 	pluginName Name of the plugin
 * @param  {function}   callback(err, module)
 */
exports.getModule = function(pluginName, callback){

	var module = null
	  , fqPlugin = ''
	  ;
	
	// Get the plugin info
	_getPlugin(pluginName, function(err, plugin){
		if(err){
			return callback(err);
		}
		// Check if the plugin is enabled
		if(plugin.enable === 'on'){
			fqPlugin = plugin.plugindir + plugin.module + '/' + pluginJSfilename;
			// Get the module handle
			try{
				module = require(fqPlugin);
				if(!module){
					return callback(new Error('Error loading Plugin ' + fqPlugin));
				}
				return callback(null, module);
			}catch(err){
				return callback(new Error('Error loading Plugin ' + fqPlugin + '. Reason: ' + err.stack));
			}
		}
		else{
			return callback(new Error('Plugin ' + pluginName + ' not enabled.'));
		}
	});
}

/**
 * Call the Service of the speficied Plugin
 * @param  {JSON}  	plugin Plugin and Service names
 *					plugin:{
 *						'name': 	<plugin name>,
 *						'service': 	<service name>
 *					}
 * @param  {JSON}  	parms Service parameters
 * @param {Function} callback(err, result)	returns the result of the called service
 * 
 */
exports.callPluginService = function(plugin, parms, callback){
	var reqId = ''; 

	if(parms && parms._Context && parms._Context.reqId) reqId = ' Req.Id=' + parms._Context.reqId + ',';

	if(!plugin || !plugin.name || !plugin.service){
		return callback(new Error('-->callPluginService:' + reqId + ' Missing plugin.name or plugin.service'));
	}

	if(logger.isDebugEnabled()) logger.debug('-->callPluginService:' + reqId + ' Plugin: ' + plugin.name + ', Service: ' + plugin.service);

	// Get the handle of the requested plugin if enabled
	exports.getModule(plugin.name, function(err, module){
		if(err) return callback(err);

		// Try to call the requested Service from the given Plugin
		try{
			module[plugin.service](parms, callback);
		}catch(err){
			return callback(new Error('Calling ' + plugin.name + '.' + plugin.service + '() failed: ' + err.message));
		}
	});
}

/**
 * Get Route State Information
 * @param  {String} routePath Route Path
 * @return {JSON}           Route State Info
 */
exports.getRouteStateInfo_notused = function(route, callback){

    let res	= undefined
      , sfwExists = _pluginExists(sfwPluginName) // V4.1.7

	if(!sfwExists) return callback() // V4.1.7
	
	res	= {
		'_Context': {
			'sfw': {'route': {'path': route.path}, 'plugin': route.plugin}
		}
	}
	
	//if(routeStateInfo && routeStateInfo[routePath]) return callback(null, routeStateInfo[routePath]);
	
	exports.callPluginService({name:'oxsnps-statefilewatcher',service:'getRouteStateInfo'}, res, callback)
	
	//function(err, result){
		//return callback(err, result)
	//})
	
	/*
	var pattern 	= undefined
	  , pathTmp = undefined
	
	if(!process.env.DMS2_STATE_FILES_PATH) return callback();

	if(routeStateInfo && routeStateInfo[routePath]) return callback(null, routeStateInfo[routePath]);

	// replace '/' with '.' in the route path
	pathTmp = routePath.replace(/\//g, '.');
	//  Route state file patten : <hostname>_<plugin name>_<route>.<state>
	//  The / of the http and dir routes will be replace by dots (.)
	pattern	= '^(' + os.hostname()+ ')'+ '_(.*)_('+ pathTmp + ')\\.('+ exports.DEACTIVATED + '|' + exports.DISABLE + '|' + exports.DISABLEBYERROR + '|' + exports.DISABLEBYSTOP  + '|' + exports.ENABLE +')?$' // State file patten

	// Read route state info from file
	_readRouteStateInfo(pattern, function(err, result){
		if(err) return callback(new Error('Unable to load state of routes. ' + err.message));
		if(!result) return callback(new Error('Unable to find state file for routes matching pattern "' + pattern + '"'));
		if(result) return callback(null, result[routePath]);
		return callback();	// state file not available
	});
	*/
}

// V3.0.26 Begin
/**
 * Read application specific plugin conf files
 * @return {function}	callback(err)
 */
exports.readPluginConfFiles = function(callback){

	let pattern 		= /\.js$/i		// match only *.js files
	  , pluginfConfDir 	= npsConf.pluginConfDir || (npsConfDir + FILE_SEP + "plugins")		// directory of application specific plugin conf files
	  , pluginname 		= undefined
	  , serverconf 		= undefined
	  , pluginconf 		= undefined

	// plugins conf dir is now npsConfDir + FILE_SEP + appname + FILE_SEP +  "plugins" if appname is defined // V5
	if(appname) pluginfConfDir = npsConf.pluginConfDir || (npsConfDir + FILE_SEP + appname + FILE_SEP + "plugins")
	pluginfConfDir = path.resolve(pluginfConfDir)
	
	npsConf.plugins = npsConf.plugins || {}	// get plugins section from server-conf.js

	// Check if directory exists
	utils.dirExists(pluginfConfDir, function(err, exists){
		if(err) return callback(err)
		if(!exists){
			if(logger.isDebugEnabled()) logger.debug('-->readPluginConfFiles: ' + pluginfConfDir  + ' directory does not exist')
			return callback()
		} 
		// Get file list from directory of application specific plugin conf files
		utils.getFiles(pluginfConfDir, pattern, false, function(err, files){
			if(err) return callback(err)
			if(!files || files.length <= 0){
				if(logger.isDebugEnabled()) logger.debug('-->readPluginConfFiles: ' + pluginfConfDir  + ' directory does not have any files')
				return callback()
			}
			if(logger.isDebugEnabled()) logger.debug('-->readPluginConfFiles: Dir: ' + pluginfConfDir + ', Files: ' + JSON.stringify(files))

			// Read conf files and add to server-conf plugins section
			files.forEach(function(file){
				// get plugin name from file name
				pluginname = utils.getLastToken(file, FILE_SEP).replace(/\.js/, '')
				serverconf = npsConf.plugins[pluginname]
				try{
					pluginconf = require(file)
					if(!pluginconf) return callback(new Error('Unable to load ' + file))
				}catch(err){
					return callback(new Error('Unable to load ' + file + ', ' + err.message))
				}
				if(serverconf){
					if(logger.isDebugEnabled()) logger.debug('-->readPluginConfFiles: Pluginname: ' + pluginname + ', pluginConf: ' + JSON.stringify(pluginconf)); 					
					if(logger.isDebugEnabled()) logger.debug('-->readPluginConfFiles: Pluginname: ' + pluginname + ', server-conf.config: ' + JSON.stringify(serverconf))
					for(var attrname in serverconf){
						if(!pluginconf[attrname]) pluginconf[attrname] = serverconf[attrname];
					}
					if(logger.isDebugEnabled()) logger.debug('-->readPluginConfFiles: Pluginname: ' + pluginname + ', MergedConf: ' + JSON.stringify(pluginconf))
				}
				npsConf.plugins[pluginname] = pluginconf
			})
			callback()
		})
	})
}

// V4.1.22 Begin
/**
 * Return swagger configuration
 * @return {function}	callback(err, result)
 *		Example: result = {"urls":[{"name":"ci.baw","url":"/ci.baw/api/rest.yaml"},{"name":"hal.finalizer","url":"/hal.finalizer/api/rest.yaml"}]}
 */
exports.getSwaggerconf = function(req, res){
    /* 
        swagger = { "urls": []}
    */
	logger.info('--->getSwaggerconf: Getting the swagger configuration...');

	// Assert plugins array
	res.json(swagger)

	logger.info('--->getSwaggerconf: Response: ' + JSON.stringify(swagger) + ', over.');
}
// V4.1.22 End

// V4.1.27 Begin
/**
 * Disable the plugin routes
 * @param  {req} 		req as GET or POST 
 * 		   GET URL: 	<server>[:port]/appservices/pm/disablepluginroute?module=oxsnps-afp2any[&reason={...}]
 * 		   POST URL: 	<server>[:port]/appservices/pm/disablepluginroute
 * 		   POST parms:	req.body = {
 * 		            		"module":"oxsnps-afp2any", 
 * 		            	 	"reason": { // optional
 * 		            		 	"desc":  res._Context.proJob.error.server + ' assertion failed',
 * 		            		 	"error":{
 * 		            		 	    //"code":   // not used 
 * 		            		 	    "msg': res._Context.proJob.error.msg
 * 		            		 	}
 * 		            	 	}
 * 		              	}
 * @param  {res}		res
 */
exports.disablePluginRoutes = function(req, res){

//utils.log4jsSetLogLevel(logger, "DEBUG")

	// Add Context to the response instance
	res._Context = {}

	// Add the Current Date to the Context
	res._Context.date = new Date()

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date)

	req.body = req.body || {}

	// Add Get Parameters to req.body
	if(req.method === 'GET'){
        req.body.module = req.query.module
        if(req.query.reason) req.body.reason = utils.parseJSON(req.query.reason)
	}

	res._Context.disableRoutesReq = req.body

	exports.disablePluginRoutesService(res, function(err){
		if(err && err instanceof warn) return res.end(err.message);
        else if(err) return res.status(404).end('Unable to disable plugin routes. Req. Id=' + res._Context.reqId + ', ' + err.message)      // plugin name comes in error message
		res.end('Plugin ' + req.body.module + ', routes has been disabled')
	})
}

/**
 * Disable the plugin routes
 * @param  {res} 		Response Object
 * 		   POST parms:	res._Context = {
 * 		            	    "reqId": <reqId>,
 * 		   					"disableRoutesReq" : {
 * 		            			"module": "oxsnps-afp2any", 
 * 		            	 	    "reason": { // optional
 * 		            		 	    "desc":  Optional. <error description>,
 * 		            		 	    "error":{
 * 		            		 	        //"code":   // not used 
 * 		            		 	        "msg': <error message>	// detailed error message
 * 		            		 	    }
 * 		            	 	    }
 * 		              	}
 * @param  {Function} 	callback(err)
 */
exports.disablePluginRoutesService = function(res, callback){

    let plugin      = undefined
      , resTmp      = undefined
      , routeTmp    = undefined
      , errors      = []
      , statefileData   = ''

//utils.log4jsSetLogLevel(logger, "DEBUG")      

    if(!res._Context || !res._Context.disableRoutesReq || !res._Context.disableRoutesReq.module) return callback(new Error('Missing parameter res._Context.disableRoutesReq.module to disable the plugin routes'))

    res._Context.reqId = res._Context.reqId || utils.buildReqId(new Date()),

    //if(logger.isDebugEnabled()) logger.debug('--->disablePluginRoutesService: Req.Id=' + res._Context.reqId + ', Disabling all the routes of a plugin: ' + res._Context.disableRoutesReq.module)
	async.series([
	    // Get Plugin        
		function(nextTask){
            if(logger.isDebugEnabled()) logger.debug('--->disablePluginRoutesService: Req.Id=' + res._Context.reqId + ', Getting information of a plugin: ' + res._Context.disableRoutesReq.module)
	        _getPlugin(res._Context.disableRoutesReq.module, function(err, retPlugin){
                if(err){   // error is thrown even if plugin is not found
                    //logger.error('--->disablePluginRoutesService: Req.Id=' + res._Context.reqId + ', Unable to disable a plugin routes, ' + err.message + ', plugin: ' + res._Context.disableRoutesReq.module)
                    errors.push(err.message)
                    return nextTask(err)    // error is thrown even if plugin is not found
                }
                plugin  =  retPlugin
                if(logger.isDebugEnabled()) logger.debug('--->disablePluginRoutesService: Req.Id=' + res._Context.reqId + ', Plugin: ' + JSON.stringify(plugin))
                return nextTask()    
            })
        },        
		function(nextTask){
            if(!plugin.routes || plugin.routes.length<=0)  return nextTask()
            // V5.0.7 Begin
            if(plugin.initErrors){
                // if plugin alreay has initialization errors, add the new error that is comming as parameter to that array. And create a content to write to state file
                if(res._Context.disableRoutesReq.reason && res._Context.disableRoutesReq.reason.error && res._Context.disableRoutesReq.reason.error.msg) plugin.initErrors.push(res._Context.disableRoutesReq.reason.error.msg)
                statefileData = plugin.initErrors.join('\n') || ''
            }
            else statefileData = res._Context.disableRoutesReq.reason && res._Context.disableRoutesReq.reason.error && res._Context.disableRoutesReq.reason.error.msg || ''
            // V5.0.7 End

            async.eachSeries(plugin.routes,
                function(route, next){
                    if(!route) return next()
                    if(route.maxJobs === undefined) return next()  // do not disable routes that are not involved in job creation     
                    
                    routeTmp = utils.clone(route)
                    routeTmp.stateInfo = {
                        'state':    exports.DISABLEBYERROR,
                        'plugin': 	res._Context.disableRoutesReq.module,
                        'route':    route.path || route.queue || route.subscription
                    }

                    resTmp  = {
                        '_Context': {
                            'reqId': res._Context.reqId,
                            'disableRouteReq': {
                                'body': {
                                    'module': res._Context.disableRoutesReq.module,
                                    'route':  routeTmp,  
                                    'reason':{
                                        //'reqId': res._Context.reqId
                                        'error':{
                                            'msg': statefileData
                                        }                                        
                                    }
                                }
                            }
                        }
                    }
                

                    if(logger.isDebugEnabled()) logger.debug('--->disablePluginRoutesService: Req.Id=' + res._Context.reqId + ', Disabling a plugin route: ' + JSON.stringify(routeTmp))
                    exports.disableRouteService(resTmp, function(err){
                        //if(err && err instanceof warn) logger.warn('--->disablePluginRoutesService: Req.Id=' + resTmp._Context.reqId + ', route: ' + JSON.stringify(routeTmp) + ', ' + err.message)
                        //else if(err) errors.push('route: ' + JSON.stringify(routeTmp) + ', ' + err.message)
                        if(err) errors.push('route: ' + JSON.stringify(routeTmp) + ', ' + err.message)  // V5.0.7 
                        next()
                    })
                },
                function(err){
                    nextTask(err)
                }
            )
        }
	],
	function(err){
        if(errors.length >0) return callback(new Error(errors.join('\n')))
        return callback()
	})
}
// V4.1.27 End

/******************  Private Functions ********************/
/**
 * _loadPlugin: 		Load and initialize a plugin
 * @param  {string}   	plugin 		the FQ plugin directory
 * @param  {function}   callback(err)
 */
function _loadPlugin(plugin, callback){
	
	var module = null
	  , fqPlugin    = plugin + '/' + pluginJSfilename
      , aPlugin     = {}	// the new plugin to be added to the list of plugins
      , sfwExists   = _pluginExists(sfwPluginName) // V4.1.7
      , resTmp      = undefined
      , initErr     = undefined
	  ;

  	// Assert fqPlugin
	if(!fqPlugin) return callback(new Error('Plugin name is null'))

	if(logger.isDebugEnabled()) logger.debug('_loadPlugin: Enabling ' + fqPlugin + ' ...')

	// Load the plugin
	try{
		// Get the module handle
		module = require(fqPlugin)
		if(!module) initErr = new Error('Error loading Plugin ' + fqPlugin)
	}catch(err){
        if(err) initErr = err
		//return callback(new Error('Error loading Plugin ' + fqPlugin + '. Reason: ' + err.stack));
    }
    if(initErr){
        if(npsConf.stopOnInitErrors) return callback(initErr)
        logger.error('--->_loadPlugin: Unable to load the plugin: ' + fqPlugin + '. Reason: ' + initErr.message)
        return callback()
    } 

	// Set the plugin props
	/*
		Structure of the plugins array
	    "plugins":[
			{
				"module": 		"Plugin Name (ex.: afp2any)",
				"plugindir": 	"FQ name of the the plugin root directory (ex.: /path/to/pluginDir/)",
				"enable": 		"on|off",
				"desc": 		"Short description of the plugin",
				"longDesc": 	"Long description of the plugin",
				"log.level":		"DEBUG|INFO|ERROR",
				"routes": []
			},
			{...}
		]
	*/
    var groups = plugin.match(/(.*\/)(.*)/);
	aPlugin.module = groups[2]; 	// Get the last dir as default plugin name
	aPlugin.plugindir = groups[1]; 	// Get the path up to last dir (i.e the plugin root dir)
    
    aPlugin.initErrors = [] // V5.0.6

	if(module.pluginConf === undefined){
		aPlugin.enable 		= 'off';
		aPlugin.desc 		= fqPlugin;
		aPlugin.longDesc	= fqPlugin;
		aPlugin.logLevel	= 'INFO';
		aPlugin.version		= 'Unknown';
	}
	else{
		//!!!aPlugin.module 	 = 	module.pluginConf.module 	|| groups[2]; //!!! Get the last dir (i.e plugin name)
		aPlugin.enable 	 = 	module.pluginConf.enable 	|| 'off';
		aPlugin.desc 	 =	module.pluginConf.desc 		|| fqPlugin;
		aPlugin.longDesc = 	module.pluginConf.longDesc 	|| fqPlugin;
		if (typeof(module.getLogLevel) === 'function') aPlugin.logLevel = module.getLogLevel();
		aPlugin.logLevel = aPlugin.logLevel || 'INFO';
		aPlugin.version  = 	module.pluginConf.version 	|| 'Unknown';
	}

    // V5.0.6 Begin
    /*
	// Initialize the plugin if the plugin is enabled
	if(aPlugin.enable === 'on'){
		_initializePlugin(module, fqPlugin, function(err, routes){
			if (err){
				aPlugin.enable = 'off'; // reset enable prop
				return callback(new Error('Could not intialize Plugin ' + fqPlugin + '. ' + err.stack));
            }
			// Set enable flag on by default for all routes
            routes.forEach(function(route){
                if(!route) return
                route.enable = route.enable || 'on' 
            })
			aPlugin.routes = routes;	// add the routes to the plugin
			plugins.push(aPlugin);		// add the plugin to the list of plugins
			return callback();
		});
	}
	else{
		aPlugin.routes = [];		// add the routes to the plugin
		plugins.push(aPlugin);		// add the plugin to the list of plugins
		return callback();
    }
    */

    if(aPlugin.enable !== 'on'){
        aPlugin.routes = []		// add the routes to the plugin
        plugins.push(aPlugin)		// add the plugin to the list of plugins
        return callback()
    }

    async.series([
        // Initialize the plugin
        function(nextTask){
            _initializePlugin(module, fqPlugin, function(err, routes){
                if(err && npsConf.stopOnInitErrors){
                    aPlugin.enable = 'off'; // reset enable prop
                    return nextTask(new Error('Could not intialize Plugin ' + fqPlugin + '. ' + err.message));
                }
                if(err){
                    logger.error('--->_loadPlugin: Unable to initialize the plugin: ' + aPlugin.module + 'v' + aPlugin.version + '. Reason: ' + err.message)
                    aPlugin.initErrors.push(err.message)
                }
                routes = routes || []
                // Set enable flag on by default for all routes
                routes.forEach(function(route){
                    if(!route) return
                    route.enable = route.enable || 'on' 
                })
                aPlugin.routes = routes	// add the routes to the plugin
                plugins.push(aPlugin)		// add the plugin to the list of plugins
                return nextTask()
            });

        },
        // Create state files if it does not exist (even in error cases since it is first loading of plugin). If it exists, update plugin routes with state flag extracted from the state file
        function(nextTask){
            if(!sfwExists) return nextTask()		  // V4.1.7
            if( sfwPluginName === module) return nextTask() // V5.0.11
            resTmp = {
                '_Context': {
                    'sfw': {
                        'plugin':	        aPlugin.module, 
                        'routes': 	        aPlugin.routes
                    }
                }
            }
            exports.callPluginService({name:'oxsnps-statefilewatcher', service:'updateRoutesWithStateInfo'}, resTmp, function(err){
                if(err && npsConf.stopOnInitErrors) return nextTask(new Error('Error in intializing plugin routes. Plugin: ' + fqPlugin + '. ' + err.message))
                if(err){
                    logger.error('--->_loadPlugin: Unable to initialize the plugin: ' + aPlugin.module + 'v' + aPlugin.version + '. Reason: ' + err.message)
                    aPlugin.initErrors.push(err.message)
                }
                nextTask()
            })
        },
        // in case of plugin initialization errors, disable plugin routes 
        function(nextTask){
            if(npsConf.stopOnInitErrors || aPlugin.initErrors.length <=0 ) return nextTask()		  
            // log error
            //logger.error('--->_loadPlugin: Unable to initialize the plugin: ' + aPlugin.module + 'v' + aPlugin.version + '. Reason: ' + initError.message)
            
            // disable plugin routes
            resTmp = {
                '_Context': {
                    'reqId': utils.buildReqId(new Date()),
                    'disableRoutesReq': {
                        'module': aPlugin.module
                    }
                }
            }
        
            if(logger.isDebugEnabled()) logger.debug('--->_loadPlugin: Req. id=' + resTmp._Context.reqId + ', Disabling all the routes of a plugin: ' + resTmp._Context.disableRoutesReq.module)

            exports.disablePluginRoutesService(resTmp, function(err){
                // V5.0.7 Begin
                if(err && err instanceof warn) logger.warn('--->_loadPlugin: Req.Id=' + resTmp._Context.reqId + ', ' + err.message)
                else if(err) logger.error('--->_loadPlugin: Req.Id=' + resTmp._Context.reqId + ', Unable to disable plugin routes. Reason: ' + err.message)     // plugin name comes in error message
                else logger.info('--->_loadPlugin: Req.Id=' + resTmp._Context.reqId + ', Disabled all the routes of a plugin: '  + resTmp._Context.disableRoutesReq.module)
                // V5.0.7 End
                nextTask()
            })
        }
    ],
    function(err){
        callback(err)
    })
   // V5.0.6 End
}

/**
 * _enablePlugin: 		Enable a plugin which is already in the list of plugins (has been once before enabled)
 * @param  {string}     plugindir The plugin directory
 * @param  {function}   callback(err)
 */
function _enablePlugin(plugin, callback){
	
	var module 		= null
	  , fqPlugin 	= null
	  , routes 		= undefined
      , resTmp		= undefined
      , sfwExists   = _pluginExists(sfwPluginName) // V4.1.7
      , initError   = undefined

	// Assert plugin parameter
	if(!plugin && !plugin.plugindir && !plugin.module){
		return callback(new Error('Plugin configuration parameter is null'));
	}
	fqPlugin = plugin.plugindir + plugin.module + '/' + pluginJSfilename;

    if(logger.isDebugEnabled()) logger.debug('_enablePlugin: Enabling ' + fqPlugin + ' ...');
    //V5.0.6 Begin
    /*
	// Get the module handle
	try{
		module = require(fqPlugin);
		if(!module){
			return callback(new Error('Error loading Plugin ' + fqPlugin));
		}
	}catch(err){
		return callback(new Error('Error loading Plugin ' + fqPlugin + '. Reason: ' + err.stack));
    }
    */    
    //V5.0.6 End
	//routeStateInfo = undefined;

	//  Route state file patten : <hostname>_<plugin name>_<route>.<state>
	//  The / of the http and dir routes will be replace by dots (.)
	//pattern	= '^(' + os.hostname() + ')'+ '_('+ plugin.module + ')_(.*)\\.('+ exports.DEACTIVATED + '|' + exports.DISABLE + '|' + exports.DISABLEBYERROR + '|' + exports.DISABLEBYSTOP + '|' + exports.ENABLE +')?$'  // State file patten does not have tenant DMS2 scenario
    
    plugin.initErrors =  []     // reset old errors

	async.series([
        function(nextTask){
            // Get the module handle
            try{
                module = require(fqPlugin)
                if(!module) initError = new Error('Error loading Plugin ' + fqPlugin)
                if(!module && npsConf.stopOnInitErrors) return nextTask(initError)
            }catch(err){
                if(err && npsConf.stopOnInitErrors) return nextTask(new Error('Error loading Plugin ' + fqPlugin + '. ' + err.message))
                initError = err
            }
            if(initError){
                logger.error('--->_enablePlugin: Unable to initialize the plugin: ' + plugin.module +  ((module.pluginConf && module.pluginConf.version) ? ('v' + module.pluginConf.version) : '')  + '. Reason: ' + initError.message)
                plugin.initErrors.push(initError.message)
            }
            return nextTask()
        },                
		// Initialize the plugin
		function(nextTask){
            if(initError) return nextTask()

			_initializePlugin(module, fqPlugin, function(err, result){
                if(err && npsConf.stopOnInitErrors) return nextTask(new Error('Could not intialize Plugin ' + fqPlugin + '. ' + err.message));
                if(err){
                    logger.error('--->_enablePlugin: Unable to initialize the plugin: ' + plugin.module +  ((module.pluginConf && module.pluginConf.version) ? ('v' + module.pluginConf.version) : '')  + '. Reason: ' + err.message)
                    plugin.initErrors.push(err.message)
                    initError = err
                }
				routes = result || []
				nextTask()
			});
		},
		// update plugin routes with state flag extracted from state file
		function(nextTask){
            if(!sfwExists || initError) return nextTask()		  // V4.1.7
            if( sfwPluginName === module) return nextTask() // V5.0.11
			resTmp = {
				'_Context': {
					'sfw': {
						'plugin':	plugin.module, 
						'routes': 	routes
					}
				}
            }
			exports.callPluginService({name:'oxsnps-statefilewatcher', service:'updateRoutesWithStateInfo'}, resTmp, function(err){
                if(err && npsConf.stopOnInitErrors) return nextTask(new Error('Error in intializing plugin routes. Plugin: ' + fqPlugin + '. ' + err.message))
                if(err){
                    logger.error('--->_enablePlugin: Unable to initialize the plugin: ' + plugin.module +  ((module.pluginConf && module.pluginConf.version) ? ('v' + module.pluginConf.version) : '')  + '. Reason: ' + err.message)
                    plugin.initErrors.push(err.message)
                    initError = err
                }
                nextTask()
            })
        },
        // in case of plugin initialization errors, disable plugin routes 
        function(nextTask){
            if(npsConf.stopOnInitErrors || !initError ) return nextTask()		  
            // log error
            //logger.error('--->_enablePlugin: Unable to initialize the plugin: ' + plugin.module +  ((module.pluginConf && module.pluginConf.version) ? ('v' + module.pluginConf.version) : '')  + '. Reason: ' + initError.message)
            // disable plugin routes
            resTmp = {
                '_Context': {
                    'reqId': utils.buildReqId(new Date()),
                    'disableRoutesReq': {
                        'module': plugin.module
                    }
                }
            }
        
            if(initError.desc) resTmp._Context.disableRoutesReq.reason.desc = plugin.module +  ((module.pluginConf && module.pluginConf.version) ? ('v' + module.pluginConf.version) : '') + ', ' +  initError.desc   // for backward compatibility
        
            if(logger.isDebugEnabled()) logger.debug('--->_enablePlugin: Req. id=' + resTmp._Context.reqId + ', Disabling all the routes of a plugin: ' + resTmp._Context.disableRoutesReq.module)

            exports.disablePluginRoutesService(resTmp, function(err){
                // V5.0.7 Begin
                if(err && err instanceof warn) logger.warn('--->_enablePlugin: Req.Id=' + resTmp._Context.reqId + ', ' + err.message)
                else if(err) logger.error('--->_enablePlugin: Req.Id=' + resTmp._Context.reqId + ', Unable to disable plugin routes. Reason: ' + err.message)      // plugin name comes in error message
                else logger.info('--->_enablePlugin: Req.Id=' + resTmp._Context.reqId + ', Disabled all the routes of a plugin: '  + resTmp._Context.disableRoutesReq.module)
                // V5.0.7 End                
                nextTask()
            })
        },
		// Activate the plugin
		function(nextTask){
			_activatePlugin(plugin, function(err){
                if (err) return nextTask(new Error('Could not activate Plugin ' + fqPlugin + '. ' + err.stack));
				// Add the list of the plugin routes to the list of the plugins array
				plugin.enable = 'on';
				module.pluginConf.enable = 'on';
				plugin.routes = routes;
				nextTask();
			});
		},
		// Save the plugin configuration file
	    function(nextTask){
			if (savePluginConf && // added "savePluginConf" flag for OTS-2326
				module.saveConfService && 
				typeof(module.saveConfService) === 'function') module.saveConfService(nextTask);
			else nextTask();
	    }
	],
	function(err){
		// While initializing plugin, we read all route state files of this plugin and keep in'routeStateInfo' to decide whether to add route or not. 
		// After initializing, reset routeStateInfo in memory. 
		// When route state info is needed later, it should be read from state file
		//routeStateInfo = undefined;
		callback(err);
	});
}

/**
 * _disablePlugin: 		Disable a plugin
 * @param  {JSON}       plugin    A JSON object that contains plugin information.
 * @param  {function}   callback(err)
 */
function _disablePlugin(plugin, callback){

	var name = null
	  , fqPlugin = null
	  , module	= undefined
	  ;

	// Assert plugin parameter
	if(!plugin || !plugin.plugindir || !plugin.module){
		return callback(new Error('Plugin configuration parameter is null'));
	}
	fqPlugin = plugin.plugindir + plugin.module + '/' + pluginJSfilename;

	if(logger.isDebugEnabled()) logger.debug('_disablePlugin: Disabling ' + fqPlugin + '  ...');

	// Get the module handle
	try{
		module = require(fqPlugin);
		if(!module){
			return callback(new Error('Error loading Plugin ' + fqPlugin));
		}
	}catch(err){
		return callback(new Error('Error loading Plugin ' + fqPlugin + '. Reason: ' + err.stack));
	}

	// Finalize the plugin
	_finalizePlugin(module, fqPlugin, function(err){
		if (err){
			return callback(new Error('Could not finalize Plugin ' + fqPlugin + '. ' + err.stack));
		}
		// do not save config file if stopInProgress is true
		if(stopInProgress) return callback(); 	// // V3.0.13 Begin
		// Add the list of the plugin routes to the list of the plugins array
		plugin.enable = 'off';
		module.pluginConf.enable = 'off';
		plugin.routes = []; // no more route available
		// Save the plugin configuration file		
		if (savePluginConf && module.saveConfService && typeof(module.saveConfService) === 'function')		 // added "savePluginConf" flag for OTS-2326
			module.saveConfService(callback);
		else callback();
	});
}

/**
 * _activatePlugin: 	Activate a plugin
 * @param  {JSON}       plugin    A JSON object that contains plugin information.
 * @param  {function}   callback(err)
 */
function _activatePlugin(plugin, callback){

	var fqPlugin    = null
      , initError   = undefined
      , resTmp      = undefined
      , module      = undefined
	  ;

	// Assert plugin parameter
	if(!plugin || !plugin.plugindir || !plugin.module){
		return callback(new Error('Plugin configuration parameter is null'));
	}
	fqPlugin = plugin.plugindir + plugin.module + '/' + pluginJSfilename;

    if(logger.isDebugEnabled()) logger.debug('_activatePlugin: Activating ' + fqPlugin + '  ...');
    
    plugin.initErrors = plugin.initErrors || []

    // V5.0.6 Begin
    /*
	// Get the module handle
	try{
		module = require(fqPlugin);
		if(!module){
			return callback(new Error('Error loading Plugin ' + fqPlugin));
		}
	}catch(err){
		return callback(new Error('Error loading Plugin ' + fqPlugin + '. Reason: ' + err.stack));
	}

	// Call activate service of the plugin if it exists
	if (typeof(module.postInitialize) === 'function'){

		// Try to call activate
		try{
			module.postInitialize(function(err){
				if(err) return callback(new Error('Could not activate Plugin ' + fqPlugin + '. ' + err.stack));		
				callback();
			});
		}catch(err){
			callback(new Error('Calling ' + plugin.module + ' .postInitialize() failed.' + err.stack));
		}
	}
	else{
		logger.warn('--->_activatePlugin: ' + plugin.module + '.postInitialize() not available.');
		callback();
    }
    */

    async.series([
        function(nextTask){
            // Get the module handle
            try{
                module = require(fqPlugin)
                //if(!module) return next(new Error('Error loading Plugin ' + fqPlugin));
                if(!module) initError = new Error('Error loading Plugin ' + fqPlugin)
                if(!module && npsConf.stopOnInitErrors) return nextTask(initError)
            }catch(err){
                if(err && npsConf.stopOnInitErrors) return nextTask(new Error('Error loading Plugin ' + fqPlugin + '. ' + err.message))
                initError = err
            }
            if(initError){
                logger.error('--->_activatePlugin: Unable to activate the plugin: ' + plugin.module +  ((module.pluginConf && module.pluginConf.version) ? ('v' + module.pluginConf.version) : '')  + '. Reason: ' + initError.message)
                plugin.initErrors.push(initError.message)
            }
            return nextTask()
        },                

        function(nextTask){
            if(initError || typeof(module.postInitialize) !== 'function') return nextTask()
            // Activate the plugin
            try{
                module.postInitialize(function(err){
                    if(err && npsConf.stopOnInitErrors) return nextTask(new Error('Could not activate Plugin ' + fqPlugin + '. ' + err.stack))
                    if(err){
                        logger.error('--->_activatePlugin: Unable to activate the plugin: ' + plugin.module +  ((module.pluginConf && module.pluginConf.version) ? ('v' + module.pluginConf.version) : '')  + '. Reason: ' + err.message)
                        plugin.initErrors.push(err.message)                        
                        initError = err
                    }
                    nextTask()
                })
            }catch(err){
                if(err && npsConf.stopOnInitErrors) return nextTask(new Error('Calling ' + plugin.module + ' .postInitialize() failed.' + err.stack))
                if(err){
                    logger.error('--->_activatePlugin: Unable to activate the plugin: ' + plugin.module +  ((module.pluginConf && module.pluginConf.version) ? ('v' + module.pluginConf.version) : '')  + '. Reason: ' + err.message)
                    plugin.initErrors.push(err.message)                        
                    initError = err
                }
                nextTask()
            }
        },
        // in case of plugin initialization errors, disable plugin routes 
        function(nextTask){
            if(npsConf.stopOnInitErrors || !initError ) return nextTask()		  
            // log error
            //logger.error('--->_activatePlugin: Unable to activate the plugin: ' + plugin.module +  ((module.pluginConf && module.pluginConf.version) ? ('v' + module.pluginConf.version) : '')  + '. Reason: ' + initError.message)
            // disable plugin routes
            resTmp = {
                '_Context': {
                    'reqId': utils.buildReqId(new Date()),
                    'disableRoutesReq': {
                        'module': plugin.module
                    }
                }
            }
        
            if(initError.desc) resTmp._Context.disableRoutesReq.reason.desc = plugin.module +  ((module.pluginConf && module.pluginConf.version) ? ('v' + module.pluginConf.version) : '') + ', ' +  initError.desc   // for backward compatibility
        
            if(logger.isDebugEnabled()) logger.debug('--->_activatePlugin: Req. id=' + resTmp._Context.reqId + ', Disabling all the routes of a plugin: ' + resTmp._Context.disableRoutesReq.module)

            exports.disablePluginRoutesService(resTmp, function(err){
                // V5.0.7 Begin
                if(err && err instanceof warn) logger.warn('--->_activatePlugin: Req.Id=' + resTmp._Context.reqId + ', ' + err.message)
                else if(err) logger.error('--->_activatePlugin: Req.Id=' + resTmp._Context.reqId + ', Unable to disable plugin routes. Reason: ' + err.message)     // plugin name comes in error message
                else logger.info('--->_activatePlugin: Req.Id=' + resTmp._Context.reqId + ', Disabled all the routes of a plugin: '  + resTmp._Context.disableRoutesReq.module)
                // V5.0.7 End
                nextTask()
            })            
        }
    ],
    function(err){
        callback(err)
    })
   // V5.0.6 End    
}

// V3.0.13 Begin
/**
 * _deactivatePlugin: 	Deactivate a plugin
 * @param  {Response}  	res Response Object
 * @param  {JSON}       plugin    A JSON object that contains plugin information.
 * @param  {function}   callback(err)
 */
function _deactivatePlugin(res, plugin, callback){

	var name 		= null
	  , fqPlugin 	= null
	  ;

	// Assert plugin parameter
	if(!plugin || !plugin.plugindir || !plugin.module){
		return callback(new Error('Plugin configuration parameter is null'));
	}
	fqPlugin = plugin.plugindir + plugin.module + '/' + pluginJSfilename;

	if(logger.isDebugEnabled()) logger.debug('_deactivatePlugin: Deactivating ' + fqPlugin + '  ...');

	// Get the module handle
	try{
		module = require(fqPlugin);
		if(!module) return callback(new Error('Error loading Plugin ' + fqPlugin));
	}catch(err){
		return callback(new Error('Error loading Plugin ' + fqPlugin + '. Reason: ' + err.stack));
	}

	// Call prefinalize service of the plugin if it exists
	if (typeof(module.preFinalize) === 'function'){

		// Try to call prefinalize 
		try{
			module.preFinalize(res, function(err){
				if(err) return callback(new Error('Could not deactivate Plugin ' + fqPlugin + '. ' + err.stack));		
				callback();
			});
		}catch(err){
			callback(new Error('Calling ' + plugin.module + ' .preFinalize() failed.' + err.stack));
		}
	}
	else{
		if(logger.isDebugEnabled()) logger.debug('--->_deactivatePlugin: ' + plugin.module + '.preFinalize() not available.');
		callback();
	}
}
// V3.0.13 End

/**
 * _intializePlugin: 	Initialize a plugin
 * @param  {module}     module    	Plugin Module
 * @param  {string} 	plugindir 	Plugin Directory
 * @param  {function}   callback(err, routes)
 */
/*
function _intializePlugin(module, plugindir, callback){

	if(logger.isDebugEnabled()) logger.debug('_intializePlugin: Intializing Plugin ' + plugindir + ' ...');

	// Initialize the plugin
	try{
		module.initialize(function(err, routes){
			if(err){
				return callback(new Error('Initializing Plugin ' + plugindir + ' failed. Reason: ' + err.stack));
			}
			routes.forEach(function(route){
				// For Queue routes add 'path' property and set queue value for 'path'
				if(route.queue)	route.path = route.queue;
			});
			return callback(null, routes);
		});
	}catch(err){
		return callback(new Error('-->Initializing Plugin ' + plugindir + ', failed. Reason: ' + err.stack));
	}
}
*/

/**
 * _initializePlugin: 	Initialize a plugin
 * @param  {module}     module    	Plugin Module
 * @param  {string} 	plugindir 	Plugin Directory
 * @param  {function}   callback(err, routes)
 */
 function _initializePlugin(module, plugindir, callback){
	
	let pluginName		= undefined
	  , pluginVersion	= undefined
      , routes      	= undefined

	if(module.pluginConf && module.pluginConf.module){
		pluginName = module.pluginConf.module
		pluginVersion = module.pluginConf.version
	} 

	//!!!	logger.info('--->_initializePlugin: Intializing Plugin: ' + plugindir + ' ...')
    if(logger.isDebugEnabled()) logger.debug('--->_initializePlugin: Intializing Plugin: ' + pluginName + '@' + pluginVersion +  ', file: ' + plugindir + ' ...')
    else logger.info('--->_initializePlugin: Intializing Plugin: ' + pluginName + '@' + pluginVersion +  ' ...')    

	// Initialize the plugin
	try{
		module.initialize(function(err, result){
            // V4.1.22 Begin
            // From oxsnpscoreV4.1.22, plugin.initialilze() can return result to pluginManager in two ways as given below
            // 1. old style: callback(err, [routes]) as well as 
            // 2. callback(err,{'routes':[routes], 'swagger':[{'name':<name>, 'url': 'url'}]})
            if(result && !Array.isArray(result)){
                routes = result.routes
                _addToSwaggerConf(result.swagger)
            }
            else routes = result    // for old compatibility(i.e for plugins returning callback(err, [routes])
            if(err) return callback(new Error('Initializing Plugin: ' + plugindir + ' failed. Reason: ' + err.message), routes)
            callback(null, routes)
			// V4.1.22 End			
		})
	}catch(err){
		return callback(new Error('-->Initializing Plugin: ' + plugindir + ', failed. Reason: ' + err.stack))
	}
}

/**
 * _initializePlugin: 	Initialize a plugin
 * @param  {module}     module    	Plugin Module
 * @param  {string} 	plugindir 	Plugin Directory
 * @param  {function}   callback(err, routes)
 */
function _initializePlugin_del(module, plugindir, callback){
	
	var routes 			= undefined
	  , confModified	= false
	  //, routeState 		= undefined
	  //, routePath 		= undefined
	  //, routeStateObj 	= undefined
	  , pluginName 		= ''
	  , res 			= undefined
	  ;

	if(module.pluginConf && module.pluginConf.module) pluginName = module.pluginConf.module;

	logger.info('--->_initializePlugin: Intializing Plugin ' + plugindir + ' ...');

	async.series([
		function(nextTask){
			// Initialize the plugin
			try{
				module.initialize(function(err, result){
					if(err) return nextTask(new Error('Initializing Plugin ' + plugindir + ' failed. Reason: ' + err.stack));
					routes = result;
					nextTask();
				});
			}catch(err){
				return nextTask(new Error('-->Initializing Plugin ' + plugindir + ', failed. Reason: ' + err.stack));
			}
		},
		// Task 2: Set state for each route
		//function(nextTask){
			/*
			if(!sfwExists) return nextTask()		  
			res = {
				'_Context': {
					'plugin':			pluginName, 
					'routeStateInfo': 	routeStateInfo,
					'routes': 			routes
				}
			}
			exports.callPluginService({name:'oxsnps-statefilewatcher',service:'updateRoutesWithStateInfo'}, res, function(err, result){
				confModified = result
				nextTask()
			})
			/*
			/*
			if(!routes || routes.length <=0) return nextTask();
			async.forEachOfLimit(routes, 1, 
				function(route, index, next){
					if(route.queue)	route.path = route.queue;	// fill route.path for MQ routes
					
					if(!process.env.DMS2_STATE_FILES_PATH) return next();

					// HTTP routes do not have states, so just return
					if(!route.method || (route.method.toLowerCase() !== 'dir' && route.method.toLowerCase() !== 'queue')
						|| pluginName.toLowerCase() === 'oxsnps-dirwatcher') return next()

					routePath 		= route.path;
					routeStateObj 	= undefined
					if(routeStateInfo && routeStateInfo[routePath])	routeStateObj = routeStateInfo[routePath];
					routeState 		= routeStateObj ? routeStateObj.state.toLowerCase(): undefined;

					if(logger.isDebugEnabled()) logger.debug('_initializePlugin: ' + pluginName + ', Route=' + routePath + 
								', StateFileInfo=' + JSON.stringify(routeStateObj) +
								', Conf.Enable=' + JSON.stringify(route.enable));

					switch(routeState){
						case undefined:
							routeStateInfo = routeStateInfo || {};
							routeStateInfo[routePath] = {
								'plugin': pluginName,
								'state':  exports.DISABLE,
								//'filename': '',
								'route': 	routePath
							};		
							// Set state from plugin conf
							if(route.enable && route.enable.toLowerCase() === 'on')
								routeStateInfo[routePath].state = exports.ENABLE;
							
							_createRouteStatefile(routeStateInfo[routePath], function(err){
								if(err)	logger.error('--->_initializePlugin: Unable to save plugin('+ pluginName + ') configuration. ' + err.message);
								next();
							});
							break;
						case exports.ENABLE:
							if(route.enable && route.enable.toLowerCase() === 'off'){
								route.enable = 'on';
								confModified = true;
							}
							else{
								if(logger.isDebugEnabled()) logger.debug('_initializePlugin: ' + pluginName + ', Route=' + routePath + ', State=' + routeState +
											', Plugin conf and routestate have enable state');
							}
							next();
							break;
						case exports.DISABLE:
						case exports.DEACTIVATED:
						case exports.DISABLEBYERROR:
						case exports.DISABLEBYSTOP:
							if(!route.enable ||	(route.enable && route.enable.toLowerCase() === 'on')){
								route.enable = 'off';
								confModified = true;
							}
							else{
								if(logger.isDebugEnabled()) logger.debug('_initializePlugin: ' + pluginName + ', Route=' + routePath + ', State=' + routeState +
											', Plugin conf and routestate file have disable/deactivate state');
							}		
							next();
							break;
						default: 
							logger.warn('_initializePlugin: ' + pluginName + ', Route=' + routePath + 
										', StateFileInfo=' + JSON.stringify(routeStateObj) +
									', Conf.Enable=' + JSON.stringify(route.enable));
							next(); break;							
					}
				},
				function(err){
					nextTask(err);
				}          
			);
			*/		
		//},
		/*
		// Task 3: If there are changes in plugin conf, save plugin configuration file
	    function(nextTask){
			// save plugin configuration file
			if(savePluginConf && confModified && module.saveConfService && typeof(module.saveConfService) === 'function'){		// added "savePluginConf" flag for OTS-2326
				if(logger.isDebugEnabled()) logger.debug('_initializePlugin: ' + pluginName + ', Saving plugin configuration file...');
				module.saveConfService(function(err){
					if(err)	logger.error('--->_initializePlugin: Unable to save plugin('+ pluginName + ') configuration. ' + err.message);
					nextTask();
				});
			}
			else nextTask();
	    }
		*/
	],
	function(err){
		callback(err, routes);
	});
}

/**
 * _finalizePlugin: 	Initialize a plugin
 * @param  {module}     module    	Plugin Module
 * @param  {string} 	plugindir 	Plugin Directory
 * @param  {function}   callback(err, routes)
 */
function _finalizePlugin(module, plugindir, callback){

	if(logger.isDebugEnabled()) logger.debug('_finalizePlugin: Finalizing Plugin ' + plugindir + ' ...');

	// Finalize the plugin
	try{
		module.finalize(function(err){
			if(err){
				return callback(new Error('Finalizing Plugin ' + plugindir + ' failed. Reason: ' + err.stack));
			}
			return callback();
		});
	}catch(err){
		return callback(new Error('Finalizing Plugin ' + plugindir + ', failed. Reason: ' + err.stack));
	}
}

/**
 * _getPlugin: 			Looks in the plugins array for the passed plugin and returns its json desc. if found, return it
 * 						If not found returns error
 * @param  {string}   	pluginName 	FQ Plugin Name
 * @param  {function}   callback(err, plugin)
 */
function _getPlugin(pluginName, callback){

	var retPlugin = null;
	
	if(pluginName === undefined || pluginName === null){
		return callback(new Error('plugin name undefined or null!'));
	}	

	// Check if the plugin is in the plugins array
	async.each(plugins,
		function(plugin, nextTask){
			if(plugin.module === pluginName){
				retPlugin = plugin;	// return the plugin
			}
			nextTask();
		},
		function(err){
			if(err){
				return callback(new Error('Error when getting Plugin ' + pluginName + ', Reason: ' + err.stack));
			}
			if(retPlugin === null || retPlugin === undefined){
				return callback(new Error('Plugin ' + pluginName + ' not found.'));
			}
			return callback(null, retPlugin);
		}
	);
}

/**
 * _logPlugins: 		Logs all loaded plugins
 */
function _logPlugins(){

 	var pluginInfo = [];

	if(logger.isDebugEnabled()){	
	
		logger.debug('_logPlugins: Listing the Plugins...');

		// Assert plugins parameter
		if(plugins === undefined || plugins === null){
			logger.debug('_logPlugins: No Plugins found in ' + npsDir + '/plugins/');
			return
		}

		// Check if the plugin is in the list and enabled
		async.each(plugins,
			function(plugin, nextTask){
				pluginInfo.push('Plugin: ' + plugin.module + (plugin.enable === 'on'? ', enabled' : ', disabled'));
				nextTask();
			},
			function(err){
				if(err){
					logger.debug(err.stack);
				}
				else{
					for (var i = pluginInfo.length - 1; i >= 0; i--) {
						logger.debug(pluginInfo[i]);
					}			
				}
			}
		)
	}
}

/**
 * _logAppRoutes: 		Display the routes of the Express App
 */
function _logAppRoutes(){

	let stack = expressApp._router.stack	// get the Express App router stack
	  , routes = []
	  , method = undefined
	  , msg = ''
	  , bDebug = logger.isDebugEnabled()

	if(bDebug) logger.debug('_logAppRoutes: Listing the Routes...');

    if(stack instanceof Array){

		stack.forEach(function(a){
			msg = ''

			/**
				a:{ 
					"name": "bound dispatch",
					"keys": [],
					"regexp": {
						"fast_star": false,
						"fast_slash": false
					},
					"route": {
						"path": "/services/tt.proxy/deleteproxyinstance",
						"stack": [
							{
								"name": "<anonymous>",
								"keys": [],
								"regexp": {
									"fast_star": false,
									"fast_slash": false
									},
								"method": "get"
							}
						],
						"methods": {
							"get": true
						}
					}
				}
				or
				a:{
					"name": "$cocheck$stag",
					"keys": [],
					"regexp": {
						"fast_star": false,
						"fast_slash": false
					}
				}
				
			 */
			var route = a.route;
            if(route){
                route.stack.forEach(function(r){
					method = r.method.toUpperCase();
					msg = method + '\t' + route.path
					routes.push({'method':method,'route':route.path})
					if(bDebug) logger.debug(msg);
					else logger.info(msg);
				})
			}
			else{ // Log entries added thru 'http-proxy-middleware' (see tt.proxy & tt.admin oxsnps plugins) V3.0.23
				let name = a.name.replace(/\$/g,"/") // Replace the '$' back to '/'
				msg = '*\t' + name
				routes.push({'method':'*','route':name})
				if(bDebug) logger.debug(msg);
				else logger.info(msg);
			}

        })
	}
	return routes
}

/**
 * Enable the route
 * @param  {Response}   res      ResponseObject
 * @param  {Function} 	callback callback(err)
 */
function _enableRoute(res, callback){

	var fqPlugin 		= ''
	  , req 			= res._Context.enableRouteReq || {}
	  , msg 			= undefined
	  , module 			= undefined
	  ;

	// If plugin is not enabled
	if(res._Context.plugin.enable && res._Context.plugin.enable !== 'on'){
		msg = req.body.module + ' is not enabled. Parms:' + JSON.stringify(req.body);
		logger.error('--->_enableRoute: Req. Id=' + res._Context.reqId + ', Error: ' + msg);
		return callback(new Error(msg));
	}

	// Fetch req.body.route.path in the plugins array
	for (var i = res._Context.plugin.routes.length - 1; i >= 0; i--){
		// If optional method parameter is given, use it to find the route with that method
		// This is useful since same path (especially HTTP Paths) can be used with different methods(get|post)
		if(req.body.route.method && req.body.route.method.length > 0){
			if(res._Context.plugin.routes[i] !== undefined && 
				res._Context.plugin.routes[i].method !== undefined && 
				res._Context.plugin.routes[i].method === req.body.route.method &&
				//res._Context.plugin.routes[i].path !== undefined && 
				(res._Context.plugin.routes[i].path === req.body.route.path || res._Context.plugin.routes[i].subscription === req.body.route.path)){
                    if( res._Context.plugin.routes[i].tenant === req.body.route.tenant){ 
                        res._Context.route = res._Context.plugin.routes[i]; // add the route to the Context
                        req.body.route = res._Context.plugin.routes[i]; 	//!!! add the route to the req
                        break;
                    }
			}
		}
		else if(res._Context.plugin.routes[i] !== undefined && 
		   //res._Context.plugin.routes[i].path !== undefined && 
		   (res._Context.plugin.routes[i].path === req.body.route.path || res._Context.plugin.routes[i].subscription === req.body.route.path)){
                if( res._Context.plugin.routes[i].tenant === req.body.route.tenant){                
				    res._Context.route = res._Context.plugin.routes[i]; // add the route to the Context
				    req.body.route = res._Context.plugin.routes[i]; 	//!!! add the route to the req
                    break;
                }
		}
	};
    //if(logger.isDebugEnabled()) logger.debug('--->_enableRoute: Req. Id=' + res._Context.reqId + ', res._Context.route: ' + JSON.stringify(res._Context.route));

	// If route does not exist, leave
	if(!res._Context.route){
		msg = (req.body.route.tenant ? ('tenant: '+  req.body.route.tenant +', ') : '') + 'plugin: ' + req.body.module + ', route: ' + req.body.route.path + ' does not exist'
		logger.warn('--->_enableRoute: Req. Id=' + res._Context.reqId + ', ' + msg);
		return callback(new warn(msg));
	}

	// If route is already enabled, leave
	if(res._Context.route.enable === 'on'){
		msg = (req.body.route.tenant ? ('tenant: '+  req.body.route.tenant + ', ') : '') + 'plugin: ' + req.body.module + ', ' + (req.body.route.method || 'get') + ' route: ' + req.body.route.path + ' already enabled'
		logger.warn('--->_enableRoute: Req. Id=' + res._Context.reqId + ', ' + msg)
		return callback(new warn(msg));
	}

	// Get the module handle
	try{
		fqPlugin = res._Context.plugin.plugindir + res._Context.plugin.module + '/' + pluginJSfilename;
		module = require(fqPlugin);
		if(!module){
			logger.error('--->_enableRoute: Req. Id=' + res._Context.reqId + ', Error loading Plugin ' + fqPlugin);
			return callback(new Error('Error loading Plugin ' + fqPlugin + ', Parms:' + JSON.stringify(req.body)));				
		}
		res._Context.pluginHandle = module;
	}catch(err){
		logger.error('--->_enableRoute: Req. Id=' + res._Context.reqId + ', Error loading Plugin ' + fqPlugin + '. Reason: ' + err.message);
		return callback(new Error('Error loading Plugin ' + fqPlugin + '. Reason: ' + err.message + ', Parms:' + JSON.stringify(req.body)));
	}

	// Call enableRoute of the plugin if it exists
	if (typeof(module.enableRoute) === 'function'){
		if(logger.isDebugEnabled()) logger.debug('--->_enableRoute: Req. Id=' + res._Context.reqId + ', Route: ' + JSON.stringify(res._Context.route) + 
					 ', Calling ' + req.body.module + '.enableRoute...');
		// route.enable must be on to enable the route in oxsnps-dirwatcher and oxsnps-ibmmq
		res._Context.route.enable = 'on';
		module.enableRoute(req, function(err, route){
			if(err){
				// route.enable to off since we are unable to enable the route
				res._Context.route.enable = 'off';
				msg = 'Module=' + req.body.module +
				 	  ', Route=' + req.body.route.path +
				 	  ', Method=' + req.body.route.method +
				 	  ', Reason: ' + err.message;
				logger.error('--->_enableRoute: Req. Id=' + res._Context.reqId + ', Unable to enable a Route. ' + err.message);
				return callback(new Error(msg));
			}
			else return callback();
		});
	}
	else{
		logger.warn('--->_enableRoute: Req. Id=' + res._Context.reqId + ', Plugin ' + req.body.module + '.enableRoute not available.');
		return callback(new warn('Plugin ' + req.body.module + '.enableRoute() not available.'));	
	}
}

/**
 * Disable the route
 * @param  {Response}   res      ResponseObject
 * @param  {Function} 	callback callback(err)
 */
function _disableRoute(res, callback){

	var fqPlugin 		= ''
	  , req 			= res._Context.disableRouteReq || {}
	  , msg 			= undefined
	  , module 			= undefined
	  ;

	// If plugin is not enabled
	if(res._Context.plugin.enable && res._Context.plugin.enable !== 'on'){
		msg = req.body.module + ' is not enabled. Parms:' + JSON.stringify(req.body);
		logger.error('--->_disableRoute: Req. Id=' + res._Context.reqId + ', Error: ' + msg);
		return callback(new Error(msg));
	}

	// Fetch  req.body.route.path in the plugins array
	for(var i = res._Context.plugin.routes.length - 1; i >= 0; i--){
		// If optional method parameter is given use it to find the route with that method
		// This is useful since same path (especially HTTP Paths) can be used with different methods(get|post)
		if(req.body.route.method && req.body.route.method.length > 0){
			if(res._Context.plugin.routes[i] !== undefined && 
				res._Context.plugin.routes[i].method !== undefined && 
				res._Context.plugin.routes[i].method === req.body.route.method &&
                //res._Context.plugin.routes[i].path !== undefined && 
    			(res._Context.plugin.routes[i].path === req.body.route.path || res._Context.plugin.routes[i].subscription === req.body.route.path)){
                    if( res._Context.plugin.routes[i].tenant === req.body.route.tenant){ 
                        res._Context.route = res._Context.plugin.routes[i]; // add the route to the Context
                        req.body.route = res._Context.plugin.routes[i]; 	//!!! add the route to the req
                         break;
                    }
			}
		}			
		else if(res._Context.plugin.routes[i] !== undefined && 
		   //res._Context.plugin.routes[i].path !== undefined && 
		   (res._Context.plugin.routes[i].path === req.body.route.path || res._Context.plugin.routes[i].subscription === req.body.route.path)){
            if( res._Context.plugin.routes[i].tenant === req.body.route.tenant){
                res._Context.route = res._Context.plugin.routes[i]; // add the route to the Context
                req.body.route = res._Context.plugin.routes[i]; 	//!!! add the route to the req
                break;
            }
		}
	};
    //if(logger.isDebugEnabled()) logger.debug('--->disableRoute: Req. Id=' + res._Context.reqId + ', Route: ' + JSON.stringify(res._Context.route));

	// If route does not exist, leave
	if(!res._Context.route){
		msg =  (req.body.route.tenant ? ('tenant: '+  req.body.route.tenant) +', ' : '') + 'plugin: ' + req.body.module + ', route: ' + req.body.route.path + ' does not exist'
		logger.warn('--->_disableRoute: Req. Id=' + res._Context.reqId + ', ' + msg)
		return callback(new warn(msg))
	}

	// If route is already disabled, leave
	if(res._Context.route.enable !== 'on'){
		msg =  (req.body.route.tenant ? ('tenant: '+  req.body.route.tenant + ', ') : '') + 'plugin: ' + req.body.module + ', ' + (req.body.route.method || 'get') + ' route: ' + req.body.route.path + ' already disabled'
		logger.warn('--->_disableRoute: Req. Id=' + res._Context.reqId + ', ' + msg)
		return callback(new warn(msg));
	}

	// Get the module handle
	try{
		fqPlugin = res._Context.plugin.plugindir + res._Context.plugin.module + '/' + pluginJSfilename;
		module = require(fqPlugin);
		if(!module){
			logger.error('--->_disableRoute: Req. Id=' + res._Context.reqId + ', Error loading Plugin ' + fqPlugin);
			return callback(new Error('Error loading Plugin ' + fqPlugin + ', Parms:' + JSON.stringify(req.body)));
		}
		res._Context.pluginHandle = module;
	}catch(err){
		logger.error('--->_disableRoute: Req. Id=' + res._Context.reqId + ', Error loading Plugin ' + fqPlugin + '. Reason: ' + err.message);
		return callback(new Error('Error loading Plugin ' + fqPlugin + '. Reason: ' + err.message + ', Parms:' + JSON.stringify(req.body)));
	}

	// Call disableRoute of the plugin if it exists
	if(typeof(module.disableRoute) === 'function'){
		if(logger.isDebugEnabled()) logger.debug('--->disableRoute: Req. Id=' + res._Context.reqId + ', Route: ' + JSON.stringify(res._Context.route) + 
					 ', Calling ' + req.body.module + '.disableRoute...');
		module.disableRoute(req, function(err, route){
            if(err && err instanceof warn) logger.warn('--->_disableRoute: Req. Id=' + res._Context.reqId + ', '+ err.message)  // V4.1.27
			else if(err){
				msg = 'Module=' + req.body.module +
				 	  ', Route=' + req.body.route.path +
				 	  ', Method=' + req.body.route.method +
				 	  ', Reason: ' + err.message;
				logger.error('--->disableRoute: Req. Id=' + res._Context.reqId + 
				 			 ', Unable to disable a Route. ' + err.message);
				return callback(new Error(msg));
            }
			res._Context.route.enable = 'off';
			return callback(err);
		});
	}
	else{
		logger.warn('--->_disableRoute: Req. Id=' + res._Context.reqId + 
					', Plugin ' + req.body.module + '.disableRoute not available.');
		return callback(new warn('Plugin ' + req.body.module + '.disableRoute not available.'));
	}
}
// V4.1.7
/**
 * Return true if plugin exists or return false
 */
function _pluginExists(pluginName){
    let plugin = undefined

    if(!plugins || plugins.length <=0) return false

    for(plugin of plugins) {
        if(plugin.module === pluginName) return true
    }
    return false
}
// V4.1.7

// V4.1.22 Begin
/**
 * Add plugin swagger conf, to swagger conf. object
 */
function _addToSwaggerConf(plguinSwaggerConf){

    let urls = undefined

    if(!plguinSwaggerConf) return

    if(!Array.isArray(plguinSwaggerConf)) plguinSwaggerConf = [plguinSwaggerConf]

    if(swagger.urls.length === 0) return swagger.urls = plguinSwaggerConf

    //check and remove plugin swagger conf. entry if it already exists
    urls = swagger.urls.filter(function(item){
        if(!item || !item.name) return false
        let duplicate = false
        plguinSwaggerConf.forEach(function(pluginItem){
            if(item.name === pluginItem.name) duplicate = true
        })
        if(duplicate) return false
        return true
    })
    //console.log('remove urls:' + JSON.stringify(urls))
    //Add plugin swagger conf, to swagger conf. object
    urls = urls.concat(plguinSwaggerConf)
    swagger.urls = urls
}
// V4.1.22 End