/**-----------------------------------------------------------------------------
 * queue.js: Node.js module that provides APIs to push tasks to a specified queue. 
 *           The task contains information to call a function in a plugin along with parameters.
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2014 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V1.0.0
 * 
 * History
 *  V100        15.11.2017     Initial release
 *
 *----------------------------------------------------------------------------*/
'use strict';

var MODULE_NAME 	= 'queue'
  , MODULE_VERSION 	= '1.0.0'
  , async	    	= require('async')
  , log4js	    	= require('log4js')  
  , npsServer		= require('./server')
  , npsConf 		= npsServer.npsConf				// server-conf.js of oxs-xxx-server
  , pluginManager 	= require('oxsnps-core/pluginManager')      
  , utils 			= require('./helpers/utils')
  , queues 			= []

// Get Logger
var logger = log4js.getLogger(MODULE_NAME);
utils.log4jsSetLogLevel(logger, (utils.getAppLogLevel(npsConf.log) || 'INFO'));

// Export Module Version
exports.version = MODULE_VERSION;

/**
 * Set the log level of this module
 * @param  {req} 	GET req   http://localhost:1026/appservices/cache/setloglevel?loglevel=DEBUG|INFO|ERROR
 * @param  {res}	res
 */
exports.setLogLevel = function(req, res){
	var loglevel = req.query.loglevel || 'INFO';
	utils.log4jsSetLogLevel(logger, loglevel);
	return res.end('--->setLogLevel: Log level of ' + NAME + ' set to ' + loglevel);
}

/**
 * Return the log level of this module
 * @param  {req} 	GET req   http://localhost:1026/appservices/cache/getloglevel
 * @param  {res}	res
 */
exports.getLogLevel = function(req, res){
	logger.info('--->getLogLevel: Getting the log level of ' + NAME + '...');
	return res.end(logger.level.levelStr);
}

/**
 * Add task to a specified queue
 * @param  {Response}	res 		Response Object
 *                      res._Context.queue = {
 *                      	'info':{
 *                      		'name': <Queue Name>,
 *                      		'concurrency': >concurrency>
 *                      	},
 *                      	'task':{
 *                      		'name':<task name>, 
 *                      		'plugin':{
 *                      			'name':<plugin name>, 
 *                      		 	'service':<plugin service name>
 *                      		 }
 *                      	}
 * @param  {Function}	callback 	callback(err, result)
 */
exports.add = function(res, callback){

	var queueRef = undefined;

	if(!res._Context.queue)
		return callback(new Error('queue.add(): Queue parameter is missing in res._Context.'));

	if(!res._Context.queue.info || !res._Context.queue.info.name)
		return callback(new Error('queue.add(): Queue name is not passed. Parms=' + JSON.stringify(res._Context.queue)));

	if(!res._Context.queue.task || !res._Context.queue.task.plugin
		 || !res._Context.queue.task.plugin.name || !res._Context.queue.task.plugin.service)
		return callback(new Error('queue.add(): task plugin name or plugin service is not passed. Parms=' + JSON.stringify(res._Context.queue)));

	if(!res._Context.queue.task.name) 
		res._Context.queue.task.name = res._Context.queue.task.plugin.name + '.' + res._Context.queue.task.plugin.service;
	// Get queue of this directory
	queueRef = _getQ(res);

	// add the task to the queue
	logger.info('--->add: Adding task '+ res._Context.queue.task.name + ' to "' + res._Context.queue.info.name + '" queue');
	queueRef.push(res, callback);
	/*
	queueRef.push(res, function(err, result){
		// control comes here only after file has been added to file system
	 	return callback(err, result);
	});
	*/
}

/**
 * Get the queue that corresponds to the passed directory.
 * If queue does not exist, create one.
 * @param  {[type]} dirname Directory name
 * @return {JSON}           Queue 
 */
function _getQ(res){

	var queueRef 	= queues[res._Context.queue.info.name];

	if(!queueRef){
		logger.debug('--->getQ: Creating queue for "' + res._Context.queue.info.name + '"...');

		queues[res._Context.queue.info.name] = {};

		// Create a queue for this directory
		queues[res._Context.queue.info.name].queue = async.queue(function(res, next){
				logger.debug('--->_getQ: Starting task '+ res._Context.queue.task.name + ' from "' + res._Context.queue.info.name + '" queue');
				pluginManager.callPluginService(res._Context.queue.task.plugin, res, function(err, result){
					logger.debug('--->_getQ: Finished task '+ res._Context.queue.task.name + ' from "'+ res._Context.queue.info.name + '" queue');
					next(err, result);
				});	 
			},
			res._Context.queue.info.concurrency || 1  	// run one task at time
		);
	}

	return queues[res._Context.queue.info.name].queue;
}

