/**-----------------------------------------------------------------------------
 * server.js :  Node.js module that creates an application server
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2021 by Maas Holding GmbH
 * Email     :  support@maas.de
 * 
 * History
 * 
 *  V5.0.37  25.08.2025  cache.js: "options.timeToKeep" parameter passed to initialize() specified the minutes to retain the files in cache directory. 
 *                                 Extended to add "options.timeUnit"/"options.unit" property to "options" parameter to specify the unit for "options.timeToKeep" parameter. 
 *                                  "options.timeUnit"/"options.unit" could be, 
 *                                  'y' --> years,    'm' --> months, 'w' --> weeks, 
 *                                  'd' --> days,     'h' --> hours,  'n' --> minutes, 
 *                                  's' --> seconds,  's' --> seconds 'x' --> milliseconds 						
 *                                  Default is 'n' --> minutes
 *  V5.0.36  21.07.2025  Exended to catch 'uncaughtException' errors (errors that are not caught by plugin or any try...catch blocks)
 *  V5.0.35  08.07.2025  HNDMS-1916: Apply oxsnps-core 4.1.x changes (upto 4.1.37) to oxsnps-core V5.x.x 
 *                       V4.1.36  22.11.2024  OXS-16233: Changes of v5.0.32 copied to 4.1.x. helpers/utils.js: Added a getFileStat() function with retry options
 *                       V4.1.35  06.09.2024  OXS-15968: Catch unhandled reject exception thrown from external modules. It is done to cathc errors like "node:internal/process/promises:279 error"
 *                       V4.1.34  17.06.2024  OXS-15593: utils.js/deleteDir: Fixed a bug that deleted dirs, that are newly created after starting/invoking deleteDir() (mainly temp dierctories)
 *                       V4.1.33  07.03.2024  OXS-15459: Extended to set content security policies in response header when given in server-conf.js
 *                       V4.1.32  09.02.2024  OXS-14542: To fix vulnerability issues, moment@2.24.0 replaced with moment@2.29.4 supplant@0.2.0 replaced with utils: cheerio module usage removed
 *                       V4.1.31  01.08.2023  OXS-14527: utils.js: Extend getAuthToken() function to use technical username/password if the passed username or password is undefined.
 *                       V4.1.30  27.07.2023  OXS-14527: Re-organize ingress external rule - support new login url (same as V5.0.18)
 *                       V4.1.29: Same as 5.0.17 
 *                       V4.1.28: already integrated
 *                       V4.1.27	09.05.2022	 OXS-13673: utils.js: Extended xml2json() function to assert if callback has been already called before calling it 
 *                       utils.js: Extended iterateJSON() to work with node16 
 *  V5.0.34  14.02.2024  OXS-16464 Hallesche : Ticket 617485 – DMS: Installation (oxs-dialog-server-3.0.2.rpm) [REWT] - hot fix
 *  V5.0.33  12.11.2024   OXS-16233: helpers/utils.js: Added a getFileStat() function with retry options
 *  V5.0.32  12.11.2024   a. OXS-16233: helpers/utils.js: Added a getFileStat() function with retry options
 *           12.11.2024   b. OXS-16464 Hallesche : Ticket 617485 – DMS: Installation (oxs-dialog-server-3.0.2.rpm) [REWT]
 *  V5.0.31  25.10.2024   OXS-xxxx:  index.js: Fixed a minor bug in _parseEnvVars() when log level is set to DEBUG
 *  V5.0.30  07.10.2024   OXS-16212:  index.js: Extended to get server assertion flags from an environment variable
 *  V5.0.29  09.05.2024   OTS-3554:  httputil.js: Log rules for retry requests for different scenarios
 *  V5.0.28  22.04.2024   OTS-3552:  Extend /isAlive API to skip retrying
 *  V5.0.27  18.04.2024   OTS-3549:  pluginManager.js: Extend to activate the plugins in the order given in server-conf.js as done for enabling the plugins
 *  V5.0.26  16.04.2024   OTS-3550:  httpUtils.js: Extend getURL()/postURL() APIs with endless retry options
 *  V5.0.25  12.04.2024   OTS-3548:  httpUtils.js: Extend assert environment variable with "retryOptions" property
 *  V5.0.24  14.12.2044   utils.js:  Added missing "iconv = require('iconv-lite')" since it is required for encodeBuffer() function
 *  V5.0.23  12.10.2023   OTS-3503:  index.js: Extended to stop the server in case of errors, when "stopOnInitErrors" is true
 *  V5.0.22  12.10.2023   OTS-xxxx:  server.js: Extended to convert npsConf.stopOnInitErrors value to boolen if given as string
 *  V5.0.21  11.10.2023   OTS-3494:  server.js: Extended to disable swagger routes for 'production' environment
 *  V5.0.20  04.10.2023   OTS-3490:  server.js: Extended to give access to document files (/doc directory) to use with Swagger interface
 *  V5.0.19  11.08.2023   OTS-3483:  httpUtils.getRequestParms(): Extended to support content-type "text/*"
 *  V5.0.18  27.07.2023   OXS-14527: Re-organize ingress external rule - support new login url
 *  V5.0.17  21.06.2023   OXS-14715: From V4.1.29: Extended to keep server status('starting', 'started', 'stoping') in export.status variabel. It is used while shutting down the server to skip some tasks (refer to oxsnps-gcpubsub._removePushSubscription())
 *           22.06.2022	  OXS-14408: From V4.1.29: utils.js: Extended getAuthToken() function to use TT_HOST_IP_CONTEXT env. variable to get tt-server hostname and port
 *  V4.1.28	 10.03.2022   OXS-14471: httpUtils.js: Cloud Run instance sends an ECONNRESET error response even after sending a successful response with status code=200 for a request. 
 *  V5.0.16 03.04.2023   OTS-3449: oxsnps-core: Extend to add a cron job to log server alive status
 *                                 oxsnps-core: index.js: Refactored init() funtion to use async.series()
 *  V5.0.15 03.04.2023             oxsnps-core: index.js: Extended init() to get plugins list from an env. variable <server_prefix>_PLUGIN_LIST
 *  V5.0.14 21.03.2023   OTS-3434: oxsnps-core: Extend to use same context for the core service APIs.
 *  V5.0.13 10.03.2022   OXS-14471: Fix from V4.1.28: httpUtils.js: Cloud Run instance sends an ECONNRESET error response even after sending a successful response with status code=200 for a request. 
 *                                  This has triggered "Callback was already called." error, since we get repeat responses for the same request.
 *                                  Extended httpUtils.js: getURL()/postURL() functions to ignore repeat responses for the same request.
 *                                  It will take into account only the first response, other responses are ignored with a warning.
 * 
 *  V5.0.12	 08.08.2022  OTS-3325:: serve.js: js: Added a route app.get('/api',  core.swagger) to call index.js/swagger() function to return swagger/index.html, instead of using "static" middleware 
 *                                   which does not work consistently with nginx contexts. Still in server-conf.js static middleware is needed to serve other files in the swagger directory as given below
 *                                            "middlewares":[
 *                                                {
 *                                                    // Add static route for /api
 *                                                    "name": 	"express",
 *                                                    "handler": 	"static",
 *                                                    "routes":[
 *                                                        {
 *                                                            "path": ["/api/", "/"],					// define http://<host>:<port>/api to map to the physical dir defined below
 *                                                            "parms": process.env["OXSNPS_SERVER_ROOTDIR"] + '/swagger'	// physical dir where the files to be served are located (i.e. oxs-xxx-server/swagger)
 *                                                        }
 *                                                    ]
 *                                                }
 *                                              ],
 *                                  index..js:  Added swagger() function to return return swagger/index.html.
 * 
 *  V5.0.11	 14.07.2022  OTS-3315:: pluginManager.js: Extended not to create the state files for router of the statefilewatcher plugin
 *  V5.0.10	 12.07.2022  utils.js:  Added getIPAddresses() function 
 *  V5.0.9 	 11.07.2022  OXS-13815: utils.js: Expose private _networkInterfaces() as exports.networkInterfaces = _networkInterfaces() so that it can be used in other plugins (oxsnps-assert) where needed
 *  V5.0.8 	 13.06.2022  OXS-13782: Rename OXSNPS_SERVER_CONFDIR as OXS_SERVER_CONFDIR
 *  V5.0.7 	 13.06.2022  OXS-13767: httpUtils.js: Added deleteURL() function to perform HTTP DELETE requests
 *  V5.0.6 	 28.04.2022  OTS-3238: pluginManager.js: Added disablePluginRoutesService() function to disable all routess of a plugin
 *                                 utils.js: Added isHTTPRoute() function to check if route is  HTTP Route or not
 *  V5.0.4	 10.02.2022	 OTS-3216: Add swagger to oxs-server
 *  V5.0.2	 12.11.2021	 Added /heapdump?dir=<dirname> HTTP API and new version of exports.buildReqId()
 *  V5.0.0	 06.10.2021	 OTS-3151:  oxs-xxx-server auf die neue Architektur bringen
 *  V4.1.25	 03.01.2022	 OXS-13057: index.js: Extended heapdump() function to check node.js version, before calling v8.writeSnapshot() that is available only from node.js 11.13.0
 *                                  OTS-3186: utils.js: Added encodeBuffer() function to encode the input buffer to the given encoding <outputEnc> (used with hal-archivegw-server)
 *  V4.1.24	 15.11.2021	 OXS-13057: helpers/utils.js: Added a function detectEncoding() to find encoding of a buffer or filename
 *  V4.1.22	 10.06.2021	 OTS-3063:  pluginManager.js/index.js: Added an HTTP API /appservices/pm/swaggerconf to return swagger configuraiton with url information for plugins having HTTP interface
 *                                  pluginManager.js: Extended _initializePlugin() to optionally get swagger configuraiton with url information from plugin.initialize()
 *  V4.1.21	 21.05.2021	 OXS-12274: utils.js: Extended getOXSEcoTechUser() to get technical username/pwd from the env. variable OXSECO_DEFAULT_TECH_USER, if the list of technical user per tenant is not available
 *  V4.1.20	 04.05.2021	 OTS-2837:  Added a utils.getAddlLogParms() to return additional log parameters
 *  V4.1.19	 04.05.2021	 OTS-2837:  Added a utils.getAddlLogParms() to return additional log parameters
 *  V4.1.18	 07.04.2021	 OXS-12062: utils.js.getAuthToken(): If process.env.OXSECO_HTTP_TIMEOUT is specified as a string, convert it to a number before passing it to  http.request() service
 *  V4.1.17	 07.04.2021	 OXS-12062: utils.js: Removed the usage of outdated 'request' module. Extended getAuthToken() to use httpUtils.postURL() that uses retry options
 *  V4.1.15  18.12.2020  OTS-2942:  pluginManager.js: Extend to pass the reason to rename the state file to disablebyerror to oxsnps-statfilewatcher.updateRouteStatefile
 *  V4.1.11	 12.10.2020  OTS-2906:  Extended to get username and password to get authorization token to use it with configuration server/ordersystem APIs from technical user JS file specified in OXSECO_TECH_USER env. variable
 *  V4.1.9 	 11.09.2020  OTS-2812:  oxsnps-core: Extend httpUtils.getURL() and postURL() function to retry failed requests based retry options parameter
 *  V4.1.8 	 01.07.2020  OTS-2753   oxsnps-core/utils.js: Extend utils.getMountInfoSync() and utils.getHostsSync() to support Windows OS  
 *  V4.1.7 	 18.06.2020  OXS-10750: pluginManager.js: Extend to check if oxnps-statefilewatcher plugin exists befor using it to to create state files
 *  V4.1.0   18.02.2020  OTS-2692:  Add middlewares at run time based on the server requirements
 *  V3.0.47  06.02.2020  OXS-10274  Extend to add bodyParser.json middleware handler only when "bodyParser" section is not in server-conf.js
 *  V3.0.39  09.08.2019  OXS-9525   Extend to complete the running jobs before closing the application
 *  V3.0.35  10.01.2019  OXS-8765   server.js: Return a failure exit code(non zero value) in case of server start errors
 *           10.01.2019  OTS-2398   deploy.js: Eliminate console messages and write the message at INFO level
 *  V3.0.34  17.12.2018  Ensured to always specify body length in uploadFileAsBuffer
 *  V3.0.32  14.11.2018  OTS-2326   pluginManager.js: Extend to provide an option to skip saving plugin configuration
 *  V3.0.32  14.11.2018  utils.js: Extended mergeJSON() function to use "deepmerge" module for merging JSON objects
 *  V3.0.32  01.11.2018  Set default bodyParser if not set via server-conf.js [npsConf.bodyParser]
 *                       default:app.use(bodyParser.urlencoded({extended: false}));
 *  V3.0.26  24.07.2018  OTS-2250   Extended to read application specific plugin confs while starting the server
 *  V3.0.25  24.07.2018  OXS-8470   Messages loop appears in Front End Console.log by resubmission
 *  V3.0.16  14.02.2018  OTS-2049   Extended to do application specific tasks after starting the server
 *  V3.0.10  23.10.2017  OTS-1897   Extended to get maximum request body size from server-conf.js
 *  V100     25.08.2014  Initial release
 *----------------------------------------------------------------------------*/
'use strict';

var path        = require('path')
  , express     = require('express')
  , favicon     = require('serve-favicon')
  , bodyParser  = require('body-parser')
  , serve_index = require('serve-index')
  , log4js      = require('log4js')
  , nodeCache   = require('node-cache')  
  , mode        = parseInt('0777',8)
  , utils       = require('./helpers/utils')
  , app         = express()
  , server      = undefined 
  , npsDir      = path.resolve(__dirname + '/../../')
  , npsConfDir  = npsDir + '/conf'
  , npsConf     = undefined // server-conf.js of oxs-xxx-server
  , npsPort     = undefined
  , appname     = undefined // OXSNPS server name (i.e. oxs-xxx-server)

  // Backward compatibily
  , packageJson = require(npsDir + '/package.json')
  , npsName     = packageJson.name || 'OXSNPS Server'
  , npsVersion  = packageJson.version || 'unknown'  // Get Server Version from package.json
  , npsDesc     = packageJson.description || 'OXSNPS Server'

process.env['OXSNPS_SERVER_ROOTDIR'] = npsDir

// Fetch appname from args
if(process.argv.length>2){appname = process.argv[2]}
if(appname){
  // Set npsConfDir based on 'OXS_SERVER_CONFDIR' if defined (i.e. "OXS_SERVER_CONFDIR": "/etc/oxs-server/")
  // If not defined, use oxs-server/conf  
  // 5.0.34 - Begin
  ///npsConfDir  = process.env['OXS_SERVER_CONFDIR'] ? path.resolve(process.env['OXS_SERVER_CONFDIR'] + '/conf') : npsConfDir    //5.0.8
  // OXS_SERVER_CONFDIR must be /usr/lib/node_modules so that it is common for all Servers
  npsConfDir  = process.env['OXS_SERVER_CONFDIR'] ? path.resolve(process.env['OXS_SERVER_CONFDIR']) + '/' + appname  + '/conf': npsConfDir    //5.0.8
  // 5.0.34 - End
  // 5.0.32 b - Begin
  //npsConf     = require(path.resolve(npsConfDir + '/' + appname + '/server-conf.js'))
  // 5.0.34 - Begin
  //let npsConfDirUpd = !npsConfDir.endsWith(appname) ? npsConfDir + '/' + appname :  npsConfDir
  //npsConf     = require(path.resolve(npsConfDirUpd + '/server-conf.js'))
  npsConf     = require(path.resolve(npsConfDir) + '/' + appname + '/server-conf.js')
  // 5.0.34 - End
  // 5.0.32 b - End
  npsName     = npsConf.name || 'OXSNPS Server'
  npsVersion  = npsConf.version || 'unknown'  // Get Server Version from package.json
  npsDesc     = npsConf.description || 'OXSNPS Server'
} 
else{
  npsConf = require(npsConfDir + '/server-conf.js') // backward compatibily
}

// Set OXSNPS server port
npsPort = npsConf.port || process.env.PORT || 3000

// Create log dir if not existing
npsConf.logDir = npsConf.logDir || '/var/log/nps';
utils.mkDirSync(npsConf.logDir, mode);

// Create Temp dir if not existing
npsConf.tempDir = path.resolve((npsConf.tempDir?npsConf.tempDir:'/tmp/nps'));
utils.mkDirSync(npsConf.tempDir, mode);

// Set the pluginDirs array
npsConf.pluginDirs = npsConf.pluginDirs || npsDir + '/node_modules/' // default plugin dir
if(!Array.isArray(npsConf.pluginDirs)) npsConf.pluginDirs = [npsConf.pluginDirs]

if(npsConf.stopOnInitErrors === undefined) npsConf.stopOnInitErrors = true  // V5.0.6

// Create Cache dir if not existing but required
if(npsConf.cache){
  npsConf.cache.dir = (npsConf.cache.dir?npsConf.cache.dir:'/var/cache/nps');
  utils.mkDirSync(npsConf.cache.dir, mode);
  
  // Export the Server Cache Directory
  exports.npsCacheDir = path.resolve(npsConf.cache.dir);

  // Create and export a Memory Cache
  //!!! Should be moved to the cache core service
  exports.npsMemCache = new nodeCache({"stdTTL":npsConf.cache.memory.ttl, "checkperiod":npsConf.cache.memory.checkperiod});  
}

// Export the Express App
exports.expressApp = app;

// Export the Application Name.
exports.npsName = npsName;

// Export the Application Version.
exports.npsVersion = npsVersion;

// Export the Server Description
exports.npsDesc = npsDesc;

// Export the Server Directory
exports.npsDir = npsDir;

// Export the Server Temp Directory
exports.npsTempDir = path.resolve(npsConf.tempDir);

// Export the Server Configuration Directory
exports.npsConfDir = npsConfDir;

// Export the Server Log File
exports.npsLogFile = path.resolve(npsConf.logDir) + '/npsServer.log'; //!!! NOT USED when controlled from PM2

// Export the Server Configuration
exports.npsConf = npsConf;

// Export the Server Port
exports.npsPort = npsPort;

// Export the Server status
// V5.0.17 Start
exports.STARTING_STATUS = 'starting'
exports.READY_STATUS    = 'ready'
exports.STOPING_STATUS  = 'stoping'
exports.status = exports.STARTING_STATUS
// V5.0.17 End

// Export the appname
exports.appname = appname

// Set log properties as defined in pluginConf.log4js
// npsConf.log.level = npsConf.log.level || 'INFO';
log4js.configure(npsConf.log, {});

var core = require('oxsnps-core');  // V3.0.16

// Get the Logger
var logger = log4js.getLogger(npsName);
utils.log4jsSetLogLevel(logger, (utils.getAppLogLevel(npsConf.log) || 'INFO'));

if(npsConf.stopOnInitErrors && typeof(npsConf.stopOnInitErrors) === 'string') npsConf.stopOnInitErrors = utils.stringToBoolean(npsConf.stopOnInitErrors)  // V5.0.22
/**
 * init:              Initialize the Server
 * @param  {function}  callback(err)
 */
function _init(callback){
  let envvar = undefined
    , retValue= undefined

  logger.debug('init: Initializing OXSNPS Server...')
  logger.info('Starting ' + (npsDesc || 'OXSNPS Server') + ' v' + npsVersion + ' (based on ' + packageJson.description + ' v' + packageJson.version + ')...')
  
  // Log Hardware Info
  logger.info(utils.getHardwareInfo())

  // Log Environment Info
  logger.info(utils.getEnvironmentInfo())

  // Log /etc/hosts
  logger.info(utils.getHostsSync())

  // Log Mount points Info
  logger.info(utils.getMountInfoSync())

  // Log Node Version
  logger.info('Running Node ' + process.version + '...')

  // Create npsTempDir
  utils.mkDir(exports.npsTempDir, parseInt('0777',8), function(err){
    if(err) logger.error('init: Error when creating Temp Dir ' + exports.npsTempDir + ', Message: ' + err.message)
  })

  // V4.1.33
  retValue = core.setResponseHeaders()
  if(retValue && retValue instanceof Error) return callback(retValue)

  // V5.0.21 Begin
  if(npsConf.disableSwagger && typeof npsConf.disableSwagger === 'string'){
    envvar = npsConf.disableSwagger
    if(process.env[envvar]){
        npsConf.disableSwagger = utils.parseJSON(process.env[envvar])
        if(npsConf.disableSwagger === null){
            logger.error('Error on parsing environment variable: ' + envvar + ', value: ' + process.env[envvar])
            npsConf.disableSwagger = []
        }
    }
    else npsConf.disableSwagger = []
  }
  if(npsConf.disableSwagger && Array.isArray(npsConf.disableSwagger) && npsConf.disableSwagger.length >0) npsConf.disableSwagger.forEach(function(item){if(item) item = item.toLowerCase()})
  // V5.0.21 End
  // Set the parameters need for the sub-modules
  app.set('port', npsPort)

// V4.1.0 Begin
  // Define routes to get version, history...
  app.get('/', core.main) 
  if(npsConf.envname && npsConf.disableSwagger && npsConf.disableSwagger.length >0 && npsConf.disableSwagger.includes(npsConf.envname.toLowerCase())){/*do not add /api route*/ } // V5.0.21
  else app.get('/api', core.swagger)  // V5.0.21
  app.get('/ping',    core.ping)
  app.post('/ping',   core.ping)
  app.get('/version', core.getVersion)
  app.get('/logserveralivestatus', function(req, res){ res.send(core.logServerAliveStatus())})	// V5.0.16
  //app.get('/isalive', core.isAlive)	// V5.0.14
  app.get('/isaliveall', core.isAlive)	// V5.0.14
  app.get('/history', core.getHistory)
  app.get(['/appservices', '/appservices/routes'], core.getRoutes)//!!! needed?
  app.get('/appservices/core/gelloglevel', core.getLogLevel)
  app.get('/appservices/core/setloglevel', core.setLogLevel)
  app.get('/quit',    core.quit)
  app.get('/http',    core.processHttpReq)
  app.get('/heapdump',core.heapdump);	// V5.0.2

  // Set favicon
  app.use(favicon(npsDir + '/public/img/favicon.ico'))
  app.use("/appservices", bodyParser.json({limit: '2mb'}))
  app.use("/", express.static(path.join(npsDir, 'public')))

  // V5.0.20  Begin
  app.use("/doc", express.static(path.join(npsDir, 'doc')))
  app.use('/doc', serve_index(path.join(npsDir, 'doc'), {hidden:true, icons:true, view:'details'}))
  // V5.0.20 End


  // Allow to browse the server's root directory
  // 1. Map http://server:port/oxsnps to the root dir
  app.use('/oxsnps', serve_index('/',{hidden:true, icons:true, view:'details'}))
  
  // 2. Allow static files stored in / to be served
  app.use('/oxsnps', express.static('/'))
// V4.1.0 End  
  // Needs to be added because of CROS (see http://johnzhang.io/options-request-in-express)
  //!!! Check whether allowing CORS for all the requests is an issue!!!
  app.options('*', function(req, res, next){
    logger.debug('--->OPTIONS Req., req.headers: ' + JSON.stringify(req.headers))
    let method=req.headers["access-control-request-method"]
    if (method) {
      res.setHeader("Access-Control-Allow-Methods","HEAD,GET,POST,PUT,DELETE,OPTIONS")
      res.setHeader("Allow","HEAD,GET,POST,PUT,DELETE")
      res.setHeader("Access-Control-Allow-Headers","Content-Type")
      res.setHeader("Access-Control-Allow-Origin","*")
    }
    res.sendStatus(200)
  })

  // Add Plugin routes
  core.init(function(err){
    return callback(err)
  })

  // Add Error Handling
  // Development error handler...will print stacktrace
  if (app.get('env') === 'development') {
    app.use(function(err, req, res, next) {
      res.status(err.status || 500).end('Error: ' + err.message + '\nStatus: ' + err.status + '\nStack: ' + err.stack)
    })
  }
  else{ // Production error handler...no stacktraces leaked to user
    app.use(function(err, req, res, next) {
      res.status(err.status || 500).end('Error: ' + err.message + '\nStatus: ' + err.status)
    })
  }
}

// Initialize the Server Application
_init(function(err){

  if(err){
    logger.error('Error while initializing ' + exports.npsDesc + ' v' + exports.npsVersion
           + ' on port ' + app.get('port') + ', Reason: ' + err.message);
    process.exit(1);  // V3.0.35
  }

  // Start the server     
  server = app.listen(app.get('port'), function(err){
    if(err){
      logger.error('Error on starting ' + exports.npsDesc + ' v' + exports.npsVersion
             + ' on port ' + app.get('port')+ ', Reason: ' + err.message);
      process.exit(1);  // V3.0.35
    }
    core.setupShutdownOptions(server)	// V3.0.39
    // V3.0.16 Begin

	// V5.0.36 Begin
    // These are errors thrown and not caught by any try...catch.
    process.on('uncaughtException', (err) => {
        console.error('Unhandled Exception:', err);
        // only catch the exception and do not shutdown the server as there may be some running jobs.
        //return process.exit(1); 
      });
// V5.0.36 End 

    // V4.1.35
    // Catch unhandled reject exception thrown from external modules.
    // These occur when a Promise is rejected and there’s no .catch() handler.
    process.on('unhandledRejection', (reason, promise) => {
      console.error('oxsnps-core: Unhandled Rejection error at:', promise, 'reason:', reason)
      // only catch the exception and do not shutdown the server as there may be some running jobs.
      //return process.exit(1); 
    })

    core.afterServerStart(function(err){
      if(err){
        logger.error('Error on doing tasks after starting ' + exports.npsDesc + ' v' + exports.npsVersion
               + ' on port ' + app.get('port')+ ', Reason: ' + err.message);
        return process.exit(1);  // V3.0.35
      }
      logger.info(exports.npsDesc + ' v' + exports.npsVersion +  ' is listening on port ' + app.get('port'));
      exports.status = exports.READY_STATUS // V5.0.17
    });
    // V3.0.16 End
  });
});

