/**-----------------------------------------------------------------------------
 * shutdown.js: 	Node.js module to setup server shutdown options
 * 
 * Author    :  AFP2web Team
 * Copyright :  (C) 2014 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  V1.0.0
 * 
 * History
 *
 *----------------------------------------------------------------------------*/
'use strict';

var MODULE_NAME 	= 'shutdown'
  , MODULE_VERSION 	= '1.0.0'
  , log4js 			= require('log4js') 
  ,	async 			= require('async')
  , npsServer		= require('./server')
  , npsConf 		= npsServer.npsConf					// holds the configuration of the Server 
  , cronjob 		= require('./cronjob')
  , pluginManager 	= require('./pluginManager')  
  , utils 			= require('./helpers/utils')

// Get Logger
let logger = log4js.getLogger(MODULE_NAME)
utils.log4jsSetLogLevel(logger, (utils.getAppLogLevel(npsConf.log) || 'INFO'))

// Export Module Version
exports.version			= MODULE_VERSION;
exports.shutdownOptions = undefined

/**
 * Setup Shutdown options
 * Shutdown options are given in env. variable <OXSNPS_SERVER_PREFIX>_SHUTDOWN_OPTIONS
 * Ex:
 *		"OXS_BATCH_IMPORT_SHUTDOWN_OPTIONS": "{\"timeout\":10000, \"signals\":[\"SIGTERM\", \"SIGINT\"], \"jobTimeout\":1000 }",	// Refer to https://github.com/godaddy/terminus
 * @param  {server} 	server Appplcation server object 
 */
exports.setupShutdownOptions = function(server){

	let options				= undefined
	  , shutdownOptions		= process.env[ process.env.OXSNPS_SERVER_PREFIX + 'SHUTDOWN_OPTIONS'] 
	  //, defaultSignalList	= ['SIGTERM', 'SIGINT']		// signals handled by oxsnps-core/server.js
	
	if(shutdownOptions) shutdownOptions = utils.parseJSON(shutdownOptions)
	if(logger.isDebugEnabled()) logger.debug('--->setupShutdownOptions: ShutdownOptions: ' + JSON.stringify(shutdownOptions))

	exports.shutdownOptions = shutdownOptions
	
	if(shutdownOptions && (shutdownOptions.signal || shutdownOptions.signals)){
		const { createTerminus } = require('@godaddy/terminus');
		
		// If shutdownOptions.signals not given, consider shutdownOptions.signal option
		if(shutdownOptions.signal && !shutdownOptions.signals) shutdownOptions.signals = [shutdownOptions.signal]
		
		// Convert signal name to uppercase
		shutdownOptions.signals.forEach(function(signal){signal = signal.toUpperCase()})

		// filter out signals that are specified in env. var from defaultSignalList		
		//defaultSignalList = defaultSignalList.filter(function(signal){
		//	if(shutdownOptions.signals.includes(signal)) return false
		//	return true
		//})
		
		// Refer to https://github.com/godaddy/terminus
		options = {
			timeout:		shutdownOptions.timeout || 1000,			// [optional = 1000] number of milliseconds before forceful exiting
			signals:		shutdownOptions.signals,                   	// [optional = []] array of signals to listen for relative to shutdown
			beforeShutdown:	_beforeShutdown,	                  	// [optional] called before the HTTP server starts its shutdown
			onShutdown:		exports.onShutdown,                      			// [optional] called right before exiting
			logger:			_logger                           			// [optional] logger function to be called with errors			
			//signal:                          							// [optional = 'SIGTERM'] what signal to listen for relative to shutdown			
			//onSignal:		shutdownOptions.                        	// [optional] cleanup function, returning a promise (used to be onSigterm)
			//onShutdown:		shutdownOptions.                      	// [optional] called right before exiting
			//onSendFailureDuringShutdown: shutdownOptions.     		// [optional] called before sending each 503 during shutdowns
		}
		
		createTerminus(server, options)	// call terminus to setup shutdown options
		return 
	}
	
	// If shutdownOptions not specified in env.var, Setup shutdown that close application immediately
	//_setupSimpleShutdown(server, defaultSignalList)
	_setupSimpleShutdown(server)
}

exports.onShutdown = function() {
  if(logger) logger.info('--->onShutdown: Shutting down the server...' )
  console.log('--->onShutdown: Shutting down the server...' )
		//console.log('onshutdown')
  Promise.resolve()
}

/**
 * Function to shutdown server gracefully
 */
exports.beforeShutdown = function(res, callback) {

	// Create res if it is undefined
	res = res || {};
	npsServer.status = npsServer.STOPING_STATUS // V5.0.17
	// V3.0.13 Begin
	// Add Context to the response instance
	res._Context = res._Context || {}

	// Add the Current Date to the Context
	res._Context.date = res._Context.date || new Date()

	// Add a Request Id to the Context
	res._Context.reqId = res._Context.reqId || utils.buildReqId(res._Context.date)

	res._Context.softquit = res._Context.softquit || true
	
	res._Context.shutdownOptions = exports.shutdownOptions
	res._Context.shutdownOptions.jobTimeout	= res._Context.shutdownOptions.jobTimeout || 60000	// default is one minute
	
	logger.info('--->beforeShutdown: Req. id=' + res._Context.reqId + ', Shutdown options: ' + JSON.stringify(res._Context.shutdownOptions) + ', Starting tasks to shutdown server ...' )

	async.series([
		// Finalize the cronjob module
		function(nextTask){
			cronjob.finalize(function(err){
				if(err) logger.	error('-->beforeShutdown: quit, Error: ' + err)
				nextTask()
			});
		},
		// Deactivate plugins
		function(nextTask){
			pluginManager.deactivatePlugins(res, nextTask)
		},
		// Disable Plugins
		function(nextTask){
			pluginManager.disablePlugins(res, nextTask)
		}		
	],
	function(err){
		callback(err)
	})
}

/**
 * Service to delete Jobs by state
 * @param  {Response}  	res Response Object
 *                      res:{
 *                      	_Context.shutdown:{
 *                      		"jobPrefix": 	"Optional. Job Prefix like plugin name",
 *                      		"routes":  		"<Routes> as defined in pluginConf or in channel properties,
 *                      		"routeTypes":  	"[<Array of Route Types>] like dir, queue, ftp", 
 *                     			"jobTimeout": 	"Optional. Job timeout in milliseconds. Time to wait before forcefully terminating a job. Default is one second"
 *                      	}
 *                      }
 * @param  {Function} 	callback(err, deletedJobInfo)
 */
exports.waitTillJobCompletion = function(res, callback){

	let timeout		= res._Context.shutdown.jobTimeout	|| pluginConf.jobTimeout 
	  , routeTypes	= res._Context.shutdown.routeTypes	|| []
	  , kue			= undefined
	  , jobs		= undefined
	  
	
	if(!res._Context.shutdown.routes || res._Context.shutdown.routes.length <=0) return callback()
	
	if(!Array.isArray(routeTypes)) routeTypes = [routeTypes]

	try{
		kue = require('kue')
	}catch(err){
		return callback(new Error('--->waitTillJobCompletion: Unable to load kue nodule: ' + err.message))
	}
	
	try{
		jobs = kue.createQueue(npsConf.kue) 
	}catch(err){
		return callback(new Error('--->waitTillJobCompletion: Unable to create kue instance: ' + err.message))
	}
	
	async.forEach(res._Context.shutdown.routes, 
		function(route, next){
			if(!route || !route.method || !routeTypes.includes(route.method)) return next()
			let type = res._Context.shutdown.jobPrefix || ''
			type += route.tenant ? (route.tenant + '_' + (route.subscription || route.path)) : (route.subscription || route.path)
			if(logger.isDebugEnabled()) logger.debug('--->waitTillJobCompletion: Req. id=' + res._Context.reqId + ', Gracefully shutting down any jobs of type: ' + type + ', Jobtimeout: ' + timeout)
			// !!1 todo: log => Gracefully shutting down running jobs for plugin: ci.orgrollkarte, tenant: wackler, subscription: fv-orgrollkarte_4017, jobs count: <count>
			jobs.shutdown(timeout, type, function(err){		// refer to https://www.npmjs.com/package/kue#graceful-shutdown
				if(logger.isDebugEnabled()) logger.debug('--->waitTillJobCompletion: Req. id=' + res._Context.reqId + ', Jobs of type: ' + type + ', are completed')
				if(!err) return setTimeout(function(){return next()}, 50)
				return next(new Error('Error while gracefully shutting down jobs of type: ' +  type + ', ' + err.message))
			})
		}, 
		function(err) {
			callback(err)
		}
	)
}

function _logger(msg, err){
	logger.error('-->logger: terminus error: ' + msg + ', ' + (err ? err.message : '' ))
}

/**
 * Function to shutdown server gracefully
 */
function _beforeShutdown(res, callback){

	return new Promise((resolve, reject) => {
		exports.beforeShutdown({}, function(err){
			if(err){
				logger.error('-->_beforeShutdown: Error on closing the server: ' + err.message)
				return reject(new Error('-->_beforeShutdown: Error on closing the server: ' + err.message))
			}
			logger.debug('--->_beforeShutdown: Over' );
			return resolve()
		})
	})
}

/*
 * Setup functions for signals specified in default signal list
 */
//function _setupSimpleShutdown(server, defaultSignalList){
function _setupSimpleShutdown(server){

	// Simple shutdown function that does not ensure running jobs are completed before closing the application. Used in Hallesche DMS2
	let shutdown = function(){
	
		logger.warn('-->shutdown: Shutting down gracefully.')
		//else console.log('-->shutdown: Shutting down gracefully.')
		
		let kue 			= undefined
		  , jobs			= undefined
		  
		try{
			kue = require('kue')
		}catch(err){
			if(logger.isDebugEnabled()) logger.debug('--->shutdown: Unable to load kue: ' + err.message)
		}
		
		if(kue){
			if(logger.isDebugEnabled()) logger.debug('--->shutdown: Kue shutting down...')
			try{
				jobs = kue.createQueue(npsConf.kue) 
				jobs.shutdown() 
			}catch(err){
				if(logger.isDebugEnabled()) logger.debug('--->shutdown: Unable shutdown jobs: ' + err.message)
			}
		}
		  
		if(server){
			if(logger.isDebugEnabled()) logger.debug('--->shutdown: Closing the server...')
			server.close(function(){
				logger.warn('-->shutdown: Closed out remaining connections.')
				process.exit(); // OXS-8765 exit() gets exit code from process.exitCode if defined. Else default exit code is zero
			})

			// If after 
			setTimeout(function(){
			  logger.warn('-->shutdown: Could not close connections in time, forcefully shutting down.')
			  process.exit() // OXS-8765: exit() gets exit code from process.exitCode if defined. Else default exit code is zero
			}, 10*1000)
		} 
		else process.exit(1);  // OXS-8765
	}
	
	//if(defaultSignalList.includes('SIGTERM')){
		if(logger.isDebugEnabled()) logger.debug('-->_setupSimpleShutdown: Adding handler for SIGTERM...')
		// Listen for TERM signal .e.g. kill 
		process.on ('SIGTERM', function(){
		  // OXS-8765 Begin
		  process.exitCode = 1	// SIGTERM is not called by PM2. But it can be called from shell to kill the process 
		  shutdown()
		  // OXS-8765 End
		})
	//}
	
	// Listen for INT signal e.g. Ctrl-C
	//if(defaultSignalList.includes('SIGINT')){
		if(logger.isDebugEnabled()) logger.debug('-->_setupSimpleShutdown: Adding handler for SIGINT...')
		process.on ('SIGINT', shutdown)   // SIGINT is called by PM2 to stop the application 		
	//} 
}		

