/**-----------------------------------------------------------------------------
 *
 * Author    :  AFP2web Team
 * Copyright :  (C) 2024-2025 by Maas Holding GmbH
 * Email     :  support@maas.de
 * 
 * oxsnps-classifier:
 * 
 * History
 * V1.0.0    07.08.2025        AFP-1257: Implement a plugin for oxsnps-office conversions
 *                                     
 *----------------------------------------------------------------------------*/
'use strict'

/**
 * Variable declaration
 * @private
 */
let fs                = require('fs')
  , path              = require('path')
  , url               = require('url')  
  , async             = require('async')  
  , log4js            = require('log4js')

  , npsServer         = require('oxsnps-core/server')
  , npsConf           = npsServer.npsConf
  //, npsConfDir        = (npsServer.appname) ? path.join(npsServer.npsConfDir, npsServer.appname) : npsServer.npsConfDir  
  //, npsName         = npsServer.npsName
  //, npsVersion      = npsServer.npsVersion
  //, npsLogFile      = npsServer.npsLogFile
  , npsLogDir         = npsConf.logDir
  , npsTempDir        = npsServer.npsTempDir

  , packageJson       = require(__dirname + '/package.json')
  , PLUGIN_NAME       = packageJson.name
  , PLUGIN_VERSION    = packageJson.version // Get Server Version from package.json
  , pluginConf        = require(__dirname + '/conf/' + PLUGIN_NAME + '.js')

  , utils             = require('oxsnps-core/helpers/utils')
  , httpUtils         = require('oxsnps-core/helpers/httpUtils')
  , dateFormat        = require('oxsnps-core/helpers/date_format')
  , expressAppUtils   = require('oxsnps-core/helpers/expressAppUtils')
  , lo                = require(path.join(__dirname, 'services', 'libreoffice.js'))
  , mso               = require(path.join(__dirname, 'services', 'msoffice.js'))
  , common            = require(path.join(__dirname, 'helpers', 'common.js'))    

  , FILE_SEP          = path.sep
  , initError         = undefined

// Set plugin name, version and longDesc in pluginConf
pluginConf.module     = PLUGIN_NAME 
pluginConf.version     = PLUGIN_VERSION 
pluginConf.longDesc = packageJson.description
pluginConf.log         = pluginConf.log || {}

// Export the Plugin's Configuration Express App
exports.pluginConf     = pluginConf
exports.router      = undefined     

// Get the Logger
let logger = log4js.getLogger(PLUGIN_NAME)
utils.log4jsSetLogLevel(logger, (pluginConf.log.level || 'INFO'))

/**************** PUBLIC FUNCTIONS ****************/
/**
 * Initialize Plugin
 * @param  {function}     callback  callback(err, routes)
 */
exports.initialize = function(callback){

    let self = this
      , appPluginConf    = (npsConf.plugins && npsConf.plugins[PLUGIN_NAME] ? npsConf.plugins[PLUGIN_NAME] : {})    

    function _handleError(err){
        initError = err 
        self._initialized = false 
        callback(err, pluginConf.routes)
    }          

    if(self._initialized) return callback()      

    if(logger.isDebugEnabled()) logger.debug('--->initializing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION)

    initError       = undefined

    // Merge application specific configuration with plugin configuration
    pluginConf = utils.mergeJSON(pluginConf, appPluginConf)

    exports.pluginConf = pluginConf  
    
    async.series([
        // Check if <pluginLogDir>' path exists
        function(nextTask){
            _parseEnvs(nextTask)
        },
        function(nextTask){
            expressAppUtils.setupRouter(self, function(err, router){      
                exports.router = router
                nextTask(err)
            })            
        }
    ],
    function(err){
        exports.pluginConf = pluginConf 
        if(err) return _handleError(err) // 4.0.0
        //logger.debug('--->pluginConf:', pluginConf)
        self._initialized = true;
        logger.info('\t\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' initialized');
        // callback(null, pluginConf.routes);
        callback(null, {'routes': pluginConf.routes, 'swagger': undefined /*{"name": PLUGIN_NAME, "url": '/' + PLUGIN_NAME + '/api/rest.yaml'}*/})        
    });    
}

/**
 * Activate Plugin. This function is called from pluginManager once all the enabled plugins have been initialize
 * @param  {function}     callback  callback(err)
 */
exports.postInitialize = function(callback){
    let self = this

    if(!self._initialized)
        return callback(new Error('Can\'t activate Plugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' as long as it has not been initialized!'))
    
    if(logger.isDebugEnabled()) logger.debug('Activating ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + '...')

    async.series([
        function(nextTask){
            lo.initialize(nextTask)
        },
        function(nextTask){
            mso.initialize(nextTask)
        },        
        function(nextTask){
            common.initialize(nextTask)
        }
    ],  
    function(err){
        if(err){
            self._initialized = false
            initError = err
        }
        return callback(err)
    })            
}

/**
 * Finalize Plugin
 * @param  {function}     callback callback(err) 
 */
exports.finalize = function(callback){

    let self = this

    if(logger.isDebugEnabled()) logger.debug('--->finalizing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION )
    if(!self._initialized) return callback()

    async.series([
        // Disable all routes
        function(nextTask){
            async.each(pluginConf.routes, 
                function(route, next){
                    let req = {'body': {'module': PLUGIN_NAME, 'route': route}}
                    exports.disableRoute(req, next)
                },
                function(err){
                    return nextTask(err)
                }
            )
        }
    ],
    function(err){
        if(err) return callback(err)
        self._initialized = false
        logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' finalized')
        callback()
    })

}

/**
 * Return Plugin Version
 * @param  {function}     callback callback(err,version)
 */
exports.getVersion = function(callback){
    return callback(null, PLUGIN_VERSION)
}

/**
 * Process a HTTP Version Request
 * @param  {Request}      req Request Object
 * @param  {Response}     res Response Object
 */
exports.version = function(req, res){

    let data = undefined

    // Add Context to the response instance
    res._Context = {}

    // Add the Current Date to the Context
    res._Context.date = new Date()

    // Add a Request Id to the Context
    res._Context.reqId = utils.buildReqId(res._Context.date)
    logger.info('--->version: Req. Id=' + res._Context.reqId)

    data = '<h3>' +  PLUGIN_NAME + ' v' + PLUGIN_VERSION + '</h3>'
    res.end(data)

    logger.info('--->version: Req. Id=' + res._Context.reqId + ', sent back ' +
                PLUGIN_NAME + ' v' + PLUGIN_VERSION + ', over')
    return
}

/**
 * Process a HTTP Configuration Request 
 *     Either as HTTP GET request:
 *          GET http://localhost:1029/services/<plugin>/conf                             --> To retrieve the configuration
 *          GET http://localhost:1029/services/<plugin>/conf?save=true&conf="{...}"     --> To pass and save the configuration
 *     or as HTTP POST request:
 *          POST http://localhost:1029/services/<plugin>/conf                         --> To pass and save the configuration
 *          req.body={
 *              "conf": {...}
 *          }
 *     
 * @param {Request}     req Request Object
 *                      req.query:{ // for HTTP GET request:
 *                          "save":"false",
 *                          "conf":"{<IMPORTANT: COMPLETE JSON Conf. as String>}"
 *                      }
 *                      req.body:{ // for HTTP POST request:
 *                          "conf":{<IMPORTANT: COMPLETE JSON Conf. as JSON Object>}
 *                      }
 * @param {Response}    res Response Object
 */
exports.conf = function(req, res){

    let reqType     = 'getconf'
      , jsonConf     = undefined
      
    // Add Context to the response instance
    res._Context = {}

    // Add the Current Date to the Context
    res._Context.date = new Date()

    // Add a Request Id to the Context
    res._Context.reqId = utils.buildReqId(res._Context.date)
    logger.info('--->conf: Req. id=' + res._Context.reqId)
    
    if(req.method.toUpperCase() === 'POST'){
        logger.info('--->conf: body=' + JSON.stringify(req.body))
        if(!req.body || !req.body.conf) {
            logger.info('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body))
            return res.end('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body))
        }
        reqType = 'saveconf'         // save request
        jsonConf = req.body.conf     // set the json conf to be passed to saveConfService
    }
    else if(req.method.toUpperCase() === 'GET'){
        logger.info('--->conf: query=' + JSON.stringify(req.query))
        if(req.query && req.query.save && req.query.save.toLowerCase() === 'true' && req.query.conf)
        {
            reqType = 'saveconf'
            jsonConf = utils.parseJSON(unescape(req.query.conf)) // remove escaped chars and set the json conf to be passed to saveConfService
        } 
    }

    switch(reqType){
        case 'saveconf':
            exports.saveConfService(jsonConf, function(err){
                logger.info('--->conf: Req. id=' + res._Context.reqId + ', over')
                res.json(err ? {'status':'error', 'message': err.message} : pluginConf)
            }) 
            break
        default:
        case 'getconf':
            logger.info('--->conf: Req. id=' + res._Context.reqId + ', over')
            res.json(pluginConf)
            break
    }
}

/**
 * Store  the Plugin conf to its configuration file (i.e. .../<plugin>/conf/<plugin>.js)
 * @param  {JSON}       conf     Plugin Configuration
 * @param  {Function}     callback
 */
exports.saveConfService = function(jsonConf, callback){

    let filename     = undefined
      , data         = undefined
      
    // PluginManager calls saveConfService without passing the conf
    if(typeof jsonConf === 'function'){
        callback = jsonConf
        jsonConf = pluginConf
    }

    // Assert jsonConf
    if(!jsonConf || jsonConf === null) return callback(new Error('--->saveConfService: empty Configuration'))
    if(logger.isDebugEnabled()) logger.debug('--->saveConfService: conf:' + JSON.stringify(jsonConf))

    // Backup the current plugin conf.in the server's backup dir and store the new one in the plugin's /conf dir
    async.series([
        // Task 1: Backup the current plugin conf.
        function(nextTask){
            filename = npsConf.backupDir + FILE_SEP + dateFormat.asString('yyyy-MM-dd_hh-mm-ss', new Date()) + '_configurationOf_' + PLUGIN_NAME + '.js' // build the Backup filename
            if(logger.isDebugEnabled()) logger.debug('--->saveConfService: Backuping plugin configuration to ' + filename)
            data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t') // build the plugin conf content

            // Write data to the Plugin configuration file
            fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
                if(err) logger.error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message) // log error & continue
                nextTask()
            })
        },
        // Task 2: Save the new plugin conf.
        function(nextTask){
            pluginConf = jsonConf // we should have here a proper conf, so overwrite the current pluginConf file
            filename = path.resolve(__dirname + '/conf/' + PLUGIN_NAME + '.js') // build the plugin configuration filename
            if(logger.isDebugEnabled()) logger.debug('--->saveConfService: Writing plugin configuration to ' + filename)
            data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t') // build the plugin conf content

            // Write data to Plugin configuration file
            fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
                if(err) nextTask(new Error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message))
                else nextTask()
            }) 
        }
    ],
    function(err){
        callback(err)
    })
}

/**
 * Process a HTTP GET Ping Request
 * 
 * @param  {Request}      req Request Object
 * @param  {Response}     res Response Object
 */
exports.ping = function(req, res){

    // Add Context to the response instance
    res._Context = {};

    // Add a Request Id to the Context
    res._Context.reqId = utils.buildReqId(new Date())
    
    logger.info('--->ping: Req. id=' + res._Context.reqId)

    // Call the ping service
    exports.pingService(res, function(err, result){
        if(err){
            logger.error('--->ping: Req. id=' + res._Context.reqId + ', Reason: ' + + err.message)
            return res.status(404).send( err.message)
        }            
        logger.info('--->ping: Req. id=' + res._Context.reqId + ', result: ' + result + ', over!')
        return res.send(result)
    })
}

/**
 * ping service that returns nopodofo external module version + oxnsps-nopodofo version
 *
 * @param  {JSON}         res
 * @param  {Function}     callback(err,result)
 */
exports.pingService = function(res, callback){

    return callback(null, 'oxsnps-office2ay version: ' + PLUGIN_VERSION)
    // Todo: Get and retturn Libreoffice version
}

/**
 * 
 * URLs: 
 *      POST http://localhost:port/services/office2any/transform
 * 
 * @param  {Request}      req Request Object
 *                            req.body:{
 *                              'input':{
 *                                  'filename': <Fully Qualified input filename>
 *                                  'type': Optional. Input file type like 'doc', 'docx', 'odt', 'rtf' or 'xls', 'xlsx', 'ods', 'csv'. Default it to take it from file extn
 *                                  'readOptions':{   
 *                                      'password': <Password for encrupted documents>
 *                                  }
 *                              },
 *                              'output':{
 *                                  path: <FQ output path>
 *                                  transformOptions:{     // libre office filter options given as key value pairs 
 *                                      'EncryptFile':          true | false,
 *                                      'DocumentOpenPassword': <Document Open Password>,
 *                                      'UserPassword':         <User Permissions password>,
 *                                      'SelectPdfVersion':       <0: standard, 1: PDF/A>
 *                                      ....
 *                                  }
 *                              },
 *                              // Exec options to use while invoking office CLI
 *                              'execOptions':{   // Refer to https://nodejs.org/api/child_process.html#child_processexecfilefile-args-options-callback
 *                                  'timeout': <Timeout in milliseconds before killing the process.>. Default is 60 seconds, If value is 0, process runs indefinitely and will NOT be killed automatically if in case it is hanging.
 *                                  //env: { ...process.env, LANG: 'en_US.UTF-8' },
 *                              },
 *                              // number of times to retry the office conversion 
 *                              'retry':{
 *                                  'count': <number of times to retry conversion>,
 *                                  'interval': <Delay in milliseconds between retry attempts>
 *                              },
 *                              engine: 'libreoffice' | 'msoffice'. Default is 'libreoffice'
 *                           }
 * @param  {Response}     res Response Object
 * On Error:
 *  Error message sent as string 
 * On Suceess:
 *  The following json will be sent
 *  {
 *      'outputFilename' : <FQ outputFilename>
 *  }
 */
exports.transform = function(req, res){

    let lp          = undefined
      , msg         = undefined
      , reqOptions  = undefined

    res._Context        = {'o2a': {}}
    res._Context.reqId  = utils.buildReqId(new Date())
    lp = '--->transform: Req. id=' + res._Context.reqId + ', '

    logger.info(lp + (req.headers['content-type'] ? 'Content-type: ' + req.headers['content-type'] :'') + ', Query Parms: ' + JSON.stringify(req.query) + ', Path Parms: ' + JSON.stringify(req.params) + ', Req.body: ' + JSON.stringify(req.body))
    
    if(!req.body){
        msg = 'Request body is missing. Req.Body : ' + JSON.stringify(req.body)
        logger.error(lp + res._Context.reqId + msg)
        return res.status(400).send(msg)
    }

    try{
        reqOptions = url.parse(req.url)
        if(!reqOptions) return res.status(400).send('Unable to parse the URL string. url: ' + req.url)
    }catch(err){
        return res.status(400).send('Unable to parse the URL string. url: ' + req.url + '. ' + err.message)
    }
    /*
        reqOptions = {
            "protocol": null,
            "slashes": null,
            "auth": null,
            "host": null,
            "port": null,
            "hostname": null,
            "hash": null,
            "search": "?enc=utf-8",
            "query": "enc=utf-8",
            "pathname": "/services/ci.baw/createdocumentindex",
            "path": "/services/ci.baw/createdocumentindex?enc=utf-8",
            "href": "/services/ci.baw/createdocumentindex?enc=utf-8"
        }
    */

    res._Context.o2a.req            = req
    res._Context.o2a.parms          = httpUtils.getRequestParms(req)
    res._Context.o2a.reqStartTime   = new Date() 
    
    exports.transformService(res, function(err){
        res._Context.o2a.reqEndTime    = new Date()
        if(err){
            msg = 'Error: ' + err.message
            logger.error(lp + msg + ', parms: ' + JSON.stringify(res._Context.o2a.parms))
            return res.status(err.code || 400).end('Req.Id=' + res._Context.reqId + ', ' + msg)
        }
        res.json(res._Context.o2a.result) // send the Response back. 
        logger.info(lp + 'over')
    })
}

/**
 * 
 * @param    {JSON}          res  Object
 *                           res._Context.o2a.parms:{
 *                              'input':{
 *                                  'filename': <Fully Qualified input filename>
 *                                  'type': Optional. Input file type like 'doc', 'docx', 'odt', 'rtf' or 'xls', 'xlsx', 'ods', 'csv'. Default it to take it from file extn
 *                                  'readOptions':{   
 *                                      'password': <Password for encrupted documents>
 *                                  }
 *                              },
 *                              'output':{
 *                                  path: <FQ output path>
 *                                  transformOptions:{     // libre office filter options given as key value pairs 
 *                                      'EncryptFile':          true | false,
 *                                      'DocumentOpenPassword': <Document Open Password>,
 *                                      'UserPassword':         <User Permissions password>,
 *                                      'SelectPdfVersion':       <0: standard, 1: PDF/A>
 *                                      ....
 *                                  }
 *                              },
 *                              // Exec options to use while invoking office CLI
 *                              'execOptions':{   // Refer to https://nodejs.org/api/child_process.html#child_processexecfilefile-args-options-callback
 *                                  'timeout': <Timeout in milliseconds before killing the process.>. Default is 60 seconds, If value is 0, process runs indefinitely and will NOT be killed automatically if in case it is hanging.
 *                                  //env: { ...process.env, LANG: 'en_US.UTF-8' },
 *                              },
 *                              // number of times to retry the office conversion 
 *                              'retry':{
 *                                  'count': <number of times to retry conversion>,
 *                                  'interval': <Delay in milliseconds between retry attempts>
 *                              },
 *                              engine: 'libreoffice' | 'msoffice'. Default is 'libreoffice'
 *                           }
 * @param    {Function}      callback    callback(err)
 * On Error:
 *  error object: err
 * On Suceess:
 *  res._Context.o2a.result : {
 *      'outputFilename' : <FQ outputFilename>
 *  }
 */
exports.transformService = function(res, callback){
    
    let lp  = '--->transformService: Req. id=' + res._Context.reqId + ', '

    // Add a Request Id and temp dir to the Context
    res._Context.reqId   = res._Context.reqId || utils.buildReqId(new Date())
    res._Context.tempDir = path.join(npsTempDir, res._Context.reqId)
    
    lo.assertParms(res, function(err){
        if(err){
            err.code = 400    // Bad request
            err.message = 'Unable to transform the document. Reason: '  + err.message
            logger.error(lp + err.message + ', Parms: ' + JSON.stringify(res._Context.o2a.parms))
            return callback(err)
        }
        lo.transform(res, function(err, result){
            if(err){
                err.message = 'Unable to transform the document. Reason: ' + err.message
                logger.error(lp + err.message + ', Parms: ' + JSON.stringify(res._Context.o2a.parms))
                return callback(err)
            }
            callback()
        })
    })
}

/************ OPTIONAL PUBLIC FUNCTIONS ***********/
/**
 * Set Log Level
 * @param {String} logLevel Log Level. values:DEBUG|INFO|WARN|ERROR 
 */
exports.setLogLevel = function(logLevel){
    pluginConf.log.level = logLevel || 'INFO'
    utils.log4jsSetLogLevel(logger, pluginConf.log.level)
    mso.setLogLevel(logLevel)
    lo.setLogLevel(logLevel)
    common.setLogLevel(logLevel)
}

/**
 * Get Log Level
 * @return {String} Log Level. values:DEBUG|INFO|WARN|ERROR
 */
exports.getLogLevel = function(){
    return pluginConf.log.level
}

/**
 * Enable the given route
 * @param  {Request}     req
 *                      req.body = {
 *                          "module":"sample", 
 *                          "route": { 
 *                              "path": "/path/to/the/service",
 *                              "method":     Optional. Values are "get|post|dw|queue",
 *                              "service":     "version"
 *                          }
 *                      }
 * @param  {function}     callback callback(err)
 */
exports.enableRoute = function(req, callback){

    let self = this

    if(logger.isDebugEnabled()) logger.debug('--->enableRoute: Route=' + JSON.stringify(req.body.route))

    if(!req.body.route) return callback(new Error('route parameter not given'))

    req.body.route.method = req.body.route.method || 'get'

    // Enable the route
    switch(req.body.route.method.toLowerCase()){
        case 'dir': // Enable Dir Route
            break
        case 'queue': // Enable Queue Route
            break
        default:
            // Enable HTTP Route
            expressAppUtils.enableRoute(self, req.body.route, callback)
    }
} 

/**
 * Disable the given route
 * @param  {Request}     request Request json has following structure.
 *                         req.body = {
 *                          "module":"sample", 
 *                          "route": { 
 *                              "path": "/path/to/the/service",
 *                              "method":     Optional. Values are "get|post|dw|queue"
 *                               }
 *                         }
 * @param  {function}     callback  callback(err)
 */
exports.disableRoute = function(req, callback){

    let self    = this  

    if(logger.isDebugEnabled()) logger.debug('--->disableRoute: Route=' + JSON.stringify(req.body.route))
    
    if(!req.body.route) return callback(new Error('route parameter not given'))

    req.body.route.method = req.body.route.method || 'get'

    // Disable the route
    switch(req.body.route.method.toLowerCase()){
        // Disable Dir Route
        case 'dir': 
            break
        case 'queue':
            break
        default:
            //logger.warn('--->disableRoute: Wrong parameters(unknown route.method)! Not able to disable the route, Parms: ' + JSON.stringify(req.body))
            //callback(new Error('Wrong parameters(unknown route.method)'))
            // Disable HTTP Route
            expressAppUtils.disableRoute(self, req.body.route, callback)
    }
} 

/**************** PRIVATE FUNCTIONS ***************/
// Parse environment variables 
function _parseEnvs(callback){

    let envvar  = undefined
      , lp       = '--->_parseEnvs: '

    if(typeof(pluginConf.docExtns) === 'string'){
        envvar = pluginConf.docExtns
        if(!process.env[envvar]) return callback(new Error('Missing environment variable: ' + envvar))
  
        pluginConf.docExtns = utils.parseJSON(process.env[envvar])
        if(pluginConf.docExtns === null) return callback(new Error('Unable to parse environment variable: ' + envvar + ', value: ' + process.env[envvar]))
    }
    if(logger.isDebugEnabled()) logger.debug(lp + 'pluginConf.docExtns: ' + JSON.stringify(pluginConf.docExtns))

    if(typeof(pluginConf.sheetExtns) === 'string'){
        envvar = pluginConf.sheetExtns
        if(!process.env[envvar]) return callback(new Error('Missing environment variable: ' + envvar))
  
        pluginConf.sheetExtns = utils.parseJSON(process.env[envvar])
        if(pluginConf.sheetExtns === null) return callback(new Error('Unable to parse environment variable: ' + envvar + ', value: ' + process.env[envvar]))
    }
    if(logger.isDebugEnabled()) logger.debug(lp + 'pluginConf.sheetExtns: ' + JSON.stringify(pluginConf.sheetExtns))

    if(typeof(pluginConf.htmlExtns) === 'string'){
        envvar = pluginConf.htmlExtns
        if(!process.env[envvar]) return callback(new Error('Missing environment variable: ' + envvar))
  
        pluginConf.htmlExtns = utils.parseJSON(process.env[envvar])
        if(pluginConf.htmlExtns === null) return callback(new Error('Unable to parse environment variable: ' + envvar + ', value: ' + process.env[envvar]))
    }
    if(logger.isDebugEnabled()) logger.debug(lp + 'pluginConf.htmlExtns: ' + JSON.stringify(pluginConf.htmlExtns))    
    
    return callback()
}

/**
 * Log a Warning if Service Response time exceeded maxresptime 
 * @param  {Response}    res      Response Object
 * @param  {String}        service service
 * @private
 */
function _logServiceRespTime(res, service, startTime, maxresptime){
}

