/**-----------------------------------------------------------------------------
 * libreoffice.js: Service to use Libre Office for transformation
 *
 * Author    :  AFP2web Team
 * Copyright :  (C) 2018 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  1.0.0
 * 
 * History
 * 
 *----------------------------------------------------------------------------*/
 'use strict'

 /**
  * letiable declaration
  * @private
  */
let async          = require('async')
  , log4js         = require('log4js')
  , path           = require('path')
  , packageJson    = require(path.join(__dirname, '..',  'package.json'))
  , PLUGIN_NAME    = packageJson.name
  //, npsServer        = require('oxsnps-core/server')
  //, npsConf         = npsServer.npsConf     
  //, pluginManager  = require('oxsnps-core/pluginManager')
  , utils          = require('oxsnps-core/helpers/utils')  
  , SERVICE_NAME   = "libreoffice"
  , logger         = log4js.getLogger(PLUGIN_NAME + '/'+ SERVICE_NAME)
  , plugin         = require(path.join('..', 'oxsnps.js'))
  , pluginConf     = undefined

const fs              = require("fs/promises")
const { constants }   = require('fs');
const { execFile }    = require("child_process")

const util            = require("util")
const execFileAsync = util.promisify(execFile)
// OR 
//const { promisify } = require("util");
//const execFileAsync = promisify(execFile);

/**
  * Set Log Level
  */  
exports.setLogLevel = function(logLevel){
    utils.log4jsSetLogLevel(logger, (logLevel || 'WARN'))
} 
 
/**
 * Initialize 
 */
exports.initialize = function(callback){

    let self    = this
      , envvar  = undefined
      , lp      = '--->initialize: '
    
    if(self._initialized) return callback()

    pluginConf = plugin.pluginConf
    utils.log4jsSetLogLevel(logger, (pluginConf.log.level || 'WARN'))      
    
    if(logger.isDebugEnabled()) logger.debug('--->initializing libreoffice.js...' )

    pluginConf.libreOffice                            = pluginConf.libreOffice                            ||  {}
    pluginConf.libreOffice.transformOptions           = pluginConf.libreOffice.transformOptions           ||  {}
    pluginConf.libreOffice.transformOptions.doc2pdf   = pluginConf.libreOffice.transformOptions.doc2pdf   ||  {'SelectPdfVersion': 0, 'ExportFormFields': true, 'MaxImageResolution': 300, 'Quality': 75, 'TaggedPDF': true}
    pluginConf.libreOffice.transformOptions.sheet2pdf = pluginConf.libreOffice.transformOptions.sheet2pdf ||  {'AllSheets': true, 'SinglePageSheets': true, 'MaxImageResolution': 300, 'Quality': 75}

    if(typeof(pluginConf.libreOffice.transformOptions.doc2pdf) === 'string'){
      envvar = pluginConf.libreOffice.transformOptions.doc2pdf
      if(!process.env[envvar]) return callback(new Error('Missing environment variable: ' + envvar))

      pluginConf.libreOffice.transformOptions.doc2pdf = utils.parseJSON(process.env[envvar])
      if(pluginConf.libreOffice.transformOptions.doc2pdf === null) return callback(new Error('Unable to parse environment variable: ' + envvar + ', value: ' + process.env[envvar]))
    }
    if(logger.isDebugEnabled()) logger.debug(lp + 'pluginConf.libreOffice.transformOptions.doc2pdf: ' + JSON.stringify(pluginConf.libreOffice.transformOptions.doc2pdf))

    if(typeof(pluginConf.libreOffice.transformOptions.sheet2pdf) === 'string'){
      envvar = pluginConf.libreOffice.transformOptions.sheet2pdf
      if(!process.env[envvar]) return callback(new Error('Missing environment variable: ' + envvar))

      pluginConf.libreOffice.transformOptions.sheet2pdf = utils.parseJSON(process.env[envvar])
      if(pluginConf.libreOffice.transformOptions.sheet2pdf === null) return callback(new Error('Unable to parse environment variable: ' + envvar + ', value: ' + process.env[envvar]))
    }

    if(logger.isDebugEnabled()) logger.debug(lp + 'pluginConf.libreOffice.transformOptions.sheet2pdf: ' + JSON.stringify(pluginConf.libreOffice.transformOptions.sheet2pdf))
    return callback()
}

/**
 * 
 * @param    {JSON}          res  Object
 *                           res._Context.o2a.parms:{
 *                              'input':{
 *                                  'filename': <Fully Qualified input filename>
 *                                  'type': Optional. Input file type like 'doc', 'docx', 'odt', 'rtf' or 'xls', 'xlsx', 'ods', 'csv'. Default it to take it from file extn
 *                                  'readOptions':{   
 *                                      'password': <Password for encrupted documents>
 *                                  }
 *                              },
 *                              'output':{
 *                                  path: <FQ output path>
 *                                  transformOptions:{     // libre office filter options given as key value pairs 
 *                                      'EncryptFile':          true | false,
 *                                      'DocumentOpenPassword': <Document Open Password>,
 *                                      'UserPassword':         <User Permissions password>,
 *                                      'SelectPdfVersion':       <0: standard, 1: PDF/A>
 *                                      ....
 *                                  }
 *                              },
 *                              'execOptions':{   // Refer to https://nodejs.org/api/child_process.html#child_processexecfilefile-args-options-callback
 *                                  'timeout': <Timeout in milliseconds before killing the process.>. Default is 60 seconds, If value is 0, process runs indefinitely and will NOT be killed automatically if in case it is hanging.
 *                                  //env: { ...process.env, LANG: 'en_US.UTF-8' },
 *                              },
 *                              'retry':{
 *                                  'count': <number of times to retry conversion>,
 *                                  'interval': <Delay in milliseconds between retry attempts>
 *                              },
 *                              engine: 'libreoffice' | 'msoffice'. Default is 'libreoffice'
 *                           }
 * @param    {Function}      callback    callback(err)
 */
exports.assertParms = function(res, callback){

    let lp    = '--->assertParms: Req. id=' + res._Context.reqId + ', '  
      , parms = res._Context.o2a && res._Context.o2a.parms

    if(logger.isDebugEnabled())  logger.debug(lp + 'Asserting request parameters, res._Context.o2a.parms: ' + JSON.stringify(res._Context.o2a.parms))
    if(!parms) return callback(new Error('Invalid request, parameters are missing'))      

    if(!parms.input) return callback(new Error('Invalid request, parms.input parameter is missing'))      

    // Assert input file
    if(!parms.input.filename)
        return callback(new Error('Invalid request, parms.input.filename parameter is missing'))      

    if(!parms.output) return callback(new Error('Invalid request, parms.output parameter is missing'))      

    // Assert output file
    if(!parms.output.path)
        return callback(new Error('Invalid request, parms.output.path parameter is missing'))      

    res._Context.o2a.opPath = path.resolve(parms.output.path)

    parms.output.transformOptions = parms.output.transformOptions || {}

    parms.execOptions         = parms.execOptions || {}
    parms.execOptions.timeout = parms.execOptions.timeout || 60000    // Timeout in milliseconds before killing the process.

    parms.retry           = parms.retry           || {} 
    parms.retry.count     = parms.retry.count     || 1      // count =1 no retry
    parms.retry.interval  = parms.retry.interval  || 1000   //

    return callback()
}

/**
 * 
 * @param    {JSON}          res  Object
 *                           res._Context.o2a.parms:{
 *                              'input':{
 *                                  'filename': <Fully Qualified input filename>
 *                                  'type': Optional. Input file type like 'doc', 'docx', 'odt', 'rtf' or 'xls', 'xlsx', 'ods', 'csv'. Default it to take it from file extn
 *                                  'readOptions':{   
 *                                      'password': <Password for encrupted documents>
 *                                  }
 *                              },
 *                              'output':{
 *                                  path: <FQ output path>
 *                                  transformOptions:{     // libre office filter options given as key value pairs 
 *                                      'EncryptFile':          true | false,
 *                                      'DocumentOpenPassword': <Document Open Password>,
 *                                      'UserPassword':         <User Permissions password>,
 *                                      'SelectPdfVersion':       <0: standard, 1: PDF/A>
 *                                      ....
 *                                  }
 *                              },
 *                              'execOptions':{   // Refer to https://nodejs.org/api/child_process.html#child_processexecfilefile-args-options-callback
 *                                  'timeout': <Timeout in milliseconds before killing the process.>. Default is 60 seconds, If value is 0, process runs indefinitely and will NOT be killed automatically if in case it is hanging.
 *                                  //env: { ...process.env, LANG: 'en_US.UTF-8' },
 *                              },
 *                              'retry':{
 *                                  'count': <number of times to retry conversion>,
 *                                  'interval': <Delay in milliseconds between retry attempts>
 *                              },
 *                              engine: 'libreoffice' | 'msoffice'. Default is 'libreoffice'
 *                           }
 * @param    {Function}      callback    callback(err)
 * On Error:
 *  error object: err
 * On Suceess:
 *  res._Context.o2a.result : {
 *      'outputFilename' : <FQ outputFilename>
 *  }
 */
exports.transform = function(res, callback){

    let lp          = '--->transform: Req. id=' + res._Context.reqId + ', '
      , parms       = res._Context.o2a.parms
      , mode        = parseInt('0777', 8)
      , ext         = undefined

    ext       = path.extname(parms.input.filename) || ''
    parms.input.type = parms.input.type || ext.slice(1).toLowerCase() // remove the dot from extn (.docx)

    res._Context.o2a.result = {}
    res._Context.o2a.baseName   = path.basename(parms.input.filename, ext)    
    res._Context.o2a.opFilename = path.join(res._Context.o2a.opPath, res._Context.o2a.baseName + '.pdf')      
        
    if(logger.isDebugEnabled()) logger.debug(lp + 'Starting to process the request, parms: ' + JSON.stringify(parms))

    async.series([
      //  Assert input file exists
      function(nextTask){
        _assertInputFile(res)
          .then(() =>{return nextTask()})
          .catch(err => {return nextTask(err)})
      },         
      // Ensure output path exist
      function(nextTask){
          if(logger.isDebugEnabled()) logger.debug(lp + 'Check and create output path: ' + res._Context.o2a.opPath)
          utils.mkDir(res._Context.o2a.opPath, mode, nextTask)
      },         
      function(nextTask){
        if(pluginConf.sheetExtns.includes(parms.input.type)) return _sheet2pdf(res, nextTask)
        if(pluginConf.htmlExtns.includes(parms.input.type)) return _sheet2pdf(res, nextTask)
        return _doc2pdf(res, nextTask)
      }        
    ],
    function(err){
      if(!err && logger.isDebugEnabled()) logger.debug(lp + 'Finished processing the request. Result: ' + JSON.stringify(res._Context.o2a.result))
      return callback(err)
    })
 }

/*
function _convertFile(res, callback){
    let lp          = '--->transform: Req. id=' + res._Context.reqId + ', '
      , parms       = res._Context.o2a.parms

    ext       = path.extname(parms.input.filename) || ''
    parms.input.type = parms.input.type || ext.slice(1).toLowerCase() // remove the dot

    res._Context.o2a.baseName   = path.basename(parms.input.filename, ext)    
    res._Context.o2a.opFilePath = path.join(res._Context.o2a.opPath, res._Context.o2a.baseName + '.pdf')      
      if(pluginConf.sheetExtns.contains(parms.input.type)) return _sheet2pdf(res, callback)
      return _sheet2pdf(res, callback)
 }
*/

/**
 * Assert if input file is readable
 */
async function _assertInputFile(res){

    let lp          = '--->_assertFileReadable: Req. id=' + res._Context.reqId + ', '
      , parms       = res._Context.o2a.parms
      , filename    = parms.input.filename

    try{
      if(logger.isDebugEnabled()) logger.debug(lp + 'Check if input file: ' + res._Context.o2a.opFilename + ' exists and is readable')
      await fs.access(filename, constants.F_OK | constants.R_OK)
      if(logger.isDebugEnabled()) logger.debug(lp + 'Input File: ' + res._Context.o2a.opFilename + ', exists and is readable')
    }
    catch(err){
      throw new Error('Input File:' + filename + ' does not exist or it is not readable,' + err.message)
    }
}

/**
 * Convert document to PDF
 */
function _doc2pdf(res, callback){

    let lp          = '--->_doc2pdf: Req. id=' + res._Context.reqId + ', '
      , parms       = res._Context.o2a.parms
      , tmp        = undefined

    // Merge request transformOptions with default plugin configuration transformOptions. 
    // If same property is used in both, request transformOptions will take the precedence
    res._Context.o2a.filterOptions = utils.mergeJSON(pluginConf.libreOffice.transformOptions.doc2pdf, parms.output.transformOptions)

    res._Context.o2a.filtername = 'writer_pdf_Export'

    if(logger.isDebugEnabled()) logger.debug(lp + 'Filter name: ' + res._Context.o2a.filtername + ', Filter Options: ' + JSON.stringify(res._Context.o2a.filterOptions))

    tmp = _convert(res)
        .then(() =>{
            return callback()
        })
        .catch(err => {
            return callback(err)
      })
      .finally(() => {
          // callback should not be invoked here
      })    
}

/**
 * Convert Excel sheet to PDF
 */
function _sheet2pdf(res, callback){

    let lp          = '--->_doc2pdf: Req. id=' + res._Context.reqId + ', '
    , parms       = res._Context.o2a.parms

    // Merge request transformOptions with default plugin configuration transformOptions. 
    // If same property is used in both, request transformOptions will take the precedence
    res._Context.o2a.filterOptions = utils.mergeJSON(pluginConf.libreOffice.transformOptions.sheet2pdf, parms.output.transformOptions)

    res._Context.o2a.filtername = 'calc_pdf_Export'

    if(logger.isDebugEnabled()) logger.debug(lp + 'Filter name: ' + res._Context.o2a.filtername + ', Filter Options: ' + JSON.stringify(res._Context.o2a.filterOptions))

    _convert(res)
      .then(() =>{
        return callback()
      })
      .catch(err => {
        return callback(err)
      })
      .finally(() => {
          // callback should not be invoked here
      })    
}

/**
 * Delay helper (ms)
 */
async function _delay(ms) {
  return new Promise(resolve => setTimeout(resolve, ms))
}

/**
 * Convert office documents(word, excel) to PDF
 */
async function _convert(res){

    let lp          = '--->_convert: Req. id=' + res._Context.reqId + ', '
      , parms       = res._Context.o2a.parms
      , password    = res._Context.o2a.parms.input.readOptions && res._Context.o2a.parms.input.readOptions.password
      , filterArgs  = undefined
      , filterStr   = ''
      , args        = undefined
      , attempts    = 1
      , keys        = Object.keys(res._Context.o2a.filterOptions) || []

    /*
    if(keys.includes('EncryptFile')){
      // if security options are given, filter options should be given as JSON string
      filterStr = "'" + res._Context.o2a.filtername + ':' + JSON.stringify(res._Context.o2a.filterOptions) + "'"
      //console.log(filterStr)  // filterStr ='writer_pdf_Export:{"SelectPdfVersion":0,"ExportFormFields":true,"MaxImageResolution":300,"Quality":75,"TaggedPDF":true,"EncryptFile":true,"EncryptDocument":true,"DocumentOpenPassword":"mht1","UserPassword":"mht1"}'
      //filterStr ='writer_pdf_Export:{"SelectPdfVersion":0,"ExportFormFields":true,"MaxImageResolution":300,"Quality":75,"TaggedPDF":true,"EncryptFile":true,"EncryptDocument":true,"DocumentOpenPassword":"mht1","UserPassword":"mht1"}'      
    }
    else{
      filterArgs  = _buildFilterOptionString(res)   
      filterStr   = filterArgs ? (res._Context.o2a.filtername + ':' + filterArgs) : filterName
    }  
    */
    filterArgs  = _buildFilterOptionString(res)   
    filterStr   = filterArgs ? (res._Context.o2a.filtername + ':' + filterArgs) : filterName
    
    args = [
      '--headless',
      '--convert-to', 
      //'-env:PWD=1',
      //'-env:UNO_USER_PASSWORD=1',
      //'-env:LIBREOFFICE_PASSWORD=1',
      //'pdf',
      'pdf:' + filterStr,
      '--outdir', 
      res._Context.o2a.opPath,      // output path
      parms.input.filename          // input filename
    ]    

    // LibreOffice CLI does not directly support password via CLI args,
    // but we can try to set an environment variable for some formats
    const env = { ...process.env }
    if(password) {     // ****** DOES NOT WORK AT ALL ******
      env['PWD'] = password; // May work for some versions
      env['UNO_USER_PASSWORD'] = password
      env['LIBREOFFICE_PASSWORD'] = password
    }

    while(attempts <= parms.retry.count){ 
      
      try {
        // libreoffice --headless --convert-to pdf:writer_pdf_Export:SelectPdfVersion=1,Quality=85 mydoc.docx
        if(logger.isDebugEnabled()) logger.debug(lp + 'Calling libreoffice for the conversion, parms: ' + JSON.stringify(args) 
                  + (password ? (', password:' + password) : ', no password')    
                  + ', Attempts: '  + attempts 
                  + (parms.execOptions ? (', execOptions:' + JSON.stringify(parms.execOptions)) : ''))    

        const {stdout, stderr} = await execFileAsync('libreoffice', args, parms.execOptions)
        const output = (stdout + stderr).trim()

        if(logger.isDebugEnabled()) logger.debug(lp + 'libreoffice stdout: ' +  stdout)
        if(logger.isDebugEnabled()) logger.debug(lp + 'libreoffice stderr: ' +  stderr)

        if (/Error|failed|not found/i.test(output)){
          throw new Error(output)
        }

        // Confirm output file exists
        if(logger.isDebugEnabled()) logger.debug(lp + 'Check if output file: ' + res._Context.o2a.opFilename + ' exists')
        await fs.access(res._Context.o2a.opFilename, constants.F_OK | constants.R_OK)
        if(logger.isDebugEnabled()) logger.debug(lp + 'Output file: ' + res._Context.o2a.opFilename + ' exists')
        res._Context.o2a.result.outputFilename = res._Context.o2a.opFilename
        return res._Context.o2a.opFilename
      } 
      catch(err){
        if (attempts < parms.retry.count) {
          logger.warn(lp + 'Attempt: ' + attempts + ' failed. Error: ' + err.message + '. Retrying in ' + parms.retry.interval + 'ms...')
          await _delay(parms.retry.interval)
          attempts++;
        } 
        else {
          throw new Error(err.message + ', Attempts tried: ' + attempts)
        }        
      }
    }
}

/**
 * Builds a LibreOffice PDF filter option string
 */
function _buildFilterOptionString(res){

  let lp          = '--->_convert: Req. id=' + res._Context.reqId + ', '
    , options     = res._Context.o2a.filterOptions
    , parts       = []
    , retValue    = undefined

  for(const [key, value] of Object.entries(options)){
    parts.push(key+'='+value)
  }
  retValue = parts.join(",")    // Ex: return value SelectPdfVersion=1,Quality=85 mydoc.docx
  if(logger.isDebugEnabled()) logger.debug(lp + 'Filter Option String: ' + retValue)
  return retValue 
}
