/**-----------------------------------------------------------------------------
 * oxsnps.js: 	
 *
 * Author    :  AFP2web Team
 * Copyright :  (C) 2017 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * Version   :  1.0.0
 * 
 * History
 * V3.0.2   20.01.2025  OXS-16546: Extend to write stdout content optionally to a file whost descriptor is passed in res._Context.options.stdio
 * V3.0.1   12.12.2024  OXS-16367: fixing cross-spawn	vulnerability CVE-2024-21538	HIGH	7.0.1	7.0.5, 6.0.6
 * V3.0.0   20.05.2022  OTS-3271: EXtended to adopt to oxsnps-core >=5.0.7 that disables routes if plugin can not be initialized
 * V2.0.0   31.03.2020  OTS-2699: Extend base and custom import/export plugins to use oxsnps-core@4.1.x
 * V1.0.2               Extended to return exit code, stdout, stderr as result JSON even in case of error so that caller can use stdout/stderr messages as he wishes
 *----------------------------------------------------------------------------*/
'use strict'

/**
 * Variable declaration
 * @private
 */
var packageJson		= require(__dirname + '/package.json')
  , PLUGIN_NAME		= packageJson.name
  , PLUGIN_VERSION	= packageJson.version // Get Server Version from package.json
  , async			= require('async')  
  , fs				= require('fs')
  , log4js			= require('log4js')
  , os				= require('os')  
  , path			= require('path')

  , npsServer		= require('oxsnps-core/server')
  , npsConf 		= npsServer.npsConf
  , npsName   		= npsServer.npsName
  , npsVersion   	= npsServer.npsVersion
  , npsLogFile 		= npsServer.npsLogFile
  , npsLogDir 		= npsConf.logDir
  , npsTempDir		= npsServer.npsTempDir

  , pluginConf		= require(__dirname + '/conf/' + PLUGIN_NAME + '.js')

  , dateFormat 		= require('oxsnps-core/helpers/date_format')
  , utils 			= require('oxsnps-core/helpers/utils')
  , expressAppUtils = require('oxsnps-core/helpers/expressAppUtils')
  , pluginLogDir	= undefined
  , FILE_SEP 		= require('path').sep

  , spawn			= require('cross-spawn')
  , initError       = undefined	// V3.0.0   // to be used when pingservice is implmented
  

// Set plugin name, version and longDesc in pluginConf
pluginConf.module 	= PLUGIN_NAME 
pluginConf.version 	= PLUGIN_VERSION 
pluginConf.longDesc = packageJson.description
pluginConf.log 		= pluginConf.log || {}

// Export the Plugin's Configuration Express App
exports.pluginConf 	= pluginConf
exports.router      = undefined     // V2.0.0

// Get the Logger
let logger = log4js.getLogger(PLUGIN_NAME)
utils.log4jsSetLogLevel(logger, (pluginConf.log.level || 'INFO'))

/**************** PUBLIC FUNCTIONS ****************/
/**
 * Initialize Plugin
 * @param  {function} 	callback  callback(err, routes)
 */
exports.initialize = function(callback){

	let self		= this
    , appPluginConf	= (npsConf.plugins && npsConf.plugins[PLUGIN_NAME] ? npsConf.plugins[PLUGIN_NAME] : {})
    
    initError = undefined   // V3.0.0
    
	if(logger.isDebugEnabled()) logger.debug('--->initializing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION )

	if(self._initialized) return callback()

	pluginLogDir = path.resolve(npsLogDir) + FILE_SEP + PLUGIN_NAME

	// Merge application specific configuration with plugin configuration
	pluginConf = utils.mergeJSON(pluginConf, appPluginConf)
	
	// Map the HTTP routes of the plugin
	expressAppUtils.setupRouter(self, function(err, router){    // V2.0.0
        if(err){
            // V2.0.0 Begin
            initError = err
            self._initialized = false 
            return callback(err, pluginConf.routes)
            // V2.0.0 End
        }          
       exports.router = router     // V2.0.0
		self._initialized = true
		logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' initialized')
		callback(null, pluginConf.routes)
	})
}

/**
 * Activate Plugin. This function is called from pluginManager once all the enabled plugins have been initialize
 * @param  {function} 	callback  callback(err)
 */
exports.postInitialize = function(callback){
	callback()
}

/**
 * Finalize Plugin
 * @param  {function} 	callback callback(err) 
 */
exports.finalize = function(callback){

	let self = this
	  , req  = {}

	if(logger.isDebugEnabled()) logger.debug('--->finalizing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION )
	if(!self._initialized) return callback()

	async.series([
		// Disable all routes
		function(nextTask){
			req.body = {'module': PLUGIN_NAME}
			async.each(pluginConf.routes, 
				function(route, next){
					req.body.route = route
					exports.disableRoute(req, next)
				},
				function(err){
					return nextTask(err)
				}
			)
		}
	],
	function(err){
		if(err) return callback(err)
		self._initialized = false
		logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' finalized')
		callback()
	})

}

/**
 * Return Plugin Version
 * @param  {function} 	callback callback(err,version)
 */
exports.getVersion = function(callback){
	return callback(null, PLUGIN_VERSION)
}

/**
 * Process a HTTP Version Request
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.version = function(req, res){

	let data = undefined

	// Add Context to the response instance
	res._Context = {}

	// Add the Current Date to the Context
	res._Context.date = new Date()

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date)
	logger.info('--->version: Req. Id=' + res._Context.reqId)

	data = '<h3>' +  PLUGIN_NAME + ' v' + PLUGIN_VERSION + '</h3>'
	res.end(data)

	logger.info('--->version: Req. Id=' + res._Context.reqId + ', sent back ' +
				PLUGIN_NAME + ' v' + PLUGIN_VERSION + ', over')
	return
}

/**
 * Process a HTTP Configuration Request 
 * 	Either as HTTP GET request:
 * 	 	GET http://localhost[:<port>]/services/<plugin>/conf 							--> To retrieve the configuration
 * 	 	GET http://localhost[:<port>]/services/<plugin>/conf?save=true&conf="{...}" 	--> To pass and save the configuration
 * 	or as HTTP POST request:
 * 	 	POST http://localhost[:<port>]/services/<plugin>/conf 						--> To pass and save the configuration
 * 	 	req.body={
 * 	 		"conf": {...}
 * 	 	}
 * 	
 * @param {Request} 	req Request Object
 *                      req.query:{ // for HTTP GET request:
 *                      	"save":"false",
 *                      	"conf":"{<IMPORTANT: COMPLETE JSON Conf. as String>}"
 *                      }
 *                      req.body:{ // for HTTP POST request:
 *                      	"conf":{<IMPORTANT: COMPLETE JSON Conf. as JSON Object>}
 *                      }
 * @param {Response}    res Response Object
 */
exports.conf = function(req, res){

	let reqType 	= 'getconf'
	  , jsonConf 	= undefined
	  
	// Add Context to the response instance
	res._Context = {}

	// Add the Current Date to the Context
	res._Context.date = new Date()

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date)
	logger.info('--->conf: Req. id=' + res._Context.reqId)
	
	if(req.method.toUpperCase() === 'POST'){
		logger.info('--->conf: body=' + JSON.stringify(req.body))
		if(!req.body || !req.body.conf) {
			logger.info('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body))
			return res.end('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body))
		}
		reqType = 'saveconf' 		// save request
		jsonConf = req.body.conf 	// set the json conf to be passed to saveConfService
	}
	else if(req.method.toUpperCase() === 'GET'){
		logger.info('--->conf: query=' + JSON.stringify(req.query))
		if(req.query && req.query.save && req.query.save.toLowerCase() === 'true' && req.query.conf)
		{
			reqType = 'saveconf'
			jsonConf = utils.parseJSON(unescape(req.query.conf)) // remove escaped chars and set the json conf to be passed to saveConfService
		} 
	}

	switch(reqType){
		case 'saveconf':
		    exports.saveConfService(jsonConf, function(err){
		    	logger.info('--->conf: Req. id=' + res._Context.reqId + ', over')
				res.json(err ? {'status':'error', 'message': err.message} : pluginConf)
		    }) 
			break
		default:
		case 'getconf':
			logger.info('--->conf: Req. id=' + res._Context.reqId + ', over')
			res.json(pluginConf)
			break
	}
}

/**
 * Store  the Plugin conf to its configuration file (i.e. .../<plugin>/conf/<plugin>.js)
 * @param  {JSON}   	conf     Plugin Configuration
 * @param  {Function} 	callback
 */
exports.saveConfService = function(jsonConf, callback){

	let filename 	= undefined
	  , data 		= undefined
	  
	// PluginManager calls saveConfService without passing the conf
	if(typeof jsonConf === 'function'){
		callback = jsonConf
		jsonConf = pluginConf
	}

	// Assert jsonConf
	if(!jsonConf || jsonConf === null) return callback(new Error('--->saveConfService: empty Configuration'))
	if(logger.isDebugEnabled()) logger.debug('--->saveConfService: conf:' + JSON.stringify(jsonConf))

	// Backup the current plugin conf.in the server's backup dir and store the new one in the plugin's /conf dir
	async.series([
		// Task 1: Backup the current plugin conf.
		function(nextTask){
			filename = npsConf.backupDir + FILE_SEP + dateFormat.asString('yyyy-MM-dd_hh-mm-ss', new Date()) + '_configurationOf_' + PLUGIN_NAME + '.js' // build the Backup filename
			if(logger.isDebugEnabled()) logger.debug('--->saveConfService: Backuping plugin configuration to ' + filename)
			data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t') // build the plugin conf content

			// Write data to the Plugin configuration file
			fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
				if(err) logger.error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message) // log error & continue
				nextTask()
			})
		},
		// Task 2: Save the new plugin conf.
		function(nextTask){
			pluginConf = jsonConf // we should have here a proper conf, so overwrite the current pluginConf file
			filename = path.resolve(__dirname + '/conf/' + PLUGIN_NAME + '.js') // build the plugin configuration filename
			if(logger.isDebugEnabled()) logger.debug('--->saveConfService: Writing plugin configuration to ' + filename)
			data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t') // build the plugin conf content

			// Write data to Plugin configuration file
			fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
				if(err) nextTask(new Error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message))
				else nextTask()
			}) 
		}
	],
	function(err){
		callback(err)
	})
}

/**
 * HTTP API to spawn a process
 * 	Either as HTTP GET request:
 * 	 	GET http://localhost[:<port>]services/spawn/execute?cmd=<process to execute>&args=[...]&options={...}
 * 	or as HTTP POST request:
 * 	 	POST http://localhost[:<port>]/services/spawn/execute
 * 	 	req.body={
 * 	 		"cmd": <process to execute>,
 * 			"args": [...]
 * 			"options": {...}
 * 	 	}
 * 	
 *	Spawn process, see https://github.com/moxystudio/node-cross-spawn
 * 	For options, see https://nodejs.org/docs/latest-v8.x/api/child_process.html#child_process_child_process_spawn_command_args_options
 * 
 * @param {Request} 	req Request Object
 *                      req.query:{ // for HTTP GET request:
 *                      	"cmd":		<process to execute,
 *                      	"args": 	[OPTIONAL] <Array of arguments>,
 *                      	"options":	[OPTIONAL] <hash of options>}
 *                      }
 *                      req.body:{ // for HTTP POST request:
 *                      	"cmd":		<process to execute,
 *                      	"args": 	[OPTIONAL] <Array of arguments>,
 *                      	"options":	[OPTIONAL] <hash of options>
 *                      }
 * @param {Response}    res Response Object
 * 
 */
exports.execute = function(req, res){

	// Add Context to the response instance
	res._Context = {}

	// Add the Current Date to the Context
	res._Context.date = new Date()

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date)

	logger.info('--->execute: Req. id=' + res._Context.reqId)
	
	if(req.method.toUpperCase() === 'POST'){
		if(!req.body || !req.body.cmd){
			logger.error('--->execute: Req. id=' + res._Context.reqId + ', Invalid POST request, req.body: ' + JSON.stringify(req.body))
			return res.status(404).end('--->execute: Req. id=' + res._Context.reqId + ', Invalid POST request, req.body: ' + JSON.stringify(req.body))
		}
		res._Context.cmd 	= req.body.cmd
		res._Context.args 	= req.body.args || []
		res._Context.options= req.body.options || {}
	}
	else if(req.method.toUpperCase() === 'GET'){
		if(!req.query || !req.query.cmd){
			logger.error('--->execute: Req. id=' + res._Context.reqId + ', Invalid GET request, req.query: ' + JSON.stringify(req.query))
			return res.status(404).end('--->execute: Req. id=' + res._Context.reqId + ', Invalid GET request, req.query: ' + JSON.stringify(req.query))
		}
		res._Context.cmd 	= req.query.cmd
		res._Context.args 	= req.query.args ? utils.parseJSON(req.query.args) : []
		res._Context.options= req.query.options ? utils.parseJSON(req.query.options) : {}
	}

	logger.debug('--->execute: Req. id=' + res._Context.reqId + ', res._Context: ' + JSON.stringify(res._Context))

	exports.executeService(res, function(err, result){
		if(err){
			logger.error('--->execute: Req.Id=' + res._Context.reqId + ', Execute Service failed, Reason: ' + err.message + 
				', Result: ' + JSON.stringify(result))
			return res.status(404).end('--->execute: Req.Id=' + res._Context.reqId + ', Execute Service failed, Reason: ' + err.message + 
									   ', Result: ' + JSON.stringify(result))
		}
		logger.info('--->execute: Req.Id=' + res._Context.reqId + '. Result: ' + JSON.stringify(result))
		res.end('--->execute: Req. id=' + res._Context.reqId + '. over. Result: ' + JSON.stringify(result))
	}) 
}

/**
 * Service to spawn a process
 * 	
 * @param {Request} 	res Request Object
 *                      res._Context:{
 *                      	"cmd":		<process to execute,
 *                      	"args": 	[OPTIONAL] <Array of arguments>,
 *                      	"options":	[OPTIONAL] <hash of options>
 *                      }
 * NOTE:
 *	Spawn process, see https://github.com/moxystudio/node-cross-spawn
 * 	For options, see https://nodejs.org/docs/latest-v8.x/api/child_process.html#child_process_child_process_spawn_command_args_options
 * 
 * @param  {Function} 	callback 	callback(err, result)
 * 						In case of success:
 * 									err: 			null
 * 									result:{
 * 										'code':		code, should be 0,
 * 										'stdout':	Data collected on stdout,
 * 										'stderr':	Data collected on stderr
 * 									}
 * 						In case of err:
 * 									err: 			Error
 * 									result:{
 * 										'code':		code, != 0,
 * 										'stdout':	Data collected on stdout,
 * 										'stderr':	Data collected on stderr
 * 									}
 */
exports.executeService = function(res, callback){

	let stdout		= ''
	  , stderr		= ''
	  , cbCalled	= false
      , self        = this 

    if(!self._initialized) return callback(new Error(PLUGIN_NAME + ' plugin is not initialized. ' + (initError ? initError.message:'')))  // 4.0.0
     
	// Spawn given command
	//const child = spawn('npm', ['list', '-g', '-depth', '0'], { stdio: 'inherit' });
	const child = spawn(res._Context.cmd, res._Context.args, res._Context.options)
    
    // V3.0.2 Begin
    //if(!res._Context.options || !res._Context.options.stdio){
    if(res._Context.options && res._Context.options.stdio && res._Context.options.stdio.length >=2 && res._Context.options.stdio[1] && Number.isFinite(res._Context.options.stdio[1])){
        //console.log('stdout goes to file')
        // File descriptor for stdout passed so donot capture the stdout here.
    }
    else {
        //console.log('stdout: handler')
        child.stdout.on('data', (data) => {
            console.log('stdout: ' + data.toString())
            stdout += data.toString()
        })
    }
    // V3.0.2 End

	child.stderr.on('data', (data) => {
        //console.log('stderr: ' + data.toString())
		stderr += data.toString()
	})

	child.on('close', (code) => {
		if(!cbCalled){
			cbCalled = true
			if(code !== 0) return callback(new Error('Error. Exit code: ' + code), {'code':code, 'stdout':stdout,'stderr':stderr})  // V1.0.2
			return callback(null,{'code':code,'stdout':stdout,'stderr':stderr})
		}
	})

	child.on('error', (err) => {
        // console.log(err)
		if(!cbCalled){
			cbCalled = true
			return callback(err)
		}
	})
}

/************ OPTIONAL PUBLIC FUNCTIONS ***********/
/**
 * Set Log Level
 * @param {String} logLevel Log Level. values:DEBUG|INFO|WARN|ERROR 
 */
exports.setLogLevel = function(logLevel){
	pluginConf.log.level = logLevel || 'INFO'
	utils.log4jsSetLogLevel(logger, pluginConf.log.level)
}

/**
 * Get Log Level
 * @return {String} Log Level. values:DEBUG|INFO|WARN|ERROR
 */
exports.getLogLevel = function(){
	return pluginConf.log.level
}

/**
 * Enable the given route
 * @param  {Request} 	req
 *					  req.body = {
 *					  	"module":"sample", 
 *					  	"route": { 
 *					  		"path": "/path/to/the/service",
 *					  		"method": 	Optional. Values are "get|post|dw|queue",
 *					  		"service": 	"version"
 *					  	}
 *					  }
 * @param  {function} 	callback callback(err)
 */
exports.enableRoute = function(req, callback){

	let self = this

	if(logger.isDebugEnabled()) logger.debug('--->enableRoute: Route=' + JSON.stringify(req.body.route))

	if(!req.body.route) return callback(new Error('route parameter not given'))

	req.body.route.method = req.body.route.method || 'get'

	// Enable the route
	switch(req.body.route.method.toLowerCase()){
		case 'dir': // Enable Dir Route
			pluginManager.callPluginService({name:'oxsnps-dirwatcher',service:'addListenersService'}, [req.body.route], callback)
			break
		case 'queue': // Enable Queue Route
	   		// Delete Queued Jobs before enabling route 
			pluginManager.callPluginService({name:'oxsnps-ibmmq', service:'addListenersService'}, [req.body.route], callback)
			break
		default:
			// Enable HTTP Route
			expressAppUtils.enableRoute(self, req.body.route, callback)
	}
} 

/**
 * Disable the given route
 * @param  {Request} 	request Request json has following structure.
 * 						req.body = {
 *					  	"module":"sample", 
 *					  	"route": { 
 *					  		"path": "/path/to/the/service",
 *					  		"method": 	Optional. Values are "get|post|dw|queue"
 * 						  	}
 * 						}
 * @param  {function} 	callback  callback(err)
 */
exports.disableRoute = function(req, callback){

	let self    = this  // V2.0.0

	if(logger.isDebugEnabled()) logger.debug('--->disableRoute: Route=' + JSON.stringify(req.body.route))
	
	if(!req.body.route) return callback(new Error('route parameter not given'))

	req.body.route.method = req.body.route.method || 'get'

	// Disable the route
	switch(req.body.route.method.toLowerCase()){
		// Disable Dir Route
		case 'dir': 
			pluginManager.callPluginService({name:'oxsnps-dirwatcher',service:'removeListenersService'}, [req.body.route], callback)
			break
		case 'queue':
			// Disable Queue Route
			pluginManager.callPluginService({name:'oxsnps-ibmmq', service:'removeListenersService'}, [req.body.route], callback)
			break
		default:
			//logger.warn('--->disableRoute: Wrong parameters(unknown route.method)! Not able to disable the route, Parms: ' + JSON.stringify(req.body))
			//callback(new Error('Wrong parameters(unknown route.method)'))
			// Disable HTTP Route
			expressAppUtils.disableRoute(self, req.body.route, callback)    // V2.0.0
	}
} 

/**************** PRIVATE FUNCTIONS ***************/ 