/**-----------------------------------------------------------------------------
 * oxsnps.js: 	
 *
 * Author    :  AFP2web Team
 * Copyright :  (C) 2019 by Maas Holding GmbH
 * Email     :  support@maas.de
 * 
 * History
 * V2.0.10	20.01.2025 	OXS-16546: Extend to write qpdf tool stdout content optionally to a file passed as HTTP request parameter. 
 * V2.0.9	13.12.2024 	OTS-16158: Added a function to extract attchments from a PDF file
 * V2.0.8	26.09.2024 	OTS-16158: Added a function to extract attchments from a PDF file
 * V2.0.7	20.09.2024	OTS-16158: Added a function to extract attchments from a PDF file
 *                      Integrated V3.0.5 changes
 * V2.0.6   07.11.2022  OTS-3317 oxs-server: Deploy stateless oxs-servers in Cloud Run 
 * V2.0.5   12.01.2021  Extend to get default retry options from env.variable 
 * V2.0.4   11.09.2020  OTS-2812: Extend to pass retry options when sending requests to oxs-qpdf-server
 * V2.0.2   08.06.2020  OTS-2804: Extend to process stdout/stderr messages of qpdf call to certain error/warning condition
 * V2.0.1   ???
 * V2.0.0   24.02.2020  OTS-2699: Extend base and custom import/export plugins to use oxsnps-core@4.1.x
 * V1.0.1   10.04.2020		OTS-2727: Extended with pageCountService()
 * V1.0.0   04.10.2019  Initial release
 *----------------------------------------------------------------------------*/
'use strict'

/**
 * Variable declaration
 * @private
 */
var packageJson		= require(__dirname + '/package.json')
  , PLUGIN_NAME		= packageJson.name
  , PLUGIN_VERSION	= packageJson.version // Get Server Version from package.json
  , async			= require('async')
  , fs				= require('fs')
  , log4js			= require('log4js')
  , os				= require('os')
  , path			= require('path')
  , FILE_SEP 		= path.sep

  , npsServer		= require('oxsnps-core/server')
  , npsConf 		= npsServer.npsConf
  , npsLogDir 		= npsConf.logDir

  , pluginConf		= require(__dirname + '/conf/' + PLUGIN_NAME + '.js')

  , dateFormat 		= require('oxsnps-core/helpers/date_format')
  , utils 			= require('oxsnps-core/helpers/utils')
  , httpUtils  		= require('oxsnps-core/helpers/httpUtils')
  , expressAppUtils = require('oxsnps-core/helpers/expressAppUtils')
  , SERVER_NAME     = "oxs-qpdf-server"     // V2.0.4

// Set plugin name, version and longDesc in pluginConf
pluginConf.module 	= PLUGIN_NAME 
pluginConf.version 	= PLUGIN_VERSION 
pluginConf.longDesc = packageJson.description
pluginConf.log 		= pluginConf.log || {}

// Export the Plugin's Configuration Express App
exports.pluginConf 	= pluginConf
exports.router      = undefined     // V2.0.0

// Get the Logger
let logger = log4js.getLogger(PLUGIN_NAME)
utils.log4jsSetLogLevel(logger, (pluginConf.log.level || 'INFO'))

/**************** PUBLIC FUNCTIONS ****************/
/**
 * Initialize Plugin
 * @param  {function} 	callback  callback(err, routes)
 */
exports.initialize = function(callback){

	let self = this
	  , appPluginConf	= (npsConf.plugins && npsConf.plugins[PLUGIN_NAME] ? npsConf.plugins[PLUGIN_NAME] : {})	
	  
	if(logger.isDebugEnabled()) logger.debug('--->initializing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION )

	if(self._initialized) return callback()

	// Merge application specific configuration with plugin configuration
	pluginConf = utils.mergeJSON(pluginConf, appPluginConf)

	/**
	 * Process pluginConf.qpdf value
	 *
	 * 	"qpdf":{
	 *		"server":{
	 *			// oxs-qpdf-sever Server
	 *			"hostname":	"localhost",
	 *			// Port
	 *			"port": 80,
	 *			// context will be prepended to the URL as http://<server>:<port>/<context>/....
	 *			"context": "qpdf",
	 *			// Request Timeout in milliseconds. Default is 10000 milli seconds
	 *			"httptimeout": 30000
	 *		},
	 *		'uris': {
	 *			<API>:	<service URI in qpdf server>
	 *		}
	 *	}
	 */
	pluginConf.qpdf	= pluginConf.qpdf || {}

	// If pluginConf.qpdf.server is a string, parse it (i.e. "{\"hostname\":\"localhost\", \"port\":80, \"httptimeout\":10000,  \"context\":\"qpdf\"}",)
	// Envar: <OXSNPS_SERVER_PREFIX> + "QPDF_SERVER"
	if(typeof pluginConf.qpdf.server === 'string'){
		let envvar = pluginConf.qpdf.server
		if(!process.env[envvar]) return callback(new Error('Missing environment variable: ' + envvar))

		pluginConf.qpdf.server = utils.parseJSON(process.env[envvar])
		if(pluginConf.qpdf.server === null) return callback(new Error('Unable to parse environment variable: ' + 
										envvar + ', value: ' + process.env[envvar]))
	} 

	pluginConf.qpdf.server = pluginConf.qpdf.server || {}

	// Set hostname
	pluginConf.qpdf.server.hostname = pluginConf.qpdf.server.hostname || 'localhost'

	// Set Port
	pluginConf.qpdf.server.port = pluginConf.qpdf.server.port || 80

	// Request Timeout in milliseconds, default is 10000
	pluginConf.qpdf.server.httptimeout = pluginConf.qpdf.server.httptimeout || 10000

	// Ensure that context is having leading slash
	let context = pluginConf.qpdf.server.context || ''
	if(context.length > 0 && !context.startsWith('/')) context = '/' + context
	pluginConf.qpdf.server.context = context

	pluginConf.qpdf.uris			= pluginConf.qpdf.uris || {}	
	pluginConf.qpdf.uris.process	= pluginConf.qpdf.uris.process || '/services/qpdf/process'
    
	// V3.0.5  // 2.0.7
    if(typeof pluginConf.qpdf.options === 'string'){
        envvar = pluginConf.qpdf.options
        if(!process.env[envvar]) return callback(new Error('Missing environment variable: ' + envvar))

        pluginConf.qpdf.options = utils.parseJSON(process.env[envvar])
        if(pluginConf.qpdf.options === null) return callback(new Error('Unable to parse environment variable: ' + envvar + ', value: ' + process.env[envvar]))
    } 
    pluginConf.qpdf.options                     = pluginConf.qpdf.options                   || {}
    pluginConf.qpdf.options.listAttachments     = pluginConf.qpdf.options.listAttachments   || ['--list-attachments']	// do not specify verbose since we parse stdout to get attachment names
    pluginConf.qpdf.options.extractAttachments  = pluginConf.qpdf.options.extractAttachments|| []						// --show-attachment=<filename> will be added at runtime
	// V3.0.5 // 2.0.7

    exports.pluginConf = pluginConf   // V2.0.0    

	// Setup router for the plugin
	expressAppUtils.setupRouter(self, function(err, router){    // V2.0.0
        if(err)	return callback(err)
        exports.router = router     // V2.0.0
		self._initialized = true
		logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' initialized')
		callback(null, pluginConf.routes)
	})
}

/**
 * Activate Plugin. This function is called from pluginManager once all the enabled plugins have been initialize
 * @param  {function} 	callback  callback(err)
 */
exports.postInitialize = function(callback){
	callback()
}

/**
 * Finalize Plugin
 * @param  {function} 	callback callback(err) 
 */
exports.finalize = function(callback){

	let self = this
	  , req  = {}

	if(logger.isDebugEnabled()) logger.debug('--->finalizing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION )
	if(!self._initialized) return callback()

	async.series([
		// Disable all routes
		function(nextTask){
			req.body = {'module': PLUGIN_NAME}
			async.each(pluginConf.routes, 
				function(route, next){
					req.body.route = route
					exports.disableRoute(req, next)
				},
				function(err){
					return nextTask(err)
				}
			)
		}
	],
	function(err){
		if(err) return callback(err)
		self._initialized = false
		logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' finalized')
		callback()
	})

}

/**
 * Return Plugin Version
 * @param  {function} 	callback callback(err,version)
 */
exports.getVersion = function(callback){
	return callback(null, PLUGIN_VERSION)
}

/**
 * Process a HTTP Version Request
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.version = function(req, res){

	let data = undefined

	// Add Context to the response instance
	res._Context = {}

	// Add the Current Date to the Context
	res._Context.date = new Date()

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date)
	logger.info('--->version: Req. Id=' + res._Context.reqId)

	data = '<h3>' +  PLUGIN_NAME + ' v' + PLUGIN_VERSION + '</h3>'
	res.end(data)

	logger.info('--->version: Req. Id=' + res._Context.reqId + ', sent back ' +
				PLUGIN_NAME + ' v' + PLUGIN_VERSION + ', over')
	return
}

/**
 * Process a HTTP Configuration Request 
 * 	Either as HTTP GET request:
 * 	 	GET http://localhost:1029/services/<plugin>/conf 							--> To retrieve the configuration
 * 	 	GET http://localhost:1029/services/<plugin>/conf?save=true&conf="{...}" 	--> To pass and save the configuration
 * 	or as HTTP POST request:
 * 	 	POST http://localhost:1029/services/<plugin>/conf 						--> To pass and save the configuration
 * 	 	req.body={
 * 	 		"conf": {...}
 * 	 	}
 * 	
 * @param {Request} 	req Request Object
 *                      req.query:{ // for HTTP GET request:
 *                      	"save":"false",
 *                      	"conf":"{<IMPORTANT: COMPLETE JSON Conf. as String>}"
 *                      }
 *                      req.body:{ // for HTTP POST request:
 *                      	"conf":{<IMPORTANT: COMPLETE JSON Conf. as JSON Object>}
 *                      }
 * @param {Response}    res Response Object
 */
exports.conf = function(req, res){

	let reqType 	= 'getconf'
	  , jsonConf 	= undefined
	  
	// Add Context to the response instance
	res._Context = {}

	// Add the Current Date to the Context
	res._Context.date = new Date()

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date)
	logger.info('--->conf: Req. id=' + res._Context.reqId)
	
	if(req.method.toUpperCase() === 'POST'){
		logger.info('--->conf: body=' + JSON.stringify(req.body))
		if(!req.body || !req.body.conf) {
			logger.info('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body))
			return res.end('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body))
		}
		reqType = 'saveconf' 		// save request
		jsonConf = req.body.conf 	// set the json conf to be passed to saveConfService
	}
	else if(req.method.toUpperCase() === 'GET'){
		logger.info('--->conf: query=' + JSON.stringify(req.query))
		if(req.query && req.query.save && req.query.save.toLowerCase() === 'true' && req.query.conf)
		{
			reqType = 'saveconf'
			jsonConf = utils.parseJSON(unescape(req.query.conf)) // remove escaped chars and set the json conf to be passed to saveConfService
		} 
	}

	switch(reqType){
		case 'saveconf':
		    exports.saveConfService(jsonConf, function(err){
		    	logger.info('--->conf: Req. id=' + res._Context.reqId + ', over')
				res.json(err ? {'status':'error', 'message': err.message} : pluginConf)
		    }) 
			break
		default:
		case 'getconf':
			logger.info('--->conf: Req. id=' + res._Context.reqId + ', over')
			res.json(pluginConf)
			break
	}
}

/**
 * Store  the Plugin conf to its configuration file (i.e. .../<plugin>/conf/<plugin>.js)
 * @param  {JSON}   	conf     Plugin Configuration
 * @param  {Function} 	callback
 */
exports.saveConfService = function(jsonConf, callback){

	let filename 	= undefined
	  , data 		= undefined
	  
	// PluginManager calls saveConfService without passing the conf
	if(typeof jsonConf === 'function'){
		callback = jsonConf
		jsonConf = pluginConf
	}

	// Assert jsonConf
	if(!jsonConf || jsonConf === null) return callback(new Error('--->saveConfService: empty Configuration'))
	if(logger.isDebugEnabled()) logger.debug('--->saveConfService: conf:' + JSON.stringify(jsonConf))

	// Backup the current plugin conf.in the server's backup dir and store the new one in the plugin's /conf dir
	async.series([
		// Task 1: Backup the current plugin conf.
		function(nextTask){
			filename = npsConf.backupDir + FILE_SEP + dateFormat.asString('yyyy-MM-dd_hh-mm-ss', new Date()) + '_configurationOf_' + PLUGIN_NAME + '.js' // build the Backup filename
			if(logger.isDebugEnabled()) logger.debug('--->saveConfService: Backuping plugin configuration to ' + filename)
			data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t') // build the plugin conf content

			// Write data to the Plugin configuration file
			fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
				if(err) logger.error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message) // log error & continue
				nextTask()
			})
		},
		// Task 2: Save the new plugin conf.
		function(nextTask){
			pluginConf = jsonConf // we should have here a proper conf, so overwrite the current pluginConf file
			filename = path.resolve(__dirname + '/conf/' + PLUGIN_NAME + '.js') // build the plugin configuration filename
			if(logger.isDebugEnabled()) logger.debug('--->saveConfService: Writing plugin configuration to ' + filename)
			data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t') // build the plugin conf content

			// Write data to Plugin configuration file
			fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
				if(err) nextTask(new Error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message))
				else nextTask()
			}) 
		}
	],
	function(err){
		callback(err)
	})
}

/**
 * Process given PDF document
 * @param	{Response}	res			Response Object
 *									res._Context:{
 *										'reqId':	<Request id>,
 *                                      "retryOptions":{
 *                                          "default":{
 *									            "interval": 10000,      // 10 secs  <interval in milliseconds>,
 *									            "count": 5,		                  
 *									            "errorCodes" : [502]   // <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          },
 *                                          "oxs-qpdf-sever":{  // <Optional>
 *									            "interval": 10000,      // 10 secs  <interval in milliseconds>,
 *								            	"count": 5,		                  
 *								            	"errorCodes" : [502]   // <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          }
 *							            },
 *										'qpdf':{ 
 *											'args':[
 * 												// Reference: http://qpdf.sourceforge.net/files/qpdf-manual.html#ref.using
 * 												<command line options of qpdf>
 *											]
 *											'processRules': [
 *												{
 *													'value':	<eyecatcher string to be searched in stderr/stdout>,
 *													'action':	<error | warn, If not specified rule will be ignored>
 *												}
 *											]
 *										}
 *									}
 * @param	{function}	callback	callback(err,result)
 */
exports.processService = function(res, callback){

    let options         = undefined
       , retryOptions   = undefined // V2.0.4    

	//V2.0.6 Begin 
/*	options	= {
		"hostname":		pluginConf.qpdf.server.hostname,
		"port": 		pluginConf.qpdf.server.port,
		"path": 		pluginConf.qpdf.server.context + pluginConf.qpdf.uris.process,	
		"httptimeout":	pluginConf.qpdf.server.httptimeout
	}
*/
	options	= 	utils.clone(pluginConf.qpdf.server)
	options.path = options.context + pluginConf.qpdf.uris.process	
	//V2.0.6 End 
	
    // V2.0.4 Begin
    options.reqId = res._Context.reqId
    if(res._Context.retryOptions && res._Context.retryOptions[SERVER_NAME]) retryOptions = res._Context.retryOptions[SERVER_NAME]
    else if(res._Context.retryOptions && res._Context.retryOptions['default']) retryOptions = res._Context.retryOptions['default']
    else retryOptions = pluginConf.qpdf.server.retryOptions     // V2.0.5
    // V2.0.4 End

	logger.info('--->processService: Req. id=' + res._Context.reqId + ', options: ' + JSON.stringify(options) + ', res._Context.qpdf: ' + JSON.stringify(res._Context.qpdf) + ', retryOptions: ' + JSON.stringify(retryOptions))
	httpUtils.postURL(options, res._Context.qpdf, retryOptions, function(err, result){  // V2.0.4
        if(err) return callback(new Error(err.message + ', options: ' + JSON.stringify(options) + ', Parms:' + JSON.stringify(res._Context.qpdf)))   // http status !== 200

        // if not an error (http status === 200) result obj must exists
        result = result ? utils.parseJSON(result) : result
        if(!result) return callback(new Error('oxsnps-qpdf Error: Result is null or result json is not parsable, Result: ' + JSON.stringify(result) + ', Parms:' + JSON.stringify(res._Context.qpdf)))                

        // V202 Begin
        // Process stdout/stderr messages to certain error/warnings
        if (res._Context.qpdf.processRules && Array.isArray(res._Context.qpdf.processRules) && res._Context.qpdf.processRules.length > 0){
			res._Context.qpdf.result = result

			_processMessages(res, function(err){
				if (err) return callback(err)
				logger.info('--->processService: Req. id=' + res._Context.reqId + ' is over, Result: ' + JSON.stringify(result))
				return callback(null, result)
			})
		}
		else {
			logger.info('--->processService: Req. id=' + res._Context.reqId + ' is over, Result: ' + JSON.stringify(result))
			return callback(null, result)
		}
        // V202 End

	})
}

/**
 * Process given PDF document
 * @param	{Response}	res			Response Object
 *									res._Context:{
 *										'qpdf':{ 
 *											'args':[
 * 												// Reference: http://qpdf.sourceforge.net/files/qpdf-manual.html#ref.using
 * 												<command line options of qpdf>
 *											]
 *										}
 *									}
 * @param	{function}	callback	callback(err,result)
 */
exports.pageCountService = function(res, callback){
    
    let pageCount   = undefined 

	exports.processService(res, function(err, result){
        if(err) return callback(err)

        if(result.code !== 0) return callback(new Error('oxsnps-qpdf Error Code: ' + result.code  + ', Result: ' + JSON.stringify(result) + ', Parms:' + JSON.stringify(res._Context.qpdf)), result)

        if(!result.stdout) return callback(new Error('oxsnps-qpdf does not return "stdout" in result. Result: ' + JSON.stringify(result) + ', Parms:' + JSON.stringify(res._Context.qpdf)), result)

        if(logger.isDebugEnabled()) logger.debug('--->pageCountService: Req. id=' + res._Context.reqId + ', Result: ' + JSON.stringify(result))

        /*  result = {"code":0,"stdout":"16\n","stderr":""} */
        result.stdout = result.stdout.trim().replace(/\r?\n/, '')  // remove new line chars
          
        if(isNaN(result.stdout)) return callback(new Error('oxsnps-qpdf does not return page count string in "stdout" in result. Result: ' + JSON.stringify(result) + ', Parms:' + JSON.stringify(res._Context.qpdf)), result)

        pageCount = parseInt(result.stdout)
        if(logger.isDebugEnabled()) logger.debug('--->pageCountService: Req. id=' + res._Context.reqId + ', PageCount: ' +  pageCount)
        return callback(null, pageCount)
	})
}

// V3.0.5 Begin	// 2.0.7
/**
 * Get Attachment List
 * @param	{Response}	res			Response Object
 *									req.body:{
 *                                      "retryOptions":{
 *                                          "default":{
 *									            "interval": 10000,      // 10 secs  <interval in milliseconds>,
 *									            "count": 5,		                  
 *									            "errorCodes" : [502]   // <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          },
 *                                          "oxs-qpdf-sever":{  // <Optional>
 *									            "interval": 10000,      // 10 secs  <interval in milliseconds>,
 *								            	"count": 5,		                  
 *								            	"errorCodes" : [502]   // <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          }
 *							            },
 *										'qpdf.input':{ 
 *											'filename': <FQ input filename>,
 *											'addlOptions': []		// addl qpdf options 
 *										},
 *									}
 * @param	{function}	callback	callback(err,result)
 * result:{
 *      "code":0,
 *      "stdout":"xxxxxx",
 *      "stderr":"",
 *      "attachmentList": []        => file attachments in the PDF
 * }
 * 
 */
 exports.listAttachments = function(req, res){	// not tested incomplete

    let lp = undefined

	// Add Context to the response instance
	res._Context = {};

	// Add a Request Id to the Context
    res._Context.reqId = utils.buildReqId(new Date())
    
    lp = '--->listAttachments: Req. id=' + res._Context.reqId + ', '
    logger.info(lp + 'Query Parms: ' + JSON.stringify(req.query) + ', Path Parms: ' + JSON.stringify(req.params) + ', Req.body: ' + JSON.stringify(req.body) + (req.headers['content-type'] ? ', Content-type: ' + req.headers['content-type'] :'') )

    res._Context.qpdf = req.body

	logger.debug(lp + 'Parms: ' + JSON.stringify(res._Context.qpdf))

	// Call the ping service
	exports.listAttachmentsService(res, function(err, result){
		if(err){
			logger.error(lp + 'Reason: ' + err.message + ', Parms: ' + JSON.stringify(res._Context.qpdf))
			return res.status(404).end(lp + 'Reason: ' + err.message)
        }			
		logger.info(lp + 'over!. Result: '+ JSON.stringify(result))
		return res.send(result)
	})
}

/**
 * Get Attachment List
 * @param	{Response}	res			Response Object
 *									res._Context:{
 *										'reqId':	<Request id>,
 *                                      "retryOptions":{
 *                                          "default":{
 *									            "interval": 10000,      // 10 secs  <interval in milliseconds>,
 *									            "count": 5,		                  
 *									            "errorCodes" : [502]   // <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          },
 *                                          "oxs-qpdf-sever":{  // <Optional>
 *									            "interval": 10000,      // 10 secs  <interval in milliseconds>,
 *								            	"count": 5,		                  
 *								            	"errorCodes" : [502]   // <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          }
 *							            },
 *										'qpdf.input':{ 
 *											'filename': <FQ input filename>,
 *											'pattern':  [Array of pattern that specify files to extract]. Default is all.
 *											'addlOptions': []		// addl qpdf options 
 *										},
 *									}
 * @param	{function}	callback	callback(err,result)
 * result:{
 *      "code":0,
 *      "stdout":"xxxxxx",
 *      "stderr":"",
 *      "attachmentList": []        => file attachments in the PDF
 * }
 * 
 */
exports.listAttachmentsService = function(res, callback){

    let lp      = '--->listAttachmentsService: Req. id=' + res._Context.reqId + ', '
      , resTmp  = undefined
      , options = []

    if(!res._Context.qpdf.input || !res._Context.qpdf.input.filename) return callback(new Error(lp + 'res._Context.qpdf.input.filename parameter is missing'))
 
    // set qpdf args
    options.push('qpdf')

    if(pluginConf.qpdf.options.listAttachments && pluginConf.qpdf.options.listAttachments.length > 0) options = options.concat(pluginConf.qpdf.options.listAttachments)
    else options.push("--listAttachments") 

    if(res._Context.qpdf.input.addlOptions && res._Context.qpdf.input.addlOptions.length > 0) options = options.concat(res._Context.qpdf.input.addlOptions)

    // Add input filename
    options.push("'" + res._Context.qpdf.input.filename + "'" )    // surrounding "'" added to avoid erros with filenames having spaces

    resTmp = { 
        '_Context': {
            'reqId':res._Context.reqId,
            'qpdf': { "args": options}
        }
    }

    if(logger.isDebugEnabled()) logger.debug(lp + 'Calling processService() to get attachment list, parms:' + JSON.stringify(resTmp._Context.qpdf))
    exports.processService(resTmp, function(err, result){
        /*  result = {
            "code":0,
            "stdout":"200.000.000.576.337.85_abrechnung.ps7 -> 81,0
                    200.000.000.576.337.85_quittung.ps7 -> 82,0
                    200.000.000.576.337.85_verordnung.ps7 -> 83,0"
            "stderr":""}
        */
        if(err) return callback(err)    // params are alread included in err
        
        if(!result) return callback(new Error('oxsnps-qpdf result is undefiend when getting list of attachments, Parms: ' + JSON.stringify(resTmp._Context.qpdf)))

        if(logger.isDebugEnabled()) logger.debug(lp + 'Result: ' + JSON.stringify(result) )
        if(result.code !== 0) return callback(new Error('oxsnps-qpdf error when getting list of attachments, Code: ' + result.code  + ', Result: ' + JSON.stringify(result) + ', Parms:' + JSON.stringify(resTmp._Context.qpdf)))

        if(result.stdout === undefined) return callback(new Error('oxsnps-qpdf does not return "stdout" in result when getting list of attachments. Result: ' + JSON.stringify(result) + ', Parms:' + JSON.stringify(resTmp._Context.qpdf)))
        res._Context.qpdf.result = result 

        result.attachmentList = _extractFilenames(res)  // Extract filenames from stdout

        if(logger.isDebugEnabled()) logger.debug(lp + 'Attachment list:' + JSON.stringify(result.attachmentList))

        return callback(null, result)
    })
}

/**
 * Extract attachment
 * @param	{Response}	res			Response Object
 *									req.body:{
 *                                      "retryOptions":{
 *                                          "default":{
 *									            "interval": 10000,      // 10 secs  <interval in milliseconds>,
 *									            "count": 5,		                  
 *									            "errorCodes" : [502]   // <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          },
 *                                          "oxs-qpdf-sever":{  // <Optional>
 *									            "interval": 10000,      // 10 secs  <interval in milliseconds>,
 *								            	"count": 5,		                  
 *								            	"errorCodes" : [502]   // <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          }
 *							            },
 *										'qpdf.input':{ 
 *											'filename': <FQ input filename>,
 *											'attachmentName': <Attachment filename as in input PDF>,
  *											'addlOptions': []		// addl qpdf options 
 *										},
 *										'qpdf.output': {}
 *											'path':	<FQ Path>
 *										}
 *									}
 * @param	{function}	callback	callback(err,result)
 * result:{
 *      "code":0,
 *      "stdout":"xxxxxx",
 *      "stderr":"",
 *      extractedAttachment: <FQ filename of extracted attachment>
 * }
  * 
 */
exports.extractAttachment = function(req, res){	// not tested incomplete
    
    let lp = undefined

	// Add Context to the response instance
	res._Context = {};

	// Add a Request Id to the Context
    res._Context.reqId = utils.buildReqId(new Date())
    
    lp = '--->extractAttachment: Req. id=' + res._Context.reqId + ', '
    logger.info(lp + 'Query Parms: ' + JSON.stringify(req.query) + ', Path Parms: ' + JSON.stringify(req.params) + ', Req.body: ' + JSON.stringify(req.body) + (req.headers['content-type'] ? ', Content-type: ' + req.headers['content-type'] :'') )

    res._Context.qpdf = req.body

	logger.debug(lp + 'Parms: ' + JSON.stringify(res._Context.qpdf))

	// Call the ping service
	exports.extractAttachmentService(res, function(err, result){
		if(err){
			logger.error(lp + 'Reason: ' + err.message + ', Parms: ' + JSON.stringify(res._Context.qpdf))
			return res.status(404).end(lp + 'Reason: ' + err.message)
        }			
		logger.info(lp + 'over!. Result: '+ JSON.stringify(result))
		return res.send(result)
	})
}

/**
 * Extract Attachment 
 * @param	{Response}	res			Response Object
 *									res._Context:{
 *										'reqId':	<Request id>,
 *                                      "retryOptions":{
 *                                          "default":{
 *									            "interval": 10000,      // 10 secs  <interval in milliseconds>,
 *									            "count": 5,		                  
 *									            "errorCodes" : [502]   // <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          },
 *                                          "oxs-qpdf-sever":{  // <Optional>
 *									            "interval": 10000,      // 10 secs  <interval in milliseconds>,
 *								            	"count": 5,		                  
 *								            	"errorCodes" : [502]   // <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          }
 *							            },
 *										'qpdf.input':{ 
 *											'filename': <FQ input filename>,
 *											'attachmentName': <Attachment filename as in input PDF>,
  *											'addlOptions': []		// addl qpdf options 
 *										},
 *										'qpdf.output': [
 *											'path':	<FQ Path>
 *										]
 *									}
 * @param	{function}	callback	callback(err,result)
 * result:{
 *      "code":0,
 *      "stdout":"xxxxxx",
 *      "stderr":"",
 *      extractedAttachment: <FQ filename of extracted attachment>
 * }
 * 
 */
 exports.extractAttachmentService = function(res, callback){

    let lp      = '--->extractAttachmentService: Req. id=' + res._Context.reqId + ', '
      , resTmp  = undefined
      , options = []
      , logRes  = undefined
      , attFile = undefined
      , buf     = undefined
      , fsWriteOptions = undefined
      , retValue    = undefined    

    if(!res._Context.qpdf.input || !res._Context.qpdf.input.filename) return callback(new Error(lp + 'res._Context.qpdf.input.filename parameter is missing'))
    if(!res._Context.qpdf.input || !res._Context.qpdf.input.attachmentName) return callback(new Error(lp + 'res._Context.qpdf.input.attachmentName parameter is missing'))
    if(!res._Context.qpdf.output || !res._Context.qpdf.output.path) return callback(new Error(lp + 'res._Context.qpdf.output.path parameter is missing'))

    retValue = _buildOutputPath(res)
    if(retValue instanceof Error) return callback(new Error(lp + retValue.message))

    fsWriteOptions = res._Context.qpdf.output.writeOptions ? utils.clone(res._Context.qpdf.output.writeOptions) : {}
	fsWriteOptions.encoding	= fsWriteOptions.encoding || "binary"
	fsWriteOptions.mode		= fsWriteOptions.mode     || parseInt('0777',8)

    // set qpdf args
    options.push('qpdf')

    if(pluginConf.qpdf.options.extractAttachments && pluginConf.qpdf.options.extractAttachments.length > 0) options = options.concat(pluginConf.qpdf.options.extractAttachments)

    // Add input filename
    options.push("--show-attachment=" + res._Context.qpdf.input.attachmentName + "" )    // surrounding "'" added to avoid erros with filenames having spaces

    if(res._Context.qpdf.input.addlOptions && res._Context.qpdf.input.addlOptions.length > 0) options = options.concat(res._Context.qpdf.input.addlOptions)
    options = options.concat(pluginConf.qpdf.options.extractAttachments)

    // Add input filename
    options.push("'" + res._Context.qpdf.input.filename + "'" )    // surrounding "'" added to avoid erros with filenames having spaces

	attFile = path.join(res._Context.qpdf.output.path, res._Context.qpdf.input.attachmentName)	// V2.0.10
    resTmp = { 
        '_Context': {
            'reqId':res._Context.reqId,
            'qpdf': { "args": options, "addDefaultArgs": false, "output": { "stdoutFilename": attFile}} // V2.0.10
        }
    }

	async.series([
		// Call pdf to extract an attachment
		function(nextTask){
            if(logger.isDebugEnabled()) logger.debug(lp + 'Calling processService() to extract attachment, parms:' + JSON.stringify(resTmp._Context.qpdf))
            exports.processService(resTmp, function(err, result){
                // result = {"code":0, "stdout":"xxxxx" "stderr":""}
                if(err) return nextTask(err)    // params are already included in err
                
                if(!result) return nextTask(new Error('oxsnps-qpdf result is undefiend when extracting an attachment, Parms: ' + JSON.stringify(resTmp._Context.qpdf)))
                logRes = {...result, stdout: ''}
                
                logRes.stdout = (result.stdout && result.stdout.length > 0) ? result.stdout.substring(0, 50) : ''
                logRes.stdout_length = (result.stdout && result.stdout.length > 0) ? result.stdout.length : 0
                if(logger.isDebugEnabled()) logger.debug(lp + 'Result: ' + JSON.stringify(logRes))

                if(result.code !== 0) return nextTask(new Error('oxsnps-qpdf error when extracting an attachment, Code: ' + result.code  + ', Result: ' + JSON.stringify(logRes) + ', Parms: ' + JSON.stringify(resTmp._Context.qpdf)))
        
                if(result.stdout === undefined) return nextTask(new Error('oxsnps-qpdf does not return "stdout" in result when extracting an attachment. Result: ' + JSON.stringify(logRes) + ', Parms:' + JSON.stringify(resTmp._Context.qpdf)))
                res._Context.qpdf.result = result
			    // V2.0.10
                res._Context.qpdf.result.extractedAttachment = {
                    'filename': attFile,
                    'simpleFilename': path.basename(attFile)
                }				
				// V2.0.10
                return nextTask()
            })    
		},
		// V2.0.10 Begin
		/*
		// Create output dir.
		function(nextTask){
            if(logger.isDebugEnabled()) logger.debug(lp + 'Check and create a directory: ' + res._Context.qpdf.output.path)
            utils.mkDir(res._Context.qpdf.output.path, parseInt('0777',8), function(err){
                if(err) return nextTask(new Error('Unable to a create directory: ' + res._Context.qpdf.output.path + ', ' + err.message))
                nextTask()
            })
		},        
		// Write to a file 
		function(nextTask){
            //buf = Buffer.from(res._Context.qpdf.result.stdout)
            //console.log("typeof(res._Context.qpdf.result.stdout)");   console.log(typeof(res._Context.qpdf.result.stdout))
            attFile = path.join(res._Context.qpdf.output.path, res._Context.qpdf.input.attachmentName)
            if(logger.isDebugEnabled()) logger.debug(lp + 'Writing the attachment: ' + res._Context.qpdf.input.attachmentName + ' to a file: ' + attFile + ', writeoptions: ' + JSON.stringify(fsWriteOptions))
            fs.writeFile(attFile, res._Context.qpdf.result.stdout, fsWriteOptions, (err) => {
                if(err) return nextTask(new Error('Unable to write the attachment: ' + res._Context.qpdf.input.attachmentName + ' to a file: ' + attFile + ', error: ' + err.message))
                if(logger.isDebugEnabled()) logger.debug(lp + 'Attachment: ' + res._Context.qpdf.input.attachmentName + ' written to a file: ' + attFile)
                res._Context.qpdf.result.extractedAttachment = {
                    'filename': attFile,
                    'simpleFilename': path.basename(attFile)
                }
                nextTask()
            })
		}
		*/
		// V2.0.10 End
	],
	function(err){
		callback(err, res._Context.qpdf.result)
	})        
}

/**
 * Get Attachment List
 * @param	{Response}	res			Response Object
 *									req.body:{
 *                                      "retryOptions":{
 *                                          "default":{
 *									            "interval": 10000,      // 10 secs  <interval in milliseconds>,
 *									            "count": 5,		                  
 *									            "errorCodes" : [502]   // <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          },
 *                                          "oxs-qpdf-sever":{  // <Optional>
 *									            "interval": 10000,      // 10 secs  <interval in milliseconds>,
 *								            	"count": 5,		                  
 *								            	"errorCodes" : [502]   // <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          }
 *							            },
 *										'qpdf.input':{ 
 *											'filename': <FQ input filename>,
 *											'pattern':  [Array of pattern that specify files to extract]. Default is all.
 *											'addlOptions': []		// addl qpdf options 
 *										},
 *										'qpdf.output': {
 *											'path':	<FQ Path>
 *										}
 *									}
 * @param	{function}	callback	callback(err,result)
 * result:{
 *      "code":0,
 *      "stdout":"xxxxxx",
 *      "stderr":"",
 *      "attachmentList": []        => file attachments in the PDF
 *      "extractedAttachments: []   => FQ filenams of extracted attachments from the PDF.  "extractedAttachments": [] => array is empty if specified attachment or no attachments in the PDF
 * }
 * result:{
    "attachmentList": [
        "200.000.000.576.337.85_abrechnung.ps7",
        "200.000.000.576.337.85_quittung.ps7",
        "200.000.000.576.337.85_verordnung.ps7"
    ],
    "extractedAttachments": [
        "/media/sf_Issues/base/qpdfclient/testdata/1/200.000.000.576.337.85_abrechnung.ps7"
    ]
}
 */
exports.extractAttachments = function(req, res){	// not tested incomplete

    let lp = undefined

	// Add Context to the response instance
	res._Context = {};

	// Add a Request Id to the Context
    res._Context.reqId = utils.buildReqId(new Date())
    lp = '--->extractAttachments: Req. id=' + res._Context.reqId + ', '
    logger.info(lp + 'Query Parms: ' + JSON.stringify(req.query) + ', Path Parms: ' + JSON.stringify(req.params) + ', Req.body: ' + JSON.stringify(req.body) + (req.headers['content-type'] ? ', Content-type: ' + req.headers['content-type'] :'') )

    res._Context.qpdf = req.body

	logger.debug(lp + 'Parms: ' + JSON.stringify(res._Context.qpdf))

	// Call the ping service
	exports.extractAttachmentsService(res, function(err, result){
		if(err){
			logger.error(lp + 'Reason: ' + err.message + ', Parms: ' + JSON.stringify(res._Context.qpdf))
			return res.status(404).end(lp + 'Reason: ' + err.message)
        }			
		logger.info(lp + 'over!. Result: '+ JSON.stringify(result))
		return res.send(result)
	})
}

/**
 * extractAttachments given PDF document
 * @param	{Response}	res			Response Object
 *									res._Context:{
 *										'reqId':	<Request id>,
 *                                      "retryOptions":{
 *                                          "default":{
 *									            "interval": 10000,      // 10 secs  <interval in milliseconds>,
 *									            "count": 5,		                  
 *									            "errorCodes" : [502]   // <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          },
 *                                          "oxs-qpdf-sever":{  // <Optional>
 *									            "interval": 10000,      // 10 secs  <interval in milliseconds>,
 *								            	"count": 5,		                  
 *								            	"errorCodes" : [502]   // <Optional>. [List of error codes for which retry should be done]. If not given retry will be done for all errors
 *                                          }
 *							            },
 *										'qpdf.input':{ 
 *											'filename': <FQ input filename>,
 *											'pattern':  [Array of pattern that specify files to extract]. Default is all.
 *											'addlOptions': []		// addl qpdf options 
 *										},
 *										'qpdf.output': {
 *											'path':	<FQ Path>
 *										}
 *									}
 * @param	{function}	callback	callback(err,result)
 * result:{
 *      "attachmentList": []        => file attachments in the PDF
 *      "extractedAttachments: []   => FQ filenams of extracted attachments from the PDF
 * }
 * 
        Error cases:
            1. input file not found => err 
            2. attachments exist but invalid reg.exp used to extract an attachment => error 
            
        Proper cases 
        3. no attachements in PDF => 	{"attachmentList": [], "extractedAttachments": []}
        4. attachments exist but specified attachment does not exist => {"attachmentList": ["factur-x.xml"], "extractedAttachments": []}
        5. attachments exist and specified attachment exist => {"attachmentList": ["factur-x.xml"], "extractedAttachments": ["/media/sf_Issues/base/qpdfclient/testdata/1/factur-x.xml"]}
        5. attachments exist and fetched all 
        {
            "attachmentList": [
                "200.000.000.576.337.85_abrechnung.ps7",
                "200.000.000.576.337.85_quittung.ps7",
                "200.000.000.576.337.85_verordnung.ps7"
            ],
            "extractedAttachments": [
                "/media/sf_Issues/base/qpdfclient/testdata/1/200.000.000.576.337.85_abrechnung.ps7",
                "/media/sf_Issues/base/qpdfclient/testdata/1/200.000.000.576.337.85_quittung.ps7",
                "/media/sf_Issues/base/qpdfclient/testdata/1/200.000.000.576.337.85_verordnung.ps7"
            ]
        }
 * 
 */
exports.extractAttachmentsService = function(res, callback){

    let self        = this 
      , retValue    = undefined
      , lp = '--->extractAttachmentsService: Req. id=' + res._Context.reqId + ', '

    if(!self._initialized) return callback(new Error(PLUGIN_NAME + ' plugin is not initialized. ' + (initError ? initError.message:'')))  // 3.0.0

    if(!res._Context.qpdf.input || !res._Context.qpdf.input.filename) return callback(new Error(lp + 'res._Context.qpdf.input.filename parameter is missing'))
    if(!res._Context.qpdf.output || !res._Context.qpdf.output.path) return callback(new Error(lp + 'res._Context.qpdf.output.path parameter is missing'))

    // Evaluate output path
    retValue = _buildOutputPath(res)
    if(retValue instanceof Error) return callback(new Error(lp + retValue.message))

    // Validate Patterns
    retValue = _buildRegEx(res)
    if(retValue instanceof Error) return callback(new Error(lp + retValue.message))

    retValue    = {'attachmentList': [], 'extractedAttachments': []}    
	async.series([
		// Get list of attachments
		function(nextTask){
            exports.listAttachmentsService(res, function(err, result){
                if(!err) retValue.attachmentList = res._Context.qpdf.attachmentList = result.attachmentList
                return nextTask(err)
            })
		},
		// extract filenames
		function(nextTask){
            if(!res._Context.qpdf.attachmentList || res._Context.qpdf.attachmentList.length <=0) return nextTask()
            _filterAndextractAttachments(res, function(err, extractedAttachments){
                if(!err) retValue.extractedAttachments = extractedAttachments
                return nextTask(err)
            })
		},
	],
	function(err){
        //if(err) logger.error(lp + 'Reason: ' + err.message)
		callback(err, retValue)
	})    
}
// 2.0.7 End
/************ OPTIONAL PUBLIC FUNCTIONS ***********/
/**
 * Set Log Level
 * @param {String} logLevel Log Level. values:DEBUG|INFO|WARN|ERROR 
 */
exports.setLogLevel = function(logLevel){
	pluginConf.log.level = logLevel || 'INFO'
	utils.log4jsSetLogLevel(logger, pluginConf.log.level)
}

/**
 * Get Log Level
 * @return {String} Log Level. values:DEBUG|INFO|WARN|ERROR
 */
exports.getLogLevel = function(){
	return pluginConf.log.level
}

/**
 * Enable the given route
 * @param  {Request} 	req
 *					  req.body = {
 *					  	"module":"sample", 
 *					  	"route": { 
 *					  		"path": "/path/to/the/service",
 *					  		"method": 	Optional. Values are "get|post|dw|queue",
 *					  		"service": 	"version"
 *					  	}
 *					  }
 * @param  {function} 	callback callback(err)
 */
exports.enableRoute = function(req, callback){

	let self = this

	if(logger.isDebugEnabled()) logger.debug('--->enableRoute: Route=' + JSON.stringify(req.body.route))

	if(!req.body.route) return callback(new Error('route parameter not given'))

	req.body.route.method = req.body.route.method || 'get'

	// Enable the route
	switch(req.body.route.method.toLowerCase()){
		case 'dir': // Enable Dir Route
			pluginManager.callPluginService({name:'oxsnps-dirwatcher',service:'addListenersService'}, [req.body.route], callback)
			break
		case 'queue': // Enable Queue Route
	   		// Delete Queued Jobs before enabling route 
			pluginManager.callPluginService({name:'oxsnps-ibmmq', service:'addListenersService'}, [req.body.route], callback)
			break
		default:
			// Enable HTTP Route
			expressAppUtils.enableRoute(self, req.body.route, callback)
	}
} 

/**
 * Disable the given route
 * @param  {Request} 	request Request json has following structure.
 * 						req.body = {
 *					  	"module":"sample", 
 *					  	"route": { 
 *					  		"path": "/path/to/the/service",
 *					  		"method": 	Optional. Values are "get|post|dw|queue"
 * 						  	}
 * 						}
 * @param  {function} 	callback  callback(err)
 */
exports.disableRoute = function(req, callback){

    let self    = this  // V2.0.0    

	if(logger.isDebugEnabled()) logger.debug('--->disableRoute: Route=' + JSON.stringify(req.body.route))
	
	if(!req.body.route) return callback(new Error('route parameter not given'))

	req.body.route.method = req.body.route.method || 'get'

	// Disable the route
	switch(req.body.route.method.toLowerCase()){
		// Disable Dir Route
		case 'dir': 
			pluginManager.callPluginService({name:'oxsnps-dirwatcher',service:'removeListenersService'}, [req.body.route], callback)
			break
		case 'queue':
			// Disable Queue Route
			pluginManager.callPluginService({name:'oxsnps-ibmmq', service:'removeListenersService'}, [req.body.route], callback)
			break
		default:
			//logger.warn('--->disableRoute: Wrong parameters(unknown route.method)! Not able to disable the route, Parms: ' + JSON.stringify(req.body))
			//callback(new Error('Wrong parameters(unknown route.method)'))
			// Disable HTTP Route
			expressAppUtils.disableRoute(self, req.body.route, callback)      // V2.0.0
	}
} 

/**************** PRIVATE FUNCTIONS ***************/
/**
 * Log a Warning if Service Response time exceeded maxresptime 
 * @param  {Response}	res  	Response Object
 * @param  {String}		service service
 * @private
 */
function _logServiceRespTime(res, service){

	let respTime 	= 0
	  , os 			= require('os')
	  

	// Check resp. time only if maxresptime is defined
   	if(pluginConf.maxresptime){
		respTime = utils.dateDiff('x', res._Context.date, new Date()) // 'x' means returns diff. in ms

   		if(respTime > pluginConf.maxresptime * 1000){
			logger.warn('--->' + service + ': Req.Id=' + res._Context.reqId + ', '
						+ ' Resp. Time: ' + respTime/1000 + 's, Max. allowed: '
						+ pluginConf.maxresptime + 's. Average Load: ' + os.loadavg())
   		}
   	}
}

// V202 Begin
/**
 * Process stdout/stderr messages to certain qpdf call has any error or warnings
 *
 * @param	{Response}	res			Response Object
 *									res._Context:{
 *										'reqId':	<Request id>,
 *										'qpdf':{ 
 *											'processRules': [
 *												{
 *													'value':	<eyecatcher string to be searched in stderr/stdout>,
 *													'action':	<error | warn, If not specified rule will be ignored>
 *												}
 *											]
 *										}
 *									}
 * @param	{function}	callback	callback(err)
 */
function _processMessages(res, callback){

    let msgs    = []    
      , result  = res._Context.qpdf.result
      , rules   = res._Context.qpdf.processRules
      , rule    = undefined
      , procErr = undefined
      , msg     = undefined
      , reRule  = undefined
      , action  = undefined

    // split stdout messages into single lines
    if(result.stdout && result.stdout.length > 0) msgs = result.stdout.toString().split(/\r?\n/) // split at newline characters
           
    // split stderr messages into single lines
    if(result.stderr && result.stderr.length > 0){
        msgs = msgs || []
        let stderrMsgs = result.stderr.toString().split(/\r?\n/) // split at newline characters
        msgs = msgs.concat(stderrMsgs)
    }

    // filter out empty lines
    msgs = msgs.filter(ln => ln.length > 0)
    if(msgs.length <= 0) return callback() // no messages to process

	// Iterate through configured rules
	for( let r = 0; r < rules.length && !procErr; r++){
		rule = rules[r]
		if (!rule) continue
		if (!rule.value) continue

		msg = undefined
		reRule = undefined
		action = undefined
		if(typeof(rule.value) === 'string'){
			reRule = new RegExp(rule.value.trim())
		}
		else if(rule.value instanceof RegExp){
			reRule = rule.value
		}

		// check whether messages array has any message that contains rule.value as substring
		msgs.some(m => {if(reRule.test(m)) msg = m})
		if(msg) action = rule.action
		if(!action) continue

		action = action.toLowerCase()
		if(action === 'warn' || action === 'warning') logger.warn('--->_processMessages: Req. id=' + res._Context.reqId  +', ' + msg)
		if(action === 'error') procErr = new Error('qpdf process failed. Exit code: ' + result.code + ', ' + msg)
	}
	return callback(procErr)
}
// V202 End

// 3.0.5 Begin	// 2.0.7
// Extract filenams from listattachment command output
function _extractFilenames(res){

    let lp      = '--->__extractFilenames: Req. id=' + res._Context.reqId + ', '
      , lines   = undefined
      , pattern = /^(?<filename>[\w\-.]+)\s*->\s*(\d+),(\d+)/
      , result  = undefined
      , list    = []

    if(!res._Context.qpdf.result.stdout || res._Context.qpdf.result.stdout.length <=0) return list
    lines = res._Context.qpdf.result.stdout.split(os.EOL)   

    if(logger.isDebugEnabled()) logger.debug(lp + 'Stdout lines:', lines)
    if(!lines || lines.length <=0) return list

    if(!Array.isArray(lines)) lines = [lines]

    lines.forEach(function(line){
        if(!line || line.trim().length <=0) return 
        line = line.trim()
        result = line.match(pattern)	
        if(logger.isDebugEnabled()) logger.debug(lp + 'line:' + line + ', pattern match result: ' +  JSON.stringify(result))
        if(result && result.groups && result.filename) list.push(result.groups.filename)
        else if(result && result.length >=2) list.push(result[1].trim())
    })
    return list
}

function _filterAndextractAttachments(res, callback){

    let lp = '--->_filterAndextractAttachments: Req. id=' + res._Context.reqId + ', '    
      , extractedAttachments    = []

    async.eachOfLimit(res._Context.qpdf.attachmentList, 1, function(attachment, index, next){
        let extract = true 
          , resTmp = undefined

        if(res._Context.qpdf.regex && res._Context.qpdf.regex.length >0){
            extract = res._Context.qpdf.regex.some(function(pattern){
                return pattern.test(attachment)
            })
        }

        if(!extract){
            if(logger.isDebugEnabled()) logger.debug(lp + 'Skipping extraction of attachment: ' + attachment + ', since it does not match any given pattern: ' + JSON.stringify(res._Context.qpdf.input.pattern))
            return next()
        }

        resTmp = {
            '_Context':{
                'reqId':     res._Context.reqId,
                'qpdf' : {
                    'input': utils.clone(res._Context.qpdf.input),
                    'output': res._Context.qpdf.output
                }
                
            }
        }        
        resTmp._Context.qpdf.input.attachmentName = attachment

        // Determine whether current page is empty or not
        if(logger.isDebugEnabled()) logger.debug(lp + 'Extracting attachment: ' + resTmp._Context.qpdf.input.attachmentName)
        exports.extractAttachmentService(resTmp, function(err, result){
            if(!err) extractedAttachments.push(result.extractedAttachment)
            return next(err)
        })
    },
    function(err){
        return callback(err, extractedAttachments)
    })
}

function _buildRegEx(res){

    let patErr = undefined

    res._Context.qpdf.regex = []

    if(!res._Context.qpdf.input.pattern || res._Context.qpdf.input.pattern.length <=0) return patErr

    if(!Array.isArray(res._Context.qpdf.input.pattern)) res._Context.qpdf.input.pattern = [res._Context.qpdf.input.pattern]
    res._Context.qpdf.input.pattern.forEach(function(item){
        if(patErr) return 
        try{
			if(item && item.pattern && item.pattern.length >0) res._Context.qpdf.regex.push(new RegExp(item.pattern, (item.flags|| 'i')))
            //res._Context.qpdf.regex.push(new RegExp(patternStr))
        }
        catch(err){
            patErr = new Error('Invalid regular expresson: ' + JSON.stringify(item) + ', ' + err.message)
        }
    })
    return patErr
}

function _buildOutputPath(res){
    let lp          = '--->_buildOutputPath: Req. id=' + res._Context.reqId + ', '
      , outputPath  = res._Context.qpdf.output.path
      , expr        = undefined
      , ctx         = res._Context  // may be used by the eval() so don't delete it

    if(!outputPath) return 

    if(typeof(outputPath) === 'string' && !outputPath.startsWith('eval:')) return 

    if (logger.isDebugEnabled()) logger.debug(lp + 'Evaluating output path: ' + outputPath + ' ...')
    expr = outputPath.substring(outputPath.indexOf(':') + 1)
    try {
        outputPath = eval(expr)
        if (logger.isDebugEnabled()) logger.debug(lp+ 'Evaluated output path: ' + outputPath)
        res._Context.qpdf.output.path = outputPath
    }
    catch(err){
        return new Error('Unable to evaluate output path from expression: ' + outputPath + ', Reason: ' + err.message)
    }
}
// 3.0.5 End	// 2.0.7