/**-----------------------------------------------------------------------------
 * oxsnps.js: 	ISH custom import plugin of the OXS.IMPORT server
 *
 * Author    :  AFP2web Team
 * Copyright :  (C) 2015 by Maas Holding GmbH
 * Email     :  support@oxseed.de
 * 
 * History
 * V2.0.0	 02.02.2022	OTS-3216: Added swagger to oxs-server
 * V1.0.6	 11.08.2021	OTS-3117: Extended to support REST ISH Stack with PDFs having more than one page. Removed the assertion that ensures input PDFs must have only one page
 * V1.0.5	 31.03.2021	OTS-2990: Fixed minor issues related to the Swagger UI
 * V1.0.0	 10.02.2021	OTS-2990: TFS-325296: ci.ish: Neue REST-Schnittstelle H5U für Inputmanagement
 *----------------------------------------------------------------------------*/
'use strict'

/**
 * Variable declaration
 * @private
 */
var packageJson 	= require('./package.json')
  , PLUGIN_NAME 	= packageJson.name
  , PLUGIN_VERSION  = packageJson.version // Get Server Version from package.json
  , async		  	= require('async')  
  , fs			    = require('fs')  
  , log4js		    = require('log4js')
  , path		    = require('path')

  , npsServer		= require('oxsnps-core/server')
  , npsConf 		= npsServer.npsConf  
  , pluginConf		= require('./conf/' + PLUGIN_NAME + '.js')
  , dateFormat 		= require('oxsnps-core/helpers/date_format')
  , utils 			= require('oxsnps-core/helpers/utils')
  , httpUtils 		= require('oxsnps-core/helpers/httpUtils')
  , expressAppUtils = require('oxsnps-core/helpers/expressAppUtils')
  , pluginManager 	= require('oxsnps-core/pluginManager')
  , FILE_SEP 		= require('path').sep
  , maxQpdfReqs		= undefined

// Set plugin name, version and longDesc in pluginConf
pluginConf.module 	= PLUGIN_NAME
pluginConf.version 	= PLUGIN_VERSION
pluginConf.longDesc = packageJson.description
pluginConf.log 		= pluginConf.log || {}

// Export the Plugin's Configuration Express App
exports.pluginConf = pluginConf

// Define router
exports.router     = undefined

// Get the Logger & set the log level
var logger = log4js.getLogger(PLUGIN_NAME)
utils.log4jsSetLogLevel(logger, (pluginConf.log.level || 'INFO'))

/**************** PUBLIC FUNCTIONS ****************/
/**
 * Initialize Plugin
 * @param  {function} 	callback  callback(err, routes)
 */
exports.initialize = function(callback){

	var self 			= this
      , appPluginConf 	= (npsConf.plugins && npsConf.plugins[PLUGIN_NAME] ? npsConf.plugins[PLUGIN_NAME] : {})

	if(logger.isDebugEnabled()) logger.debug('--->initializing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION )
	if(self._initialized) return callback()
    
	// Write the server port needed for http://localhost:<port>/api/index.html to ./node_modules/rest.ish/swagger/oxs/utils.js
	/*TODO: REMOVE not needed anymore
	let oxsUtilsjs = path.join(__dirname,'swagger/oxs/utils.js')
	let writeStream = fs.createWriteStream(oxsUtilsjs)
	writeStream.write('let oxs={"port":' + npsServer.npsPort + '}') // writing 'let oxs={"port":8080}' to ./public/swagger/oxs/utils.js
	writeStream.end()
	*/
	// Merge application specific configuration with plugin configuration
    pluginConf = utils.mergeJSON(pluginConf, appPluginConf)

	// Maximum of concurrent qpdf requests; 5 is default if HAL_REST_ISH_MAX_QPDF_REQS is not defined
	if(pluginConf.maxOccurences && pluginConf.maxOccurences.qpdf ) maxQpdfReqs = pluginConf.maxOccurences.qpdf || 5

	// Export the modified and merged pluginConf
	exports.pluginConf = pluginConf

	// Add a router for the plugin routes and its middlewares
	expressAppUtils.setupRouter(self, function(err,router){
		if(err) return callback(err)
		exports.router = router
		self._initialized = true
		logger.info('\t\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' initialized')
        callback(null, {'routes': pluginConf.routes, 'swagger':{"name": PLUGIN_NAME, "url": '/' + PLUGIN_NAME + '/api/rest.yaml'}}) // V2.0.0
	})
}

/**
 * Activate Plugin. This function is called from pluginManager once all the enabled plugins have been initialize
 * @param  {function} 	callback  callback(err)
 */
exports.postInitialize = function(callback){
	
	let self 		= this
	  
	if(!self._initialized)
		return callback(new Error('Can\'t activate Plugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' as long as it has not been initialized!'))
	
	if(logger.isDebugEnabled()) logger.debug('Activating ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + '...')
	callback()
}

/**
 * Finalize Plugin
 * @param  {function} 	callback callback(err) 
 */
exports.finalize = function(callback){

	var self = this
	  , req  = {}

	if(logger.isDebugEnabled()) logger.debug('--->finalizing ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION )
	if(!self._initialized) return callback()

	async.series([
		// Disable all routes
		function(nextTask){
			req.body = {'module': PLUGIN_NAME}
			async.each(pluginConf.routes, 
				function(route, next){
					req.body.route = route
					exports.disableRoute(req, next)
				},
				function(err){
					return nextTask(err)
				}
			)
		}
	],
	function(err){
		if(err) return callback(err)
		self._initialized = false
		logger.info('\t\tPlugin ' + PLUGIN_NAME + ' v'+ PLUGIN_VERSION + ' finalized')
		callback()
	})
}

/**
 * Return Plugin Version
 * @param  {function} 	callback callback(err,version)
 */
exports.getVersion = function(callback){
	return callback(null, PLUGIN_VERSION)
}

/**
 * Process a HTTP Version Request
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.version = function(req, res){

	// Add Context to the response instance
	res._Context = {}

	// Add the Current Date to the Context
	res._Context.date = new Date()

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date)
	logger.info('--->version: Req. id=' + res._Context.reqId)

	var data = '<h3>' +  PLUGIN_NAME + ' v' + PLUGIN_VERSION + '</h3>'
	res.end(data)
	
	logger.info('--->version: Req. id=' + res._Context.reqId + ', sent back ' +
				PLUGIN_NAME + ' v' + PLUGIN_VERSION + ', over')
	return
}

/**
 * Process a HTTP Configuration Request 
 * 	Either as HTTP GET request:
 * 	 	GET http://localhost:1029/services/<plugin>/conf 							--> To retrieve the configuration
 * 	 	GET http://localhost:1029/services/<plugin>/conf?save=true&conf="{...}" 	--> To pass and save the configuration
 * 	or as HTTP POST request:
 * 	 	POST http://localhost:1029/services/<plugin>/conf 						--> To pass and save the configuration
 * 	 	req.body={
 * 	 		"conf": {...}
 * 	 	}
 * 	
 * @param {Request} 	req Request Object
 *                      req.query:{ // for HTTP GET request:
 *                      	"save":"false",
 *                      	"conf":"{<IMPORTANT: COMPLETE JSON Conf. as String>}"
 *                      }
 *                      req.body:{ // for HTTP POST request:
 *                      	"conf":{<IMPORTANT: COMPLETE JSON Conf. as JSON Object>}
 *                      }
 * @param {Response}    res Response Object
 */
exports.conf = function(req, res){

	var reqType 	= 'getconf'
	  , jsonConf 	= undefined

	// Add Context to the response instance
	res._Context = {}

	// Add the Current Date to the Context
	res._Context.date = new Date()

	// Add a Request Id to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date)
	logger.info('--->conf: Req. id=' + res._Context.reqId)
	
	if(req.method.toUpperCase() === 'POST'){
		logger.info('--->conf: body=' + JSON.stringify(req.body))
		if(!req.body || !req.body.conf) {
			logger.info('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body))
			return res.end('--->conf: Invalid POST configuration request, req.body=' + + JSON.stringify(req.body))
		}
		reqType = 'saveconf'		// save request
		jsonConf = req.body.conf	// set the json conf to be passed to saveConfService
	}
	else if(req.method.toUpperCase() === 'GET'){
		logger.info('--->conf: query=' + JSON.stringify(req.query))
		if(req.query && req.query.save && req.query.save.toLowerCase() === 'true' && req.query.conf)
		{
			reqType = 'saveconf'
			jsonConf = utils.parseJSON(unescape(req.query.conf)) // remove escaped chars and set the json conf to be passed to saveConfService
		} 
	}

	switch(reqType){
		case 'saveconf':
		    exports.saveConfService(jsonConf, function(err){
		    	logger.info('--->conf: Req. id=' + res._Context.reqId + ', over')
				res.json(err ? {'status':'error', 'message': err.message} : pluginConf)
		    }) 
			break
		default:
		case 'getconf':
			logger.info('--->conf: Req. id=' + res._Context.reqId + ', over')
			res.json(pluginConf)
			break
	}
}

/**
 * Store  the Plugin conf to its configuration file (i.e. .../<plugin>/conf/<plugin>.js)
 * @param  {JSON}   	conf     Plugin Configuration
 * @param  {Function} 	callback
 */
exports.saveConfService = function(jsonConf, callback){

	var filename 	= undefined
	  , data 		= undefined
	
	// PluginManager calls saveConfService without passing the conf
	if(typeof jsonConf === 'function'){
		callback = jsonConf
		jsonConf = pluginConf
	}

	// Assert jsonConf
	if(!jsonConf || jsonConf === null) return callback(new Error('--->saveConfService: empty Configuration'))
	if(logger.isDebugEnabled()) logger.debug('--->saveConfService: conf:' + JSON.stringify(jsonConf))

	// Backup the current plugin conf.in the server's backup dir and store the new one in the plugin's /conf dir
	async.series([
		// Task 1: Backup the current plugin conf.
		function(nextTask){
			filename = npsConf.backupDir + FILE_SEP + dateFormat.asString('yyyy-MM-dd_hh-mm-ss', new Date()) + '_configurationOf_' + PLUGIN_NAME + '.js' // build the Backup filename
			if(logger.isDebugEnabled()) logger.debug('--->saveConfService: Backuping plugin configuration to ' + filename)
			data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t') // build the plugin conf content

			// Write data to the Plugin configuration file
			fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
				if(err) logger.error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message) // log error & continue
				nextTask()
			})
		},
		// Task 2: Save the new plugin conf.
		function(nextTask){
			pluginConf = jsonConf // we should have here a proper conf, so overwrite the current pluginConf file
			filename = path.resolve(__dirname + '/conf/' + PLUGIN_NAME + '.js') // build the plugin configuration filename
			if(logger.isDebugEnabled()) logger.debug('--->saveConfService: Writing plugin configuration to ' + filename)
			data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t') // build the plugin conf content

			// Write data to Plugin configuration file
			fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
				if(err) nextTask(new Error('--->saveConfService: Unable to write ' + filename + ', Reason: ' + err.message))
				else nextTask()
			}) 
		}
	],
	function(err){
		callback(err)
	})
}

/**
 * HTTP GET Request to get the given GUID
 * 
 * GET /services/document/:guid
 * 
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.getGuid = function(req, res){

	_buildResContext(req, res)

	//res._Context.parms.guid
	// Swagger UI workaround; for the optional guid, Swagger does sometimes pass ',' or '{guid}'
	if(res._Context.parms.guid === '*' || res._Context.parms.guid === ',' || res._Context.parms.guid === '{guid}') res._Context.parms.guid = undefined
	logger.info('--->getGuid: Req. id=' + res._Context.reqId + ', API: GET /services/document/' + (res._Context.parms.guid ? res._Context.parms.guid : ''))
	if(logger.isDebugEnabled()) logger.debug('--->getGuid: Req. id=' + res._Context.reqId + ', res._Context: ' + JSON.stringify(res._Context))
	exports.getGuidService(res, function(err,result){
		if(err){
			logger.error('--->getGuid: Req. id=' + res._Context.reqId + ', not able to get guid: ' + res._Context.parms.guid + '. Reason: ' + err.message)
			return res.status(res._Context.httpCode||500).json({code:err.errno||res._Context.httpCode||500,message:err.message,reqId:res._Context.reqId})
		}
		logger.info('--->getGuid: Req. id=' + res._Context.reqId + ',  guid: ' + (res._Context.parms.guid ? res._Context.parms.guid : 'ALL') + '. File list: ' + JSON.stringify(result))
		res.status(200).json(result) // result is an array of files or dirs
	})
}

/**
 * HTTP POST Request to create the given GUID
 * 
 * POST /services/document/:guid
 * 
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.createGuid = function(req, res){

	_buildResContext(req, res)

	logger.info('--->createGuid: Req. id=' + res._Context.reqId + ', API: POST /services/document/' + res._Context.parms.guid)
	if(logger.isDebugEnabled()) logger.debug('--->createGuid: Req. id=' + res._Context.reqId + ', res._Context: ' + JSON.stringify(res._Context))
	exports.createGuidService(res, function(err,result){
		if(err){
			logger.error('--->createGuid: Req. id=' + res._Context.reqId + ', not able to create a dir for guid: ' + res._Context.parms.guid + '. Reason: ' + err.message)
			return res.status(500).json({code:err.errno||500,message:err.message,reqId:res._Context.reqId})
		}
		logger.info('--->createGuid: Req. id=' + res._Context.reqId + ', dir: ' + result + ' created')
		res.status(200).json({dir:result})
	})
}

/**
 * HTTP DELETE Request to delete the given GUID
 * 
 * DELETE /services/document/:guid
 * 
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.deleteGuid = function(req, res){

	_buildResContext(req, res)

	logger.info('--->deleteGuid: Req. id=' + res._Context.reqId + ', API: DELETE /services/document/' + res._Context.parms.guid)
	if(logger.isDebugEnabled()) logger.debug('--->deleteGuid: Req. id=' + res._Context.reqId + ', res._Context: ' + JSON.stringify(res._Context))
	exports.deleteGuidService(res, function(err,result){
		if(err){
			logger.error('--->deleteGuid: Req. id=' + res._Context.reqId + ', not able to delete guid: ' + res._Context.parms.guid + '. Reason: ' + err.message)
			return res.status(res._Context.httpCode||500).json({code:err.errno||res._Context.httpCode||500,message:err.message,reqId:res._Context.reqId})
		}
		logger.info('--->deleteGuid: Req. id=' + res._Context.reqId + ',  guid: ' + res._Context.parms.guid + ', dir: ' + result + ' deleted.')
		res.status(200).json({dir:result})
	})
}

/**
 * HTTP POST Request to add an image to the given GUID
 * 
 * POST /services/document/:guid/image/:imageid
 * 
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.addImage = function(req, res){

	_buildResContext(req, res)

	logger.info('--->addImage: Req. id=' + res._Context.reqId + ', API: POST /services/document/' + res._Context.parms.guid + '/image/' + res._Context.parms.imageid)
	if(logger.isDebugEnabled()) logger.debug('--->addImage: Req. id=' + res._Context.reqId + ', res._Context: ' + JSON.stringify(res._Context))

	// Add the request to the Context; used in oxsnps-httpimport's uploadService()
	res._Context.req = req
	
	exports.addImageService(res, function(err){
		if(err){
			logger.error('--->addImage: Req. id=' + res._Context.reqId + ', not able to add image to guid: ' + res._Context.parms.guid + '. Reason: ' + err.message)
			return res.status(res._Context.httpCode||500).json({code:err.errno||res._Context.httpCode||500,message:err.message,reqId:res._Context.reqId})
		}
		logger.info('--->addImage: Req. id=' + res._Context.reqId + ',  added image: ' + res._Context.file + ' to guid: ' + res._Context.parms.guid)
		res.status(200).json({file:res._Context.file})
	})
}

/**
 * HTTP POST Request to add Metadata to the given GUID
 * 
 * POST /services/document/:guid/meta
 * 
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.addMeta = function(req, res){
	
	res._Context = {}	// Add Context to the response instance
	res._Context.date = new Date()	// Add the Current Date to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date) // Add a Request Id to the Context
	res._Context.parms = {}
	res._Context.parms.guid = req.params.guid
	res._Context.parms.body = req.body

	logger.info('--->addMeta: Req. id=' + res._Context.reqId + ', API: POST /services/document/' + res._Context.parms.guid + '/meta')
	if(logger.isDebugEnabled()) logger.debug('--->addMeta: Req. id=' + res._Context.reqId + ', res._Context: ' + JSON.stringify(res._Context))
	exports.addMetaService(res, function(err){
		if(err){
			logger.error('--->addMeta: Req. id=' + res._Context.reqId + ', not able to add metadata to guid: ' + res._Context.parms.guid + '. Reason: ' + err.message)
			return res.status(res._Context.httpCode||500).json({code:err.errno||res._Context.httpCode||500,message:err.message,reqId:res._Context.reqId})
		}
		logger.info('--->addMeta: Req. id=' + res._Context.reqId + ',  added metadata to guid: ' + res._Context.parms.guid)
		res.status(200).json({})
	})
}

/**
 * HTTP POST Request to complete the given GUID
 * 
 * POST /services/document/:guid/operation/complete
 * 
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
exports.complete = function(req, res){
	
	_buildResContext(req, res)

	logger.info('--->complete: Req. id=' + res._Context.reqId + ', API: POST /services/document/' + res._Context.parms.guid + '/operation/complete')
	if(logger.isDebugEnabled()) logger.debug('--->complete: Req. id=' + res._Context.reqId + ', res._Context: ' + JSON.stringify(res._Context))
	exports.completeService(res, function(err){
		if(err){
			logger.error('--->complete: Req. id=' + res._Context.reqId + ', not able to process guid: ' + res._Context.parms.guid + '. Reason: ' + err.message)
			return res.status(res._Context.httpCode||500).json({code:err.errno||res._Context.httpCode||500,message:err.message,reqId:res._Context.reqId})
		}
		logger.info('--->complete: Req. id=' + res._Context.reqId + ',  guid: ' + res._Context.parms.guid + ' completed')
		res.status(200).json({guid:res._Context.parms.guid})
	})
}

/************ OPTIONAL PUBLIC FUNCTIONS ***********/
/**
 * Set Log Level
 * @param {String} logLevel Log Level. values:DEBUG|INFO|ERROR 
 */
exports.setLogLevel = function(logLevel){
	pluginConf.log.level = logLevel || 'INFO'
	utils.log4jsSetLogLevel(logger, pluginConf.log.level)
}

/**
 * Get Log Level
 * @return {String} Log Level. values:DEBUG|INFO|ERROR
 */
exports.getLogLevel = function(){
	return pluginConf.log.level
}

/**
 * Enable the given route
 * @param  {Request} 	req
 *                      req.body = {
 *                      	"module":"sample", 
 *                      	"route": { 
 *                      		"path": "/path/to/the/service",
 *                      		"method": 	Optional. Values are "get|post|dw|queue",
 *                      		"service": 	"version"
 *                      	}
 *                      }
 * @param  {function} 	callback callback(err)
 */
exports.enableRoute = function(req, callback){

	var self = this

	if(logger.isDebugEnabled()) logger.debug('--->enableRoute: Route=' + JSON.stringify(req.body.route))

	if(!req.body.route) return callback(new Error('route parameter not given'))

	req.body.route.method = req.body.route.method || 'get'

	// Enable the route
	switch(req.body.route.method.toLowerCase()){
		case 'post':
		case 'get':
		case 'delete':
			// Enable HTTP Route
			expressAppUtils.enableRoute(self, req.body.route, callback)
			break
		default:
			callback(new Error('Wrong parameters(unknown route.method)'))
	}
} 

/**
 * Disable the given route
 * @param  {Request} 	request Request json has following structure.
 * 		            	req.body = {
 *                      	"module":"sample", 
 *                      	"route": { 
 *                      		"path": "/path/to/the/service",
 *                      		"method": 	Optional. Values are "get|post|dw|queue"
 * 		            	  	}
 * 		                }
 * @param  {function} 	callback  callback(err)
 */
exports.disableRoute = function(req, callback){
	let self    = this

	if(logger.isDebugEnabled()) logger.debug('--->disableRoute: Route=' + JSON.stringify(req.body.route))
	
	if(!req.body.route) return callback(new Error('route parameter not given'))

	req.body.route.method = req.body.route.method || 'get'

	// Disable the route
	switch(req.body.route.method.toLowerCase()){
		case 'post':
		case 'get':
		case 'delete':
			expressAppUtils.disableRoute(self, req.body.route, callback) // disable HTTP Route
			break
		default:
			callback(new Error('Wrong parameters(unknown route.method)'))
	}
} 

/**
 * Service to get the list of files contained in the dir of the given GUID
 * @param  {Response}  	res Response Object
 *                      	res._Context:{
 * 								parms:{
 * 									guid: <guid>
 * 								}
 *                          }
 * @param  {Function} 	callback(err,result)
 * 							result: <array of files>
 */
exports.getGuidService = function(res, callback){
	 
	let dir = path.resolve(process.env["REST_ISH_INPUT_CHANNEL_DIR"])
	let fileList = undefined

	async.series([
		// Task 1: If guid is speified, assert the corr. dir
	    function(nextTask){
			// Ensure that guid dir exists; if not return 404
			if(res._Context.parms.guid){
				_checkIfDirExists(res,function(err,found){
					if(err || !found){
						res._Context.httpCode = 404
						return nextTask(err ? err : new Error('guid: ' + res._Context.parms.guid + ' not found'))
					}
					return nextTask()
				})
			}
			else nextTask()
	    },
		// Task 2: Get the list of the files of the guid dir or the list of the guid dirs
	    function(nextTask){
			// Get the list of the files of the guid dir
			if(res._Context.parms.guid){
				dir = path.resolve(dir, res._Context.parms.guid) // append the guid dir
				utils.getFiles(dir, '.*', true, function(err,result){ // recursive:true
					fileList = result ? result : undefined
					nextTask(err)
				})
			}
			// No guid, so get the list of the guid dirs
			else{
				utils.getFsEntries(dir,{'recursive':false,'dirPattern':'.*'}, function(err,result){
					fileList = result ? result : undefined
					nextTask(err)
				})
			}
		}
	],
	function(err){
		callback(err,fileList)
	})
}

/**
 * Service to create the given GUID
 * @param  {Response}  	res Response Object
 *                      	res._Context:{
 * 								parms:{
 * 									guid: <guid>
 * 								}
 *                          }
 * @param  {Function} 	callback(err,dir)
 * 							dir: <the fully qualified created guid dir>
 */
exports.createGuidService = function(res, callback){
	 
	let mode = parseInt('0777',8)
	  , dir = path.resolve(process.env["REST_ISH_INPUT_CHANNEL_DIR"], res._Context.parms.guid)
	utils.mkDir(dir,mode,function(err){
		return callback(err,dir)
	})
}

/**
 * Service to delete the given GUID
 * @param  {Response}  	res Response Object
 *                      	res._Context:{
 * 								parms:{
 * 									guid: <guid>
 * 								}
 *                          }
 * @param  {Function} 	callback(err)
 */
exports.deleteGuidService = function(res, callback){

	let dir = undefined

	async.series([
		// Task 1: Assert guid dir
	    function(nextTask){
			// Ensure that guid dir exists; if not return 404
			_checkIfDirExists(res,function(err,found){
				if(err || !found){
					res._Context.httpCode = 404
					return nextTask(err ? err : new Error('guid: ' + res._Context.parms.guid + ' not found'))
				}
				return nextTask()
			})
	    },
		// Task 2: Upload file to the guid dir
	    function(nextTask){
			dir = path.resolve(process.env["REST_ISH_INPUT_CHANNEL_DIR"], res._Context.parms.guid)
			utils.deleteDir(dir, true, 'x', 50, true, function(err){ /* dir, true if root dir should be deleted, x=unit is ms, older than 50ms, true to delete recursively */
				return nextTask(err)
			})
		}
	],
	function(err){
		callback(err,dir)
	})	
}

/**
 * Service to add Metadata to the given GUID
 * @param  {Response}  	res Response Object
 *                      	res._Context:{
 * 								parms:{
 * 									guid: <guid>,
 * 									body: <JSON Metadata>
 * 								}
 *                          }
 * @param  {Function} 	callback(err)
 * 
 * NOTE: This service stores (i.e. overwrites) the metada as Import.json into the guid dir
 */
exports.addMetaService = function(res, callback){

	async.series([
		// Task 1: Assert guid dir
	    function(nextTask){
			// Ensure that guid dir exists; if not return 404
			_checkIfDirExists(res,function(err,found){
				if(err || !found){
					res._Context.httpCode = 404
					return nextTask(err ? err : new Error('guid: ' + res._Context.parms.guid + ' not found'))
				}
				return nextTask()
			})
	    },
		// Task 2: Assert Import.json and the files of the guid dir
	    function(nextTask){
			// Assert res._Context.parms.body; should not be empty
			let body = res._Context.parms.body
			if(!body.STACK || !body.STACK.DEVICE){
				res._Context.httpCode = 415 // 415 Unsupported Media Type - If incorrect content type was provided as part of the request
				return nextTask(new Error("Missing metadata or metadata is not JSON compliant"))
			}
	
			// Write data to <ISH input channel>./<guid>/Import.json 
			let filename = path.resolve(process.env["REST_ISH_INPUT_CHANNEL_DIR"], res._Context.parms.guid, 'Import.json')
			fs.writeFile(filename,JSON.stringify(res._Context.parms.body),{'encoding': 'utf8'},nextTask)
		}
	],
	function(err){
		callback(err)
	})
}

/**
 * Service to add an image to the given GUID
 * @param  {Response}  	res Response Object
 *                      	res._Context:{
 * 								parms:{
 * 									guid: 		<guid>,
 * 									imageid: 	<Image filename>,
 * 									req: 		<express req>
 * 								}
 *                          }
 * @param  {Function} 	callback(err)
 */
exports.addImageService = function(res, callback){

	async.series([
		// Task 1: Assert guid dir
	    function(nextTask){
			// Ensure that guid dir exists; if not return 404
			_checkIfDirExists(res,function(err,found){
				if(err || !found){
					res._Context.httpCode = 404
					return nextTask(err ? err : new Error('guid: ' + res._Context.parms.guid + ' not found'))
				}
				return nextTask()
			})
	    },
		// Task 2: Upload file to the guid dir
	    function(nextTask){
			res._Context.file = path.resolve(process.env["REST_ISH_INPUT_CHANNEL_DIR"], res._Context.parms.guid, res._Context.parms.imageid)
			pluginManager.callPluginService({name:'oxsnps-httpimport',service:'uploadService'},res,nextTask)
		}
	],
	function(err){
		callback(err)
	})
}

/**
 * Service to complete the given GUID
 * @param  {Response}  	res Response Object
 *                      	res._Context:{
 * 								parms:{
 * 									guid: <guid>
 * 								}
 *                          }
 * @param  {Function} 	callback(err)
 * 
 */
exports.completeService = function(res, callback){

	// Add fully qualified guid dir and Import.json to res._Context
	res._Context.parms.dir = path.resolve(process.env["REST_ISH_INPUT_CHANNEL_DIR"], res._Context.parms.guid)
	res._Context.parms.importFilename = path.resolve(res._Context.parms.dir, 'Import.json')
	if(logger.isDebugEnabled()) logger.debug('--->completeService: Req. id=' + res._Context.reqId + ', res._Context: ' + JSON.stringify(res._Context))

	async.series([
		// Task 1: Assert guid dir
	    function(nextTask){
			// Ensure that guid dir exists; if not return 404
			_checkIfDirExists(res,function(err,found){
				if(err || !found){
					res._Context.httpCode = 404
					return nextTask(err ? err : new Error('guid: ' + res._Context.parms.guid + ' not found'))
				}
				return nextTask()
			})
	    },
		// Task 2: Read Import.json
	    function(nextTask){
			utils.parseJSONFile(res._Context.parms.importFilename,function(err,data){
				if(err) return nextTask(err)
				res._Context.parms.importJSON = data
				nextTask()
			})
	    },
		// Task 3: Assert Import.json and the files of the guid dir
	    function(nextTask){
			_assertInput(res,function(err){
				if(err) res._Context.httpCode = 422 // 422 Unprocessable Entity - Used for validation errors
				return nextTask(err)
			})
	    },
		// Task 4: Rename folder as _ready
	    function(nextTask){
			_renameDir(res,'_ready',nextTask)
	    }
	],
	function(err){
		callback(err)
	})
}

/**************** PRIVATE FUNCTIONS ***************/
/**
 * Fill res._Context
 * 
 * @param  {Request}  	req Request Object
 * @param  {Response} 	res Response Object
 */
function _buildResContext(req, res){
		
	res._Context = {}	// Add Context to the response instance
	res._Context.date = new Date()	// Add the Current Date to the Context
	res._Context.reqId = utils.buildReqId(res._Context.date) // Add a Request Id to the Context
	res._Context.parms = httpUtils.getRequestParms(req)	// get the request parms
}

/**
 * Rename the GUID dir as <guid>_ready
 * @param  {Response}  	res Response Object
 *                      	res._Context:{
 * 								parms:{
 * 									guid: 	<guid>
 * 									dir:	<fully qualified guid dir>
 * 								}
 *                          }
 * @param  {String}     changeAs  	Suffix string (i.e. '_ready')
 * @param  {function}   callback  	callback(err)
 */
function _renameDir(res, changeAs, callback){

	let newDir = res._Context.parms.dir + changeAs

  	// Rename folder
  	if(logger.isDebugEnabled()) logger.debug('--->_renameDir: Req.Id=' + res._Context.reqId + ', Renaming ' + 
	  res._Context.parms.dir +' as ' + newDir + '...')
	fs.rename(res._Context.parms.dir, newDir,  callback)
}

/**
 * Check that content of Import.json matches exaclty the dir content
 * @param  {Response}  	res Response Object
 *                      	res._Context:{
 * 								parms:{
 * 									guid: 			<guid>
 * 									dir:			<fully qualified guid dir>
 * 									importFilename:	<fully qualified filename of Import.json>
 * 									importJSON: 	<Content of Import.json>
 * 								}
 *                          }
 * @param  {function}   callback  	callback(err)
 */
function _assertInput(res, callback){

	let stack = res._Context.parms.importJSON.STACK
	let device = res._Context.parms.importJSON.STACK.DEVICE

	// Verify that STACK, STACK.IMAGE and DEVICE are defined
	if(!stack || !stack.IMAGE || !device) return callback(new Error('Missing STACK, STACK.IMAGE or DEVICE for guid: ' + res._Context.parms.guid))

	let images = stack.IMAGE
	let files = undefined
	let file = undefined
	let dir = res._Context.parms.dir

	async.series([
		// Task 1: Get list of files of the guid dir
	    function(nextTask){
			utils.getFiles(dir, '.*\\.pdf$|.*\\.tif$|.*\\.tiff$|.*\\.jpg$|.*\\.jpeg$', true, /* true means get files of the sub dirs too */ function(err,result){
				if(err) return nextTask(err)
				res._Context.parms.files = utils.clone(result) // clone the list of files
				files = result 
				nextTask()
			})
	    },
		// Task 2: Check that the files defined in Import.json are available in the guid dir
	    function(nextTask){ 
			for(let i=0;i<images.length;i++){
				if(logger.isDebugEnabled()) logger.debug('--->_assertInput: Req.Id=' + res._Context.reqId + ', checking Image: ' + JSON.stringify(images[i]))
				file = path.resolve(res._Context.parms.dir, images[i].FileName)
		
				for(let j=0;j<files.length;j++){
					if(file===files[j]){
						files.splice(j,1) 		// remove file from files array
						images[i].found = true	// image has been found in the guid dir 
						break
					}
				}
			}
			if(logger.isDebugEnabled()) logger.debug('--->_assertInput: Req.Id=' + res._Context.reqId + ', remaining files ' + JSON.stringify(files))
			nextTask()
	    },
		// Task 3: Check the file list of Import.json against the file list of the guid dir and and the other way around
	    function(nextTask){
			// Ensure that we have not more files in the guid dir than listed in Import.json
			if(files.length!==0){
				return nextTask(new Error('the files listed in: ' + res._Context.parms.importFilename + '  do not match the files in the guid dir: ' + dir))
			}

			// Ensure that we found all the files listed in Import.json
			for(let i=0;i<images.length;i++){
				if(!images[i].found)
					return nextTask(new Error('Missing file: ' + images[i].FileName + ' in: ' + res._Context.parms.dir))
			}
			nextTask()
	    },
		// V1.0.6 Begin
		/*
		// Task 4: Ensure that none of the PDF files has more than 1 page
	    function(nextTask){
			async.forEachOfLimit(res._Context.parms.files,maxQpdfReqs,function(file,index,next){
				if(path.extname(file)!=='.pdf') return next() // skip if not a pdf file
				let resTmp={
					'_Context':{
						'reqId': 		res._Context.reqId,
						'inputFile': 	file
					}
				}
				_getPDFPageCount(resTmp,function(err, result){
					if (err) return next(err)
					if(logger.isDebugEnabled()) logger.debug('--->_assertInput: Req. id=' + res._Context.reqId + ', result: ' + JSON.stringify(result))
					if(result!==1) return next(new Error('PDF file: ' + file + ' has: ' + result + ' pages, but should have only one.'))
					next()
				})
			},
			function(err){
				nextTask(err)
			})		
	    }
		*/
		// V1.0.6 End
	],
	function(err){
		callback(err)
	})
}

/*
 * Get PDF page count using qpdf tool
 * @param  {Response}  	res Response Object
 *                      	res._Context:{
 * 								inputFile:	<fully qualified name of the PDF file>
 *                          }
 * @param  {Function} callback 	callback(err)
 */
function _getPDFPageCount(res, callback){

	// Set qpdf arguments. Surround filename with "'" to avoid errors if filenames contains spaces
    res._Context.qpdf= {'args':['qpdf', '--show-npages', "'" + res._Context.inputFile + "'"]}
    
    if(logger.isDebugEnabled()) logger.debug('--->_getPDFPageCount: Req.Id=' + res._Context.reqId + ', Calling oxnps-qpdfclient.pageCountService() to get PDF page count, parms:' + JSON.stringify(res._Context.qpdf))
    pluginManager.callPluginService({'name':'oxsnps-qpdfclient','service':'pageCountService'},res,callback)
}

/*
 * Get PDF page count using qpdf tool
 * @param  {Response}  	res Response Object
 *                      	res._Context:{
 * 								guid:	<fully qualified name of the PDF file>
 *                          }
 * @param  {Function} callback(err,true|false)	true if dir exists
 */
function _checkIfDirExists(res,callback){

	let dir = path.resolve(process.env["REST_ISH_INPUT_CHANNEL_DIR"], res._Context.parms.guid)
	return utils.dirExists(dir,callback)
}
