/**-----------------------------------------------------------------------------
 * oxsnps.js:
 * Workflow controlled custom import of a stack
 *
 * - For a new stack, a Order/Process will be created
 * - Process gets workflow and workflow decides the stack import flow
 *
 *
 * Author    :  Panneer
 * Copyright :  (C) 2022-24 by Maas Holding GmbH
 *
 * History
 * 1.0.17   19.09.2025    OXS-17221: Added gcqPubSubTest() funtion to test ordered delivery of Pub/Sub messages
 * 1.0.16   16.05.2025    OXS-16900: Ensure a transition is allowed from current state before invoking the set signal for that transition
 * 1.0.15   29.11.2024    OXS-16426: Wackler: Cepra ICs: Fixed minor bug in accessing stack process indexes for the multiple input import
 * 1.0.14   13.06.2024    OXS-15779: In case of technical errors, acknowledge the message in the plugin itself since process has not been created or process might not be not in proper state to ack the msg from workflow
 * 1.0.13   13.06.2024    OXS-15733: End the job from workflow instead of from plugin
 * 1.0.12   17.05.2024    OXS-15660: Wackler: Ensured to use tech user (as default) for config authorization instead of channel defined user
 * 1.0.11   09.05.2024    Wackler: cepra IC: Ensured to rename the input with error status when import fails with assertion of servers
 * 1.0.10   25.01.2024    OXS-15306/OXS-15150: Wackler: Ausgangsrechnung IC: Initialized global vars in "initialize" to reload this module dynamically
 * 1.0.9    10.11.2023    a. OXS-14965: Wackler: Bescheinigung IC: Extended to detect input format (based on a flag in channel) for multiple inputs
 *                        b. OXS-14965: Wackler: Bescheinigung IC: Fixed minor bug in getting wait period in seconds for multiple inputs
 * 1.0.8    01.11.2023    a. OXS-15160: Hartmann: Tankrechnung IC: Extended to remove '.' (dots) in group id for multiple inputs
 *                        b. OXS-15160: Hartmann: Tankrechnung ICs Extended to reset group id (to process info) cache for multiple inputs
 * 1.0.7    12.09.2023    a. OXS-15009: Wackler: Cepra ICs: Fixed issues with synchronized signal handling
 *                        b. OXS-15009: Wackler: Cepra ICs: Extended "awaitMultipleInputs" channel defined attribute to say
 *                           minimum and maximum await period for the inputs
 * 1.0.6    01.09.2023    a. OXS-15009: Wackler: Cepra ICs: Extended to create process first and then to set START signal separately
 *                        b. OXS-15009: Wackler: Cepra ICs: Extended to use "<channel>->wsInitial" given value as start signal
 * 1.0.5    23.08.2023    a. OXS-14974: Wackler: Extended to synchronize the importMultipleFiles calls using following techniques
 *                           - async.queue
 *                           - proper-lockfile (module)
 *          10.08.2023    b. OXS-14890: Wackler: importFile.js: Added "stack_info_json.keys_json" to process additional info (as in importMultipleFiles.js)
 * 1.0.4    05.07.2023    OXS-14816: Wackler: cepra-ablieferbeleg IC: Extended to import multiple inputs
 * 1.0.3    16.06.2023    OXS-14729: Wackler: ablieferbeleg-extern IC: Extended to import stack directory or file
 * 1.0.2    13.06.2023    OXS-14729: Wackler: ablieferbeleg-extern IC: Extended to subscribe both FTP and SFTP import channels
 * 1.0.1    28.07.2022    OXS-13855 : Wackler: IC import_sped_beschein - Part 1 spedbeschein (Ascii to PDF)
 * 1.0.0    07.04.2022    OXS-13557: Initial Release
 *----------------------------------------------------------------------------*/
'use strict';

/**
 * Variable declaration
 * @private
 */
let packageJson      = require(__dirname + '/package.json')
  , PLUGIN_NAME      = packageJson.name
  , PLUGIN_VERSION   = packageJson.version // Get Server Version from package.json
  , pluginNameNVer   = PLUGIN_NAME + ' v' + PLUGIN_VERSION

  , npsServer        = require('oxsnps-core/server')
  , npsConf          = npsServer.npsConf

  , npsTempDir       = npsServer.npsTempDir

  , pluginConf       = require(__dirname + '/conf/' + PLUGIN_NAME + '.js')

  , utils            = require('oxsnps-core/helpers/utils')
  , httpUtils        = require('oxsnps-core/helpers/httpUtils')
  , shutdown         = require('oxsnps-core/shutdown')
  , dateFormat       = require('oxsnps-core/helpers/date_format')
  , pluginManager    = require('oxsnps-core/pluginManager')
  , expressAppUtils  = require('oxsnps-core/helpers/expressAppUtils')

  , fs               = require('fs')
  , url              = require('url') // V1.0.8b Change
  , path             = require('path')
  , async            = require('async')
  , log4js           = require('log4js')

  , taskBlocker      = require('proper-lockfile') // V1.0.5a Change
  // Collection of queues to serialize multiple inputs of same channel
  , multipleInputQueues = {} // V1.0.5a Change

  , FILE_SEP         = path.sep

  , serviceHandles   = {
      'importMultipleFiles':  require(__dirname + '/services/importMultipleFiles'), // V1.0.4 Change
      'importFile':           require(__dirname + '/services/importFile'),          // V1.0.3b Change
      'importStack':          require(__dirname + '/services/importStack')
  }

// Set plugin name, version and longDesc in pluginConf
pluginConf.module    = PLUGIN_NAME
pluginConf.version   = PLUGIN_VERSION
pluginConf.longDesc  = packageJson.description

pluginConf.log       = pluginConf.log || {}

// Export the Plugin's Configuration Express App
exports.pluginConf   = pluginConf
exports.router       = undefined // For oxsnps-core V4.1.x

// Get the Logger
let logger = log4js.getLogger(PLUGIN_NAME)
utils.log4jsSetLogLevel(logger, (utils.getAppLogLevel(pluginConf.log) || 'INFO'))

/**************** PUBLIC FUNCTIONS ****************/
/**
 * Initialize Plugin
 * @param  {function}    callback  callback(err, routes)
 */
exports.initialize = function (callback){

    let self          = this
      , appPluginConf = (npsConf.plugins && npsConf.plugins[PLUGIN_NAME] ? npsConf.plugins[PLUGIN_NAME] : {})
      , logPrefix     = '--->initializing '

    if(logger.isDebugEnabled()) logger.debug(logPrefix + pluginNameNVer)

    if (self._initialized) return callback()

    multipleInputQueues = {} // V1.0.10 Change

    // Merge application specific configuration with plugin configuration
    pluginConf = utils.mergeJSON(pluginConf, appPluginConf)
    exports.pluginConf = pluginConf

    // Add a router for the plugin routes and its middlewares
    expressAppUtils.setupRouter(self, function(err, router){
        if(err) return callback(err)

        exports.router = router
        self._initialized = true
        logger.info('Plugin ' + pluginNameNVer + ' initialized')

        return callback(null, pluginConf.routes)
    })
}

/** * Activate Plugin. This function is called from pluginManager once all the enabled plugins have been initialize
 * @param  {function}    callback  callback(err)
 */
exports.postInitialize = function (callback) {
    let self      = this
      , waitTime  = pluginConf.waitTime || 5000 // Wait time in milliseconds before enabling listeners
      , channels  = undefined
      , resTmp    = undefined
      , gpsRoutes = []    // google pub/sub routes
      , logPrefix = '--->postInitialize: '

    if(!self._initialized) return callback(new Error('Can\'t activate Plugin ' + pluginNameNVer + ' as long as it has not been initialized!'))
    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Activating ' + pluginNameNVer + '...')

    async.series([
        // Task 1: Get Import Channels
        function(nextTask){
            resTmp = {
                '_Context':{
                    'channel': { 
                        'plugin':    PLUGIN_NAME,
                        //'source':    'FTP', // V1.0.3a Change: get both ftp and sftp type channels
                        'type':     'import'
                    }
                }
            }

            pluginManager.callPluginService({name:'oxsnps-channel',service:'getPluginChannelService'}, resTmp, function(err, result){
                if(err) return nextTask(new Error('postInitialize Error: ' + err.message))

                logger.debug(logPrefix + 'Got routes from config. server for current plugin: ' + PLUGIN_NAME + ', result: ' + JSON.stringify(result))
                channels = result
                return nextTask()
            })
        }, 
        // Task 2: Filter rules
        function(nextTask){
            if(!channels || !channels.routes){
                if(routesCount<=0) logger.warn(logPrefix + 'No import channels are available in config. server for plugin: ' + PLUGIN_NAME)
                return nextTask()
            }

            let routesCount = 0
            channels.routes.forEach(function(route){
                // V1.0.3a Begin
                // Updated in reference with ci.condor.rechnung plugin's postInitialize
                //
                // If route is invalid, return
                if(!route || !route.type) return

                // If route is not of FTP or SFTP type, return
                if(route.type.toLowerCase().indexOf('ftp') < 0) return

                // Use the plugin service as configured in the channel
                let serviceName = 'gcqImportStack' // default service
                if(route.pluginService && route.pluginService.name){
                    let plsName = route.pluginService.name.trim()
                    if(plsName.length > 0) serviceName = plsName
                }
                route.action = {'props':{'module': PLUGIN_NAME, 'service': serviceName}}
                // V1.0.3a End

                if(route.method === 'gcpubsub') gpsRoutes.push(route) // google pub/sub route

                pluginConf.routes.push(route)
                routesCount++
            })

            logger.debug(logPrefix + 'Plugin: ' + PLUGIN_NAME + ' routes: ' + JSON.stringify(pluginConf.routes))
            return nextTask()
        },
        // Task 3: Add google pub/sub listeners (if any configured)
        function(nextTask){
            if(gpsRoutes.length <= 0) return nextTask()

            resTmp = {
                '_Context': {
                    'reqId':     utils.buildReqId(new Date()),
                    'gcpubsub': {'routes': gpsRoutes}
                }
            }
            setTimeout(function(){
                pluginManager.callPluginService({name:'oxsnps-gcpubsub', service:'addListenersService'}, resTmp, function(err){
                    if(err) logger.error('postInitialize Error: ' + err.message)
                })
            }, waitTime)
            return nextTask()
        },
        // V1.0.5 Begin
        // Task 4: Create async queues for multiple input channels
        function(nextTask){
            if(gpsRoutes.length <= 0) return nextTask()

            gpsRoutes.forEach(r => {
                // Skip single input channels
                if(r.multipleInputs === undefined || r.multipleInputs === false) return

                let channelName = r.name || r.path
	            if(!multipleInputQueues[channelName]){
	                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Creating a queue for the channel: ' + channelName + ' ...')
	                multipleInputQueues[channelName] = {
	                    'queue': async.queue(
	                                 // This function called for each item in the queue
	                                 function(item, next){
                                         _groupAndImportMultipleFiles(item.res, next)
                                     },
                                     // run one task at time
                                     1
	                             )
	                }
	                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Created a queue for the channel: ' + channelName)
	            }
            })

            return nextTask()
        }
        // V1.0.5 End
    ],
    function(err){
        return callback(err)
    })
}

/**
 * Prefinalize Plugin
 * @param  {Response}      res Response Object
 * @param  {function}     callback callback(err) 
 */
exports.preFinalize = function(res, callback){

    var self      = this
      , req       = {}
      , softQuit  = false
      , resTmp    = {}
      , logPrefix = '--->preFinalize: Req. id=' + res._Context.reqId + ', '

    if(logger.isDebugEnabled()) logger.debug(logPrefix + pluginNameNVer)
    if(!self._initialized) return callback()

    // Use case: HTTP request: http://localhost:1033/quit/?soft=true|false
    if(res._Context.query && res._Context.query.soft && res._Context.query.soft.toLowerCase() === 'true') softQuit = true
    // Use case: For signals 'SIGTERM' | 'SIGINT'
    else if (res._Context.softquit) softQuit = res._Context.softquit
    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'softQuit: ' + softQuit)
    
    resTmp._Context = {
        'reqId':            res._Context.reqId,
        'softquit':         softQuit,
        'shutdownOptions':  res._Context.shutdownOptions,
        'shutdown': {
            'jobPrefix':    PLUGIN_NAME + '_',
            'routes':       pluginConf.routes,
            'routeTypes':   ['dir', 'gcpubsub'],
            'jobTimeout':   res._Context.shutdownOptions.jobTimeout
        }
    }

    async.series([
        // Disable all routes. It also deletes Queued Jobs
        function(nextTask){
            req.body = {'module': PLUGIN_NAME}
            async.each(pluginConf.routes, 
                function(route, next){
                    req.body.route = route
                    exports.disableRoute(req, next)
                },
                function(err){
                    return nextTask(err)
                }
            )
        },
        // Wait for job completion, if soft quit is triggered.
        function(nextTask){
            if(!softQuit) return nextTask()
            // Shutdown jobs gracefully 
            shutdown.waitTillJobCompletion(resTmp, nextTask)
        }
    ],
    function(err){
        if(logger.isDebugEnabled()) logger.debug(logPrefix + (err ? ('error: ' + err.message) : ' over.'))
        if(err) return callback(err)
        logger.info('\t\tPlugin ' + pluginNameNVer + ' prefinalize over')
        return callback()
    })    
}

/**
 * Finalize Plugin
 * @param  {function}     callback callback(err)
 */
exports.finalize = function(callback){

    let self      = this
      , logPrefix = '--->finalize: '

    if(logger.isDebugEnabled()) logger.debug(logPrefix + pluginNameNVer)
    if(!self._initialized) return callback()

    self._initialized = false
    logger.info('\t\tPlugin ' + pluginNameNVer + ' finalized')
    return callback()
}

/**
 * Return Plugin Version
 * @param  {function}    callback callback(err,version)
 */
exports.getVersion = function (callback) {
    return callback(null, PLUGIN_VERSION)
}

/**
 * Process a HTTP Version Request
 * @param  {Request}    req Request Object
 * @param  {Response}    res Response Object
 */
exports.version = function (req, res) {

    let ctDate    = new Date()
      , logPrefix = '--->version: '

    // Add Context to the response instance
    res._Context = {
        'date': ctDate,
        'reqId': utils.buildReqId(ctDate)
    }
    logger.info(logPrefix + 'Req. Id=' + res._Context.reqId)

    res.end('<h3>' + pluginNameNVer + '</h3>')
    logger.info(logPrefix + 'Req. Id=' + res._Context.reqId + ', sent back ' + pluginNameNVer + ', over')
}

/**
 * Process a HTTP Configuration Request
 *    Either as HTTP GET request:
 *        GET http://localhost:1029/services/<plugin>/conf                            --> To retrieve the configuration
 *        GET http://localhost:1029/services/<plugin>/conf?save=true&conf="{...}"    --> To pass and save the configuration
 *    or as HTTP POST request:
 *        POST http://localhost:1029/services/<plugin>/conf                        --> To pass and save the configuration
 *        req.body={
 *              "conf": {...}
 *          }
 *
 * @param {Request}    req Request Object
 *                      req.query:{ // for HTTP GET request:
 *                          "save":"false",
 *                          "conf":"{<IMPORTANT: COMPLETE JSON Conf. as String>}"
 *                      }
 *                      req.body:{ // for HTTP POST request:
 *                          "conf":{<IMPORTANT: COMPLETE JSON Conf. as JSON Object>}
 *                      }
 * @param {Response}    res Response Object
 */
exports.conf = function (req, res) {

    let ctDate    = new Date()
      , reqMethod = req.method.toUpperCase()
      , reqType   = 'getconf'
      , jsonConf  = undefined
      , logPrefix = '--->conf: '

    // Add Context to the response instance
    res._Context = {
        'date': ctDate,
        'reqId': utils.buildReqId(ctDate)
    }
    logger.info(logPrefix + 'Req. Id=' + res._Context.reqId)

    switch (reqMethod){
        case 'POST' :
            logger.info(logPrefix + 'Req. Id=' + res._Context.reqId + ' body=' + JSON.stringify(req.body))
            if(!req.body){
                logger.info(logPrefix + 'Req. Id=' + res._Context.reqId + ' Invalid POST configuration request, Reason: req.body is undefined')
                return res.end(logPrefix + 'Req. Id=' + res._Context.reqId + ' Invalid POST configuration request, Reason: req.body is undefined')
            }
            if(!req.body.conf){
                logger.info(logPrefix + 'Req. Id=' + res._Context.reqId + ' Invalid POST configuration request, Reason: req.body.conf is undefined')
                return res.end(logPrefix + 'Req. Id=' + res._Context.reqId + ' Invalid POST configuration request, Reason: req.body.conf is undefined')
            }

            reqType = 'saveconf'     // save request
            jsonConf = req.body.conf // set the json conf to be passed to saveConfService
            break

        case 'GET' :
            logger.info(logPrefix + 'Req. Id=' + res._Context.reqId + ' query=' + JSON.stringify(req.query))
            if (   req.query
                && req.query.save
                && req.query.save.toLowerCase() === 'true'
                && req.query.conf
               ){
                reqType = 'saveconf'      // save request
                jsonConf = req.query.conf // set the json conf to be passed to saveConfService
            }
            break
    }

    switch (reqType){
        case 'saveconf' :
            exports.saveConfService(jsonConf, function (err) {
                logger.info(logPrefix + 'Req. Id=' + res._Context.reqId + ', over')
                res.json(err ? {'status': 'error', 'message': err.message} : pluginConf)
            })
            break

        default :
        case 'getconf' :
            logger.info(logPrefix + 'Req. Id=' + res._Context.reqId + ', over')
            res.json(pluginConf)
            break
    }
}

/**
 * Store  the Plugin conf to its configuration file (i.e. .../<plugin>/conf/<plugin>.js)
 * @param  {JSON}        jsonConf     Plugin Configuration
 * @param  {Function}    callback     callback(err)
 */
exports.saveConfService = function (jsonConf, callback) {

    let filename  = undefined
      , logPrefix = '--->saveConfService: '

    // PluginManager calls saveConfService without passing the conf
    if(typeof jsonConf === 'function'){
        callback = jsonConf
        jsonConf = pluginConf
    }

    // Assert jsonConf
    if (!jsonConf || jsonConf === null) return callback(new Error(logPrefix + 'empty Configuration'))
    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'conf:' + JSON.stringify(jsonConf))

    // Backup the current plugin conf.in the server's backup dir and store the new one in the plugin's /conf dir
    async.series([
        // Task 1: Backup the current plugin conf.
        function(nextTask){
            // Build the Backup filename
            filename = npsConf.backupDir + '/'
                       + dateFormat.asString('yyyy-MM-dd_hh-mm-ss', new Date())
                       + '_configurationOf_'
                       + PLUGIN_NAME + '.js'
            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Backuping plugin configuration to ' + filename)

            // Build the plugin conf content
            let data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t')

            // Write data to the Plugin configuration file
            fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
                // log error & continue
                if (err) logger.error(logPrefix + 'Unable to write ' + filename + ', Reason: ' + err.message)
                nextTask()
            })
        },
        // Task 2: Save the new plugin conf.
        // we should have here a proper conf, so overwrite the current pluginConf file
        function(nextTask){
            pluginConf = jsonConf

            // Build the plugin configuration filename
            filename = path.resolve(__dirname + '/conf/' + PLUGIN_NAME + '.js')

            // Build the plugin conf content
            let data = 'module.exports = ' + JSON.stringify(pluginConf, null, '\t')

            // Write data to Plugin configuration file
            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Writing plugin configuration to ' + filename)
            fs.writeFile(filename, data, {'encoding': 'utf8'}, function(err){
                if (err) return nextTask(new Error(logPrefix + 'Unable to write ' + filename + ', Reason: ' + err.message))
                else return nextTask()
            })
        }
    ],
    function(err){
        return callback(err)
    })
}

/**
 * For testing oxsnps-gcpubusb message ordering
 *
 * @param    {Function}   callback       callback(err)
 */
exports.gcqPubSubTest = function(res, callback){
    
    let msgData        = undefined
      , message        = undefined
      , orgMsgData     = undefined
      , logPrefix      = '--->gcqPubSubTest: Req.Id=' + res._Context.reqId + ', '
	  , filename		= ''
	  , extname			= ''

    if(res._Context && res._Context.message){
        message = res._Context.message
        if(message.data) orgMsgData = message.data.toString()
        if(logger.isDebugEnabled()) logger.debug(logPrefix + 'message Id: ' + message.id + ', Data: ' + orgMsgData)
    }
    
    callback() // Call back right away so that the oxsnps-gcpubsub won't wait

    // Parse message payload data
    if(orgMsgData) msgData = utils.parseJSON(orgMsgData)

    if(!msgData){
        logger.error(logPrefix + 'Unable to parse message data as JSON. Data: ' + orgMsgData)
        setTimeout(() => {message.ack()}, 1 * 1000)
        return
    }

	let processingTime = msgData.processingTime || 5*1000
	
	if(msgData.filename){
		filename = path.basename(msgData.filename)
		extname = path.extname(filename)
		if(extname && extname.toLowerCase() === '.pdf') processingTime = 120*1000
	}
		
	logger.debug(logPrefix + 'Start mid: ' + message.id + ', filename ' + filename + ', extname: ' + extname + ', orderingKey: ' + res._Context.orderingKey + ', processingTime: ' + processingTime + ', msgData: ' + JSON.stringify(msgData))
	
	setTimeout(
		function(){
			if( (extname && extname.toLowerCase() === 'pdf1') ||(extname && extname.toLowerCase() === '.pdf1')){
				message.nack()
				logger.debug(logPrefix + 'Nacked,  mid: ' + message.id + ', filename: ' + filename)
			}
			else {
				try{
					fs.renameSync(msgData.pubfilename, msgData.pubfilename+'.done')
				}catch(err){
					logger.error(logPrefix + err.message)
				}
				message.ack()
				logger.debug(logPrefix + 'Acked,  mid: ' + message.id + ', filename: ' + filename)
			}
			logger.debug(logPrefix + 'Finish, mid: ' + message.id + ', filename ' + filename + ', orderingKey: ' + res._Context.orderingKey + ', processingTime: ' + processingTime )
			
		},
		processingTime 
	)
}

/**
 * GC pub/sub API to import a stack
 *
 * @param    {Object}     res            JSON Object
 *                                       res._Context: {
 *                                           route: <route>, 
 *                                           message : {
 *                                              "id":   <message id>,
 *                                              "data": {
 *                                                  "event":          <Event that triggered this method, Ex: RNTO means ReNameTo>,
 *                                                  "fqpath":         <Fully qualified input stack dir, Ex: /mnt/shared/wackler/support.wackler/import/cargoline/job1_uploaded>,
 *                                                  "patternMatches": <Array of pattern matched path elements, Ex: ["import/cargoline/job1_uploaded","cargoline", "job1","_uploaded"]>,
 *                                                  "channel":        <channel as defined in config server>
 *                                              }
 *                                           }
 *                                       }
 * @param    {Function}   callback       callback(err)
 */
exports.gcqImportStack = function(res, callback){
    
    let ctDate         = new Date()
      , msgData        = undefined
      , message        = undefined
      , channel        = undefined
      , tenant         = ''

      , orgMsgData     = undefined
      , resTmp         = undefined

      , service        = 'importStack'
      , serviceContext = service.toLowerCase()

      // V1.0.3b Begin
      , triggerEvent   = 'rnto'  // default import trigger event
      , importEntity   = 'stack' // default import entity
      // V1.0.3b End
	  , ackFromWF	   = false // V1.0.13	

      , logPrefix      = '--->gcqImportStack: Req.Id=' + res._Context.reqId + ', '

    if(res._Context && res._Context.message){
        message = res._Context.message
        if(message.data) orgMsgData = message.data.toString()
        if(logger.isDebugEnabled()) logger.debug(logPrefix + 'message Id: ' + message.id + ', Data: ' + orgMsgData)
    }
    
    callback() // Call back right away so that the oxsnps-gcpubsub won't wait

    // Parse message payload data
    if(orgMsgData) msgData = utils.parseJSON(orgMsgData)
	// V1.0.13 Begin
    if(!msgData){
        logger.error(logPrefix + 'Unable to parse message data as JSON. Data: ' + orgMsgData)
        setTimeout(() => {message.ack()}, 1 * 1000)
        return
    }

	if(msgData && msgData.channel && msgData.channel.gcpubsub && msgData.channel.gcpubsub.ackFromWF) ackFromWF = true 
	if(!ackFromWF){
		// V1.0.5 Begin
		// Tasks are now queued in async.queue per channel basis and when the scheduling takes a while
		// from the queue, pub/sub re-issues the same message as ack time elapsed.
		// To avoid this scenario, ensure to ack with a delay of 1 second
		//
		// Acknowledge the message
		setTimeout(() => {message.ack()}, 1 * 1000)
		// V1.0.5 End
	}
	// V1.0.13 End
	
	
    // V1.0.3b Begin
    channel = msgData.channel
    tenant  = channel.tenant
    if(   msgData.event && msgData.event.toLowerCase() === 'stor' // FTP message
       || channel.event && channel.event.toLowerCase() === 'stor' // SFTP message
      ){
        // V1.0.4 Begin
        if(channel.multipleInputs){
            triggerEvent   = 'stor'
            importEntity   = 'file'
            service        = 'importMultipleFiles'
            serviceContext = service.toLowerCase()
        }
        else {
            triggerEvent   = 'stor'
            importEntity   = 'file'
            service        = 'importFile'
            serviceContext = service.toLowerCase()
        }
        // V1.0.4 End
    }

    // Build new res context with only required properties
    resTmp = {
      '_Context' :{
            'tenant':        tenant,
            'path':          tenant + '_' + (channel.subscription || channel.path),
            'date':          res._Context.date || ctDate,
            'reqId':         res._Context.reqId || utils.buildReqId(res._Context.date || ctDate),
            'service':       service,
			'redisKey':      res._Context.redisKey	// V1.0.13 
        }
    }

    // Create a separate JSON object to store values so that it does
    // not mingle with other service's context values
    resTmp._Context[serviceContext] = {
        'channelName':      channel.name  ? channel.name : '',
        'source':           channel.topic ? channel.topic : '',
        'route': {
            'tenant':       tenant,
            'path':         channel.subscription,
            'subscription': channel.subscription,
            'method':       channel.method || 'FTP' // !! todo: remove 'ftp' once oxsnps-ftpserver plugin uses oxsnps-channel 
        },
        'msgData':          msgData,
        'rootDir':          msgData.ftpDir // Ex: /mnt/shared/wackler/support.wackler
    }
    logger.info(logPrefix + 'Calling service ' + service + ', resTmp._Context: ' + JSON.stringify(resTmp._Context))
    exports[service](resTmp, function(err){
        if(err) {
            logger.error('Importing ' + importEntity + ': ' + msgData.ftpDir + ' failed, Reason: ' + err.message + ', rediskey: ' + res._Context.redisKey)
            // V1.0.14: In case of technical errors, acknowledge the message in the plugin itself, since OS process has not been created or process might not be not in a proper state to ack the msg from workflow               
	    	if(ackFromWF) setTimeout(() => {message.ack()}, 1 * 1000)
		}
        else logger.info('Initiated ' + importEntity + ': ' + msgData.ftpDir + ' processing, Workflow will take care of import ...')
    })
    // V1.0.3b End
}

// V1.0.3b Begin
/**
 * Import stack service
 * @param    {Object}      res         res object
 *                                     res._Context: {
 *                                         'importstack':{
 *                                             'channelName': <Channel name>,
 *                                             'source':      <Channel topic>,
 *                                             'route':       <route configuration>,
 *                                             'msgData':     <Message payload data>
 *                                         }
 *                                     }
 * @param    {Function}    callback    callback(err)
 */
exports.importStack = function(res, callback){

    let service        = res._Context.service
      , serviceContext = service.toLowerCase()
      , srvCtxt        = res._Context[serviceContext]
      , srvHandle      = serviceHandles[service]

      , msgData        = srvCtxt.msgData
      , stackDir       = msgData.fqpath
      , stackId        = path.basename(stackDir)

      , logPrefix      = '--->' + service + ': Req.Id=' + res._Context.reqId + ', '

    // Instantiate service
    if(srvHandle === undefined) return callback(new Error('Unknown service: ' + service + ' to process synchronous'))
    srvHandle.setLogLevel(pluginConf.log.level)

    // Add service specific attributes
    srvCtxt.stackDir       = stackDir
    res._Context.stackId   = stackId
    res._Context.titleInfo = 'stackDir: ' + stackDir
    res._Context.tempDir   = path.resolve(npsTempDir + FILE_SEP + res._Context.reqId)

    logger.info(logPrefix + 'res._Context: ' + JSON.stringify(res._Context))

    srvHandle.initialize(function(err){
        if(err) return callback(err)

        srvHandle.process(res, function(err){
            if(err) logger.error(logPrefix + 'failed, Reason: ' + err.message)
            else logger.info(logPrefix + 'over')

            return callback(err)
        })
    })
}

/**
 * Import file service
 * @param    {Object}      res         res object
 *                                     res._Context: {
 *                                         'importfile':{
 *                                             'channelName': <Channel name>,
 *                                             'source':      <Channel topic>,
 *                                             'route':       <route configuration>,
 *                                             'msgData':     <Message payload data>
 *                                         }
 *                                     }
 * @param    {Function}    callback    callback(err)
 */
exports.importFile = function(res, callback){

    let service        = res._Context.service
      , serviceContext = service.toLowerCase()
      , srvCtxt        = res._Context[serviceContext]
      , srvHandle      = serviceHandles[service]

      , msgData        = srvCtxt.msgData
      , filename       = msgData.filename
      , pubFilename    = msgData.pubfilename
      , stackDir       = path.dirname(filename)
      , stackFile      = path.basename(filename)
      , stackId        = path.basename(filename, path.extname(filename))

      , logPrefix      = '--->' + service + ': Req.Id=' + res._Context.reqId + ', '

    // Instantiate service
    if(srvHandle === undefined) return callback(new Error('Unknown service: ' + service + ' to process synchronous'))
    srvHandle.setLogLevel(pluginConf.log.level)

    // Add service specific attributes
    srvCtxt.stackDir       = stackDir
    srvCtxt.fqFilename     = filename
    srvCtxt.fqPubFilename  = pubFilename
    res._Context.stackId   = stackId
    res._Context.titleInfo = 'stackFile: ' + stackFile
    res._Context.tempDir   = path.resolve(npsTempDir + FILE_SEP + res._Context.reqId)

    logger.info(logPrefix + 'res._Context: ' + JSON.stringify(res._Context))

    srvHandle.initialize(function(err){
        if(err) return callback(err)

        srvHandle.process(res, function(err){
            if(err) logger.error(logPrefix + 'failed, Reason: ' + err.message)
            else logger.info(logPrefix + 'over')

            return callback(err)
        })
    })
}
// V1.0.3b End

// V1.0.4 Begin
/**
 * Import multiple files service
 * @param    {Object}      res         res object
 *                                     res._Context: {
 *                                         'importmultiplefiles':{
 *                                             'channelName': <Channel name>,
 *                                             'source':      <Channel topic>,
 *                                             'route':       <route configuration>,
 *                                             'msgData':     <Message payload data>
 *                                         }
 *                                     }
 * @param    {Function}    callback    callback(err)
 */
exports.importMultipleFiles = function(res, callback){

    // V1.0.5a Begin
    let service        = res._Context.service
      , serviceContext = service.toLowerCase()
      , srvCtxt        = res._Context[serviceContext]

      , msgData        = srvCtxt.msgData
      , filename       = msgData.filename
      , stackFile      = path.basename(filename)

      , multiInQueue   = undefined

      , logPrefix      = '--->' + service + ': Req.Id=' + res._Context.reqId + ', '

    logger.info(logPrefix + 'Parms: ' + JSON.stringify(srvCtxt))
    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Importing input: ' + stackFile + ' in synchronized way ..., Parms: ' + JSON.stringify(srvCtxt))

    // Get the input serializer queue for current group
    res._Context.groupId = path.basename(filename, path.extname(filename))
    multiInQueue = multipleInputQueues[srvCtxt.channelName].queue
    if(!multiInQueue){
        let errMsg   = 'Importing input: ' + stackFile + ' failed, Reason: Queue to serialize inputs could not be created'
          , allocErr = new Error(errMsg)
        logger.error(logPrefix + errMsg)

        return callback(allocErr)
    }

    // Push the task (res) to the queue
    multiInQueue.push({'res': res}, (err) => {
        if(err) logger.error(logPrefix + 'Importing input: ' + stackFile + ' failed, Reason: ' + err.message)
        else logger.info(logPrefix + 'over')
        return callback(err)
    })
    // V1.0.5a End
}
// V1.0.4 End

// V1.0.8b Begin
/**
 * [POST|GET] HTTP request to reset multiple input group cache
 *
 * @param    {Request}    req    Request Object
 *                               req.body || req.query:{
 *                                   'groupid':     <Unique group identifier of multiple inputs>
 *                               }
 * @param    {Response}   res    Response Object
 *                               Returns 200 in case of success and not in case of error
 *
 */
exports.resetMultipleInputGroupCache = function(req, res){

    let service        = 'resetMultipleInputGroupCache'
      , serviceContext = service.toLowerCase()
      , resTmp         = _createRequestContext(req, service)
      , logPrefix      = '--->' + service + ': Req.Id=' + resTmp._Context.reqId + ', '

    logger.info(logPrefix + 'Resetting multiple input group cache with parms: ' + JSON.stringify(resTmp._Context[serviceContext].parms))

    exports.resetMultipleInputGroupCacheService(resTmp, function(err, result){
        if(err){
            logger.error(logPrefix + err.message)
            return res.status(err.code || 404).end('Req.Id=' + resTmp._Context.reqId + ', ' + err.message)
        }

        logger.info(logPrefix + 'Resetting multiple input group cache over')
        return res.end(logPrefix + 'over')
    })
}

/**
 * Service to reset multiple input group cache
 * @param    {Response}   res         Response Object
 *                                    res._Context: {
 *                                        'resetmultipleinputgroupcache': {
 *                                            'parms': {
 *                                                'groupid':     <Unique group identifier of multiple inputs>
 *                                            }
 *                                        }
 *                                    }
 * @param    {Function}   callback    callback(err)
 *
 */
exports.resetMultipleInputGroupCacheService = function(res, callback){
    let service           = 'resetMultipleInputGroupCache'
      , serviceLC         = service.toLowerCase()
      , rmigContext       = res._Context[serviceLC]
      , parms             = rmigContext.parms
      , groupid           = parms.groupid
      , logPrefix         = '--->' + service + 'Service: Req.Id=' + res._Context.reqId + ', '
      , srvMultiInput     = serviceHandles['importMultipleFiles']
      , self              = this

    if(!self._initialized) return callback(new Error(PLUGIN_NAME + ' plugin is not initialized. ' + (initError ? initError.message:'')))
    if(!srvMultiInput) return callback() // service not loaded, just return
    
    // Check and get key to delete in the cache
    if (groupid){
        srvMultiInput.processInfoCache.del(groupid)
        if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Cleared multiple input group: ' + groupid + ' from cache')
    }
    else {
        let allKeys = srvMultiInput.processInfoCache.keys()
        srvMultiInput.processInfoCache.del(allKeys)
        if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Cleared all multiple input groups: ' + allKeys.join(', ') + ' from cache')
    }

    return callback()
}
// V1.0.8b End

/************ OPTIONAL PUBLIC FUNCTIONS ***********/
/**
 * Set Log Level
 * @param {String} logLevel Log Level. values:DEBUG|INFO|WARN|ERROR
 */
exports.setLogLevel = function (logLevel) {
    /*
     "log": {
     // Available Log Levels are DEBUG, INFO, WARN and ERROR
     "levels": {
     "[all]": "INFO"
     },
     */
    pluginConf.log.level = logLevel || 'INFO'
    utils.log4jsSetLogLevel(logger, pluginConf.log.level)

    // Set Log level for services
    let services = Object.keys(serviceHandles)

    // Loop thru the list of services and set log level
    async.eachOfLimit(services, 1, function(service, index, next){
        try{
            let module = serviceHandles[service]
            if(!module) return next()
            if(typeof(module.setLogLevel) !== 'function') return next()
            module.setLogLevel(pluginConf.log.level)
            return next()
        }
        catch(err){
            logger.warn('--->setLogLevel: Unable to set log level: ' + pluginConf.log.level + ' in service: ' + service + ', Reason: ' + err.message)
            return next()
        }
    },
    function(err){
        // do nothing
        return
    })
}

/**
 * Get Log Level
 * @return {String} Log Level. values:DEBUG|INFO|WARN|ERROR
 */
exports.getLogLevel = function () {
    return pluginConf.log.level
}

/**
 * Enable the given route
 * @param  {Request}    req
 *                      req.body = {
 *                          module: 'sample,
 *                          route: {
 *                              path:         '/path/to/the/service',
 *                              method:     'get|post',
 *                              service:     'version'
 *                          }
 *                      }
 * @param  {function}    callback callback(err)
 */
exports.enableRoute = function (req, callback) {

    let self   = this
      , resTmp = undefined

    if(logger.isDebugEnabled()) logger.debug('--->enableRoute: Route=' + JSON.stringify(req.body.route))
    if(!req.body.route) return callback(new Error('route parameter not given'))
    req.body.route.method = req.body.route.method || 'get'

    // Enable the route
    switch(req.body.route.method.toLowerCase()){
        case 'gcpubsub':
            resTmp = {
                '_Context': {
                    'reqId':    utils.buildReqId(new Date()),
                    'gcpubsub': { 'routes': [req.body.route]}
                }
            }
            // Add pubsub Route
            pluginManager.callPluginService({name:'oxsnps-gcpubsub', service:'addListenersService'}, resTmp, callback)
            break

        default:
            // Enable HTTP Route
            expressAppUtils.enableRoute(self, req.body.route, callback)
    }
}

/**
 * Disable the given route
 * @param  {Request}    req
 *                        req.body = {
 *                          module:'sample',
 *                          route: {
 *                              path: '/path/to/the/service'
 *                               }
 *                         }
 * @param  {function}    callback  callback(err)
 */
exports.disableRoute = function (req, callback) {

    let self   = this
      , resTmp = undefined

    if(logger.isDebugEnabled()) logger.debug('--->disableRoute: Route=' + JSON.stringify(req.body.route))
    if(!req.body.route) return callback(new Error('route parameter not given'))
    req.body.route.method = req.body.route.method || 'get'

    // Disable the route
    switch(req.body.route.method.toLowerCase()){
        case 'gcpubsub':
            resTmp = {
                '_Context': {
                    'reqId':    utils.buildReqId(new Date()),
                    'gcpubsub': { 'routes': [req.body.route]}
                }
            }

            // Disable pubsub Route
            pluginManager.callPluginService({name:'oxsnps-gcpubsub', service:'removeListenersService'}, resTmp, function(err){
                if(err) logger.error('--->disableRoute: Removing pubsub listener failed, Reason: ' + err.message)
                return callback(err)
            })
            break

        default:
            expressAppUtils.disableRoute(self, req.body.route, callback)
    }
}

/**************** PRIVATE FUNCTIONS ***************/
// V1.0.5a Begin
/**
 * Group multiple files and import them
 * @param    {Object}      res         res object
 *                                     res._Context: {
 *                                         'groupId':         <Unique id for the grouping of multiple inputs>,
 *                                         'importmultiplefiles':{
 *                                             'channelName': <Channel name>,
 *                                             'source':      <Channel topic>,
 *                                             'route':       <route configuration>,
 *                                             'msgData':     <Message payload data>
 *                                         }
 *                                     }
 * @param    {Function}    callback    callback(err)
 */
function _groupAndImportMultipleFiles(res, callback){

    let service        = res._Context.service
      , serviceContext = service.toLowerCase()
      , srvCtxt        = res._Context[serviceContext]

      , msgData        = srvCtxt.msgData
      , channel        = msgData.channel
      , filename       = msgData.filename
      , stackFile      = path.basename(filename)

      , groupId        = res._Context.groupId
      , lockSimplename = [res._Context.tenant, srvCtxt.channelName, groupId].join('_')
      , lockFile       = path.join(npsTempDir, lockSimplename)
      , fileExists     = false

      , retry          = channel.retry || {}
      , lockRetry      = retry.lockfile || {}

      , logPrefix      = '--->_groupAndImportMultipleFiles: Req.Id=' + res._Context.reqId + ', '

    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Detecting input for the group: ' + groupId + ' already arrived or not ...')
    srvCtxt.lockFile = lockFile
    res._Context.lockFile = lockFile

    // Fill in defaults for the lock retry
    if(!lockRetry.retries) lockRetry.retries = 5
    if(!lockRetry.factor) lockRetry.factor = 2
    if(!lockRetry.minTimeout) lockRetry.minTimeout = 3 * 1000
    if(!lockRetry.maxTimeout) lockRetry.maxTimeout = 9 * 1000

    async.series([
        // Check whether file already exists or not
        function(nextTask){
            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Checking lock file: ' + lockFile + ' exists or not ...')
            utils.fileExists(lockFile, (err, exists) => {
                if(err){
                    let errMsg = 'Could not check whether lock file: ' + lockFile + ' exists or not, Reason: ' + err.message
                    if(logger.isDebugEnabled()) logger.debug(logPrefix + errMsg)
                    return nextTask(new Error(errMsg))
                }

                fileExists = exists
                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Lock file: ' + lockFile + ' exists: ' + exists)
                return nextTask()
            })
        },
        // Check and start import
        function(nextTask){
            // Following input of this group, initiate import after a while (30s by default or configured via channel.multipleInputsDelay)
            if(fileExists){
                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Stack file: ' + stackFile + ' is the following file of the group: ' + groupId)
                return nextTask()
            }
            // First input of this group, initiate import immediately
            else {
                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Stack file: ' + stackFile + ' is the first file of the group: ' + groupId)
                // Create the lock file for given input's group
                _createLockFile(res, nextTask)
            }

        },
        // Import the input
        function(nextTask){
            // Blocking the import tasks using the file lock
            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Locking the import tasks using the file: ' + lockFile + ' lock ...')
            taskBlocker.lock(lockFile, {'retries': lockRetry})
            .then((release) => {
                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Locked the import tasks using the file: ' + lockFile + ' and starting the import')
                _importMultipleFiles(res, (importErr) => {
                    // First release the file lock
                    release()
                    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Unlocked the file: ' + lockFile)

                    if(importErr && logger.isDebugEnabled()) logger.debug(logPrefix + 'Importing the stack file: ' + stackFile + ' failed, Reason: ' + importErr.message)
                    return nextTask(importErr)
                })
            })
            .catch((err) => {
                return nextTask(err)
            })
        }
    ],
    function(err){
        if(!err && logger.isDebugEnabled()) logger.debug(logPrefix + 'Detected input of the group: ' + groupId + ' and delegated importing to workflow')
        return callback(err)
    })
}
// V1.0.5a End

// V1.0.4 Begin
/**
 * Create lock file
 *
 * @param    {Object}      res         res object
 *                                     res._Context: {
 *                                         'groupId':         <Unique id for the grouping of multiple inputs>,
 *                                         'lockFile':        <Group specific lock filename>
 *                                     }
 *
 * @param    {Function}    callback    callback(err)
 *
 */
function _createLockFile(res, callback){

    let groupId  = res._Context.groupId
      , filename = res._Context.lockFile

      , logPrefix = '--->_createLockFile: Req. id=' + res._Context.reqId + ', '

    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Creating a lock file: ' + filename + ' for group id: ' + groupId + ' ...')
    fs.writeFile(filename, '', (err) => {
        if(err){
            let errMsg = 'Could not create lock file: ' + filename + ', Reason: ' + err.message
            if(logger.isDebugEnabled()) logger.debug(logPrefix + errMsg)
            return callback(new Error(errMsg))
        }

        if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Created lock file: ' + filename + ' for group id: ' + groupId)
        return callback()
    })
}

/**
 * Import multiple files service (low level API)
 * @param    {Object}      res         res object
 *                                     res._Context: {
 *                                         'importmultiplefiles':{
 *                                             'channelName': <Channel name>,
 *                                             'source':      <Channel topic>,
 *                                             'route':       <route configuration>,
 *                                             'msgData':     <Message payload data>
 *                                         }
 *                                     }
 * @param    {Function}    callback    callback(err)
 */
function _importMultipleFiles(res, callback){

    // V1.0.5a Begin
    let service        = res._Context.service
      , serviceContext = service.toLowerCase()
      , srvCtxt        = res._Context[serviceContext]
      , srvHandle      = serviceHandles[service]

      , msgData        = srvCtxt.msgData
      , stackFile      = path.basename(msgData.filename)

      , logPrefix      = '--->_importMultipleFiles: Req.Id=' + res._Context.reqId + ', '

    // Instantiate service
    if(srvHandle === undefined) return callback(new Error('Unknown service: ' + service + ' to process synchronous'))
    srvHandle.setLogLevel(pluginConf.log.level)

    // Add service specific attributes
    srvCtxt.stackDir       = path.dirname(msgData.filename)
    srvCtxt.fqFilename     = msgData.filename
    srvCtxt.fqPubFilename  = msgData.pubfilename
    res._Context.stackId   = path.basename(msgData.filename, path.extname(msgData.filename))
    res._Context.titleInfo = 'stackFile: ' + stackFile
    res._Context.tempDir   = path.resolve(npsTempDir + FILE_SEP + res._Context.reqId)

    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'res._Context: ' + JSON.stringify(res._Context))

    srvHandle.initialize(function(err){
        if(err) return callback(err)

        srvHandle.process(res, function(err){
            
            if(err) logger.error(logPrefix + 'failed, Reason: ' + err.message)
            else if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Imported input: ' + stackFile + ' in synchronized way')

            return callback(err)
        })
    })
    // V1.0.5a End
}
// V1.0.4 End

// V1.0.8b Begin
/**
 * _createRequestContext
 *
 * Create request context for a service
 *
 * @param    {Request}    req     Request Object
 *                                req.body || req.query:{
 *                                  'plugin':   <Plugin name, whose jobs should be cleaned up>,
*                                   'pattern':  [OPTIONAL] <Pattern to filter jobs>,
 *                                  'jobstate': [OPTIONAL] <Specific state of jobs that should be cleaned up>
 *                                }
 * @returns    {Object}    res    Request context
 *                                res._Context: {
 *                                    'date':                    <current date>,
 *                                    'reqId':                <Request id>,
 *                                    'route':                <request route info: path, method etc>,
 *                                    'path':                    <request url path>,
 *                                    <service context>:    {
 *                                        'headers':            <request header parameters>,
 *                                        'parms':            <request query/body parameters>,
 *                                        'authorization':    <request header given authorization>
 *                                    }
 *                                }
 */
function _createRequestContext(req, service){

    let ctDate          = new Date()
      , reqId           = utils.buildReqId(ctDate)
      , parms           = httpUtils.getRequestParms(req)
      , headers         = req.headers
      , serviceContext  = service.toLowerCase()
      , resTmp          = {}

    // Create request context
    resTmp._Context = {
        'date':          ctDate,
        'reqId':         reqId,
        'tenant':        parms.tenant,
        'route':{
            'path':      req.url,
            'method':    req.method || 'POST'
        },
        'path':          url.parse(req.url).pathname, // extract the path from url, i.e. services/<plugin name>/...
    }

    // Create service specific context (a separate JSON object)
    resTmp._Context[serviceContext] = {
        'headers'       : headers,
        'parms'         : parms,
        'authorization' : headers['Authorization'] || undefined
    }

    return resTmp
}
// V1.0.8b End
