/**-----------------------------------------------------------------------------
 * importMultipleFiles.js:
 *
 * Service to create multiple files process and let workflow control the import
 * 1. Channel must be defined with following attributes
 *    a. multipleInputs=true, to force multiple input processing
 *    b. wsInputHandler=<workstep name>, to set signal about the importable file
 *    c. pi_multiple_input_id_str=<unique group id>, to group the multiple inputs
 * 2. Processing will do the following
 *    a. Search orderbase for existing process
 *       - OB Query: pi_process_type_str=<process type> AND pi_multiple_input_id_str=<unique group id>
 *    b. When process found in orderbase
 *       - Set the '<wsInputHandler value>' signal in the process
 *    c. When process NOT found in orderbase
 *       - Create a process (where pi_process_type_str=<process type> and pi_multiple_input_id_str=<unique group id>)
 *       - Set the '<wsInputHandler value>' signal in the process
 *    d. Process will wait for multiple inputs and will start import only when all inputs arrived
 *
 * Author       :  AFP2web Team
 * Copyright    :  (C) 2023 by Maas Holding GmbH
 * Email        :  support@maas.de
 * 
 * History
 * 1.0.0    05.07.2023    OXS-14816: Wackler: cepra-ablieferbeleg IC: Initial release
 * Refer to ../oxsnps.js header for further changes
 *----------------------------------------------------------------------------*/
'use strict'

/**
 * letiable declaration
 * @private
 */
let packageJson       = require(__dirname + '/../package.json')
  , PLUGIN_NAME       = packageJson.name
  , plugin            = require(__dirname + '/../oxsnps.js')
  , pluginManager     = require('oxsnps-core/pluginManager')

  , serviceScript     = __filename
  , serviceScriptName = serviceScript.slice(__dirname.length + 1)
  , SERVICE_NAME      = serviceScriptName.replace(/\.js$/, "") // remove .js ext
  , serviceContext    = SERVICE_NAME.toLowerCase()

  , npsServer         = require('oxsnps-core/server')
  , npsConf           = npsServer.npsConf
  , npsTempDir        = npsServer.npsTempDir

  , dateFormat        = require('oxsnps-core/helpers/date_format')
  , utils             = require('oxsnps-core/helpers/utils')

  , serviceUtilsThe   = require(__dirname + '/serviceUtils')
  , serviceUtils      = undefined

  , fileTypeDetector  = require('file-type') // V1.0.9a Change
  , nodeCache         = require('node-cache')
  , log4js            = require('log4js')
  , logger            = log4js.getLogger(PLUGIN_NAME + ':' + SERVICE_NAME)
  , async             = require('async')
  , path              = require('path')
  , FILE_SEP          = path.sep

const DEF_PROCESS_INFO_ALIVE_PERIOD = '15m' // in minutes. Default value is 15 mins // V1.0.7b Change

// Cache to keep info about the stack processes created (over multiple inputs)
exports.processInfoCache = new nodeCache({'useClones': false})

exports.setLogLevel = function(logLevel){
    utils.log4jsSetLogLevel(logger, (logLevel || 'WARN'))
    if(logger.isWarnEnabled()) logger.warn('--->setLogLevel: plugin: ' + PLUGIN_NAME + ' service: ' + SERVICE_NAME + ' set with log level: ' + (logLevel || 'WARN'))
}

/**
 * Initialize
 */
exports.initialize = function(callback){

    let self = this

    if(self._initialized) return callback()

    if(logger.isDebugEnabled()) logger.debug('--->initializing ' + serviceScriptName + '...')

    self._initialized = true

    return callback()
}

/**
 * Process service
 * @param  {Response}    res        Response Object
 *                                  res = {
 *                                      '_Context' :{
 *                                          'importmultiplefiles':{ 
 *                                              'channelName':   <channel name>,
 *                                              'source':        <channel source>,    
 *                                              'stackDir':      <stack directory>,
 *                                              'fqFilename':    <Fully qualified filename>,
 *                                              'fqPubFilename': <Fully qualified published filename>,
 *                                              'route':         <route>,
 *                                              'msgData':       <message data>,
 *                                              'lockFile':      <Lock file used to synchronize the files of this grouped multiple inputs>
 *                                          }
 *                                      }
 *                                  }
 * @param  {Function}    callback   callback(err)
 */
exports.process = function(res, callback){

    let impMultiFiles     = res._Context[serviceContext]
      , msgData           = impMultiFiles.msgData
      , channel           = msgData.channel

      , filepath          = impMultiFiles.stackDir
      , filename          = impMultiFiles.fqFilename
      , pubFilename       = impMultiFiles.fqPubFilename
      , workFilename      = pubFilename // V1.0.11 Change

      // V1.0.7b Begin
      , awaitPeriods      = (Array.isArray(channel.awaitMultipleInputs) && channel.awaitMultipleInputs.length > 0)
                            ? channel.awaitMultipleInputs
                            : [DEF_PROCESS_INFO_ALIVE_PERIOD]
      , minAwaitPeriod    = channel.awaitMultipleInputs[0]
      , maxAwaitPeriod    = (channel.awaitMultipleInputs.length > 1) ? channel.awaitMultipleInputs[1] : minAwaitPeriod
      , waitingTime       = _getWaitPeriodInSeconds(maxAwaitPeriod)
      // V1.0.7b End

      , activeImportState = msgData && msgData.pathSuffix
      , logPrefix         = '--->' + SERVICE_NAME + '->process: Req.Id=' + res._Context.reqId + ', '

    // Instantiate service utils to keep context specific to this service
    if(!serviceUtils) serviceUtils = new serviceUtilsThe({'serviceName': SERVICE_NAME, 'plugin': this, 'pluginName': PLUGIN_NAME, 'logger': logger})

    // Add the temp Dir to the Context
    res._Context.tempDir = res._Context.tempDir || npsTempDir + FILE_SEP + res._Context.reqId

    logger.info(logPrefix + 'Creating process for stack file: ' + filename)
    
    async.series([
        // Task 1: Ensure state file has enable state and HA servers are running
        function(nextTask){
            // if no job conditions given, skip HA server assertion
            if(!channel.jobconditions) return nextTask() 

            // if job conditions.enable = off, skip HA server assertion
            if(channel.jobconditions.enable !== undefined && channel.jobconditions.enable.toLowerCase() !== 'on' ) return nextTask()
            
            // Set job conditions in 'res'            
            res._Context.assert = {}
            res._Context.assert.conditions = utils.clone(channel.jobconditions)
            res._Context.assert.module     = PLUGIN_NAME
            res._Context.assert.route      = utils.clone(impMultiFiles.route)
            res._Context.assert.tenant     = res._Context.tenant // set tenant
            pluginManager.callPluginService({name:'oxsnps-assert',service:'assertJobConditions'}, res, function(err){
                if(err) err.dsCode = 101
                return nextTask(err)
            })
        },
        // Task 2: Assert message payload data
        function(nextTask){
            let resTmp = {
                '_Context':{
                    'reqId':   res._Context.reqId,
                    'msgData': msgData
                }
            }
            _validateMessageData(resTmp, function(err){
                if(err){
                    if (!err.dsCode) err.dsCode = 102
                    return nextTask(err)
                }

                // Fill in default import trigger (last element of patternMatches)
                if(msgData.patternMatches){
                    let pmLen = msgData.patternMatches.length
                    impMultiFiles.importTrigger = msgData.patternMatches[(pmLen - 1)]
                }

                return nextTask()
            })
        },
        // Task 3: Build indexes of stack process
        function(nextTask){
            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Getting stack process indexes ...')
            _getStackIndexes(res, function(err, result){
                if(err){
                    if (!err.dsCode) err.dsCode = 103
                    return nextTask(err)
                }

                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Got stack process indexes, indexes: ' + JSON.stringify(result))
                impMultiFiles.indexes = result

                // Fill in additional indexes
                if(impMultiFiles.indexes['pi_import_trigger_str']) impMultiFiles.importTrigger = impMultiFiles.indexes['pi_import_trigger_str']
                if (impMultiFiles.importTrigger) activeImportState = impMultiFiles.importTrigger
                impMultiFiles.indexes['pi_import_dir_str'] = impMultiFiles.indexes['pi_import_dir_str'] || impMultiFiles.stackDir

                // Evaluate stack id index
                impMultiFiles.stackId = impMultiFiles.indexes['pi_stack_id_str']
                if(!impMultiFiles.stackId){
                    impMultiFiles.stackId = filename ? path.basename(filename, impMultiFiles.importTrigger) : ''
                    if(impMultiFiles.stackId) impMultiFiles.indexes['pi_stack_id_str'] = impMultiFiles.stackId
                }

                // Get the unique group id for multiple inputs
                impMultiFiles.groupId = impMultiFiles.indexes['pi_multiple_input_id_str']
                if(!impMultiFiles.groupId) return nextTask(new Error('pi_multiple_input_id_str index value is undefined, Stack process can neither be created nor be retrieved from ordersystem'))

                return nextTask()
            })
        },
        // Task 4: Rename stack file as "_work_<timestamp>"
        function(nextTask){
            let findStr    = msgData.pathSuffix
              , workSuffix = npsConf.workDirSuffix.replace(/^_+/, '')
              , replaceStr = '.' + workSuffix + '_' + dateFormat.getAsString(res._Context.date, 'YYYYMMDDHHmmss')
            logger.debug(logPrefix + 'renaming stack file "'+ pubFilename + '" with ' + replaceStr + ' suffix')

            // 1. In file event handler, rename the stack file from "<stack id>.<ext>.pub" to "<stack id>.<ext>.work_<timestamp>"
            // 2. Later based on transformation status, rename stack file as "<stack id>.<ext>.error_<timestamp>" or "<stack id>.<ext>.done_<timestamp>"
            res._Context.file = pubFilename
            serviceUtils.renameFile(res, findStr, replaceStr, function(err){
                if(err){
                    err.dsCode = 104
                    return nextTask(err)
                }

                workFilename = res._Context.renamedFile
                impMultiFiles.workFilename = workFilename
                activeImportState = npsConf.workDirSuffix
                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'renamed stack file as: ' + workFilename)
                return nextTask()
            })
        },
        // V1.0.9a Begin
        // Task 5: Detect the input format
        function(nextTask){
            // Do not detect input format, if not configured in the channel
            if(typeof(channel.detectFormat) !== 'boolean') return nextTask()
            if(!channel.detectFormat) return nextTask()

            // Detect input format, if configured in the channel
            fileTypeDetector.fromFile(impMultiFiles.workFilename)
            .then(result => {
                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Detected input: ' + impMultiFiles.workFilename + ' format, result: ' + JSON.stringify(result))
                if(result){
                    impMultiFiles.detectedFormat = result
                    impMultiFiles.detectedFormat.status_str = 'success'
                }
                else {
                    impMultiFiles.detectedFormat = {
                        'status_str': 'error',
                        'reason_str': 'Could not detect the format of the input: ' + impMultiFiles.workFilename + ', Reason: result is undefined'
                    }
                }

                return nextTask()
            })
            .catch(err => {
                let errMsg = 'Could not detect the format of the input: ' + impMultiFiles.workFilename + ', Reason: ' + err.message
                if(logger.isDebugEnabled()) logger.debug(logPrefix + errMsg)
                impMultiFiles.detectedFormat = {
                    'status_str': 'error',
                    'reason_str': errMsg
                }

                // Ignore the error (as format detection is an additional info not mandatory)
                return nextTask()
            })
        },
        // V1.0.9a End
        // Task 6: Build additional info of stack process
        function(nextTask){
            let extName    = path.extname(filename)
              , baseName   = path.basename(filename)
              //, simpleName = path.basename(filename, extName).replace(/\./g, '_') // V1.0.8a Change
              //, keyName    = simpleName + extName.toLowerCase().replace(/^\./, '_')
              , simpleName = path.basename(filename, extName).replace(/[\.,]/g, '_') // V1.0.8a Change  // V1.0.16 replace '.' ',' with underscore in the base name 
              , keyName    = simpleName + extName.toLowerCase().replace(/[\.,]/g, '_')                  // V1.0.16 replace '.' ',' with underscore in the ext name  


            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Getting stack process additional info ...')
            impMultiFiles.additionalInfo = {
                'stack_info_json': {
                    'keys_json': [keyName]
                }
            }
            let stackInfo = impMultiFiles.additionalInfo.stack_info_json
            stackInfo[keyName] = {
                'filename_str':          impMultiFiles.workFilename,
                'input_filename_str':    baseName,
                'fq_input_filename_str': filename,
                'work_filename_str':     path.basename(impMultiFiles.workFilename),
                'fq_work_filename_str':  impMultiFiles.workFilename,
                'fq_lock_filename_str':  impMultiFiles.lockFile,
                'message_json':          msgData,
                'id_str':                impMultiFiles.stackId,
                'indexes_json':          impMultiFiles.indexes,
                'format_info_json':      impMultiFiles.detectedFormat // V1.0.9a Change
            }
            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Got stack process additional info, additionalInfo: ' + JSON.stringify(impMultiFiles.additionalInfo))
            return nextTask()
        },
        // Task 7: Get jwt token for the tenant
        function(nextTask){
            // V1.0.12 Begin
            // fill res._Context.login to get JWT token
            //res._Context.login = {
            //    'tenant':   res._Context.tenant,
            //    'username': channel.username,
            //    'password': channel.password
            //}

            // By default, authorize using tech user
            res._Context.login = {'tenant': res._Context.tenant}
            if(   channel.authorizer
               && typeof(channel.authorizer) === 'string'
               && channel.authorizer.toLowerCase() !== 'techuser'
              ){
                // Authorize using channel given user (channel.authorizer=channeluser)
                res._Context.login.username = channel.username
                res._Context.login.password = channel.password
            }
            // V1.0.12 End

            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Getting auth. token, options: ' + JSON.stringify(res._Context.login))
            utils.getAuthToken(res, function(err, auth_token){    
                if(err){
                    let tokErr = new Error('Config.Server error: ' + err.message + ', parms: ' + JSON.stringify(res._Context.login))
                    tokErr.dsCode = 105
                    return nextTask(tokErr)
                }
                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'auth_token: ' + auth_token)
                res._Context.auth_token = auth_token
                return nextTask()
            })
        },
        // Task 8: Check whether stack process exists already or not
        function(nextTask){
            // Check whether process info exists in mem cache
            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Looking stack process in mem cache ...')
            let stackProcessInfo = exports.processInfoCache.get(impMultiFiles.groupId)
            if(stackProcessInfo){
                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Found stack process in mem cache, info: ' + JSON.stringify(stackProcessInfo))

                // Fill in stack process info
                res._Context.course          = '0.2'
                res._Context.orderId         = stackProcessInfo.orderId
                res._Context.processId       = stackProcessInfo.processId
                impMultiFiles.stackOrderId   = stackProcessInfo.orderId
                impMultiFiles.stackProcessId = stackProcessInfo.processId

                return nextTask()
            }

            // Get stack process from the ordersystem
            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Looking stack process in ordersystem ...')
            _getStackProcess(res, function(err){
                if(err) return nextTask(err)
                return nextTask()
            })
        },
        //1.0.16 Begin
        function(nextTask){
            if(!channel.wsInputHandler) return nextTask() // don't set any signal if input handler workstep name is defined in channel
			if(impMultiFiles.stackOrderId && impMultiFiles.stackProcessId){ 
				res._Context.serviceContextName = serviceContext
				serviceUtils.assertProcessTransition(res, function(err){
					if(err){
						// error message already logged in serviceUtils.assertProcessTransition. no need to forward the error. 
						// reset 
						res._Context.orderId = res._Context.processId = undefined
						impMultiFiles.stackOrderId = impMultiFiles.stackProcessId = undefined
					}
					return nextTask()
				})
			}
			else return nextTask()
        },
        //1.0.16 End        
        // Task 9: Create stack process
        function(nextTask){
            // If process already exists, don't create new
            if(impMultiFiles.stackOrderId && impMultiFiles.stackProcessId) return nextTask()

            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Creating process for StackId: ' + impMultiFiles.stackId)
            _createStackProcess(res, function(err){
                if (err){
                    err.dsCode = 106
                    res._Context.obError = true
                    return nextTask(err)
                }

                // V1.0.6 Begin
                //impMultiFiles.newStack = true
                if(impMultiFiles.startSignalled){
                    impMultiFiles.newStack = true
                }
                else {
                    channel.wsInputHandler = 'START'
                }
                // V1.0.6 End

                // Add process info in mem cache
                exports.processInfoCache.set(
                      impMultiFiles.groupId
                    , {'orderId': impMultiFiles.stackOrderId, 'processId': impMultiFiles.stackProcessId}
                    , waitingTime // in seconds // V1.0.7b Change
                )

                return nextTask()
            })
        },
        // Task 10: Signal to input handler (only when process already exists)
        function(nextTask){
            if(impMultiFiles.newStack) return nextTask()  // don't set any signal for newly created process
            if(!channel.wsInputHandler) return nextTask() // don't set any signal if input handler workstep name is defined in channel

            // Fill in signal info
            // Following attributes were set by _createStackProcess or by _getStackProcess
            //res._Context.course
            //res._Context.orderId
            //res._Context.processId
            res._Context.rp         = {'rp_parms': impMultiFiles.additionalInfo}
            // V1.0.6 Begin
            if(channel.syncSignals){
                if(channel.wsInputHandler.indexOf('#') > 0){
                    res._Context.transition = channel.wsInputHandler
                }
                else {
                    let channelName = impMultiFiles.channelName
                    if(impMultiFiles.indexes['pi_inputchannel_str']) channelName = impMultiFiles.indexes['pi_inputchannel_str']
                    // Default value: <tenant>_<channel name>#<transition name>
                    res._Context.transition = res._Context.tenant + '_' + channelName + '#' + channel.wsInputHandler
                    if(channel.syncSignalsQueue && channel.syncSignalsQueue.import){
                        res._Context.transition = channel.syncSignalsQueue.import + '#' + channel.wsInputHandler
                    }
                }
            }
            else {
                res._Context.transition = channel.wsInputHandler
            }
            // V1.0.6 End
            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Setting signal: ' + res._Context.transition + ' ...')

            // Set signal
            serviceUtils.setSignal(res, function(err){
                if(err){
                    res._Context.obError = true
                    return nextTask(err)
                }
                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Signal: ' + res._Context.transition + ' set')
                return nextTask()
            })
        }
    ],
    function(err){
        // Success case
        if(!err) return callback()

        // Error case
        res._Context.err = err

        // If error occurred before or when renaming import file, then don't rename import file again
        if (workFilename === undefined || activeImportState === undefined) return callback(err)

        // there is no process created, we can not set signal, so plugin has to rename the file as error
        res._Context.file = workFilename
        serviceUtils.renameFile(res, activeImportState, npsConf.errorDirSuffix, function(err){
            if(err) logger.error(logPrefix + 'Could not rename stack file. Reason:' + err.message)
            return callback(res._Context.err || err)
        })
    })
}

/**
 * Validate message payload data for the required attributes
 * @param    {Response}    res         Response Object
 *                                     res._Context = {
 *                                         'reqId':    res._Context.reqId,
 *                                         'msgData':    <Message payload data>
 *                                     }
 * @param    {Function}    callback    callback(err)
 */
function _validateMessageData(res, callback){
    
    let msgData = res._Context.msgData

    if(logger.isDebugEnabled()) logger.debug('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Validating message data ...')

    /**
     * Assert the required attributes
     * msgData.patternMatches
     * msgData.channel
     * msgData.channel.tenant
     * msgData.channel.process.indexes
     * msgData.channel.process.indexes.pi_process_type_str
     * msgData.channel.document
     * msgData.channel.document[<document type>].archiveType
     * msgData.channel.document[<document type>].indexes
     * msgData.channel.document[<document type>].indexes.document_type_str
     * ...
     */
    if(!msgData.patternMatches){
        return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing or invalid patternMatches in message data:' + JSON.stringify(msgData)))
    }
    if(!msgData.channel){
        return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing channel in message data:' + JSON.stringify(msgData)))
    }
    if(!msgData.channel.tenant){
        return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing tenant in channel:' + JSON.stringify(msgData.channel)))
    }
    if(!msgData.channel.process){
        return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing process in channel:' + JSON.stringify(msgData.channel)))
    }
    let processTypes = Object.keys(msgData.channel.process)
    if(processTypes.length <= 0){
        return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing process types in process:' + JSON.stringify(msgData.channel.process)))
    }
    logger.debug('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', channel defined process types: ' + JSON.stringify(processTypes))
    for(let pt = 0; pt < processTypes.length; pt++){
        let proctmp = msgData.channel.process[processTypes[pt]]
        if(!proctmp.indexes){
            return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing indexes in process:' + JSON.stringify(proctmp) + ' for process type: ' + processTypes[pt]))
        }
        if(!proctmp.indexes.pi_process_type_str){
            return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing pi_process_type_str in indexes:' + JSON.stringify(proctmp.indexes) + ' for process: ' + JSON.stringify(proctmp) + ' with type: ' + processTypes[pt]))
        }
    }
    if(msgData.channel.document){
        let docTypes = Object.keys(msgData.channel.document)
        if(docTypes.length <= 0){
            return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing document types in document:' + JSON.stringify(msgData.channel.document)))
        }
        logger.debug('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', channel defined document types: ' + JSON.stringify(docTypes))
        for(let dct = 0; dct < docTypes.length; dct++){
            let doctmp = msgData.channel.document[docTypes[dct]]
            if(!doctmp.archiveType){
                return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing archiveType in document:' + JSON.stringify(doctmp) + ' for document type: ' + docTypes[dct]))
            }
            if(!doctmp.indexes.document_type_str){
                return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing document_type_str in indexes:' + JSON.stringify(doctmp.indexes) + ' for document: ' + JSON.stringify(doctmp) + ' with type: ' + docTypes[dct]))
            }
        }
    }

    if(logger.isDebugEnabled()) logger.debug('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Validated message data')
    return callback()
}

/**
 * Get Stack process info (process indexes) from channel definition
 * @param  {Response}   res         Response Object 
 *                                  res._Context = {
 *                                      'importmultifiles':{ 
 *                                          'stackDir':             <stackDir>,
 *                                          'msgdata': {
 *                                                "event":          "RNTO" | "STOR",
 *                                                "fqpath":         "/mnt/shared/wackler/support.wackler/import/import_invoice/job1_uploaded",
 *                                                "patternMatches": ["import/import_invoice/job1_uploaded","import_invoice", "job1","_uploaded"],
 *                                                "channel":        <channel as defined in config server>
 *                                          }
 *                                      }
 *                                  }
 * @param  {Function}   callback    callback(err, indexes)
 */
function _getStackIndexes(res, callback){
    
    let name                = undefined
      , value               = undefined
      , expr                = undefined
      , impMultiFiles       = res._Context[serviceContext]
      , msgData             = impMultiFiles.msgData
      , channel             = msgData.channel
      , processConfig       = channel.process['stackProcess']
      , indexes             = utils.clone(processConfig.indexes)
      , pi_input_path_str_m = msgData.patternMatches
      , logPrefix           = '--->' + SERVICE_NAME + '->_getStackIndexes: Req.Id=' + res._Context.reqId + ', '

    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Getting indexes for stack process, stackFile: ' + impMultiFiles.fqFilename + ', indexes: ' + JSON.stringify(indexes))

    // iterate through process indexes and if some indexes starts with "eval", use eval() function to get the value
    for(name in indexes){
        if(!name) continue
        value = indexes[name]
        if(typeof value !== 'string' || !value.startsWith('eval:')) continue

        if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Processing stack index: ' + name + ': ' + value)
        expr = value.substring(value.indexOf(':')+1)
        try{
            indexes[name] = eval(expr)
            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Processed stack index: ' + name + ': ' + indexes[name])
        }
        catch(err){
            if(err){
                if (    err.message.includes('is not defined')
                     || err.message.includes(' of undefined')
                   ){
                    logger.warn('Unable to evaluate Javascript expression: ' + expr + ', Reason: ' + err.message)
                    indexes[name] = undefined
                }
                else {
                    return callback(new Error('Unable to evaluate Javascript expression: ' + expr + ', Reason: ' + err.message))
                }
            }
        }
    }

    // Fill in additional process indexes
    if(pi_input_path_str_m && pi_input_path_str_m.length > 0) indexes['pi_input_path_str_m'] = pi_input_path_str_m

    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Got indexes for stack process, stackFile: ' + impMultiFiles.fqFilename + ', indexes: ' + JSON.stringify(indexes))
    return callback(null, indexes)
}

/**
 * Get stack process from ordersystem
 * @param  {Response}  res       Response Object
 * @param  {function}  callback  callback(err)
 */
function _getStackProcess(res, callback){

    let impMultiFiles   = res._Context[serviceContext]
      , msgData         = impMultiFiles.msgData
      , channel         = msgData.channel
      , indexes         = impMultiFiles.indexes
      , additionalInfo  = impMultiFiles.additionalInfo

      , query           =   'pi_process_type_str=' + indexes.pi_process_type_str
                          + ' AND pi_multiple_input_id_str=' + indexes.pi_multiple_input_id_str
                          + ' AND pi_inputchannel_str=' + indexes.pi_inputchannel_str
      , expectedFields  = 'pi_order_id_str,pi_process_id_str,pi_course_id_str,pi_transition_info_str_m'

      , procInfo        = undefined
      , stackProcess    = undefined

      , osIndexQuery    = {'name': 'oxsnps-ordersystem', 'service': 'indexQueryService'}
      , osGetCourses    = {'name': 'oxsnps-ordersystem', 'service': 'getCoursesService'}
      , logPrefix       = '--->' + SERVICE_NAME + '->_getStackProcess: Req.Id=' + res._Context.reqId + ', '

    res._Context.orderParms = {
        'tenant':     res._Context.tenant,
        'auth_token': res._Context.auth_token,
        'where':      query,
        'fields':     expectedFields
    }

    async.series([
        // Query ordersystem and get stack process
        function(nextTask){
            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Querying ' + query + ' for the stack process ...')
            pluginManager.callPluginService(osIndexQuery, res, function(err, result){
                if(err) return nextTask(err)
                if(!result.orders || result.orders.length <= 0){
                    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Unable to find stack process, Reason: Invalid result: ' + JSON.stringify(result))
                    return nextTask()
                }
                let orders = result.orders
                if(!orders[0].processes || orders[0].processes.length <= 0){
                    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Unable to find stack process, Reason: No process found in result: ' + JSON.stringify(result))
                    return nextTask()
                }
                let process     = orders[0].processes[0]
                  , procIndexes = process && process.indexes
                if(!procIndexes || !procIndexes['pi_order_id_str'] || !procIndexes['pi_process_id_str']){
                    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Unable to find stack process, Reason: Expected fields: pi_order_id_str and/or pi_process_id_str not found in result: ' + JSON.stringify(result))
                    return nextTask()
                }
                procInfo = {'orderId': procIndexes['pi_order_id_str'], 'processId': procIndexes['pi_process_id_str']}
                if(procIndexes && !procIndexes['pi_transition_info_str_m']){
                    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Stack process: ' + JSON.stringify(procInfo) + ' is archived, hence set signal is not possible')
                    return nextTask()
                }

                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Query result: ' + JSON.stringify(result))
                stackProcess = process

                // Fill in info to get the course allowed transitions
                res._Context.orderParms.course = procIndexes['pi_course_id_str'] || '0.2'
                res._Context.orderParms.orderId = procIndexes['pi_order_id_str']
                res._Context.orderParms.processId = procIndexes['pi_process_id_str']

				// V1.0.16 Begin
                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Found stack process: ' + JSON.stringify(procInfo))
				procIndexes = stackProcess && stackProcess.indexes // V1.0.15 Change
				res._Context.course          = procIndexes['pi_course_id_str'] || '0.2'
				res._Context.orderId         = procIndexes['pi_order_id_str']
				res._Context.processId       = procIndexes['pi_process_id_str']
				impMultiFiles.stackOrderId   = procIndexes['pi_order_id_str']
				impMultiFiles.stackProcessId = procIndexes['pi_process_id_str']
                return nextTask()
            })
        },
        // Validate the current state of process
        function(nextTask){
			return nextTask() // Validating the current state of process is done by serviceutils.assertProcessTransition() // V1.0.16
			/*
            if(!stackProcess) return nextTask()
            if(!channel.wsInputHandler) return nextTask()

            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Getting current possible worksteps from the stack process ...')
            pluginManager.callPluginService(osGetCourses, res, function(err, result){
                if(err) return nextTask(err)
                if(   !result
                   || result.course !== res._Context.orderParms.course
                   || !result.transitions
                   || result.transitions.length <= 0){
                    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Unable to get possible worksteps from the stack process, Reason: Invalid result: ' + JSON.stringify(result))
                    return nextTask()
                }

                // Get allowed transitions
                let courseTransitions = result.transitions
                  , allowedTransitions = courseTransitions.map(t => (t.location) ? t.location.split(':')[1] : undefined).filter(t => t !== undefined)
                if(!allowedTransitions.includes(channel.wsInputHandler)){
                    let errMsg =   'Stack process course does not allow to signal: ' + channel.wsInputHandler
                                 + ', Reason: Current workstep: ' + result.state + ' does not allow transition to ' + channel.wsInputHandler
                                 + ', Allowed transitions: ' + allowedTransitions.join(',')
                    if(logger.isDebugEnabled()) logger.debug(logPrefix + errMsg)
                    return nextTask()
                }

                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Found valid stack process: ' + JSON.stringify(procInfo))
                let procIndexes = stackProcess && stackProcess.indexes // V1.0.15 Change
                res._Context.course          = procIndexes['pi_course_id_str'] || '0.2'
                res._Context.orderId         = procIndexes['pi_order_id_str']
                res._Context.processId       = procIndexes['pi_process_id_str']
                impMultiFiles.stackOrderId   = procIndexes['pi_order_id_str']
                impMultiFiles.stackProcessId = procIndexes['pi_process_id_str']
                return nextTask()
            })
            */
        }
    ],
    function(err){
        if(err){
            let errMsg = 'Could not find stack process matching the query: ' + query + ', Reason: ' + err.message
            if(logger.isDebugEnabled()) logger.debug(logPrefix + errMsg)

            // Ignore the error, caller will continue to create new process
            return callback()
        }

        if(impMultiFiles.stackOrderId && logger.isDebugEnabled()) logger.debug(logPrefix + 'Found stack process, order: ' + impMultiFiles.stackOrderId + ' process: ' + impMultiFiles.stackProcessId) // V1.0.7 Change
        return callback()
    })
}

/**
 * Create a stack process
 * @param  {Response}  res       Response Object
 * @param  {function}  callback  callback(err)
 */
function _createStackProcess(res, callback){

    let impMultiFiles   = res._Context[serviceContext]
      , process         = {
              'document':       []
            , 'indexes':        impMultiFiles.indexes
            , 'additionalInfo': impMultiFiles.additionalInfo
        }
      , impOrderService = {'name': 'oxs.import', 'service':'importJSONService'}
      , msgData         = impMultiFiles.msgData
      , channel         = msgData.channel // V1.0.6 Change
      , logPrefix       = '--->' + SERVICE_NAME + '->_createStackProcess: Req.Id=' + res._Context.reqId + ', '

    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Importing a process for the stackFile: ' + impMultiFiles.fqFilename)
    res._Context.orderParms = {
        'tenant':         res._Context.tenant,
        'processes':      [process],
        'auth_token':     res._Context.auth_token,
        'setStartSignal': (channel.setStartSignal !== undefined) ? channel.setStartSignal : true // V1.0.6 change
    }

    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Parameters: ' + JSON.stringify(res._Context.orderParms))
    pluginManager.callPluginService(impOrderService, res, function(err, result){
        if(err){
            return callback(new Error('Ordersystem error: Unable to create a stack process. Reason: ' +  err.message + ', Parms: ' + JSON.stringify(res._Context.orderParms)))
        }

        if(logger.isDebugEnabled())
            logger.debug(  logPrefix + 'Process imported for the Stack: ' + impMultiFiles.fqFilename
                         + '. OrderId: ' + res._Context.orderId
                         + ', ProcessId: ' + res._Context.processId
                        )

        impMultiFiles.stackOrderId   = res._Context.orderId
        impMultiFiles.stackProcessId = res._Context.processId
        if(!res._Context.course) res._Context.course = '0.2' // V1.0.6 change
        impMultiFiles.startSignalled = res._Context.orderParms.setStartSignal // V1.0.6 change
        return callback()
    })
}

// V1.0.7b Begin
/**
 * Get waiting period in seconds (converted from given string)
 *
 * @param    {String}    waitPeriod    String with waiting period specified with 'm' or 'h' suffixes
 *                                     - m means the values specified is in minutes (Ex: 15m means 15 minutes)
 *                                     - h means the values specified is in hours (Ex: 2h means 2 hours)
 *
 * @returns
 * waitPeriod string covverted to seconds and same is returned
 */
function _getWaitPeriodInSeconds(waitPeriod){
    let reWaitPeriod    = /^(\d+)(m|h)?$/
      // V1.0.9b Begin
      , waitPeriodArray = (waitPeriod) ? waitPeriod.match(reWaitPeriod) : [1, 'm']
      , wpaIsArray      = Array.isArray(waitPeriodArray)
      , wpaLength       = (wpaIsArray) ? waitPeriodArray.length : 0
      , waitPeriodValue = (wpaLength > 1) ? waitPeriodArray[1] : 1
      , waitPeriodUnit  = (wpaLength > 2) ? waitPeriodArray[2] : 'm'
      // V1.0.9b End

    waitPeriodUnit = waitPeriodUnit.toLowerCase()
    switch(waitPeriodUnit){
        case 'h':
            waitPeriodValue = waitPeriodValue * 3600 // (60 * 60)
            break

        default:
        case 'm':
            waitPeriodValue = waitPeriodValue * 60
            break
    }

    return waitPeriodValue
}
// V1.0.7b End
