/**-----------------------------------------------------------------------------
 * importStack.js:
 *
 * Service to create stack directory process and let workflow control the import
 *
 * Author       :  AFP2web Team
 * Copyright    :  (C) 2022-2023 by Maas Holding GmbH
 * Email        :  support@maas.de
 * 
 * History
 * Refer to ../oxsnps.js header
 *----------------------------------------------------------------------------*/
'use strict'

/**
 * letiable declaration
 * @private
 */
let packageJson       = require(__dirname + '/../package.json')
  , PLUGIN_NAME       = packageJson.name
  , plugin            = require(__dirname + '/../oxsnps.js')
  , pluginManager     = require('oxsnps-core/pluginManager')

  , serviceScript     = __filename
  , serviceScriptName = serviceScript.slice(__dirname.length + 1)
  , SERVICE_NAME      = serviceScriptName.replace(/\.js$/, "") // remove .js ext
  , serviceContext    = SERVICE_NAME.toLowerCase()

  , npsServer         = require('oxsnps-core/server')
  , npsConf           = npsServer.npsConf
  , npsTempDir        = npsServer.npsTempDir

  , dateFormat        = require('oxsnps-core/helpers/date_format')
  , utils             = require('oxsnps-core/helpers/utils')

  , serviceUtilsThe   = require(__dirname + '/serviceUtils')
  , serviceUtils      = undefined

  , log4js            = require('log4js')
  , logger            = log4js.getLogger(PLUGIN_NAME + ':' + SERVICE_NAME)
  , async             = require('async')
  , path              = require('path')
  , FILE_SEP          = path.sep

exports.setLogLevel = function(logLevel){
    utils.log4jsSetLogLevel(logger, (logLevel || 'WARN'))
    if(logger.isWarnEnabled()) logger.warn('--->setLogLevel: plugin: ' + PLUGIN_NAME + ' service: ' + SERVICE_NAME + ' set with log level: ' + (logLevel || 'WARN'))
}

/**
 * Initialize
 */
exports.initialize = function(callback){

    let self = this

    if(self._initialized) return callback()

    if(logger.isDebugEnabled()) logger.debug('--->initializing ' + serviceScriptName + '...')

    self._initialized = true

    return callback()
}

/**
 * Process service
 * @param  {Response}    res        Response Object
 *                                  res = {
 *                                      '_Context' :{
 *                                          'importStack':{ 
 *                                              'channelName': <channel name>,
 *                                              'source':      <channel source>,    
 *                                              'stackDir':    <stack directory>,
 *                                              'route':       <route>,
 *                                              'msgData':     <message data>
 *                                          }
 *                                      }
 *                                  }
 * @param  {Function}    callback   callback(err)
 */
exports.process = function(res, callback){

    let importStack       = res._Context[serviceContext]
      , msgData           = importStack.msgData
      , channel           = msgData.channel
      , spoolPath         = importStack.stackDir
      , activeImportState = (msgData && msgData.pathSuffix || npsConf.uploadedDirSuffix)
      , logPrefix         = '--->' + SERVICE_NAME + '->process: Req.Id=' + res._Context.reqId + ', '

    // Instantiate service utils to keep context specific to this service
    if(!serviceUtils) serviceUtils = new serviceUtilsThe({'serviceName': SERVICE_NAME, 'plugin': this, 'pluginName': PLUGIN_NAME, 'logger': logger})

    // Add the temp Dir to the Context
    res._Context.tempDir = res._Context.tempDir || npsTempDir + FILE_SEP + res._Context.reqId
    importStack.importTrigger = msgData.patternMatches[4] // default

    logger.info(logPrefix + 'Creating process for stack: ' + importStack.stackDir)
    
    async.series([
        // Task 1: Ensure state file has enable state and HA servers are running
        function(nextTask){
            // if no job conditions given, skip HA server assertion
            if(!channel.jobconditions) return nextTask() 

            // if job conditions.enable = off, skip HA server assertion
            if(channel.jobconditions.enable !== undefined && channel.jobconditions.enable.toLowerCase() !== 'on' ) return nextTask()
            
            // Set job conditions in 'res'            
            res._Context.assert = {}
            res._Context.assert.conditions = utils.clone(channel.jobconditions)
            res._Context.assert.module     = PLUGIN_NAME
            res._Context.assert.route      = utils.clone(importStack.route)
            res._Context.assert.tenant     = res._Context.tenant // set tenant
            pluginManager.callPluginService({name:'oxsnps-assert',service:'assertJobConditions'}, res, function(err){
                if(err) err.dsCode = 101
                return nextTask(err)
            })
        },
        // Task 2: Assert message payload data
        function(nextTask){
            let resTmp = {
                '_Context':{
                    'reqId':   res._Context.reqId,
                    'msgData': msgData
                }
            }
            _validateMessageData(resTmp, function(err){
                if(err){ if (!err.dsCode) err.dsCode = 102 }
                return nextTask(err)
            })
        },
        // Task 3: Build indexes of stack process
        function(nextTask){
            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Getting stack process indexes ...')
            _getStackIndexes(res, function(err, result){
                if(err){
                    if (!err.dsCode) err.dsCode = 103
                    return nextTask(err)
                }

                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Got stack process indexes, indexes: ' + JSON.stringify(result))
                importStack.indexes = result

                if(importStack.indexes['pi_import_trigger_str']) importStack.importTrigger = importStack.indexes['pi_import_trigger_str']
                if (importStack.importTrigger) activeImportState = importStack.importTrigger

                // Evaluate stack id index
                importStack.stackId = importStack.indexes['pi_stack_id_str']
                if(!importStack.stackId){
                    let re = new RegExp(importStack.importTrigger + '$', 'i')
                    importStack.stackId = msgData.fqpath ? path.basename(msgData.fqpath).replace(re, '') : ''
                    if(importStack.stackId) importStack.indexes['pi_stack_id_str'] = importStack.stackId
                }
                return nextTask()
            })
        },
        // Task 4: Rename stack directory as "_work_<timestamp>"
        function(nextTask){
            let workSuffix = npsConf.workDirSuffix + '_' + dateFormat.getAsString(res._Context.date, 'YYYYMMDDHHmmss')
            logger.debug(logPrefix + 'renaming stack directory "'+ importStack.stackDir + '" as ... ' + workSuffix)

            // 1. In directory event handler, rename the stack directory from "<stack id>_uploaded" to "<stack id>_work_<timestamp>"
            // 2. Later based on transformation status, rename stack directory as "<stack id>_error_<timestamp>" or "<stack id>_done_<timestamp>"
            res._Context.dir = importStack.stackDir
            serviceUtils.renameDirectory(res, importStack.importTrigger, workSuffix, function(err){
                if(err){
                    err.dsCode = 104
                    return nextTask(err)
                }
                spoolPath = res._Context.renamedDir
                res._Context.spoolPath = spoolPath
                activeImportState = npsConf.workDirSuffix
                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'renamed stack directory as: ' + spoolPath)
                return nextTask()
            })
        },
        // Task 4a: Get jwt token for the tenant
        function(nextTask){
            // V1.0.12 Begin
            // fill res._Context.login to get JWT token
            //res._Context.login = {
            //    'tenant':   res._Context.tenant,
            //    'username': channel.username,
            //    'password': channel.password
            //}

            // By default, authorize using tech user
            res._Context.login = {'tenant': res._Context.tenant}
            if(   channel.authorizer
               && typeof(channel.authorizer) === 'string'
               && channel.authorizer.toLowerCase() !== 'techuser'
              ){
                // Authorize using channel given user (channel.authorizer=channeluser)
                res._Context.login.username = channel.username
                res._Context.login.password = channel.password
            }
            // V1.0.12 End

            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Getting auth. token, options: ' + JSON.stringify(res._Context.login))
            utils.getAuthToken(res, function(err, auth_token){    
                if(err){
                    let tokErr = new Error('Config.Server error: ' + err.message + ', parms: ' + JSON.stringify(res._Context.login))
                    tokErr.dsCode = 105
                    return nextTask(tokErr)
                }
                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'auth_token: ' + auth_token)
                res._Context.auth_token = auth_token
                return nextTask()
            })
        },
        // Task 5: Create stack process
        function(nextTask){
            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Creating process for StackId: ' + importStack.stackId)
            _createStackProcess(res, function(err){
                if (err){
                    err.dsCode = 106
                    res._Context.obError = true
                    return nextTask(err)
                }
                return nextTask()
            })
        },
        // Task 6: Start the stack processing
        function(nextTask){
            if(!channel.wsInitial) return nextTask()

            // Fill in signal info
            // Following attributes were set by _createStackProcess
            //res._Context.course
            //res._Context.orderId
            //res._Context.processId
            // V1.0.6 Begin
            if(channel.syncSignals){
                if(channel.wsInitial.indexOf('#') > 0){
                    res._Context.transition = channel.wsInitial
                }
                else {
                    let channelName = importStack.channelName
                    if(importStack.indexes['pi_inputchannel_str']) channelName = importStack.indexes['pi_inputchannel_str']
                    // Default value: <tenant>_<channel name>#<transition name>
                    res._Context.transition = res._Context.tenant + '_' + channelName + '#' + channel.wsInitial
                    if(channel.syncSignalsQueue && channel.syncSignalsQueue.import){
                        res._Context.transition = channel.syncSignalsQueue.import + '#' + channel.wsInitial
                    }
                }
            }
            else {
                res._Context.transition = channel.wsInitial
            }
            // V1.0.6 End
            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Setting signal: ' + res._Context.transition + ' ...')

            // Set signal
            serviceUtils.setSignal(res, function(err){
                if(err){
                    res._Context.obError = true
                    return nextTask(err)
                }
                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Signal: ' + res._Context.transition + ' set')
                return nextTask()
            })
        }
    ],
    function(err){
        // Success case
        if(!err) return callback()

        // Error case
        res._Context.err = err

        // If error occurred before or when renaming import dir, then don't rename import dir again
        if (spoolPath === undefined || activeImportState === undefined) return callback(err)

        // there is no process created, we can not set signal, so plugin has to rename the directory as error
        res._Context.dir = spoolPath
        serviceUtils.renameDirectory(res, activeImportState, npsConf.errorDirSuffix, function(err){
            if(err) logger.error(logPrefix + 'Could not rename stack directory. Reason:' + err.message)
            return callback(res._Context.err || err)
        })
    })
}

/**
 * Validate message payload data for the required attributes
 * @param    {Response}    res         Response Object
 *                                     res._Context = {
 *                                         'reqId':    res._Context.reqId,
 *                                         'msgData':    <Message payload data>
 *                                     }
 * @param    {Function}    callback    callback(err)
 */
function _validateMessageData(res, callback){
    
    let msgData = res._Context.msgData

    if(logger.isDebugEnabled()) logger.debug('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Validating message data ...')

    /**
     * Assert the required attributes
     * msgData.patternMatches
     * msgData.channel
     * msgData.channel.tenant
     * msgData.channel.process.indexes
     * msgData.channel.process.indexes.pi_process_type_str
     * msgData.channel.document
     * msgData.channel.document[<document type>].archiveType
     * msgData.channel.document[<document type>].indexes
     * msgData.channel.document[<document type>].indexes.document_type_str
     * ...
     */
    if(!msgData.patternMatches){
        return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing or invalid patternMatches in message data:' + JSON.stringify(msgData)))
    }
    if(!msgData.channel){
        return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing channel in message data:' + JSON.stringify(msgData)))
    }
    if(!msgData.channel.tenant){
        return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing tenant in channel:' + JSON.stringify(msgData.channel)))
    }
    if(!msgData.channel.process){
        return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing process in channel:' + JSON.stringify(msgData.channel)))
    }
    let processTypes = Object.keys(msgData.channel.process)
    if(processTypes.length <= 0){
        return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing process types in process:' + JSON.stringify(msgData.channel.process)))
    }
    logger.debug('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', channel defined process types: ' + JSON.stringify(processTypes))
    for(let pt = 0; pt < processTypes.length; pt++){
        let proctmp = msgData.channel.process[processTypes[pt]]
        if(!proctmp.indexes){
            return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing indexes in process:' + JSON.stringify(proctmp) + ' for process type: ' + processTypes[pt]))
        }
        if(!proctmp.indexes.pi_process_type_str){
            return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing pi_process_type_str in indexes:' + JSON.stringify(proctmp.indexes) + ' for process: ' + JSON.stringify(proctmp) + ' with type: ' + processTypes[pt]))
        }
    }
    if(msgData.channel.document){
        let docTypes = Object.keys(msgData.channel.document)
        if(docTypes.length <= 0){
            return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing document types in document:' + JSON.stringify(msgData.channel.document)))
        }
        logger.debug('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', channel defined document types: ' + JSON.stringify(docTypes))
        for(let dct = 0; dct < docTypes.length; dct++){
            let doctmp = msgData.channel.document[docTypes[dct]]
            if(!doctmp.archiveType){
                return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing archiveType in document:' + JSON.stringify(doctmp) + ' for document type: ' + docTypes[dct]))
            }
            if(!doctmp.indexes.document_type_str){
                return callback(new Error('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Missing document_type_str in indexes:' + JSON.stringify(doctmp.indexes) + ' for document: ' + JSON.stringify(doctmp) + ' with type: ' + docTypes[dct]))
            }
        }
    }

    if(logger.isDebugEnabled()) logger.debug('--->_validateMessageData: Req.Id=' + res._Context.reqId + ', Validated message data')
    return callback()
}

/**
 * Get Stack process info (process indexes) from channel definition
 * @param  {Response}   res         Response Object 
 *                                  res._Context = {
 *                                      'importStack':{ 
 *                                          'stackDir':             <stackDir>,
 *                                          'msgdata': {
 *                                                "event":          "RNTO",
 *                                                "fqpath":         "/mnt/shared/wackler/support.wackler/import/import_invoice/job1_uploaded",
 *                                                "patternMatches": ["import/import_invoice/job1_uploaded","import_invoice", "job1","_uploaded"],
 *                                                "channel":        <channel as defined in config server>
 *                                          }
 *                                      }
 *                                  }
 * @param  {Function}   callback    callback(err, indexes)
 */
function _getStackIndexes(res, callback){
    
    let name                = undefined
      , value               = undefined
      , expr                = undefined
      , importStack         = res._Context[serviceContext]
      , msgData             = importStack.msgData
      , channel             = msgData.channel
      , processConfig       = channel.process['stackProcess']
      , indexes             = utils.clone(processConfig.indexes)
      , pi_input_path_str_m = msgData.patternMatches
      , logPrefix           = '--->' + SERVICE_NAME + '->_getStackIndexes: Req.Id=' + res._Context.reqId + ', '

    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Getting indexes for stack process, stackDir: ' + importStack.stackDir + ', indexes: ' + JSON.stringify(indexes))

    // iterate through process indexes and if some indexes starts with "eval", use eval() function to get the value
    for(name in indexes){
        if(!name) continue
        value = indexes[name]
        if(typeof value !== 'string' || !value.startsWith('eval:')) continue

        if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Processing stack index: ' + name + ': ' + value)
        expr = value.substring(value.indexOf(':')+1)
        try{
            indexes[name] = eval(expr)
            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Processed stack index: ' + name + ': ' + indexes[name])
        }
        catch(err){
            if(err){
                if (    err.message.includes('is not defined')
                     || err.message.includes(' of undefined')
                   ){
                    logger.warn('Unable to evaluate Javascript expression: ' + expr + ', Reason: ' + err.message)
                    indexes[name] = undefined
                }
                else {
                    return callback(new Error('Unable to evaluate Javascript expression: ' + expr + ', Reason: ' + err.message))
                }
            }
        }
    }

    // Fill in additional process indexes
    if(pi_input_path_str_m && pi_input_path_str_m.length > 0) indexes['pi_input_path_str_m'] = pi_input_path_str_m

    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Got indexes for stack process, stackDir: ' + importStack.stackDir + ', indexes: ' + JSON.stringify(indexes))
    return callback(null, indexes)
}

/**
 * Get stack process additional info
 * @param {Response} res      Response Object
 * @param {Function} callback callback(err)
 */
function _getStackInfo(res, callback){

    let importStack  = res._Context[serviceContext]
      , spoolPath    = res._Context.spoolPath
      , pattern      = new RegExp('.*', 'i')
      , msgData      = importStack.msgData
      , channel      = msgData.channel
      , inputFilter  = channel.inputFilter
      , inputFiles   = []
      , logPrefix    = '--->' + SERVICE_NAME + '->_getStackInfo: Req.Id=' + res._Context.reqId + ', '

    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Processing stack: ' + importStack.stackId + ', input channel directory: ' + spoolPath)
    if(!inputFilter){
        if(logger.isDebugEnabled()) logger.debug(logPrefix + 'No input filter configured in channel, skipped getting stack filenames ...')
        return callback()
    }

    utils.getFiles(spoolPath, pattern, false /*recursive=false*/, function(err, result){ // get list of files
        if(err) return callback(new Error(logPrefix + 'unable to get input filenames from "' + spoolPath + '" dir. Reason: ' + err.message))

        // Make error if input directory is empty or has no valid files to process
        if (!result || result.length <= 0){
            return callback(new Error('Unable to import the stack dir: ' + spoolPath + ', Reason: Directory is empty or No files found matching "' + JSON.stringify(pattern) + '" pattern'))
        }
        if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Stack files:' + JSON.stringify(result))

        if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Filtering stack files based on inputFilter: ' + JSON.stringify(inputFilter))
        //importStack.spoolFilesList = result
        for( let x = 0; x < inputFilter.length; x++ ){
            let fnPattern = new RegExp(inputFilter[x], 'i')
              , fnList    = result.filter(fn => fnPattern.test(fn))

            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Filtered input files: ' + JSON.stringify(fnList) + ' using filter: ' + inputFilter[x])
            if(fnList && fnList.length > 0) inputFiles = inputFiles.concat(fnList)
        }
        if(inputFiles.length <= 0) return callback(new Error("No files found in stack: " + spoolPath))

        importStack.spoolFilesList = inputFiles

        if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Input files=' + JSON.stringify(importStack.spoolFilesList))
        return callback()
    })
}

/**
 * Create a stack process
 * @param  {Response}  res       Response Object
 * @param  {function}  callback  callback(err)
 */
function _createStackProcess(res, callback){

    let importStack  = res._Context[serviceContext]
      , process      = {'document': [], 'indexes': importStack.indexes}
      , msgData      = importStack.msgData
      , channel      = msgData.channel // V1.0.6 Change
      , logPrefix    = '--->' + SERVICE_NAME + '->_createStackProcess: Req.Id=' + res._Context.reqId + ', '

    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Importing a process for the stackdir: ' + importStack.stackDir)

    async.series([
        // Task 1: Build additional info of stack process
        function(nextTask){
            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Getting stack process additional info ...')

            // Fill in additional indexes
            process.indexes['pi_import_dir_str'] = process.indexes['pi_import_dir_str'] || res._Context.spoolPath

            // Get stack info
            _getStackInfo(res,function(err){
                if(err) return nextTask(err)

                process.additionalInfo = {
                    'stack_info_json': {
                        'path_str':        res._Context.spoolPath,
                        'files_str_m':     importStack.spoolFilesList,
                        'files_count_int': (importStack.spoolFilesList) ? importStack.spoolFilesList.length : undefined,
                        'message_json':    msgData,
                        'id_str':          importStack.stackId
                    }
                }
                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Got stack process additional info, additionalInfo: ' + JSON.stringify(process.additionalInfo))

                return nextTask()
            })
        },
        // Task 3: Create the stack process
        function(nextTask){
            res._Context.orderParms = {
                'tenant':         res._Context.tenant,
                'processes':      [process],
                'auth_token':     res._Context.auth_token,
                'setStartSignal': (channel.setStartSignal !== undefined) ? channel.setStartSignal : true // V1.0.6 change
            }

            // Create stack process
            if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Creating a process with parms:' + JSON.stringify(res._Context.orderParms))
            pluginManager.callPluginService({'name': 'oxs.import', 'service':'importJSONService'}, res, function(err, result){
                if(err) return nextTask(new Error('Ordersystem error: Unable to create a stack process. ' +  err.message + ', Stackdir: ' + importStack.stackDir + ', Parms: ' + JSON.stringify(res._Context.orderParms)))

                if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Process created for the Stack: ' + importStack.stackDir +
                                                         '. OrderId: ' + res._Context.orderId +
                                                         ', ProcessId: ' + res._Context.processId)

                importStack.stackOrderId   = res._Context.orderId
                importStack.stackProcessId = res._Context.processId
                if(!res._Context.course) res._Context.course = '0.2' // V1.0.6 change
                return nextTask()
            })
        }
    ],
    function(err){
        return callback(err)
    })
}