/**-----------------------------------------------------------------------------
 * serviceUtils.js: Utility functions for services
 *
 * Author	:  AFP2web Team
 * Copyright :  (C) 2019-2021 by Maas Holding GmbH
 * Email	 :  support@oxseed.de
 * 
 * History
 * V107 25.03.2021  OXS-11200: Extended to retry service requests based on "retry" config in channel def
 * V106	21.07.2020  OXS-11032: Extended to process possible documents in case of functional error
 * V105 17.07.2020  Fixed minor bug in setting wsError signal when PDF spool could not be archived (due to size limit)
 * V104	07.05.2020	OXS-10616: Treat eval error as warning, if the error is due to "<var> is not defined" issue
 * V103	08.01.2020	a. OXS-10114: Extended for redesigned process flow of document splitter
 * V102	26.11.2019	a. OXS-10005: Extended with renameDirectory() to rename input directory in case ftp export error
 * V101	23.11.2019	a. OXS-9652: Converted '_evalExpression()' as public function "evalExpression()"
 * V100	18.10.2019	Initial release
 *----------------------------------------------------------------------------*/
'use strict'

/**
 * letiable declaration
 * @private
 */
let npsServer				= require('oxsnps-core/server')
  , npsTempDir				= npsServer.npsTempDir
  , utils					= require('oxsnps-core/helpers/utils')
  , pluginManager			= require('oxsnps-core/pluginManager')
  , log4js					= require('log4js')
  , async					= require('async')
  , fs						= require('fs')
  , path					= require('path')
  , FILE_SEP				= path.sep

/************ OPTIONAL PUBLIC FUNCTIONS ***********/
//------------------------------- Constructor -------------------------------//
/**
 * Creates service utils
 *
 * @param {JSON} props Config optioins
 *					 props = {
 *		 					"serviceName":	<Name of service in which these utility functions will be used>,
 *		 					"plugin":		<parent plugin>,
 *		 					"pluginName":	<parent plugin name>,
 *		 					"logger":		<parent plugin logger instance>
 *					 }
 * @class
 */
function serviceUtils(props){
	this.SERVICE_NAME	= props.serviceName
	this.plugin			= props.plugin
	this.PLUGIN_NAME	= props.pluginName
	this.logger			= props.logger
	this.serviceContext	= props.serviceName.toLowerCase()
}

//--------------------------------- Exports ---------------------------------//
module.exports = serviceUtils

//------------------------------- Destructor --------------------------------//
/**
 * DESTROY
 *
 * Delete job utils
 *
 * @instance
 */
serviceUtils.prototype.DESTROY = function(){
	this.SERVICE_NAME	= undefined
	this.plugin			= undefined
	this.PLUGIN_NAME	= undefined
	this.logger			= undefined
	this.serviceContext	= undefined
}

//----------------------------- Object functions -----------------------------//

/**
 * Get process from orderbase
 *
 * @param  {Response}	res			Response Object
 *									res._Context: {
 *										'reqId':			<Request Id>,
 *										'orderParms': {
 *											'tenant':		<tenant>,
 *											'orderId':		<order id>,
 *											'processId':	<process id>,
 *											'auth_token':	<authentication token>
 *										} 
 *									} 
 * @param  {function}	callback	callback(err)
 *									Updates res._Context with result
 *									res._Context.obResult = <order base result>
 */
serviceUtils.prototype.getProcess = function(res, callback){

	let logger = this.logger

	if(logger.isDebugEnabled()) logger.debug('--->getProcess: Req. id=' + res._Context.reqId + 
		', Calling oxsnps-ordersystem.obQueryService() to get a process, Parms: ' + JSON.stringify(res._Context.orderParms))

	pluginManager.callPluginService({'name':'oxsnps-ordersystem','service':'obQueryService'}, res, function(err, result){
		if(err) return callback(err)
		if(!result) return callback(new Error('Unable to get process with orderparms:' + JSON.stringify(res._Context.orderparms)))
		if(logger.isDebugEnabled()) logger.debug('--->getProcess: Req. id=' + res._Context.reqId + ', Result: ' + JSON.stringify(result))

		if(result) result = utils.parseJSON(result)
		res._Context.obResult = result

		return callback()
	})
}

/**
 * Get process additional info from orderbase
 *
 * @param  {Response}	res			Response Object
 *									res._Context: {
 *										'reqId':		<Request Id>,
 *										'tenant':		<tenant>,
 *										'orderId':		<order id>,
 *										'processId':	<process id>,
 *										'auth_token':	<authentication token>
 *									} 
 * @param  {function}	callback	callback(err)
 *									Updates res._Context with result
 *									res._Context.obResult = <order base result>
 *									Ex: res._Context.obResult.additionalInfo = process.additionalInfo
 */
serviceUtils.prototype.getProcessAdditionalInfo = function(res, callback){

	let logger = this.logger

	pluginManager.callPluginService({'name':'oxsnps-ordersystem','service':'obQueryService'}, res, function(err, result){
		if(err) return callback(err)
		if(!result) return callback(new Error('Unable to get process with orderparms:' + JSON.stringify(res._Context.orderparms)))
		if(logger.isDebugEnabled()) logger.debug('--->getProcess: Req. id=' + res._Context.reqId + ', Result: ' + JSON.stringify(result))

		if(result) result = utils.parseJSON(result)
		res._Context.obResult = result

		return callback()
	})

	res._Context.rest = {
		'tenant':		res._Context.tenant,
		'auth_token':	res._Context.auth_token,
		'method':		'GET',
		'path':			'/order/' + res._Context.orderId + '/process/' + res._Context.processId + '/additionalProcessInfo?response-format=json'
	}
	if(logger.isDebugEnabled()) logger.debug('--->getProcessAdditionalInfo: Req.Id=' + res._Context.reqId +
											 ', Calling oxsnps-ordersystem.restService() to get process additional info, Parms=' + JSON.stringify(res._Context.rest))
	
	// Call rest service and get process additional info
	pluginManager.callPluginService({'name':'oxsnps-ordersystem','service':'restService'}, res, function(err, result){
		if(err) return callback(err)
		if(!result) return callback(new Error('Unable to get process additional info with parms:' + JSON.stringify(res._Context.rest)))
		if(logger.isDebugEnabled()) logger.debug('--->getProcessAdditionalInfo: Req. id=' + res._Context.reqId + ', Result: ' + JSON.stringify(result))

		res._Context.obResult = result
		return callback()
	})
}

/**
 * Filter process documents based on given attribute and it's value
 * 
 * @param  {Response}	res			Response Object 
 *									res._Context: {
 *										'reqId':			<Request Id>,
 *										'process':			<Process JSON>,
 *										'documents':		<array of documents>,
 *										'filterBy':			<An expression with document indexes that should return a boolean value to filter the docs>
 *									} 
 * @param  {function}	callback	callback(err)
 *									Updates res._Context with result
 *									res._Context.filteredDocuments = <Filtered documents based on filterBy rule>
 */
serviceUtils.prototype.filterDocs = function(res, callback){

	let logger = this.logger

	if(logger.isDebugEnabled()) logger.debug('--->filterDocs: Req. id=' + res._Context.reqId + ', Filter documents of process, documents: ' + JSON.stringify(res._Context.documents) + ', filterBy: ' + res._Context.filterBy)
	
	res._Context.filteredDocuments = []
	res._Context.expr = res._Context.filterBy
	
	// Loop thru all the documents of the process
	res._Context.documents.forEach(function(document){
		res._Context.document = document
		this.evalExpression(res, function(err, flag){ // V101 Change
			if(err) return callback(err)

			// Assert filterBy attribute data type and convert it to boolean if needed
			logger.debug('--->filterDocs: Req. id=' + res._Context.reqId + ', Evaluated filter value: ' + flag)
			if(typeof(flag) === 'string') flag = (flag.toLowerCase() === 'true')

			// Add document to filtered documents list
			if(flag) res._Context.filteredDocuments.push(document)
		})
	})
	logger.debug('--->filterDocs: Req. id=' + res._Context.reqId + ', Filtered documents: ' + JSON.stringify(res._Context.filteredDocuments))
	return callback()
}

/**
 * Retrieve documents from the Archive Server and store them in temp dir
 * 
 * @param  {Response}	res			Response Object 
 *									res._Context: {
 *										'reqId':				<Request Id>,
 *										'filteredDocuments':	<Array of documents to be retrieved>,
 *										'filename':				<Filename, where filename might contain process/document indexes as eyecatchers>,
 *										'tempDir':				<Temp directory to store retrieved documents>,
 *										'process':				<Process JSON>,
 *										'retrieveMaxOccurence':	<Maximum asynchronous retrieval limit>
 *									} 
 * @param  {function}	callback	callback(err)
 *									Updates res._Context with result
 *									res._Context.retrievedDocuments = <Retrieved documents>
 */
serviceUtils.prototype.retrieveDocs = function(res, callback){

	let logger	= this.logger
	  , self	= this

	logger.debug('--->retrieveDocs: Req.Id=' + res._Context.reqId + ', Retrieving documents: ' + JSON.stringify(res._Context.filteredDocuments) + ', from archive server')

	// Iterate through each document
	res._Context.retrievedDocuments = []
	async.eachLimit(res._Context.filteredDocuments, res._Context.retrieveMaxOccurence /*limit=1*/, function(doc, next){
		let docData			= undefined
		  , outputFilename	= undefined

		async.series([
			// 1. Build filename for current document
			function(nextTask){
				logger.debug('--->retrieveDocs: Req.Id=' + res._Context.reqId + ', Building filename for document: ' + doc.id)
				res._Context.document = doc
				self.buildDocFilename(res, function(err, filename){
					res._Context.document = undefined
					if(err) return nextTask(new Error('Unable to build name for document: ' + doc.id + ' from filename pattern: ' + res._Context.filename + '. Reason: ' + err.message))

					outputFilename = path.resolve(res._Context.tempDir + FILE_SEP + filename)
					logger.debug('--->retrieveDocs: Req.Id=' + res._Context.reqId + ', Built filename: ' + outputFilename + ' for document: ' + doc.id)
					return nextTask()
				})
			},
			// 2. Retrieve the document from archive server
			function(nextTask){
				if(logger.isDebugEnabled()) logger.debug('--->retrieveDocs: Req. id=' + res._Context.reqId + ', Retrieving document: ' + doc.id)
				res._Context.documentId = doc.id
				pluginManager.callPluginService({name:'oxsnps-archive',service:'getDocService'}, res, function(err,result){
					if(err) return nextTask(new Error('Unable to retrieve document: ' + doc.id + ' from archive server. Reason: ' + err.message))

					if(logger.isDebugEnabled()) logger.debug('--->retrieveDocs: Req. id=' + res._Context.reqId + ', Retrieved document: ' + doc.id)
					docData = result
					return nextTask()
				})
			},
			// 3. Store the document in file
			function(nextTask){
				if(logger.isDebugEnabled()) logger.debug('--->retrieveDocs: Req. id=' + res._Context.reqId + ', Storing document: ' + doc.id + ' as file: ' + outputFilename)
				fs.writeFile(outputFilename, docData, function(err){
					if(err) return nextTask(new Error('Unable to store document: ' + doc.id + ' as file: ' + outputFilename + '. Reason: ' + err.message))

					if(logger.isDebugEnabled()) logger.debug('--->retrieveDocs: Req. id=' + res._Context.reqId + ', Stored document: ' + doc.id + ' as file: ' + outputFilename)
					res._Context.retrievedDocuments.push(outputFilename)
					nextTask()
				})
			}
		],
		function(err){
			return next(err)
		})
	},
	function(err){
		if(err) return callback(err)
		logger.debug('--->retrieveDocs: Req.Id=' + res._Context.reqId + ', Retrieved documents: ' + JSON.stringify(res._Context.retrievedDocuments) + ', from archive server')
		return callback()
	})
}

// V101 Begin
/**
 * Add documents to process
 *
 * @param  {Response}	res			Response Object 
 *									res._Context = {
 *										'reqId':			<request id>,
 *										'tenant':			<tenant>,
 *										'auth_token':		<Authenticaion token>,
 *										'orderId':			<Order id>,
 *										'processId':		<Process id>
 *										<serviceContext>:	{
 *											'doc':			<Array of documents to be added>,
 *											'msgData':		<Message payload data>
 *										}
 *									}
 * @param  {function}	callback	callback(err)
 */
serviceUtils.prototype.addDocuments = function(res, callback){
	let apiContext	= res._Context[this.serviceContext] || undefined
	if(!apiContext) return callback(new Error('Missing API context (' + this.serviceContext + ') in res._Context: ' + JSON.stringify(res._Context)))

	res._Context.orderParms = {
		'tenant': 		res._Context.tenant,
		'auth_token': 	res._Context.auth_token,
		'orderId': 		res._Context.orderId,
		'processId': 	res._Context.processId,
		'document': 	undefined
	}
	if(apiContext.doc) res._Context.orderParms.document = apiContext.doc
	if(res._Context.orderParms.document === undefined) return callback(new Error('No document to add in process: ' + res._Context.processId))

	let logger	= this.logger
	  , self	= this
	logger.debug('--->addDocuments: Req.Id=' + res._Context.reqId + ', Adding documents, parms:' + JSON.stringify(res._Context.orderParms))

	let msgData			= apiContext.msgData
	  , signPDF			= false
	  , signPDFConf		= undefined
	  , signPDFConfType	= undefined
	  , channel			= msgData.channel // V107 Change
	  , retry			= (channel.retry) ? channel.retry : undefined // V107 Change
	  , serverRetry		= (retry && retry.servers) ? retry.servers : undefined // V107 Change
	  , retryOptions	= (retry && retry.fs_access) ? utils.clone(retry.fs_access) : ((channel.fs_access_retry) ? utils.clone(channel.fs_access_retry) : undefined) // V107 Change

    res._Context.retryOptions = serverRetry // V107 Change

	async.series([
		// Task 1: Get configuration from the channel def
		function(nextTask){
			// Assert msgData and channel def
			if(!msgData) return nextTask(new Error('Missing msgData in API context: ' + JSON.stringify(apiContext)))
			if(!msgData.channel) return nextTask(new Error('Missing channel in msgData: ' + JSON.stringify(msgData)))
			if(msgData.channel.signPDF === undefined) return nextTask()

			// Get signPDF configuration
			signPDFConf = msgData.channel.signPDF
			signPDFConfType = typeof(signPDFConf)

			// Check whether value is given as string, if yes interpret the value as boolean
			if(signPDFConfType === 'string' && signPDFConf.toLowerCase() !== 'false'){
				signPDF = true
				return nextTask()
			}

			// Check whether value is given as boolean, if yes use it AS IS
			if(signPDFConfType === 'boolean') signPDF = signPDFConf

			nextTask()
		},
		// Task 2: Check and sign the document before adding it to process
		function(nextTask){
			// Iterate through each document and sign them
			async.eachLimit(res._Context.orderParms.document, 1 /*limit=1*/, function(doc, next){
				if(!doc) return next()
				let signDoc	= signPDF

				// Determine whether document should be signed or not
				// Check whether value is given as object, if yes use it as value[<document type>]
				if(!signDoc && signPDFConfType === 'object') signDoc = signPDFConf[doc.indexes.document_type_str]
				if(!signDoc) return self.buildDocSizeIndex(doc, next)

				// Sign the document
				self.signDocument(res, doc, signDoc, retryOptions, next)
			},
			function(err){
				return nextTask(err)
			})
		},
		// Task 3: Add document to process
		function(nextTask){
			pluginManager.callPluginService({'name':'oxs.import', 'service':'addDocumentsService'}, res, function(err){
				//if(err) res._Context.obError = true // V105 Change
				if(err) res._Context.archiveError = true // V105 Change
				return nextTask(err)
			})
		}
	],
	function(err){
		if(logger.isDebugEnabled()) logger.debug('--->addDocuments: Req.Id=' + res._Context.reqId + ((err) ? ', failed. Reason: ' + err.message : ', Added documents (' + JSON.stringify(res._Context.orderParms.document) + ') to process'))
		return callback(err)
	})
}

/**
 * Evaluate expression
 * 
 * @param  {Response}	res			Response Object 
 *									res._Context: {
 *										'expr':		<Expression to evaluate. Starts with prefix 'eval:...'>,
 *										'process':	<Process JSON>,
 *										'document':	<Document JSON>
 *									} 
 * @param  {function}	callback	callback(err, evaluatedValue)
 */
serviceUtils.prototype.evalExpression = function(res, callback){

	let logger = this.logger

	logger.debug('--->evalExpression: Req.Id=' + res._Context.reqId + ', Evaluating expression: ' + res._Context.expr)

	let process		= res._Context.process						// might be used in expression
	  , document	= res._Context.document || res._Context.documents || process.document	// might be used in expression
	  , expr		= undefined
	  , value		= res._Context.expr
	
	// Evaluate value if it has index eyecatchers
	if(typeof(res._Context.expr) === 'string' && res._Context.expr.startsWith('eval:')){
		try{
			expr = res._Context.expr.substring(res._Context.expr.indexOf(':')+1) // get expression from 'eval: <expression>' string

			value = eval(expr)
		}
		catch(err){
			// V104 Begin
			if (err){
				if (    err.message.includes('is not defined')
				     || err.message.includes(' of undefined') // V106 Change
				   ){
					logger.warn('Unable to evaluate Javascript expression: ' + expr + ', Reason: ' + err.message)
					value = undefined // V106 Change
				}
				else {
					return callback(new Error('Unable to evaluate Javascript expression: ' + expr + ', Reason: ' + err.message))
				}
			}
			// V104 End
		}
	}
	logger.debug('--->evalExpression: Req.Id=' + res._Context.reqId + ', Evaluated value: ' + value + ' for expression: ' + res._Context.expr)
	return callback(null, value)
}
// V101 End

// V102 Begin
/**
 * Rename directory
 * @param	{Response}	res			Response Object
 *									res._Context:{
 *										'dir': <Directory to be renamed>
 *									}
 * @param	{String}	find		Suffix string to find
 * @param	{String}	replace  	Suffix string to replace
 * @param	{function}	callback	callback(err)
 *									Also updates res._Context with 'renamedDir' as shown below
 *									res._Context:{
 *										'renamedDir': <directory after renaming>
 *									}
 */
serviceUtils.prototype.renameDirectory = function(res, find, replace, callback){

	let src = res._Context.dir
	  , logger = this.logger

	logger.debug('--->renameDirectory: Req.Id=' + res._Context.reqId + ', find: ' + find + ', replace: ' + replace + ', src: ' + src)

	// Check and rename directory
	utils.dirExists(src, function(err, exists){
		if(err){
			if(err) logger.error('--->renameDirectory: Req.Id=' + res._Context.reqId + ', Unable to locate directory: ' + src + '. Reason: ' + err.message)
			return callback(err)
		}
		let dst = src + replace // if find is undefined, then dst should be src + replace
		if(find) dst = src.replace(new RegExp(find, 'i'), replace)

		if(!exists) return callback(new Error('Unable to rename ' + src + ' as ' + dst + '. Reason: Directory not found'))

		res._Context.renamedDir = dst
		logger.debug('--->renameDirectory: Req.Id=' + res._Context.reqId + ', Renaming ' + src + ' as ' + dst + '...')
		utils.move(src, dst, {}, function(err){
			if(err) logger.error('--->renameDirectory: Req.Id=' + res._Context.reqId + ', Unable to rename directory: ' + src + ' as: ' + dst + '. Reason: ' + err.message)
			else logger.debug('--->renameDirectory: Req.Id=' + res._Context.reqId + ', Renamed ' + src + ' as ' + dst)
			return callback(err)
		})
	})
}
// V102 End

// V103 Begin
/**
 * Rename file
 * @param	{Response}	res			Response Object
 *									res._Context:{
 *										'file': <file to be renamed>
 *									}
 * @param	{String}	find		Suffix string to find
 * @param	{String}	replace  	Suffix string to replace
 * @param	{function}	callback	callback(err)
 *									Also updates res._Context with 'renamedFile' as shown below
 *									res._Context:{
 *										'renamedFile': <file after renaming>
 *									}
 */
serviceUtils.prototype.renameFile = function(res, find, replace, callback){

	let src = res._Context.file
	  , logger = this.logger

	// Check and rename file
	utils.fileExists(src, function(err, exists){
		if(err){
			if(err) logger.error('--->renameFile: Req.Id=' + res._Context.reqId + ', Unable to locate file: ' + src + '. Reason: ' + err.message)
			return callback(err)
		}
		let dst = src + replace // if find is undefined, then dst should be src + replace
		if(find) dst = src.replace(new RegExp(find, 'i'), replace)

		if(!exists) return callback(new Error('Unable to rename ' + src + ' as ' + dst + '. Reason: File not found'))

		res._Context.renamedFile = dst
		logger.debug('--->renameFile: Req.Id=' + res._Context.reqId + ', src: ' + src + ', dst: ' + dst)
		utils.move(src, dst, {}, function(err){
			if(err) logger.error('--->renameFile: Req.Id=' + res._Context.reqId + ', Unable to rename file: ' + src + ' as: ' + dst + '. Reason: ' + err.message)
			return callback(err)
		})
	})
}

/**
 * Set signal to process
 * @param  {Response}	res  	  Response Object
 * @param  {function} 	callback  callback(err)
 */
serviceUtils.prototype.setSignal = function(res, callback){

	let logger = this.logger

	if(!res._Context.transition) return callback()	// transition is optional

	if(logger.isDebugEnabled()) logger.debug('--->setSignal: Req.Id=' + res._Context.reqId +
											 ', Set signal: ' + res._Context.transition + ', Parms: '+ JSON.stringify(res._Context))

	pluginManager.callPluginService({'name':'oxs.import','service':'setSignalService'}, res, function(err){
		if(err) logger.error('--->setSignal: Req.Id=' + res._Context.reqId + ', Setting ' + res._Context.transition + ' signal failed. Reason:' + err.message)
		return callback(err)
	})
}

/**
 * Restore date
 * @param  {Response}	res  	  Response Object
 */
serviceUtils.prototype.restoreDateAndRequestId = function(res){
	// When Job is saved, kue converts res._Context.date to string, so convert it back to date object
	if(res._Context.date && typeof res._Context.date === 'string') res._Context.date = new Date(res._Context.date)
	res._Context.date = res._Context.date || new Date()

	// Add Request id to the Context
	res._Context.reqId = res._Context.reqId || utils.buildReqId(res._Context.date)
}

/**
 * Build process specific temp directory name
 * @param  {Response}	res  	Response Object
 * 								res._Context:{
 * 									'tenant':		<tenant>,
 * 									'orderId':		<order id>,
 * 									'processId':	<process id>
 * 								}
 */
serviceUtils.prototype.buildProcessTempDir = function(res){
	if(!res._Context.tenant || !res._Context.processId) return undefined
	return path.resolve(npsTempDir + FILE_SEP + res._Context.tenant + FILE_SEP + res._Context.processId)
}

/**
 * Build document filename
 * 
 * @param  {Response}	res			Response Object
 *									res._Context: {
 *										'reqId':				<Request Id>,
 *										'filteredDocuments':	<Array of documents to be retrieved>,
 *										'filename':				<Filename, where filename might contain process/document indexes as eyecatchers>,
 *										'tempDir':				<Temp directory to store retrieved documents>,
 *										'process':				<Process JSON>,
 *										'document':				<Document JSON>
 *									} 
 * @param  {function}	callback	callback(err, filename)
 */
serviceUtils.prototype.buildDocFilename = function(res, callback){

	let filename = res._Context.filename || res._Context.document.id + '.pdf' // default name

	res._Context.expr = filename
	return this.evalExpression(res, callback) // V101 Change
}

/**
 * Build document size index
 *
 * 1. Runs fs.stat for given document filename
 * 2. Updates document.indexes with 'size_int=<document size in Kb>' index
 *
 * @param  {Object}		document	Document object
 *									document: {
 *										'file':		<document filename>,
 *										'indexes':	<document indexes>
 *									} 
 * @param  {function}	callback	callback(err)
 */
serviceUtils.prototype.buildDocSizeIndex = function(document, callback){
	fs.stat(document.file, function(err, stats){
		if(err) return callback(err)

		document.indexes['size_int'] = Math.round(stats.size / 1024) || 1 	// round to nearest integer, if value is zero set file size to 1 KB
		return callback()
	})
}

/**
 * Sign document
 *
 * 1. Signs given document based on flag
 * 2. Updates document.indexes with 'size_int=<document size in Kb>' index
 *
 * @param  {Response}	res			Response object
 * @param  {Object}		document	Document object
 *									document: {
 *										'file':		<document filename>,
 *										'indexes':	<document indexes>
 *									}
 * @param  {Boolean}	sign		Sign flag
 * @param  {Object}		retryOpts	Retry options
 * @param  {function}	callback	callback(err)
 */
serviceUtils.prototype.signDocument = function(res, document, sign, retryOpts, callback){
	let logger		= this.logger
	  , self		= this
	  , docSigned	= false
	  , accessMode	= (retryOpts) ? retryOpts.mode : fs.constants.R_OK

	async.series([
		// Task 1: Sign document when sign flag is set
		function(nextTask){
			if(!sign) return nextTask()

			// Fill sign info in res context
			res._Context.npdf = {
				'input':		{ 'file': document.file, 'opts': {'forUpdate': true} },
				'output':		{ 'path': res._Context.signOutputPath, 'filenamePattern':	'%(simpleFilename)s_signed.pdf', 'writeMode' : 'clean' },	// for PDF/A output
				'signOption':	sign
			}

			if(logger.isDebugEnabled()) logger.debug('--->signDocument: Req.Id=' + res._Context.reqId + ', Signing document: ' + JSON.stringify(document))
			pluginManager.callPluginService({'name':'oxsnps-nopodofoclient', 'service':'signDocumentService'}, res, function(err, result){
				if(err) return nextTask(new Error('Unable to sign the document: ' + JSON.stringify(document) + ', Reason: ' + err.message))
				if(result && result.status === 'signed'){
					// Update document info with signed output details
					docSigned = true
					document.file = result.outputFile
					if(logger.isDebugEnabled()) logger.debug('--->signDocument: Req.Id=' + res._Context.reqId + ', Signed the document: ' + JSON.stringify(document))
				}
				return nextTask()
			})
		},
		// Task 2: When document is signed, check for file accessibility of signed output
		function(nextTask){
			if(!docSigned || !retryOpts) return nextTask()

			res._Context.fa = {'path': document.file, 'mode': accessMode, 'retryOptions': retryOpts}
			if(logger.isDebugEnabled()) logger.debug('--->signDocument: Req.Id=' + res._Context.reqId + ', Check for file accessibility. Parms: ' + JSON.stringify(res._Context.fa))
			utils.checkFileAccess(res, function(err){
				if(err) return nextTask(new Error('Could not access signed doc: ' + res._Context.fa.path + ', Reason: ' + err.message))
				if(logger.isDebugEnabled()) logger.debug('--->signDocument: Req.Id=' + res._Context.reqId + ', ' + res._Context.fa.path + ' is accessible')
				return nextTask()
			})
		},
		// Task 3: Evaluate the size_int index for document
		function(nextTask){
			if(logger.isDebugEnabled()) logger.debug('--->signDocument: Req.Id=' + res._Context.reqId + ', Evaluating the size_int index for document')
			self.buildDocSizeIndex(document, function(err){
				if(err) return nextTask(err)
				if(logger.isDebugEnabled()) logger.debug('--->signDocument: Req.Id=' + res._Context.reqId + ', Evaluated the index size_int: ' + document.indexes['size_int'] + ' for document')
				return nextTask()
			})
		}
	],
	function(err){
		if(err && logger.isDebugEnabled()) logger.debug('--->signDocument: Req.Id=' + res._Context.reqId + ', failed. Reason: ' + err.message)
		return callback(err)
	})
}
// V103 End

// V1.0.16 Begin
serviceUtils.prototype.assertProcessTransition = function(res, callback){

    let serviceContext  = res._Context.serviceContextName
      , impMultiFiles   = res._Context[serviceContext]
      , msgData         = impMultiFiles && impMultiFiles.msgData
      , channel         = msgData && msgData.channel
      , resTmp            =  {'_Context': {}}
      , logPrefix       = '--->assertProcessTransition: Req.Id=' + res._Context.reqId + ', '
      , errMsg            = undefined
      , retryCount        = channel && channel.wsInputHandlerRetryCount        || 60        // Default is 60 for 2 mins
      , retryInterval    = channel && channel.wsInputHandlerRetryInterval    || 2000        // Default is 2 seconds 
	  , logger = this.logger
	  
	const self = this

    if(impMultiFiles.retryCount === undefined)    impMultiFiles.retryCount = 0

    resTmp._Context = {
        'reqId':            res._Context.reqId,
        'orderParms': {
            'tenant':        res._Context.tenant,
            'auth_token':    res._Context.auth_token,
            'orderId':       res._Context.orderId    || res._Context.orderParms.orderId,        
            'processId':     res._Context.processId  || res._Context.orderParms.processId,
            'course':        res._Context.course     || res._Context.orderParms.course
        }
    }

    if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Getting current possible worksteps from the stack order, Parms: ' + JSON.stringify(resTmp._Context.orderParms))
    pluginManager.callPluginService({'name': 'oxsnps-ordersystem', 'service': 'getCoursesService'}, resTmp, function(err, result){
		/*
			result = {
				"course": "0.2",
				"workflow": "ais_tagesreport_import",
				"state": "Warte auf die Eingabe",
				"owner": null,
				"transitions": [{
						"location": "ais_tagesreport_import:_Behandeln Sie die Eingabe",
						"permitted": true
					}, {
						"location": "basic-technical:Vorgang löschen",
						"permitted": true
					}
				],
				"attributes": {
					"tf_reminder_to_str": "katja.maas@maas.de",
					"previous_state": "Pubsub aktion ausgefuehrt",
					"tf_reminder_str": "1m",
					"tf_export_to_str": "katja.maas@maas.de",
					"stack_fehler_boolean": false,
					"last_scheduled_operation_long": 26591,
					"tf_error_to_str": "katja.maas@maas.de",
					"workbasket_str": "ais-tagesreport-4730",
					"workstate_read_boolean": false,
					"previous_workflow": "multiple_input_job_release",
					"rentschler_sachbearbeiter_email_to_str": "katja.maas@maas.de"
				}
			}
		*/

        if(logger.isDebugEnabled()) logger.debug(logPrefix + 'Possible worksteps from the stack order: ' + (result ? JSON.stringify(result):"undefined"))

        if(err) errMsg = 'Unable to get possible worksteps from the stack order: ' + resTmp._Context.orderParms.orderId + ', Reason:' + err.message
        else if(   !result
           || result.course !== resTmp._Context.orderParms.course
           || !result.transitions
           || result.transitions.length <= 0){
			errMsg = 'Unable to get possible worksteps from the stack order: ' + resTmp._Context.orderParms.orderId + ', Reason: Invalid result: ' + JSON.stringify(result)
        }
        else {
            // Get allowed transitions
            let courseTransitions = result.transitions || []
              , allowedTransitions = courseTransitions.map(t => (t.location) ? t.location.split(':')[1] : undefined).filter(t => t !== undefined)

			//if(logger.isDebugEnabled()) logger.debug(logPrefix + 'AllowedTransitions: ' + JSON.stringify(allowedTransitions))

			allowedTransitions = allowedTransitions.map(str => str.replace(/^_/, ''))	// remove leading _ from transitions
            
			if(logger.isDebugEnabled()) logger.debug(logPrefix + 'AllowedTransitions after removing leading underscore: ' + JSON.stringify(allowedTransitions))

			if(allowedTransitions.includes(channel.wsInputHandler)) return callback()

            errMsg =   'Stack  order: ' + resTmp._Context.orderParms.orderId + ' course does not allow to signal: ' + channel.wsInputHandler
                         + ', Reason: Current workstep: ' + result.state + ' does not allow transition to ' + channel.wsInputHandler
                         + ', Allowed transitions: ' + allowedTransitions.join(',')
        }

        if(impMultiFiles.retryCount >= retryCount){
			/*
            return callback(new Error(  'Stack  order: ' + resTmp._Context.orderParms.orderId
                                      + ' not in proper course/transition('+ channel.wsInputHandler + ') even after retries. RetryCount: '
                                      + impMultiFiles.retryCount + ', retry interval in milliseconds: ' + retryInterval
                                      + ', Reason: ' + errMsg
                                     )
                           )
			*/
			let errTmp = new Error(errMsg + ', RetryCount: ' + impMultiFiles.retryCount + ', retry interval: ' + retryInterval + ' ms')
			logger.warn(logPrefix + errTmp.message)
			return callback(errTmp)
		}
        
        impMultiFiles.retryCount++
        if(logger.isDebugEnabled()) logger.debug(logPrefix + errMsg + ', Retry attempt: ' + impMultiFiles.retryCount+ ', getting current possible worksteps scheduled after ' + retryInterval + ' ms')
        setTimeout(function(){self.assertProcessTransition(res, callback);}, retryInterval) // Wait for 2s
    })
}    
// V1.0.16 End