#-------------------------------------------------------------------------------
#  addAnnotations.pm: Scripting Facility to add annotation to the page
#
#
#  Call:
#
#  On Windows:	afp2web.exe -q -c -doc_cold -sp:addAnnotations.pm samples\insure.afp
#
#  On Unix:	./afp2web   -q -c -doc_cold -sp:addAnnotations.pm samples/insure.afp
#
#  Author  : Fa. Maas
#  Date    : 2008-07-10
#  Version : 1.0.1
#
#  $V100   2005-08-01	Initial Release
#
#  $V101   2008-07-10	Extended to add script file path as module search path
#                       on "BEGIN" block
#-------------------------------------------------------------------------------

# $V101 Begin
#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );

    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
}
# $V101 End

use a2w::Config;
use a2w::Document;
use a2w::Font;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::MediumMap;
use a2w::NOP;
use a2w::Overlay;
use a2w::Page;
use a2w::PSEG;
use a2w::Text;
#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;	# TRUE  boolean value
    $FALSE = 0;	# FALSE boolean value

    #---- Set/Reset Logging
	$bLog = $FALSE;
	if (index( lc($a2wConfigPar->getAttribute("LoggingLevel")), "sf") >= 0 ){
		$bLog = $TRUE;
	}

    my $svScriptProcTmp = $a2wConfigPar->getAttribute("ScriptProcedure");
    my $svScriptArgsTmp	= $a2wConfigPar->getScriptArgs();
    $svIndexFilePath	= $a2wConfigPar->getIndexFilePath();
    $svOutputFilePath	= $a2wConfigPar->getOutputFilePath();
    $svSpoolFilename	= $a2wKernelPar->getSpoolFilename();

    if ( $bLog == $TRUE ){
	    print "Running $svScriptProcTmp...\n";
        print "initialize(): Processing $svSpoolFilename\n";
		print "initialize(): Args: $svScriptArgsTmp, IndexFilePath: $svIndexFilePath, OutputFilePath: $svOutputFilePath\n";
    }

    #---- Reset Page Id
    $PageId = 0;

    #---- Set AutoSplit on
	$a2wConfigPar->setAttribute( "AutoSplit", "on" );

	# -- Assuming width of a character as 2.53 millimeter
	# -- It is used to calculate Hyper Link Rectangle Width as below
	# -- Hyper Link Rect Width (in millimeters) = Hyper Link Text Length * $CharWidth 
	$CharWidth = 2.53;

    #--- Hyper Link Rectangle Height values in millimeters --- 
	$HPLHeight	= 2.6;	
	
	#--- Hyper Link Rectangle Border Width value in millimeters (approximately 1.0 Point)
	#--- Set BorderWidth to zero if border is not needed.
	$BorderWidth = 0.35;

	#--- Hyper Link Action URL
	$UrlText = "http://afp2web.com";

	#--- Hyper Link Border color as RGB value (default color blue is assigned here)
	$HPLColor=0x000000FF;		#---- Blue (0x00RRGGBB)
#	$HPLColor=0x00FF0000;		#---- Red
#	$HPLColor=0x0000FF00;		#---- Green
#	$HPLColor=0x00FFFFFF;		#---- White
#	$HPLColor=0x00000000;		#---- Black

    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

	if ( $bLog == $TRUE ){
		print "initializeDoc(): DocId " . $a2wDocumentPar->getId() . "\n";
	}

    #---- Reset Page Id
    $PageId = 0;

    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){
        print "initializePage()\n";
    }
    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#  	  < 0:	error
# 		0:	append page to Current Document
# 		1:	skip page
# 		2:	first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

	if ( $bLog == $TRUE ){
        print "afp2web(): PageId " . $a2wPagePar->getParseId() . "\n";
	}

    $APPEND	= 0;	# append page to Current Document
    $SKIP	= 1;	# skip page
    $NEWDOC	= 2;	# new document

	#---- Set default return value
	my $svRetTmp = $APPEND; # default: append page

    #---- Increment Page Id
	$PageId++;

	#---- Add annotation to the first page of every document.
	if( $PageId == 1 ) {

		#---- We are looking for a text object that has a insured customer name 
		#---- for example "Geoffrey R Stephens" in the first page of insure.afp
		#---- Area: 2438<= x <= 2458 and 5337 <= y <= 5357

		#---- Fetch first Text Object
		my $a2wTextTmp = $a2wPagePar->getFirstText();

		#---- Define temp variables
		my $svTextTmp		= "";
		my $svTextXPosTmp	= 0;
		my $svTextYPosTmp	= 0;
		my $svTextLenTmp	= 0 ;

		#--- Get Page Resolution 
		my $PageResTmp	= $a2wPagePar->getResolution();

		#--- Convert Rectangle Height from millimeters to AFP units
		my $HeightTmp	= $PageResTmp * $HPLHeight / 25.4;

		#--- Convert Rectangle Border Width from millimeters to AFP units
		my $BorderWidthTmp = $PageResTmp * $BorderWidth / 25.4;

		#--- Border is drawn on top and bottom sides of a rectangle
		#--- So increase height to include border width
		$HeightTmp	= $HeightTmp + ( 2 * $BorderWidthTmp );

		#---- Loop thru all the Text Objects
		while ( $a2wTextTmp != 0 ){
			
			$svTextTmp = $a2wTextTmp->getText();
			$svTextXPosTmp = $a2wTextTmp->getXPos();
			$svTextYPosTmp = $a2wTextTmp->getYPos();

			#---- If true ==> text object is a customer name 
			if ( $svTextXPosTmp >= 2438	&& 
				 $svTextXPosTmp <= 2458	&& 
				 $svTextYPosTmp >= 5337	&&
				 $svTextYPosTmp <= 5357	){

				$svTextLenTmp = length( $svTextTmp );

				#--- Calcualte HyperLink Rectangle Width
				$WidthTmp = $svTextLenTmp * $CharWidth * $PageResTmp / 25.4;

				#--- Border is drawn on both left and right sides
				#--- So increase width to include border width
				$WidthTmp	= $WidthTmp  + ( 2 * $BorderWidthTmp );

				#--- CheckXPosTmp is X Position of lower left corner of a TextBox.
				#--- PDF Draws border inside the rectangle.
				#--- So move CheckXPosTmp position to the left by $BorderWidthTmp 
				#--- so that it does not overlap the first character
				my $CheckXPosTmp = $svTextXPosTmp  - $BorderWidthTmp ;

				#--- CheckYPosTmp is Y Position of lower left corner of a TextBox.
				#--- PDF Draws border inside the rectangle, so move CheckYPosTmp position 
				#--- to the bottom by $BorderWidthTmp so that it does not overlap the characters
				#--- Since addAnnotation API requires Upper left corner of rectangle,
				#--- subtract rectangle height from CheckYPosTmp 
				my $CheckYPosTmp = $svTextYPosTmp + $BorderWidthTmp - $HeightTmp;

				$a2wPagePar->addAnnotation( $CheckXPosTmp,	#---- Upper left X position of annotation box (in AFP page units)
										 $CheckYPosTmp,		#---- Upper left Y position of annotation box (in AFP page units)
										 $WidthTmp,			#---- Width of annotation box (in AFP page units)
										 $HeightTmp,		#---- Height of annotation box (in AFP page units)
										 $UrlText,			#---- URL
										 $BorderWidthTmp,	#---- Annotation border width
										 $HPLColor);		#---- Annotation border color (Format=0x00BBGGRR)
		
				if ( $bLog == $TRUE ){
					print "Annotation added @(" . $CheckXPosTmp . "," . $CheckYPosTmp . ")>" . $svTextTmp . "<\n";
				}
				last; # leave while loop
			}

			#---- Fetch next Text Object
			$a2wTextTmp = $a2wPagePar->getNextText();
		}
	}
    return $svRetTmp;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

	if ( $bLog == $TRUE ){
		print "finalizePage()\n";
	}
    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){
        print "finalizeDoc()\n";
    }
    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){
        print "finalize()\n";
    }
    return 0;
}

__END__
