#-------------------------------------------------------------------------------
#  afp2xml.pm: Scripting Facility to create XML output
#
#
#  Call:
#
#  On Windows:
#			FOR /F %%i IN ( 'perl -e "{$,=';'; print @INC;}"' ) DO @SET PERLLIB=%%i
#			afp2web.exe -q -c -doc_cold -sp:afp2xml.pm samples\insure.afp
#
#  On Unix:
#			export PERLLIB=`perl -e "{$,=' '; print @INC;}"`
#			./afp2web   -q -c -doc_cold -sp:afp2xml.pm samples/insure.afp
# 
#  Author  : Fa. Maas
#  Date    : 2008-07-10
#  Version : 1.0.1
#
#  $V100   2005-08-01	Initial Release
#
#  $V101   2008-07-10	Extended to add script file path as module search path
#                       on "BEGIN" block
#-------------------------------------------------------------------------------

# $V101 Begin
#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );

    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
}
# $V101 End

use a2w::Config;
use a2w::Document;
use a2w::Font;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::MediumMap;
use a2w::NOP;
use a2w::Overlay;
use a2w::Page;
use a2w::PSEG;
use a2w::Text;

#---- Required Perl Modules for XML output
use IO::File;
use XML::Writer;
#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;	# TRUE  boolean value
    $FALSE = 0;	# FALSE boolean value

    #---- Set/Reset Logging
	$bLog = $FALSE;
	if (index( lc($a2wConfigPar->getAttribute("LoggingLevel")), "sf") >= 0 ){
		$bLog = $TRUE;
	}

    my $svScriptProcTmp = $a2wConfigPar->getAttribute("ScriptProcedure");
    my $svScriptArgsTmp	= $a2wConfigPar->getScriptArgs();
    $svIndexFilePath	= $a2wConfigPar->getIndexFilePath();
    $svOutputFilePath	= $a2wConfigPar->getOutputFilePath();
    $svSpoolFilename	= $a2wKernelPar->getSpoolFilename();

    if ( $bLog == $TRUE ){
	    print "Running $svScriptProcTmp...\n";
        print "initialize(): Processing $svSpoolFilename\n";
		print "initialize(): Args: $svScriptArgsTmp, IndexFilePath: $svIndexFilePath, OutputFilePath: $svOutputFilePath\n";
    }

    #---- Reset Page Id
    $PageId = 0;

    #---- Set AutoSplit on
	$a2wConfigPar->setAttribute( "AutoSplit", "on" );

	#---- afp2xml.pm Version
	$Version = "v1.0.0";

    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

	#---- Fetch DocId
	my $DocIdTmp = $a2wDocumentPar->getId();

	if ( $bLog == $TRUE ){
		print "initializeDoc(): DocId $DocIdTmp\n";
	}

	#---- Generating FileHandle for XML Writer (needed since a2wDocumentPar->getSimpleFilename() not available from initializeDoc())
   	$xmlDocIdFilename = $svOutputFilePath . $DocIdTmp . ".xml";
   	$xmlDocOutput = new IO::File( ">" . $xmlDocIdFilename );

	#---- Generating XML Writer Object
	$xmlDocWriter = new XML::Writer(OUTPUT		=> $xmlDocOutput, 
									NAMESPACES	=> 1,
									);

	#---- Processing Instruction
	$xmlDocWriter->xmlDecl("ISO-8859-1");
	$xmlDocWriter->comment("Generated by $svScriptProc " . $Version . ", " . localtime());
	$xmlDocWriter->doctype('Document', "", 'afp2xml.dtd');

	#---- Starting with root element
    $xmlDocWriter->startTag('Document', 'Id'	=> $DocIdTmp,					# Doc Id element
							'Name'				=> $a2wDocumentPar->getName(),
							'Spool'				=> $svSpoolFilename,
							'PID'				=> $a2wDocumentPar->getPID(),
							);

    #---- Reset Page Id
    $PageId = 0;

    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){
        print "initializePage()\n";
    }
    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#  	  < 0:	error
# 		0:	append page to Current Document
# 		1:	skip page
# 		2:	first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

	if ( $bLog == $TRUE ){
        print "afp2web(): PageId " . $a2wPagePar->getParseId() . "\n";
	}

    $APPEND	= 0;	# append page to Current Document
    $SKIP	= 1;	# skip page
    $NEWDOC	= 2;	# new document

	#---- Set default return value
	my $svRetTmp = $APPEND; # default: append page

    #---- Increment PageId
    $PageId++;

    #---- Add Document Indexes to Index List
    if ($PageId == 1){
		addIndexes( $a2wDocumentPar );
		$xmlDocWriter->startTag('PageList');	# <PageList>
	}

    #---- Starting with Element Page
    $xmlDocWriter->startTag('Page', 'Id'	=> $PageId,						# Page Id element
							'Name'			=> $a2wPagePar->getName(),
							'Width'			=> $a2wPagePar->getWidth(),
							'Height'		=> $a2wPagePar->getHeight(),
							'Resolution'	=> $a2wPagePar->getResolution() 
							);

    #---- Add Page Applied Medium Map Info
	addPageMediumMapInfo();

    #---- Add Page Applied Medium Overlay List
    addPageMediumOverlayList();

    #---- Add Page Included Overlay List
    addIncludedOverlayList( $a2wPagePar );

    #---- Add Page Included PageSegment List
    addIncludedPageSegmentList( $a2wPagePar );

    #---- Add Page Indexes to IndexList of Page
	addIndexes( $a2wPagePar );

    #---- Add NOPS to NOPList of Page
    addPageNOPs();

    #---- Add Text Objects to TextObjectList of Page
    addTextObjects( $a2wPagePar );

    #---- Add Line Objects to LineObjectList of Page
    addLineObjects( $a2wPagePar );

    return $svRetTmp;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){
        print "finalizePage()\n";
    }
    $xmlDocWriter->endTag();					# closing Element Page ==> </Page>

    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){
        print "finalizeDoc()\n";
    }

	#---- Closing Element PageList
	$xmlDocWriter->endTag();	# </PageList>

	#---- Closing Element Document
	$xmlDocWriter->endTag();	# </Document>

	#---- Writing Buffer into the File
	$xmlDocWriter->end();

	#---- Closing FileHandle
	$xmlDocOutput->close();

	#---Rename DocId.xml to DocumentOutputFilename.xml
	rename( $xmlDocIdFilename , $svOutputFilePath . $a2wDocumentPar->getSimpleFilename() . ".xml" );

	return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){
        print "finalize()\n";
    }
    return 0;
}

#-----------------------------------------------------------------------
# Add Indexes to Index List
#-----------------------------------------------------------------------
sub addIndexes(){

    #---- Get Parameter (Page or Overlay)
    my $svMediaPar = $_[ 0 ];

    if ( $bLog == $TRUE ){
	    print "addIndexes()\n";
    }

    #---- Fetch first Index
	my $a2wIndexTmp = $svMediaPar->getFirstIndex();

    if ( $a2wIndexTmp != 0 ){

    	$xmlDocWriter->startTag('IndexList');	# opening element IndexList ==> <IndexList>

		#---- Loop thru the Doc Indexes
		while ( $a2wIndexTmp != 0 ){

			#---- Write Index Name & Value
			$xmlDocWriter->dataElement('Index',	   $a2wIndexTmp->getValue(),
									   'Name'	=> $a2wIndexTmp->getName());

			#---- Fetch next Index
			$a2wIndexTmp = $svMediaPar->getNextIndex();
		}
    	$xmlDocWriter->endTag();				# closing Element IndexList ==> </IndexList>
	}
}

#-----------------------------------------------------------------------
# Add Page Applied Medium Map Info
#-----------------------------------------------------------------------
sub addPageMediumMapInfo(){

    if ( $bLog == $TRUE ){
        print "addPageMediumMapInfo()\n";
    }

    #---- Fetch applied medium map
	my $a2wMediumMapTmp = $a2wPagePar->getAppliedMediumMap();

    if ( $a2wMediumMapTmp != 0 ){
        #---- Starting with Medium Map
        $xmlDocWriter->startTag( 'MediumMap',                       # opening Element MediumMap ==> <MediumMap>
                                 'Name'             => $a2wMediumMapTmp->getName(),
								 'Formdef'			=> $a2wMediumMapTmp->getFormdefName(),
                                 'Width'            => $a2wMediumMapTmp->getWidth(),
                                 'Height'           => $a2wMediumMapTmp->getHeight(),
                                 'Resolution'       => $a2wMediumMapTmp->getResolution(),
                                 'DuplexControl'    => $a2wMediumMapTmp->getDuplexControl(),
                                 'N-UpControl'      => $a2wMediumMapTmp->getNupControl(),
                               );
                            
		#---- Closing medium map
		$xmlDocWriter->endTag();				# </MediumMap>
    }
}

#-----------------------------------------------------------------------
# Add Page Applied Medium Overlay List
#-----------------------------------------------------------------------
sub addPageMediumOverlayList(){

    if ( $bLog == $TRUE ){
        print "addPageMediumOverlayList()\n";
    }

    #---- Fetch first Page Medium Overlay
	my $a2wOverlayTmp = $a2wPagePar->getFirstMediumOverlay();

	if ( $a2wOverlayTmp != 0 ){

		$xmlDocWriter->startTag('MediumOverlayList');	# opening Element MediumOverlayList ==> <MediumOverlayList>

		#---- Loop thru the Page Medium Overlays
		while ( $a2wOverlayTmp != 0 ){

			#---- Starting with Overlay
			$xmlDocWriter->startTag( 'Overlay',
									 'Name'	        => $a2wOverlayTmp->getName,             # Overlay name
									 'Width'		=> $a2wOverlayTmp->getWidth(),
									 'Height'		=> $a2wOverlayTmp->getHeight(),
									 'Resolution'	=> $a2wOverlayTmp->getResolution(),
									 'XPos'			=> $a2wOverlayTmp->getIncludedXPosition(),
									 'YPos'			=> $a2wOverlayTmp->getIncludedYPosition(),
									 );

			#---- Closing overlay
			$xmlDocWriter->endTag();				# </Overlay>

			#---- Fetch next Page Medium Overlay
			$a2wOverlayTmp = $a2wPagePar->getNextMediumOverlay();
		}
    	$xmlDocWriter->endTag();				# closing Element MediumOverlayList ==> </MediumOverlayList>
    }
}

#-----------------------------------------------------------------------
# Add Included Overlay List
#-----------------------------------------------------------------------
sub addIncludedOverlayList(){

    #---- Get Parameter (Page or Overlay)
    my $svMediaPar = $_[ 0 ];

    if ( $bLog == $TRUE ){
        print "addIncludedOverlayList()\n";
    }

    #---- Fetch first Included Overlay
	my $a2wOverlayTmp = $svMediaPar->getFirstOverlay();

	if ( $a2wOverlayTmp != 0 ){

		$xmlDocWriter->startTag('IncludedOverlayList');	# opening Element IncludedOverlayList ==> <IncludedOverlayList>

		#---- Loop thru the Included Overlays
		while ( $a2wOverlayTmp != 0 ){

			#---- Starting with Overlay
			$xmlDocWriter->startTag( 'Overlay',
									 'Name'	        => $a2wOverlayTmp->getName,             # Overlay name
									 'Width'		=> $a2wOverlayTmp->getWidth(),
									 'Height'		=> $a2wOverlayTmp->getHeight(),
									 'Resolution'	=> $a2wOverlayTmp->getResolution(),
									 'XPos'			=> $a2wOverlayTmp->getIncludedXPosition(),
									 'YPos'			=> $a2wOverlayTmp->getIncludedYPosition(),
									 );

			#---- Closing overlay
			$xmlDocWriter->endTag();				# </Overlay>

			#---- Fetch next Included Overlay
			$a2wOverlayTmp = $svMediaPar->getNextOverlay();
		}
    	$xmlDocWriter->endTag();				# closing Element IncludedOverlayList ==> </IncludedOverlayList>
    }
}

#-----------------------------------------------------------------------
# Add Included Page Segment List
#-----------------------------------------------------------------------
sub addIncludedPageSegmentList(){

    #---- Get Parameter (Page or Overlay)
    my $svMediaPar = $_[ 0 ];

    if ( $bLog == $TRUE ){
        print "addIncludedPageSegmentList()\n";
    }

    #---- Fetch first Included Page Segment
	my $a2wPageSegmentTmp = $svMediaPar->getFirstPageSegment();

	if ( $a2wPageSegmentTmp != 0 ){

		$xmlDocWriter->startTag('IncludedPageSegmentList');	# opening Element IncludedPageSegmentList ==> <IncludedPageSegmentList>

		#---- Loop thru the Included Page Segments
		while ( $a2wPageSegmentTmp != 0 ){

			#---- Starting with Page Segment
			$xmlDocWriter->startTag( 'PageSegment',
									 'Name'	        => $a2wPageSegmentTmp->getName(),				# Page Segment name
									 'XPos'			=> $a2wPageSegmentTmp->getIncludedXPosition(),	# XPosition
									 'YPos'			=> $a2wPageSegmentTmp->getIncludedYPosition(),	# YPosition
									 );     

			#---- Closing PageSegment
			$xmlDocWriter->endTag();				# </PageSegment>

			#---- Fetch next Included Page Segment
			$a2wPageSegmentTmp = $svMediaPar->getNextPageSegment();
		}
    	$xmlDocWriter->endTag();				# closing Element IncludedPageSegmentList ==> </IncludedPageSegmentList>
    }
}

#-----------------------------------------------------------------------
# Add NOPs to NOPList
#-----------------------------------------------------------------------
sub addPageNOPs(){

    if ( $bLog == $TRUE ){
        print "addPageNOPs()\n";
    }

	#---- Fetch first NOP
	my $a2wNOPTmp = $a2wPagePar->getFirstNOP();

	if ( $a2wNOPTmp != 0 ){

		$xmlDocWriter->startTag('NOPList');	# starting Element NOPList ==> <NOPList>

		#---- Loop thru the NOPs
		while ( $a2wNOPTmp != 0 ){

			#---- Starting Elmement TextObject
			$xmlDocWriter->dataElement('NOP', $a2wNOPTmp->getValue());

			#---- Fetch next NOP
			$a2wNOPTmp = $a2wPagePar->getNextNOP();
		}
		$xmlDocWriter->endTag();	# closing Element NOPList ==> </NOPList>
	}
}

#-----------------------------------------------------------------------
# Add Text Objects to TextObjectList of Page
#-----------------------------------------------------------------------
sub addTextObjects(){

    #---- Get Parameter (Page or Overlay)
    my $svMediaPar = $_[ 0 ];

    if ( $bLog == $TRUE ){
        print "addTextObjects()\n";
    }

	#---- Fetch first Text Object
	my $a2wTextTmp = $svMediaPar->getFirstText();

	my $a2wFontTmp = "";

	if ( $a2wTextTmp != 0 ){

		$xmlDocWriter->startTag('TextObjectList');	# starting Element TextObjectList ==> <TextObjectList>

		#---- Loop thru the TextObjects
		while ( $a2wTextTmp != 0 ){

			$a2wFontTmp = $a2wTextTmp->getFont();

			#---- Starting Elmement TextObject
			$xmlDocWriter->startTag( 'TextObject',
									 'XPos'			=> $a2wTextTmp->getXPos(),
									 'YPos'			=> $a2wTextTmp->getYPos(),
									 'CodedFont'	=> $a2wFontTmp->getCodedFontName(),
									 'CharacterSet'	=> $a2wFontTmp->getCharacterSetName(),
									 'CodePage'		=> $a2wFontTmp->getCodePageName(),
									 'Typeface'		=> $a2wFontTmp->getTypefaceName(),
									 'Height'		=> $a2wFontTmp->getHeight(),
									 'Width'		=> $a2wFontTmp->getWidth(),
									 'Weight'		=> $a2wFontTmp->isBold()		? "Bold"	: ($a2wFontTmp->isLight() ? "Light" : "Normal"),
									 'Slant'		=> $a2wFontTmp->isItalic()		? "Italic"	: "Medium",
									 'Underline'	=> $a2wFontTmp->isUnderline()	? "yes"		: "no",
									 'Strikeover'	=> $a2wFontTmp->isStrikeover()	? "yes"		: "no",
									 );

			$xmlDocWriter->characters($a2wTextTmp->getText());

			$xmlDocWriter->endTag();	# closing Element TextObject

			#---- Fetch next Text Object
			$a2wTextTmp = $svMediaPar->getNextText();
		}
		$xmlDocWriter->endTag();	# closing Element TextObjectList ==> </TextObjectList>
	}
}

#-----------------------------------------------------------------------
# Add Line Objects to LineObjectList of Page
#-----------------------------------------------------------------------
sub addLineObjects(){

    #---- Get Parameter (Page or Overlay)
    my $svMediaPar = $_[ 0 ];

    if ( $bLog == $TRUE ){
        print "addLineObjects()\n";
    }

	#---- Fetch first Line Object
	my $a2wLineTmp = $svMediaPar->getFirstLine();

	if ( $a2wLineTmp != 0 ){

		$xmlDocWriter->startTag('LineObjectList');	# starting Element LineObjectList ==> <LineObjectList>

		my $LineColorTmp = "";
		my $NegativeLineTmp = "";
		my $LineOrientationTmp = "";

		#---- Loop thru the LineObjects
		while ( $a2wLineTmp != 0 ){

			#---- Starting Elmement LineObject
			$xmlDocWriter->startTag( 'LineObject',
									 'XPos'			=> $a2wLineTmp->getXPos(),
									 'YPos'			=> $a2wLineTmp->getYPos(),
									 'Length'		=> $a2wLineTmp->getLength(),
									 'Width'		=> $a2wLineTmp->getWidth(),
									 'Color'		=> $a2wLineTmp->getColor(),
									 'Orientation'	=> $a2wLineTmp->isHorizontal() ? "Horizontal" : "Vertical",
									 'Negative'		=> $a2wLineTmp->isNegative() ? "yes" : "no",
									 );

			#---- Closing Element LineObject
			$xmlDocWriter->endTag();

			#---- Fetch next Line Object
			$a2wLineTmp = $svMediaPar->getNextLine();
		}
		$xmlDocWriter->endTag();	# closing Element LineObjectList ==> </LineObjectList>
	}
}

__END__
