#-------------------------------------------------------------------------------
#  eyecatcher_lpd.pm:
#  Scripting Facility Module to demonstrate how to process & format LPD input
#  documents and to collect indexes
#
#  Call:
#
#  On Windows:    afp2web.exe -q -c -lpdin -doc_cold -sp:sfsamples\eyecatcher_lpd.pm -sft:ASA_EBCDIC_FIXED_148  samples\lpdsample.lpd
#
#  On Unix   :    ./afp2web   -q -c -lpdin -doc_cold -sp:sfsamples/eyecatcher_lpd.pm -sft:ASA_EBCDIC_FIXED_148  samples/lpdsample.lpd
#
#  Author  : Fa. OXSEED
#  Date    : 2010-07-27
#  Version : 1.0.0
#
#  $V100   2010-07-27    Initial Release
#
#-------------------------------------------------------------------------------

#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );

    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
}

use a2w::Config;
use a2w::Document;
use a2w::Font;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::MediumMap;
use a2w::NOP;
use a2w::Overlay;
use a2w::Page;
use a2w::PSEG;
use a2w::Text;
use a2w::ConfigConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;
use a2w::FontConstants;

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sf") >= 0 ){
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sIndexFilePath    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::INDEXPATH );
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
    $sSpoolFileType    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SPOOLFILETYPE );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    if ( $bLog == $TRUE ){
        printf STDERR ( "Running $sScriptProcTmp...\n" );
        printf STDERR ( "initialize(): Processing $sSpoolFilename\n" );
        printf STDERR ( "initialize(): Args: $sScriptArgsTmp, IndexFilePath: $sIndexFilePath, OutputFilePath: $sOutputFilePath\n" );
    }

    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document

    #---- Initialize Page Id
    $iPageId = 0;

    #---- Page formatting constants
    $iPageWidth  = 842.0;
    $iPageHeight = 595.0;

    $iLeftMargin  = 15.0;
    $iTopMargin   = 0.0;

    $iLineSpacing = 9;

    #---- Create default font
    $a2wDefaultFont = new a2w::Font( $a2w::FontConstants::TYPE_TYPE1, "Courier" );
    $a2wDefaultFont->setHeight( $iLineSpacing );

    $bTRCExist = $FALSE;    # Flag indicating whether records have TRC or not

    #---- Initialize current page index fields line
    $iIndexFieldsLineNr = 64;
    $sIndexFieldsLine   = "";

    #---- Current document id
    $sCurrentDocumentId = "";

    #---- Hash of index field line of documents
    #
    # %hshlstDocIndexFieldsLine{
    #    <First page id as in input spool> = <Index fields line>
    # }
    %hshlstDocIndexFieldsLine = {};

    #---- Set script unit base as Pixels    
    $sScriptUnitBase = "pixel";
    $a2wConfigPar->setAttribute( $a2w::ConfigConstants::SCRIPTUNITBASE, $sScriptUnitBase );

    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializeDoc(): DocId " . $a2wDocumentPar->getId() . "\n" );
    }    
    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializePage()\n" );
    }

    #---- Initialize page index fields line
    $sIndexFieldsLine = "";

    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "afp2web(): PageId " . $a2wPagePar->getParseId() . "\n" );
    }

    #---- Set default return value
    my $iRetTmp = $APPEND; # default: append page

    #---- Format page ----#
    my ( $iRcTmp, $sMsgTmp ) = _formatPage( $a2wPagePar );
    if ( $iRcTmp < 0 ){
        return ( $iRcTmp, $sMsgTmp );
    }

    #---- Identify new document ----#
    # New document is identified whenever the document id on index fields line is changing
    #
    # Syntax of index fields line:
    #  Index:          0         0         0         0         0         0         0         0         0         1         1         1         1         1
    #                  1         2         3         4         5         6         7         8         9         0         1         2         3         4
    #         12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890
    #        >                  0010000001                                                                                       324-1443255-11   03-25-53<
    #                           \_/
    #                            |---> Document ID
    #
    my $sDocIdTmp = "";
    if ( $sIndexFieldsLine =~ /^.{18}(.{3}).*/ ){
        $sDocIdTmp = $1;    # $1 will have document id
    }

    if ( $sCurrentDocumentId ne $sDocIdTmp ){
        #---- Store current document id
        $sCurrentDocumentId = $sDocIdTmp;

        #---- Reset Page Id
        $iPageId = 0;

        $iRetTmp = $NEWDOC;
        if ( $bLog == $TRUE ){
            printf STDERR "Found New Document\n";
        }

        #---- Store index fields of current document
        %hshlstDocIndexFieldsLine->{ $a2wPagePar->getParseId() } = $sIndexFieldsLine;
    }

    #---- Increment Page Id
    $iPageId++;

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizePage()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizeDoc()\n" );
    }

    #---- Process and fetch document indexes ----#
    my $a2wFirstPageTmp = $a2wDocumentPar->getFirstPage();

    #---- Fetch page id
    my $iPageIdTmp = $a2wFirstPageTmp->getParseId();

    #---- Fetch current document index fields
    my $sIndexFieldsTmp = %hshlstDocIndexFieldsLine->{ $iPageIdTmp };
    
    #---- Process index fields
    my $hshIndexesTmp = _processAndFetchIndexFields( $sIndexFieldsTmp );
    
    #---- Reset current document index fields
    %hshlstDocIndexFieldsLine->{ $iPageIdTmp } = "";

    #---- Assert processed index value
    if ( $hshIndexesTmp == undef ){
        return ( -999, "Unable to fetch indexes for document " . $a2wDocumentPar->getId() );
    }

    #---- Write index to file ----#
    #---- Build Index Filename
    my $sSimpleFilenameTmp = $a2wDocumentPar->getSimpleFilename();    # get document simple filename
    if ( $sSimpleFilenameTmp eq "" ){
        return ( -997, "Invalid Simple Filename(" . $sSimpleFilenameTmp . ") for document " . $a2wDocumentPar->getId() );
    }

    my $IndexFilenameTmp = $sIndexFilePath . $sSimpleFilenameTmp . ".idx"; # build Index Filename
    if ( $bLog == $TRUE ){
        printf STDERR ( "Writing Index File ==> $IndexFilenameTmp\n" );
    }

    #---- Open Index file
    my $bFileOpenSuccessTmp = open( fIndexFile, ">$IndexFilenameTmp" );
    if ( !$bFileOpenSuccessTmp ){
        return ( -998, "Unable to open index file $IndexFilenameTmp, rc=" . $bFileOpenSuccessTmp . "msg=" . $! );
    }

    #---- Fetch index name list
    my @arrIndexNameListTmp = sort keys( %{ $hshIndexesTmp } );
    for ( my $i = 0; $i < @arrIndexNameListTmp; $i++ ){
        print fIndexFile ( $arrIndexNameListTmp[ $i ] . "=" . $hshIndexesTmp->{ $arrIndexNameListTmp[ $i ] } . "\n" );
    }
    close( fIndexFile );

    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalize()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Process index fields
#
# Parameter(s)
# 1. String having index fields
#
# Returns
# - undef in case of error
#
# - Hash reference having indexes in case of success, hash elements would
#   be as given below
#   %hshCurrentDocIndexes = {
#         'I_DOC_ID'      => ""         # Current document id (e.g. 001)
#       , 'I_INSURED'     => ""         # Insured (e.g. Geoffrey R Stephens)
#       , 'I_POLICY_NR'   => ""         # Policy Number (e.g. 324-1443255-11)
#       , 'I_ISSUED_DATE' => ""         # Date of issue (e.g. 03-25-53)
#   };
#
# Remarks
# Processes given index fields line, splits indexes and returns the same
#
#-----------------------------------------------------------------------
sub _processAndFetchIndexFields(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "_processAndFetchIndexFields()\n" );
    }

    #---- Get parameter(s)
    #
    # 1. String having index fields
    #
    my $sIndexFieldsTmp = shift;

    if ( $bLog == $TRUE ){
        printf STDERR ( "\tIndexFields=" . $sIndexFieldsTmp . "\n" );
    }

    #---- Return value
    my $hshIndexFieldsTmp = undef;

    # Syntax of index fields line:
    #  Index:          0         0         0         0         0         0         0         0         0         1         1         1         1         1
    #                  1         2         3         4         5         6         7         8         9         0         1         2         3         4
    #         12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890
    #        >                  0010000001 Geofrey R Stephens                                                                    324-1443255-11   03-25-53<
    #                           \_/\_____/ \________________/                                                                    \____________/   \______/
    #                            |    |           |                                                                                      |             |
    #  Document Id <--------------    |           ----------> Insured                                      Policy Number <----------------             |
    #  Page Id     <-------------------                                                                    Date of issue <------------------------------
    #
    if ( $sIndexFieldsTmp =~ /^.{18}(.{3}).{8}(.{30}).{56}(.{14}).{3}(.{8}).*$/ ){
        $hshIndexFieldsTmp = {};
        $hshIndexFieldsTmp->{ 'I_DOC_ID' }      = $1;
        $hshIndexFieldsTmp->{ 'I_INSURED' }     = $2;
        $hshIndexFieldsTmp->{ 'I_POLICY_NR' }   = $3;
        $hshIndexFieldsTmp->{ 'I_ISSUED_DATE' } = $4;

        if ( $bLog == $TRUE ){
            printf STDERR ( "Index: DocumentID=$1, Insured=$2, PolicyNr=$3, Date of Issue=$4\n" );
        }

        #---- Strip trailing spaces
        $hshIndexFieldsTmp->{ 'I_INSURED' } =~ s/\s+$//g;
    }

    return $hshIndexFieldsTmp;
}

#-----------------------------------------------------------------------
# Format page
#
# Parameter(s)
# 1. Page (of type a2w::Page)
#
# Returns
# >0 in case of success
# <0 (+ the reason message) in case of error
#
# Remarks
# Processes each text (aka line text) of input and formats it based on
# ASA carriage controls
#
# *** IMPORTANT NOTE ON PROCESSING LINES ***:
# AFP2web will process each line of input and will make a text object
# out of each line of text. So processing a text object mean processing
# a line from input.
#
# Also the text (aka line) will have Carriage Control character at the
# beginning which can be processed in below function for formatting the
# lines
#
#-----------------------------------------------------------------------
sub _formatPage(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "_formatPage()\n" );
    }

    #---- Get parameter(s)
    #
    # 1. Page (of type a2w::Page)
    #
    my $a2wCurrentPageTmp = shift;

    #---- Fetch page resolution
    my $iPageResTmp = $a2wCurrentPageTmp->getResolution();

    #---- Set new page width/height
    $a2wCurrentPageTmp->setHeight( $iPageHeight );
    $a2wCurrentPageTmp->setWidth( $iPageWidth );

    #---- Evaluate line height
    my $fFactorTmp = $iPageResTmp / 72;

    if ( lc( $sScriptUnitBase ) eq "mm" ){
        $fFactorTmp = 25.4 / 72;
    }

    #---- Initialize LineId
    my $iLineIdTmp = 0;

    #---- Origin is at the bottom left corner
    my $iLeftMarginTmp = $iLeftMargin * $fFactorTmp;
    my $iTopMarginTmp  = $iTopMargin * $fFactorTmp;

    #---- Modify boundary evaluation based on Top-Left co-ordinate system
    my $iXPosTmp = $iLeftMarginTmp;    # X Position starts from Left Margin
    my $iYPosTmp = $iTopMarginTmp;     # Y Position is "Top Margin"

    #---- Process and format page content ----#
    my $a2wTextTmp = $a2wCurrentPageTmp->getFirstText();

    #---- Format each and every text (aka line text)
    my $sTextTmp  = "";
    my $cCCCharTmp = "";
    my $cTRCharTmp = "";
    my $sTextWithoutCCTmp = "";

    #---- Index Field Insured Name
    my $sInsuredNameTmp = "";
    
    while ( $a2wTextTmp != 0 ){
        $iLineIdTmp++;                 # Increment line number
        $iXPosTmp = $iLeftMarginTmp;   # X Position starts from Left Margin

        #---- Assert Y position
        #---- Modify boundary evaluation based on Top-Left co-ordinate system
        if ( $iYPosTmp > $iPageHeight ){
            if ( $bLog == $TRUE ){
                printf STDERR "\tERROR! Spool File Type (" . $sSpoolFileType . "), Line=" . $iLineIdTmp . ": Not enough space on page.\n";
            }

            return ( -98, "Spool File Type (" . $sSpoolFileType . "), Line=" . $iLineIdTmp . ": Not enough space on page." );
        }

        #---- Process current line and add texts appropriately
        $sTextWithoutCCTmp = "";
        $sTextTmp = $a2wTextTmp->getText();

        #---- Collect current page's index fields line
        if ( $iLineIdTmp == $iIndexFieldsLineNr ){
            $sIndexFieldsLine = $sTextTmp;
        }

        if ( $bTRCExist == $TRUE ){
            if ( $sTextTmp =~ /(.)(.)(.*)/ ){
                $cCCCharTmp        = $1;   # Contains CC
                $cTRCharTmp        = $2;   # Contains TRC
                $sTextWithoutCCTmp = $3;   # Contains line data
            }
        }
        else {
            if ( $sTextTmp =~ /(.)(.*)/ ){
                $cCCCharTmp        = $1;   # Contains CC
                $sTextWithoutCCTmp = $2;   # Contains line data
            }
        }

        #----- ASA Carriage Control action is applied before printing current line
        #      so process apply actions for given CC character
        #
        if ( $cCCCharTmp eq "+" ){                   # Do not space
            # Overwrite existing line
            $iYPosTmp -= $iLineSpacing;
        }
        elsif ( $cCCCharTmp eq " " ){                # Space one line
            # One line space is already evaluated,
            # so nothing to do here
        }
        elsif ( $cCCCharTmp eq "0" ){                # Space two lines
            # One line space is already evaluated,
            # so leave only one line space now
            $iYPosTmp += $iLineSpacing;
        }
        elsif ( $cCCCharTmp eq "-" ){                # Space three lines
            # One line space is already evaluated,
            # so leave only two line space now
            $iYPosTmp += $iLineSpacing;
            $iYPosTmp += $iLineSpacing;
        }
        elsif ( $cCCCharTmp eq "1" ){                # Skip to channel 1/New page
            # A2W kernel itself identifies new page,
            # so nothing to do here
        }
        elsif ( $cCCCharTmp eq "2" ){                # Skip to channel 2
        }
        elsif ( $cCCCharTmp eq "3" ){                # Skip to channel 3
        }
        elsif ( $cCCCharTmp eq "4" ){                # Skip to channel 4
        }
        elsif ( $cCCCharTmp eq "5" ){                # Skip to channel 5
        }
        elsif ( $cCCCharTmp eq "6" ){                # Skip to channel 6
        }
        elsif ( $cCCCharTmp eq "7" ){                # Skip to channel 7
        }
        elsif ( $cCCCharTmp eq "8" ){                # Skip to channel 8
        }
        elsif ( $cCCCharTmp eq "9" ){                # Skip to channel 9
        }
        elsif ( $cCCCharTmp eq "A" ){                # Skip to channel 10
        }
        elsif ( $cCCCharTmp eq "B" ){                # Skip to channel 11
        }
        elsif ( $cCCCharTmp eq "C" ){                # Skip to channel 12
        }

        #---- Skip trailing spaces
        $sTextWithoutCCTmp =~ s/\s+$//g;

        #---- Skip empty lines
        if ( length( $sTextWithoutCCTmp ) <= 0 ){
            #---- Evaluate next YPos (YPos = YPos - FontHeight)
            $iYPosTmp += $iLineSpacing;

            #---- Don't present this text on output
            $a2wTextTmp->remove();
        }
        else {
            if ( $bLog == $TRUE ){
                printf STDERR ( "(" . $iXPosTmp . ", " . $iYPosTmp . ")=>" . $sTextWithoutCCTmp . "<=\n" );
            }

            #---- Fill in text details
            $a2wTextTmp->setText( $sTextWithoutCCTmp ); # Text value
            $a2wTextTmp->setXPos( $iXPosTmp );          # Text X position
            $a2wTextTmp->setYPos( $iYPosTmp );          # Text Y position
            $a2wTextTmp->setFont( $a2wDefaultFont );    # Text font

            #---- Evaluate next YPos
            $iYPosTmp += $iLineSpacing;
        }

        #---- Get next text
        $a2wTextTmp = $a2wCurrentPageTmp->getNextText();
    }

    if ( $bLog == $TRUE ){
        printf STDERR ( "\tIndexFields=" . $sIndexFieldsLine . "\n" );
    }

    return 0;
}

__END__
