#-------------------------------------------------------------------------------
#  addUserData.pm: Scripting Facility Module to add user data and to fetch those
#                  added values at Java SDK
#
#  Call:
#
#  On Windows:    afp2web.exe -q -c -doc_cold -sp:sfsamples\addUserData.pm samples\insure.afp
#
#  On Unix:    ./afp2web -q -c -doc_cold -sp:sfsamples/addUserData.pm samples/insure.afp
#
#  Author  : Fa. OXSEED
#  Date    : 2010-01-04
#  Version : 1.0.1
#
#  $V100   2009-02-06    Initial Release
#
#  $V101   2010-01-04    a. Extended to utilize
#                           "a2w::ConfigConstants"
#                           "a2w::DocumentConstants"
#                           "a2w::PageConstants"
#                           "a2w::FontConstants"
#                           packages
#                        b. Optimized constant declarations
#                        c. Upgraded to V4.x standards
#                        d. Modified PERL "@INC" extending to consider the
#                           Scripting Facility samples grouping under "sfsamples"
#                           directory
#-------------------------------------------------------------------------------

#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );
    # $V101 Begin
    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }
    # $V101 End

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
}

use a2w::Config;
use a2w::Document;
use a2w::Font;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::MediumMap;
use a2w::NOP;
use a2w::Overlay;
use a2w::Page;
use a2w::PSEG;
use a2w::Text;
use a2w::UserData;
# $V101 Begin
use a2w::ConfigConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;
use a2w::FontConstants;
# $V101 End
#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sf") >= 0 ){    # $V101 Change
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );    # $V101 Change
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sIndexFilePath    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::INDEXPATH );
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
    $sDocType          = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFORMAT );    
    $sMemoryStream     = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::MEMORYOUTPUTSTREAM );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    if ( $bLog == $TRUE ){
        print "Running $sScriptProcTmp...\n";
        print "initialize(): Processing $sSpoolFilename\n";
        print "initialize(): Args: $sScriptArgsTmp, IndexFilePath: $sIndexFilePath, OutputFilePath: $sOutputFilePath\n";
    }

    # $V101 Begin
    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document
    # $V101 End

    #---- Initialize Page Id
    $PageId = 0;

    #---- Set AutoSplit to true
#INFO: If your AFP Spool contains Page Named Groups and Tag Logical 
#INFO: Elements (TLEs) and if you want AFP2web to automatically split
#INFO: the documents, set AutoSplit on, if not remove following line
    $a2wConfigPar->setAttribute( $a2w::ConfigConstants::AUTOSPLIT, "on" );    # $V101 Change

    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializeDoc(): DocId " . $a2wDocumentPar->getId() . "\n" );
    }

    #---- Reset Page Id
    $PageId = 0;

#INFO: This is a sample for fetching the document indexes as one String
    my $sIndexDataTmp = fetchDocumentIndexes();

#INFO: This is a sample for adding user data to document
    #---- Create user data object
    my $a2wUserDataTmp = new a2w::UserData();

    #---- Set document id as user data id
    $a2wUserDataTmp->setId( "DOCINDEX" );

    #---- Set document index as data
    $a2wUserDataTmp->setData( $sIndexDataTmp, length( $sIndexDataTmp )  );

    #---- Add data to document
    $a2wDocumentPar->addUserData( $a2wUserDataTmp );

    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializePage()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "afp2web(): PageId " . $a2wPagePar->getParseId() . "\n" );
    }

    #---- Set default return value
    my $iRetTmp = $APPEND; # append page to Current Document

    #---- Increment Page Id
    $PageId++;

#INFO: If you need to get Information from the Page (NOPs, TLEs, Overlays,etc...), do it here.
#INFO: If you need to add/remove Information to/from the Page, do it here.

#INFO: This is a sample for fetching the page indexes as one String
    my $sIndexDataTmp = fetchPageIndexes();

#INFO: This is a sample for adding user data to document
    #---- Create user data object
    my $a2wUserDataTmp = new a2w::UserData();

    #---- Set document and page id as user data id
    $a2wUserDataTmp->setId( "PAGE" . "_" . $PageId . "_" . "INDEX" );

    #---- Set current page index as data
    $a2wUserDataTmp->setData( $sIndexDataTmp, length( $sIndexDataTmp )  );

    #---- Add data to document
    $a2wDocumentPar->addUserData( $a2wUserDataTmp );

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizePage()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizeDoc()\n" );
    }

#INFO: If you need to write Index Files, do it here.
#INFO: If you need to post process the Output File (for example, sending 
#INFO: it to the archive, do it here.
#INFO: You'll find an example in autosplit.pm and eyecatcher.pm

    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalize()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Fetch Document Indexes
#-----------------------------------------------------------------------
sub fetchDocumentIndexes(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "fetchDocumentIndexes()\n" );
    }

    #---- Prepare Document Id as index
    my $sRetIndexDataTmp = "DocId=" . $a2wDocumentPar->getId() . "\r\n";

    #---- Fetch first Document Index
    my $a2wDocIndexTmp = $a2wDocumentPar->getFirstIndex();

    #---- Add PageGroup Name
    if ( $a2wDocIndexTmp != 0 ){

        #---- Add PageGroup Name 
        $sRetIndexDataTmp .= "PageGroup=" . $a2wDocIndexTmp->getIndexedObjectName() . "\r\n";
    }

    while ( $a2wDocIndexTmp != 0 ){

        #---- Add Index Record
        $sRetIndexDataTmp .= $a2wDocIndexTmp->getName() . "=" . $a2wDocIndexTmp->getValue() . "\r\n";

        #---- Fetch next Document Index
        $a2wDocIndexTmp = $a2wDocumentPar->getNextIndex();
    }

    return $sRetIndexDataTmp;
}

#-----------------------------------------------------------------------
# Fetch Page Indexes
#-----------------------------------------------------------------------
sub fetchPageIndexes(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "fetchPageIndexes()\n" );
    }

    #---- Prepare Page Id
    my $sRetIndexDataTmp = "PageId=$PageId\r\n";

    #---- Fetch first Page Index
    my $a2wPageIndexTmp = $a2wPagePar->getFirstIndex();

    while ( $a2wPageIndexTmp != 0 ){

        #---- Add Index Record
        $sRetIndexDataTmp .= $a2wPageIndexTmp->getName() . "=" . $a2wPageIndexTmp->getValue() . "\r\n";

        #---- Fetch next Page Index
        $a2wPageIndexTmp = $a2wPagePar->getNextIndex();
    }

    return $sRetIndexDataTmp;
}

__END__
