/**
 * Custom document writer
 *
 *  Author  : Fa. OXSEED
 *  Date    : 2009-03-24
 *  Version : 1.0.1
 *
 *  $V100   2009-03-20    Initial Release
 *
 *  $V101   2009-03-24	  a. Extended ini path fetching at process beginning (fetching after will lose
 *                           ini path value as it will be removed by Java SDK APIs)
 *                        b. Extended to consider INIFILENAME property value for building output filenames
 *
 *  $V102   2009-09-25    Extended with "PNG" output support
 */

// Import required packages
import de.maas.afp2web.A2WDocumentWriter;

import de.maas.afp2web.A2WSDK;
import de.maas.afp2web.A2WIniConstants;
import de.maas.afp2web.A2WDocumentObject;
import de.maas.afp2web.A2WIndex;
import de.maas.afp2web.A2WUserData;

import java.lang.Exception;

import java.io.FileOutputStream;
import java.io.ByteArrayOutputStream;

import java.util.LinkedList;
import java.util.ListIterator;
import java.util.Properties;

/**
 * <p>
 * Custom document writer.
 *
 * Stores passed document buffer on file and write down the indexes and user data elements
 * </p>
 */
public class CustomDocumentWriter implements A2WDocumentWriter{
    // Attributes
    // A2W SDK parameters
    Properties propSDKParams;

    // Previous spool name
    private String sPreviousSpoolname;

    // Unique id
    private String sUniqueId;

    // Output document id
    private long lDocumentId;

    // $V101 Begin
    // Ini path
    private String sA2WIniPath;
    // $V101 End

    // Default contructor
    public CustomDocumentWriter(){
        sPreviousSpoolname = null;
        sUniqueId = null;
        lDocumentId = 0;
        propSDKParams = null;
        // $V101 Begin
        sA2WIniPath = null;
        // $V101 End
    }

    // Mutators
    /**
     * Set SDK parameters
     */
    void setSDKParams( Properties propSDKParamsPar ){
        propSDKParams = propSDKParamsPar;

        // $V101 Begin
        if ( propSDKParams != null ){
            // Fetch ini path right here, fetching later will lose value
            sA2WIniPath = propSDKParams.getProperty( A2WIniConstants.INIPATH );

            // Try "IniPath" constant as key for backward compatibility
            if ( sA2WIniPath == null ){
                sA2WIniPath = propSDKParams.getProperty( "IniPath" );
            }

            // Fetch ini filename
            String sA2WIniFilenameTmp = propSDKParams.getProperty( A2WIniConstants.INIFILENAME );
            if ( sA2WIniFilenameTmp != null ){
                // Extract path
                int iLastSlashIndexTmp = sA2WIniFilenameTmp.lastIndexOf( '\\' );
                if ( iLastSlashIndexTmp == -1 ){
                    iLastSlashIndexTmp = sA2WIniFilenameTmp.lastIndexOf( '/' );
                }

                if ( iLastSlashIndexTmp != -1 ){
                    sA2WIniPath = sA2WIniFilenameTmp.substring( 0, iLastSlashIndexTmp  );
                }
            }

            // Check and add slash at end
            if ( sA2WIniPath != null ){
                if ( !sA2WIniPath.endsWith( "\\" )
                     && !sA2WIniPath.endsWith( "/" )
                   ){
                    sA2WIniPath += "/";
                }
            }
        }
        // $V101 End
    }

    // Accessors
    /**
     * Get SDK parameters
     */
    Properties getSDKParams(){
        return propSDKParams;
    }

    // Workers
    /**
     * Callback to write document with default definition
     *
     * <p>
     * @param a2wDocObjectPar Document object that has output buffer, index list and user data list
     * <br>
     * <br>
     * @return 
     * <PRE>
     *  0 on Success
     * <0 on Error
     * </PRE>
     * </p>
     */
    public int write( A2WDocumentObject a2wDocObjectPar ){
        //---- Assert parameters ----//
        if ( a2wDocObjectPar == null ){
            return -1;
        }

        //---- Fetch document buffer, index list and user data list ----//
        ByteArrayOutputStream baosDocBufferTmp = a2wDocObjectPar.getOutputStream();
        LinkedList lstDocIndexTmp              = a2wDocObjectPar.getIndexList();
        LinkedList lstDocUserDataTmp           = a2wDocObjectPar.getUserDataList();

        //---- Assert parameters ----//
        if ( baosDocBufferTmp == null
             || baosDocBufferTmp.size() <= 0 ){
            return -1;
        }

        //---- Fetch AFP2web SDK singleton instance ----//
        A2WSDK a2wsdkThe = A2WSDK.getSingleton();

        //---- Fetch active spool name ----//
        String sActiveSpoolnameTmp = a2wsdkThe.getActiveSpoolname();
        if ( !sActiveSpoolnameTmp.equals( sPreviousSpoolname ) ){
            sPreviousSpoolname = sActiveSpoolnameTmp;

            // Generate unique id for each spool
            sUniqueId = generateUniqueId();

            // Reset output document id
            lDocumentId = 0;
        }

        String sFileCreationModeTmp = null;
        String sOutputFormatTmp     = null;
        String sOutputFilePathTmp   = null;

        //--- Get attributes from A2WSDK parameters
        if ( propSDKParams != null ){
            //--- Get FileCreationMode
            sFileCreationModeTmp = propSDKParams.getProperty( A2WIniConstants.FILECREATIONMODE );

            //--- Get Output Format 
            sOutputFormatTmp = propSDKParams.getProperty( A2WIniConstants.OUTPUTFORMAT );

            //---- Get output filepath
            sOutputFilePathTmp = propSDKParams.getProperty( A2WIniConstants.OUTPUTFILEPATH );
        }
        // Assert parameters and assume default if not specified
        if ( sFileCreationModeTmp  == null
             || sFileCreationModeTmp .length() <= 0 ){
            sFileCreationModeTmp = "all" ;
        }
        if ( sOutputFormatTmp == null
             || sOutputFormatTmp.length() <= 0 ){
            sOutputFormatTmp = "pdf" ;
        }
        if ( sOutputFilePathTmp == null
             || sOutputFilePathTmp.length() <= 0 ){
            sOutputFilePathTmp = "pdf" ;
        }

        String sSimpleSpoolnameTmp = extractSimpleFilename( sActiveSpoolnameTmp );

        // Increment output document id
        lDocumentId++;

        //---- Build output filename ----//
        String sSimpleOutputFilenameTmp = sSimpleSpoolnameTmp;

        if ( sFileCreationModeTmp != null 
             && ( sFileCreationModeTmp.equalsIgnoreCase( "doc_index" )
                  || sFileCreationModeTmp.equalsIgnoreCase( "doc_cold" )
                  || sFileCreationModeTmp.equalsIgnoreCase( "page" )
                 )
            ){
            sSimpleOutputFilenameTmp += "_" + sUniqueId + "." + lDocumentId;
        }

        //---- Process index details ----//
        if ( lstDocIndexTmp != null ){
            String sIndexFilenameTmp = null;
            if ( sA2WIniPath != null ){
                sIndexFilenameTmp = sA2WIniPath + sOutputFilePathTmp + "/" + sSimpleOutputFilenameTmp + ".idx";
            }
            else {
                sIndexFilenameTmp = sOutputFilePathTmp + "/" + sSimpleOutputFilenameTmp + ".idx";
            }

            try {
                // Create index file stream
                FileOutputStream fosIndexTmp = new FileOutputStream( sIndexFilenameTmp );

                ListIterator lstitrIndexTmp = lstDocIndexTmp.listIterator( 0 );

                if ( lstitrIndexTmp != null ){
                    String sIndexEntryTmp = null;
                    A2WIndex a2wIndexTmp = null;

                    while ( lstitrIndexTmp.hasNext() ){
                        a2wIndexTmp = (A2WIndex)lstitrIndexTmp.next();

                        // Build index entry and write it on index file
                        if ( a2wIndexTmp != null ){
                            sIndexEntryTmp = a2wIndexTmp.getName() + "=" + a2wIndexTmp.getValue() + "\n";

                            fosIndexTmp.write( sIndexEntryTmp.getBytes() );
                        }
                    }
                }
            }
            catch ( Exception eFileErrorPar ){
                System.out.println( eFileErrorPar );
            }
        }

        //---- Process UserData details ----//
        if ( lstDocUserDataTmp != null ){
            String sUserDataBaseFilenameTmp = null;
            String sUserDataFilenameTmp = null;
            if ( sA2WIniPath != null ){
                sUserDataBaseFilenameTmp = sA2WIniPath + sOutputFilePathTmp + "/" + sSimpleOutputFilenameTmp;
            }
            else {
                sUserDataBaseFilenameTmp  = sOutputFilePathTmp + "/" + sSimpleOutputFilenameTmp;
            }

            try {
                
                A2WUserData a2wUserDataTmp = null;    
                FileOutputStream fosUserDataTmp = null;
                
                ListIterator lstitrUserDataTmp = lstDocUserDataTmp.listIterator( 0 );

                if ( lstitrUserDataTmp != null ){
                    while ( lstitrUserDataTmp.hasNext() ){
                        a2wUserDataTmp = (A2WUserData)lstitrUserDataTmp.next();

                        // Write User Data to a File 
                        if ( a2wUserDataTmp != null
                              && a2wUserDataTmp.getData() != null
                            ){
                            
                            // Build filename to store User Data
                            sUserDataFilenameTmp = sUserDataBaseFilenameTmp + "_" + a2wUserDataTmp.getId() + ".dat";
                            
                            // Create file stream for User data
                            fosUserDataTmp = new FileOutputStream( sUserDataFilenameTmp  );
                            
                            fosUserDataTmp.write( a2wUserDataTmp.getData() );
                            fosUserDataTmp.close();
                            fosUserDataTmp = null;
                        } //if
                    }// while
                }//if
            } //try
            catch ( Exception eFileErrorPar ){
                System.out.println( eFileErrorPar );
            }
        }        
        
        //---- Process document buffer ----//
        try {
            String sOutputFilenameTmp = null;
            if ( sA2WIniPath != null ){
                sOutputFilenameTmp = sA2WIniPath + sOutputFilePathTmp + "/" + sSimpleOutputFilenameTmp + fetchExtension( sOutputFormatTmp );
            }
            else {
                sOutputFilenameTmp = sOutputFilePathTmp + "/" + sSimpleOutputFilenameTmp + fetchExtension( sOutputFormatTmp );
            }

            // Create output file stream
            FileOutputStream fosDocumentTmp = new FileOutputStream( sOutputFilenameTmp );

            // Write document to file
            baosDocBufferTmp.writeTo( fosDocumentTmp );
        }
        catch ( Exception eFileErrorPar ){
            System.out.println( eFileErrorPar );
        }

        return 0;
    }

    /**
     * Simple filename extractor
     *
     * <p>
     * @param sFilenamePar Filename
     * <br>
     * <br>
     * @return 
     * "Simple filename" as String
     * </p>
     */
    private String extractSimpleFilename( String sFilenamePar ){
        //---- Assert parameter ----//
        if ( sFilenamePar == null
             || sFilenamePar.length() <= 0 ){
            return null;
        }

        // Extract simple filename
        int iLastSlashIndexTmp = sFilenamePar.lastIndexOf( '\\' );
        if ( iLastSlashIndexTmp == -1 ){
            iLastSlashIndexTmp = sFilenamePar.lastIndexOf( '/' );
        }

        String sSimpleFilenameTmp = null;
        if ( iLastSlashIndexTmp != -1 ){
            sSimpleFilenameTmp = sFilenamePar.substring( iLastSlashIndexTmp, sFilenamePar.lastIndexOf( '.' ) );
        }
        else {
            sSimpleFilenameTmp = sFilenamePar;
        }

        return sSimpleFilenameTmp;
    }

    /**
     * Generates unique id
     *
     * <p>
     * @return 
     * Generated unique id
     * </p>
     */
    private String generateUniqueId(){
        int iMaxUniqueIdLenTmp = 5;

        char cRandomCharTmp = '\0';
        String sUniqueIdTmp = "";
        String sPossibleCharsTmp = "abcdefghijklmonpqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
        int iPossibleCharsLenTmp = sPossibleCharsTmp.length();

        //---- Generate unique id ----//
        while( sUniqueIdTmp.length() < iMaxUniqueIdLenTmp ){
            // Get a random character
            cRandomCharTmp = (char)sPossibleCharsTmp.charAt( (int)( java.lang.Math.random() * iPossibleCharsLenTmp ) );

            // Add above generated random char to unique string, only if it is not added already
            if ( sUniqueIdTmp.indexOf( cRandomCharTmp ) < 0 ){
                sUniqueIdTmp += cRandomCharTmp;
            }
        }

        return sUniqueIdTmp;
    }

    /**
     * Fetch filename extension for given output format
     *
     * <p>
     * @return 
     * Filename extension as string
     * </p>
     */
    private String fetchExtension( String sFormatPar ){
        if ( sFormatPar == null ){
            return "";
        }

        String sExtensionTmp = "";

        // PDF and PDF/A format
        if ( sFormatPar.equalsIgnoreCase( "pdf" )
             || sFormatPar.equalsIgnoreCase( "pdfa" ) ){
            sExtensionTmp = ".pdf";
        }
        // TIFF format
        else if ( sFormatPar.equalsIgnoreCase( "tif" )
              || sFormatPar.equalsIgnoreCase( "tiff" ) ){
            sExtensionTmp = ".tif";
        }
        // ASCII format
        else if ( sFormatPar.equalsIgnoreCase( "ascii" ) ){
            sExtensionTmp = ".txt";
        }
        // AFP format
        else if ( sFormatPar.equalsIgnoreCase( "afp" ) ){
            sExtensionTmp = ".afp";
        }
        // JPEG format
        else if ( sFormatPar.equalsIgnoreCase( "jpg" )
              || sFormatPar.equalsIgnoreCase( "jpeg" ) ){
            sExtensionTmp = ".jpg";
        }
        // $V102 Begin
        // PNG format
        else if ( sFormatPar.equalsIgnoreCase( "png" ) ){
            sExtensionTmp = ".png";
        }
        // $V102 End

        return sExtensionTmp;
    }
}
