/**
 * Custom document handler
 *
 *  Author  : Fa. OXSEED
 *  Date    : 2009-03-24
 *  Version : 1.0.5
 *
 *  $V100   2006-06-02    Initial Release
 *  $V101   2006-10-05    Extended for PDF/A output
 *  $V102   2007-05-03    Extended for AFP and JPEG outputs
 *  $V103   2009-02-03	  Extended with overloaded "handle( A2WDocumentObject )" abstract implementation
 *  $V104   2009-03-20	  a. Removed overloaded "handle" (to pass in user data list)
 *                        b. Extended output file storing to consider "OutputFilePath" INI attribute
 *  $V105   2009-03-24	  a. Extended ini path fetching at process beginning (fetching after will lose
 *                           ini path value as it will be removed by Java SDK APIs)
 *                        b. Extended to consider INIFILENAME property value for building output filenames
 *
 *  $V106   2009-09-25  Extended with "PNG" output support
 */

// Import required packages
import de.maas.afp2web.A2WSDK;
import de.maas.afp2web.A2WIniConstants;
import de.maas.afp2web.A2WIndex;
import de.maas.afp2web.A2WDocumentHandler;

import java.lang.Exception;

import java.io.FileOutputStream;
import java.io.ByteArrayOutputStream;

import java.util.LinkedList;
import java.util.ListIterator;
import java.util.Properties;

/**
 * <p>
 * Custom document handler.
 *
 * Stores passed document buffer on file and write down the indexes
 * </p>
 */
public class CustomDocumentHandler implements A2WDocumentHandler{
    // Attributes
    // A2W SDK parameters
    Properties propSDKParams;
     
    // Previous spool name
    private String sPreviousSpoolname;

    // Unique id
    private String sUniqueId;

    // Output document id
    private long lDocumentId;

    // $V105 Begin
    // Ini path
    private String sA2WIniPath;
    // $V105 End

    // Default contructor
    public CustomDocumentHandler(){
        sPreviousSpoolname = null;
        sUniqueId = null;
        lDocumentId = 0;
        propSDKParams = null;
        // $V105 Begin
        sA2WIniPath = null;
        // $V105 End
    }

    // Mutators
    /**
     * Set SDK parameters
     */
    void setSDKParams( Properties propSDKParamsPar ){
        propSDKParams = propSDKParamsPar;

        // $V105 Begin
        if ( propSDKParams != null ){
            // Fetch ini path right here, fetching later will lose value
            sA2WIniPath = propSDKParams.getProperty( A2WIniConstants.INIPATH );

            // Try "IniPath" constant as key for backward compatibility
            if ( sA2WIniPath == null ){
                sA2WIniPath = propSDKParams.getProperty( "IniPath" );
            }

            // Fetch ini filename
            String sA2WIniFilenameTmp = propSDKParams.getProperty( A2WIniConstants.INIFILENAME );
            if ( sA2WIniFilenameTmp != null ){
                // Extract path
                int iLastSlashIndexTmp = sA2WIniFilenameTmp.lastIndexOf( '\\' );
                if ( iLastSlashIndexTmp == -1 ){
                    iLastSlashIndexTmp = sA2WIniFilenameTmp.lastIndexOf( '/' );
                }

                if ( iLastSlashIndexTmp != -1 ){
                    sA2WIniPath = sA2WIniFilenameTmp.substring( 0, iLastSlashIndexTmp  );
                }
            }

            // Check and add slash at end
            if ( sA2WIniPath != null ){
                if ( !sA2WIniPath.endsWith( "\\" )
                     && !sA2WIniPath.endsWith( "/" )
                   ){
                    sA2WIniPath += "/";
                }
            }
        }
        // $V105 End
    }

    // Accessors
    /**
     * Get SDK parameters
     */
    Properties getSDKParams(){
        return propSDKParams;
    }

    // Workers
    /**
     * <p>
     * Callback to handle document archiving with default definition 
     * <br>
     * 
     * @param baosOutputStreamPar Stream (java.io.ByteArrayOutputStream) to receive output document buffer
     * <br>
     * @param lstDocumentIndexPar Linked list (java.util.LinkedList) of indexes (de.maas.afp2web.A2WIndex) to receive output document indexes
     * <br>
     * <br>
     * @return 
     * <PRE>
     *  0 on Success
     * <0 on Error
     * </PRE>
     * </p>
     */     
    public int handle( ByteArrayOutputStream baosOutputStreamPar,
                       LinkedList lstDocumentIndexPar
                     ){
        //---- Assert parameters ----//
        if ( baosOutputStreamPar == null
             || baosOutputStreamPar.size() <= 0 ){
            return -1;
        }

        //---- Fetch AFP2web SDK singleton instance ----//
        A2WSDK a2wsdkThe = A2WSDK.getSingleton();

        //---- Fetch active spool name ----//
        String sActiveSpoolnameTmp = a2wsdkThe.getActiveSpoolname();
        if ( !sActiveSpoolnameTmp.equals( sPreviousSpoolname ) ){
            sPreviousSpoolname = sActiveSpoolnameTmp;

            // Generate unique id for each spool
            sUniqueId = generateUniqueId();

            // Reset output document id
            lDocumentId = 0;
        }

        String sFileCreationModeTmp = null;
        String sOutputFormatTmp     = null;
        String sOutputFilePathTmp   = null;            // $V104 Change

        //--- Get attributes from A2WSDK parameters
        if ( propSDKParams != null ){
            //--- Get FileCreationMode
            sFileCreationModeTmp = propSDKParams.getProperty( A2WIniConstants.FILECREATIONMODE );

            //--- Get Output Format 
            sOutputFormatTmp = propSDKParams.getProperty( A2WIniConstants.OUTPUTFORMAT );

            //---- Get output filepath
            sOutputFilePathTmp = propSDKParams.getProperty( A2WIniConstants.OUTPUTFILEPATH );
        }

        // $V104 Begin
        // Assert parameters and assume default if not specified
        if ( sFileCreationModeTmp  == null
             || sFileCreationModeTmp .length() <= 0 ){
            sFileCreationModeTmp = "all" ;
        }
        if ( sOutputFormatTmp == null
             || sOutputFormatTmp.length() <= 0 ){
            sOutputFormatTmp = "pdf" ;
        }
        if ( sOutputFilePathTmp == null
             || sOutputFilePathTmp.length() <= 0 ){
            sOutputFilePathTmp = "pdf" ;
        }
        // $V104 End

        String sSimpleSpoolnameTmp = extractSimpleFilename( sActiveSpoolnameTmp );

        // Increment output document id
        lDocumentId++;

        //---- Build output filename ----//
        String sSimpleOutputFilenameTmp = sSimpleSpoolnameTmp;

        if ( sFileCreationModeTmp != null 
             && ( sFileCreationModeTmp.equalsIgnoreCase( "doc_index" )
                  || sFileCreationModeTmp.equalsIgnoreCase( "doc_cold" )
                 )
            ){
            sSimpleOutputFilenameTmp += "_" + sUniqueId + "." + lDocumentId;
        }

        //---- Process index details ----//
        if ( lstDocumentIndexPar != null ){
            String sIndexFilenameTmp = null;
            if ( sA2WIniPath != null ){
                // $V104 Begin
                //sIndexFilenameTmp = sA2WIniPath + "pdf/" + sSimpleOutputFilenameTmp + ".idx";
                sIndexFilenameTmp = sA2WIniPath + sOutputFilePathTmp + "/" + sSimpleOutputFilenameTmp + ".idx";
                // $V104 End
            }
            else {
                // $V104 Begin
                //sIndexFilenameTmp = "pdf/" + sSimpleOutputFilenameTmp + ".idx";
                sIndexFilenameTmp = sOutputFilePathTmp + "/" + sSimpleOutputFilenameTmp + ".idx";
                // $V104 End
            }

            try {
                // Create index file stream
                FileOutputStream fosIndexTmp = new FileOutputStream( sIndexFilenameTmp );

                ListIterator lstitrIndexTmp = lstDocumentIndexPar.listIterator( 0 );

                if ( lstitrIndexTmp != null ){
                    String sIndexEntryTmp = null;
                    A2WIndex a2wIndexTmp = null;

                    while ( lstitrIndexTmp.hasNext() ){
                        a2wIndexTmp = (A2WIndex)lstitrIndexTmp.next();

                        // Build index entry and write it on index file
                        if ( a2wIndexTmp != null ){
                            sIndexEntryTmp = a2wIndexTmp.getName() + "=" + a2wIndexTmp.getValue() + "\n";

                            fosIndexTmp.write( sIndexEntryTmp.getBytes() );
                        }
                    }
                }
            }
            catch ( Exception eFileErrorPar ){
                System.out.println( eFileErrorPar );
            }
        }

        //---- Process document buffer ----//
        try {
            String sOutputFilenameTmp = null;
            if ( sA2WIniPath != null ){
                // $V104 Begin
                //sOutputFilenameTmp = sA2WIniPath + "pdf/" + sSimpleOutputFilenameTmp + fetchExtension( sOutputFormatTmp );
                sOutputFilenameTmp = sA2WIniPath + sOutputFilePathTmp + "/" + sSimpleOutputFilenameTmp + fetchExtension( sOutputFormatTmp );
                // $V104 End 
                }
            else {
                // $V104 Begin
                //sOutputFilenameTmp = "pdf/" + sSimpleOutputFilenameTmp + fetchExtension( sOutputFormatTmp );
                sOutputFilenameTmp = sOutputFilePathTmp + "/" + sSimpleOutputFilenameTmp + fetchExtension( sOutputFormatTmp );
                // $V104 End
            }

            // Create output file stream
            FileOutputStream fosDocumentTmp = new FileOutputStream( sOutputFilenameTmp );

            // Write document to file
            baosOutputStreamPar.writeTo( fosDocumentTmp );
        }
        catch ( Exception eFileErrorPar ){
            System.out.println( eFileErrorPar );
        }

        return 0;
    }

    /**
     * Simple filename extractor
     *
     * <p>
     * @param sFilenamePar Filename
     * <br>
     * <br>
     * @return 
     * "Simple filename" as String
     * </p>
     */
    private String extractSimpleFilename( String sFilenamePar ){
        //---- Assert parameter ----//
        if ( sFilenamePar == null
             || sFilenamePar.length() <= 0 ){
            return null;
        }

        // Extract simple filename
        int iLastSlashIndexTmp = sFilenamePar.lastIndexOf( '\\' );
        if ( iLastSlashIndexTmp == -1 ){
            iLastSlashIndexTmp = sFilenamePar.lastIndexOf( '/' );
        }

        String sSimpleFilenameTmp = null;
        if ( iLastSlashIndexTmp != -1 ){
            sSimpleFilenameTmp = sFilenamePar.substring( iLastSlashIndexTmp, sFilenamePar.lastIndexOf( '.' ) );
        }
        else {
            sSimpleFilenameTmp = sFilenamePar;
        }

        return sSimpleFilenameTmp;
    }

    /**
     * Generates unique id
     *
     * <p>
     * @return 
     * Generated unique id
     * </p>
     */
    private String generateUniqueId(){
        int iMaxUniqueIdLenTmp = 5;

        char cRandomCharTmp = '\0';
        String sUniqueIdTmp = "";
        String sPossibleCharsTmp = "abcdefghijklmonpqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
        int iPossibleCharsLenTmp = sPossibleCharsTmp.length();

        //---- Generate unique id ----//
        while( sUniqueIdTmp.length() < iMaxUniqueIdLenTmp ){
            // Get a random character
            cRandomCharTmp = (char)sPossibleCharsTmp.charAt( (int)( java.lang.Math.random() * iPossibleCharsLenTmp ) );

            // Add above generated random char to unique string, only if it is not added already
            if ( sUniqueIdTmp.indexOf( cRandomCharTmp ) < 0 ){
                sUniqueIdTmp += cRandomCharTmp;
            }
        }

        return sUniqueIdTmp;
    }

    /**
     * Fetch filename extension for given output format
     *
     * <p>
     * @return 
     * Filename extension as string
     * </p>
     */
    private String fetchExtension( String sFormatPar ){
        if ( sFormatPar == null ){
            return "";
        }

        String sExtensionTmp = "";

        // PDF and PDF/A format
        // $V101 Begin
        //if ( sFormatPar.equalsIgnoreCase( "pdf" ) ){
        if ( sFormatPar.equalsIgnoreCase( "pdf" )
             || sFormatPar.equalsIgnoreCase( "pdfa" ) ){
        // $V101 End
            sExtensionTmp = ".pdf";
        }
        // TIFF format
        else if ( sFormatPar.equalsIgnoreCase( "tif" )
              || sFormatPar.equalsIgnoreCase( "tiff" ) ){
            sExtensionTmp = ".tif";
        }
        // ASCII format
        else if ( sFormatPar.equalsIgnoreCase( "ascii" ) ){
            sExtensionTmp = ".txt";
        }
        // $V102 Begin
        // AFP format
        else if ( sFormatPar.equalsIgnoreCase( "afp" ) ){
            sExtensionTmp = ".afp";
        }
        // JPEG format
        else if ( sFormatPar.equalsIgnoreCase( "jpg" )
              || sFormatPar.equalsIgnoreCase( "jpeg" ) ){
            sExtensionTmp = ".jpg";
        }
        // $V102 End
        // $V106 Begin
        // PNG format
        else if ( sFormatPar.equalsIgnoreCase( "png" ) ){
            sExtensionTmp = ".png";
       }
       // $V106 End

        return sExtensionTmp;
    }
}
