#-------------------------------------------------------------------------------
#  a2w::core::dm::Config.pm:
#  Data Mining Framework config
#
#  Author  : AFP2web Team, Maas Holding GmbH
#
#  V101   2021-06-01    AFP-1046: Extended with output config to have block specific config, content to be embedded etc in output
#
#  V100   2014-05-08    Initial Release
#
#-------------------------------------------------------------------------------
package a2w::core::dm::Config;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;
use a2w::core::dm::Constants;

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    my $this = {
          'BlockCount' => 0      # Blocks count
        , 'BlockIDs'   => undef  # Array of Block IDs
        , 'Blocks'     => undef  # Blocks collection
        , 'ChainedIDs' => undef  # Hash reference, IDs of blocks that are in chain with a parent block
    };

    bless( $this, $class );

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}
    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------
sub setBlocks{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setBlocks()" );
    #}

    $this->{ 'Blocks' } = shift;

    #---- Evaluate and set blocks info
    my @arrBlocksTmp = sort keys( %{ $this->{ 'Blocks' } } );
    $this->{ 'BlockCount' } = @arrBlocksTmp;
    $this->{ 'BlockIDs' }   = \@arrBlocksTmp;
}

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------
sub getBlocks{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getBlocks()" );
    #}

    return $this->{ 'Blocks' };
}

sub getBlockCount{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getBlockCount()" );
    #}

    return $this->{ 'BlockCount' };
}

sub getBlockIDs{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getBlockIDs()" );
    #}

    return $this->{ 'BlockIDs' };
}

sub getChainedIDs{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getChainedIDs()" );
    #}

    return $this->{ 'ChainedIDs' };
}

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
sub addBlock{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "addBlock()" );
    #}

    #---- Parameter
    #
    # 1. Block
    #
    my $blkToAddPar = shift;

    $this->{ 'BlockCount' } += 1;
    $this->{ 'Blocks' }{ $blkToAddPar->getId() } = $blkToAddPar;
    my @arrBlocksTmp = sort keys( %{ $this->{ 'Blocks' } } );
    $this->{ 'BlockIDs' }  = \@arrBlocksTmp;
}

sub getBlock{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getBlock()" );
    #}

    #---- Parameter
    #
    # 1. Block Id
    #
    my $sBlkIdPar = shift;

    return $this->{ 'Blocks' }{ $sBlkIdPar };
}

sub resolveNextChain{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "resolveNextChain()" );
    }

    #---- Iterate through all blocks and resolve next links if exists ----#
    my $blkNexTmp    = undef;
    my $blkCurrentmp = undef;
    my $sNextIdTmp   = "";
    my $sCDTypeTmp   = "";
    my $iRetTmp      = 0;

    my @arrBlocksTmp = @{ $this->{ 'BlockIDs' } };
    foreach my $sBlkIdTmp ( @arrBlocksTmp ){
        #---- Assert whether block is chained
        $blkCurrentmp = $this->{ 'Blocks' }{ $sBlkIdTmp };
        if ( $blkCurrentmp->isChained() == $TRUE ){
            #---- Link the next block
            $sNextIdTmp = $blkCurrentmp->getNext();

            #---- Collect the chained id
            $this->{ 'ChainedIDs' }{ $sNextIdTmp } = $TRUE;

            #---- Get next block reference
            $blkNexTmp = $this->{ 'Blocks' }{ $sNextIdTmp };
            if ( $blkNexTmp != undef ){
                $blkCurrentmp->setNextRef( $blkNexTmp );

                if ( $bLog == $TRUE ){
                    $theLogger->logMessage( "Next block (" . $sNextIdTmp . ") found and chained with block (" . $sBlkIdTmp . ")" );
                }

                #---- Assert whether current and next blocks have same content types
                #     For example, a table can continued in two blocks must have same
				#     content definitions otherwise formatting might go wrong
                $sCDTypeTmp = $blkCurrentmp->getContentDefType();
                if ( $sCDTypeTmp ne $blkNexTmp->getContentDefType() ){
                    $iRetTmp = -1;
                    if ( $bLog == $TRUE ){
                        $theLogger->logMessage( "  Error! Mismatching content definition type (" . $sCDTypeTmp . "!=" . $blkNexTmp->getContentDefType() . ") between current block ("  . $sBlkIdTmp . ") and next block (" . $sNextIdTmp . ")" );
                    }
                }
            }
            else {
                $iRetTmp = -2;
                if ( $bLog == $TRUE ){
                    $theLogger->logMessage( "  Error! No next block (" . $sNextIdTmp . ") defined for block (" . $sBlkIdTmp . ")" );
                }
            }
        }
    }
    return $iRetTmp;
}

sub createDefaultBlock{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "createDefaultBlock()" );
    #}

    #---- Parameter
    #
    # 1. Block Id
    #
    my $sBlkIdPar = shift;
    if ( $sBlkIdPar eq "" ){
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Empty/Invalid block id to create default block" );
        }
        return undef;
    }
    my $blkDefTmp = new a2w::core::dm::Block();

    #---- Fill in block
    $blkDefTmp->setId( $sBlkIdPar );

    #---- Create default content definition of type paragraph
    $blkDefTmp->createAndSetContentDef( {   'Type' => 'Paragraph'
                                          , 'Definition' => {
                                                  'Group' => $FALSE
                                                , 'GroupBy' => "line"
                                                , 'Separator' => ""
                                            }
                                        }
                                      );

    return $blkDefTmp;
}

# V101 Begin
sub createSkipBlock{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "createSkipBlock()" );
    #}

    #---- Parameter
    #
    # 1. Block Id
    #
    my $sBlkIdPar = shift;
    if ( $sBlkIdPar eq "" ){
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Empty/Invalid block id to create skip block" );
        }
        return undef;
    }
    my $blkSkipTmp = new a2w::core::dm::Block();

    #---- Fill in block
    $blkSkipTmp->setId( $sBlkIdPar );

    #---- Create content definition of type skip
    $blkSkipTmp->createAndSetContentDef( { 'Type' => 'Skip' } );

    return $blkSkipTmp;
}
# V101 End

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
