#-------------------------------------------------------------------------------
#  a2w::core::dm::Constraint.pm:
#  Data Mining Framework constraint
#
#  Author  : AFP2web Team, Maas Holding GmbH
#
#  $V100   2014-05-08    Initial Release
#
#  $V101   2018-01-19    Extended to process container objects
#
#  $V102   2018-01-24    a. Fixed minor bug in splitting condition value
#                        b. Extended with 'any' object type to detect begin/end of page
#
#  $V103   2020-04-02    Fixed minor bug in checking object type
#
#  $V104   2020-07-01    AFP-951: Extended to process/write annotation objects
#
#-------------------------------------------------------------------------------
package a2w::core::dm::Constraint;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;
use a2w::core::dm::Constants;

use a2w::core::ext::Annotation; # V104 Change
use a2w::core::ext::Image;
use a2w::core::ext::Container; # $V101 Change
use a2w::core::ext::Line;
use a2w::core::ext::Text;
use a2w::core::ext::Vector;

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    my $this = {
        # Type of object to be checked by condition
          'ObjectType'     => 0
        # Conditions count
        , 'ConditionCount' => 0
        # Array of Conditions
        , 'Conditions'     => []
        # Array of Operator callbacks
        # Signature: bool opcb( Value1, Value2[, Constraint ] )
        , 'OpCallbacks'    => []
        # Result
        , 'Result'         => undef
    };

    bless( $this, $class );

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new" );
    #}

    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------
sub setObjectType{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setObjectType()" );
    #}

    $this->{ 'ObjectType' } = shift;
}

sub setConditions{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setConditions()" );
    #}

    #---- Get parameter
    #
    # 1. Conditions (as hash or array reference)
    #
    my $ConditionsTmp = shift;

    my $sRefTmp = lc( ref( $ConditionsTmp ) );
    if ( $sRefTmp eq "hash" ){
        $this->setConditionsHashRef( $ConditionsTmp );
    }
    elsif ( $sRefTmp eq "array" ){
        $this->setConditionsArrayRef( $ConditionsTmp );
    }
    else {
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Warning! Ignored invalid conditions value (" . $ConditionsTmp . ")" );
        }
    }
}

sub setConditionsHashRef{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setConditionsHashRef()" );
    #}

    #---- Get parameter
    #
    # 1. Conditions (hash reference)
    #
    $this->{ 'Conditions' } = shift;

    #---- Evaluate and set conditions info
    my @arrConditionsTmp = @{ $this->{ 'Conditions' } };
    $this->{ 'ConditionCount' } = @arrConditionsTmp;
}

sub setConditionsArrayRef{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setConditionsArrayRef()" );
    #}

    #---- Get parameter
    #
    # 1. Conditions (array reference)
    #
    my $ConditionsTmp = shift;
    my @arrCondsTmp = @{ $ConditionsTmp };

    #---- Iterate and add conditions ----#
    foreach my $sCondTmp ( @arrCondsTmp ){
        $this->addCondition( $sCondTmp );
    }
}

sub setResult{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setResult()" );
    #}

    $this->{ 'Result' } = shift;
}

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------
sub getObjectType{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getObjectType()" );
    #}

    return $this->{ 'ObjectType' };
}

sub getConditions{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getConditions()" );
    #}

    return $this->{ 'Conditions' };
}

sub getResult{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getResult()" );
    #}

    return $this->{ 'Result' };
}

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
sub addCondition{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "addCondition()" );
    #}

    #---- Get parameter
    #
    # 1. Condition string
    #
    my $sCondPar = shift;
    if ( $sCondPar eq "" ){
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Warning! Empty condition" );
        }
        return;
    }

    #---- Fetch attribute name, operator and limit
    my $sAttrNameTmp = "";
    my $sOperatorTmp = "";
    my $sValueTmp    = "";
    # $V102 Begin
    #if ( $sCondPar =~ /^ATTR_(.*)\s+(.{2})\s+(.*)$/ ){
    if ( $sCondPar =~ /^ATTR_(.*)\s+(eq|eQ|Eq|EQ|ne|nE|Ne|NE|lt|lT|Lt|LT|gt|gT|Gt|GT|le|lE|Le|LE|ge|gE|Ge|GE)\s+(.*)$/ ){
    # $V102 End
        $sAttrNameTmp = $1;
        $sOperatorTmp = $2;
        $sValueTmp    = $3;
    }
    if (    $sAttrNameTmp eq ""
         || $sOperatorTmp eq ""
         || $sValueTmp eq ""
       ){
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Warning! Empty condition or missing name/operator/value" );
        }
        return;
    }
    #else {
    #    if ( $bLog == $TRUE ){
    #        $theLogger->logMessage( "Condition: Attribute=>" . $sAttrNameTmp . "< Operator=>" . $sOperatorTmp . "< Value=>" . $sValueTmp . "<" );
    #    }
    #}

    #---- Determine operator to assert condition ----#
    my $sOpTmp = lc( $sOperatorTmp );
    # Numeric value
    if ( $sValueTmp =~ m/^[-+]?\d+(?:\.\d*)?(?:[eE][-+]?\d+(?:\.\d*)?)?$/ ){
        if ( $sOpTmp eq "eq" ){
            $this->{ 'OpCallbacks' }[ $this->{ 'ConditionCount' } ] = sub {
                return ( $_[0] == $_[1] ) ? $TRUE : $FALSE;
            };
        }
        elsif ( $sOpTmp eq "ne" ){
            $this->{ 'OpCallbacks' }[ $this->{ 'ConditionCount' } ] = sub {
                return ( $_[0] != $_[1] ) ? $TRUE : $FALSE;
            };
        }
        elsif ( $sOpTmp eq "lt" ){
            $this->{ 'OpCallbacks' }[ $this->{ 'ConditionCount' } ] = sub {
                return ( $_[0] < $_[1] ) ? $TRUE : $FALSE;
            };
        }
        elsif ( $sOpTmp eq "gt" ){
            $this->{ 'OpCallbacks' }[ $this->{ 'ConditionCount' } ] = sub {
                return ( $_[0] > $_[1] ) ? $TRUE : $FALSE;
            };
        }
        elsif ( $sOpTmp eq "le" ){
            $this->{ 'OpCallbacks' }[ $this->{ 'ConditionCount' } ] = sub {
                return ( $_[0] <= $_[1] ) ? $TRUE : $FALSE;
            };
        }
        elsif ( $sOpTmp eq "ge" ){
            $this->{ 'OpCallbacks' }[ $this->{ 'ConditionCount' } ] = sub {
                return ( $_[0] >= $_[1] ) ? $TRUE : $FALSE;
            };
        }
    }
    # Binary operator (for regular expression)
    elsif ( $sValueTmp =~ m/^qr\/(.*)\/$/ ){
        $sValueTmp = $1;
        if ( $sOpTmp eq "eq" ){
            $this->{ 'OpCallbacks' }[ $this->{ 'ConditionCount' } ] = sub {
                my $reTmp = qr/$_[1]/;
                my $bRetTmp = $FALSE;
                if (    @_ > 2
                     && $_[ 2 ] != undef
                   ){
                    if ( $_[0] =~ /$reTmp/ ){
                        $_[ 2 ]->setResult( $1 );
                        $bRetTmp = $TRUE;
                    }
                }
                else {
                    if ( $_[0] =~ /$reTmp/ ){
                        $bRetTmp = $TRUE;
                    }
                }
                return $bRetTmp;
            };
        }
        elsif ( $sOpTmp eq "ne" ){
            $this->{ 'OpCallbacks' }[ $this->{ 'ConditionCount' } ] = sub {
                my $reTmp = qr/$_[1]/;
                my $bRetTmp = $FALSE;
                if (    @_ > 2
                     && $_[ 2 ] != undef
                   ){
                    if ( $_[0] !~ /$reTmp/ ){
                        $_[ 2 ]->setResult( $1 );
                        $bRetTmp = $TRUE;
                    }
                }
                else {
                    if ( $_[0] !~ /$reTmp/ ){
                        $bRetTmp = $TRUE;
                    }
                }
                return $bRetTmp;
            };
        }
    }
    # String value
    else {
        if ( $sOpTmp eq "eq" ){
            $this->{ 'OpCallbacks' }[ $this->{ 'ConditionCount' } ] = sub {
                return ( $_[0] eq $_[1] ) ? $TRUE : $FALSE;
            };
        }
        elsif ( $sOpTmp eq "ne" ){
            $this->{ 'OpCallbacks' }[ $this->{ 'ConditionCount' } ] = sub {
                return ( $_[0] ne $_[1] ) ? $TRUE : $FALSE;
            };
        }
        elsif ( $sOpTmp eq "lt" ){
            $this->{ 'OpCallbacks' }[ $this->{ 'ConditionCount' } ] = sub {
                return ( $_[0] lt $_[1] ) ? $TRUE : $FALSE;
            };
        }
        elsif ( $sOpTmp eq "gt" ){
            $this->{ 'OpCallbacks' }[ $this->{ 'ConditionCount' } ] = sub {
                return ( $_[0] gt $_[1] ) ? $TRUE : $FALSE;
            };
        }
        elsif ( $sOpTmp eq "le" ){
            $this->{ 'OpCallbacks' }[ $this->{ 'ConditionCount' } ] = sub {
                return ( $_[0] le $_[1] ) ? $TRUE : $FALSE;
            };
        }
        elsif ( $sOpTmp eq "ge" ){
            $this->{ 'OpCallbacks' }[ $this->{ 'ConditionCount' } ] = sub {
                return ( $_[0] ge $_[1] ) ? $TRUE : $FALSE;
            };
        }
    }

    #---- Add to conditions list
    $this->{ 'Conditions' }[ $this->{ 'ConditionCount' } ] = {
          'ATTR_NAME' => $sAttrNameTmp
        , 'OPERATOR'  => $sOpTmp
        , 'VALUE'     => $sValueTmp
    };

    #---- Increment condition count
    $this->{ 'ConditionCount' }++;
}

sub assert{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Object
    # 2. POM object wrapper (optional)
    #
    my $a2wObjectPar = shift;
    my $pomObjectPar = undef;
    if ( @_ > 0 ){
        $pomObjectPar = shift;
    }

    #---- Assert object type
    if ( $a2wObjectPar == undef ){ return $FALSE; } # V103 Change
    if (    lc( $this->{ 'ObjectType' } ) ne "any" # $V102 Change
         && lc( $this->{ 'ObjectType' } ) ne $a2wObjectPar->_getType()
       ){
        #if ( $bLog == $TRUE ){
        #    $theLogger->logMessage( "  Condition failed due to object type mismatch" );
        #}
        return $FALSE;
    }

    #---- Get conditions and operators
    my $iCondCntTmp = $this->{ 'ConditionCount' };
    my @arrCondsTmp = @{ $this->{ 'Conditions' } };
    my @arrOpsTmp   = @{ $this->{ 'OpCallbacks' } };

    #---- Iterate through conditions and assert against given object ----#
    my $attrValTmp = '';
    my $condTmp    = undef;
    my $funcOpTmp  = undef;
    my $bRetTmp    = $TRUE;
    for (   my $i = 0
          ; (    $bRetTmp == $TRUE
              && $i < $iCondCntTmp
            )
          ; $i++
        ){
        $condTmp = @arrCondsTmp[ $i ];

        #---- Get object's attribute value
        $attrValTmp = $a2wObjectPar->_getAttribute( $condTmp->{ 'ATTR_NAME' }, $pomObjectPar );

        #---- Compare attribute value
        $funcOpTmp = @arrOpsTmp[ $i ];
        $bRetTmp &&= $funcOpTmp->( $attrValTmp, $condTmp->{ 'VALUE' }, $this );

        if ( $bLog == $TRUE ){
            $theLogger->logMessage(   "  Condition ("
                                    . $condTmp->{ 'ATTR_NAME' } . "{" . $attrValTmp . "}"
                                    . " " . uc( $condTmp->{ 'OPERATOR' } )
                                    . " " . $condTmp->{ 'VALUE' }
                                    . "):"
                                    . ( ( $bRetTmp == $TRUE ) ? " Ok" : " failed" )
                                  );
        }
    }
    return $bRetTmp;
}

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
