#-------------------------------------------------------------------------------
#  a2w/core/dm/Constraints.pm
#
#  Collection of constraints (base for following modules for common usage of constraints)
#  - Anchor
#  - Index
#  - FindAndReplace
#  - FindAndPrepend
#  - FindAndAppend
#
#  Author   : Panneer, AFP2web Team
#  Date     : 2021-09-21
#  Version  : 1.0.3
#
#  V100   2014-02-14    Initial Release
#
#  V101   2018-01-26    Redesigned to have multiple constraints to have OR conditionals
#
#  V102   2018-04-27    Fixed minor bug in countinig conditions on constraints
#
#  V103   2021-09-21    AFP-1068: Extended with FindAndPrepend and FindAndAppend content processing objects
#
#-------------------------------------------------------------------------------
package a2w::core::dm::Constraints;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;

use a2w::core::dm::Constants;
use a2w::core::dm::Constraint;

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;
	
    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    my $this = {
        # Constraints collection (of type hash)
        # Each constraints will have one or more constraint (conditions and result of conditions)
          'Constraints'       => undef
        # Recently matched constraints
        , 'RecentConstraints' => undef
        # Mentions to which output formats these constraints are applicable
        # Default is for all output formats currently generated by SF DMF
        , 'ApplicableOutputs' => ''       # V103 Change
    };

    bless( $this, $class );

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}

    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------
# V103 Begin
#===================================
# setApplicableOutputs
#
# Set to which output formats these constraints are applicable
#===================================
sub setApplicableOutputs{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setApplicableOutputs()" );
    #}

    #---- Get parameter
    #
    # 1. List of output formats separated by comma (string)
    my $sApplicableOutputsPar = shift;

    $this->{ 'ApplicableOutputs' } = $sApplicableOutputsPar;
}
# V103 End

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------
sub isSkipped{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "isSkipped()" );
    #}

    # Returns the skip flag of given constraints or by default returns the recently matched constraints skip flag #
    #
    #---- Get parameter
    #
    # 1. Constraints Id
    my $sCIdPar = shift;

    #---- Get skipped flag
    my $bSkippedTmp = $this->{ 'RecentConstraints' }{ 'Skip' };
    if ( $CIdPar ne "" && $this->{ 'Constraints' }{ $CIdPar }{ 'Skip' } == $TRUE ){ $bSkippedTmp = $TRUE; }

    return $bSkippedTmp;
}

sub getResult{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getResult()" );
    #}

    # Returns the result of given constraints or by default returns the recently matched constraints result #
    #
    #---- Get parameter
    #
    # 1. Constraints Id
    my $sCIdPar = shift;

    #---- Get result
    my $sResultTmp = $this->{ 'RecentConstraints' }{ 'Result' };
    if ( $CIdPar ne "" && $this->{ 'Constraints' }{ $CIdPar }{ 'Result' } ne "" ){ $sResultTmp = $this->{ 'Constraints' }{ $CIdPar }{ 'Result' }; }

    return $sResultTmp;
}

# V103 Begin
#===================================
# getApplicableOutputs
#
# Get to which output formats these constraints are applicable
#===================================
sub getApplicableOutputs{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getApplicableOutputs()" );
    #}

    return $this->{ 'ApplicableOutputs' };
}
# V103 End

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
sub resetResults{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "resetResults()" );
    #}

    #---- Get list of constraints (sorted based on sequence id)
    my @arrConstraintsTmp = keys ( %{ $this->{ 'Constraints' } } );
    foreach my $cid ( @arrConstraintsTmp ){
        if ( $this->{ 'Constraints' }{ $cid } == undef ){ next; }
        $this->{ 'Constraints' }{ $cid }->{ 'Result' } = '';
    }
}

sub createAndAddConstraints{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "createAndAddConstraints()" );
    }

    #---- Parameter
    #
    # 1. Constraints Id
    # 2. Constraints config hash reference
    #
    my $sConstraintsIdPar  = shift;
    my $hrefConstraintsPar = shift;
    my @arrConstraintTmp = sort keys( %{ $hrefConstraintsPar } );

    #---- Store constraints on collection
    @arrConstraintTmp = grep { lc( $_ ) =~ /cond*/ } @arrConstraintTmp; # V102 Change
    $this->{ 'Constraints' }{ $sConstraintsIdPar } = {
        # Constraints config reference
          'ConfigRef' => $hrefConstraintsPar
        # Conditions array
        , 'Conditions' => undef
        # Sequence id
        , 'SequenceId' => ( @arrConstraintTmp + 0 ) # Added with zero to force array context as scalar # V102 Change
        # Flag indicating whether constraints matching object is part of block content or not, default is part of block content
        , 'Skip' => $FALSE
    };

    #---- Create constraint
    my $conCurrentTmp = undef;
    foreach my $sIdTmp ( @arrConstraintTmp ){
        if ( $bLog == $TRUE ){
            $theLogger->logMessage( "Constraint: Id=>" . $sIdTmp . "< ObjType=>" . $hrefConstraintsPar->{ $sIdTmp }{ 'OBJ_TYPE' } . "< Cond=>" . $hrefConstraintsPar->{ $sIdTmp }{ 'CONDITION' } . "<" );
        }

        #---- Create constraint and fill in details ---#
        if (    $conCurrentTmp == undef
             || $conCurrentTmp->getObjectType() != $hrefConstraintsPar->{ $sIdTmp }{ 'OBJ_TYPE' }
           ){
            $conCurrentTmp = new a2w::core::dm::Constraint();
            my @arrCondsTmp = @{ $this->{ 'Constraints' }{ $sConstraintsIdPar }{ 'Conditions' } };
            $this->{ 'Constraints' }{ $sConstraintsIdPar }{ 'Conditions' }[ @arrCondsTmp ] = $conCurrentTmp;

            #---- Set object type
            $conCurrentTmp->setObjectType( $hrefConstraintsPar->{ $sIdTmp }{ 'OBJ_TYPE' } );

            #---- Check and Set skip
            if ( defined $hrefConstraintsPar->{ 'Skip' } ){ $this->{ 'Constraints' }{ $sConstraintsIdPar }{ 'Skip' } = $hrefConstraintsPar->{ 'Skip' }; }
        }

        #---- Add condition
        $conCurrentTmp->addCondition( $hrefConstraintsPar->{ $sIdTmp }{ 'CONDITION' } );
    }
}

sub doesConstraintsMatch{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "doesConstraintsMatch()" );
    }

    #---- Parameter
    #
    # 1. Object
    # 2. POM object wrapper (optional)
    #
    my $a2wObjectPar = shift;
    my $pomObjectPar = undef;
    if ( @_ > 0 ){
        $pomObjectPar = shift;
    }

    #---- Get list of constraints (sorted based on sequence id)
    my @arrConstraintsTmp = sort { $this->{ 'Constraints' }{ $a }{ 'SequenceId' } <=> $this->{ 'Constraints' }{ $b }{ 'SequenceId' } } keys ( %{ $this->{ 'Constraints' } } );
    if ( @arrConstraintsTmp <= 0 ){ return $FALSE; }
    
    #---- Iterate through constraints and assert them against object ----#
    my $c = undef;
    my @arrCndsTmp = undef;
    my $bMatchTmp = $FALSE;
    foreach my $cid ( @arrConstraintsTmp ){
        #---- If one of constraints matched already, break the loop
        if ( $bMatchTmp == $TRUE ){ last; }

        #---- Get constraints
        $c = $this->{ 'Constraints' }{ $cid };
        if ( $c == undef ){ next; }

        #---- Assert conditions of current contraints
        @arrCndsTmp = @{ $c->{ 'Conditions' } };
        if ( @arrCndsTmp <= 0 ){ next; }

        my $bCndMatchTmp = $TRUE;
        for (   my $i = 0
              ; (    $bCndMatchTmp == $TRUE
                  && $i < @arrCndsTmp
                )
              ; $i++
            ){
            #---- Assert for constraint
            $bCndMatchTmp &&= $arrCndsTmp[ $i ]->assert( $a2wObjectPar, $pomObjectPar );
            if (    $bCndMatchTmp == $TRUE
                 && $c->{ 'Result' } == undef
               ){
                $c->{ 'Result' } = $arrCndsTmp[ $i ]->getResult();
            }
        }
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "$cid " . ( ( $bCndMatchTmp ) ? ">matched<" : ">not matched<" ) ); }

        #---- Update matched flag with current constraints result
        $this->{ 'RecentConstraints' } = $c;
        $bMatchTmp ||= $bCndMatchTmp;
    }

    return $bMatchTmp;
}

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
