#-------------------------------------------------------------------------------
#  a2w/core/dm/ContentParser.pm
#
#  Perl module to parse page content and to build a database out of page content
#
#  Author   : Panneer, AFP2web Team
#
#  $V100   2014-02-14    Initial Release
#
#  $V101   2015-08-27    Fixed minor bug in evaluating output file name when page
#                        output is turned on
#
#  $V102   2017-10-30    a. Fixed minor bug in evaluating absolute position of page
#                           inline images when page and image resolution differ.
#                        b. Extended to fill in page reference on Database to handle
#                           empty cells for ADA output
#                        c. Extended to process page segment resources also
#
#  $V103   2018-01-10    a. Extended with container objects parsing (AFP-456, AFP-623)
#                        b. Extended to handle included position of objects within resources
#
#  $V104   2018-05-18    a. Removed evaluating absolute position of page inline images
#                           for Script Visitor process (since AFP2web core already pass
#                           absolute position of image)
#
#  $V105   2018-08-09    - AFP-735: Extended to handle tagging for reusable objects (with different tag info
#                          at each presentation of same object)
#                        - AFP-734: Extended to write tagging info at document level instead of page level
#
#  $V106   2018-09-11    - OXS-8490: Extended with parser anchors to preprocess objects and to invoke callback
#                          when anchor matching object is found
#
#  $V107   2018-10-22    - AFP-743: Extended to parse page included resources recursively
#
#  $V108   2020-03-05    AFP-929: Fixed minor bug in processing object color value when adding object
#
#  $V109   2020-09-29    AFP-974: Extended to handle Annotation object
#
#  $V110   2021-04-10    OXS-12071: Fixed minor bug in matching object against parser rules
#
#-------------------------------------------------------------------------------
package a2w::core::dm::ContentParser;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;

use a2w::core::dm::Constants;
use a2w::core::dm::Constraint;
use a2w::core::dm::Database;

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    my $this = {
          'POM'             => undef    # Current page database reference
        , 'ObjectCount'     => 0        # Object count # V106 Change
        , 'PageLastObject'  => undef    # Page last object # V106 Change
        , 'Rules'           => undef    # Array reference with parser rules # V106 Change
    };

    bless( $this, $class );

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}

    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------
# V106 Begin
sub setRules{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setRules()" );
    #}

    $this->{ 'Rules' } = shift; # Array reference with parser rules
}
# V106 End

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------
# V106 Begin
sub getRules{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getRules()" );
    #}

    return $this->{ 'Rules' }; # Array reference with parser rules
}
# V106 End

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# Parse (generic interface)
# Returns a database containing the page content
#-----------------------------------------------------------------------
sub parse{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "parse" );
    }

    require a2w::Page;    # Include page module

    #---- Get parameter of parse ( Par: Page instance[, Flag to include page resource contents] )
    my $a2wPagePar = shift;
    my $bIncludePageResPar = $FALSE;

    #---- Get include page resource contents flag, if exist
    if ( @_ ){
        $bIncludePageResPar = shift;
    }

    #---- Parse page content and building database
    my $iPageIdTmp = $a2wPagePar->getParseId();

    #---- Create database
    my $dbPageTmp = new a2w::core::dm::Database( $iPageIdTmp );

    #---- Fill in page details
    $dbPageTmp->setPageRes( $a2wPagePar->getResolution() );
    $dbPageTmp->setPageWidth( $a2wPagePar->getWidth() );
    $dbPageTmp->setPageHeight( $a2wPagePar->getHeight() );
    $dbPageTmp->setPageBackground( $a2wPagePar->getOutputFilename() );
    $dbPageTmp->setPageReference( $a2wPagePar ); # $V102 Change
    $this->{ 'ObjectCount' } = 0; # V106 Change
    $this->{ 'PageLastObject' } = undef; # V106 Change

    # V107 Begin
    #---- Prepare parent context
    # syntax:
    # Array = (
    #  {
    #      'STATE' => <parent object>
    #    , 'INC_X' => <included x position>
    #    , 'INC_Y' => <included y position>
    #    , 'RES'   => <parent resolution>
    #    , 'TYPE'  => <parent type>
    #  },
    #  {
    #      'STATE' => <parent object>
    #    , 'INC_X' => <included x position>
    #    , 'INC_Y' => <included y position>
    #    , 'RES'   => <parent resolution>
    #    , 'TYPE'  => <parent type>
    #  }
    # )
    #
    # Use case 1: Page define texts
    # Array of parent = (
    #  {
    #     'STATE' => <page>
    #   , 'INC_X' => 0
    #   , 'INC_Y' => 0
    #   , 'RES'   => <page resolution>
    #   , 'TYPE'  => <page>
    #  },
    #  {
    #     'STATE' => <text>
    #   , 'INC_X' => <text x position>
    #   , 'INC_Y' => <text y position>
    #   , 'RES'   => <text resolution>
    #   , 'TYPE'  => <text>
    #  }
    # )
    #
    # Use case 2: Page includes overlay and overlay define texts
    # Array of parent = (
    #  {
    #      'STATE' => <page>
    #    , 'INC_X' => <overlay included x position>
    #    , 'INC_Y' => <overlay included y position>
    #    , 'RES'   => <page resolution>
    #    , 'TYPE'  => <page>
    #  },
    #  {
    #      'STATE' => <overlay>
    #    , 'INC_X' => 0
    #    , 'INC_Y' => 0
    #    , 'RES'   => <overlay resolution>
    #    , 'TYPE'  => <overlay>
    #  },
    #  {
    #      'STATE' => <text>
    #    , 'INC_X' => <text x position>
    #    , 'INC_Y' => <text y position>
    #    , 'RES'   => <text resolution>
    #    , 'TYPE'  => <text>
    #  }
    # )
    #
    # Use case 3: Page includes overlay, overlay includes page segment and page segment define image
    # Array of parent = (
    #  {
    #      'STATE' => <page>
    #    , 'INC_X' => <overlay included x position>
    #    , 'INC_Y' => <overlay included y position>
    #    , 'RES'   => <page resolution>
    #    , 'TYPE'  => <page>
    #  },
    #  {
    #      'STATE' => <overlay>
    #    , 'INC_X' => <page segment included x position>
    #    , 'INC_Y' => <page segment included y position>
    #    , 'RES'   => <overlay resolution>
    #    , 'TYPE'  => <overlay>
    #  },
    #  {
    #      'STATE' => <page segment>
    #    , 'INC_X' => 0
    #    , 'INC_Y' => 0
    #    , 'RES'   => 0
    #    , 'TYPE'  => <page segment>
    #  },
    #  {
    #      'STATE' => <image>
    #    , 'INC_X' => <image x position>
    #    , 'INC_Y' => <image y position>
    #    , 'RES'   => <image resolution>
    #    , 'TYPE'  => <image>
    #  }
    # )
    #
    my @arrParentContextTmp = (
        { 'STATE' => $a2wPagePar, 'INC_X' => 0, 'INC_Y' => 0, 'RES' => $a2wPagePar->getResolution(), 'TYPE' => $a2w::core::dm::Constants::PARENT_TYPE_PAGE }
	);

    #---- Fill Database with page content
    #
    #---- Add Text Objects
    $this->_addTextObjs( $a2wPagePar, $dbPageTmp, \@arrParentContextTmp );

    #---- Add Line Objects
    $this->_addLineObjs( $a2wPagePar, $dbPageTmp, \@arrParentContextTmp );

    #---- Add Vector Objects
    $this->_addVectorObjs( $a2wPagePar, $dbPageTmp, \@arrParentContextTmp );

    #---- Add Image Objects
    $this->_addImageObjs( $a2wPagePar, $dbPageTmp, \@arrParentContextTmp );

    #---- Add Container Objects
    $this->_addContainerObjs( $a2wPagePar, $dbPageTmp, \@arrParentContextTmp ); # $V103 Change

    #---- Add Overlays
    if ( $bIncludePageResPar ){ $this->_addOverlays( $a2wPagePar, $dbPageTmp, \@arrParentContextTmp ); }

    # $V102 Begin
    #---- Add Page Segments
    if ( $bIncludePageResPar ){ $this->_addPageSegments( $a2wPagePar, $dbPageTmp, \@arrParentContextTmp ); }
    # $V102 End
    # V107 End

    # V106 Begin
    #---- Process last object of page
    my $hrefPOMObjTmp = { $a2w::core::dm::Constants::AT_PAGELAST => $TRUE };
    my $hrefObjContextTmp = $this->_createObjectContext( $this->{ 'PageLastObject' } );
    $this->_preProcessAndAddObject( $a2wPagePar, $dbPageTmp, $hrefObjContextTmp, $hrefPOMObjTmp );
    # V106 End

    return $dbPageTmp;
}

#-----------------------------------------------------------------------
# Initialize page
#
# Returns page database (POM) in case of success, undef in case of error
#-----------------------------------------------------------------------
sub initializePage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "initializePage" );
    }

    require a2w::Page;    # Include page module

    #---- Get parameter ( Par: Page instance )
    my $a2wPagePar = shift;

    #---- Parse page content and building database
    my $iPageIdTmp = $a2wPagePar->getParseId();

    #---- Create database
    $this->{ 'POM' } = new a2w::core::dm::Database( $iPageIdTmp );
    my $dbPageTmp = $this->{ 'POM' };

    #---- Fill in page details
    $dbPageTmp->setPageRes( $a2wPagePar->getResolution() );
    $dbPageTmp->setPageWidth( $a2wPagePar->getWidth() );
    $dbPageTmp->setPageHeight( $a2wPagePar->getHeight() );
    $dbPageTmp->setPageReference( $a2wPagePar ); # $V102 Change
    $this->{ 'ObjectCount' } = 0; # V106 Change
    $this->{ 'PageLastObject' } = undef; # V106 Change

    # $V101 Begin
    #return 0;
    return $dbPageTmp;
    # $V101 End
}

#-----------------------------------------------------------------------
# Add object
#
# Adds given object to active page database (POM)
#
# Returns 0 in case of success, <0 in case of error
#-----------------------------------------------------------------------
sub addObject{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Object
    #
    my $a2wObjPar = shift;

    # V106 Begin
    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "addObject" ); }

    my $dbPageTmp   = $this->{ 'POM' };
    my $a2wPageTmp  = $dbPageTmp->getPageReference();

    #---- Create object context
    my $hrefObjContextTmp = $this->_createObjectContext( $a2wObjPar );

    #---- Preprocess and add object to database
    $this->_preProcessAndAddObject( $a2wPageTmp, $dbPageTmp, $hrefObjContextTmp );
    # V106 End

    return 0;
}

#-----------------------------------------------------------------------
# Finalize page
#
# Returns page database (POM) in case of success, undef in case of error
#-----------------------------------------------------------------------
sub finalizePage{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "finalizePage" );
    }

    return $this->{ 'POM' };
}

#-----------------------------------------------------------------------
#
# Add the Text objects to the database
#
#-----------------------------------------------------------------------
sub _addTextObjs{
    require a2w::Text;    # Include text module

    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addTextObjs" );
    }

    #---- Get parameter of _addTextObjs ( Par: Page/Overlay instance, Database instance, Position Adjustment )
    my $a2wContentObjectPar  = shift;
    my $dbContentPar         = shift;
    # V107 Begin
    # Parent context (of type array of hash references, refer "parse" function for more details)
    #
    my $arefParentContextPar = shift;

    #---- Fetch parent context value
    my @arrPtCtxtTmp = @{ $arefParentContextPar };
    my @arrPtCtxt1Tmp = ();
    # V107 End

    my $a2wPageTmp = $dbContentPar->getPageReference(); # V106 Change

    #---- Fill with text contents
    #---- Fetch first text
    my $a2wCurTextTmp = $a2wContentObjectPar->getFirstText();

    #---- Loop thru all texts
    while ( $a2wCurTextTmp != 0 ){
        # V107 Begin
        #---- Update parent context with object info
        @arrPtCtxt1Tmp = @arrPtCtxtTmp; # make a copy

        #---- Add object entry on parent context
        push( @arrPtCtxt1Tmp, { 'STATE' => $a2wCurTextTmp, 'INC_X' => $a2wCurTextTmp->getXPos(), 'INC_Y' => $a2wCurTextTmp->getYPos(), 'RES' => $arrPtCtxt1Tmp[ $#arrPtCtxt1Tmp ]->{ 'RES' }, 'TYPE' => $a2w::core::dm::Constants::PARENT_TYPE_TEXT } );

        #---- Get object position in page
        my $hrefObjPosTmp = $this->_getObjectPositionInPage( \@arrPtCtxt1Tmp );
        # V107 End

        # V106 Begin
        #---- Get text attributes and Create object context
        my $hrefObjContextTmp = {
              'OBJECT' => $a2wCurTextTmp
            , 'TYPE'   => $a2w::core::dm::Constants::OT_TEXT
            , 'XPOS'   => $hrefObjPosTmp->{ 'X' } # Get X position # V107 Change
            , 'YPOS'   => $hrefObjPosTmp->{ 'Y' } # Get Y position # V107 Change
            , 'COLOR'  => $a2wCurTextTmp->getColor() # Get color
            , 'ANGLE'  => $a2wCurTextTmp->getAngle() # Get angle
        };

        #---- Preprocess and add object to page
        $this->_preProcessAndAddObject( $a2wPageTmp, $dbContentPar, $hrefObjContextTmp );
        # V106 End

        #---- Get the next text
        $a2wCurTextTmp = $a2wContentObjectPar->getNextText();
    } # end-while
}

#-----------------------------------------------------------------------
#
# Add the Line objects to the database
#
#-----------------------------------------------------------------------
sub _addLineObjs{
    require a2w::Line;    # Include line module

    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addLineObjs" );
    }

    #---- Get parameter of _addLineObjs ( Par: Page/Overlay instance, Database instance, Position Adjustment )
    my $a2wContentObjectPar  = shift;
    my $dbContentPar         = shift;
    # V107 Begin
    # Parent context (of type array of hash references, refer "parse" function for more details)
    #
    my $arefParentContextPar = shift;

    #---- Fetch parent context value
    my @arrPtCtxtTmp = @{ $arefParentContextPar };
    my @arrPtCtxt1Tmp = ();
    # V107 End

    my $a2wPageTmp = $dbContentPar->getPageReference(); # V106 Change

    #---- Fill with line contents
    #---- Fetch first line
    my $a2wCurLineTmp = $a2wContentObjectPar->getFirstLine();

    #---- Loop thru all lines
    while ( $a2wCurLineTmp != 0 ){
        # V107 Begin
        #---- Update parent context with object info
        @arrPtCtxt1Tmp = @arrPtCtxtTmp; # make a copy

        #---- Add object entry on parent context
        push( @arrPtCtxt1Tmp, { 'STATE' => $a2wCurLineTmp, 'INC_X' => $a2wCurLineTmp->getXPos(), 'INC_Y' => $a2wCurLineTmp->getYPos(), 'RES' => $arrPtCtxt1Tmp[ $#arrPtCtxt1Tmp ]->{ 'RES' }, 'TYPE' => $a2w::core::dm::Constants::PARENT_TYPE_LINE } );

        #---- Get object position in page
        my $hrefObjPosTmp = $this->_getObjectPositionInPage( \@arrPtCtxt1Tmp );
        # V107 End

        # V106 Begin
        #---- Get line attributes and Create object context
        my $hrefObjContextTmp = {
              'OBJECT' => $a2wCurLineTmp
            , 'TYPE'   => $a2w::core::dm::Constants::OT_LINE
            , 'XPOS'   => $hrefObjPosTmp->{ 'X' } # Get X position # V107 Change
            , 'YPOS'   => $hrefObjPosTmp->{ 'Y' } # Get Y position # V107 Change
            , 'COLOR'  => $a2wCurLineTmp->getColor() # Get color
            , 'ANGLE'  => 0 #!!! Line does not have Angle property !!!
        };

        #---- Preprocess and add object to page
        $this->_preProcessAndAddObject( $a2wPageTmp, $dbContentPar, $hrefObjContextTmp );
        # V106 End

        #---- Get the next line
        $a2wCurLineTmp = $a2wContentObjectPar->getNextLine();
    } # end-while
}

#-----------------------------------------------------------------------
#
# Add the Vector objects to the database
#
#-----------------------------------------------------------------------
sub _addVectorObjs{
    require a2w::Vector;    # Include vector module

    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addVectorObjs" );
    }

    #---- Get parameter of _addVectorObjs ( Par: Page/Overlay/PageSegment instance, Database instance, Position Adjustment )
    my $a2wContentObjectPar  = shift;
    my $dbContentPar         = shift;
    # V107 Begin
    # Parent context (of type array of hash references, refer "parse" function for more details)
    #
    my $arefParentContextPar = shift;

    #---- Fetch parent context value
    my @arrPtCtxtTmp = @{ $arefParentContextPar };
    my @arrPtCtxt1Tmp = ();
    # V107 End

    my $a2wPageTmp = $dbContentPar->getPageReference(); # V106 Change

    #---- Fill with vector contents
    #---- Fetch first vector
    my $a2wCurVectorTmp = $a2wContentObjectPar->getFirstVector();
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "$a2wContentObjectPar=$a2wCurVectorTmp" ); }

    #---- Loop thru all vectors
    while ( $a2wCurVectorTmp != 0 ){
        # V107 Begin
        #---- Update parent context with object info
        @arrPtCtxt1Tmp = @arrPtCtxtTmp; # make a copy

        #---- Add object entry on parent context
        push( @arrPtCtxt1Tmp, { 'STATE' => $a2wCurVectorTmp, 'INC_X' => $a2wCurVectorTmp->getXPos(), 'INC_Y' => $a2wCurVectorTmp->getYPos(), 'RES' => $arrPtCtxt1Tmp[ $#arrPtCtxt1Tmp ]->{ 'RES' }, 'TYPE' => $a2w::core::dm::Constants::PARENT_TYPE_VECTOR } );

        #---- Get object position in page
        my $hrefObjPosTmp = $this->_getObjectPositionInPage( \@arrPtCtxt1Tmp );
        # V107 End

        # V106 Begin
        #---- Get vector attributes and Create object context
        my $hrefObjContextTmp = {
              'OBJECT' => $a2wCurVectorTmp
            , 'TYPE'   => $a2w::core::dm::Constants::OT_VECTOR
            , 'XPOS'   => $hrefObjPosTmp->{ 'X' } # Get X position # V107 Change
            , 'YPOS'   => $hrefObjPosTmp->{ 'Y' } # Get Y position # V107 Change
            , 'COLOR'  => $a2wCurVectorTmp->getColor() # Get color
            , 'ANGLE'  => $a2wCurVectorTmp->getAngle() # Get angle
        };

        #---- Preprocess and add object to page
        $this->_preProcessAndAddObject( $a2wPageTmp, $dbContentPar, $hrefObjContextTmp );
        # V106 End

        #---- Get the next vector
        $a2wCurVectorTmp = $a2wContentObjectPar->getNextVector();
    } # end-while
}

#-----------------------------------------------------------------------
#
# Add the Image objects to the database
#
#-----------------------------------------------------------------------
sub _addImageObjs{
    require a2w::Image;    # Include image module

    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addImageObjs" );
    }

    #---- Get parameter of _addImageObjs ( Par: Page/Overlay/PageSegment instance, Database instance, Position Adjustment )
    my $a2wContentObjectPar  = shift;
    my $dbContentPar         = shift;
    # V107 Begin
    # Parent context (of type array of hash references, refer "parse" function for more details)
    #
    my $arefParentContextPar = shift;

    #---- Fetch parent context value
    my @arrPtCtxtTmp = @{ $arefParentContextPar };
    my @arrPtCtxt1Tmp = ();
    # V107 End

    my $a2wPageTmp = $dbContentPar->getPageReference(); # V106 Change

    #---- Fill with image contents
    #---- Fetch first image
    my $a2wCurImageTmp = $a2wContentObjectPar->getFirstImage();

    #---- Loop thru all images
    while ( $a2wCurImageTmp != 0 ){
        # V107 Begin
        #---- Update parent context with object info
        @arrPtCtxt1Tmp = @arrPtCtxtTmp; # make a copy

        #---- Add object entry on parent context
        push( @arrPtCtxt1Tmp, { 'STATE' => $a2wCurImageTmp, 'INC_X' => $a2wCurImageTmp->getXPos(), 'INC_Y' => $a2wCurImageTmp->getYPos(), 'RES' => $a2wCurImageTmp->getResolution(), 'TYPE' => $a2w::core::dm::Constants::PARENT_TYPE_IMAGE } );

        #---- Get object position in page
        my $hrefObjPosTmp = $this->_getObjectPositionInPage( \@arrPtCtxt1Tmp );
        # V107 End

        # V106 Begin
        #---- Get image attributes and Create object context
        my $hrefObjContextTmp = {
              'OBJECT' => $a2wCurImageTmp
            , 'TYPE'   => $a2w::core::dm::Constants::OT_IMAGE
            , 'XPOS'   => $hrefObjPosTmp->{ 'X' } # Get X position # V107 Change
            , 'YPOS'   => $hrefObjPosTmp->{ 'Y' } # Get Y position # V107 Change
            , 'COLOR'  => $a2wCurImageTmp->getColor() # Get color # V108 Change
            , 'ANGLE'  => $a2wCurImageTmp->getAngle() # Get angle
        };

        #---- Preprocess and add object to page
        $this->_preProcessAndAddObject( $a2wPageTmp, $dbContentPar, $hrefObjContextTmp );
        # V106 End

        #---- Get the next image
        $a2wCurImageTmp = $a2wContentObjectPar->getNextImage();
    } # end-while
}

# $V103 Begin
#-----------------------------------------------------------------------
#
# Add the Container objects to the database
#
#-----------------------------------------------------------------------
sub _addContainerObjs{
    require a2w::Container;    # Include container module

    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addContainerObjs" );
    }

    #---- Get parameter of _addImageObjs ( Par: Page/Overlay instance, Database instance, Position Adjustment )
    my $a2wContentObjectPar  = shift;
    my $dbContentPar         = shift;

    # V107 Begin
    # Parent context (of type array of hash references, refer "parse" function for more details)
    #
    my $arefParentContextPar = shift;

    #---- Fetch parent context value
    my @arrPtCtxtTmp = @{ $arefParentContextPar };
    my @arrPtCtxt1Tmp = ();
    # V107 End

    my $a2wPageTmp = $dbContentPar->getPageReference(); # V106 Change

    #---- Fill with container contents
    #---- Fetch first container
    my $a2wCurContainerTmp = $a2wContentObjectPar->getFirstContainer();

    #---- Loop thru all containers
    while ( $a2wCurContainerTmp != 0 ){
        # V107 Begin
        #---- Update parent context with object info
        @arrPtCtxt1Tmp = @arrPtCtxtTmp; # make a copy
        $arrPtCtxt1Tmp[ $#arrPtCtxt1Tmp ]->{ 'INC_X' } = $a2wCurContainerTmp->getIncludedXPosition(); # Set container object included position on container object's parent
        $arrPtCtxt1Tmp[ $#arrPtCtxt1Tmp ]->{ 'INC_Y' } = $a2wCurContainerTmp->getIncludedYPosition(); # Set container object included position on container object's parent

        #---- Add object entry on parent context
        push( @arrPtCtxt1Tmp, { 'STATE' => $a2wCurContainerTmp, 'INC_X' => $a2wCurContainerTmp->getXPos(), 'INC_Y' => $a2wCurContainerTmp->getYPos(), 'RES' => $a2wCurContainerTmp->getResolution(), 'TYPE' => $a2w::core::dm::Constants::PARENT_TYPE_CONTAINER } );

        #---- Get object position in page
        my $hrefObjPosTmp = $this->_getObjectPositionInPage( \@arrPtCtxt1Tmp );
        # V107 End

        # V106 Begin
        #---- Get vector attributes and Create object context
        my $hrefObjContextTmp = {
              'OBJECT' => $a2wCurContainerTmp
            , 'TYPE'   => $a2w::core::dm::Constants::OT_CONTAINER
            , 'XPOS'   => $hrefObjPosTmp->{ 'X' } # Get X position # V107 Change
            , 'YPOS'   => $hrefObjPosTmp->{ 'Y' } # Get Y position # V107 Change
            , 'COLOR'  => $a2wCurContainerTmp->getColor() # Get color # V108 Change
            , 'ANGLE'  => $a2wCurContainerTmp->getAngle() # Get angle
        };

        #---- Preprocess and add object to page
        $this->_preProcessAndAddObject( $a2wPageTmp, $dbContentPar, $hrefObjContextTmp );
        # V106 End

        #---- Get the next image
        $a2wCurContainerTmp = $a2wContentObjectPar->getNextContainer();
    } # end-while
}
# $V103 End

#-----------------------------------------------------------------------
#
# Add the Overlays to the database
#
#-----------------------------------------------------------------------
sub _addOverlays{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addOverlays" );
    }

    #---- Get parameter of _addOverlays ( Par: Overlay instance, Database instance )
    my $a2wContentObjectPar = shift;
    my $dbContentPar        = shift;

    # V107 Begin
    # Parent context (of type array of hash references, refer "parse" function for more details)
    #
    my $arefParentContextPar = shift;

    #---- Fetch parent context value
    my @arrPtCtxtTmp = @{ $arefParentContextPar };
    my @arrPtCtxt1Tmp = ();
    # V107 End

    require a2w::Overlay;    # Include overlay module

    #---- Fetch 1st Overlay
    my $a2wOlyTmp = $a2wContentObjectPar->getFirstOverlay();

    #---- Loop thru all the Overlays
    # $V103 Begin
    my $iResIdTmp     = 0; # Resource id in page # V105 Change

    while ( $a2wOlyTmp != 0 ){
        #---- Fill database with overlay contents
        #
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Overlay (" . $a2wOlyTmp->getName() . ") having sequence id " . $a2wOlyTmp->getSequenceId() . " is included at position (" . $a2wOlyTmp->getIncludedXPosition() . "," . $a2wOlyTmp->getIncludedYPosition() . ")" ); }

        # V107 Begin
        #---- Update parent context with overlay info
        @arrPtCtxt1Tmp = @arrPtCtxtTmp; # make a copy
        $arrPtCtxt1Tmp[ $#arrPtCtxt1Tmp ]->{ 'INC_X' } = $a2wOlyTmp->getIncludedXPosition(); # Set overlay included position on overlay's parent
        $arrPtCtxt1Tmp[ $#arrPtCtxt1Tmp ]->{ 'INC_Y' } = $a2wOlyTmp->getIncludedYPosition(); # Set overlay included position on overlay's parent

        #---- Add overlay entry on parent context
        push( @arrPtCtxt1Tmp, { 'STATE' => $a2wOlyTmp, 'INC_X' => 0, 'INC_Y' => 0, 'RES' => $a2wOlyTmp->getResolution(), 'TYPE' => $a2w::core::dm::Constants::PARENT_TYPE_OVERLAY } );

        # V105 Begin
        #---- Update resource id in database context
        $iResIdTmp = $a2wOlyTmp->getSequenceId();
        if ( $iResIdTmp > 0 ){ $dbContentPar->setResId( $iResIdTmp ); }
        # V105 End

        #---- Fill with text contents
        $this->_addTextObjs( $a2wOlyTmp, $dbContentPar, \@arrPtCtxt1Tmp );

        #---- Fill with line contents
        $this->_addLineObjs( $a2wOlyTmp, $dbContentPar, \@arrPtCtxt1Tmp );

        #---- Add Vector Objects
        $this->_addVectorObjs( $a2wOlyTmp, $dbContentPar, \@arrPtCtxt1Tmp );

        #---- Add Image Objects
        $this->_addImageObjs( $a2wOlyTmp, $dbContentPar, \@arrPtCtxt1Tmp );

        #---- Add Container Objects
        $this->_addContainerObjs( $a2wOlyTmp, $dbContentPar, \@arrPtCtxt1Tmp );

        #---- Add Overlays
        $this->_addOverlays( $a2wOlyTmp, $dbContentPar, \@arrPtCtxt1Tmp );

        #---- Add Page Segments
        $this->_addPageSegments( $a2wOlyTmp, $dbContentPar, \@arrPtCtxt1Tmp );
        # V107 End

        #---- Reset resource id in database context
        $dbContentPar->setResId( 0 ); # V105 Change

        #---- Get the next Overlay
        $a2wOlyTmp = $a2wContentObjectPar->getNextOverlay();
    } # while ( $a2wOlyTmp != 0 )
    # $V103 End
}

# $V102 Begin
#-----------------------------------------------------------------------
#
# Add the Page Segments to the database
#
#-----------------------------------------------------------------------
sub _addPageSegments{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "_addPageSegments" );
    }

    #---- Get parameter of _addPageSegments ( Par: Page segment instance, Database instance )
    my $a2wContentObjectPar = shift;
    my $dbContentPar        = shift;

    # V107 Begin
    # Parent context (of type array of hash references, refer "parse" function for more details)
    #
    my $arefParentContextPar = shift;

    #---- Fetch parent context value
    my @arrPtCtxtTmp = @{ $arefParentContextPar };
    my @arrPtCtxt1Tmp = ();
    # V107 End

    require a2w::PSEG;    # Include Page Segment module

    #---- Fetch 1st Page Segment
    my $a2wPSEGTmp = $a2wContentObjectPar->getFirstPageSegment();

    #---- Loop thru all the Page Segments
    # $V103 Begin
    my $iResIdTmp     = 0; # Resource id in page # V105 Change

    while ( $a2wPSEGTmp != 0 ){
        #---- Fill database with page segment contents
        #
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Page Segment (" . $a2wPSEGTmp->getName() . ") having sequence id " . $a2wPSEGTmp->getSequenceId() . " is included at position (" . $a2wPSEGTmp->getIncludedXPosition() . "," . $a2wPSEGTmp->getIncludedYPosition() . ")" ); }

        # V107 Begin
        #---- Update parent context with page segment info
        @arrPtCtxt1Tmp = @arrPtCtxtTmp; # make a copy
        $arrPtCtxt1Tmp[ $#arrPtCtxt1Tmp ]->{ 'INC_X' } = $a2wPSEGTmp->getIncludedXPosition(); # Set page segment included position on page segment's parent
        $arrPtCtxt1Tmp[ $#arrPtCtxt1Tmp ]->{ 'INC_Y' } = $a2wPSEGTmp->getIncludedYPosition(); # Set page segment included position on page segment's parent

        #---- Add page segment entry on parent context
        push( @arrPtCtxt1Tmp, { 'STATE' => $a2wPSEGTmp, 'INC_X' => 0, 'INC_Y' => 0, 'RES' => 0, 'TYPE' => $a2w::core::dm::Constants::PARENT_TYPE_PAGE_SEGMENT } );

        # V105 Begin
        #---- Update resource id in database context
        $iResIdTmp = $a2wPSEGTmp->getSequenceId();
        if ( $iResIdTmp > 0 ){ $dbContentPar->setResId( $iResIdTmp ); }
        # V105 End

        #---- Add Image Objects
        $this->_addImageObjs( $a2wPSEGTmp, $dbContentPar, \@arrPtCtxt1Tmp );

        #---- Add Vector Objects
        #$this->_addVectorObjs( $a2wPSEGTmp, $dbContentPar, \@arrPtCtxt1Tmp ); # !!!Uncomment after fixing core bug!!!
        # V107 End

        #---- Reset resource id in database context
        $dbContentPar->setResId( 0 ); # V105 Change

        #---- Get the next Page Segment
        $a2wPSEGTmp = $a2wContentObjectPar->getNextPageSegment();
    } # while ( $a2wPSEGTmp != 0 )
    # $V103 End
}
# $V102 End

# $V106 Begin
#-----------------------------------------------------------------------
#
# Create object context from given core object
#
#-----------------------------------------------------------------------
sub _createObjectContext{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_createObjectContext" ); }

    #---- Get parameter of _createObjectContext ( Par: a2w::<Container|Image|Line|Text|Vector>Object )
    my $a2wObjPar = shift;

    #---- Collect object info
    my $sObjTypeTmp = $a2wObjPar->_getType();
    my $iObjTypeTmp = 0;
    my $iAngleTmp   = 0;
    my $iColorTmp   = 0;
    if ( $sObjTypeTmp eq "text" ){
        $iObjTypeTmp = $a2w::core::dm::Constants::OT_TEXT;
        $iAngleTmp   = $a2wObjPar->getAngle();
        $iColorTmp   = $a2wObjPar->getColor();
    }
    elsif ( $sObjTypeTmp eq "line" ){
        $iObjTypeTmp = $a2w::core::dm::Constants::OT_LINE;
        $iAngleTmp   = 0; #!!! Line does not have Angle property !!!
        $iColorTmp   = $a2wObjPar->getColor();
    }
    elsif ( $sObjTypeTmp eq "image" ){
        # $V102 Begin
        #---- Assert image/page resolution for inline image and evaluate proper position of image
        #my $iXPosTmp = $a2wObjPar->getXPos();
        #my $iYPosTmp = $a2wObjPar->getYPos();
        # $V104 Begin
        #if ( $a2wObjPar->isInline() == $TRUE ){
        #    my $iImResTmp = $a2wObjPar->getResolution();
        #    my $iPgResTmp = $dbPageTmp->getPageRes();
        #
        #    if ( $iImResTmp != $iPgResTmp ){
        #        my $fScaleTmp = $iPgResTmp / $iImResTmp;
        #        $iXPosTmp *= $fScaleTmp;
        #        $iYPosTmp *= $fScaleTmp;
        #    }
        #}
        # $V104 End
        # $V102 End
        $iObjTypeTmp = $a2w::core::dm::Constants::OT_IMAGE;
        $iAngleTmp   = $a2wObjPar->getAngle();
        $iColorTmp   = $a2wObjPar->getColor(); # V108 Change
    }
    # $V103 Begin
    elsif ( $sObjTypeTmp eq "container" ){
        $iObjTypeTmp = $a2w::core::dm::Constants::OT_CONTAINER;
        $iAngleTmp   = $a2wObjPar->getAngle();
        $iColorTmp   = $a2wObjPar->getColor(); # V108 Change
    }
    # $V103 End
    elsif ( $sObjTypeTmp eq "vector" ){
        $iObjTypeTmp = $a2w::core::dm::Constants::OT_VECTOR;
        $iAngleTmp   = $a2wObjPar->getAngle();
        $iColorTmp   = $a2wObjPar->getColor();
    }
    # V109 Begin
    elsif ( $sObjTypeTmp eq "annotation" ){
        $iObjTypeTmp = $a2w::core::dm::Constants::OT_ANNOTATION;
        $iAngleTmp   = 0; #!!! Annotation does not have Angle property !!!
        $iColorTmp   = $a2wObjPar->getColor();
    }
    # V109 End

    #---- Create object context
    my $hrefObjContextTmp = {
          'OBJECT' => $a2wObjPar
        , 'TYPE'   => $iObjTypeTmp
        , 'XPOS'   => $a2wObjPar->getXPos()
        , 'YPOS'   => $a2wObjPar->getYPos()
        , 'COLOR'  => $iColorTmp # V108 Change
        , 'ANGLE'  => $iAngleTmp # V108 Change
    };

    return $hrefObjContextTmp;
}

#-----------------------------------------------------------------------
#
# _preProcessAndAddObject
#
# Analyze object against preprocess anchors. If object match anchor, then invoke the
# callback and get result objects to add on page
#
# Parameters
# a2wPagePar    a2w::Page                  Page reference
# dbPagePar     a2w::core::dm::Database    Page database to add objects
# objContextPar Hash Reference             Object context hash
#                                          ObjectContext => {
#                                              'OBJECT' => a2w::<Container|Image|Line|Text|Vector>Object
#                                              'TYPE'   => <object type>
#                                              'XPOS'   => <x position>
#                                              'YPOS'   => <y position>
#                                              'COLOR'  => <color>
#                                              'ANGLE'  => <angle>
#                                          }
# pomObjectPar  Hash Reference             OPTIONAL, Database specific object info context
#
# Callback prototype:
# <Array of objects> <function name>( a2w::core::dm::Anchor, a2w::Page, a2w::<Container|Image|Line|Text|Vector>Object )
#
# Callback must preprocess the object and return array of objects to add on the page
#
# NOTE: Object passed in must also be included on returned array of objects if it
#       has to be added on page otherwise the object will be skipped
#
#-----------------------------------------------------------------------
sub _preProcessAndAddObject{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_preProcessAndAddObject" ); }

    #---- Get parameter of _preProcess
    my $a2wPagePar    = shift;
    my $dbPagePar     = shift;
    my $objContextPar = shift;
    my $pomObjectPar  = undef; # Optional
    if ( @_ > 0 ){ $pomObjectPar = shift; }

    #---- Assert parameters
    if ( $a2wPagePar == undef || $dbPagePar == undef || $objContextPar == undef ){ return; }
    $this->{ 'ObjectCount' }++;
    $this->{ 'PageLastObject' } = $objContextPar->{ 'OBJECT' };
    my $arefRulesTmp = $this->{ 'Rules' };
    my @arrRulesTmp = @{ $arefRulesTmp };

    #---- No rules, nothing to preprocess. Just add object to database
    my $bLastObjTmp = $FALSE;
    if ( $pomObjectPar != undef && $pomObjectPar->{ $a2w::core::dm::Constants::AT_PAGELAST } == $TRUE ){ $bLastObjTmp = $TRUE; }
    if (    @arrRulesTmp <= 0
	     && $bLastObjTmp == $FALSE # to avoid duplicate addition for last object when no rules is given
	   ){
        $dbPagePar->addObject(   $objContextPar->{ 'TYPE' }    # Object type
                               , $objContextPar->{ 'XPOS' }    # X Position
                               , $objContextPar->{ 'YPOS' }    # Y Position
                               , $objContextPar->{ 'ANGLE' }   # Angle
                               , $objContextPar->{ 'COLOR' }   # Color
                               , $objContextPar->{ 'OBJECT' }  # Object reference
                             );
        return;
    }

    #---- Iterate through rules
    my $bObjAddedTmp = $FALSE;
    my $bRuleMatchTmp = $FALSE;
    foreach my $r ( @arrRulesTmp ){
        if ( $r->{ 'ANCHOR' } == undef || $r->{ 'CALLBACK' } == undef ){ next; } # Invalid rule, skip to next

        #---- Assert object against rule
        $bRuleMatchTmp = $r->{ 'ANCHOR' }->doesConstraintsMatch( $objContextPar->{ 'OBJECT' } );
        # Last object case
        if (    $bRuleMatchTmp == $FALSE # V110 Change
             && $pomObjectPar != undef
           ){
            $bRuleMatchTmp = $r->{ 'ANCHOR' }->doesConstraintsMatch( $objContextPar->{ 'OBJECT' }, $pomObjectPar );
        }
        # First object case
        elsif (    $bRuleMatchTmp == $FALSE # V110 Change
                && $this->{ 'ObjectCount' } == 1
              ){
            my $hrefPOMObjTmp = { $a2w::core::dm::Constants::AT_PAGEFIRST => $TRUE };
            $bRuleMatchTmp = $r->{ 'ANCHOR' }->doesConstraintsMatch( $objContextPar->{ 'OBJECT' }, $hrefPOMObjTmp );
        }
        if ( $bRuleMatchTmp == $TRUE ){
            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Object match rule " . $r->{ 'ANCHOR' }->getId() ); }

            #---- Invoke callback
            my $fnCBTmp = $r->{ 'CALLBACK' };
            my $arefPPObjsTmp = &$fnCBTmp( $r->{ 'ANCHOR' }, $a2wPagePar, $objContextPar->{ 'OBJECT' } );

            #---- Process callback returned objects and add them to page
            my @arrPPObjsTmp = @{ $arefPPObjsTmp };
            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Object preprocessing callback returned " . @arrPPObjsTmp . " objects, adding them to page" ); }
            foreach my $o ( @arrPPObjsTmp ){
                if ( $o == undef ){ next; } # Invalid object, skip to next

                #---- Handle object being checked separately (to use x, y positions passed in object context)
                my $osid = $o->getSequenceId();
                my $aosid = $objContextPar->{ 'OBJECT' }->getSequenceId();
                if (    $osid > 0
				     && $aosid > 0
				     && $osid == $aosid
				   ){
                    $bObjAddedTmp = $TRUE;
                    $dbPagePar->addObject(   $objContextPar->{ 'TYPE' }    # Object type
                                           , $objContextPar->{ 'XPOS' }    # X Position
                                           , $objContextPar->{ 'YPOS' }    # Y Position
                                           , $objContextPar->{ 'ANGLE' }   # Angle
                                           , $objContextPar->{ 'COLOR' }   # Color
                                           , $objContextPar->{ 'OBJECT' }  # Object reference
                                         );
                }
                else {
                    my $hrefObjContextTmp = $this->_createObjectContext( $o );
                    $dbPagePar->addObject(   $hrefObjContextTmp->{ 'TYPE' }    # Object type
                                           , $hrefObjContextTmp->{ 'XPOS' }    # X Position
                                           , $hrefObjContextTmp->{ 'YPOS' }    # Y Position
                                           , $hrefObjContextTmp->{ 'ANGLE' }   # Angle
                                           , $hrefObjContextTmp->{ 'COLOR' }   # Color
                                           , $hrefObjContextTmp->{ 'OBJECT' }  # Object reference
                                         );
                }
            }
        }
    }

    #---- Add object alone if not added so far (default action)
    if ( $bObjAddedTmp == $FALSE && $pomObjectPar == undef ){
        $dbPagePar->addObject(   $objContextPar->{ 'TYPE' }    # Object type
                               , $objContextPar->{ 'XPOS' }    # X Position
                               , $objContextPar->{ 'YPOS' }    # Y Position
                               , $objContextPar->{ 'ANGLE' }   # Angle
                               , $objContextPar->{ 'COLOR' }   # Color
                               , $objContextPar->{ 'OBJECT' }  # Object reference
                             );
    }
}
# $V106 End

# V107 Begin
#-----------------------------------------------------------------------
# _getObjectPositionInPage
#
# Evaluates given object position in page and returns the position
#
# Return value
# { 'X' => <object x postion>, 'Y' => <object y postion> }
#
#-----------------------------------------------------------------------
sub _getObjectPositionInPage{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_getObjectPositionInPage" ); }

    #---- Get parameter of _getObjectPositionInPage
    # Parent context (of type array of hash references, refer "parse" function for more details)
    #
    my $arefParentContextPar = shift;

    #---- Fetch parent context value
    my @arrPtCtxtTmp = @{ $arefParentContextPar };

    #---- Iterate through the parent context array and evaluate the position of object in page
    my $hrefObjPosInPageTmp = { 'X' => 0, 'Y' => 0 };
    my $prevParent = undef;
    my $parent = undef;
    for ( my $i = $#arrPtCtxtTmp; $i >= 0; $i-- ){
        $parent = $arrPtCtxtTmp[ $i ];

        if (    $parent->{ 'TYPE' } == $a2w::core::dm::Constants::PARENT_TYPE_TEXT
		     || $parent->{ 'TYPE' } == $a2w::core::dm::Constants::PARENT_TYPE_LINE
		     || $parent->{ 'TYPE' } == $a2w::core::dm::Constants::PARENT_TYPE_IMAGE
		     || $parent->{ 'TYPE' } == $a2w::core::dm::Constants::PARENT_TYPE_VECTOR
		     || $parent->{ 'TYPE' } == $a2w::core::dm::Constants::PARENT_TYPE_CONTAINER
		   ){
		    $hrefObjPosInPageTmp->{ 'X' } = $parent->{ 'INC_X' };
		    $hrefObjPosInPageTmp->{ 'Y' } = $parent->{ 'INC_Y' };
		    $prevParent = $parent;
        }
        elsif ( $parent->{ 'TYPE' } == $a2w::core::dm::Constants::PARENT_TYPE_PAGE ){
            #---- Factorize object position
            if ( $parent->{ 'RES' } != $prevParent->{ 'RES' } ){
                my $factor = ( $parent->{ 'RES' } / $prevParent->{ 'RES' } );

		        $hrefObjPosInPageTmp->{ 'X' } *= $factor;
		        $hrefObjPosInPageTmp->{ 'Y' } *= $factor;
            }

		    #---- Add included position
		    $hrefObjPosInPageTmp->{ 'X' } += $parent->{ 'INC_X' };
		    $hrefObjPosInPageTmp->{ 'Y' } += $parent->{ 'INC_Y' };
        }
        elsif ( $parent->{ 'TYPE' } == $a2w::core::dm::Constants::PARENT_TYPE_OVERLAY ){
            #---- Factorize object position
            if ( $parent->{ 'RES' } != $prevParent->{ 'RES' } ){
                my $factor = ( $parent->{ 'RES' } / $prevParent->{ 'RES' } );

		        $hrefObjPosInPageTmp->{ 'X' } *= $factor;
		        $hrefObjPosInPageTmp->{ 'Y' } *= $factor;
            }

		    #---- Add included position
		    $hrefObjPosInPageTmp->{ 'X' } += $parent->{ 'INC_X' };
		    $hrefObjPosInPageTmp->{ 'Y' } += $parent->{ 'INC_Y' };
            $prevParent = $parent;
        }
        #elsif ( $parent->{ 'TYPE' } == $a2w::core::dm::Constants::PARENT_TYPE_PAGE_SEGMENT ){
        # do nothing
        #}
    }

    return $hrefObjPosInPageTmp;
}
# V107 End

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__

