#-------------------------------------------------------------------------------
#  a2w/core/dm/ContentProcessor.pm
#
#  Perl module to process block contents
#
#  Author   : Panneer, AFP2web Team
#  Date     : 2021-09-21
#  Version  : 1.0.3
#
#  V100   2014-02-14    Initial Release
#
#  V101   2018-01-26    Redesigned to have multiple constraints to have OR conditionals
#
#  V102   2018-09-04    Fixed minor bug (when one index condition fails all the other indexes
#                        following were ignored) in collecting index
#
#  V103   2021-09-21    AFP-1068: Extended with FindAndPrepend and FindAndAppend content processing objects
#
#-------------------------------------------------------------------------------
package a2w::core::dm::ContentProcessor;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;

use a2w::core::dm::Constants;
use a2w::core::dm::Constraint;
use a2w::core::dm::FindAndReplace;
use a2w::core::dm::FindAndPrepend;   # V103 Change
use a2w::core::dm::FindAndAppend;    # V103 Change
use a2w::core::dm::Index;

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    my $this = {
          'FNRList'      => []    # List of Find and Replace
        , 'FNPList'      => []    # List of Find and Prepend   # V103 Change
        , 'FNAList'      => []    # List of Find and Append    # V103 Change
        , 'IDXList'      => []    # List of indexes
    };

    bless( $this, $class );

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}

    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------
sub setFindAndReplaceList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setFindAndReplaceList" );
    #}

    $this->{ 'FNRList' } = shift;
}

# V103 Begin
sub setFindAndPrependList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setFindAndPrependList" );
    #}

    $this->{ 'FNPList' } = shift;
}

sub setFindAndAppendList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setFindAndAppendList" );
    #}

    $this->{ 'FNAList' } = shift;
}
# V103 End

sub setIndexList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "setIndexList" );
    #}

    $this->{ 'IDXList' } = shift;
}

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------
sub getFindAndReplaceList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getFindAndReplaceList" );
    #}

    return $this->{ 'FNRList' };
}

# V103 Begin
sub getFindAndPrependList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getFindAndPrependList" );
    #}

    return $this->{ 'FNPList' };
}

sub getFindAndAppendList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getFindAndAppendList" );
    #}

    return $this->{ 'FNAList' };
}
# V103 End

sub getIndexList{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getIndexList" );
    #}

    return $this->{ 'IDXList' };
}

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
sub hasFNRs{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "hasFNRs" );
    #}

    my @arrFNRsTmp = @{ $this->{ 'FNRList' } };
    my $bRetTmp = $FALSE;
    if ( @arrFNRsTmp > 0 ){
        $bRetTmp = $TRUE;
    }

    return $bRetTmp;
}

# V103 Begin
sub hasFNPs{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "hasFNPs" );
    #}

    my @arrFNPsTmp = @{ $this->{ 'FNPList' } };
    return ( @arrFNPsTmp > 0 ) ? $TRUE : $FALSE;
}

sub hasFNAs{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "hasFNAs" );
    #}

    my @arrFNAsTmp = @{ $this->{ 'FNAList' } };
    return ( @arrFNAsTmp > 0 ) ? $TRUE : $FALSE;
}
# V103 End

sub hasIndexes{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "hasIndexes" );
    #}

    my @arrIndexesTmp = @{ $this->{ 'IDXList' } };
    my $bRetTmp = $FALSE;
    if ( @arrIndexesTmp > 0 ){
        $bRetTmp = $TRUE;
    }

    return $bRetTmp;
}

sub getFNRCount{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getFNRCount" );
    #}

    my @arrFNRsTmp = @{ $this->{ 'FNRList' } };
    my $iCountTmp = @arrFNRsTmp;
    return $iCountTmp;
}

# V103 Begin
sub getFNRCount{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getFNRCount" );
    #}

    my @arrFNPsTmp = @{ $this->{ 'FNPList' } };
    my $iCountTmp = @arrFNPsTmp;
    return $iCountTmp;
}

sub getFNACount{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getFNACount" );
    #}

    my @arrFNAsTmp = @{ $this->{ 'FNAList' } };
    my $iCountTmp = @arrFNAsTmp;
    return $iCountTmp;
}
# V103 End

sub getIndexesCount{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getIndexesCount" );
    #}

    my @arrIndexesTmp = @{ $this->{ 'IDXList' } };
    my $iCountTmp = @arrIndexesTmp;
    return $iCountTmp;
}

sub addFindAndReplace{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "addFindAndReplace" );
    #}

    #---- Get parameter
    #
    # 1. Name
    # 2. Find and Replace (config hash)
    #
    my $sNamePar     = shift;
    my $configFNRPar = shift;

    #---- Assert parameter
    if (    $sNamePar eq ""
         || $configFNRPar->{ 'Constraints' } == undef
         || $configFNRPar->{ 'Replace' } eq ""
       ){
        if ( $bLog == $TRUE ){
            if ( $sNamePar ne "" ){
                $theLogger->logMessage( "Warning! Ignored config Find and Replace (" . $sNamePar . ") due to missing Constraints or Replace" );
            }
            else{
                $theLogger->logMessage( "Warning! Ignored config Find and Replace due to missing Name" );
            }
        }
        return -1;
    }

    #---- Create find and replace
    my $fnrTmp = new a2w::core::dm::FindAndReplace();

    #---- Fill in details
    $fnrTmp->setName( $sNamePar );
    $fnrTmp->setReplace( $configFNRPar->{ 'Replace' } );
    $fnrTmp->setApplicableOutputs( $configFNRPar->{ 'ApplicableOutputs' } ); # V103 Change

    # V101 Begin
    #$fnrTmp->createAndAddConstraints( $configFNRPar->{ 'Constraints' } );
    #
    my @arrConsTmp = grep { lc( $_ ) =~ /constraints*/ } keys( %{ $configFNRPar } );
    foreach my $cons ( @arrConsTmp ){
        $fnrTmp->createAndAddConstraints( $cons, $configFNRPar->{ $cons } );
    }
    # V101 End

    if ( defined $configFNRPar->{ 'Column' } ){
        $fnrTmp->setColumn( $configFNRPar->{ 'Column' } );
    }

    #---- Add FNR to list
    $this->{ 'FNRList' }[ @{ $this->{ 'FNRList' } } ] = $fnrTmp;

    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Added FNR (N=>" . $sNamePar . "< R=>" . $configFNRPar->{ 'Replace' } . "<)" );
    }

    return 0;
}

# V103 Begin
sub addFindAndPrepend{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "addFindAndPrepend" );
    #}

    #---- Get parameter
    #
    # 1. Name
    # 2. Find and Prepend (config hash)
    #
    my $sNamePar     = shift;
    my $configFNPPar = shift;

    #---- Assert parameter
    if (    $sNamePar eq ""
         || $configFNPPar->{ 'Constraints' } == undef
         || $configFNPPar->{ 'Prepend' } eq ""
       ){
        if ( $bLog == $TRUE ){
            if ( $sNamePar ne "" ){
                $theLogger->logMessage( "Warning! Ignored config Find and Prepend (" . $sNamePar . ") due to missing Constraints or Prepend" );
            }
            else{
                $theLogger->logMessage( "Warning! Ignored config Find and Prepend due to missing Name" );
            }
        }
        return -1;
    }

    #---- Create find and prepend
    my $fnpTmp = new a2w::core::dm::FindAndPrepend();

    #---- Fill in details
    $fnpTmp->setName( $sNamePar );
    $fnpTmp->setPrepend( $configFNPPar->{ 'Prepend' } );
    $fnpTmp->setApplicableOutputs( $configFNPPar->{ 'ApplicableOutputs' } );

    my @arrConsTmp = grep { lc( $_ ) =~ /constraints*/ } keys( %{ $configFNPPar } );
    foreach my $cons ( @arrConsTmp ){
        $fnpTmp->createAndAddConstraints( $cons, $configFNPPar->{ $cons } );
    }

    if ( defined $configFNPPar->{ 'Column' } ){
        $fnpTmp->setColumn( $configFNPPar->{ 'Column' } );
    }

    #---- Add FNP to list
    $this->{ 'FNPList' }[ @{ $this->{ 'FNPList' } } ] = $fnpTmp;

    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Added FNP (N=>" . $sNamePar . "< P=>" . $configFNPPar->{ 'Prepend' } . "<)" );
    }

    return 0;
}

sub addFindAndAppend{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "addFindAndAppend" );
    #}

    #---- Get parameter
    #
    # 1. Name
    # 2. Find and Append (config hash)
    #
    my $sNamePar     = shift;
    my $configFNAPar = shift;

    #---- Assert parameter
    if (    $sNamePar eq ""
         || $configFNAPar->{ 'Constraints' } == undef
         || $configFNAPar->{ 'Append' } eq ""
       ){
        if ( $bLog == $TRUE ){
            if ( $sNamePar ne "" ){
                $theLogger->logMessage( "Warning! Ignored config Find and Append (" . $sNamePar . ") due to missing Constraints or Append" );
            }
            else{
                $theLogger->logMessage( "Warning! Ignored config Find and Append due to missing Name" );
            }
        }
        return -1;
    }

    #---- Create find and append
    my $fnaTmp = new a2w::core::dm::FindAndAppend();

    #---- Fill in details
    $fnaTmp->setName( $sNamePar );
    $fnaTmp->setAppend( $configFNAPar->{ 'Append' } );
    $fnaTmp->setApplicableOutputs( $configFNAPar->{ 'ApplicableOutputs' } );

    my @arrConsTmp = grep { lc( $_ ) =~ /constraints*/ } keys( %{ $configFNAPar } );
    foreach my $cons ( @arrConsTmp ){
        $fnaTmp->createAndAddConstraints( $cons, $configFNAPar->{ $cons } );
    }

    if ( defined $configFNAPar->{ 'Column' } ){
        $fnaTmp->setColumn( $configFNAPar->{ 'Column' } );
    }

    #---- Add FNA to list
    $this->{ 'FNAList' }[ @{ $this->{ 'FNAList' } } ] = $fnaTmp;

    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Added FNA (N=>" . $sNamePar . "< A=>" . $configFNAPar->{ 'Append' } . "<)" );
    }

    return 0;
}
# V103 End

sub addIndex{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "addIndex" );
    #}

    #---- Get parameter
    #
    # 1. Name
    # 2. Index (config hash)
    #
    my $sNamePar     = shift;
    my $configIDXPar = shift;

    #---- Assert parameter
    if (    $sNamePar eq ""
         || not defined $configIDXPar->{ 'Constraints' }
         || not defined $configIDXPar->{ 'Value' }
       ){
        if ( $bLog == $TRUE ){
            if ( $sNamePar ne "" ){
                $theLogger->logMessage( "Warning! Ignored config Index (" . $sNamePar . ") due to missing Constraints or Value" );
            }
            else{
                $theLogger->logMessage( "Warning! Ignored config Index due to missing Name" );
            }
        }
        return -1;
    }

    #---- Create index
    my $idxTmp = new a2w::core::dm::Index();

    #---- Fill in details
    $idxTmp->setName( $sNamePar );
    $idxTmp->setValue( $configIDXPar->{ 'Value' } );
    $idxTmp->setApplicableOutputs( $configIDXPar->{ 'ApplicableOutputs' } ); # V103 Change

    # V101 Begin
    #$idxTmp->createAndAddConstraints( $configIDXPar->{ 'Constraints' } );
    #
    my @arrConsTmp = grep { lc( $_ ) =~ /constraints*/ } keys( %{ $configIDXPar } );
    foreach my $cons ( @arrConsTmp ){
        $idxTmp->createAndAddConstraints( $cons, $configIDXPar->{ $cons } );
    }
    # V101 End

    #---- Add index to list
    $this->{ 'IDXList' }[ @{ $this->{ 'IDXList' } } ] = $idxTmp;

    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Added index (N=>" . $sNamePar . "< V=>" . $configIDXPar->{ 'Value' } . "<)" );
    }

    return 0;
}

sub collectIndex{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "collectIndex" );
    #}

    #---- Get parameter
    #
    # 1. Object
    # 2. POM object wrapper
    #
    my $a2wObjectPar = shift;
    my $pomObjectPar = shift;

    #---- Iterate through indexes and evaluate value ----#
    my $idxTmp  = undef;
    my $iRetTmp = -1;
    my @arrIndexesTmp = @{ $this->{ 'IDXList' } };

    #---- Filter out detected indexes
    @arrIndexesTmp = grep { $_->getResult() eq '' } @arrIndexesTmp;
    if ( @arrIndexesTmp <= 0 ){
        return 0;
    }

    for (   my $i = 0
          # V102 Begin
          ; (    $i < @arrIndexesTmp
              #&& $iRetTmp < 0
          # V102 End
            )
          ; $i++
        ){
        $idxTmp = @arrIndexesTmp[ $i ];
        $iRetTmp = $idxTmp->evaluate( $a2wObjectPar, $pomObjectPar );
    }

    return $iRetTmp;
}

sub doesAllIndexesCollected{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "doesAllIndexesCollected" );
    #}

    #---- Iterate through indexes and evaluate value ----#
    my @arrIndexesTmp = @{ $this->{ 'IDXList' } };

    #---- Filter out detected indexes
    @arrIndexesTmp = grep { $_->getResult() eq '' } @arrIndexesTmp;

    return ( @arrIndexesTmp == 0 ) ? $TRUE : $FALSE;
}

sub getIndex{
    my $this = shift;

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "getIndex" );
    #}

    #---- Get parameter
    #
    # 1. Name
    #
    my $sNamePar = shift;

    #---- Fetch index matching given index name ----#
    my @arrIndexesTmp = @{ $this->{ 'IDXList' } };

    #---- Filter out needed index
    @arrIndexesTmp = grep { $_->getName() eq $sNamePar } @arrIndexesTmp;

    return ( @arrIndexesTmp > 0 ) ? @arrIndexesTmp[ 0 ] : undef;
}

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__

