#-------------------------------------------------------------------------------
#  a2w::core::dm::ListBlockFormatter.pm
#  Data Mining Framework List Block Formatter
#
#  Author  : AFP2web Team, Maas Holding GmbH
#
#  $V100   2018-11-29    Initial Release
#
#-------------------------------------------------------------------------------
package a2w::core::dm::ListBlockFormatter;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;
use a2w::core::dm::Constants;
use a2w::core::dm::BlockFormatter;

#-----------------------------------------------------------------------
# Inherit from base class
#-----------------------------------------------------------------------
our @ISA = qw( a2w::core::dm::BlockFormatter );

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Instantiate from base class to inherit base class attributes
    my $this = a2w::core::dm::BlockFormatter->new( @_ );

    #---- Add this derived class specific attributes
    #$this->{ 'someattribute' } = 0;

    bless( $this, $class );

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "new()" ); }
    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# writeFormattedBlock
#
# Writes block content formatted as list entries
#
# >=0, in case of successful writing
# < 0, in case of error
#
#-----------------------------------------------------------------------
sub writeFormattedBlock{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeFormattedBlock()" ); }

    #---- Parameter
    #
    # 1. Block
    # 2. Visitor
    #
    my $blkCurrentPar = shift;
    my $outVisitorPar = shift;

    #---- Fetch content definition of block
    my $contDefTmp = $blkCurrentPar->getContentDef();
    if ( lc( $contDefTmp->getType() ) ne "list" ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Warning! List block formatter can not process other type (" . $contDefTmp->getType() . ") of blocks<" ); }
        return -1;
    }

    #---- Assert block has content or not ----#
    my $arefObjsListTmp = $blkCurrentPar->getObjects();
    my @arrObjsListTmp = @{ $arefObjsListTmp };

    #---- Get next block in chain
    my $blkNextTmp = $blkCurrentPar->getNextRef();
    if ( @arrObjsListTmp <= 0 ){
        # When current block (or chained block) is empty, ensure to continue with next block in
        # chain (if any exists) else return no error (so that formatting completed properly)
        $blkCurrentPar->setFlushed( $TRUE );
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Warning! Skipped writing empty block (" . $blkCurrentPar->getId() . ")" ); }

        #---- Write chained block contents ----#
        if ( $blkNextTmp != undef ){
            #---- Check and create formatter
            my $fmtNextTmp = $blkNextTmp->getFormatter();
            if ( $fmtNextTmp == undef ){
                $blkNextTmp->_createContentFormatter();
                $fmtNextTmp = $blkNextTmp->getFormatter();
            }

            #---- Write next block
            $iRetTmp = $fmtNextTmp->writeFormattedContent( # Block
                                                             $blkNextTmp
                                                           # Visitor
                                                           , $outVisitorPar
                                                           # List item count
                                                           , 0
                                                         );
            if ( $iRetTmp < 0 ){ return $iRetTmp; }
        }
        return 0;
    }

    #---- Evaluate column positions
    #
    # 1. Ordered list will have two columns (label and body)
    # 2. Unordered list will have only one column (body)
    #
    my $hrefColsTmp = { 'label' => $blkCurrentPar->getStartX(), 'body' => $contDefTmp->getBodyPosition() };
    my @arrColsTmp = sort { $hrefColsTmp->{ $a } <=> $hrefColsTmp->{ $b } } keys( %{ $hrefColsTmp } );
    my @arrColXPosTmp = map { $hrefColsTmp->{ $_ } } @arrColsTmp;

    #---- Detect first list item Y position
    my $iCurrentYTmp = 0;
    my $iFontHeightTmp = 0;
    foreach my $a2wFirstCellTmp ( @arrObjsListTmp ){
        if (    $a2wFirstCellTmp != undef
             && $outVisitorPar->isWritable( $a2wFirstCellTmp ) == $TRUE
             && $this->_isSupported( $a2wFirstCellTmp ) == $TRUE
           ){
            $iCurrentYTmp = $a2wFirstCellTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_ADJ_YPOS };
            my $a2wFontTmp = $a2wFirstCellTmp->{ 'A2WOBJ' }->getFont();
            if ( $a2wFontTmp != undef ){
                $iFontHeightTmp = $a2wFontTmp->getHeight();
                $iFontHeightTmp *= $a2wFirstCellTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_RESOLUTION } / 72;
            }
            last;
        }
    }

    #---- Begin List
    my $iRetTmp = $outVisitorPar->beginList( $blkCurrentPar, \@arrColXPosTmp, ( $iCurrentYTmp - $iFontHeightTmp ) );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- Write formatted content of block ----#
    $iRetTmp = $this->_writeFormattedContent( # Block
                                                $blkCurrentPar
                                              # Visitor
                                              , $outVisitorPar
                                              # List item count
                                              , 0
                                            );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- End List
    if ( $this->isFinalized() == $FALSE ){ $iRetTmp = $outVisitorPar->endList( $blkCurrentPar ); }
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    return 0;
}

#-----------------------------------------------------------------------
# _writeFormattedContent
#
# Writes block content formatted as list entries (low level implementation)
#
# >=0, in case of successful writing
# < 0, in case of error
#
#-----------------------------------------------------------------------
sub _writeFormattedContent{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_writeFormattedContent()" ); }

    #---- Parameter
    #
    # 1. Block
    # 2. Visitor
    # 3. List item count
    #
    my $blkCurrentPar = shift;
    my $outVisitorPar = shift;
    my $iListItemCountPar  = shift;
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Block: Id=>" . $blkCurrentPar->getId() . "<" ); }

    #---- Set flushed
    $blkCurrentPar->setFlushed( $TRUE );

    #---- Fetch content definition of block
    my $contDefTmp = $blkCurrentPar->getContentDef();
    if ( lc( $contDefTmp->getType() ) ne "list" ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Warning! Table block formatter can not process other type (" . $contDefTmp->getType() . ") of blocks" ); }
        return -1;
    }

    #---- Evaluate column positions
    #
    # 1. Ordered list will have two columns (label and body)
    # 2. Unordered list will have only one column (body)
    #
    my $hrefColsTmp = { 'label' => $blkCurrentPar->getStartX(), 'body' => $contDefTmp->getBodyPosition() };
    my @arrColsTmp = sort { $hrefColsTmp->{ $a } <=> $hrefColsTmp->{ $b } } keys( %{ $hrefColsTmp } );
    my @arrColXPosTmp = map { $hrefColsTmp->{ $_ } } @arrColsTmp;

    #---- Assert block has content or not ----#
    my $arefObjsListTmp = $blkCurrentPar->getObjects();
    my @arrObjsListTmp = @{ $arefObjsListTmp };

    #---- Get next block in chain
    my $blkNextTmp = $blkCurrentPar->getNextRef();
    if ( @arrObjsListTmp <= 0 ){
        # When current block (or chained block) is empty, ensure to continue with next block in
        # chain (if any exists) else return no error (so that formatting completed properly)
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Warning! Skipped writing empty block (" . $blkCurrentPar->getId() . ")" ); }

        #---- Write chained block contents ----#
        if ( $blkNextTmp != undef ){
            #---- Check and create formatter
            my $fmtNextTmp = $blkNextTmp->getFormatter();
            if ( $fmtNextTmp == undef ){
                $blkNextTmp->_createContentFormatter();
                $fmtNextTmp = $blkNextTmp->getFormatter();
            }

            #---- Write next block
            $iRetTmp = $fmtNextTmp->_writeFormattedContent( # Block
                                                              $blkNextTmp
                                                            # Visitor
                                                            , $outVisitorPar
                                                            # List item count
                                                            , $iListItemCountPar
                                                          );
            if ( $iRetTmp < 0 ){ return $iRetTmp; }
        }
        return 0;
    }

    #---- Detect first list item Y position
    my $iCurrentYTmp = 0;
    my $iObjResolutionTmp = 0;
    foreach my $a2wFirstCellTmp ( @arrObjsListTmp ){
        if (    $a2wFirstCellTmp != undef
             && $outVisitorPar->isWritable( $a2wFirstCellTmp ) == $TRUE
             && $this->_isSupported( $a2wFirstCellTmp ) == $TRUE
           ){
            $iCurrentYTmp = $a2wFirstCellTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_ADJ_YPOS };
            $iObjResolutionTmp = $a2wFirstCellTmp->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_RESOLUTION };
            last;
        }
    }

    my $iRetTmp = -1;
    my $a2wObjTmp = undef;
    my $pomObjTmp = undef;
    my $elemObjTmp = undef;
    my $bOrderedTmp = $contDefTmp->isOrdered();

    my %hshLineTmp = map { $_, undef } @arrColsTmp;
    my $hrefLineTmp = \%hshLineTmp;

    my $iLineCountTmp = 0;
    my $iObjCountTmp = @arrObjsListTmp;
    my @arrListLinesTmp = ();

    my $iLastObjectIndexTmp = -1;
    my $bLastLineProcessedTmp = $FALSE;

    #---- Process and collect list lines ----#
    for ( my $o = 0; $o < $iObjCountTmp; $o++ ){
        $elemObjTmp = $arrObjsListTmp[ $o ]; # Get object from list

        #---- Assert whether visitor supports this object type ----#
        if (    $outVisitorPar->isWritable( $elemObjTmp ) == $FALSE
             || $this->_isSupported( $elemObjTmp ) == $FALSE
           ){
            #---- Trick to process last line (assert for last object and extend the iteration to run exactly once)
            if ( $o == ( $iObjCountTmp - 1 ) && $bLastLineProcessedTmp == $FALSE ){
                $bLastLineProcessedTmp = $TRUE;
                $iCurrentYTmp = 0;
                $o = $iLastObjectIndexTmp - 1;
            }

            next; # go to next object
        }

        $iLastObjectIndexTmp = $o;
        $a2wObjTmp = $elemObjTmp->{ 'A2WOBJ' };
        $pomObjTmp = $elemObjTmp->{ 'POMOBJ' };
        #if ( $bLog == $TRUE && $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE } == $a2w::core::dm::Constants::OT_TEXT && $bLastLineProcessedTmp == $FALSE ){ $theLogger->logMessage( "Object $o:>" . $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE } . "<" ); }

        if ( $iCurrentYTmp != $pomObjTmp->{ $a2w::core::dm::Constants::AT_ADJ_YPOS } ){
            $arrListLinesTmp[ $iLineCountTmp ] = $hrefLineTmp;
            $iLineCountTmp++;
            $iCurrentYTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_ADJ_YPOS };

            if ( $bLog == $TRUE ){
                my $sLineTmp = "|";
                foreach my $colId ( @arrColsTmp ){
                    my $sSeparatorTmp = '';
                    my @arrElementsTmp = @{ $hrefLineTmp->{ $colId } };
                    foreach my $elem ( @arrElementsTmp ){
                        if ( $elem->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } != $a2w::core::dm::Constants::OT_TEXT ){ next; }
                        $sLineTmp .= $sSeparatorTmp . $elem->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
                        $sSeparatorTmp = ' ';
                    }
                    $sLineTmp .= "|";
                }
                $theLogger->logMessage( "Detected Line $iLineCountTmp: $sLineTmp" );
            }

            #---- Revert object list array index to start over collection of line objects from current object again
            if ( $bLastLineProcessedTmp == $FALSE ){ $o--; }

            #---- Reinitialize line
            my %hshLineTmp = map { $_, undef } @arrColsTmp;
            $hrefLineTmp = \%hshLineTmp;

            next;
        }

        #---- Detect to which column the current object belongs to ----#
        for ( my $i = @arrColsTmp - 1; $i >= 0; $i-- ){
            if ( $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS } >= $hrefColsTmp->{ @arrColsTmp[ $i ] } ){
                $hrefLineTmp->{ @arrColsTmp[ $i ] }[ @{ $hrefLineTmp->{ @arrColsTmp[ $i ] } } ] = $elemObjTmp;
                last; # break this loop
            }
        } # for ( my $i = @arrColsTmp - 1; $i >= 0; $i-- ){

        #---- Trick to process last line (assert for last object and extend the iteration to run exactly once)
        if ( $o == ( $iObjCountTmp - 1 ) && $bLastLineProcessedTmp == $FALSE ){
            $bLastLineProcessedTmp = $TRUE;
            $iCurrentYTmp = 0;
            $o = $iLastObjectIndexTmp - 1;
        }
    }

    #---- Process list lines and make list items out of them appropriately using following rules ----#
    # Rules:
    # 1. If first line is caption, then write first line as caption and continue with further lines as usual
    # 2. A line having valid value for label cell means beginning of list item
    # 3. If current line is last line of list, then write current list item
    # 4. A line having no value for all primary column cells means line is part of current list item (list item has multi line value)
    #
    my $tl = 0;
    my $lstLineTmp = undef;
    my $lstLineNrTmp = 0;
    my $lstLineCountTmp = @arrListLinesTmp;

    # 1. If first line is caption, then write first line as caption and continue with further lines as usual
    my $tlc = 0;
    my $iListItemCountTmp = $iListItemCountPar;
    if ( $contDefTmp->hasCaption() == $TRUE ){
        my $captionLineTmp = $arrListLinesTmp[ $tl++ ]; # fetch first line
        $tlc++;

        #---- Collect caption line objects
        my @arrCaptionObjsTmp = ();
        foreach my $colId ( @arrColsTmp ){
            my @arrElementsTmp = @{ $captionLineTmp->{ $colId } };
            push( @arrCaptionObjsTmp, @arrElementsTmp );
        } # foreach my $colId ( @arrColsTmp )

        #---- Write first line as caption
        if ( @arrCaptionObjsTmp > 0 ){ $outVisitorPar->writeListCaption( \@arrCaptionObjsTmp ); }
    }

    #---- Iterate through list lines and detect list items ----#
    my @arrLinePrimaryColsTmp = ();
    my @arrPrimaryColsTmp = ( 'label' );

    for ( ; $tl < $lstLineCountTmp; $tl++, $tlc++ ){
        $lstLineNrTmp = $tl + 1;
        $lstLineTmp = $arrListLinesTmp[ $tl ];

        if ( $bLog == $TRUE ){
            my $sLineTmp = "|";
            foreach my $colId ( @arrColsTmp ){
                my $sSeparatorTmp = '';
                my @arrElementsTmp = @{ $lstLineTmp->{ $colId } };
                foreach my $elem ( @arrElementsTmp ){
                    if ( $elem->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJTYPE } != $a2w::core::dm::Constants::OT_TEXT ){ next; }
                    $sLineTmp .= $sSeparatorTmp . $elem->{ 'POMOBJ' }->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_VALUE };
                    $sSeparatorTmp = ' ';
                }
                $sLineTmp .= "|";
            }
            $theLogger->logMessage( "Line $lstLineNrTmp: $sLineTmp" );
        }

        # 2. A line having valid value for label cell means beginning of list item
        @arrLinePrimaryColsTmp = grep { $lstLineTmp->{ $_ } == undef } @arrPrimaryColsTmp;
        if ( @arrLinePrimaryColsTmp <= 0 ){ # line has all primary column cells
            if ( $hrefCurrentListItemTmp == undef ){
                if ( $bLog == $TRUE ){ $theLogger->logMessage( "Line $lstLineNrTmp is new list item but held from writing to collect following lines (if any exist)" ); }
        
                #---- Assign current line as new list item
                $hrefCurrentListItemTmp = $lstLineTmp;
            }
            else {
                if ( $bLog == $TRUE ){ $theLogger->logMessage( "Line $lstLineNrTmp is new list item." ); }

                #---- Write list item (collected so far)
                $iListItemCountTmp++;
                if ( $bLog == $TRUE ){ $theLogger->logMessage( "Writing list item $iListItemCountTmp" ); }
                $iRetTmp = $this->_writeListItem( # Visitor
                                                    $outVisitorPar
                                                  # List item
                                                  , $hrefCurrentListItemTmp
                                                  # List item number
                                                  , $iListItemCountTmp
                                                  # Column names
                                                  , \@arrColsTmp
                                                  # Column positions
                                                  , $hrefColsTmp
                                                  # Ordered or not flag
                                                  , $bOrderedTmp
                                                );
                if ( $iRetTmp < 0 ){ return $iRetTmp; }

                #---- Assign current line as new list item
                $hrefCurrentListItemTmp = $lstLineTmp;
            }
        }

        # 3. If current line is last line of list, then write current list item (after merging current line with current list item)
        elsif ( $lstLineNrTmp == $lstLineCountTmp ){
            if ( $blkNextTmp == undef ){ # Ensures this is last line of list
                if ( $bLog == $TRUE ){ $theLogger->logMessage( "Line $lstLineNrTmp is last line of current list item due to page end" ); }

                #---- Merge last line with current list item
                foreach my $colId ( @arrColsTmp ){
                    my @arrElementsTmp = @{ $lstLineTmp->{ $colId } };
                    foreach my $elem ( @arrElementsTmp ){
                        $hrefCurrentListItemTmp->{ $colId }[ @{ $hrefCurrentListItemTmp->{ $colId } } ] = $elem;
                    }
                }

                #---- Write list item (collected so far)
                $iListItemCountTmp++;
                if ( $bLog == $TRUE ){ $theLogger->logMessage( "Writing list item $iListItemCountTmp" ); }
                $iRetTmp = $this->_writeListItem( # Visitor
                                                    $outVisitorPar
                                                  # List item
                                                  , $hrefCurrentListItemTmp
                                                  # List item number
                                                  , $iListItemCountTmp
                                                  # Column names
                                                  , \@arrColsTmp
                                                  # Column positions
                                                  , $hrefColsTmp
                                                  # Ordered or not flag
                                                  , $bOrderedTmp
                                                );
                if ( $iRetTmp < 0 ){ return $iRetTmp; }

                #---- Clean up current list item
                $hrefCurrentListItemTmp = undef;
            }
            else {
                #---- Merge last line with current list item
                foreach my $colId ( @arrColsTmp ){
                    my @arrElementsTmp = @{ $lstLineTmp->{ $colId } };
                    foreach my $elem ( @arrElementsTmp ){
                        $hrefCurrentListItemTmp->{ $colId }[ @{ $hrefCurrentListItemTmp->{ $colId } } ] = $elem;
                    }
                }

                # Retain the current item of list until all lines are collected (where lines may run across chained blocks or across pages)
                $this->{ 'PageLastRow' } = $hrefCurrentListItemTmp;
            }
        }

        # 4. A line having no value for all primary column cells means line is part of current list item (list item has multi line cell value)
        else {
            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Line $lstLineNrTmp is following line of list item " . ( $iListItemCountTmp + 1 ) . ", appending line to list item." ); }

            #---- Merge last line with current list item
            foreach my $colId ( @arrColsTmp ){
                my @arrElementsTmp = @{ $lstLineTmp->{ $colId } };
                foreach my $elem ( @arrElementsTmp ){
                    $hrefCurrentListItemTmp->{ $colId }[ @{ $hrefCurrentListItemTmp->{ $colId } } ] = $elem;
                }
            }
        }
    }

    #---- Write list item (collected so far)
    if ( $hrefCurrentListItemTmp != undef ){
        $iListItemCountTmp++;
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Writing list item $iListItemCountTmp" ); }
        $iRetTmp = $this->_writeListItem( # Visitor
                                            $outVisitorPar
                                          # List item
                                          , $hrefCurrentListItemTmp
                                          # List item number
                                          , $iListItemCountTmp
                                          # Column names
                                          , \@arrColsTmp
                                          # Column positions
                                          , $hrefColsTmp
                                          # Ordered or not flag
                                          , $bOrderedTmp
                                        );
        if ( $iRetTmp < 0 ){ return $iRetTmp; }

        #---- Clean up current list item
        $hrefCurrentListItemTmp = undef;
    }

    #---- Write chained block contents ----#
    if ( $blkNextTmp != undef ){
        #---- Check and create formatter
        my $fmtNextTmp = $blkNextTmp->getFormatter();
        if ( $fmtNextTmp == undef ){
            $blkNextTmp->_createContentFormatter();
            $fmtNextTmp = $blkNextTmp->getFormatter();
        }

        #---- Assert whether next block is of same type
        # a. If it is same, continue current block state on visitor
        # b. If it is NOT same, finalize current block state on visitor and let next block started properly
        if ( lc( $blkNextTmp->getContentDefType() ) eq "list" ){
            #---- Continue next block writing
            $iRetTmp = $fmtNextTmp->_writeFormattedContent( # Block
                                                              $blkNextTmp
                                                            # Visitor
                                                            , $outVisitorPar
                                                            # List item count
                                                            , $iListItemCountTmp
                                                          );
            $this->setFinalized( $fmtNextTmp->isFinalized() ); # Current block is also finalized, if block is finalized on next block
        }
        else {
            #---- Finalize current block
            $this->setFinalized( $TRUE );
            $iRetTmp = $outVisitorPar->endList( $blkCurrentPar );
            if ( $iRetTmp < 0 ){ return $iRetTmp; }

            #---- Write next block
            $iRetTmp = $blkNextTmp->write( $outVisitorPar );
        }
        if ( $iRetTmp < 0 ){ return $iRetTmp; }
    }

    return 0;
}

#-----------------------------------------------------------------------
# _isSupported
#
# Asserts whether given object is supported to be list content
#
# Returns
# TRUE means object can be list content
# FALSE means object can not be list content
#
#-----------------------------------------------------------------------
sub _isSupported{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_isSupported()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Object hash (having both kernel object and pom object)
    #
    my $hrefObjPar = shift;

    #---- Assert object
    my $a2wObjTmp = $hrefObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefObjPar->{ 'POMOBJ' };
    my $bRetTmp   = $TRUE;

    my $iObjTypeTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE };
    unless ( $iObjTypeTmp == $a2w::core::dm::Constants::OT_TEXT ){
        $bRetTmp = $FALSE;
    }

    return $bRetTmp;
}

#-----------------------------------------------------------------------
# _writeListItem
#
# write an item of list
#
# Returns
# >=0 in case of success
# <0 in case of error
#
#-----------------------------------------------------------------------
sub _writeListItem{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_writeListItem()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Visitor (a2w::core::visitor::Visitor object)
    # 2. List item cells (Hash Reference)
    # 3. List item number (integer)
    # 4. Column names (Array reference)
    # 5. Column positions (Hash reference)
    # 6. Ordered or not flag (integer)
    #
    my $outVisitorPar   = shift;
    my $hrefListItemPar = shift;
    my $iListItemNrPar  = shift;
    my $arefColNamesPar = shift;
    my $hrefColPosPar   = shift;
    my $bOrderedPar     = shift;

    #---- Determine the page where the list item starts
    my $iRetTmp = 0;
    my @arrColsTmp = @{ $arefColNamesPar };

    my @arrCellObjsTmp = ();
    my $arefCellObjsTmp = undef;
    my $iListItemStartPageIdTmp = -1;
    for ( my $c = 0; $c < @arrColsTmp; $c++ ){
        $arefCellObjsTmp = $hrefListItemPar->{ @arrColsTmp[ $c ] };
        if ( $arefCellObjsTmp == undef ){ next; }

        @arrCellObjsTmp = @{ $arefCellObjsTmp };
        if ( @arrCellObjsTmp <= 0 ){ next; }

        $iListItemStartPageIdTmp = @arrCellObjsTmp[ 0 ]->{ 'POMOBJ' }{ $a2w::core::dm::Constants::AT_PAGE_ID };
        last; # break the loop after identifying the list item start page id
    }

    #---- Write a list item ----#
    $iRetTmp = $outVisitorPar->beginListItem( # List item number
                                                $iListItemNrPar
                                              # Page id where list item started
                                              , $iListItemStartPageIdTmp
                                            );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- Write Ordered list ----#
    if ( $bOrderedPar == $TRUE ){
        #---- Write label and body of list item
        my $sColTmp = '';
        for ( my $i = 0; $i < @arrColsTmp; $i++ ){
            $sColTmp = @arrColsTmp[ $i ];

            #---- Write label/body
            if ( lc( $sColTmp ) eq 'label' ){
                $iRetTmp = $outVisitorPar->writeListLabel( $hrefListItemPar->{ $sColTmp } );
            }
            elsif ( lc( $sColTmp ) eq 'body' ){
                $iRetTmp = $outVisitorPar->writeListBody( $hrefListItemPar->{ $sColTmp } );
            }
            if ( $iRetTmp < 0 ){ return $iRetTmp; }

            $hrefListItemPar->{ $sColTmp } = undef; # Reset label/body value
        }
    }
    #---- Write Unordered list ----#
    else {
        #---- Merge all column objects as one and write as list item body ----#
        my $sColTmp = '';
        my @arrBodyObjsTmp = ();
        for ( my $i = 0; $i < @arrColsTmp; $i++ ){
            $sColTmp = @arrColsTmp[ $i ];
            if ( ref( $hrefListItemPar->{ $sColTmp } ) eq "ARRAY" ){
                push( @arrBodyObjsTmp, @{ $hrefListItemPar->{ $sColTmp } } );
            }
        }

        #---- Write list item body
        $iRetTmp = $outVisitorPar->writeListBody( \@arrBodyObjsTmp );
        if ( $iRetTmp < 0 ){ return $iRetTmp; }

        #---- Cleanup
        for ( my $i = 0; $i < @arrColsTmp; $i++ ){ $hrefListItemPar->{ $arrColsTmp[ $i ] } = undef; }
    }

    #---- End a list item
    $iRetTmp = $outVisitorPar->endListItem( $iListItemNrPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    return 0;
}

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
